#charset "utf-8"

/*
 *   Czech version is copyright 2010 Tomas Blaha.
 *
 *   Based on English language module by Michael J. Roberts with permisson.
 *   Original copyright below:
 *
 *   Copyright (c) 2000, 2006 Michael J. Roberts.  All Rights Reserved. 
 *
 *   TADS 3 Library - "neutral" messages for US English
 *
 *   This module provides standard library messages with a parser/narrator 
 *   that's as invisible (neutral) as possible.  These messages are designed 
 *   to reduce the presence of the computer as mediator in the story, to 
 *   give the player the most direct contact that we can with the scenario.
 *
 *   The parser almost always refers to itself in the third person (by 
 *   calling itself something like "this story") rather than in the first 
 *   person, and, whenever possible, avoids referring to itself in the first 
 *   place.  Our ideal phrasing is either second-person, describing things 
 *   directly in terms of the player character's experience, or "no-person," 
 *   simply describing things without mentioning the speaker or listener at 
 *   all.  For example, rather than saying "I don't see that here," we say 
 *   "you don't see that here," or "that's not here."  We occasionally stray 
 *   from this ideal where achieving it would be too awkward.
 *
 *   In the earliest days of adventure games, the parser was usually a 
 *   visible presence: the early parsers frequently reported things in the 
 *   first person, and some even had specific personalities.  This 
 *   conspicuous parser style has become less prevalent in modern games, 
 *   though, and authors now usually prefer to treat the parser as just 
 *   another part of the user interface, which like all good UI's is best 
 *   when the user doesn't notice it.  
 */

#include "adv3.h"
#include "cs_cz.h"

/* ------------------------------------------------------------------------ */
/*
 *   Build a message parameter string with the given parameter type and
 *   name.
 *   
 *   This is useful when we have a name from a variable, and we need to
 *   build the message substitution string for embedding in a larger
 *   string.  We can't just embed the name variable using <<var>>, because
 *   that would process the output piecewise - the output filter needs to
 *   see the whole {typ var} expression in one go.  So, instead of writing
 *   this:
 *   
 *.     {The/he <<var>>} {is} ...
 *   
 *   write this:
 *   
 *.     <<buildParam('The/he', var)>> {is} ...
 */
buildParam(typeString, nm)
{
    return '{' + typeString + ' ' + nm + '}';
}

/*
 *   Synthesize a message parameter, and build it into a parameter string
 *   with the given substitution type.
 *   
 *   For example, buildSynthParam('abc', obj) returns '{abc xxx}', where
 *   'xxx' is a synthesized message parameter name (created using
 *   gSynthMessageParam) for the object obj.  
 */
buildSynthParam(typeString, obj)
{
    return '{' + typeString + ' ' + gSynthMessageParam(obj) + '}';
}


/* ------------------------------------------------------------------------ */
/*
 *   Library Messages 
 */
libMessages: MessageHelper
    /*
     *   Flag: offer an explanation of the "OOPS" command when it first
     *   comes up.  We'll only show this the first time the player enters
     *   an unknown word.  If you never want to offer this message at all,
     *   simply set this flag to nil initially.
     *   
     *   See also oopsNote() below.  
     */
    offerOopsNote = true

    /*
     *   some standard commands for insertion into <a> tags - these are in
     *   the messages so they can translated along with the command set
     */
    commandLookAround = 'rozhlédni se'
    commandFullScore = 'detailní skóre'
    
    /* announce a completely remapped action */
    announceRemappedAction(action)
    {
        return '<./p0>\n<.assume>' + action.getParticiplePhrase()
            + '<./assume>\n';
    }

    /*
     *   Get a string to announce an implicit action.  This announces the
     *   current global action.  'ctx' is an ImplicitAnnouncementContext
     *   object describing the context in which the message is being
     *   displayed.  
     */
    announceImplicitAction(action, ctx)
    {
        /* build the announcement from the basic verb phrase */
        return ctx.buildImplicitAnnouncement(action.getImplicitPhrase(ctx));
    }

    /*
     *   Announce a silent implied action.  This allows an implied action
     *   to work exactly as normal (including the suppression of a default
     *   response message), but without any announcement of the implied
     *   action. 
     */
    silentImplicitAction(action, ctx) { return ''; }

    /*
     *   Get a string to announce that we're implicitly moving an object to
     *   a bag of holding to make room for taking something new.  If
     *   'trying' is true, it means we want to phrase the message as merely
     *   trying the action, not actually performing it.  
     */
    announceMoveToBag(action, ctx)
    {
        /* build the announcement, adding an explanation */
        return ctx.buildImplicitAnnouncement(
            action.getImplicitPhrase(ctx) + ' k uvolnění místa');
    }

    /* show a library credit (for a CREDITS listing) */
    showCredit(name, byline) { "<<name>> <<byline>>"; }

    /* show a library version number (for a VERSION listing) */
    showVersion(name, version) { "<<name>> verze <<version>>"; }

    /* there's no "about" information in this game */
    noAboutInfo = "<.parser>Příběh nemá žádné informace O HŘE.<./parser> "

    /*
     *   Show a list state name - this is extra state information that we
     *   show for an object in a listing involving the object.  For
     *   example, a light source might add a state like "(providing
     *   light)".  We simply show the list state name in parentheses.  
     */
    showListState(state) { " (<<state>>)"; }

    /* a set of equivalents are all in a given state */
    allInSameListState(lst, stateName)
    {
        " (<<lst.length() == 2 ? (lst[0].gender <= 2 ? 'oba' : 'obě') :
        (lst[0].gender == 1 ? 'všichni' : lst[0].gender == 4 ? 'všechna' :
        'všechny')>> <<stateName>>)";
    }

    /* generic long description of a Thing from a distance */
    distantThingDesc(obj)
    {
        gMessageParams(obj);
        "{Je|Byl[a obj]} příliš daleko, než a{bys} zahlédl{a} jakékoliv detaily. ";
    }

    /* generic long description of a Thing under obscured conditions */
    obscuredThingDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Pohledem skrz {kohoco obs} {nevid[íš]|[jsi] neviděl[a]} žádné detaily. ";
    }

    /* generic "listen" description of a Thing at a distance */
    distantThingSoundDesc(obj)
        { "Z té dálky {neslyš[íš]|[jsi] neslyšel[a]} nic konkrétního. "; }

    /* generic obscured "listen" description */
    obscuredThingSoundDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Přes {kohoco obs} {neslyš[íš]|[jsi] neslyšel[a]} nic konkrétního. ";
    }

    /* generic "smell" description of a Thing at a distance */
    distantThingSmellDesc(obj)
        { "Na tu dálku {|[jsi]} skoro nic necít{[íš]|il[a]}. "; }

    /* generic obscured "smell" description */
    obscuredThingSmellDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Přes {kohoco obs} {|[jsi]} skoro nic necít{[íš]|il[a]}. ";
    }

    /* generic "taste" description of a Thing */
    thingTasteDesc(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} chutn{á|al[a obj]} v podstatě tak, jak {[bys]|[jsi]}
            očekával{a}. ";
    }

    /* generic "feel" description of a Thing */
    thingFeelDesc(obj)
        { "Nenahmat{áv[áš]|al[a] [jsi]} nic neobvyklého. "; }

    /* obscured "read" description */
    obscuredReadDesc(obj)
    {
        gMessageParams(obj);
        "Nevid{[íš]|ěl[a] [jsi]} {kohoco obj} tak dobře, aby {jsi} {ten obj}
            mohl{a} přečíst. ";
    }

    /* dim light "read" description */
    dimReadDesc(obj)
    {
        gMessageParams(obj);
        "Na přečtení {kohočeho obj} tu {je|bylo} moc málo světla. ";
    }

    /* lit/unlit match description */
    litMatchDesc(obj) { "{Kdoco obj} {je} zapálen{ý}. "; }
    unlitMatchDesc(obj) { "{Kdoco obj} {je} obyčejn{ý} zápalka. "; }

    /* lit candle description */
    litCandleDesc(obj) { "{Kdoco obj} {je} zapálen{ý}. "; }

    /*
     *   Prepositional phrases for putting objects into different types of
     *   objects. 
     */
    putDestContainer(obj) { return 'do ' + obj.nameKohoCeho; }
    putDestSurface(obj) { return 'na ' + obj.nameKohoCo; }
    putDestUnder(obj) { return 'pod ' + obj.nameKohoCo; }
    putDestBehind(obj) { return 'za ' + obj.nameKohoCo; }
    putDestFloor(obj) { return 'na ' + obj.nameKohoCo; }

    /* the list separator character in the middle of a list */
    listSepMiddle = ", "

    /* the list separator character for a two-element list */
    listSepTwo = " a "

    /* the list separator for the end of a list of at least three elements */
    listSepEnd = " a "

    /*
     *   the list separator for the middle of a long list (a list with
     *   embedded lists not otherwise set off, such as by parentheses) 
     *
     *   TODO: Proč si myslí, že exitlister je long list? Jinak by tu klidně
     *   mohl být ten středník pro skutečně dlouhé listy.
     */
    longListSepMiddle = ", "

    /* the list separator for a two-element list of sublists */
    longListSepTwo = " a "

    /* the list separator for the end of a long list */
    longListSepEnd = " a "

    /* show the basic score message */
    showScoreMessage(points, maxPoints, turns)
    {
        local n = turns, m = maxPoints;

        while(n > 9) n /= 10;
        while(m > 9) m /= 10;

        "<<[2, 3, 4].indexOf(n) || (n >= 12 && n <= 14) || (n >= 100
            && n <= 199) ? 'Ve' : 'V'>> <<turns>> <<turns == 1 ? 'tahu' :
            'tazích'>> jsi získal{a} <<points>> <<[2, 3, 4, 6, 7].indexOf(m)
            || (m >= 12 && m <= 14) || (m >= 16 && m <= 17) || (m >= 100 &&
            m <= 199) ? 'ze' : 'z'>> <<maxPoints>> maximálně <<maxPoints == 1
            ? 'možného bodu' : 'možných bodů'>>. ";
    }

    /* show the basic score message with no maximum */
    showScoreNoMaxMessage(points, turns)
    {
        local n = turns;

        while(n > 9) n /= 10;

        "<<[2, 3, 4].indexOf(n) || (n >= 12 && n <= 14) || (n >= 100
            && n <= 199) ? 'Ve' : 'V'>> <<turns>> <<turns == 1 ? 'tahu' :
            'tazích'>> jsi získal{a} <<points>> <<points == 1 ? 'bod'
            : 'bodů'>>. ";
    }

    /* show the full message for a given score rank string */
    showScoreRankMessage(msg) { "To z tebe dělá <<msg>>. "; }

    /*
     *   show the list prefix for the full score listing; this is shown on
     *   a line by itself before the list of full score items, shown
     *   indented and one item per line 
     */
    showFullScorePrefix = "Tvé skóre zahrnuje:"

    /*
     *   show the item prefix, with the number of points, for a full score
     *   item - immediately after this is displayed, we'll display the
     *   description message for the achievement 
     */
    fullScoreItemPoints(points)
    {
        "<<points>> <<points == 1 ? 'bod' : (points < 5 ? 'body' : 'bodů')>> za ";
    }

    /* score change - first notification */
    firstScoreChange(delta)
    {
        scoreChange(delta);
        scoreChangeTip.showTip();
    }

    /* score change - notification other than the first time */
    scoreChange(delta)
    {
        "<.commandsep><.notification><<
        basicScoreChange(delta)>><./notification> ";
    }

    /*
     *   basic score change notification message - this is an internal
     *   service routine for scoreChange and firstScoreChange 
     */
    basicScoreChange(delta)
    {
        "Tvé <<aHref(commandFullScore, 'skóre', 'Ukázat detailní skóre')>>
        se právě <<delta > 0 ? 'zvýšilo' : 'snížilo'>> o
        <<spellInt(delta > 0 ? delta : -delta, 2)>>
        <<delta is in (1, -1) ? 'bod' : (delta is in (4, 3, 2, -2, -3, -4) ?
        'body' : 'bodů')>>.";
    }

    /* acknowledge turning tips on or off */
    acknowledgeTipStatus(stat)
    {
        "<.parser>Tipy jsou nyní <<stat ? 'zapnuté' : 'vypnuté'>>.<./parser> ";
    }

    /* describe the tip mode setting */
    /* TODO: Kde se tohle zobrazuje? Nemělo by být místo "zapnuté" "zapnout"?  */
    tipStatusShort(stat)
    {
        "TIPY <<stat ? 'ZAPNUTÉ' : 'VYPNUTÉ'>>";
    }

    /* get the string to display for a footnote reference */
    footnoteRef(num)
    {
        local str;

        /* set up a hyperlink for the note that enters the "note n" command */
        str = '<sup>[<a href="footnote ' + num + '"><.a>';

        /* show the footnote number in square brackets */
        str += num;
        
        /* end the hyperlink */
        str += '<./a></a>]</sup>';

        /* return the text */
        return str;
    }

    /* first footnote notification */
    firstFootnote()
    {
        footnotesTip.showTip();
    }

    /* there is no such footnote as the given number */
    noSuchFootnote(num)
    {
        "<.parser>Příběh na takovou poznámku neodkazoval.<./parser> ";
    }

    /* show the current footnote status */
    showFootnoteStatus(stat)
    {
        "Nynější nastavení je POZNÁMKY ";
        switch(stat)
        {
        case FootnotesOff:
            "VYPNUTÉ, které skrývá všechny odkazy na poznámky pod čarou.
            Zadej <<aHref('poznámky středně', 'POZNÁMKY STŘEDNĚ',
            'Nastavit poznámky pod čarou na střední zobrazování')>> a
            budou se ukazovat odkazy na všechny poznámky kromě těch, které jsi
            už četl, nebo <<aHref('poznámky naplno', 'POZNÁMKY NAPLNO',
            'Nastavit poznámky na plnou intenzitu')>> a budou se ukazovat
            odkazy na všechny poznámky. ";
            break;

        case FootnotesMedium:
            "STŘEDNĚ, které ukazuje odkazy na zatím nepřečtené poznámky pod
            čarou. Zadej <<aHref('poznámky vypnout', 'POZNÁMKY VYPNOUT',
            'Vypnout poznámky pod čarou')>> a poznámky se zcela vypnou, nebo
            <<aHref('poznámky naplno', 'POZNÁMKY NAPLNO','Nastavit poznámky pod
            čarou na plné zobrazování')>> a budou se ukazovat úplně všechny
            odkazy včetně poznámek, které už jsi četl. ";
            break;

        case FootnotesFull:
            "NAPLNO, které ukazuje odkazy na úplně všechny poznámky pod čarou.
            Zadej <<aHref('poznámky středně', 'POZNÁMKY STŘEDNĚ', 'Nastavit
            poznámky pod čarou na střední zobrazování')>> a budou se ukazovat
            odkazy na všechny poznámky kromě těch, které jsi už četl, nebo
            <<aHref('poznámky vypnout', 'POZNÁMKY VYPNOUT', 'Vypnout poznámky
            pod čarou')>> a poznámky se zcela vypnou. ";
            break;
        }
    }

    /* acknowledge a change in the footnote status */
    acknowledgeFootnoteStatus(stat)
    {
        "<.parser>Nynější nastavení je
        <<shortFootnoteStatus(stat)>>.<./parser> ";
    }

    /* show the footnote status, in short form */
    shortFootnoteStatus(stat)
    {
        "POZNÁMKY <<
          stat == FootnotesOff ? 'VYPNUTÉ'
          : stat == FootnotesMedium ? 'STŘEDNĚ'
          : 'NAPLNO' >>";
    }

    /*
     *   Show the main command prompt.
     *   
     *   'which' is one of the rmcXxx phase codes indicating what kind of
     *   command we're reading.  This default implementation shows the
     *   same prompt for every type of input, but games can use the
     *   'which' value to show different prompts for different types of
     *   queries, if desired.  
     */
    mainCommandPrompt(which) { "\b&gt;"; }

    /*
     *   Show a pre-resolved error message string.  This simply displays
     *   the given string.  
     */
    parserErrorString(actor, msg) { say(msg); }

    /* show the response to an empty command line */
    emptyCommandResponse = "<.parser>Cože {|[jsi]} to {[chceš]|chtěl[a]}?<./parser> "

    /* invalid token (i.e., punctuation) in command line */
    invalidCommandToken(ch)
    {
        "<.parser>Příběh neví, jak se vypořádat se znakem &bdquo;<<ch>>&ldquo;
            v příkazu.<./parser> ";
    }

    /*
     *   Command group prefix - this is displayed after a command line and
     *   before the first command results shown after the command line.
     *   
     *   By default, we'll show the "zero-space paragraph" marker, which
     *   acts like a paragraph break in that it swallows up immediately
     *   following paragraph breaks, but doesn't actually add any space.
     *   This will ensure that we don't add any space between the command
     *   input line and the next text.  
     */
    commandResultsPrefix = '<.p0>'

    /*
     *   Command "interruption" group prefix.  This is displayed after an
     *   interrupted command line - a command line editing session that
     *   was interrupted by a timeout event - just before the text that
     *   interrupted the command line.
     *   
     *   By default, we'll show a paragraph break here, to set off the
     *   interrupting text from the command line under construction.  
     */
    commandInterruptionPrefix = '<.p>'

    /*
     *   Command separator - this is displayed after the results from a
     *   command when another command is about to be executed without any
     *   more user input.  That is, when a command line contains more than
     *   one command, this message is displayed between each successive
     *   command, to separate the results visually.
     *   
     *   This is not shown before the first command results after a
     *   command input line, and is not shown after the last results
     *   before a new input line.  Furthermore, this is shown only between
     *   adjacent commands for which output actually occurs; if a series
     *   of commands executes without any output, we won't show any
     *   separators between the silent commands.
     *   
     *   By default, we'll just start a new paragraph.  
     */
    commandResultsSeparator = '<.p>'

    /*
     *   "Complex" result separator - this is displayed between a group of
     *   messages for a "complex" result set and adjoining messages.  A
     *   command result list is "complex" when it's built up out of
     *   several generated items, such as object identification prefixes
     *   or implied command prefixes.  We use additional visual separation
     *   to set off these groups of messages from adjoining messages,
     *   which is especially important for commands on multiple objects,
     *   where we would otherwise have several results shown together.  By
     *   default, we use a paragraph break.  
     */
    complexResultsSeparator = '<.p>'

    /*
     *   Internal results separator - this is displayed to visually
     *   separate the results of an implied command from the results for
     *   the initiating command, which are shown after the results from
     *   the implied command.  By default, we show a paragraph break.
     */
    internalResultsSeparator = '<.p>'

    /*
     *   Command results suffix - this is displayed just before a new
     *   command line is about to be read if any command results have been
     *   shown since the last command line.
     *   
     *   By default, we'll show nothing extra.  
     */
    commandResultsSuffix = ''

    /*
     *   Empty command results - this is shown when we read a command line
     *   and then go back and read another without having displaying
     *   anything.
     *   
     *   By default, we'll return a message indicating that nothing
     *   happened.  
     */
    commandResultsEmpty =
        ('Nic očividného se nestalo.<.p>')

    /*
     *   Intra-command report separator.  This is used to separate report
     *   messages within a single command's results.  By default, we show
     *   a paragraph break.  
     */
    intraCommandSeparator = '<.p>'

    /*
     *   separator for "smell" results - we ordinarily show each item's
     *   odor description as a separate paragraph 
     */
    smellDescSeparator()
    {
        "<.p>";
    }

    /*
     *   separator for "listen" results 
     */
    soundDescSeparator()
    {
        "<.p>";
    }

    /* a command was issued to a non-actor */
    cannotTalkTo(targetActor, issuingActor)
    {
        "\^<<targetActor.name>> není něco, s čím {[bys]|[jsi]} mohl{a} mluvit. ";
    }

    /* greeting actor while actor is already talking to us */
    alreadyTalkingTo(actor, greeter)
    {
        "\^Už {m[áš]|jsi měl[a]} <<actor.nameKohoCeho>> pozornost. ";
    }

    /* no topics to suggest when we're not talking to anyone */
    noTopicsNotTalking = "<.parser>{Momentálně|Tehdy} {|[jsi]} s nikým nemluv{[íš]|il[a]}.<./parser> "

    /*
     *   Show a note about the OOPS command.  This is, by default, added
     *   to the "I don't know that word" error the first time that error
     *   occurs.  
     */
    oopsNote()
    {
        oopsTip.showTip();
    }

    /* can't use OOPS right now */
    oopsOutOfContext = "<.parser>OPRAVA může být použita jen v ten okamžik,
        kdy se příběh zeptá na neznámé slovo.<./parser> "

    /* OOPS in context, but without the word to correct */
    oopsMissingWord = "<.parser>OPRAVA překlepů se používá tak, že se za slovo
        OPRAVA napíše správné slovo. Např. <q>OPRAVA KNIHA</q>.<./parser> "

    /* acknowledge setting VERBOSE mode (true) or TERSE mode (nil) */
    acknowledgeVerboseMode(verbose)
    {
        if (verbose)
            "<.parser>Je vybrán UPOVÍDANÝ režim.<./parser> ";
        else
            "<.parser>Je vybrán STRUČNÝ režim.<./parser> ";
    }

    /* show the current VERBOSE setting, in short form */
    shortVerboseStatus(stat) { "<<stat ? 'UPOVÍDANÝ' : 'STRUČNÝ'>> režim"; }

    /* show the current score notify status */
    showNotifyStatus(stat)
    {
        "<.parser>Upozorňování na změnu skóre je <<stat ? 'zapnuté'
            : 'vypnuté'>>.<./parser> ";
    }

    /* show the current score notify status, in short form */
    shortNotifyStatus(stat) { "UPOZORŇOVÁNÍ <<stat ? 'ZAPNUTÉ' : 'VYPNUTÉ'>>"; }

    /* acknowledge a change in the score notification status */
    acknowledgeNotifyStatus(stat)
    {
        "<.parser>Upozorňování na změnu skóre je teď <<stat ? 'zapnuté'
            : 'vypnuté'>>.<./parser> ";
    }

    /*
     *   Announce the current object of a set of multiple objects on which
     *   we're performing an action.  This is used to tell the player
     *   which object we're acting upon when we're iterating through a set
     *   of objects specified in a command targeting multiple objects.  
     */
    announceMultiActionObject(obj, whichObj, action)
    {
        return '<./p0>\n<.announceObj>'
            + obj.getAnnouncementDistinguisher().name(obj)
            + ':<./announceObj> <.p0>';
    }

    /*
     *   Announce a singleton object that we selected from a set of
     *   ambiguous objects.  This is used when we disambiguate a command
     *   and choose an object over other objects that are also logical but
     *   are less likely.  In such cases, it's courteous to tell the
     *   player what we chose, because it's possible that the user meant
     *   one of the other logical objects - announcing this type of choice
     *   helps reduce confusion by making it immediately plain to the
     *   player when we make a choice other than what they were thinking.  
     */
    announceAmbigActionObject(obj, whichObj, action)
    {
        /* announce the object in "assume" style, ending with a newline */
        return '<.assume>'
            + obj.getAnnouncementDistinguisher().name(obj)
            + '<./assume>\n';
    }

    /*
     *   Announce a singleton object we selected as a default for a
     *   missing noun phrase.
     *   
     *   'resolvedAllObjects' indicates where we are in the command
     *   processing: this is true if we've already resolved all of the
     *   other objects in the command, nil if not.  We use this
     *   information to get the phrasing right according to the situation.
     */
    announceDefaultObject(obj, whichObj, action, resolvedAllObjects)
    {
        /*
         *   put the action's default-object message in "assume" style,
         *   and start a new line after it 
         */
        return '<.assume>'
            + action.announceDefaultObject(obj, whichObj, resolvedAllObjects)
            + '<./assume>\n';
    }

    /* 'again' used with no prior command */
    noCommandForAgain()
    {
        "<.parser>Není tu nic, co by se dalo opakovat.<./parser> ";
    }

    /* 'again' cannot be directed to a different actor */
    againCannotChangeActor()
    {
        "<.parser>Abys zopakoval příkaz typu <q>želvo, jdi severně,</q>
        tak napiš jen <q>znovu,</q> a ne <q>želvo, znovu.</q><./parser> ";
    }

    /* 'again': can no longer talk to target actor */
    againCannotTalkToTarget(issuer, target)
    {
        "\^<<issuer.name>> nem{ůž[eš]|ohl[a]} zopakovat příkaz. ";
    }

    /* the last command cannot be repeated in the present context */
    againNotPossible(issuer)
    {
        "Teď tento příkaz nemůže být zopakován. ";
    }

    /* system actions cannot be directed to non-player characters */
    systemActionToNPC()
    {
        "<.parser>Tento příkaz nemůže být směrován jiné postavě příběhu.
        <./parser> ";
    }

    /* confirm that we really want to quit */
    confirmQuit()
    {
        "Opravdu chceš skončit?\ (<<aHref('a', 'A', 'Potvrdit ukončení')
        >> znamená souhlas) >\ ";
    }


    /*
     *   QUIT message.  We display this to acknowledge an explicit player
     *   command to quit the game.  This is the last message the game
     *   displays on the way out; there is no need to offer any options at
     *   this point, because the player has decided to exit the game.
     *   
     *   By default, we show nothing; games can override this to display an
     *   acknowledgment if desired.  Note that this isn't a general
     *   end-of-game 'goodbye' message; the library only shows this to
     *   acknowledge an explicit QUIT command from the player.  
     */
    okayQuitting() { }

    /*
     *   "not terminating" confirmation - this is displayed when the
     *   player doesn't acknowledge a 'quit' command with an affirmative
     *   response to our confirmation question 
     */
    notTerminating()
    {
        "<.parser>Příběh pokračuje.<./parser> ";
    }

    /* confirm that they really want to restart */
    confirmRestart()
    {
        "Opravdu chceš začít od znovu?\ (<<aHref('A', 'A',
        'Potvrdit restart')>> znamená souhlas) >\ ";
    }

    /* "not restarting" confirmation */
    notRestarting() { "<.parser>Příběh pokračuje.<./parser> "; }

    /*
     *   Show a game-finishing message - we use the conventional "*** You
     *   have won! ***" format that text games have been using since the
     *   dawn of time. 
     */
    showFinishMsg(msg) { "<.p>*** <<msg>>\ ***<.p>"; }

    /* standard game-ending messages for the common outcomes */
    finishDeathMsg = 'ZEMŘEL{A pc} JSI'
    finishVictoryMsg = ('VYHRÁL{A} JSI')
    finishFailureMsg = ('NEUSPĚL JSI')
    finishGameOverMsg = 'KONEC HRY'

    /*
     *   Get the save-game file prompt.  Note that this must return a
     *   single-quoted string value, not display a value itself, because
     *   this prompt is passed to inputFile(). 
     */
    getSavePrompt =
        'Zvolte prosím soubor k uložení pozice'

    /* get the restore-game prompt */
    getRestorePrompt = 'Zvolte prosím soubor s uloženou pozicí'

    /* successfully saved */
    saveOkay() { "<.parser>Uloženo.<./parser> "; }

    /* save canceled */
    saveCanceled() { "<.parser>Přerušeno.<./parser> "; }

    /* saved failed due to a file write or similar error */
    saveFailed(exc)
    {
        "<.parser>Nepovedlo se. Možná dochází místo na disku a nebo nemáte
        dostatečná práva k uložení souboru.<./parser> ";
    }

    /* note that we're restoring at startup via a saved-position launch */
    noteMainRestore() { "<.parser>Nahrávám uloženou hru...<./parser>\n"; }

    /* successfully restored */
    restoreOkay() { "<.parser>Hra obnovena.<./parser> "; }

    /* restore canceled */
    restoreCanceled() { "<.parser>Přerušeno.<./parser> "; }

    /* restore failed because the file was not a valid saved game file */
    restoreInvalidFile()
    {
        "<.parser>Chyba: To není platný soubor s uloženou pozicí.<./parser> ";
    }

    /* restore failed because the file was corrupted */
    restoreCorruptedFile()
    {
        "<.parser>Selhalo: Soubor s uloženou pozicí vypadá poškozeně. To se
        může stát, pokud byl modifikován jinou aplikací, pokud byl překopírován
        mezi počítači v nebinárním režimu nebo pokud je médium
        poškozené.<./parser> ";
    }

    /* restore failed because the file was for the wrong game or version */
    restoreInvalidMatch()
    {
        "<.parser>Selhalo: Soubor nebyl uložen tímto příběhem (nebo byl uložen
        nekompatibliní verzí příběhu).<./parser> ";
    }

    /* restore failed for some reason other than those distinguished above */
    restoreFailed(exc)
    {
        "<.parser>Selhalo: Nepovedlo se obnovit pozici ze souboru.<./parser> ";
    }

    /* error showing the input file dialog (or whatever) */
    filePromptFailed()
    {
        "<.parser>Nastala systémová chyba při dotazování se na název souboru.
        Počítači se může nedostávat operační paměti nebo má nějaký problém s
        konfugurací.<./parser> ";
    }

    /* PAUSE prompt */
    pausePrompt()
    {
        "<.parser>Příběh je pozastaven. Zmáčkni prosím mezerník, jakmile
        budeš připraven pokračovat nebo zmáčkni klávesu &lsquo;S&rsquo; k
        uložení současné pozice.<./parser><.p>";
    }

    /* saving from within a pause */
    pauseSaving()
    {
        "<.parser>Ukládám příběh...<./parser><.p>";
    }

    /* PAUSE ended */
    pauseEnded()
    {
        "<.parser>Obnovuji příběh.<./parser> ";
    }

    /* acknowledge starting an input script */
    inputScriptOkay(fname)
    {
        "<.parser>Čtu příkazy z from <q><<fname.htmlify()
         >></q>...<./parser>\n ";
    }

    /* get the scripting inputFile prompt message */
    getScriptingPrompt = 'Zadejte prosím název souboru pro zápis průběhu hry'

    /* acknowledge scripting on */
    scriptingOkay()
    {
        "<.parser>Nyní bude průběh hry zapisován do souboru. Zadejte <<aHref(
            'vypnout zápis', 'VYPNOUT ZÁPIS', 'Vypnout zápis průběhu hry')>>
            k ukončení zaznamenávání.<./parser> ";
    }

    /* acknowledge cancellation of script file dialog */
    scriptingCanceled = "<.parser>Zastaveno.<./parser> "

    /* acknowledge scripting off */
    scriptOffOkay = "<.parser>Zápis průběhu hry skončil.<./parser> "

    /* SCRIPT OFF ignored because we're not in a script file */
    scriptOffIgnored = "<.parser>Průběh hry nebyl zaznamenáván.<./parser> "

    /* get the RECORD prompt */
    getRecordingPrompt = 'Zadejte prosím jméno souboru pro záznam příkazů'

    /* acknowledge recording on */
    recordingOkay = "<.parser>Příkazy budou zaznamenávány. Zadejte <<aHref(
        'vypnout záznam', 'VYPNOUT ZÁZNAM', 'Vypnout zaznamenávání příkazů')>>
        k přerušení záznamu příkazů.<.parser> "

    /* acknowledge cancellation */
    recordingCanceled = "<.parser>Zastaveno.<./parser> "

    /* recording turned off */
    recordOffOkay = "<.parser>Záznam příkazů skončil.<./parser> "

    /* RECORD OFF ignored because we're not recording commands */
    recordOffIgnored = "<.parser>Žádný záznam příkazů nebyl
        pořizován.<./parser> "

    /* REPLAY prompt */
    getReplayPrompt = 'Zadejte prosím název souboru s příkazy k přehrání'

    /* REPLAY file selection canceled */
    replayCanceled = "<.parser>Zrušeno.<./parser> "

    /* undo command succeeded */
    undoOkay(actor, cmd)
    {
        "<.parser>Odvolává se jeden tah: <q>";

        /* show the target actor prefix, if an actor was specified */
        if (actor != nil)
            "<<actor>>, ";

        /* show the command */
        "<<cmd>></q>.<./parser><.p>";
    }

    /* undo command failed */
    undoFailed()
    {
        "<.parser>Více kroků k odvolání tahu není k dispozici.<./parser> ";
    }

    /* comment accepted, with or without transcript recording in effect */
    noteWithScript = "<.parser>Komentář zaznamenán.<./parser> "
    noteWithoutScript = "<.parser>Komentář <b>nebyl</b> zaznamenán.<./parser> "

    /* on the first comment without transcript recording, warn about it */
    noteWithoutScriptWarning = "<.parser>Komentář <b>nebyl</b> zaznamenán.
        Zadejte <<aHref('skript', 'SKRIPT', 'Začít zaznamenávat transkript')
          >>, pokud chcete začít zaznamenáva transkript.<./parser> "

    /* invalid finishGame response */
    invalidFinishOption(resp)
    {
        "\bTo není jedna z možností. ";
    }

    /* acknowledge new "exits on/off" status */
    exitsOnOffOkay(stat, look)
    {
        if (stat && look)
            "<.parser>Nyní bude seznam směrů ukazován ve stavové řádce i jako
            součást popisu každé místnosti.<./parser> ";
        else if (!stat && !look)
            "<.parser>Seznam směrů už nebude vypisován ani ve stavové řádce,
            ani jako součást popisu místností.<./parser> ";
        else
            "<.parser>Seznam směrů <<stat ? 'bude' : 'nebude'>> ukazován ve
            stavové řádce a <<look ? 'bude' : 'nebude'>> zahrnut do popisů
            místností.<./parser> ";
    }

    /* explain how to turn exit display on and off */
    explainExitsOnOff()
    {
        exitsTip.showTip();
    }

    /* describe the current EXITS settings */
    currentExitsSettings(statusLine, roomDesc)
    {
        "VÝCHODY ";
        if (statusLine && roomDesc)
            "ZAPNUTY";
        else if (statusLine)
            "V ŘÁDCE";
        else if (roomDesc)
            "V MÍSTNOSTI";
        else
            "VYPNUTY";
    }

    /* acknowledge HINTS OFF */
    hintsDisabled = '<.parser>Nápovědy jsou vypnuty.<./parser> '

    /* rebuff a request for hints when they've been previously disabled */
    sorryHintsDisabled = '<.parser>Promiň, ale jak jsi chtěl{a}, jsou nápovědy
        zablokovány v této relaci. Pokud jsi změnil{a} názor, budeš si muset
        uložit pozici, vyskočit z interpretru a pak ho znovu spustit a začít
        tak nové sezení.<./parser> '

    /* this game has no hints */
    hintsNotPresent = '<.parser>Promiň, ale tento příběh nemá žádné vestavěné
        nápovědy.<./parser> '

    /* there are currently no hints available (but there might be later) */
    currentlyNoHints = '<.parser>Promiň, ale žádné nápovědy  současné době
        nejsou k dispozici. Zkus to později.<./parser> '

    /* show the hint system warning */
    showHintWarning =
       "<.notification>Upozornění: Někteří lidé nemají v oblibě vestavěné
       nápovědy, protože pokušení si předčasně říci o pomoc může být velké,
       když jsou nápovědy tak snadno dosažitelné. Pokud máš pocit, že pokušení
       nevydržíš, můžeš nápovědy vypnout do konce sezení zadáním
       <<aHref('nápovědy vypnout', 'NÁPOVĚDY VYPNOUT')>>. Pokud i přesto chceš
       získat nápovědu, napiš <<aHref('nápověda', 'NÁPOVĚDA')>>.
       <./notification> "

    /* done with hints */
    hintsDone = '<.parser>Hotovo.<./parser> '

    /* optional command is not supported in this game */
    commandNotPresent = "<.parser>Tento příkaz není v tomto příběhu potřeba.
        <./parser> "

    /* this game doesn't use scoring */
    scoreNotPresent = "<.parser>Tento příběh nepočítá skóre.<./parser> "

    /* mention the FULL SCORE command */
    mentionFullScore()
    {
        fullScoreTip.showTip();
    }

    /* SAVE DEFAULTS successful */
    savedDefaults()
    {
        "<.parser>Současné nastavení bylo uloženo jako výchozí nastavení pro
            další hry. Uložená nastavení jsou: ";

        /* show all of the settings */
        settingsUI.showAll();

        ". Většina nových her automaticky převezme tato nastavení, jakmile
        začneš (nebo RESTARTuješ) hry, ale některé starší hry ne.<./parser> ";
    }

    /* RESTORE DEFAULTS successful */
    restoredDefaults()
    {
        "<.parser>Uložené výchozí nastavení vstoupilo v platnost. Nová
            nastavení jsou: ";

        /* show all of the settings */
        settingsUI.showAll();

        ".<./parser> ";
    }

    /* show a separator for the settingsUI.showAll() list */
    settingsItemSeparator = ", "

    /* SAVE/RESTORE DEFAULTS not supported (old interpreter version) */
    defaultsFileNotSupported = "<.parser>Verze TADS interpretru, kterou
        používáš bohužel nepodporuje ukládání a načítání výchozího nastavení.
        Abys mohl tuto vlastnost používat, budeš muset nainstalovat novější
        verzi.<./parser> "

    /* SAVE DEFAULTS file creation error */
    defaultsFileWriteError = "<.parser>Při ukládání výchozího nastavení došlo
        k chybě. Nastavení nebylo uloženo. Možná ti dochází místo na disku nebo
        nemáš dostatečné oprávnění k zápisu souboru.<./parser> "

    /*
     *   Command key list for the menu system.  This uses the format
     *   defined for MenuItem.keyList in the menu system.  Keys must be
     *   given as lower-case in order to match input, since the menu
     *   system converts input keys to lower case before matching keys to
     *   this list.  
     *   
     *   Note that the first item in each list is what will be given in
     *   the navigation menu, which is why the fifth list contains 'ENTER'
     *   as its first item, even though this will never match a key press.
     */
    menuKeyList = [
                   ['o'],
                   ['p', '[left]', '[bksp]', '[esc]'],
                   ['n', '[up]'],
                   ['d', '[down]'],
                   ['ENTER', '\n', '[right]', ' ']
                  ]

    /* link title for 'previous menu' navigation link */
    prevMenuLink = '<font size=-1>Předchozí</font>'

    /* link title for 'next topic' navigation link in topic lists */
    nextMenuTopicLink = '<font size=-1>Následující</font>'

    /*
     *   main prompt text for text-mode menus - this is displayed each
     *   time we ask for a keystroke to navigate a menu in text-only mode 
     */
    textMenuMainPrompt(keylist)
    {
        "\bVyber číslo tématu nebo zmáčkni &lsquo;<<keylist[M_PREV][1]>>&rsquo;
        pro návrat do předchozí nabídky nebo &lsquo;<<keylist[M_QUIT][1]
        >>&rsquo; k opuštění nabídky:\ ";
    }

    /* prompt text for topic lists in text-mode menus */
    textMenuTopicPrompt()
    {
        "\bPro zobrazení další řádky zmáčkni mezerník, pro předchozí nabídku
        &lsquo;<b>P</b>&rsquo;, nebo pro opuštění nabidky
        &lsquo;<b>Q</b>&rsquo;.\b";
    }

    /*
     *   Position indicator for topic list items - this is displayed after
     *   a topic list item to show the current item number and the total
     *   number of items in the list, to give the user an idea of where
     *   they are in the overall list.  
     */
    menuTopicProgress(cur, tot) { " [<<cur>>/<<tot>>]"; }

    /*
     *   Message to display at the end of a topic list.  We'll display
     *   this after we've displayed all available items from a
     *   MenuTopicItem's list of items, to let the user know that there
     *   are no more items available.  
     */
    menuTopicListEnd = '[konec]'

    /*
     *   Message to display at the end of a "long topic" in the menu
     *   system.  We'll display this at the end of the long topic's
     *   contents.  
     */
    menuLongTopicEnd = '[konec]'

    /*
     *   instructions text for banner-mode menus - this is displayed in
     *   the instructions bar at the top of the screen, above the menu
     *   banner area 
     */
    menuInstructions(keylist, prevLink)
    {
        "<tab align=right ><b>\^<<keylist[M_QUIT][1]>></b>=Odejít <b>\^<<
        keylist[M_PREV][1]>></b>=Předchozí nabídka<br>
        <<prevLink != nil ? aHrefAlt('předchozí', prevLink, '') : ''>>
        <tab align=right ><b>\^<<keylist[M_UP][1]>></b>=Nahoru <b>\^<<
        keylist[M_DOWN][1]>></b>=Dolů <b>\^<<
        keylist[M_SEL][1]>></b>=Vybrat<br>";
    }

    /* show a 'next chapter' link */
    menuNextChapter(keylist, title, hrefNext, hrefUp)
    {
        "Další: <a href='<<hrefNext>>'><<title>></a>;
        <b>\^<<keylist[M_PREV][1]>></b>=<a href='<<hrefUp>>'>Nabídka</a>";
    }

    /*
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObject(obj)
    {
        "Na {kohoco obj} {nedosáhn[eš]|[jsi] nedosáhl[a]}. ";
    }

    /*
     *   cannot reach an object, because the object is inside the given
     *   container 
     */
    cannotReachContents(obj, loc)
    {
        gMessageParams(obj, loc);
        return 'Skrz {kohoco loc} {|[jsi]} na {kohoco obj} '
            + 'nedosáh{n[eš]|l[a]}. ';
    }

    /* cannot reach an object because it's outisde the given container */
    cannotReachOutside(obj, loc)
    {
        gMessageParams(obj, loc);
        return 'Skrz {kohoco loc} {|[jsi]} na {kohoco obj} '
            + 'nedosáh{n[eš]|l[a]}. ';
    }

    /* sound is coming from inside/outside a container */
    soundIsFromWithin(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} z
            {kohočeho loc}. ";
    }
    soundIsFromWithout(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} do {kohočeho loc}
            z venku. ";
    }

    /* odor is coming from inside/outside a container */
    smellIsFromWithin(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} z
            {kohočeho loc}. ";
    }
    smellIsFromWithout(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} do {kohočeho loc}
            z venku. ";
    }

    /* default description of the player character */
    pcDesc(actor)
    {
        "Vypad{[áš]|al[a]} stejně, jako obvykle. ";
    }

    /*
     *   Show a status line addendum for the actor posture, without
     *   mentioning the actor's location.  We won't mention standing, since
     *   this is the default posture.  
     */
    roomActorStatus(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        /* mention any posture other than standing */
        if (actor.posture != standing)
            " (<<actor.posture.participle>> {|[jsi]})";
    }

    /* show a status line addendum: standing in/on something */
    actorInRoomStatus(actor, room)
    {
        local poser = actor;
        gMessageParams(poser);

        " (<<actor.posture.participle>> {|[jsi]} <<room.actorNaKomCem>>)";
    }

    /* generic short description of a dark room */
    roomDarkName = 'Ve tmě'

    /* generic long description of a dark room */
    roomDarkDesc = "{Je|Byla} tu naprostá tma. "

    /*
     *   mention that an actor is here, without mentioning the enclosing
     *   room, as part of a room description 
     *
     *   "Ty tu stojí." (Součást vylistování místnosti, když ho provádí
     *   NPC a vidí přitom PC -- ">Sally, rozhlédni se".)
     *
     *   TODO: Proc ignoruje {|[jsi]}? Pise "Ty tu stala".
     */
    roomActorHereDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        "{Ty poser} {|[jsi]} tu <<actor.posture.participle>>. ";
    }

    /*
     *   mention that an actor is visible at a distance or remotely,
     *   without mentioning the enclosing room, as part of a room
     *   description 
     */
    roomActorThereDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>> poblíž. ";
    }

    /*
     *   Mention that an actor is in a given local room, as part of a room
     *   description.  This is used as a default "special description" for
     *   an actor.  
     */
    actorInRoom(actor, cont)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>>
        <<cont.actorInName>>. ";
    }

    /*
     *   Describe an actor as standing/sitting/lying on something, as part
     *   of the actor's EXAMINE description.  This is additional
     *   information added to the actor's description, so we refer to the
     *   actor with a pronoun ("He's standing here").  
     *
     *   TODO: "Sally sedíš na zemi." -> sedí
     */
    actorInRoomPosture(actor, room)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>>
        <<room.actorInName>>. ";
    }

    /*
     *   Describe an actor's posture, as part of an actor's "examine"
     *   description.  If the actor is standing, don't bother mentioning
     *   anything, as standing is the trivial default condition.  
     */
    roomActorPostureDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        if (actor.posture != standing)
            "\^<<actor.itIs>> <<actor.posture.participle>>. ";
    }

    /*
     *   mention that the given actor is visible, at a distance or
     *   remotely, in the given location; this is used in room
     *   descriptions when an NPC is visible in a remote or distant
     *   location 
     *
     *   Listuje actora, který je v jiné lokaci propojené sense konektorem.
     *   Když napíšu v posture.participle parametr vztahující se k actor, tak
     *   je tím ale implicitně míněn PC vykonávající příkaz a ne proměnná actor
     *   předaná do fukce definující viděnou NPC. Musím si tedy v definici
     *   posture.participle specifikovat parametr jiného názvu a tady si ho
     *   naplnit actorem.
     */
    actorInRemoteRoom(actor, room, pov)
    {
        local poser = actor;
        gMessageParams(poser);

        /* say that the actor is in the room, using its remote in-name */
        "Vid{[íš]|ěl[a] [jsi]} odsud <<room.destName>>. \^<<actor.name>> tam
            <<actor.posture.participle>>. ";
    }

/* ODSUD NAHORU JE HOTOVO */

    /*
     *   mention that the given actor is visible, at a distance or
     *   remotely, in the given nested room within the given outer
     *   location; this is used in room descriptions 
     */
    actorInRemoteNestedRoom(actor, inner, outer, pov)
    {
        local poser = actor;
        gMessageParams(poser);

        /*
         *   say that the actor is in the nested room, in the current
         *   posture, and add then add that we're in the outer room as
         *   well 
         */
        "\^<<actor.name>> x {je} <<outer.inRoomName(pov)>>,
        <<actor.posture.participle>> <<inner.actorInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors are in the local room in a nominal
     *   container that we want to mention: "Bob and Bill are sitting on
     *   the couch."
     *
     *   TODO: Jak s parametry posture.partivipl vs. mnoho postav?
     */
    actorInGroupPrefix(posture, cont, lst) { "\^"; }
    actorInGroupSuffix(posture, cont, lst)
    {
        " <<lst.length() == 1 ? tSel('is', 'was') : tSel('are', 'were')>>
        <<posture.participle>> <<cont.actorInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors are inside a nested room that's inside
     *   a remote location: "Bob and Bill are in the courtyard, sitting on
     *   the bench." 
     */
    actorInRemoteGroupPrefix(pov, posture, cont, remote, lst) { "\^"; }
    actorInRemoteGroupSuffix(pov, posture, cont, remote, lst)
    {
        " <<lst.length() == 1 ? tSel('is', 'was') : tSel('are', 'were')>>
        <<remote.inRoomName(pov)>>, <<posture.participle>>
        <<cont.actorInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors' nominal container cannot be seen or is
     *   not to be stated: "Bob and Bill are standing here."
     *   
     *   Note that we don't always want to state the nominal container,
     *   even when it's visible.  For example, when actors are standing on
     *   the floor, we don't bother saying that they're on the floor, as
     *   that's stating the obvious.  The container will decide whether or
     *   not it wants to be included in the message; containers that don't
     *   want to be mentioned will use this form of the message.  
     */
    actorHereGroupPrefix(posture, lst) { "\^"; }
    actorHereGroupSuffix(posture, lst)
    {
        " <<lst.length() == 1 ? tSel('is', 'was') : tSel('are', 'were')>>
        <<posture.participle>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors' immediate container cannot be seen or
     *   is not to be stated, and the actors are in a remote location:
     *   "Bob and Bill are in the courtyard."  
     */
    actorThereGroupPrefix(pov, posture, remote, lst) { "\^"; }
    actorThereGroupSuffix(pov, posture, remote, lst)
    {
        " <<lst.length() == 1 ? tSel('is', 'was') : tSel('are', 'were')>>
        <<posture.participle>> <<remote.inRoomName(pov)>>. ";
    }

    /* a traveler is arriving, but not from a compass direction */
    sayArriving(traveler)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerLocName>>. ";
    }

    /* a traveler is departing, but not in a compass direction */
    sayDeparting(traveler)
    {
        local nm = traveler.travelerRemoteLocName;
        "\^<<traveler.travelerName(nil)>> {odchází|ode[šel]} <<nm != '' ? nm
            : 'pryč'>>. ";
    }

    /*
     *   a traveler is arriving locally (staying within view throughout the
     *   travel, and coming closer to the PC) 
     */
    sayArrivingLocally(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerLocName>>. ";
    }

    /*
     *   a traveler is departing locally (staying within view throughout
     *   the travel, and moving further away from the PC) 
     */
    sayDepartingLocally(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> <<traveler.verbToLeave>>
        <<traveler.travelerLocName>>. ";
    }

    /*
     *   a traveler is traveling remotely (staying within view through the
     *   travel, and moving from one remote top-level location to another) 
     */
    sayTravelingRemotely(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> <<traveler.verbToGo>> to
        <<traveler.travelerLocName>>. ";
    }

    /* a traveler is arriving from a compass direction */
    sayArrivingDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerRemoteLocName>> <<dirName>>. ";
    }

    /* a traveler is leaving in a given compass direction */
    sayDepartingDir(traveler, dirName)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToLeave>>
        to the <<dirName>><<nm != '' ? ' from ' + nm : ''>>. ";
    }
    
    /* a traveler is arriving from a shipboard direction */
    sayArrivingShipDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingSEd>>
        <<traveler.travelerRemoteLocName>> from <<dirName>>. ";
    }

    /* a traveler is leaving in a given shipboard direction */
    sayDepartingShipDir(traveler, dirName)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToLeave>>
        to <<dirName>><<nm != '' ? ' from ' + nm : ''>>. ";
    }

    /* a traveler is going aft */
    sayDepartingAft(traveler)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToGo>>
        aft<<nm != '' ? ' from ' + nm : ''>>. ";
    }

    /* a traveler is going fore */
    sayDepartingFore(traveler)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(nil)>> <<traveler.verbToGo>>
        forward<<nm != '' ? ' from ' + nm : ''>>. ";
    }

    /* a shipboard direction was attempted while not onboard a ship */
    notOnboardShip = "That direction {is|was}n&rsquo;t meaningful {|t}here. "

    /* a traveler is leaving via a passage */
    sayDepartingThroughPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToLeave>>
        <<traveler.travelerRemoteLocName>> through <<passage.theNameObj>>. ";
    }

    /* a traveler is arriving via a passage */
    sayArrivingThroughPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingSEd>>
        <<traveler.travelerRemoteLocName>> through <<passage.theNameObj>>. ";
    }

    /* a traveler is leaving via a path */
    sayDepartingViaPath(traveler, passage)
    {
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToLeave>>
        <<traveler.travelerRemoteLocName>> via <<passage.theNameObj>>. ";
    }

    /* a traveler is arriving via a path */
    sayArrivingViaPath(traveler, passage)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingSEd>>
        <<traveler.travelerRemoteLocName>> via <<passage.theNameObj>>. ";
    }

    /* a traveler is leaving up a stairway */
    sayDepartingUpStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToGo>>
        up <<stairs.theNameObj>>. ";
    }

    /* a traveler is leaving down a stairway */
    sayDepartingDownStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToGo>>
        down <<stairs.theNameObj>>. ";
    }

    /* a traveler is arriving by coming up a stairway */
    sayArrivingUpStairs(traveler, stairs)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(true)>> <<traveler.verbToCome>>
        up <<stairs.theNameObj>><<nm != '' ? ' to ' + nm : ''>>. ";
    }

    /* a traveler is arriving by coming down a stairway */
    sayArrivingDownStairs(traveler, stairs)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(true)>> <<traveler.verbToCome>>
        down <<stairs.theNameObj>><<nm != '' ? ' to ' + nm : ''>>. ";
    }

    /* acompanying another actor on travel */
    sayDepartingWith(traveler, lead)
    {
        "\^<<traveler.travelerName(nil)>> <<traveler.verbToCome>>
        with <<lead.theNameObj>>. ";
    }

/* ODSUD DOLU JE HOTOVO */

    /*
     *   Accompanying a tour guide.  Note the seemingly reversed roles:
     *   the lead actor is the one initiating the travel, and the tour
     *   guide is the accompanying actor.  So, the lead actor is
     *   effectively following the accompanying actor.  It seems
     *   backwards, but really it's not: the tour guide merely shows the
     *   lead actor where to go, but it's up to the lead actor to actually
     *   initiate the travel.  
     */
    sayDepartingWithGuide(guide, lead)
    {
        gMessageParams(guide, lead);
        "{Kdoco lead} nech{á|al[a lead]} {kohoco guide} vést. ";
    }

    /* note that a door is being opened/closed remotely */
    sayOpenDoorRemotely(door, stat)
    {
        "Někdo <<stat ? 'otev' : 'zav'>>{írá|řel} <<door.nameKohoCo>> z druhé
            strany. ";
    }

    /*
     *   open/closed status - these are simply adjectives that can be used
     *   to describe the status of an openable object 
     */
    openMsg(obj) { return 'otevřen{ý obj}'; }
    closedMsg(obj) { return 'zavřen{ý obj}'; }

    /* object is currently open/closed */
    currentlyOpen = 'Momentálně {je dobj} otevřen{ý}. '
    currentlyClosed = 'Momentálně {je dobj} zavřen{ý}. '

    /* stand-alone independent clause describing current open status */
    openStatusMsg(obj)
    {
        gMessageParams(obj);
        return '{Je obj} ' + obj.openDesc;
    }

    /* locked/unlocked status - adjectives describing lock states */
    lockedMsg(obj) { return 'zamčen{ý obj}'; }
    unlockedMsg(obj) { return 'odemčen{ý obj}'; }

    /* object is currently locked/unlocked */
    currentlyLocked = 'Momentálně {je dobj} zamčen{ý}. '
    currentlyUnlocked = 'Momentálně {je dobj} odemčen{ý}. '

    /*
     *   on/off status - these are simply adjectives that can be used to
     *   describe the status of a switchable object 
     */
    onMsg(obj) { return 'zapnut{ý obj}'; }
    offMsg(obj) { return 'vypnut{ý obj}'; }

    /* daemon report for burning out a match */
    matchBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} dohořel{a obj} a plamen {mizí|zmizel} v obláčku dýmu. ";
    }

    /* daemon report for burning out a candle */
    candleBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} už vyhořel{a obj} až na konec a zrovna
            zhas{íná|l[a obj]}. ";
    }

    /* daemon report for burning out a generic fueled light source */
    objBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} dohořel{a obj} a zrovna {zhasíná|zhasl[a obj]}. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Player Character messages.  These messages are generated when the
 *   player issues a regular command to the player character (i.e.,
 *   without specifying a target actor).  
 */
playerMessages: libMessages
    /* invalid command syntax */
    commandNotUnderstood(actor)
    {
        "<.parser>Příběh tomuto příkazu nerozum{í|ěl}.<./parser> ";
    }

    /* a special topic can't be used right now, because it's inactive */
    specialTopicInactive(actor)
    {
        "<.parser>{Zrovna teď|V tu chvíli [jsi]} tento příkaz {nemůž[eš]|nemohl[a]} použít.<./parser> ";
    }

    /* no match for a noun phrase */
    noMatch(actor, action, txt) { action.noMatch(self, actor, txt); }

    /*
     *   No match message - we can't see a match for the noun phrase.  This
     *   is the default for most verbs. 
     */
    noMatchCannotSee(actor, txt)
        { "Nic takového jako <<txt>> {|[jsi]} tu nevid{[íš]|ěl[a]}. "; }

    /*
     *   No match message - we're not aware of a match for the noun phrase.
     *   Some sensory actions, such as LISTEN TO and SMELL, use this
     *   variation instead of the normal version; the things these commands
     *   refer to tend to be intangible, so "you can't see that" tends to
     *   be nonsensical. 
     */
    noMatchNotAware(actor, txt)
        { "Ničeho takového jako <<txt>> {|[jsi]} si {[nejsi/není]|nebyl[a]} vědom{a}. "; }

    /* 'all' is not allowed with the attempted action */
    allNotAllowed(actor)
    {
        "<.parser>Slovo <q>vše</q> nelze použít s tímto slovesem.<./parser> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>Nevid{[íš]|ěl[a] [jsi]} tu nic odpovídajícího.<./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>Nic jiného tu {nevid[íš]|[jsi] neviděl[a]}.<./parser> ";
    }

    /* nothing left in a plural phrase after removing 'except' items */
    noMatchForListBut(actor) { noMatchForAllBut(actor); }

    /* no match for a pronoun */
    noMatchForPronoun(actor, typ, pronounWord)
    {
        /* show the message */
        "<.parser>Slovo <q><<pronounWord>></q> {teď zrovna|v tom okamžiku} na nic
        neodkaz{uje|ovalo}.<./parser> ";
    }

    /*
     *   Ask for a missing object - this is called when a command is
     *   completely missing a noun phrase for one of its objects.  
     *
     *   action.getQuestionInf(which) da cele "dát it"
     *   Komu cxhceš qdát it? 
     *   Komu ji chces dat? / Komu ji chce actor dat?
     *
     *   Ale pak je tu věta "Ven z čeho se chceš dostat?", kde musí být
     *   "se". To je zvratné osobní zájmeno a používá se, pokud je podmět
     *   i předmět totožný, tj. *ty* chceš *sebe* dostat.
     *
     *   Parametr which má hodnotu DirectObject nebo IndirectObject.
     */
    askMissingObject(actor, action, which)
    {
        local pronoun = action.getOtherMessageObjectPronoun(which);
        local zajmeno = action.getZvratneZajmeno(which);

        reportQuestion('<.parser>\^' + action.whatObj(which)
            + ' {|[jsi]} '
            + (zajmeno ? ' ' + zajmeno : '')    /* koho *se* chces zeptat */
            + (pronoun ? ' ' + pronoun : '')    /* komu *ji* chces dat */
            + ' {[chceš]|chtěl[a]} '
            /*+ (actor.referralPerson == ThirdPerson ? actor.name : '') tohle vygeneruje she a ne jmeno, proc? */
            + action.getQuestionInf(which) + '?<./parser> ');
    }

/* ODSUD NAHORU JE HOTOVO */

    /*
     *   An object was missing - this is called under essentially the same
     *   circumstances as askMissingObject, but in cases where interactive
     *   resolution is impossible and we simply wish to report the problem
     *   and do not wish to ask for help.
     */
    missingObject(actor, action, which)
    {
        "<.parser>You need to be more specific
        about <<action.whatObj(which)>> you want
        <<actor.referralPerson == ThirdPerson ? actor.name : ''>>
        to <<action.getQuestionInf(which)>>.<./parser> ";
    }

    /*
     *   Ask for a missing literal phrase. 
     */
    askMissingLiteral(actor, action, which)
    {
        /* use the standard missing-object message */
        askMissingObject(actor, action, which);
    }

    /*
     *   Show the message for a missing literal phrase.
     */
    missingLiteral(actor, action, which)
    {
        "<.parser>Please be more specific
        about <<action.whatObj(which)>> to
        <<action.getQuestionInf(which)>>.  Try, for example,
        <<action.getQuestionInf(which)>> <q>something</q>.<./parser> ";
    }

    /* reflexive pronoun not allowed */
    reflexiveNotAllowed(actor, typ, pronounWord)
    {
        "<.parser>The story doesn&rsquo;t understand how to use the word
        <q><<pronounWord>></q> like that.<./parser> ";
    }

    /*
     *   a reflexive pronoun disagrees in gender, number, or something
     *   else with its referent 
     */
    wrongReflexive(actor, typ, pronounWord)
    {
        "<.parser>The story doesn&rsquo;t understand what the
        word <q><<pronounWord>></q> refers to.<./parser> ";
    }

    /* no match for a possessive phrase */
    noMatchForPossessive(actor, owner, txt)
    {
        "<.parser>\^<<owner.nameDoes>>
        not appear to have any such thing.<./parser> ";
    }

    /* no match for a plural possessive phrase */
    noMatchForPluralPossessive(actor, txt)
    {
        "<.parser>\^They <<tSel('do', 'did')>> not appear to have any such
        thing.<./parser> ";
    }

    /* no match for a containment phrase */
    noMatchForLocation(actor, loc, txt)
    {
        "<.parser>\^<<actor.nameSees>>
        no <<loc.childInName(txt)>>.<./parser> ";
    }

    /* nothing in a container whose contents are specifically requested */
    nothingInLocation(actor, loc)
    {
        "<.parser>\^<<actor.nameSees>>
        <<loc.childInName('nothing unusual')>>.<./parser> ";
    }

/* ODSUD DOLU JE HOTOVO */

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /*
         *   show the message, leaving the <.parser> tag mode open - we
         *   always show another disambiguation prompt after this message,
         *   so we'll let the prompt close the <.parser> mode 
         */
        "<.parser>To nebyla jedna z nabízených možností. ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "<.parser>Vypadá to, že jsi vynechal pár slov.<./parser> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "<.parser>S nulovým množstvím {nemůž[eš]|[jsi] nemohl[a]} nic dělat.
            <./parser> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>Tolik <<txt>> {|[jsi]} tu nevid{[íš]|ěl[a]}.<./parser> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "<.parser>Tady nemůžeš použít více objektů.<./parser> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "<.parser>S tímto příkazem není dovoleno použít více objektů.
            <./parser> ";
    }

    /*
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the <.parser> tag open, for the re-prompt that will follow */
        "<.parser>Tolik možností tu nebylo. ";
    }

    /*
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed; and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.
     *   
     *   If askingAgain is true, it means that we're asking the question
     *   again because we got an invalid response to the previous attempt
     *   at the same prompt.  We will have explained the problem, and now
     *   we're going to give the user another crack at the same response.
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* mark this as a question report with a dummy report */
        reportQuestion('');
        
        /*
         *   Open the "<.parser>" tag, if we're not "asking again."  If we
         *   are asking again, we will already have shown a message
         *   explaining why we're asking again, and that message will have
         *   left us in <.parser> tag mode, so we don't need to open the
         *   tag again. 
         */
        if (!askingAgain)
            "<.parser>";
        
        /*
         *   the question varies depending on whether we want just one
         *   object or several objects in the final result 
         */
        if (requiredNum == 1)
        {
            /*
             *   One object needed - use the original text in the query.
             *   
             *   Note that if we're "asking again," we will have shown an
             *   additional message first explaining *why* we're asking
             *   again, and that message will have left us in <.parser>
             *   tag mode; so we need to close the <.parser> tag in this
             *   case, but we don't need to show a new one. 
             */
            local lst = ['kterého', 'který', 'kterou', 'které'];
            if (askingAgain)
            {
                "Tak <<lst[matchList[1].obj_.gender]>> {m[áš]|[jsi] měl[a]} na
                    mysli? \^<<askDisambigList(matchList, fullMatchList, nil,
                    dist)>>?";
            }
            else
            {
                "A <<lst[matchList[1].obj_.gender]>> <<originalText>>
                    {m[áš]|[jsi] měl[a]} na mysli? \^<<askDisambigList(
                    matchList, fullMatchList, nil, dist)>>?";
            }
        }
        else
        {
            /*
             *   Multiple objects required - ask by number, since we can't
             *   easily guess what the plural might be given the original
             *   text.
             *   
             *   As above, we only need to *close* the <.parser> tag if
             *   we're asking again, because we will already have shown a
             *   prompt that opened the tag in this case.  
             */
            if (askingAgain)
                "Tak které <<spellInt(requiredNum, 2)>> (z <<askDisambigList(
                    matchList, fullMatchList, true, dist)>>) {m[áš]|[jsi]
                    měl[a]} na mysli?";
            else
                "A které <<spellInt(requiredNum, 2)>> (z <<askDisambigList(
                    matchList, fullMatchList, true, dist)>>) {m[áš]|[jsi]
                    měl[a]} na mysli?";
        }

        /* close the <.parser> tag */
        "<./parser> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     *
     *   TODO: Přizpůsobit se rodu slovem který
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>Příběh nev{í|ěděl}, který <<originalText>>
            myslíš.<./parser> ";
    }

    /* the actor is missing in a command */
    missingActor(actor)
    {
        "<.parser>Mus{[íš]|el[a] [jsi]} být specifičtější o tom, koho
            {chc[eš]|[jsi] chtěl[a]} oslovit.<./parser> ";
    }

    /* only a single actor can be addressed at a time */
    singleActorRequired(actor)
    {
        "<.parser>Najednou {můž[eš]|[jsi] mohl[a]} oslovit jen jednu osobou.
            <./parser> ";
    }

    /* cannot change actor mid-command */
    cannotChangeActor()
    {
        "<.parser>V tomto příběhu {nemůž[eš]|[jsi] nemohl[a]} oslovit více než
            jednu postavu na jedné příkazové řádce.<./parser> ";
    }

    /*
     *   tell the user they entered a word we don't know, offering the
     *   chance to correct it with "oops" 
     */
    askUnknownWord(actor, txt)
    {
        /* start the message */
        "<.parser>Slovo <q><<txt>></q> v tomto příběhu {není|nebylo} důležité.<./parser> ";

        /* mention the OOPS command, if appropriate */
        oopsNote();
    }

    /*
     *   tell the user they entered a word we don't know, but don't offer
     *   an interactive way to fix it (i.e., we can't use OOPS at this
     *   point) 
     */
    wordIsUnknown(actor, txt)
    {
        "<.parser>Tomuto příkazu příběh nerozum{í|ěl}.<./parser> ";
    }

    /* the actor refuses the command because it's busy with something else */
    refuseCommandBusy(targetActor, issuingActor)
    {
        gMessageParams(targetActor);
        "{Kdoco targetActor} {je} zaneprázdněn{ý}. ";
    }

    /* cannot speak to multiple actors */
    cannotAddressMultiple(actor)
    {
        "<.parser>{Kdoco} {nemůže|nemohl[a]} oslovit najednou více postav.
            <./parser> ";
    }

    /* 
     *   Remaining actions on the command line were aborted due to the
     *   failure of the current action.  This is just a hook for the game's
     *   use, if it wants to provide an explanation; by default, we do
     *   nothing.  Note that games that override this will probably want to
     *   use a flag property so that they only show this message once -
     *   it's really only desirable to explain the the mechanism, not to
     *   flag it every time it's used.  
     */
    explainCancelCommandLine()
    {
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Non-Player Character (NPC) messages - parser-mediated format.  These
 *   versions of the NPC messages report errors through the
 *   parser/narrator.
 *   
 *   Note that a separate set of messages can be selected to report
 *   messages in the voice of the NPC - see npcMessagesDirect below.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessages: playerMessages
    /* the target cannot hear a command we gave */
    commandNotHeard(actor)
    {
        "{Kdoco} neodpov{ídá|ěděl[a]}. ";
    }

/* ODSUD NAHORU JE HOTOVO */

    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
        { "\^<<actor.nameSees>> no <<txt>>. "; }
    noMatchNotAware(actor, txt)
        { "\^<<actor.nameIs>> not aware of any <<txt>>. "; }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>\^<<actor.nameDoes>>n&rsquo;t see anything
        suitable.<./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>\^<<actor.nameSees>> nothing
        else.<./parser> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>\^<<actor.nameDoes>>n&rsquo;t see that many
         <<txt>>.<./parser> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>\^<<actor.nameDoes>>n&rsquo;t know which
        <<originalText>> you mean<<tSel('', 't')>>.<./parser> ";
    }

    /*
     *   Missing object query and error message templates 
     *
     *   Místo getQuestionInf použijeme getQuestionPast, protože chceme ve větě
     *   "Koho chceš, aby Sally *následovala*?" mít minulý čas.
     *
     *   TODO: Jak to, že i když je tu použit actor.name, tak se neshoduje se
     *   šablonou {kdoco}? Měl jsem pocit, že actor je výchozím, vždy přítomným
     *   parametrem. Přitom zde actorem je jiná postava, než PC. Tohle je asi ten
     *   duvod, proč někde jsou ve zprávách špičaté závorky, i když by zdánlivě
     *   šla použít šablona. Ohlídat - poznám to podle parametru funkce actor,
     *   ktery je explicitně předán?
     */
    askMissingObject(actor, action, which)
    {
        local pc = actor;
        gMessageParams(pc);
        reportQuestion('<.parser>\^' + action.whatObj(which)
                       + ' {chceš pc}, aby ' + actor.name + ' '
                       + action.getQuestionPast(which)
                       +'?<./parser> ');
    }
    missingObject(actor, action, which)
    {
        "<.parser>You must be more specific
        about <<action.whatObj(which)>> you want <<actor.theNameObj>> to
        <<action.getQuestionInf(which)>>.<./parser> ";
    }

    /* missing literal phrase query and error message templates */
    missingLiteral(actor, action, which)
    {
        "<.parser>You must be more specific about <<action.whatObj(which)>>
        you want <<actor.theNameObj>> to <<action.getQuestionInf(which)>>.
        For example: <<actor.name>>, <<action.getQuestionInf(which)>>
        <q>something</q>.<./parser> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   This default implementation simply doesn't report deferred errors at
 *   all.  The default message voice is the parser/narrator character, and
 *   there is simply no good way for the parser/narrator to say that a
 *   command failed in the past for a given character: "Bob looks like he
 *   didn't know which box you meant" just doesn't work.  So, we'll simply
 *   not report these errors at all.
 *   
 *   To report messages in the NPC's voice directly, modify the NPC's
 *   Actor object, or the Actor base class, to return
 *   npcDeferredMessagesDirect rather than this object from
 *   getParserMessageObj().  
 */
npcDeferredMessages: object
;

/* ------------------------------------------------------------------------ */
/*
 *   NPC messages, reported directly in the voice of the NPC.  These
 *   messages are not selected by default, but a game can use them instead
 *   of the parser-mediated versions by modifying the actor object's
 *   getParserMessageObj() to return these objects.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessagesDirect: npcMessages
    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
    {
        "\^<<actor.name>> look around. <q>I don&rsquo;t
        see any <<txt>>.</q> ";
    }
    noMatchNotAware(actor, txt)
    {
        "<q>I&rsquo;m not aware of any <<txt>>,</q> {říká|řekl[a]} {kdoco}. ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>I don&rsquo;t see anything suitable.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>I see nothing else here.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I can&rsquo;t do that to zero of something.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I don&rsquo;t see that many <<txt>> here.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I can&rsquo;t use multiple objects like that.</q> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I can&rsquo;t use multiple objects like that.</q> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /* leave the quote open for the re-prompt */
        "{Kdoco} {říká|řekl[a]}:
        <q>That was not one of the choices. ";
    }

    /*
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the quote open for the re-prompt */
        "{Kdoco} {říká|řekl[a]}:
        <q>There weren&rsquo;t that many choices. ";
    }

    /*
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed, and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.  
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* mark this as a question report */
        reportQuestion('');

        /* the question depends on the number needed */
        if (requiredNum == 1)
        {
            /* one required - ask with the original text */
            if (!askingAgain)
                "\^<<actor.name>> ask, <q>";
            
            "Which <<originalText>> do you mean, <<
            askDisambigList(matchList, fullMatchList, nil, dist)>>?</q> ";
        }
        else
        {
            /*
             *   more than one required - we can't guess at the plural
             *   given the original text, so just use the number 
             */
            if (!askingAgain)
                "\^<<actor.name>> ask, <q>";
            
            "Which <<spellInt(requiredNum, 2)>> (of <<
            askDisambigList(matchList, fullMatchList, true, dist)>>)
            do you mean?</q> ";
        }
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I don&rsquo;t know which <<originalText>> you mean.</q> ";
    }

    /*
     *   Missing object query and error message templates 
     */
    askMissingObject(actor, action, which)
    {
        reportQuestion('{Kdoco} {říká|řekl[a]}: <q>\^' + action.whatObj(which)
            + ' do you want me to ' + action.getQuestionInf(which) + '?</q> ');
    }
    missingObject(actor, action, which)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I don&rsquo;t know <<action.whatObj(which)>>
        you want me to <<action.getQuestionInf(which)>>.</q> ";
    }
    missingLiteral(actor, action, which)
    {
        /* use the same message we use for a missing ordinary object */
        missingObject(actor, action, which);
    }

    /* tell the user they entered a word we don't know */
    askUnknownWord(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I don&rsquo;t know the word <q><<txt>></q>.</q> ";
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>You used a word I don&rsquo;t know.</q> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   Some messages will never be deferred:
 *   
 *   commandNotHeard - if a command is not heard, it will never enter an
 *   actor's command queue; the error is given immediately in response to
 *   the command entry.
 *   
 *   refuseCommandBusy - same as commandNotHeard
 *   
 *   noMatchDisambig - interactive disambiguation will not happen in a
 *   deferred response situation, so it is impossible to have an
 *   interactive disambiguation failure.  
 *   
 *   disambigOrdinalOutOfRange - for the same reason noMatchDisambig can't
 *   be deferred.
 *   
 *   askDisambig - if we couldn't display a message, we definitely
 *   couldn't perform interactive disambiguation.
 *   
 *   askMissingObject - for the same reason that askDisambig can't be
 *   deferred
 *   
 *   askUnknownWord - for the same reason that askDisambig can't be
 *   deferred.  
 */
npcDeferredMessagesDirect: npcDeferredMessages
    commandNotUnderstood(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I didn&rsquo;t understand what you meant.</q> ";
    }

    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>I didn&rsquo;t see any <<txt>>.</q> ";
    }
    noMatchNotAware(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>I wasn&rsquo;t aware of any <<txt>>.</q> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>I didn&rsquo;t see anything suitable.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I didn&rsquo;t see what you meant.</q> ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>You left some words out.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I didn&rsquo;t understand what you meant.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I didn&rsquo;t see enough <<txt>>.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>I didn&rsquo;t understand what you meant.</q> ";
    }

/* ODSUD DOLU JE HOTOVO */

    /* a unique object is required, but multiple objects were specified */
    /* TODO: Poslední šablona by měla být podle cílového actora. */
    singleObjectRequired(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nerozumím tomu, co jsi myslel{a}.</q> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     *
     *   TODO: Řídit se rodem podle matchListu, ale bez testu to nedam.
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "{Kdoco} {říká|řekl[a]}:
            <q>Nejsem si jist{ý actor}, který <<originalText>> máš na
            mysli.</q> ";
    }

    /* an object phrase was missing */
    askMissingObject(actor, action, which)
    {
        reportQuestion('{Kdoco} {říká|řekl[a]}: <q>Nevím '
            + action.whatObj(which) + ' jsem měl{a} '
            + action.getQuestionInf(which) + '.</q> ');
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Použil jsi slovo, které neznám.</q> ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Verb messages for standard library verb implementations for actions
 *   performed by the player character.  These return strings suitable for
 *   use in VerifyResult objects as well as for action reports
 *   (defaultReport, mainReport, and so on).
 *   
 *   Most of these messages are generic enough to be used for player and
 *   non-player character alike.  However, some of the messages either are
 *   too terse (such as the default reports) or are phrased awkwardly for
 *   NPC use, so the NPC verb messages override those.  
 */
playerActionMessages: MessageHelper
    /*
     *   generic "can't do that" message - this is used when verification
     *   fails because an object doesn't define the action ("doXxx")
     *   method for the verb 
     */
    cannotDoThatMsg = 'To {nemůž[eš]|[jsi] nemohl[a]} udělat. '

    /* must be holding something before a command */
    mustBeHoldingMsg(obj)
    {
        gMessageParams(obj);
        return 'Abys to mohl{a} udělat, musel{a} {[bys]|[jsi]} {kohoco obj}
            držet. ';
    }

    /* it's too dark to do that */
    tooDarkMsg = 'Na to {je tu|tu byla} moc velká tma. '

    /* object must be visible */
    mustBeVisibleMsg(obj)
    {
        gMessageParams(obj);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten obj} vidět. ';
    }

    /* object can be heard but not seen */
    heardButNotSeenMsg(obj)
    {
        gMessageParams(obj);
        return 'Sice {|[jsi]} {kohoco obj} slyš{[íš]|el[a]}, ale nevid{[íš]|ěl[a]
            [jsi]} {ten obj}. ';
    }

    /* object can be smelled but not seen */
    smelledButNotSeenMsg(obj)
    {
        gMessageParams(obj);
        return 'Sice {|[jsi]} {kohoco obj} cít{[íš]|il[a]}, ale nevid{[íš]|ěl[a]
            [jsi]} {ten obj}. ';
    }

    /* cannot hear object */
    cannotHearMsg(obj)
    {
        gMessageParams(obj);
        return 'Neslyš{[íš]|el[a] [jsi]} {ten obj}. ';
    }

    /* cannot smell object */
    cannotSmellMsg(obj)
    {
        gMessageParams(obj);
        return 'Necít{[íš]|il[a] [jsi]} {ten obj}. ';
    }

    /* cannot taste object */
    cannotTasteMsg(obj)
    {
        gMessageParams(obj);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten obj} ochutnat. ';
    }

/* ODSUD NAHORU JE HOTOVO */

    /* must remove an article of clothing before a command */
    cannotBeWearingMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} sundat {kohoco obj}
                before {it actor/he} {can} do that. ';
    }

    /* all contents must be removed from object before doing that */
    mustBeEmptyMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} take everything out of {the obj/him}
                before {it actor/he} {can} do that. ';
    }

    /* object must be opened before doing that */
    mustBeOpenMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} open {the obj/him}
                before {it actor/he} {can} do that. ';
    }

    /* object must be closed before doing that */
    mustBeClosedMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} close {the obj/him}
               before {it actor/he} {can} do that. ';
    }

    /* object must be unlocked before doing that */
    mustBeUnlockedMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} unlock {the obj/him}
                before {it actor/he} {can} do that. ';
    }

    /* no key is needed to lock or unlock this object */
    noKeyNeededMsg = '{Kdoco dobj} {does} not appear to take a key. '

    /* actor must be standing before doing that */
    mustBeStandingMsg = '{You/he} {mus[íš]|musel[a] [jsi]} stand up before {it actor/he}
                      {can} do that. '

    /* must be sitting on/in chair */
    mustSitOnMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} sit {in obj} first. ';
    }

    /* must be lying on/in object */
    mustLieOnMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} lie {in obj} first. ';
    }

    /* must get on/in object */
    mustGetOnMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} get {in obj} first. ';
    }

    /* object must be in loc before doing that */
    mustBeInMsg(obj, loc)
    {
        gMessageParams(obj, loc);
        return '{Kdoco obj} {mus[íš]|musel[a] [jsi]} be {in loc} before {you/he}
            {can} do that. ';
    }

    /* actor must be holding the object before we can do that */
    mustBeCarryingMsg(obj, actor)
    {
        gMessageParams(obj, actor);
        return '{The actor/he} {mus[íš]|musel[a] [jsi]} be holding {the obj/him}
            before {it actor/he} {can} do that. ';
    }

/* ODSUD DOLU JE HOTOVO */

    /* generic "that's not important" message for decorations */
    decorationNotImportantMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {není} důležit{ý}. ';
    }

    /* generic "you don't see that" message for "unthings" */
    unthingNotHereMsg(obj)
    {
        gMessageParams(obj);
        return 'Nevid{[íš]|el[a] [jsi]} {ten obj} tady. ';
    }

    /* generic "that's too far away" message for Distant items */
    tooDistantMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} příliš daleko. ';
    }

    /* generic "no can do" message for intangibles */
    notWithIntangibleMsg(obj)
    {
        gMessageParams(obj);
        return 'To {|[jsi]} s {kýmčím} nem{ůž[eš]|ohl[a]} udělat. ';
    }

    /* generic failure message for varporous objects */
    notWithVaporousMsg(obj)
    {
        gMessageParams(obj);
        return 'To {|[jsi]} s {kýmčím} nem{ůž[eš]|ohl[a]} udělat. ';
    }

    /* look in/look under/look through/look behind/search vaporous */
    lookInVaporousMsg(obj)
    {
        gMessageParams(obj);
        return 'Vid{[íš]|ěl[a] [jsi]} prostě jen {kohoco obj}. ';
    }

    /*
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObjectMsg(obj)
    {
        gMessageParams(obj);
        return 'Na {kohoco obj} {nedosáhn[eš]|[jsi] nedosáhl[a]}. ';
    }

    /* cannot reach an object through an obstructor */
    cannotReachThroughMsg(obj, loc)
    {
        gMessageParams(obj, loc);
        return ' Skrz {kohoco loc} {nemůž[eš]|[jsi] nemohl[a]} šahat na {kohoco obj}.';
    }

    /* generic long description of a Thing */
    thingDescMsg(obj)
    {
        gMessageParams(obj);
        return 'Na {tom obj} {nevid[íš]|[jsi] neviděl[a]} nic mimořádného. ';
    }

    /* generic LISTEN TO description of a Thing */
    thingSoundDescMsg(obj)
        { return 'Neslyš{[íš]|el[a] [jsi]} nic mimořádného. '; }

    /* generic "smell" description of a Thing */
    thingSmellDescMsg(obj)
        { return 'Necít{[íš]|il[a] [jsi]} nic mimořádného. '; }

    /* default description of a non-player character */
    npcDescMsg(npc)
    {
        gMessageParams(npc);
        return 'Nevid{[íš]|ěl[a] [jsi]} na {tom npc} nic neobvyklého. ';
    }

    /* generic messages for looking prepositionally */
    nothingInsideMsg =
        'V {komčem dobj} {není|nebylo} nic neobvyklého. '
    nothingUnderMsg =
        'Pod {kýmčím dobj} {není|nebylo} nic neobvyklého. '
    nothingBehindMsg =
        'Za {kýmčím dobj} {není|nebylo} nic neobvyklého. '
    nothingThroughMsg =
        'Skrz {kohoco dobj} {|[jsi]} nic nevid{[íš]|ěl[a]}. '

    /* this is an object we can't look behind/through */
    cannotLookBehindMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat za {kohoco dobj}. '
    cannotLookUnderMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat pod {kohoco dobj}. '
    cannotLookThroughMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat skrz
        {kohoco dobj}. '

/* ODSUD NAHORU JE HOTOVO */

    /* looking through an open passage */
    nothingThroughPassageMsg = '{You/he} {can\'t} see much through
        {kohoco dobj} from {|t}here. '

    /* there's nothing on the other side of a door we just opened */
    nothingBeyondDoorMsg = 'Opening {kohoco dobj} reveal{s|ed} nothing
        unusual. '

    /* there's nothing here with a specific odor */
    nothingToSmellMsg =
        '{You/he} smell{s/ed} nothing out of the ordinary. '

    /* there's nothing here with a specific noise */
    nothingToHearMsg = '{You/he} hear{s/d} nothing out of the ordinary. '

    /* a sound appears to be coming from a source */
    noiseSourceMsg(src)
    {
        return '{Kdoco dobj} seem{s/ed} to be coming from '
            + src.theNameObj + '. ';
    }

    /* an odor appears to be coming from a source */
    odorSourceMsg(src)
    {
        return '{Kdoco dobj} seem{s/ed} to be coming from '
            + src.theNameObj + '. ';
    }

/* ODSUD DOLU JE HOTOVO */

    /* an item is not wearable */
    notWearableMsg =
        'To {není} něco, co {[bys]|[jsi]} mohl{a} na sebe vzít. '

    /* doffing something that isn't wearable */
    notDoffableMsg =
        'To {není} něco, co {[bys]|[jsi]} mohl{a} sundat. '

    /* already wearing item */
    alreadyWearingMsg = 'Už {|[jsi]} {ten dobj} {m[áš]|měl[a]} na sobě. '

    /* not wearing (item being doffed) */
    notWearingMsg = '{Kohoco dobj} nem{áš} na sobě. '

    /* default response to 'wear obj' */
    okayWearMsg = 'Hotovo, {m[áš]|měl[a] [jsi]} na sobě {kohoco dobj}. '

    /* default response to 'doff obj' */
    okayDoffMsg = 'Hotovo, už {|[jsi]} na sobě ne{m[áš]|měl[a]} {kohoco dobj}. '

    /* default response to open/close */
    okayOpenMsg = shortTMsg(
        'Hotovo. ', 'Otev{ír[áš]|řel[a] [jsi]} {kohoco dobj}. ')
    okayCloseMsg = shortTMsg(
        'Hotovo. ', 'Zav{ír[áš]|řel[a] [jsi]} {kohoco dobj}. ')

    /* default response to lock/unlock */
    okayLockMsg = shortTMsg(
        'Hotovo. ', 'Zam{yk[áš]|kl[a] [jsi]} {kohoco dobj}. ')
    okayUnlockMsg = shortTMsg(
        'Hotovo. ', 'Odem{yk[áš]|kl[a] [jsi]} {kohoco dobj}. ')

    /* cannot dig here */
    cannotDigMsg = 'Ke kopání do {toho dobj} {nem[áš]|[jsi] neměl[a]} žádný
        důvod. '

    /* not a digging implement */
    cannotDigWithMsg =
        'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {ten iobj} použít jako
            lopatu. '

    /* taking something already being held */
    alreadyHoldingMsg = '{Kohoco dobj} {|[jsi]} už drž{[íš]|el[a]}. '

    /* actor taking self ("take me") */
    takingSelfMsg = 'Nemůž{eš} vzít sebe. '

    /* dropping an object not being carried */
    notCarryingMsg = '{Kohoco dobj} nem{áš} u sebe. '

    /* actor dropping self */
    droppingSelfMsg = 'Nemůž{eš} položit sebe. '

    /* actor putting self in something */
    puttingSelfMsg = 'Tohle se sebou udělat nem{ůž[eš]|ohl[a]}. '

    /* actor throwing self */
    throwingSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} hodit sebe. '

    /* we can't put the dobj in the iobj because it's already there */
    alreadyPutInMsg = '{Kdoco dobj} už {je} v {komčem iobj}. '

    /* we can't put the dobj on the iobj because it's already there */
    alreadyPutOnMsg = '{Kdoco dobj} už {je} na {komčem iobj}. '

    /* we can't put the dobj under the iobj because it's already there */
    alreadyPutUnderMsg = '{Kdoco dobj} už {je} pod {kýmčím iobj}. '

    /* we can't put the dobj behind the iobj because it's already there */
    alreadyPutBehindMsg = '{Kdoco dobj} už {je} za {kýmčím iobj}. '

    /*
     *   trying to move a Fixture to a new container by some means (take,
     *   drop, put in, put on, etc) 
     */
    cannotMoveFixtureMsg = 'S {kýmčím dobj} se ned{á|al[a dobj]} hýbat. '

    /* trying to take a Fixture */
    cannotTakeFixtureMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} vzít. '

    /* trying to put a Fixture in something */
    cannotPutFixtureMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} nikam dát. '

    /* trying to take/move/put an Immovable object */
    cannotTakeImmovableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} vzít. '
    cannotMoveImmovableMsg = 'S {kýmčím dobj} se ned{á|al[a dobj]} hýbat. '
    cannotPutImmovableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} nikam dát. '

    /* trying to take/move/put a Heavy object */
    cannotTakeHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '
    cannotMoveHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '
    cannotPutHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '

    /* trying to move a component object */
    cannotMoveComponentMsg(loc)
    {
        return '{Kdoco dobj} {je} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* trying to take a component object */
    cannotTakeComponentMsg(loc)
    {
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} mít, protože
            {je dobj} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* trying to put a component in something */
    cannotPutComponentMsg(loc)
    {
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} nikam dát, protože {je}
            součástí ' + loc.nameKohoCeho + '. ';
    }

    /* specialized Immovable messages for TravelPushables */
    cannotTakePushableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} vzít, ale
        mohl{a} {[bys]|[jsi]} {ten dobj} zkusit někam odtlačit. '
    cannotMovePushableMsg = 'Bezcílným posouváním {kohočeho dobj} {[bys]|[jsi]}
        ničeho nedosáhl{a}, ale mohl{a} {[bys]|[jsi]} {ten dobj} zkusit
        odtlačit určitým směrem. '
    cannotPutPushableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} nikam dát,
        ale mohl{a} {[bys]|[jsi]} {ten dobj} zkusit někam odtlačit. '

    /* can't take something while occupying it */
    cannotTakeLocationMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} vzít, dokud
        {|[jsi]} {ten dobj} zabír{[áš]|al[a]}. '

    /* can't REMOVE something that's being held */
    cannotRemoveHeldMsg = '{Není|Nebylo} z čeho {kohoco dobj} vyndat. '

    /* default 'take' response */
    /*
     *   Mělo by dávat smysl i na objekty sebrané ze země/lokace, i vyndané
     *   z kontejneru, který držím. Mám pocit, že nedokonavé fungovaly trochu
     *   víc přirozeně "Beru.", než přesnější překlad "Uchopeno.". "Vzato." je
     *   divné a "Sebráno." nefunguje na kontejner. Kromě toho řada hlášek
     *   obsahuje jen "Hotovo.", které zase nemá nedokonavý tvar a mělo by to
     *   všechno působit jednotně.
     */
    okayTakeMsg = shortTMsg(
        'Hotovo. ', 'Ber{eš} {kohoco dobj}. ')

    /* default 'drop' response */
    okayDropMsg = shortTMsg(
        'Hotovo. ', 'Od{klád[áš]|ložil[a] [jsi]} {kohoco dobj}. ')

    /* dropping an object */
    droppingObjMsg(dropobj)
    {
        gMessageParams(dropobj);
        return 'Od{klád[áš]|ložil[a] [jsi]} {kohoco dropobj}. ';
    }

    /* default receiveDrop suffix for floorless rooms */
    floorlessDropMsg(dropobj)
    {
        gMessageParams(dropobj);
        return 'Pad{á|al[a dropobj]} kamsi dolů z dohledu. ';
    }

    /* default successful 'put in' response */
    okayPutInMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} do {kohočeho iobj}. ')

    /* default successful 'put on' response */
    okayPutOnMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} na {kohoco iobj}. ')

    /* default successful 'put under' response */
    okayPutUnderMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} pod {kohoco iobj}. ')

    /* default successful 'put behind' response */
    okayPutBehindMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} za {kohoco iobj}. ')

    /* try to take/move/put/taste an untakeable actor */
    cannotTakeActorMsg = 'To {ti} {kdoco dobj} nedovolí. '
    cannotMoveActorMsg = 'To {ti} {kdoco dobj} nedovolí. '
    cannotPutActorMsg = 'To {ti} {kdoco dobj} nedovolí. '
    cannotTasteActorMsg = 'To {ti} {kdoco dobj} nedovolí. '

    /* trying to take/move/put/taste a person */
    cannotTakePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotMovePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotPutPersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotTastePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '

    /* cannot move obj through obstructor */
    cannotMoveThroughMsg(obj, obs)
    {
        gMessageParams(obj, obs);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten obj} sunout skrz '
            + '{kohoco obs}. ';
    }

    /* cannot move obj in our out of container cont */
    cannotMoveThroughContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ten obj} sunout skrz '
            + '{kohoco cont}. ';
    }

    /* cannot move obj because cont is closed */
    cannotMoveThroughClosedMsg(obj, cont)
    {
        gMessageParams(cont);
        return 'To nem{ůž[eš]|ohl[a] [jsi]} udělat, protože {kdoco cont} '
            + '{je} zavřen{ý}. ';
    }

    /* cannot fit obj into cont through cont's opening */
    cannotFitIntoOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To nem{ůž[eš]|ohl[a] [jsi]} udělat, protože {kdoco obj} '
            + '{je} příliš velk{ý} a do {kohoco cont} se '
            + 'neve{jd[eš obj]|šl[a obj]}. ';
    }

    /* cannot fit obj out of cont through cont's opening */
    cannotFitOutOfOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} {je}
            příliš velký na vyndání z {kohočeho cont}. ';
    }

    /* actor 'obj' cannot reach in our out of container 'cont' */
    cannotTouchThroughContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} nem{ůže|ohl[a obj]} skrz {kohoco cont} s ničím
            manipulovat. ';
    }

    /* actor 'obj' cannot reach through cont because cont is closed */
    cannotTouchThroughClosedMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco obj} nem{ůže|ohl[a obj]} udělat, protože {kdoco cont}
            {je} zavřen{ý}. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachIntoOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Komučemu obj} se do {kohočeho cont} neve{jde|šla} ruka. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachOutOfOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} neprotáhn{e|ul[a obj]} ruku ven z
            {kohočeho cont}. ';
    }

    /* the object is too large for the actor to hold */
    tooLargeForActorMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} na {tebe} moc velký. ';
    }

    /* the actor doesn't have room to hold the object */
    handsTooFullForMsg(obj)
    {
        return '{Tvoje} ruce {jsou|byly} přeplněné, nezvládneš držet '
            + obj.nameKohoCo + '. ';
    }

    /* the object is becoming too large for the actor to hold */
    becomingTooLargeForActorMsg(obj)
    {
        gMessageParams(obj);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} by
            byl[a obj] moc velk{ý} a neuzdvihl{a} bys {ten obj}. ';
    }

    /* the object is becoming large enough that the actor's hands are full */
    handsBecomingTooFullForMsg(obj)
    {
        gMessageParams(obj);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {tvoje} ruce by
            pak byly moc plné a neudržel{a} bys {ten obj}. ';
    }

    /* the object is too heavy (all by itself) for the actor to hold */
    tooHeavyForActorMsg(obj)
    {
        gMessageParams(obj);
        return '{Kohoco obj} {nezdvihn[eš]|[jsi] neuzdvihl[a]}, protože {je obj}
            moc těžk{ý}. ';
    }

    /*
     *   the object is too heavy (in combination with everything else
     *   being carried) for the actor to pick up 
     */
    totalTooHeavyForMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} moc těžk{ý}, {bud[eš] muset|musel[a] [jsi]}
            nejprve něco odložit. ';
    }

    /* object is too large for container */
    tooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, do {kohoco cont} se
            neve{jde obj}. ';
    }

    /* object is too large to fit under object */
    tooLargeForUndersideMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, pod {kohoco cont} se
            neve{jde obj}. ';
    }

    /* object is too large to fit behind object */
    tooLargeForRearMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, za {kohoco cont} se
            neve{jde obj}. ';
    }

    /* container doesn't have room for object */
    containerTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco cont} {je} moc pln{ý}, {kdoco obj} se už do {toho cont}
            neve{jde obj}. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Na {komčem cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* underside doesn't have room for object */
    undersideTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Pod {kýmčím cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* rear surface/space doesn't have room for object */
    rearTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Za {kýmčím cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* the current action would make obj too large for its container */
    becomingTooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} by
            {pak|} byl{a obj} moc velk{ý} na {kohoco cont}. ';
    }

    /*
     *   the current action would increase obj's bulk so that container is
     *   too full 
     */
    containerBecomingTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj}
            by se už neve{šel obj} do {kohočeho cont}. ';
    }

    /* trying to put an object in a non-container */
    notAContainerMsg = 'Do {kohočeho iobj} se ned{á|alo} nic dát. '

    /* trying to put an object on a non-surface */
    notASurfaceMsg = 'Na {komčem iobj} {není|nebyla} žádná vhodná plocha. '

    /* can't put anything under iobj */
    cannotPutUnderMsg =
        'Pod {ten iobj} {nemůž[eš]|[jsi] nemohl[a]} nic dát. '

    /* nothing can be put behind the given object */
    cannotPutBehindMsg = 'Za {kohoco iobj} {nemůž[eš]|[jsi] nemohl[a]} nic
        dát. '

    /* trying to put something in itself */
    cannotPutInSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} do sebe sama. '

    /* trying to put something on itself */
    cannotPutOnSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} na sebe sama. '

    /* trying to put something under itself */
    cannotPutUnderSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} pod seebe
        sama. '

    /* trying to put something behind itself */
    cannotPutBehindSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} za
        sebe sama. '

    /* can't put something in/on/etc a restricted container/surface/etc */
    cannotPutInRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} dát do {kohočeho iobj}. '
    cannotPutOnRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} dát na {kohoco iobj}. '
    cannotPutUnderRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} dát pod {kohoco iobj}. '
    cannotPutBehindRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} dát za {kohoco iobj}. '

    /* trying to return something to a remove-only dispenser */
    cannotReturnToDispenserMsg = '{Kohoco dobj} {nemůžeš|[jsi] nemohl[a]} vrátit
        zpátky do {kohočeho iobj}. '

    /* wrong item type for dispenser */
    cannotPutInDispenserMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} do {kohočeho iobj}. '

    /* the dobj doesn't fit on this keyring */
    objNotForKeyringMsg = '{Kdoco dobj} {nepadne|nepdal[a dobj]} na
        {kohoco iobj}. '

    /* the dobj isn't on the keyring */
    keyNotOnKeyringMsg = '{Kdoco dobj} {není} navlečen{ý} na {komčem iobj}. '

    /* can't detach key (with no iobj specified) because it's not on a ring */
    keyNotDetachableMsg = '{Kdoco dobj} {není} na ničem navlečen{ý}. '

    /* we took a key and attached it to a keyring */
    takenAndMovedToKeyringMsg(keyring)
    {
        gMessageParams(keyring);
        return '{Ber[eš]|Vzal[a] [jsi]} {kohoco dobj} a navlék{[áš]|l[a] [jsi]}
            {ten dobj} na {kohoco keyring}. ';
    }

    /* we attached a key to a keyring automatically */
    movedKeyToKeyringMsg(keyring)
    {
        gMessageParams(keyring);
        return 'Navlék{[áš]|l[a] [jsi]} {kohoco dobj} na {kohoco keyring}. ';
    }

    /* we moved several keys to a keyring automatically */
    /* TODO: Prozkoumat a nahradit your/his */
    movedKeysToKeyringMsg(keyring, keys)
    {
        gMessageParams(keyring);
        return 'Navlék{[áš]|l[a] [jsi]} {your/his} voln' + (keys.length() > 1
            ? 'é' : 'ý') + ' klíč' + (keys.length() > 1 ? 'e' : '') + ' na
            {kohoco keyring}. ';
    }

    /* putting y in x when x is already in y */
    circularlyInMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            v {komčem y}. ';
    }

    /* putting y in x when x is already on y */
    circularlyOnMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            na {komčem y}. ';
    }

    /* putting y in x when x is already under y */
    circularlyUnderMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            pod {kýmčím y}. ';
    }

    /* putting y in x when x is already behind y */
    circularlyBehindMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            za {kýmčím y}. ';
    }

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotInMsg = '{Kdoco dobj} v {tom iobj} {není dobj}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOnMsg = '{Kdoco dobj} na {tom iobj} {není}. '

    /* taking dobj from under something, but dobj isn't under iobj */
    takeFromNotUnderMsg = '{Kdoco dobj} pod {tím iobj} {není}. '

    /* taking dobj from behind something, but dobj isn't behind iobj */
    takeFromNotBehindMsg = '{Kdoco dobj} za {tím iobj} {není}. '

    /* taking dobj from an actor, but actor doesn't have iobj */
    takeFromNotInActorMsg = '{Kdoco iobj} {ten dobj}
        ne{m[áš iobj]|měl[a iobj]}. '

    /* actor won't let go of a possession */
    willNotLetGoMsg(holder, obj)
    {
        gMessageParams(holder, obj);
        return 'Nedovol{[íš holder]|il[a holder] [jsi holder]} {komučemu} {ten obj} mít. ';
    }

    /* must say which way to go */
    whereToGoMsg = 'Budeš muset říci, kterým směrem se vydat. '

    /* travel attempted in a direction with no exit */
    cannotGoThatWayMsg = 'Tímto směrem {se|[ses]} pohybovat {nemůž[eš]|nemohl[a]}. '

    /* travel attempted in the dark in a direction with no exit */
    cannotGoThatWayInDarkMsg = '{Je|Byla} tu příliš velká tma, než a{bys}
        viděl{a} na cestu. '

    /* we don't know the way back for a GO BACK */
    cannotGoBackMsg = 'Nev{[íš]|ěděl[a] [jsi]}, jak se odsud dostat zpátky. '

    /* cannot carry out a command from this location */
    cannotDoFromHereMsg = 'Odtud {|[jsi]} to nem{ůž[eš]|ohl[a]} udělat. '

    /* can't travel through a close door */
    cannotGoThroughClosedDoorMsg(door)
    {
        gMessageParams(door);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco door}
               {je} zavřen{ý}. ';
    }

    /* cannot carry out travel while 'dest' is within 'cont' */
    invalidStagingContainerMsg(cont, dest)
    {
        gMessageParams(cont, dest);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {kdoco dest} {je}
            v {komčem cont}. ';
    }

    /* cannot carry out travel while 'cont' (an actor) is holding 'dest' */
    invalidStagingContainerActorMsg(cont, dest)
    {
        gMessageParams(cont, dest);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {kdoco cont}
            drž{í|el[a cont]} {kohoco dest}. ';
    }

/* ODSUD NAHORU JE HOTOVO */

    /* can't carry out travel because 'dest' isn't a valid staging location */
    invalidStagingLocationMsg(dest)
    {
        gMessageParams(dest);
        return '{You/he} nem{ůže|ohl[a]} get {in dest}. ';
    }

    /* destination is too high to enter from here */
    nestedRoomTooHighMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} too high to reach from {|t}here. ';
    }

    /* enclosing room is too high to reach by GETTING OUT OF here */
    nestedRoomTooHighToExitMsg(obj)
    {
        return 'It{&rsquo;s| was} too long a drop to do that from {|t}here. ';
    }

    /* cannot carry out a command from a nested room */
    cannotDoFromMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} nem{ůže|ohl[a]} do that from {the obj/him}. ';
    }

    /* cannot carry out a command from within a vehicle in a nested room */
    vehicleCannotDoFromMsg(obj)
    {
        local loc = obj.location;
        gMessageParams(obj, loc);
        return '{You/he} nem{ůže|ohl[a]} do that while {kdoco obj} {je} {in loc}. ';
    }

    /* cannot go that way in a vehicle */
    cannotGoThatWayInVehicleMsg(traveler)
    {
        gMessageParams(traveler);
        return '{You/he} nem{ůže|ohl[a]} do that {in traveler}. ';
    }

    /* cannot push an object that way */
    cannotPushObjectThatWayMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} nem{ůže|ohl[a]} go that way pushing {the obj/him}. ';
    }

    /* cannot push an object to a nested room */
    cannotPushObjectNestedMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} nem{ůže|ohl[a]} push {the obj/him} there. ';
    }

    /* cannot enter an exit-only passage */
    cannotEnterExitOnlyMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} nem{ůže|ohl[a]} enter {the obj/him} from {|t}here. ';
    }

    /* must open door before going that way */
    mustOpenDoorMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} open {the obj/him} first. ';
    }

    /* door closes behind actor during travel through door */
    doorClosesBehindMsg(obj)
    {
        gMessageParams(obj);
        return '<.p>After {you/he} {goes} through {the obj/him}, {it/he}
                close{s/d} behind {it actor/him}. ';
    }

/* ODSUD DOLU JE HOTOVO */

    /* the stairway does not go up/down */
    stairwayNotUpMsg = '{Kdoco dobj} odsud {vede|vedl[a dobj]} jen dolů. '
    stairwayNotDownMsg = '{Kdoco dobj} odsud {vede|vedl[a dobj]} jen nahoru. '

    /* "wait" */
    timePassesMsg = 'Čas plyn{e|ul}... '

    /* "hello" with no target actor */
    sayHelloMsg = (addressingNoOneMsg)

    /* "goodbye" with no target actor */
    sayGoodbyeMsg = (addressingNoOneMsg)

    /* "yes"/"no" with no target actor */
    sayYesMsg = (addressingNoOneMsg)
    sayNoMsg = (addressingNoOneMsg)

    /* an internal common handler for sayHelloMsg, sayGoodbyeMsg, etc */
    addressingNoOneMsg
    {
        return '{Mus[íš]|Musel[a]} {|jsi} být specifičtější o tom, koho {|jsi}
            {[chceš]|chtěl[a]} oslovit. ';
    }

    /* "yell" */
    okayYellMsg = 'Křič{[íš]|el[a] [jsi]} tak hlasitě, jak {|[jsi]} jen
        doved{[eš]|l[a]}. '

    /* "jump" */
    okayJumpMsg = 'Kousek {vyskaku[ješ]|[jsi] vyskočil[a]} a
        dopad{[áš]|l[a]} zpátky na stejném místě. '

    /* cannot jump over object */
    cannotJumpOverMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} přeskočit. '

    /* cannot jump off object */
    cannotJumpOffMsg = 'Z {toho dobj} {nemůž[eš]|[jsi] nemohl[a]} seskočit. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHereMsg = 'Odsud {není|nebylo} kam vyskočit. '

    /* failed to find a topic in a consultable object */
    cannotFindTopicMsg =
        'To se {ti} ne{daří|podařilo} najít v {komčem dobj}. '

    /* an actor doesn't accept a command from another actor */
    refuseCommand(targetActor, issuingActor)
    {
        gMessageParams(targetActor, issuingActor);
        return '{Kdoco targetActor} odmít{á|nul[a targetActor]} {tvoji}
            žádost. ';
    }

    /* cannot talk to an object (because it makes no sense to do so) */
    notAddressableMsg(obj)
    {
        gMessageParams(obj);
        return 'S {tím obj} {nemůž[eš]|[jsi] nemohl[a]} mluvit. ';
    }

    /* actor won't respond to a request or other communicative gesture */
    noResponseFromMsg(other)
    {
        gMessageParams(other);
        return '{Kdoco other} neodpovíd{á|al[a other]}. ';
    }

    /* trying to give something to someone who already has the object */
    giveAlreadyHasMsg = '{Kdoco iobj} už {ten dobj} {má|měl[a iobj]}. '

    /* can't talk to yourself */
    cannotTalkToSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't ask yourself about anything */
    cannotAskSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't ask yourself for anything */
    cannotAskSelfForMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't tell yourself about anything */
    cannotTellSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

/* ODSUD NAHORU JE HOTOVO */

    /* can't give yourself something */
    cannotGiveToSelfMsg = 'Giving {kohoco dobj} to {yourself/himself}
        {won&rsquo;t|wouldn&rsquo;t} accomplish anything. '

    /* can't give something to itself */
    cannotGiveToItselfMsg = 'Giving {kohoco dobj} to {itself}
        {won&rsquo;t|wouldn&rsquo;t} accomplish anything. '

    /* can't show yourself something */
    cannotShowToSelfMsg = 'Showing {kohoco dobj} to {yourself/himself}
        {won&rsquo;t|wouldn&rsquo;t} accomplish anything. '

    /* can't show something to itself */
    cannotShowToItselfMsg = 'Showing {kohoco dobj} to {itself}
        {won&rsquo;t|wouldn&rsquo;t} accomplish anything. '

    /* can't give/show something to a non-actor */
    cannotGiveToMsg = '{You/he} nem{ůže|ohl[a]} give anything to {an iobj/him}. '
    cannotShowToMsg = '{You/he} nem{ůže|ohl[a]} show anything to {an iobj/him}. '

    /* actor isn't interested in something being given/shown */
    notInterestedMsg(actor)
    {
        return '\^' + actor.nameDoes + ' not appear interested. ';
    }

    /* vague ASK/TELL (for ASK/TELL <actor> <topic> syntax errors) */
    askVagueMsg = '<.parser>The story doesn&rsquo;t understand that command.
        Please use ASK ACTOR ABOUT TOPIC (or just A TOPIC).<./parser> '
    tellVagueMsg = '<.parser>The story doesn&rsquo;t understand that command.
        Please use TELL ACTOR ABOUT TOPIC (or just T TOPIC).<./parser> '

    /* object cannot hear actor */
    objCannotHearActorMsg(obj)
    {
        return '\^' + obj.nameDoes
            + ' not appear to hear {you/him}. ';
    }

    /* actor cannot see object being shown to actor */
    actorCannotSeeMsg(actor, obj)
    {
        return '\^' + actor.nameDoes + ' not appear to be able to see '
            + obj.theNameObj + '. ';
    }

    /* not a followable object */
    notFollowableMsg = '{You/he} nem{ůže|ohl[a]} follow {that dobj/him}. '

/* ODSUD DOLU JE HOTOVO */

    /* cannot follow yourself */
    cannotFollowSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} následovat sebe. '

    /* following an object that's in the same location as the actor */
    followAlreadyHereMsg = '{Kdoco dobj} {je} přímo tady. '

/* ODSUD NAHORU JE HOTOVO */

    /*
     *   following an object that we *think* is in our same location (in
     *   other words, we're already in the location where we thought we
     *   last saw the object go), but it's too dark to see if that's
     *   really true 
     */
    followAlreadyHereInDarkMsg = '{Kdoco dobj} should {be|have been} right
        {|t}here, but {you/he} nem{ůže|ohl[a]} see {it dobj/him}. '

    /* trying to follow an object, but don't know where it went from here */
    followUnknownMsg = '{You\'re} not sure where {Kdoco dobj}
        {went|had gone} from {|t}here. '

    /*
     *   we're trying to follow an actor, but we last saw the actor in the
     *   given other location, so we have to go there to follow 
     */
    cannotFollowFromHereMsg(srcLoc)
    {
        return 'The last place {you/he} {saw|had seen} {kohoco dobj} was '
            + srcLoc.getDestName(gActor, gActor.location) + '. ';
    }

    /* acknowledge a 'follow' for a target that was in sight */
    okayFollowInSightMsg(loc)
    {
        return '{You/he} follow{s/ed} {kohoco dobj} '
            + loc.actorIntoName + '. ';
    }

    /* obj is not a weapon */
    notAWeaponMsg = '{You/he} nem{ůže|ohl[a]} attack anything with {kýmčím iobj}. '

    /* no effect attacking obj */
    uselessToAttackMsg = '{You/he} nem{ůže|ohl[a]} attack {that dobj/him}. '

    /* pushing object has no effect */
    pushNoEffectMsg = 'Pushing {kohoco dobj} {has|had} no effect. '

    /* default 'push button' acknowledgment */
    okayPushButtonMsg = '<q>Click.</q> '

    /* lever is already in pushed state */
    alreadyPushedMsg =
        '{It\'s dobj} already pushed as far as {it dobj/he} {will} go. '

    /* default acknowledgment to pushing a lever */
    okayPushLeverMsg = '{You/he} push{es/ed} {kohoco dobj} to
                     {its/her dobj} stop. '

    /* pulling object has no effect */
    pullNoEffectMsg = 'Pulling {kohoco dobj} {has|had} no effect. '

    /* lever is already in pulled state */
    alreadyPulledMsg =
        '{It\'s dobj} already pulled as far as {it dobj/he} {will} go. '

    /* default acknowledgment to pulling a lever */
    okayPullLeverMsg = '{You/he} pull{s/ed} {kohoco dobj} to
                     {its/her dobj} stop. '

    /* default acknowledgment to pulling a spring-loaded lever */
    okayPullSpringLeverMsg = '{You/he} pull{s/ed} {kohoco dobj}, which
        spr{ing[s]|ung} back to {its/her} starting position as soon as
        {you/he} let{[s]|} go of {it dobj/him}. '

/* ODSUD DOLU JE HOTOVO */

    /* moving object has no effect */
    moveNoEffectMsg = 'Posunutí {kohočeho dobj} by {|bývalo} nemělo žádný '
        + 'efekt. '

    /* cannot move object to other object */
    moveToNoEffectMsg = 'Tím {bys} {|bývalo} ničeho nedosáhl{a}.  '

    /* cannot push an object through travel */
    cannotPushTravelMsg = 'To by {|bývalo} nemělo žádný efekt. '

    /* acknowledge pushing an object through travel */
    okayPushTravelMsg(obj)
    {
        return '<.p>Tlač{[íš]|il[a] jsi} ' + obj.nameKohoCo
            + ' do volného prostoru. ';
    }

    /* cannot use object as an implement to move something */
    cannotMoveWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic posunout. '

    /* cannot set object to setting */
    cannotSetToMsg = 'Na {tom dobj} {nemůž[eš]|[jsi] nemohl[a]} nic nastavit. '

    /* invalid setting for generic Settable */
    setToInvalidMsg = '{Kdoco dobj} {nemá|neměl[a dobj]} takové nastavení. '

    /* default 'set to' acknowledgment */
    okaySetToMsg(val)
        { return 'Fajn, {kdoco dobj} {je} {teď|} nastaven{a dobj} na ' + val + '. '; }

    /* cannot turn object */
    cannotTurnMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {tím dobj} otočit. '

    /* must specify setting to turn object to */
    mustSpecifyTurnToMsg = '{Mus[íš]|musel[a] [jsi]} specifikovat polohu, do '
        + 'které {ten dobj} {chc[eš]|[jsi] chtěl[a]} natočit. '

    /* cannot turn anything with object */
    cannotTurnWithMsg =
        '{Tím dobj} {nemůž[eš]|[jsi] nemohl[a]} nic otočit. '

    /* invalid setting for dial */
    turnToInvalidMsg = '{Kdoco dobj} {nemá|neměl[a dobj]} takovou polohu. '

    /* default 'turn to' acknowledgment */
    okayTurnToMsg(val)
        { return 'Fajn, {kdoco dobj} {je} {teď|} natočen{a dobj} na ' + val + '. '; }

    /* switch is already on/off */
    alreadySwitchedOnMsg = '{Kdoco dobj} už {je} zapnut{ý}. '
    alreadySwitchedOffMsg = '{Kdoco dobj} už {je} vypnut{ý}. '

    /* default acknowledgment for switching on/off */
    okayTurnOnMsg = 'Fajn, {kdoco dobj} {je} {teď|od té chvíle} zapnut{ý}. '
    okayTurnOffMsg = 'Fajn, {kdoco dobj} {je} {teď|od té chvíle} vypnut{ý}. '

    /* flashlight is on but doesn't light up */
    flashlightOnButDarkMsg = 'Zapnul{a} {jsi} {kohoco dobj}, ale nezdá se, '
        + 'že by se cokoliv stalo. '

    /* default acknowledgment for eating something */
    okayEatMsg = 'Snědl{a} {jsi} {kohoco dobj}. '

    /* object must be burning before doing that */
    mustBeBurningMsg(obj)
    {
        return 'Nejprve {mus[íš]|[jsi] musel[a]} zapálit ' + obj.nameKohoCo
            + ', než {můž[eš]|[jsi] mohl[a]} tohle udělat. ';
    }

    /* match not lit */
    matchNotLitMsg = '{Kdoco dobj} {není} zapálen{ý}. '

    /* lighting a match */
    okayBurnMatchMsg =
        'Škrtnutím {kohočeho dobj} {zapaluj[eš]|[jsi] zapálil[a]} malý plamínek. '

    /* extinguishing a match */
    okayExtinguishMatchMsg = 'Zhas{ín[áš]|il[a] [jsi]} {kohoco dobj} a plamen '
        + 'se rozpl{ývá|ynul} v obláčku dýmu. '

    /* trying to light a candle with no fuel */
    candleOutOfFuelMsg =
        '{Kdoco dobj} {je} vyhořel{ý}, {nemůž[eš]|mohl[a] [jsi]} {ten dobj} zapálit. '

    /* lighting a candle */
    okayBurnCandleMsg = 'Zap{aluj[eš]|álil[a] [jsi]} {kohoco dobj}. '

    /* extinguishing a candle that isn't lit */
    candleNotLitMsg = '{Kdoco dobj} {není} zapálen{ý}. '

    /* extinguishing a candle */
    okayExtinguishCandleMsg = 'Hotovo. '

    /* cannot consult object */
    cannotConsultMsg =
        'To {není|nebylo} něco, s čím {[bys]|[jsi]} se mohl{a} poradit. '

    /* cannot type anything on object */
    cannotTypeOnMsg = 'Na {tom dobj} {nemůž[eš]|[jsi] nemohl[a]} nic napsat. '

    /* cannot enter anything on object */
    cannotEnterOnMsg = 'Na {tom dobj} {nemůž[eš]|[jsi] nemohl[a]} nic zadat. '

    /* cannot switch object */
    cannotSwitchMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} přepnout. '

    /* cannot flip object */
    cannotFlipMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} obrátit. '

    /* cannot turn object on/off */
    cannotTurnOnMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} zapnout. '
    cannotTurnOffMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} vypnout. '

    /* cannot light */
    cannotLightMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} zapálit. '

    /* cannot burn */
    cannotBurnMsg = 'To {není|nebylo} něco, co {|by} mohlo hořet. '
    cannotBurnWithMsg =
        '{Tím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic spálit. '

    /* cannot burn this specific direct object with this specific iobj */
    cannotBurnDobjWithMsg = '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} zapálit '
        + '{kohoco dobj}. '

    /* object is already burning */
    alreadyBurningMsg = '{Kdoco dobj} už {hoří|hořel[a dobj]}. '

    /* cannot extinguish */
    cannotExtinguishMsg = '{Tím dobj} {nemůž[eš]|[jsi] nemohl[a]} nic uhasit. '

    /* cannot pour/pour in/pour on */
    cannotPourMsg = 'To {není} něco, co {[bys]|[jsi]} mohl{a} lít. '
    cannotPourIntoMsg =
        'Do {toho iobj} {nemůž[eš]|[jsi] nemohl[a]} nic nalít. '
    cannotPourOntoMsg =
        'Na {ten iobj} {nemůž[eš]|[jsi] nemohl[a]} nic vylít. '

/* ODSUD NAHORU JE HOTOVO */

    /* cannot attach object to object */
    cannotAttachMsg =
        '{You/he} nem{ůže|ohl[a]} attach {that dobj/him} to anything. '
    cannotAttachToMsg =
        '{You/he} nem{ůže|ohl[a]} attach anything to {that iobj/him}. '

    /* cannot attach to self */
    cannotAttachToSelfMsg =
        '{You/he} nem{ůže|ohl[a]} attach {kohoco dobj} to {itself}. '

    /* cannot attach because we're already attached to the given object */
    alreadyAttachedMsg =
        '{Kdoco dobj} {je} already attached to {komučemu iobj}. '

    /*
     *   dobj and/or iobj can be attached to certain things, but not to
     *   each other 
     */
    wrongAttachmentMsg =
        '{You/he} nem{ůže|ohl[a]} attach {that dobj/him} to {komučemu iobj}. '

    /* dobj and iobj are attached, but they can't be taken apart */
    wrongDetachmentMsg =
        '{You/he} nem{ůže|ohl[a]} detach {that dobj/him} from {kohočeho iobj}. '

    /* must detach the object before proceeding */
    mustDetachMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} {mus[íš]|musel[a] [jsi]} detach {the obj/him} before {it actor/he}
            {can} do that. ';
    }

/* ODSUD DOLU JE HOTOVO */

    /* default message for successful Attachable attachment */
    okayAttachToMsg = 'Hotovo. '

    /* default message for successful Attachable detachment */
    okayDetachFromMsg = 'Hotovo. '

    /* cannot detach object from object */
    cannotDetachMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} oddělat. '
    cannotDetachFromMsg =
        'Z {toho iobj} {nemůž[eš]|[jsi] nemohl[a]} nic oddělat. '

    /* no obvious way to detach a permanent attachment */
    cannotDetachPermanentMsg = 'Ne{ní|bylo} {ti} jasné, jak {ten dobj}
        oddělat. '

    /* dobj isn't attached to iobj */
    notAttachedToMsg = '{Kdoco dobj} k {tomu iobj} {není dobj} připevněn. '

    /* breaking object would serve no purpose */
    shouldNotBreakMsg = '{Toho dobj} rozbití {by|} nebylo k ničemu. '

    /* cannot cut that */
    cutNoEffectMsg = '{Kdoco iobj} {kohoco dobj} nepřeříz{ne|l[a iobj]}. '

    /* can't use iobj to cut anything */
    cannotCutWithMsg = '{Kýmčím iobj} {|[jsi]} nic
        {nepřeřízn[eš]|nemohl[a] přeříznout}. '

    /* cannot climb object */
    cannotClimbMsg = 'To {není|nebylo} něco, po čem {by|} se dalo šplhat. '

    /* object is not openable/closable */
    cannotOpenMsg = 'To {není|nebylo} něco, co {by|} se dalo otevřít. '
    cannotCloseMsg = 'To {není|nebylo} něco, co {by|} se dalo zavřít. '

    /* already open/closed */
    alreadyOpenMsg = '{Kdoco dobj} už {je} otevřen{ý}. '
    alreadyClosedMsg = '{Kdoco dobj} už {je} zavřen{ý}. '

    /* already locked/unlocked */
    alreadyLockedMsg = '{Kdoco dobj} už {je} zamčen{ý}. '
    alreadyUnlockedMsg = '{Kdoco dobj} už {je} odemčen{ý}. '

    /* cannot look in container because it's closed */
    cannotLookInClosedMsg = '{Kdoco dobj} {je} zavřen{ý}. '

    /* object is not lockable/unlockable */
    cannotLockMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} zamknout. '
    cannotUnlockMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} odemknout. '

    /* attempting to open a locked object */
    cannotOpenLockedMsg = '{Kdoco dobj} {vypad[áš dobj]|vypadal[a dobj]}
        zamčeně. '

    /* object requires a key to unlock */
    unlockRequiresKeyMsg = 'Vypad{á|alo} to, že
        {potřebuj[eš]|[jsi] potřeboval[a]} k odemčení {kohočeho dobj} klíč. '

    /* object is not a key */
    cannotLockWithMsg =
        '{Kdoco iobj} nevypad{á|al[a iobj]} vhodně na zamykání. '
    cannotUnlockWithMsg =
        '{Kdoco iobj} nevypad{á|al[a iobj]} vhodně na odemykání. '

    /* we don't know how to lock/unlock this */
    unknownHowToLockMsg =
        '{Není|Nebylo} {ti} jasné, jak zamknout {kohoco dobj}. '
    unknownHowToUnlockMsg =
        '{Není|Nebylo} {ti} jasné, jak odemknout {kohoco dobj}. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLockMsg = '{Kdoco iobj} nepas{uje|oval[a iobj]} do zámku. '

    /* found key on keyring */
    foundKeyOnKeyringMsg(ring, key)
    {
        gMessageParams(ring, key);
        return 'Zkusil{a} {jsi} každý klíč na {komčem ring} a zjistil{a} {jsi},
            že {kdoco key} padn{e|ul[a key]} do zámku. ';
    }

    /* failed to find a key on keyring */
    foundNoKeyOnKeyringMsg(ring)
    {
        gMessageParams(ring);
        return 'Zk{ouš[íš]|usil[a] [jsi]} každý klíč na {komčem ring}, ale
            nem{ůž[eš]|ohl[a] [jsi]} najít žádný, který by pasoval do zámku. ';
    }

    /* not edible/drinkable */
    cannotEatMsg = '{Kdoco dobj} nevypad{á|al[a dobj]} poživatelně. '
    cannotDrinkMsg = '{Kdoco dobj} nevypad{á|al[a dobj]} pitně. '

    /* cannot clean object */
    cannotCleanMsg = 'Nev{[íš]|ěděl[a] [jsi]} jak {ten dobj} vyčistit. '
    cannotCleanWithMsg = '{Tím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic vyčistit. '

    /* cannot attach key (dobj) to (iobj) */
    cannotAttachKeyToMsg =
        'Na {ten iobj} {nemůžeš|[jsi] nemohl[a]} {kohoco dobj} navéknout. '

    /* actor cannot sleep */
    cannotSleepMsg = 'Nech{ce|tělo} se {ti} spát. '

    /* cannot sit/lie/stand/get on/get out of */
    cannotSitOnMsg =
        'To {není|nebylo} něco, na čem by se dalo sedět. '
    cannotLieOnMsg =
        'To {není|nebylo} něco, na čem by se dalo ležet. '
    cannotStandOnMsg = 'Na {tom dobj} {nemůž[eš]|[jsi] nemohl[a]} stát. '
    cannotBoardMsg = 'Na {ten dobj} {nemůž[eš]|[jsi] nemohl[a]} nastoupit. '
    cannotUnboardMsg = 'Z {toho dobj}{nemůž[eš]|[jsi] nemohl[a]} vystoupit. '
    cannotGetOffOfMsg = 'Z {toho dobj}{nemůž[eš]|[jsi] nemohl[a]} sestoupit. '

    /* standing on a PathPassage */
    cannotStandOnPathMsg = 'Pokud {chc[eš]|[jsi] chtěl[a]} následovat
        {kohoco dobj}, odpověz ne. '

    /* cannot sit/lie/stand on something being held */
    cannotEnterHeldMsg = 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {|[jsi]}
        {kohoco dobj} drž{[íš]|el[a]}. '

    /* cannot get out (of current location) */
    cannotGetOutMsg = '{[Nejsi/není]|Nebyl[a] [jsi]} v ničem, co {[bys]|[jsi]}
        mohl{a} opustit. '

    /* actor is already in a location */
    alreadyInLocMsg = 'Už {jsi/je} {v dobj}. '

    /* actor is already standing/sitting on/lying on */
    alreadyStandingMsg = '{Ty} už {stoj[íš]|[jsi] stál[a]}. '
    alreadyStandingOnMsg = 'Už {|[jsi]} st{oj[íš]|ál[a]} na {komčem dobj}. '
    alreadySittingMsg = '{Ty} už {sed[íš]|[jsi] seděl[a]}. '
    alreadySittingOnMsg = 'Už {|[jsi]} sed{[íš]|ěl[a]} na {komčem dobj}. '
    alreadyLyingMsg = '{Ty} už {lež[íš]|[jsi] ležel[a]}. '
    alreadyLyingOnMsg = 'Už {|[jsi]} lež{[íš]|el[a]} na {komčem dobj}. '

    /* getting off something you're not on */
    notOnPlatformMsg = '{Nejsi/není} {na dobj}. '

    /* no room to stand/sit/lie on dobj */
    noRoomToStandMsg =
        '{Na dobj} {není|nebylo} pro {tebe} dost místa ke stání. '
    noRoomToSitMsg =
        '{Na dobj} {není|nebylo} pro {tebe} dost místa k sezení. '
    noRoomToLieMsg =
        '{Na dobj} {není|nebylo} pro {tebe} dost místa k ležení. '

    /* default report for standing up/sitting down/lying down */
    /*
     *   Trochu špinavý trik - v posture.participle je šablona parametrizovaná
     *   poserem, ale tady nemam předaného actora, abych ho nabídl, tak objekt
     *   ze šablony zahodím a tím se bude vztahovat automaticky k actorovi.
     */
    okayPostureChangeMsg(posture)
    {
        return 'Fajn, {teď|} ' + posture.participle.findReplace(' poser', '',
            ReplaceAll) + ' {|[jsi]}. ';
    }

    /* default report for standing/sitting/lying in/on something */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        return 'Fajn, {teď|} ' + posture.participle.findReplace(' poser', '',
            ReplaceAll) + ' {|[jsi]} na {komčem obj}. ';
    }

    /* default report for getting off of a platform */
    /*
     *   TODO: Jak to, že to funguje? Používá to ['na', &actorInName, nil,
     *   nil, nil], ale nevím, co to vlastně má za význam - proc actorInName?
     */
    okayNotStandingOnMsg = 'Fajn, už {nejsi/není} {na dobj/komčem}. '

    /* cannot fasten/unfasten */
    cannotFastenMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} zapnout. '
    cannotFastenToMsg =
        'K {komučemu iobj} {nemůžeš|[jsi] nemohl[a]} nic připoutat. '
    cannotUnfastenMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} rozepnout. '
    cannotUnfastenFromMsg =
        'Od {kohočeho iobj} {nemůžeš|[jsi] nemohl[a]} nic odpoutat. '

    /* cannot plug/unplug */
    cannotPlugInMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        zapojit. '
    cannotPlugInToMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak cokoliv
        zapojit do {kohočeho iobj}. '
    cannotUnplugMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        vypojit. '
    cannotUnplugFromMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak cokoliv
        vypojit z {kohočeho iobj}. '

    /* cannot screw/unscrew */
    cannotScrewMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        zašroubovat. '
    cannotScrewWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic přišroubovat. '
    cannotUnscrewMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak
        {kohoco dobj} vyšroubovat. '
    cannotUnscrewWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic odšroubovat. '

    /* cannot enter/go through */
    cannotEnterMsg =
        'To {není|nebylo} něco, do čeho {[bys]|[jsi]} mohl{a} vstoupit. '
    cannotGoThroughMsg =
        'To {není|nebylo} něco, čím {[bys]|[jsi]} mohl{a} projít. '

    /* can't throw something at itself */
    cannotThrowAtSelfMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ten dobj} hodit na sebe sama. '

    /* can't throw something at an object inside itself */
    cannotThrowAtContentsMsg = 'Než {bud[eš] moci|[jsi] mohl[a]} tohle udělat,
        {mus[íš]|musel[a] [jsi]} odstranit {kohoco iobj} z {kohočeho dobj}. '

    /* shouldn't throw something at the floor */
    shouldNotThrowAtFloorMsg =
        'Raději {[bys]|[jsi]} {ten dobj} měl{a} prostě položit. '

    /* THROW <obj> <direction> isn't supported; use THROW AT instead */
    dontThrowDirMsg =
        '<.parser>Mus{[íš]|el[a] [jsi]} být specifičtější o tom, po čem
            {chc[eš]|[jsi] chtěl[a]} {kohoco dobj} vrhnout.<./parser> '

/*
 *   TODO: Bylo nize uvedene resici formulaci pro treti osobu. Nebude se to
 *   nekde hodit?
 *
 *   what ' + (gActor.referralPerson == ThirdPerson ? 'you want {the actor/him}'
 *   : '')
 */

    /* thrown object bounces off target (short report) */
    throwHitMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhnul[a projectile]}
            {kohoco target} bez znatelného efektu. ';
    }

    /* thrown object lands on target */
    throwFallMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhl[a projectile]}
            {kohoco target}. ';
    }

    /* thrown object bounces off target and falls to destination */
    throwHitFallMsg(projectile, target, dest)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhl[a projectile]} bez znatelného
            efektu {kohoco target} a {padá|spadl[a]} ' + dest.putInName + '. ';
    }

    /* thrown object falls short of distant target (sentence prefix only) */
    throwShortMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} nedoletěl{a projectile} až k
            {komučemu target}. ';
    }

    /* thrown object falls short of distant target */
    throwFallShortMsg(projectile, target, dest)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} nedoletěl{a projectile} až k
            {komučemu target} a {padá|spadla} ' + dest.putInName + '. ';
    }

    /* target catches object */
    throwCatchMsg(obj, target)
    {
        gMessageParams(obj, target);
        return '{Kdoco target} chyt{á|il[a target]} {kohoco obj}. ';
    }

    /* we're not a suitable target for THROW TO (because we're not an NPC) */
    cannotThrowToMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} nic hodit na {kohoco iobj}. '

    /* target does not want to catch anything */
    willNotCatchMsg(catcher)
    {
        gMessageParams(catcher);
        return '{Kdoco catcher} nevypad{á|al[a catcher]}, že {by|}
            chtěl{a catcher} něco chytat. ';
    }

    /* cannot kiss something */
    cannotKissMsg = 'Políbení {kohočeho dobj} {nemá|nemělo} znatelný efekt. '

    /* person uninterested in being kissed */
    cannotKissActorMsg
        = 'To by se {komučemu dobj} nejspíš nelíbilo. '

    /* cannot kiss yourself */
    cannotKissSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} políbit sebe. '

    /* it is now dark at actor's location */
    newlyDarkMsg = 'Najednou tu {je|byla} naprostá tma. '
;

/*
 *   Non-player character verb messages.  By default, we inherit all of
 *   the messages defined for the player character, but we override some
 *   that must be rephrased slightly to make sense for NPC's.
 */
npcActionMessages: playerActionMessages
    /* "wait" */
    timePassesMsg = '{Kdoco} ček{á|al[a]}... '

    /* trying to move a Fixture/Immovable */
    cannotMoveFixtureMsg = '{Kdoco} nem{ůže|ohl[a]} pohnout {kýmčím dobj}. '
    cannotMoveImmovableMsg = '{Kdoco} nem{ůže|ohl[a]} pohnout {kýmčím dobj}. '

    /* trying to take/move/put a Heavy object */
    cannotTakeHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj} na zdvihnutí. '
    cannotMoveHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj} na posunutí. '
    cannotPutHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj} na hýbání. '

    /* trying to move a component object */
    cannotMoveComponentMsg(loc)
    {
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco dobj}
            {je} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* default successful 'take' response */
    okayTakeMsg = '{Kdoco} {bere|vzal[a]} {kohoco dobj}. '

    /* default successful 'drop' response */
    okayDropMsg = '{Kdoco} {pokádá|položil[a]} {kohoco dobj}. '

    /* default successful 'put in' response */
    okayPutInMsg = '{Kdoco} {vkládá|vložil[a]} {kohoco dobj} do {kohočeho iobj}. '

    /* default successful 'put on' response */
    okayPutOnMsg = '{Kdoco} {pokádá|položil[a]} {kohoco dobj} na {kohoco iobj}. '

    /* default successful 'put under' response */
    okayPutUnderMsg =
        '{Kdoco} {dává|dal[a]} {kohoco dobj} pod {kohoco iobj}. '

    /* default successful 'put behind' response */
    okayPutBehindMsg =
        '{Kdoco} {dává|dal[a]} {kohoco dobj} za {kohoco iobj}. '

    /* default succesful response to 'wear obj' */
    okayWearMsg =
        '{Kdoco} {bere|vzal[a]} {kohoco dobj} na sebe. '

    /* default successful response to 'doff obj' */
    okayDoffMsg = '{Kdoco} {sundavá|sundal[a]} {kohoco dobj}. '

    /* default successful responses to open/close */
    okayOpenMsg = '{Kdoco} otev{írá|řel[a]} {kohoco dobj}. '
    okayCloseMsg = '{Kdoco} zav{írá|řel[a]} {kohoco dobj}. '

    /* default successful responses to lock/unlock */
    okayLockMsg = '{Kdoco} zam{yká|kl[a]} {kohoco dobj}. '
    okayUnlockMsg = '{Kdoco} odem{yká|kl[a]} {kohoco dobj}. '

    /* push/pull/move with no effect */
    pushNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} tlačit {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    pullNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} táhnout {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    moveNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} posunout {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    moveToNoEffectMsg = '{Kdoco} {nechává|zanechal[a]} {kohoco dobj}, '
        + 'kde byl{a dobj}. '

    whereToGoMsg =
        'Měl{a} {[bys]|[jsi]} říci, kterým směrem {má|měl[a]} {kdoco} jít. '

    /* object is too large for container */
    tooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se do
            {kohočeho cont} neve{jde|[šel obj]}. ';
    }

    /* object is too large for underside */
    tooLargeForUndersideMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se pod
            {kohoco cont} neve{jde|[šel obj]}. ';
    }

    /* object is too large to fit behind something */
    tooLargeForRearMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se za
            {kohoco cont} neve{jde|[šel obj]}. ';
    }

    /* container doesn't have room for object */
    containerTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco cont} {je}
            už moc plný a {kdoco obj} se do {toho cont} nevejde. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože na {komčem cont}
	    pro {kohoco obj} {není|nebylo} místo. ';
    }

    /* the dobj doesn't fit on this keyring */
    /* TODO: parametr ten tu urcite neni spravne! */
    objNotForKeyringMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {ten dobj} nepas{uje|oval[a dobj]} do {kohočeho iobj}. '

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotInMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} v {tom iobj} {není}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOnMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} na {tom iobj} {není}. '

    /* taking dobj under something, but dobj isn't under iobj */
    takeFromNotUnderMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} pod {tím iobj} {není}. '

    /* taking dobj from behind something, but dobj isn't behind iobj */
    takeFromNotBehindMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} {není} za {tím iobj}. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHereMsg = '{Kdoco} nem{á|ěl[a]} kam vyskočit. '

    /* should not break object */
    shouldNotBreakMsg = '{Kdoco} {ten dobj} nech{ce|těl[a]} rozbít. '

    /* report for standing up/sitting down/lying down */
    okayPostureChangeMsg(posture)
        { return '{Kdoco} ' + posture.msgVerbI + '. '; }

    /*
     *   report for standing/sitting/lying in/on something
     *
     *   >sally, stoupni si na židli
     *   Sally stoupá na dřevěnou židli.
     */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        return '{Kdoco} ' + posture.msgVerbT + ' {na obj/kohoco}. ';
    }

    /* report for getting off a platform */
    okayNotStandingOnMsg = '{Kdoco} sest{upuje|oupil[a]} {offof dobj}. '

    /* default 'turn to' acknowledgment */
    okayTurnToMsg(val)
        { return '{Kdoco} {otáčí|natočil[a]} {kohoco dobj} na ' + val + '. '; }

    /* default 'push button' acknowledgment */
    okayPushButtonMsg = '{Kdoco} {mačká|zmáčkl[a]} {kohoco dobj}. '

    /* default acknowledgment for switching on/off */
    okayTurnOnMsg = '{Kdoco} zap{íná|nul[a]} {kohoco dobj}. '
    okayTurnOffMsg = '{Kdoco} vyp{íná|nul[a]} {kohoco dobj}. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLockMsg = '{Kdoco} zk{ouší|usil[a]} {kohoco iobj}, ale '
        + 'do zámku nepadn{e|ul[a iobj]}. '

    /* acknowledge entering "follow" mode */
    /*
     *   TODO: Pokud má jít s jinou NPC, tak asi jméno, jinak pro PC je
     *   v pořádku "tebou".
     */
    okayFollowModeMsg = '<q>Dobrá, půjdu s tebou.</q> '

    /* note that we're already in "follow" mode */
    alreadyFollowModeMsg = '<q>Já už přeci jdu s {kýmčím dobj}.</q> '

    /* extinguishing a candle */
    okayExtinguishCandleMsg = '{Kdoco} zh{áší|asil[a]} {kohoco dobj}. '

    /* acknowledge attachment */
    okayAttachToMsg =
        '{Kdoco} přip{íná|nul[a]} {kohoco dobj} k {komučemu iobj}. '

    /* acknowledge detachment */
    okayDetachFromMsg =
        '{Kdoco} od{píná|epnul[a]} {kohoco dobj} od {kohočeho iobj}. '

    /*
     *   the PC's responses to conversational actions applied to oneself
     *   need some reworking for NPC's 
     */
    cannotTalkToSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotAskSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotAskSelfForMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotTellSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotGiveToSelfMsg = '{Kdoco} ničeho nedosáhne, když dá {kohoco dobj}
        sobě. '
    cannotShowToSelfMsg = '{Kdoco} ničeho nedosáhne, když ukáže {kohoco dobj}
        sobě. '
;

/* ------------------------------------------------------------------------ */
/*
 *   Standard tips
 */

scoreChangeTip: Tip
    "Pokud bys příště raději neslyšel{a} o změnách skóre, napiš
    <<aHref('vypni upozorňování', 'VYPNI UPOZORŇOVÁNÍ', 'Vypnout upozorňování na změnu skóre')>>."
;

/* ODSUD NAHORU JE HOTOVO */

footnotesTip: Tip
    "A number in [square brackets] like the one above refers to a footnote,
    which you can read by typing FOOTNOTE followed by the number:
    <<aHref('footnote 1', 'FOOTNOTE 1', 'Show footnote [1]')>>, for example.
    Footnotes usually contain added background information that might be
    interesting but isn&rsquo;t essential to the story. If you&rsquo;d
    prefer not to see footnotes at all, you can control their appearance by
    typing <<aHref('footnotes', 'FOOTNOTES', 'Control footnote
    appearance')>>."
;

oopsTip: Tip
    "Pokud jste se omylem překlepli, můžete to teď napravit zadáním OPRAVA
    (nebo jen O) následovaném opraveným slovem. Kdykoliv vás příběh upozorní
    na neznámé slovo, můžete se opravit použitím slova OPRAVA jako dalšího
    zadaného příkazu."
;

fullScoreTip: Tip
    "To see a detailed accounting of your score, type
    <<aHref('full score', 'FULL SCORE')>>."
;

exitsTip: Tip
    "You can control the exit listings with the EXITS command.
    <<aHref('exits status', 'EXITS STATUS',
            'Turn on status line exit listings')>>
    shows the exit list in the status line,
    <<aHref('exits look', 'EXITS LOOK', 'List exits in room descriptions')>>
    shows a full exit list in each room description,
    <<aHref('exits on', 'EXITS ON', 'Turn on all exit lists')>>
    shows both, and
    <<aHref('exits off', 'EXITS OFF', 'Turn off all exit lists')>>
    turns off both kinds of exit lists."
;

undoTip: Tip
    "If this didn't turn out quite the way you expected, note that you
    can always take back a turn by typing <<aHref('undo', 'UNDO',
        'Take back the most recent command')>>.  You can even use
    UNDO repeatedly to take back several turns in a row. "
;


/* ODSUD DOLU JE HOTOVO */

/* ------------------------------------------------------------------------ */
/*
 *   Listers
 */

/*
 *   The basic "room lister" object - this is the object that we use by
 *   default with showList() to construct the listing of the portable
 *   items in a room when displaying the room's description.  
 */
roomLister: Lister
    /* show the prefix/suffix in wide mode */
    showListPrefixWide(itemCount, pov, parent, lst) { "Vid{[íš]|ěl[a] [jsi]} tu "; }
    showListSuffixWide(itemCount, pov, parent) { ". "; }

    /* show the tall prefix */
    showListPrefixTall(itemCount, pov, parent) { "Vid{[íš]|ěl[a] [jsi]} tu:"; }
;

/*
 *   The basic room lister for dark rooms 
 */
darkRoomLister: Lister
    showListPrefixWide(itemCount, pov, parent, lst)
        { "V přítmí {vid[íš]|[jsi] viděl[a]} "; }

    showListSuffixWide(itemCount, pov, parent) { ". "; }

    showListPrefixTall(itemCount, pov, parent)
        { "V přítmí {vid[íš]|[jsi] viděl[a]}:"; }
;

/*
 *   A "remote room lister".  This is used to describe the contents of an
 *   adjoining room.  For example, if an actor is standing in one room,
 *   and can see into a second top-level room through a window, we'll use
 *   this lister to describe the objects the actor can see through the
 *   window. 
 */
class RemoteRoomLister: Lister
    construct(room) { remoteRoom = room; }
    
    showListPrefixWide(itemCount, pov, parent, lst)
        { "\^<<remoteRoom.inRoomName(pov)>> {vid[íš]|[jsi] viděl[a]} "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent)
        { "\^<<remoteRoom.inRoomName(pov)>> {vid[íš]|[jsi] viděl[a]}:"; }

    /* the remote room we're viewing */
    remoteRoom = nil
;

/*
 *   A simple customizable room lister.  This can be used to create custom
 *   listers for things like remote room contents listings.  We act just
 *   like any ordinary room lister, but we use custom prefix and suffix
 *   strings provided during construction.  
 */
class CustomRoomLister: Lister
    construct(prefix, suffix)
    {
        prefixStr = prefix;
        suffixStr = suffix;
    }

    showListPrefixWide(itemCount, pov, parent, lst) { "<<prefixStr>> "; }
    showListSuffixWide(itemCount, pov, parent) { "<<suffixStr>> "; }
    showListPrefixTall(itemCount, pov, parent) { "<<prefixStr>>:"; }

    /* our prefix and suffix strings */
    prefixStr = nil
    suffixStr = nil
;

/*
 *   Single-list inventory lister.  This shows the inventory listing as a
 *   single list, with worn items mixed in among the other inventory items
 *   and labeled "(being worn)".  
 */
actorSingleInventoryLister: InventoryLister
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        gMessageParams(parent);
        "{Kdoco parent} nes{e|l[a parent]} ";
    }

    showListSuffixWide(itemCount, pov, parent)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent)
    {
        gMessageParams(parent);
        "{Kdoco parent} nes{e|l[a parent]}:";
    }

    showListContentsPrefixTall(itemCount, pov, parent)
    {
        gMessageParams(parent);
        "{Kdoco parent}, kter{ý} nes{e|l[a parent]}:";
    }

    showListEmpty(pov, parent)
    {
        gMessageParams(parent);
        "{Kdoco parent} {má|měl[a parent]} prázdné ruce. ";
    }
;

/*
 *   Standard inventory lister for actors - this will work for the player
 *   character and NPC's as well.  This lister uses a "divided" format,
 *   which segregates the listing into items being carried and items being
 *   worn.  We'll combine the two lists into a single sentence if the
 *   overall list is short, otherwise we'll show two separate sentences for
 *   readability.  
 */
actorInventoryLister: DividedInventoryLister
    /*
     *   Show the combined inventory listing, putting together the raw
     *   lists of the items being carried and the items being worn. 
     */
    showCombinedInventoryList(parent, carrying, wearing)
    {
        /* if one or the other sentence is empty, the format is simple */
        if (carrying == '' && wearing == '')
        {
            /* the parent is completely empty-handed */
            showInventoryEmpty(parent);
        }
        else if (carrying == '')
        {
            /* the whole list is being worn */
            showInventoryWearingOnly(parent, wearing);
        }
        else if (wearing == '')
        {
            /* the whole list is being carried */
            showInventoryCarryingOnly(parent, carrying);
        }
        else
        {
            /*
             *   Both listings are populated.  Count the number of
             *   comma-separated or semicolon-separated phrases in each
             *   list.  This will give us an estimate of the grammatical
             *   complexity of each list.  If we have very short lists, a
             *   single sentence will be easier to read; if the lists are
             *   long, we'll show the lists in separate sentences.  
             */
            if (countPhrases(carrying) + countPhrases(wearing)
                <= singleSentenceMaxNouns)
            {
                /* short enough: use a single-sentence format */
                showInventoryShortLists(parent, carrying, wearing);
            }
            else
            {
                /* long: use a two-sentence format */
                showInventoryLongLists(parent, carrying, wearing);
            }
        }
    }

    /*
     *   Count the noun phrases in a string.  We'll count the number of
     *   elements in the list as indicated by commas and semicolons.  This
     *   might not be a perfect count of the actual number of noun phrases,
     *   since we could have commas setting off some other kind of clauses,
     *   but it nonetheless will give us a good estimate of the overall
     *   complexity of the text, which is what we're really after.  The
     *   point is that we want to break up the listings if they're long,
     *   but combine them into a single sentence if they're short.  
     */
    countPhrases(txt)
    {
        local cnt;
        
        /* if the string is empty, there are no phrases at all */
        if (txt == '')
            return 0;

        /* a non-empty string has at least one phrase */
        cnt = 1;

        /* scan for commas and semicolons */
        for (local startIdx = 1 ;;)
        {
            local idx;
            
            /* find the next phrase separator */
            idx = rexSearch(phraseSepPat, txt, startIdx);

            /* if we didn't find it, we're done */
            if (idx == nil)
                break;

            /* count it */
            ++cnt;

            /* continue scanning after the separator */
            startIdx = idx[1] + idx[2];
        }

        /* return the count */
        return cnt;
    }

    phraseSepPat = static new RexPattern(',(?! and )|;| and |<rparen>')

    /*
     *   Once we've made up our mind about the format, we'll call one of
     *   these methods to show the final sentence.  These are all separate
     *   methods so that the individual formats can be easily tweaked
     *   without overriding the whole combined-inventory-listing method. 
     *
     *   TODO: vyzkoušet prázdný inventář - proč v originálu bylo volání build?
     */
    showInventoryEmpty(parent)
    {
        /* empty inventory */
        "{Nem[áš]|Neměl[a] [jsi]} u sebe nic. ";
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "Nic {nenes[eš]|[jsi] nenesl[a]}, {m[áš]|měl[a] [jsi]} ale na sobě <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        /* short lists - combine carried and worn in a single sentence */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>> a na sobě {m[áš]|[jsi] měl[a]} <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        /* long lists - show carried and worn in separate sentences */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>>.
        Na sobě {m[áš]|[jsi] měl[a]} <<wearing>>. ";
    }

/* ODSUD NAHORU JE HOTOVO */

    /*
     *   For 'tall' listings, we'll use the standard listing style, so we
     *   need to provide the framing messages for the tall-mode listing.  
     *
     *   TODO: Prozkoušet tall režim.
     */
    showListPrefixTall(itemCount, pov, parent)
        { "U sebe m{áš}:"; }
    showListContentsPrefixTall(itemCount, pov, parent)
        { "<<buildSynthParam('A/he', parent)>>, who {is} carrying:"; }
    showListEmpty(pov, parent)
        { "<<buildSynthParam('The/he', parent)>> {is} empty-handed. "; }
;

/*
 *   Special inventory lister for non-player character descriptions - long
 *   form lister.  This is used to display the inventory of an NPC as part
 *   of the full description of the NPC.
 *   
 *   This long form lister is meant for actors with lengthy descriptions.
 *   We start the inventory listing on a new line, and use the actor's
 *   full name in the list preface.  
 */
actorHoldingDescInventoryListerLong: actorInventoryLister
    showInventoryEmpty(parent)
    {
        /* empty inventory - saying nothing in an actor description */
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "<.p><<buildSynthParam('The/he', parent)>> {is}
        wearing <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "<.p><<buildSynthParam('Kdoco', parent)>> <<tSel('nes'
            + buildSynthParam('eš', parent), 'nesl'
            + buildSynthParam('a', parent))>> <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* short lists - combine carried and worn in a single sentence */
        "<.p><<buildParam('The/he', nm)>> {is} carrying <<carrying>>,
        and <<buildParam('it\'s', nm)>>{subj} wearing <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* long lists - show carried and worn in separate sentences */
        "<.p><<buildParam('The/he', nm)>> {is} carrying <<carrying>>.
        <<buildParam('It\'s', nm)>> wearing <<wearing>>. ";
    }
;

/* short form of non-player character description inventory lister */
actorHoldingDescInventoryListerShort: actorInventoryLister
    showInventoryEmpty(parent)
    {
        /* empty inventory - saying nothing in an actor description */
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "<<buildSynthParam('It/he', parent)>> {is} wearing <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "<<buildSynthParam('It/he', parent)>> {is} carrying <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* short lists - combine carried and worn in a single sentence */
        "<<buildParam('It/he', nm)>> {is} carrying <<carrying>>, and
        <<buildParam('it\'s', nm)>>{subj} wearing <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* long lists - show carried and worn in separate sentences */
        "<<buildParam('It/he', nm)>> {is} carrying <<carrying>>.
        <<buildParam('It\'s', nm)>> wearing <<wearing>>. ";
    }
;

/*
 *   Base contents lister for things.  This is used to display the contents
 *   of things shown in room and inventory listings; we subclass this for
 *   various purposes 
 */
class BaseThingContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        gMessageParams(parent);
        "\^<<parent.name>> obsah{uje|oval[a parent]} ";
    }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }
    showListPrefixTall(itemCount, pov, parent)
    {
        gMessageParams(parent);
        "\^<<parent.name>> obsah{uje|oval[a parent]}:";
    }
    showListContentsPrefixTall(itemCount, pov, parent)
    {
        gMessageParams(parent);
        "<<parent.name>>, kter{ý parent} obsahuje:";
    }
;

/*
 *   Contents lister for things.  This is used to display the second-level
 *   contents lists for things listed in the top-level list for a room
 *   description, inventory listing, or object description.  
 */
thingContentsLister: ContentsLister, BaseThingContentsLister
;

/*
 *   Contents lister for openable things.
 */
openableContentsLister: thingContentsLister
    showListEmpty(pov, parent)
    {
        "\^<<parent.openStatus>>. ";
    }
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        "\^<<parent.openStatus>>, and contain<<parent.verbEndingSEd>> ";
    }
;

/*
 *   Contents lister for descriptions of things - this is used to display
 *   the contents of a thing as part of the long description of the thing
 *   (in response to an "examine" command); it differs from a regular
 *   thing contents lister in that we use a pronoun to refer to the thing,
 *   rather than its full name, since the full name was probably just used
 *   in the basic long description.  
 */
thingDescContentsLister: DescContentsLister, BaseThingContentsLister
    showListPrefixWide(itemCount, pov, parent, lst)
        { "\^Obsah<<tSel('uje', 'oval' + buildSynthParam('a', parent))>> "; }
;

/*
 *   Base contents lister for "LOOK <PREP>" commands (LOOK IN, LOOK UNDER,
 *   LOOK BEHIND, etc).  This can be subclasses for the appropriate LOOK
 *   <PREP> command matching the container type - LOOK UNDER for
 *   undersides, LOOK BEHIND for rear containers, etc.  To use this class,
 *   combine it via multiple inheritance with the appropriate
 *   Base<Prep>ContentsLister for the preposition type.  
 */
class LookWhereContentsLister: DescContentsLister
    showListEmpty(pov, parent)
    {
        /* show a default message indicating the surface is empty */
        gMessageParams(parent);
        defaultDescReport('{You/he} {sees} nothing ' + parent.objInPrep
                          + ' {the parent/him}. ');
    }
;

/*
 *   Contents lister for descriptions of things whose contents are
 *   explicitly inspected ("look in").  This differs from a regular
 *   contents lister in that we explicitly say that the object is empty if
 *   it's empty.
 */
thingLookInLister: LookWhereContentsLister, BaseThingContentsLister
    showListEmpty(pov, parent)
    {
        /*
         *   Indicate that the list is empty, but make this a default
         *   descriptive report.  This way, if we report any special
         *   descriptions for items contained within the object, we'll
         *   suppress this default description that there's nothing to
         *   describe, which is clearly wrong if there are
         *   specially-described contents. 
         */
        gMessageParams(parent);
        defaultDescReport('{Kdoco} {sees} nothing unusual
            in {the parent/him}. ');
    }
;

/*
 *   Default contents lister for newly-revealed objects after opening a
 *   container.  
 */
openableOpeningLister: BaseThingContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        /*
         *   This list is, by default, generated as a replacement for the
         *   default "Opened" message in response to an OPEN command.  We
         *   therefore need different messages for PC and NPC actions,
         *   since this serves as the description of the entire action.
         *   
         *   Note that if you override the Open action response for a given
         *   object, you might want to customize this lister as well, since
         *   the message we generate (especially for an NPC action)
         *   presumes that we'll be the only response the command.  
         */
        gMessageParams(pov, parent);
        if (pov.isPlayerChar())
            "Otevření {kohočeho parent} odhalilo ";
        else
            "{Kdoco pov} otevřel[a pov] {kohoco parent} a odhalil[a pov] ";
    }
;

/*
 *   Base contents lister.  This class handles contents listings for most
 *   kinds of specialized containers - Surfaces, RearConainers,
 *   RearSurfaces, and Undersides.  The main variation in the contents
 *   listings for these various types of containers is simply the
 *   preposition that's used to describe the relationship between the
 *   container and the contents, and for this we can look to the objInPrep
 *   property of the container.
 *
 *   V češtině v minulém čase se musí rozlišit ještě rod objektu, resp. prvního
 *   objektu v seznamu více objektů. Např. "Na staré kredenci je/byla malá
 *   plechovka." vs. "Na pultu je/byl mosazný zvonek."
 *
 *   TODO: Pro množné číslo je potřeba spočítat ženské a mužské rody a podle
 *   toho zvolit byli/byly.
 *   TODO: šabloby pro "bylo"
 */
class BaseContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        "\^<<parent.objInPrep>> <<parent.nameKomCem>>
        <<itemCount == 1 ? tSel('je', lst[1].gender == 1 || lst[1].gender == 2 ? 'byl' : lst[1].gender == 3 ?
            'byla' : 'bylo') : tSel('jsou', 'byly')>> ";
    }
    showListSuffixWide(itemCount, pov, parent)
    {
        ". ";
    }
    showListPrefixTall(itemCount, pov, parent)
    {
        "\^<<parent.objInPrep>> <<parent.theNameObj>>
        <<itemCount == 1 ? tSel('is', 'was') : tSel('are', 'were')>>:";
    }
    showListContentsPrefixTall(itemCount, pov, parent)
    {
        "<<parent.name>>, <<parent.objInPrep>> which
        <<itemCount == 1 ? tSel('is', 'was') : tSel('are', 'were')>>:";
    }
;


/*
 *   Base class for contents listers for a surface 
 *
 *   Tato třída se stará o vypisování objeků, pokud jsou položeny
 *   na nějakém povrchu. Proto si zde vlajkou zasignalizujeme, že chceme
 *   vypsat položené objekty v prvním pádu, místo obvyklého čtvrtého.
 *   "Stará kredenc obsahuje malou modrou svítilnu. Na staré kredenci je
 *   *malá plechovka*."
 */
class BaseSurfaceContentsLister: BaseContentsLister
    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKdoCo, pov, infoTab);
    }
;

/*
 *   Contents lister for a surface 
 */
surfaceContentsLister: ContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for explicitly looking in a surface 
 */
surfaceLookInLister: LookWhereContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for a surface, used in a long description. 
 */
surfaceDescContentsLister: DescContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for room parts
 */
roomPartContentsLister: surfaceContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    /* the part I'm listing */
    part_ = nil
;

/*
 *   contents lister for room parts, used in a long description 
 */
roomPartDescContentsLister: surfaceDescContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;

/*
 *   Look-in lister for room parts.  Most room parts act like surfaces
 *   rather than containers, so base this lister on the surface lister.  
 */
roomPartLookInLister: surfaceLookInLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;
                         
/*
 *   Base class for contents listers for an Underside.  
 */
class BaseUndersideContentsLister: BaseContentsLister
;

/* basic contents lister for an Underside */
undersideContentsLister: ContentsLister, BaseUndersideContentsLister
;

/* contents lister for explicitly looking under an Underside */
undersideLookUnderLister: LookWhereContentsLister, BaseUndersideContentsLister
;

/* contents lister for moving an Underside and abandoning its contents */
undersideAbandonContentsLister: undersideLookUnderLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, lst)
        { "Moving <<parent.theNameObj>> reveal<<tSel('s', 'ed')>> "; }
    showListSuffixWide(itemCount, pov, parent)
        { " underneath. "; }
    showListPrefixTall(itemCount, pov, parent)
        { "Moving <<parent.theNameObj>> reveal<<tSel('s', 'ed')>>:"; }
;
 
/* contents lister for an Underside, used in a long description */
undersideDescContentsLister: DescContentsLister, BaseUndersideContentsLister
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        "Under <<parent.theNameObj>>
        <<itemCount == 1 ? tSel('is', 'was')
                         : tSel('are', 'were')>> ";
    }
;

/*
 *   Base class for contents listers for an RearContainer or RearSurface 
 */
class BaseRearContentsLister: BaseContentsLister
;

/* basic contents lister for a RearContainer or RearSurface */
rearContentsLister: ContentsLister, BaseRearContentsLister
;

/* contents lister for explicitly looking behind a RearContainer/Surface */
rearLookBehindLister: LookWhereContentsLister, BaseRearContentsLister
;
 
/* lister for moving a RearContainer/Surface and abandoning its contents */
rearAbandonContentsLister: undersideLookUnderLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, lst)
        { "Moving <<parent.theNameObj>> reveal<<tSel('s', 'ed')>> "; }
    showListSuffixWide(itemCount, pov, parent)
        { " behind <<parent.itObj>>. "; }
    showListPrefixTall(itemCount, pov, parent)
        { "Moving <<parent.theNameObj>> reveal<<tSel('s', 'ed')>>:"; }
;
 
/* long-description contents lister for a RearContainer/Surface */
rearDescContentsLister: DescContentsLister, BaseRearContentsLister
    showListPrefixWide(itemCount, pov, parent, lst)
    {
        "Behind <<parent.theNameObj>>
        <<itemCount == 1 ? tSel('is', 'was')
                         : tSel('are', 'were')>> ";
    }
;


/*
 *   Base class for specialized in-line contents listers.  This shows the
 *   list in the form "(<prep> which is...)", with the preposition obtained
 *   from the container's objInPrep property.  
 *
 *   TODO: Pro množné číslo je potřeba spočítat ženské a mužské rody a podle
 *   toho zvolit byli/byly.
 *   TODO: šabloby pro "bylo"
 */
class BaseInlineContentsLister: ContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(cnt, pov, parent, lst)
    {
    	local kt = ['kterém', 'kterém', 'které', 'kterém'];
        " (<<parent.objInPrep>> <<kt[parent.gender]>> <<cnt == 1 ? tSel('je',
    	    lst[1].gender == 1 || lst[1].gender == 2 ? 'byl' : lst[1].gender == 3 ? 'byla' : 'bylo') :
    	    tSel('jsou', 'byly')>> ";
    }
    showListSuffixWide(itemCount, pov, parent)
        { ")"; }
;

/* ODSUD DOLU JE HOTOVO */

/*
 *   Contents lister for a generic in-line list entry.  We customize the
 *   wording slightly here: rather than saying "(in which...)" as the base
 *   class would, we use the slightly more readable "(which contains...)".
 *
 *   TODO: také množné číslo - dveře obsahovaly
 */
inlineListingContentsLister: BaseInlineContentsLister
    showListPrefixWide(cnt, pov, parent, lst)
        {
    	    local kt = ['který', 'který', 'která', 'které'];
    	    local ob = ['obsahoval', 'obsahoval', 'obsahovala', 'obsahovalo'];
    	    " (<<kt[parent.gender]>> <<tSel('obsahuje', ob[parent.gender])>> ";
    	}
;

/* in-line contents lister for a surface */
surfaceInlineContentsLister: BaseInlineContentsLister
;

/* in-line contents lister for an Underside */
undersideInlineContentsLister: BaseInlineContentsLister
;

/* in-line contents lister for a RearContainer/Surface */
rearInlineContentsLister: BaseInlineContentsLister
;

/*
 *   Contents lister for keyring list entry.  This is used to display a
 *   keyring's contents in-line with the name of the keyring,
 *   parenthetically. 
 */
keyringInlineContentsLister: inlineListingContentsLister
    showListPrefixWide(cnt, pov, parent, lst)
        { " (s navlečen<<lst[1].gender == 3 ? 'ou' : 'ým'>> "; }
    showListSuffixWide(cnt, pov, parent)
        { ")"; }
    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKymCim, pov, infoTab);
    }
;

/* ODSUD NAHORU JE HOTOVO */

/*
 *   Contents lister for "examine <keyring>" 
 */
keyringExamineContentsLister: DescContentsLister
    showListEmpty(pov, parent)
    {
        "\^<<parent.nameIs>> empty. ";
    }
    showListPrefixWide(cnt, pov, parent, lst)
    {
        "Attached to <<parent.theNameObj>>
        <<cnt == 1 ? tSel('is', 'was')
                   : tSel('are', 'were')>> ";
    }
    showListSuffixWide(itemCount, pov, parent)
    {
        ". ";
    }
;

/*
 *   Lister for actors aboard a traveler.  This is used to list the actors
 *   on board a vehicle when the vehicle arrives or departs a location.  
 */
aboardVehicleLister: Lister
    showListPrefixWide(itemCount, pov, parent, lst)
        { " (veze "; }
    showListSuffixWide(itemCount, pov, parent)
        { ")"; }

    /* list anything whose isListedAboardVehicle returns true */
    isListed(obj) { return obj.isListedAboardVehicle; }
;

/*
 *   A simple lister to show the objects to which a given Attachable
 *   object is attached.  This shows the objects which have symmetrical
 *   attachment relationships to the given parent object, or which are
 *   "major" items to which the parent is attached.  
 */
class SimpleAttachmentLister: Lister
    construct(parent) { parent_ = parent; }
    
    showListEmpty(pov, parent)
        { /* say nothing when there are no attachments */ }
    
    showListPrefixWide(cnt, pov, parent, lst)
        { "<.p>\^<<parent.nameIs>> attached to "; }
    showListSuffixWide(cnt, pov, parent)
        { ". "; }

    /* ask the parent if we should list each item */
    isListed(obj) { return parent_.isListedAsAttachedTo(obj); }

    /*
     *   the parent object - this is the object whose attachments are being
     *   listed 
     */
    parent_ = nil
;

/*
 *   The "major" attachment lister.  This lists the objects which are
 *   attached to a given parent Attachable, and for which the parent is
 *   the "major" item in the relationship.  The items in the list are
 *   described as being attached to the parent.  
 */
class MajorAttachmentLister: SimpleAttachmentLister
    showListPrefixWide(cnt, pov, parent, lst) { "<.p>\^"; }
    showListSuffixWide(cnt, pov, parent)
    {
        " <<cnt == 1 ? tSel('je', 'byl/a/o')
                     : tSel('jsou', 'byli/y/a')>>
        připevněn/a/o/y k <<parent.nameKomuCemu>>. ";
    }

    /* ask the parent if we should list each item */
    isListed(obj) { return parent_.isListedAsMajorFor(obj); }
;

/* ODSUD DOLU JE HOTOVO */

/*
 *   Finish Options lister.  This lister is used to offer the player
 *   options in finishGame(). 
 */
finishOptionsLister: Lister
    showListPrefixWide(cnt, pov, parent, lst)
    {
        "<.p>Chceš ";
    }
    showListSuffixWide(cnt, pov, parent)
    {
        /* end the question, add a blank line, and show the ">" prompt */
        "?\b&gt;";
    }
    
    isListed(obj) { return obj.isListed; }
    listCardinality(obj) { return 1; }
    
    showListItem(obj, options, pov, infoTab)
    {
        /* obj is a FinishOption object; show its description */
        obj.desc;
    }
    
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   for the last separator, show "or" rather than "and"; for
         *   others, inherit the default handling 
         */
        if (curItemNum + 1 == totalItems)
        {
                " nebo ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }
;

/*
 *   Equivalent list state lister.  This shows a list of state names for a
 *   set of otherwise indistinguishable items.  We show the state names in
 *   parentheses, separated by commas only (i.e., no "and" separating the
 *   last two items); we use this less verbose format so that we blend
 *   into the larger enclosing list more naturally.
 *   
 *   The items to be listed are EquivalentStateInfo objects.  
 */
equivalentStateLister: Lister
    showListPrefixWide(cnt, pov, parent, lst) { " ("; }
    showListSuffixWide(cnt, pov, parent) { ")"; }
    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }
    showListItem(obj, options, pov, infoTab)
    {
        "<<spellIntBelow(obj.getEquivCount(), obj.gender, 100)>> <<obj.getName()>>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        if (curItemNum < totalItems)
            ", ";
    }
;

/* in case the exits module isn't included in the build */
property destName_, destIsBack_, others_, enableHyperlinks;

/*
 *   Basic room exit lister.  This shows a list of the apparent exits from
 *   a location.
 *   
 *   The items to be listed are DestInfo objects.  
 */
class ExitLister: Lister
    showListPrefixWide(cnt, pov, parent, lst)
    {
        if (cnt == 1)
            "Jediný zřetelný směr ved{e|l} ";
        else
            "Zřetelné směry ved{ou|ly} ";
    }
    showListSuffixWide(cnt, pov, parent) { ". "; }

    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }

    showListItem(obj, options, pov, infoTab)
    {
        /*
         *   Show the back-to-direction prefix, if we don't know the
         *   destination name but this is the back-to direction: "back to
         *   the north" and so on. 
         */
        if (obj.destIsBack_ && obj.destName_ == nil)
            say(obj.dir_.backToPrefix + ' ');
        
        /* show the direction */
        showListItemDirName(obj, nil);

        /* if the destination is known, show it as well */
        if (obj.destName_ != nil)
        {
            /*
             *   if we have a list of other directions going to the same
             *   place, show it parenthetically 
             */
            otherExitLister.showListAll(obj.others_, 0, 0);

            /* if this is the way back, say so */
            if (obj.destIsBack_)
                " zpátky";

            /* show the destination */
            " <<obj.destName_>>";
        }
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   if we have a "back to" name, use the "long" notation - this is
         *   important because we'll use commas in the directions with
         *   known destination names 
         */
        if ((options & hasBackNameFlag) != 0)
            options |= ListLong;

        /*
         *   for a two-item list, if either item has a destination name,
         *   show a comma or semicolon (depending on 'long' vs 'short' list
         *   mode) before the "and"; for anything else, use the default
         *   handling 
         */
        if (curItemNum == 1
            && totalItems == 2
            && (options & hasDestNameFlag) != 0)
        {
            if ((options & ListLong) != 0)
                " a ";
            else
                " a ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }

    /* show a direction name, hyperlinking it if appropriate */
    showListItemDirName(obj, initCap)
    {
        local dirname;
        
        /* get the name */
        dirname = obj.dir_.name;

        /* capitalize the first letter, if desired */
        if (initCap)
            dirname = dirname.substr(1,1).toUpper() + dirname.substr(2);

        /* show the name with a hyperlink or not, as configured */
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(dirname, dirname));
        else
            say(dirname);
    }

    /* this lister shows destination names */
    listerShowsDest = true

    /*
     *   My special options flag: at least one object in the list has a
     *   destination name.  The caller should set this flag in our options
     *   if applicable. 
     */
    hasDestNameFlag = ListerCustomFlag(1)
    hasBackNameFlag = ListerCustomFlag(2)
    nextCustomFlag = ListerCustomFlag(3)
;

/*
 *   Show a list of other exits to a given destination.  We'll show the
 *   list parenthetically, if there's a list to show.  The objects to be
 *   listed are Direction objects.  
 */
otherExitLister: Lister
    showListPrefixWide(cnt, pov, parent, lst) { " (nebo "; }
    showListSuffixWide(cnt, pov, parent) { ")"; }

    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }

    showListItem(obj, options, pov, infoTab)
    {
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(obj.name, obj.name));
        else
            say(obj.name);
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   simply show "or" for all items (these lists are usually
         *   short, so omit any commas) 
         */
        if (curItemNum != totalItems)
            " nebo ";
    }
;

/*
 *   Show room exits as part of a room description, using the "verbose"
 *   sentence-style notation.  
 */
lookAroundExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent, lst)
    {
        /* add a paragraph break before the exit listing */
        "<.roompara>";

        /* inherit default handling */
        inherited(cnt, pov, parent);
    }    
;

/*
 *   Show room exits as part of a room description, using the "terse"
 *   notation. 
 */
lookAroundTerseExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent, lst)
    {
        "<.roompara><.parser>Očividné směry: ";
    }
    showListItem(obj, options, pov, infoTab)
    {
        /* show the direction name */
        showListItemDirName(obj, true);
    }
    showListSuffixWide(cnt, pov, parent)
    {
        "<./parser> ";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a comma between items */
        if (curItemNum != totalItems)
            ", ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

/*
 *   Show room exits in response to an explicit request (such as an EXITS
 *   command).  
 */
explicitExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "{Nejsou|Nebyly tu} žádné očividné východy. ";
    }
;

/*
 *   Show room exits in the status line (used in HTML mode only)
 */
statuslineExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "<br><b>Směry:</b> <i>žádné</i><br>";
    }
    showListPrefixWide(cnt, pov, parent, lst)
    {
        "<br><b>Směry:</b> ";
    }
    showListSuffixWide(cnt, pov, parent)
    {
        "<br>";
    }
    showListItem(obj, options, pov, infoTab)
    {
        "<a plain href='<<obj.dir_.name>>'><<obj.dir_.name>></a>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a space between items */
        if (curItemNum != totalItems)
            " &nbsp; ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

/*
 *   Implied action announcement grouper.  This takes a list of
 *   ImplicitActionAnnouncement reports and returns a single message string
 *   describing the entire list of actions.  
 */
implicitAnnouncementGrouper: object
    /*
     *   Configuration option: keep all failures in a list of implied
     *   announcements.  If this is true, then we'll write things like
     *   "trying to unlock the door and then open it"; if nil, we'll
     *   instead write simply "trying to unlock the door".
     *   
     *   By default, we keep only the first of a group of failures.  A
     *   group of failures is always recursively related, so the first
     *   announcement refers to the command that actually failed; the rest
     *   of the announcements are for the enclosing actions that triggered
     *   the first action.  All of the enclosing actions failed as well,
     *   but only because the first action failed.
     *   
     *   Announcing all of the actions is too verbose for most tastes,
     *   which is why we set the default here to nil.  The fact that the
     *   first action in the group failed means that we necessarily won't
     *   carry out any of the enclosing actions, so the enclosing
     *   announcements don't tell us much.  All they really tell us is why
     *   we're running the action that actually failed, but that's almost
     *   always obvious, so suppressing them is usually fine.  
     */
    keepAllFailures = nil

    /* build the composite message */
    compositeMessage(lst)
    {
        local txt;
        local ctx = new ListImpCtx();

        /* add the text for each item in the list */
        for (txt = '', local i = 1, local len = lst.length() ; i <= len ; ++i)
        {
            local curTxt;

            /* get this item */
            local cur = lst[i];

            /* we're not in a 'try' or 'ask' sublist yet */
            ctx.isInSublist = nil;

            /* set the underlying context according to this item */
            ctx.setBaseCtx(cur);

            /*
             *   Generate the announcement for this element.  Generate the
             *   announcement from the message property for this item using
             *   our running list context.  
             */
            curTxt = cur.getMessageText(
                cur.getAction().getOriginalAction(), ctx);

            /*
             *   If this one is an attempt only, and it's followed by one
             *   or more other attempts, the attempts must all be
             *   recursively related (in other words, the first attempt was
             *   an implied action required by the second attempt, which
             *   was required by the third, and so on).  They have to be
             *   recursively related, because otherwise we wouldn't have
             *   kept trying things after the first failed attempt.
             *   
             *   To make the series of failed attempts sound more natural,
             *   group them into a single "trying to", and keep only the
             *   first failure: rather than "trying to unlock the door,
             *   then trying to open the door", write "trying to unlock the
             *   door and then open it".
             *   
             *   An optional configuration setting makes us keep only the
             *   first failed operation, so we'd instead write simply
             *   "trying to unlock the door".
             *   
             *   Do the same grouping for attempts interrupted for an
             *   interactive question.  
             */
            while ((cur.justTrying && i < len && lst[i+1].justTrying)
                   || (cur.justAsking && i < len && lst[i+1].justAsking))
            {
                local addTxt;
                
                /*
                 *   move on to the next item - we're processing it here,
                 *   so we don't need to handle it again in the main loop 
                 */
                ++i;
                cur = lst[i];

                /* remember that we're in a try/ask sublist */
                ctx.isInSublist = true;

                /* add the list entry for this action, if desired */
                if (keepAllFailures)
                {
                    /* get the added text */
                    addTxt = cur.getMessageText(
                        cur.getAction().getOriginalAction(), ctx);

                    /*
                     *   if both the text so far and the added text are
                     *   non-empty, string them together with 'and then';
                     *   if one or the other is empty, use the non-nil one 
                     */
                    if (addTxt != '' && curTxt != '')
                        curTxt += ' and then ' + addTxt;
                    else if (addTxt != '')
                        curTxt = addTxt;
                }
            }

            /* add a separator before this item if it isn't the first */
            if (txt != '' && curTxt != '')
                txt += ', potom ';

            /* add the current item's text */
            txt += curTxt;
        }

        /* if we ended up with no text, the announcement is silent */
        if (txt == '')
            return '';

        /* wrap the whole list in the usual full standard phrasing */
        return standardImpCtx.buildImplicitAnnouncement(txt);
    }
;

/*           
 *   Suggested topic lister. 
 */
class SuggestedTopicLister: Lister
    construct(asker, askee, explicit)
    {
        /* remember the actors */
        askingActor = asker;
        targetActor = askee;

        /* remember whether this is explicit or implicit */
        isExplicit = explicit;

        /* cache the actor's scope list */
        scopeList = asker.scopeList();
    }
    /* tady budeme chtit postrpocess */
    showList(pov, parent, lst, options, indent, infoTab, parentGroup)
    {
        local groups;
        local groupTab;
        local singles;
        local origLst;
        local itemCount;

        /* remember the original list */
        origLst = lst;

        /* filter the list to get only the items we actually will list */
        lst = getFilteredList(lst, infoTab);

        /* create a lookup table to keep track of the groups we've seen */
        groupTab = new LookupTable();
        groups = new Vector(10);

        /* set up a vector to keep track of the singles */
        singles = new Vector(10);

        /* figure the groupings */
        itemCount = getListGrouping(groupTab, groups, singles,
                                    lst, parentGroup);

        /*
         *   Now that we've figured out what's in the list and how it's
         *   arranged into groups, show the list.
         */
        local text = outputManager.curOutputStream.captureOutput({:
            showArrangedList(pov, parent, lst, options, indent, infoTab,
                         itemCount, singles, groups, groupTab, origLst)});
        text = text.findReplace('bys se',  'by ses', ReplaceAll);
        say(text);

        /*
         *   If the list is recursive, mention the contents of any items
         *   that weren't listed in the main list, and of any contents
         *   that are specifically to be listed out-of-line.  Don't do
         *   this if we're already recursively showing such a listing,
         *   since if we did so we could show items at recursive depths
         *   more than once; if we're already doing a recursive listing,
         *   our caller will itself recurse through all levels of the
         *   tree, so we don't have to recurse any further ourselves.  
         */
        if ((options & ListRecurse) != 0
            && indent == 0
            && (options & ListContents) == 0)
        {
            /* show the contents of each object we didn't list */
            showSeparateContents(pov, origLst,
                                 options | ListContents, infoTab);
        }
    }

    showListPrefixWide(cnt, pov, parent, lst)
    {
        /* add the asking and target actors as global message parameters */
        gMessageParams(askingActor, targetActor);

        /* show the prefix; include a paren if not in explicit mode */
        /* TODO: Potřeba rozlišit čas, osobu */
        "<<isExplicit ? '' : '('>>Mohl{a} {[bys]|[jsi]} ";
    }
    showListSuffixWide(cnt, pov, parent)
    {
        /* end the sentence; include a paren if not in explicit mode */
        ".<<isExplicit? '' : ')'>> ";
    }
    showListEmpty(pov, parent)
    {
        /*
         *   say that the list is empty if it was explicitly requested;
         *   say nothing if the list is being added by the library 
         */
        if (isExplicit)
        {
            gMessageParams(askingActor, targetActor);
            "<<isExplicit ? '' : '('>>Nem{[áš]|ěl[a] [jsi]} na mysli nic konkrétního, o čem
            by {ses} chtěl{a} bavit s {kýmčím targetActor}.<<isExplicit ? '' : ')'>> ";
        }
    }

    showListSeparator(options, curItemNum, totalItems)
    {
        /* use "or" as the conjunction */
        if (curItemNum + 1 == totalItems)
            " nebo ";
        else
            inherited(options, curItemNum, totalItems);
    }

    /* list suggestions that are currently active */
    isListed(obj) { return obj.isSuggestionActive(askingActor, scopeList); }

    /* each item counts as one item grammatically */
    listCardinality(obj) { return 1; }

    /* suggestions have no contents */
    contentsListed(obj) { return nil; }

    /* get the list group */
    listWith(obj) { return obj.suggestionGroup; }

    /* mark as seen - nothing to do for suggestions */
    markAsSeen(obj, pov) { }

    /* show the item - show the suggestion's theName */
    showListItem(obj, options, pov, infoTab)
    {
        /* note that we're showing the suggestion */
        obj.noteSuggestion();

        /* show the name */
        say(obj.fullName);
    }

    /* don't use semicolons, even in long lists */
    longListSepTwo { listSepTwo; }
    longListSepMiddle { listSepMiddle; }
    longListSepEnd { listSepEnd; }

    /* flag: this is an explicit listing (i.e., a TOPICS command) */
    isExplicit = nil

    /* the actor who's asking for the topic list (usually the PC) */
    askingActor = nil

    /* the actor we're talking to */
    targetActor = nil

    /* our cached scope list for the actor */
    scopeList = nil
;

/* ASK/TELL suggestion list group base class */
class SuggestionListGroup: ListGroupPrefixSuffix
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /*
         *   show the short name of the item - the group prefix will have
         *   shown the appropriate long name 
         */
        say(obj.name);
    }
;

/* ASK ABOUT suggestion list group */
suggestionAskGroup: SuggestionListGroup
    groupPrefix = "se {tě targetActor} zeptat na "
;

/* TELL ABOUT suggestion list group */
suggestionTellGroup: SuggestionListGroup
    groupPrefix = "{ti targetActor} říci o "
;

/* ASK FOR suggestion list group */
suggestionAskForGroup: SuggestionListGroup
    groupPrefix = "{tě targetActor} požádat o "
;

/* GIVE TO suggestions list group */
suggestionGiveGroup: SuggestionListGroup
    groupPrefix = "{ti targetActor} dát "
;

/* SHOW TO suggestions */
suggestionShowGroup: SuggestionListGroup
    groupPrefix = "{ti targetActor} ukázat "
;

/* YES/NO suggestion group */
suggestionYesNoGroup: SuggestionListGroup
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /*
         *   if we have one each of YES and NO responses, make the entire
         *   list "say yes or no"; otherwise, use the default behavior 
         */
        if (lst.length() == 2
            && lst.indexWhich({x: x.ofKind(SuggestedYesTopic)}) != nil
            && lst.indexWhich({x: x.ofKind(SuggestedNoTopic)}) != nil)
        {
            /* we have a [yes, no] group - use the simple message */
            "odpovědět ano nebo ne";
        }
        else
        {
            /* inherit the default behavior */
            inherited(pov, lister, lst, options, indent, infoTab);
        }
    }
    groupPrefix = "odpovědět";
;
