#charset "utf-8"

/*
 *   TADS 3 Library - "neutral" messages for Czech language
 *   Copyright (c) 2010-2012 Tomas Blaha. All Rights Reserved.
 *
 *   Based on English language module by Michael J. Roberts with permission.
 *
 *.                                  -----
 *
 *   Copyright (c) 2000, 2006 Michael J. Roberts.  All Rights Reserved. 
 *
 *   TADS 3 Library - "neutral" messages for US English
 *
 *   This module provides standard library messages with a parser/narrator 
 *   that's as invisible (neutral) as possible.  These messages are designed 
 *   to reduce the presence of the computer as mediator in the story, to 
 *   give the player the most direct contact that we can with the scenario.
 *
 *   The parser almost always refers to itself in the third person (by 
 *   calling itself something like "this story") rather than in the first 
 *   person, and, whenever possible, avoids referring to itself in the first 
 *   place.  Our ideal phrasing is either second-person, describing things 
 *   directly in terms of the player character's experience, or "no-person," 
 *   simply describing things without mentioning the speaker or listener at 
 *   all.  For example, rather than saying "I don't see that here," we say 
 *   "you don't see that here," or "that's not here."  We occasionally stray 
 *   from this ideal where achieving it would be too awkward.
 *
 *   In the earliest days of adventure games, the parser was usually a 
 *   visible presence: the early parsers frequently reported things in the 
 *   first person, and some even had specific personalities.  This 
 *   conspicuous parser style has become less prevalent in modern games, 
 *   though, and authors now usually prefer to treat the parser as just 
 *   another part of the user interface, which like all good UI's is best 
 *   when the user doesn't notice it.  
 */

#include "adv3.h"
#include "cs_cz.h"

/* ------------------------------------------------------------------------ */
/*
 *   Build a message parameter string with the given parameter type and
 *   name.
 *   
 *   This is useful when we have a name from a variable, and we need to
 *   build the message substitution string for embedding in a larger
 *   string.  We can't just embed the name variable using <<var>>, because
 *   that would process the output piecewise - the output filter needs to
 *   see the whole {typ var} expression in one go.  So, instead of writing
 *   this:
 *   
 *.     {The/he <<var>>} {is} ...
 *   
 *   write this:
 *   
 *.     <<buildParam('The/he', var)>> {is} ...
 */
buildParam(typeString, nm)
{
    return '{' + typeString + ' ' + nm + '}';
}

/*
 *   Synthesize a message parameter, and build it into a parameter string
 *   with the given substitution type.
 *   
 *   For example, buildSynthParam('abc', obj) returns '{abc xxx}', where
 *   'xxx' is a synthesized message parameter name (created using
 *   gSynthMessageParam) for the object obj.  
 */
buildSynthParam(typeString, obj)
{
    return '{' + typeString + ' ' + gSynthMessageParam(obj) + '}';
}


/* ------------------------------------------------------------------------ */
/*
 *   Library Messages 
 */
libMessages: MessageHelper
    /*
     *   Flag: offer an explanation of the "OOPS" command when it first
     *   comes up.  We'll only show this the first time the player enters
     *   an unknown word.  If you never want to offer this message at all,
     *   simply set this flag to nil initially.
     *   
     *   See also oopsNote() below.  
     */
    offerOopsNote = true

    /*
     *   some standard commands for insertion into <a> tags - these are in
     *   the messages so they can translated along with the command set
     */
    commandLookAround = 'rozhlédni se'
    commandFullScore = 'detailní skóre'
    
    /* announce a completely remapped action */
    announceRemappedAction(action)
    {
        return '<./p0>\n<.assume>' + action.getParticiplePhrase()
            + '<./assume>\n';
    }

    /*
     *   Get a string to announce an implicit action.  This announces the
     *   current global action.  'ctx' is an ImplicitAnnouncementContext
     *   object describing the context in which the message is being
     *   displayed.  
     */
    announceImplicitAction(action, ctx)
    {
        /* build the announcement from the basic verb phrase */
        return ctx.buildImplicitAnnouncement(action.getImplicitPhrase(ctx));
    }

    /*
     *   Announce a silent implied action.  This allows an implied action
     *   to work exactly as normal (including the suppression of a default
     *   response message), but without any announcement of the implied
     *   action. 
     */
    silentImplicitAction(action, ctx) { return ''; }

    /*
     *   Get a string to announce that we're implicitly moving an object to
     *   a bag of holding to make room for taking something new.  If
     *   'trying' is true, it means we want to phrase the message as merely
     *   trying the action, not actually performing it.  
     */
    announceMoveToBag(action, ctx)
    {
        /* build the announcement, adding an explanation */
        return ctx.buildImplicitAnnouncement(
            action.getImplicitPhrase(ctx) + ' k uvolnění místa');
    }

    /* show a library credit (for a CREDITS listing) */
    showCredit(name, byline) { "<<name>> <<byline>>"; }

    /* show a library version number (for a VERSION listing) */
    showVersion(name, version) { "<<name>> verze <<version>>"; }

    /* there's no "about" information in this game */
    noAboutInfo = "<.parser>Příběh nemá žádné informace O HŘE.<./parser> "

    /*
     *   Show a list state name - this is extra state information that we
     *   show for an object in a listing involving the object.  For
     *   example, a light source might add a state like "(providing
     *   light)".  We simply show the list state name in parentheses.  
     */
    showListState(state) { " (<<state>>)"; }

    /* a set of equivalents are all in a given state */
    allInSameListState(lst, stateName)
    {
        " (<<lst.length() == 2 ? (lst[0].gender <= 2 ? 'oba' : 'obě') :
        (lst[0].gender == 1 ? 'všichni' : lst[0].gender == 4 ? 'všechna' :
        'všechny')>> <<stateName>>)";
    }

    /* generic long description of a Thing from a distance */
    distantThingDesc(obj)
    {
        gMessageParams(obj);
        "{Je|Byl[a obj]} příliš daleko, než a{bys} zahlédl{a} jakékoliv detaily. ";
    }

    /* generic long description of a Thing under obscured conditions */
    obscuredThingDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Pohledem skrz {kohoco obs} {nevid[íš]|[jsi] neviděl[a]} žádné detaily. ";
    }

    /* generic "listen" description of a Thing at a distance */
    distantThingSoundDesc(obj)
        { "Z té dálky {neslyš[íš]|[jsi] neslyšel[a]} nic konkrétního. "; }

    /* generic obscured "listen" description */
    obscuredThingSoundDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Přes {kohoco obs} {neslyš[íš]|[jsi] neslyšel[a]} nic konkrétního. ";
    }

    /* generic "smell" description of a Thing at a distance */
    distantThingSmellDesc(obj)
        { "Na tu dálku {|[jsi]} skoro nic necít{[íš]|il[a]}. "; }

    /* generic obscured "smell" description */
    obscuredThingSmellDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "Přes {kohoco obs} {|[jsi]} skoro nic necít{[íš]|il[a]}. ";
    }

    /* generic "taste" description of a Thing */
    thingTasteDesc(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} chutn{[áš]|al[a obj]} v podstatě tak, jak {[bys]|[jsi]}
            očekával{a}. ";
    }

    /* generic "feel" description of a Thing */
    thingFeelDesc(obj)
        { "Nenahmat{áv[áš]|al[a] [jsi]} nic neobvyklého. "; }

    /* obscured "read" description */
    obscuredReadDesc(obj)
    {
        gMessageParams(obj);
        "Nevid{[íš]|ěl[a] [jsi]} {kohoco obj} tak dobře, aby {jsi} {ho/ji obj}
            mohl{a} přečíst. ";
    }

    /* dim light "read" description */
    dimReadDesc(obj)
    {
        gMessageParams(obj);
        "Na přečtení {kohočeho obj} tu {je|bylo} moc málo světla. ";
    }

    /* lit/unlit match description */
    litMatchDesc(obj) { "{Kdoco obj} {je} zapálen{ý}. "; }
    unlitMatchDesc(obj) { "{Kdoco obj} {je} obyčejn{ý} zápalka. "; }

    /* lit candle description */
    litCandleDesc(obj) { "{Kdoco obj} {je} zapálen{ý}. "; }

    /*
     *   Prepositional phrases for putting objects into different types of
     *   objects. 
     */
    putDestContainer(obj) { return 'do ' + obj.nameKohoCeho; }
    putDestSurface(obj) { return 'na ' + obj.nameKohoCo; }
    putDestUnder(obj) { return 'pod ' + obj.nameKohoCo; }
    putDestBehind(obj) { return 'za ' + obj.nameKohoCo; }
    putDestFloor(obj) { return 'na ' + obj.nameKohoCo; }
    putDestRoom(obj) { return 'do ' + obj.nameKohoCeho; }

    /* the list separator character in the middle of a list */
    listSepMiddle = ", "

    /* the list separator character for a two-element list */
    listSepTwo = " a "

    /* the list separator for the end of a list of at least three elements */
    listSepEnd = " a "

    /*
     *   the list separator for the middle of a long list (a list with
     *   embedded lists not otherwise set off, such as by parentheses) 
     *
     *   TODO: Proč si myslí, že exitlister je long list? Jinak by tu klidně
     *   mohl být ten středník pro skutečně dlouhé listy.
     */
    longListSepMiddle = ", "

    /* the list separator for a two-element list of sublists */
    longListSepTwo = " a "

    /* the list separator for the end of a long list */
    longListSepEnd = " a "

    /* show the basic score message */
    showScoreMessage(points, maxPoints, turns)
    {
        local n = turns, m = maxPoints;

        while(n > 9) n /= 10;
        while(m > 9) m /= 10;

        "<<[2, 3, 4].indexOf(n) || (n >= 12 && n <= 14) || (n >= 100
            && n <= 199) ? 'Ve' : 'V'>> <<turns>> <<turns == 1 ? 'tahu' :
            'tazích'>> jsi získal{a} <<points>> <<[2, 3, 4, 6, 7].indexOf(m)
            || (m >= 12 && m <= 14) || (m >= 16 && m <= 17) || (m >= 100 &&
            m <= 199) ? 'ze' : 'z'>> <<maxPoints>> maximálně <<maxPoints == 1
            ? 'možného bodu' : 'možných bodů'>>. ";
    }

    /* show the basic score message with no maximum */
    showScoreNoMaxMessage(points, turns)
    {
        local n = turns;

        while(n > 9) n /= 10;

        "<<[2, 3, 4].indexOf(n) || (n >= 12 && n <= 14) || (n >= 100
            && n <= 199) ? 'Ve' : 'V'>> <<turns>> <<turns == 1 ? 'tahu' :
            'tazích'>> jsi získal{a} <<points>> <<points == 1 ? 'bod'
            : 'bodů'>>. ";
    }

    /* show the full message for a given score rank string */
    showScoreRankMessage(msg) { "To z tebe dělá <<msg>>. "; }

    /*
     *   show the list prefix for the full score listing; this is shown on
     *   a line by itself before the list of full score items, shown
     *   indented and one item per line 
     */
    showFullScorePrefix = "Tvé skóre zahrnuje:"

    /*
     *   show the item prefix, with the number of points, for a full score
     *   item - immediately after this is displayed, we'll display the
     *   description message for the achievement 
     */
    fullScoreItemPoints(points)
    {
        "<<points>> <<points == 1 ? 'bod' : (points < 5 ? 'body' : 'bodů')>> za ";
    }

    /* score change - first notification */
    firstScoreChange(delta)
    {
        scoreChange(delta);
        scoreChangeTip.showTip();
    }

    /* score change - notification other than the first time */
    scoreChange(delta)
    {
        "<.commandsep><.notification><<
        basicScoreChange(delta)>><./notification> ";
    }

    /*
     *   basic score change notification message - this is an internal
     *   service routine for scoreChange and firstScoreChange 
     */
    basicScoreChange(delta)
    {
        "Tvé <<aHref(commandFullScore, 'skóre', 'Ukázat detailní skóre')>>
        se právě <<delta > 0 ? 'zvýšilo' : 'snížilo'>> o
        <<spellInt(delta > 0 ? delta : -delta, 2)>>
        <<delta is in (1, -1) ? 'bod' : (delta is in (4, 3, 2, -2, -3, -4) ?
        'body' : 'bodů')>>.";
    }

    /* acknowledge turning tips on or off */
    acknowledgeTipStatus(stat)
    {
        "<.parser>Tipy jsou nyní <<stat ? 'zapnuté' : 'vypnuté'>>.<./parser> ";
    }

    /* describe the tip mode setting */
    /* TODO: Kde se tohle zobrazuje? Nemělo by být místo "zapnuté" "zapnout"?  */
    tipStatusShort(stat)
    {
        "TIPY <<stat ? 'ZAPNUTÉ' : 'VYPNUTÉ'>>";
    }

    /* get the string to display for a footnote reference */
    footnoteRef(num)
    {
        /* set up a hyperlink for the note that enters the "note n" command */
        return '<sup>[<<aHref('footnote ' + num, toString(num))>>]</sup>';
    }

    /* first footnote notification */
    firstFootnote()
    {
        footnotesTip.showTip();
    }

    /* there is no such footnote as the given number */
    noSuchFootnote(num)
    {
        "<.parser>Příběh na takovou poznámku neodkazoval.<./parser> ";
    }

    /* show the current footnote status */
    showFootnoteStatus(stat)
    {
        "Nynější nastavení je POZNÁMKY ";
        switch(stat)
        {
        case FootnotesOff:
            "VYPNUTÉ, které skrývá všechny odkazy na poznámky pod čarou.
            Zadej <<aHref('poznámky středně', 'POZNÁMKY STŘEDNĚ',
            'Nastavit poznámky pod čarou na střední zobrazování')>> a
            budou se ukazovat odkazy na všechny poznámky kromě těch, které jsi
            už četl, nebo <<aHref('poznámky naplno', 'POZNÁMKY NAPLNO',
            'Nastavit poznámky na plnou intenzitu')>> a budou se ukazovat
            odkazy na všechny poznámky. ";
            break;

        case FootnotesMedium:
            "STŘEDNĚ, které ukazuje odkazy na zatím nepřečtené poznámky pod
            čarou. Zadej <<aHref('poznámky vypnout', 'POZNÁMKY VYPNOUT',
            'Vypnout poznámky pod čarou')>> a poznámky se zcela vypnou, nebo
            <<aHref('poznámky naplno', 'POZNÁMKY NAPLNO','Nastavit poznámky pod
            čarou na plné zobrazování')>> a budou se ukazovat úplně všechny
            odkazy včetně poznámek, které už jsi četl. ";
            break;

        case FootnotesFull:
            "NAPLNO, které ukazuje odkazy na úplně všechny poznámky pod čarou.
            Zadej <<aHref('poznámky středně', 'POZNÁMKY STŘEDNĚ', 'Nastavit
            poznámky pod čarou na střední zobrazování')>> a budou se ukazovat
            odkazy na všechny poznámky kromě těch, které jsi už četl, nebo
            <<aHref('poznámky vypnout', 'POZNÁMKY VYPNOUT', 'Vypnout poznámky
            pod čarou')>> a poznámky se zcela vypnou. ";
            break;
        }
    }

    /* acknowledge a change in the footnote status */
    acknowledgeFootnoteStatus(stat)
    {
        "<.parser>Nynější nastavení je
        <<shortFootnoteStatus(stat)>>.<./parser> ";
    }

    /* show the footnote status, in short form */
    shortFootnoteStatus(stat)
    {
        "POZNÁMKY <<
          stat == FootnotesOff ? 'VYPNUTÉ'
          : stat == FootnotesMedium ? 'STŘEDNĚ'
          : 'NAPLNO' >>";
    }

    /*
     *   Show the main command prompt.
     *   
     *   'which' is one of the rmcXxx phase codes indicating what kind of
     *   command we're reading.  This default implementation shows the
     *   same prompt for every type of input, but games can use the
     *   'which' value to show different prompts for different types of
     *   queries, if desired.  
     */
    mainCommandPrompt(which) { "\b&gt;"; }

    /*
     *   Show a pre-resolved error message string.  This simply displays
     *   the given string.  
     */
    parserErrorString(actor, msg) { say(msg); }

    /* show the response to an empty command line */
    emptyCommandResponse = "<.parser>Cože {|[jsi]} to {[chceš]|chtěl[a]}?<./parser> "

    /* invalid token (i.e., punctuation) in command line */
    invalidCommandToken(ch)
    {
        "<.parser>Příběh neví, jak se vypořádat se znakem &bdquo;<<ch>>&ldquo;
            v příkazu.<./parser> ";
    }

    /*
     *   Command group prefix - this is displayed after a command line and
     *   before the first command results shown after the command line.
     *   
     *   By default, we'll show the "zero-space paragraph" marker, which
     *   acts like a paragraph break in that it swallows up immediately
     *   following paragraph breaks, but doesn't actually add any space.
     *   This will ensure that we don't add any space between the command
     *   input line and the next text.  
     */
    commandResultsPrefix = '<.p0>'

    /*
     *   Command "interruption" group prefix.  This is displayed after an
     *   interrupted command line - a command line editing session that
     *   was interrupted by a timeout event - just before the text that
     *   interrupted the command line.
     *   
     *   By default, we'll show a paragraph break here, to set off the
     *   interrupting text from the command line under construction.  
     */
    commandInterruptionPrefix = '<.p>'

    /*
     *   Command separator - this is displayed after the results from a
     *   command when another command is about to be executed without any
     *   more user input.  That is, when a command line contains more than
     *   one command, this message is displayed between each successive
     *   command, to separate the results visually.
     *   
     *   This is not shown before the first command results after a
     *   command input line, and is not shown after the last results
     *   before a new input line.  Furthermore, this is shown only between
     *   adjacent commands for which output actually occurs; if a series
     *   of commands executes without any output, we won't show any
     *   separators between the silent commands.
     *   
     *   By default, we'll just start a new paragraph.  
     */
    commandResultsSeparator = '<.p>'

    /*
     *   "Complex" result separator - this is displayed between a group of
     *   messages for a "complex" result set and adjoining messages.  A
     *   command result list is "complex" when it's built up out of
     *   several generated items, such as object identification prefixes
     *   or implied command prefixes.  We use additional visual separation
     *   to set off these groups of messages from adjoining messages,
     *   which is especially important for commands on multiple objects,
     *   where we would otherwise have several results shown together.  By
     *   default, we use a paragraph break.  
     */
    complexResultsSeparator = '<.p>'

    /*
     *   Internal results separator - this is displayed to visually
     *   separate the results of an implied command from the results for
     *   the initiating command, which are shown after the results from
     *   the implied command.  By default, we show a paragraph break.
     */
    internalResultsSeparator = '<.p>'

    /*
     *   Command results suffix - this is displayed just before a new
     *   command line is about to be read if any command results have been
     *   shown since the last command line.
     *   
     *   By default, we'll show nothing extra.  
     */
    commandResultsSuffix = ''

    /*
     *   Empty command results - this is shown when we read a command line
     *   and then go back and read another without having displaying
     *   anything.
     *   
     *   By default, we'll return a message indicating that nothing
     *   happened.  
     */
    commandResultsEmpty =
        ('Nic očividného se nestalo.<.p>')

    /*
     *   Intra-command report separator.  This is used to separate report
     *   messages within a single command's results.  By default, we show
     *   a paragraph break.  
     */
    intraCommandSeparator = '<.p>'

    /*
     *   separator for "smell" results - we ordinarily show each item's
     *   odor description as a separate paragraph 
     */
    smellDescSeparator()
    {
        "<.p>";
    }

    /*
     *   separator for "listen" results 
     */
    soundDescSeparator()
    {
        "<.p>";
    }

    /* a command was issued to a non-actor */
    cannotTalkTo(targetActor, issuingActor)
    {
        "\^<<targetActor.name>> není něco, s čím {[bys]|[jsi]} mohl{a} mluvit. ";
    }

    /* greeting actor while actor is already talking to us */
    alreadyTalkingTo(actor, greeter)
    {
        "\^Pozornost <<actor.nameKohoCeho>> {|[jsi]} už {m[áš]|měl[a]}. ";
    }

    /* no topics to suggest when we're not talking to anyone */
    noTopicsNotTalking = "<.parser>{Momentálně|Tehdy} {|[jsi]} s nikým nemluv{[íš]|il[a]}.<./parser> "

    /*
     *   Show a note about the OOPS command.  This is, by default, added
     *   to the "I don't know that word" error the first time that error
     *   occurs.  
     */
    oopsNote()
    {
        oopsTip.showTip();
    }

    /* can't use OOPS right now */
    oopsOutOfContext = "<.parser>OPRAVA může být použita jen v ten okamžik,
        kdy se příběh zeptá na neznámé slovo.<./parser> "

    /* OOPS in context, but without the word to correct */
    oopsMissingWord = "<.parser>OPRAVA překlepů se používá tak, že se za slovo
        OPRAVA napíše správné slovo. Např. <q>OPRAVA KNIHA</q>.<./parser> "

    /* acknowledge setting VERBOSE mode (true) or TERSE mode (nil) */
    acknowledgeVerboseMode(verbose)
    {
        if (verbose)
            "<.parser>Je vybrán UPOVÍDANÝ režim.<./parser> ";
        else
            "<.parser>Je vybrán STRUČNÝ režim.<./parser> ";
    }

    /* show the current VERBOSE setting, in short form */
    shortVerboseStatus(stat) { "<<stat ? 'UPOVÍDANÝ' : 'STRUČNÝ'>> režim"; }

    /* show the current score notify status */
    showNotifyStatus(stat)
    {
        "<.parser>Upozorňování na změnu skóre je <<stat ? 'zapnuté'
            : 'vypnuté'>>.<./parser> ";
    }

    /* show the current score notify status, in short form */
    shortNotifyStatus(stat) { "UPOZORŇOVÁNÍ <<stat ? 'ZAPNUTÉ' : 'VYPNUTÉ'>>"; }

    /* acknowledge a change in the score notification status */
    acknowledgeNotifyStatus(stat)
    {
        "<.parser>Upozorňování na změnu skóre je teď <<stat ? 'zapnuté'
            : 'vypnuté'>>.<./parser> ";
    }

    /*
     *   Announce the current object of a set of multiple objects on which
     *   we're performing an action.  This is used to tell the player
     *   which object we're acting upon when we're iterating through a set
     *   of objects specified in a command targeting multiple objects.  
     */
    announceMultiActionObject(obj, whichObj, action)
    {
        /* 
         *   get the display name - we only need to differentiate this
         *   object from the other objects in the iteration 
         */
        local nm = obj.getAnnouncementDistinguisher(
            action.getResolvedObjList(whichObj)).name(obj);

        /* build the announcement */
        return '<./p0>\n<.announceObj>' + nm + ':<./announceObj> <.p0>';
    }

    /*
     *   Announce a singleton object that we selected from a set of
     *   ambiguous objects.  This is used when we disambiguate a command
     *   and choose an object over other objects that are also logical but
     *   are less likely.  In such cases, it's courteous to tell the
     *   player what we chose, because it's possible that the user meant
     *   one of the other logical objects - announcing this type of choice
     *   helps reduce confusion by making it immediately plain to the
     *   player when we make a choice other than what they were thinking.  
     */
    announceAmbigActionObject(obj, whichObj, action)
    {
        /*
         *   get the display name - distinguish the object from everything
         *   else in scope, since we chose from a set of ambiguous options 
         */
        local nm = obj.getAnnouncementDistinguisher(gActor.scopeList())
            .name(obj);

        /* announce the object in "assume" style, ending with a newline */
        return '<.assume>' + nm + '<./assume>\n';
    }

    /*
     *   Announce a singleton object we selected as a default for a
     *   missing noun phrase.
     *   
     *   'resolvedAllObjects' indicates where we are in the command
     *   processing: this is true if we've already resolved all of the
     *   other objects in the command, nil if not.  We use this
     *   information to get the phrasing right according to the situation.
     */
    announceDefaultObject(obj, whichObj, action, resolvedAllObjects)
    {
        /*
         *   put the action's default-object message in "assume" style,
         *   and start a new line after it 
         */
        return '<.assume>'
            + action.announceDefaultObject(obj, whichObj, resolvedAllObjects)
            + '<./assume>\n';
    }

    /* 'again' used with no prior command */
    noCommandForAgain()
    {
        "<.parser>Není tu nic, co by se dalo opakovat.<./parser> ";
    }

    /* 'again' cannot be directed to a different actor */
    againCannotChangeActor()
    {
        "<.parser>Abys zopakoval příkaz typu <q>želvo, jdi severně,</q>
        tak napiš jen <q>znovu,</q> a ne <q>želvo, znovu.</q><./parser> ";
    }

    /* 'again': can no longer talk to target actor */
    againCannotTalkToTarget(issuer, target)
    {
        "\^<<issuer.name>> nem{ůž[eš]|ohl[a]} zopakovat příkaz. ";
    }

    /* the last command cannot be repeated in the present context */
    againNotPossible(issuer)
    {
        "Teď tento příkaz nemůže být zopakován. ";
    }

    /* system actions cannot be directed to non-player characters */
    systemActionToNPC()
    {
        "<.parser>Tento příkaz nemůže být směrován jiné postavě příběhu.
        <./parser> ";
    }

    /* confirm that we really want to quit */
    confirmQuit()
    {
        "Opravdu chceš skončit?\ (<<aHref('a', 'A', 'Potvrdit ukončení')
        >> znamená souhlas) >\ ";
    }


    /*
     *   QUIT message.  We display this to acknowledge an explicit player
     *   command to quit the game.  This is the last message the game
     *   displays on the way out; there is no need to offer any options at
     *   this point, because the player has decided to exit the game.
     *   
     *   By default, we show nothing; games can override this to display an
     *   acknowledgment if desired.  Note that this isn't a general
     *   end-of-game 'goodbye' message; the library only shows this to
     *   acknowledge an explicit QUIT command from the player.  
     */
    okayQuitting() { }

    /*
     *   "not terminating" confirmation - this is displayed when the
     *   player doesn't acknowledge a 'quit' command with an affirmative
     *   response to our confirmation question 
     */
    notTerminating()
    {
        "<.parser>Příběh pokračuje.<./parser> ";
    }

    /* confirm that they really want to restart */
    confirmRestart()
    {
        "Opravdu chceš začít od začátku?\ (<<aHref('A', 'A',
        'Potvrdit restart')>> znamená souhlas) >\ ";
    }

    /* "not restarting" confirmation */
    notRestarting() { "<.parser>Příběh pokračuje.<./parser> "; }

    /*
     *   Show a game-finishing message - we use the conventional "*** You
     *   have won! ***" format that text games have been using since the
     *   dawn of time. 
     */
    showFinishMsg(msg) { "<.p>*** <<msg>>\ ***<.p>"; }

    /* standard game-ending messages for the common outcomes */
    finishDeathMsg = 'ZEMŘEL{A pc} JSI'
    finishVictoryMsg = 'VYHRÁL{A} JSI'
    finishFailureMsg = 'NEUSPĚL JSI'
    finishGameOverMsg = 'KONEC HRY'

    /*
     *   Get the save-game file prompt.  Note that this must return a
     *   single-quoted string value, not display a value itself, because
     *   this prompt is passed to inputFile(). 
     */
    getSavePrompt = 'Uložit hru do souboru'

    /* get the restore-game prompt */
    getRestorePrompt = 'Obnovit hru ze souboru'

    /* successfully saved */
    saveOkay() { "<.parser>Uloženo.<./parser> "; }

    /* save canceled */
    saveCanceled() { "<.parser>Přerušeno.<./parser> "; }

    /* saved failed due to a file write or similar error */
    saveFailed(exc)
    {
        "<.parser>Nepovedlo se. Možná dochází místo na disku a nebo nemáte
        dostatečná práva k uložení souboru.<./parser> ";
    }

    /* save failed due to storage server request error */
    saveFailedOnServer(exc)
    {
        "<.parser>Failed, because of a problem accessing the storage server:
        <<makeSentence(exc.errMsg)>><./parser>";
    }

    /* 
     *   make an error message into a sentence, by capitalizing the first
     *   letter and adding a period at the end if it doesn't already have
     *   one 
     */
    makeSentence(msg)
    {
        return rexReplace(
            ['^<space>*[a-z]', '(?<=[^.?! ])<space>*$'], msg,
            [{m: m.toUpper()}, '.']);
    }

    /* note that we're restoring at startup via a saved-position launch */
    noteMainRestore() { "<.parser>Nahrávám uloženou hru...<./parser>\n"; }

    /* successfully restored */
    restoreOkay() { "<.parser>Hra obnovena.<./parser> "; }

    /* restore canceled */
    restoreCanceled() { "<.parser>Přerušeno.<./parser> "; }

    /* restore failed due to storage server request error */
    restoreFailedOnServer(exc)
    {
        "<.parser>Failed, because of a problem accessing the storage server:
        <<makeSentence(exc.errMsg)>><./parser>";
    }

    /* restore failed because the file was not a valid saved game file */
    restoreInvalidFile()
    {
        "<.parser>Chyba: To není platný soubor s uloženou pozicí.<./parser> ";
    }

    /* restore failed because the file was corrupted */
    restoreCorruptedFile()
    {
        "<.parser>Selhalo: Soubor s uloženou pozicí vypadá poškozeně. To se
        může stát, pokud byl modifikován jinou aplikací, pokud byl překopírován
        mezi počítači v nebinárním režimu nebo pokud je médium
        poškozené.<./parser> ";
    }

    /* restore failed because the file was for the wrong game or version */
    restoreInvalidMatch()
    {
        "<.parser>Selhalo: Soubor nebyl uložen tímto příběhem (nebo byl uložen
        nekompatibliní verzí příběhu).<./parser> ";
    }

    /* restore failed for some reason other than those distinguished above */
    restoreFailed(exc)
    {
        "<.parser>Selhalo: Nepovedlo se obnovit pozici ze souboru.<./parser> ";
    }

    /* error showing the input file dialog (or character-mode equivalent) */
    filePromptFailed()
    {
        "<.parser>Nastala systémová chyba při dotazování se na název souboru.
        Počítači se může nedostávat operační paměti nebo má nějaký problém s
        konfugurací.<./parser> ";
    }

    /* error showing the input file dialog, with a system error message */
    filePromptFailedMsg(msg)
    {
        "<.parser>Failed: <<makeSentence(msg)>><./parser> ";
    }

    /* Web UI inputFile error: uploaded file is too large */
    webUploadTooBig = 'The file you selected is too large to upload.'

    /* PAUSE prompt */
    pausePrompt()
    {
        "<.parser>Příběh je pozastaven. Zmáčkni prosím mezerník, jakmile
        budeš připraven pokračovat nebo zmáčkni klávesu &lsquo;S&rsquo; k
        uložení současné pozice.<./parser><.p>";
    }

    /* saving from within a pause */
    pauseSaving()
    {
        "<.parser>Ukládám příběh...<./parser><.p>";
    }

    /* PAUSE ended */
    pauseEnded()
    {
        "<.parser>Obnovuji příběh.<./parser> ";
    }

    /* acknowledge starting an input script */
    inputScriptOkay(fname)
    {
        "<.parser>Čtu příkazy z <q><<
        File.getRootName(fname).htmlify()>></q>...<./parser>\n ";
    }
    
    /* error opening input script */
    inputScriptFailed = "<.parser>Selhalo; nepodařilo se otevřít soubor se
        vstupním skriptem.<./parser> "

    /* exception opening input script */
    inputScriptFailedException(exc)
    {
        "<.parser>Selhalo; <<exc.displayException>><./parser> ";
    }

    /* get the scripting inputFile prompt message */
    getScriptingPrompt = 'Zadejte prosím název souboru pro zápis průběhu hry'

    /* acknowledge scripting on */
    scriptingOkay()
    {
        "<.parser>Nyní bude průběh hry zapisován do souboru. Zadejte <<aHref(
            'vypnout zápis', 'VYPNOUT ZÁPIS', 'Vypnout zápis průběhu hry')>>
            k ukončení zaznamenávání.<./parser> ";
    }

    scriptingOkayWebTemp()
    {
        "<.parser>Nyní bude průběh hry zapisován. Zadejte <<aHref(
            'vypnout zápis', 'VYPNOUT ZÁPIS', 'Vypnout zápis průběhu hry')>>
            k ukončení zaznamenávání a stažení zaznamenaného zápisu
            hry.<./parser> ";
    }

    /* scripting failed */
    scriptingFailed = "<.parser>Selhalo: nastala chyba při otevírání souboru
    se skriptem.<./parser> "

    /* scripting failed with an exception */
    scriptingFailedException(exc)
    {
        "<.parser>Selhalo; <<exc.displayException>><./parser>";
    }

    /* acknowledge cancellation of script file dialog */
    scriptingCanceled = "<.parser>Zastaveno.<./parser> "

    /* acknowledge scripting off */
    scriptOffOkay = "<.parser>Zápis průběhu hry skončil.<./parser> "

    /* SCRIPT OFF ignored because we're not in a script file */
    scriptOffIgnored = "<.parser>Průběh hry nebyl zaznamenáván.<./parser> "

    /* get the RECORD prompt */
    getRecordingPrompt = 'Zadejte prosím jméno souboru pro záznam příkazů'

    /* acknowledge recording on */
    recordingOkay = "<.parser>Příkazy budou zaznamenávány. Zadejte <<aHref(
        'vypnout záznam', 'VYPNOUT ZÁZNAM', 'Vypnout zaznamenávání příkazů')>>
        k přerušení záznamu příkazů.<.parser> "

    /* recording failed */
    recordingFailed = "<.parser>Failed; an error occurred opening
        the command recording file.<./parser> "

    /* recording failed with exception */
    recordingFailedException(exc)
    {
        "<.parser>Selhalo; <<exc.displayException()>><./parser> ";
    }

    /* acknowledge cancellation */
    recordingCanceled = "<.parser>Zastaveno.<./parser> "

    /* recording turned off */
    recordOffOkay = "<.parser>Záznam příkazů skončil.<./parser> "

    /* RECORD OFF ignored because we're not recording commands */
    recordOffIgnored = "<.parser>Žádný záznam příkazů nebyl
        pořizován.<./parser> "

    /* REPLAY prompt */
    getReplayPrompt = 'Zadejte prosím název souboru s příkazy k přehrání'

    /* REPLAY file selection canceled */
    replayCanceled = "<.parser>Zrušeno.<./parser> "

    /* undo command succeeded */
    undoOkay(actor, cmd)
    {
        "<.parser>Odvolává se jeden tah: <q>";

        /* show the target actor prefix, if an actor was specified */
        if (actor != nil)
            "<<actor>>, ";

        /* show the command */
        "<<cmd>></q>.<./parser><.p>";
    }

    /* undo command failed */
    undoFailed()
    {
        "<.parser>Další kroky k odvolání tahu nejsou k dispozici.<./parser> ";
    }

    /* comment accepted, with or without transcript recording in effect */
    noteWithScript = "<.parser>Komentář zaznamenán.<./parser> "
    noteWithoutScript = "<.parser>Komentář <b>nebyl</b> zaznamenán.<./parser> "

    /* on the first comment without transcript recording, warn about it */
    noteWithoutScriptWarning = "<.parser>Komentář <b>nebyl</b> zaznamenán.
        Zadejte <<aHref('skript', 'SKRIPT', 'Začít zaznamenávat transkript')
          >>, pokud chcete začít zaznamenáva transkript.<./parser> "

    /* invalid finishGame response */
    invalidFinishOption(resp)
    {
        "\bTo není jedna z možností. ";
    }

    /* acknowledge new "exits on/off" status */
    exitsOnOffOkay(stat, look)
    {
        if (stat && look)
            "<.parser>Nyní bude seznam směrů ukazován ve stavové řádce i jako
            součást popisu každé místnosti.<./parser> ";
        else if (!stat && !look)
            "<.parser>Seznam směrů už nebude vypisován ani ve stavové řádce,
            ani jako součást popisu místností.<./parser> ";
        else
            "<.parser>Seznam směrů <<stat ? 'bude' : 'nebude'>> ukazován ve
            stavové řádce, ale <<look ? 'bude' : 'nebude'>> zahrnut do popisů
            místností.<./parser> ";
    }

    /* explain how to turn exit display on and off */
    explainExitsOnOff()
    {
        exitsTip.showTip();
    }

    /* describe the current EXITS settings */
    currentExitsSettings(statusLine, roomDesc)
    {
        "SMĚRY ";
        if (statusLine && roomDesc)
            "ZAPNUTY";
        else if (statusLine)
            "V ZÁHLAVÍ";
        else if (roomDesc)
            "V POPISU";
        else
            "VYPNUTY";
    }

    /* acknowledge HINTS OFF */
    hintsDisabled = '<.parser>Rady jsou vypnuty.<./parser> '

    /* rebuff a request for hints when they've been previously disabled */
    sorryHintsDisabled = '<.parser>Promiň, ale rady jsou na tvé přání v této
        relaci zablokovány. Pokud jsi změnil{a} názor, budeš si muset uložit
        pozici, vyskočit z interpretru a pak ho znovu spustit a začít tak nové
        sezení.<./parser> '

    /* this game has no hints */
    hintsNotPresent = '<.parser>Promiň, ale tento příběh nemá žádné vestavěné
        rady.<./parser> '

    /* there are currently no hints available (but there might be later) */
    currentlyNoHints = '<.parser>Promiň, ale žádné rady nejsou v současné době
        k dispozici. Zkus to později.<./parser> '

    /* show the hint system warning */
    showHintWarning =
       "<.notification>Upozornění: Někteří lidé nemají vestavěné rady v oblibě,
       protože pokušení říci si předčasně o pomoc může být velké, když jsou rady
       tak snadno dosažitelné. Pokud máš pocit, že pokušení nevydržíš, můžeš
       rady vypnout až do konce sezení zadáním <<aHref('vypnout rady', 'VYPNOUT
       RADY')>>. Pokud i přesto chceš získat radu, napiš znovu příkaz
       <<aHref('rady', 'RADY')>>.<./notification> "

    /* done with hints */
    hintsDone = '<.parser>Hotovo.<./parser> '

    /* optional command is not supported in this game */
    commandNotPresent = "<.parser>Tento příkaz není v tomto příběhu potřeba.
        <./parser> "

    /* this game doesn't use scoring */
    scoreNotPresent = "<.parser>Tento příběh nepočítá skóre.<./parser> "

    /* mention the FULL SCORE command */
    mentionFullScore()
    {
        fullScoreTip.showTip();
    }

    /* SAVE DEFAULTS successful */
    savedDefaults()
    {
        "<.parser>Současné nastavení bylo uloženo jako výchozí nastavení pro
            další hry. Uložená nastavení jsou: ";

        /* show all of the settings */
        settingsUI.showAll();

        ". Většina nových her automaticky převezme tato nastavení, jakmile
        začneš (nebo RESTARTuješ) hry, ale některé starší hry ne.<./parser> ";
    }

    /* RESTORE DEFAULTS successful */
    restoredDefaults()
    {
        "<.parser>Uložené výchozí nastavení vstoupilo v platnost. Nová
            nastavení jsou: ";

        /* show all of the settings */
        settingsUI.showAll();

        ".<./parser> ";
    }

    /* show a separator for the settingsUI.showAll() list */
    settingsItemSeparator = ", "

    /* SAVE/RESTORE DEFAULTS not supported (old interpreter version) */
    defaultsFileNotSupported = "<.parser>Verze TADS interpretru, kterou
        používáš bohužel nepodporuje ukládání a načítání výchozího nastavení.
        Abys mohl tuto vlastnost používat, budeš muset nainstalovat novější
        verzi.<./parser> "

    /* RESTORE DEFAULTS file open/read error */
    defaultsFileReadError(exc)
    {
        "<.parser>An error occurred reading the default settings
        file.  The global defaults can't be restored.<./parser> ";
    }

    /* SAVE DEFAULTS file creation error */
    defaultsFileWriteError = "<.parser>Při ukládání výchozího nastavení došlo
        k chybě. Nastavení nebylo uloženo. Možná ti dochází místo na disku nebo
        nemáš dostatečné oprávnění k zápisu souboru.<./parser> "

    /*
     *   Command key list for the menu system.  This uses the format
     *   defined for MenuItem.keyList in the menu system.  Keys must be
     *   given as lower-case in order to match input, since the menu
     *   system converts input keys to lower case before matching keys to
     *   this list.  
     *   
     *   Note that the first item in each list is what will be given in
     *   the navigation menu, which is why the fifth list contains 'ENTER'
     *   as its first item, even though this will never match a key press.
     */
    menuKeyList = [
                   ['o'],
                   ['p', '[left]', '[bksp]', '[esc]'],
                   ['n', '[up]'],
                   ['d', '[down]'],
                   ['ENTER', '\n', '[right]', ' ']
                  ]

    /* link title for 'previous menu' navigation link */
    prevMenuLink = '<font size=-1>Předchozí</font>'

    /* link title for 'next topic' navigation link in topic lists */
    nextMenuTopicLink = '<font size=-1>Následující</font>'

    /*
     *   main prompt text for text-mode menus - this is displayed each
     *   time we ask for a keystroke to navigate a menu in text-only mode 
     */
    textMenuMainPrompt(keylist)
    {
        "\bVyber číslo tématu nebo zmáčkni &lsquo;<<keylist[M_PREV][1]>>&rsquo;
        pro návrat do předchozí nabídky nebo &lsquo;<<keylist[M_QUIT][1]
        >>&rsquo; k opuštění nabídky:\ ";
    }

    /* prompt text for topic lists in text-mode menus */
    textMenuTopicPrompt()
    {
        "\bPro zobrazení další řádky zmáčkni mezerník, pro předchozí nabídku
        &lsquo;<b>P</b>&rsquo;, nebo pro opuštění nabidky
        &lsquo;<b>Q</b>&rsquo;.\b";
    }

    /*
     *   Position indicator for topic list items - this is displayed after
     *   a topic list item to show the current item number and the total
     *   number of items in the list, to give the user an idea of where
     *   they are in the overall list.  
     */
    menuTopicProgress(cur, tot) { " [<<cur>>/<<tot>>]"; }

    /*
     *   Message to display at the end of a topic list.  We'll display
     *   this after we've displayed all available items from a
     *   MenuTopicItem's list of items, to let the user know that there
     *   are no more items available.  
     */
    menuTopicListEnd = '[konec]'

    /*
     *   Message to display at the end of a "long topic" in the menu
     *   system.  We'll display this at the end of the long topic's
     *   contents.  
     */
    menuLongTopicEnd = '[konec]'

    /*
     *   instructions text for banner-mode menus - this is displayed in
     *   the instructions bar at the top of the screen, above the menu
     *   banner area 
     */
    menuInstructions(keylist, prevLink)
    {
        "<tab align=right ><b>\^<<keylist[M_QUIT][1]>></b>=Odejít <b>\^<<
        keylist[M_PREV][1]>></b>=Předchozí nabídka<br>
        <<prevLink != nil ? aHrefAlt('předchozí', prevLink, '') : ''>>
        <tab align=right ><b>\^<<keylist[M_UP][1]>></b>=Nahoru <b>\^<<
        keylist[M_DOWN][1]>></b>=Dolů <b>\^<<
        keylist[M_SEL][1]>></b>=Vybrat<br>";
    }

    /* show a 'next chapter' link */
    menuNextChapter(keylist, title, hrefNext, hrefUp)
    {
        "Další: <<aHref(hrefNext, title)>>;
        <b>\^<<keylist[M_PREV][1]>></b>=<<aHref(hrefUp, 'Nabídka')>>";
    }

    /*
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObject(obj)
    {
        "Na {kohoco obj} {nedosáhn[eš]|[jsi] nedosáhl[a]}. ";
    }

    /*
     *   cannot reach an object, because the object is inside the given
     *   container 
     */
    cannotReachContents(obj, loc)
    {
        gMessageParams(obj, loc);
        return 'Skrz {kohoco loc} {|[jsi]} na {kohoco obj} '
            + 'nedosáh{n[eš]|l[a]}. ';
    }

    /* cannot reach an object because it's outisde the given container */
    cannotReachOutside(obj, loc)
    {
        gMessageParams(obj, loc);
        return 'Skrz {kohoco loc} {|[jsi]} na {kohoco obj} '
            + 'nedosáh{n[eš]|l[a]}. ';
    }

    /* sound is coming from inside/outside a container */
    soundIsFromWithin(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} {z/ze loc}
            {kohočeho loc}. ";
    }
    soundIsFromWithout(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} do {kohočeho loc}
            z venku. ";
    }

    /* odor is coming from inside/outside a container */
    smellIsFromWithin(obj, loc)
    {
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} {z/ze loc}
            {kohočeho loc}. ";
    }
    smellIsFromWithout(obj, loc)
    {
        gMessageParams(obj);
        "Vypad{á|alo} to, že {kdoco obj} přicház{í|el[a obj]} do {kohočeho loc}
            z venku. ";
    }

    /* default description of the player character */
    pcDesc(actor)
    {
        "\^{kdoco/postava} vypad{[áš]|al[a] [jsi]} stejně, jako obvykle. ";
    }

    /*
     *   Show a status line addendum for the actor posture, without
     *   mentioning the actor's location.  We won't mention standing, since
     *   this is the default posture.
     *
     *   Zajímavé je, že v následujících dvou funkcích šablona {jsi} funguje
     *   na správného actora, i když jsem obecně musel zavést posera kvůli
     *   parametrům v posture.participle. Přesto ale posera do {jsi} přidám.
     *
     *   TODO: V attachment se tohle volá hned na začátku při uvodním
     *   rozhlédnutí a v tu dobu je gAction nenastavené, takže skončí
     *   stacktracem. Jak to korektně vyřešit?
     */
    roomActorStatus(actor)
    {
        local poser = actor;
        if(gAction != nil) gMessageParams(poser);

        /* mention any posture other than standing */
        if (actor.posture != standing)
            " (<<actor.posture.participle>>{| [jsi poser]})";
    }

    /* show a status line addendum: standing in/on something */
    actorInRoomStatus(actor, room)
    {
        local poser = actor;
        if(gAction != nil) gMessageParams(poser); // TODO: viz roomActorStatus kvuli ifu

        " (<<actor.posture.participle>> {|[jsi poser]} <<room.objInName>>)";
    }

    /* generic short description of a dark room */
    roomDarkName = 'Ve tmě'

    /* generic long description of a dark room */
    roomDarkDesc = "{Je|Byla} tu naprostá tma. "

    /*
     *   mention that an actor is here, without mentioning the enclosing
     *   room, as part of a room description 
     *
     *   "Ty tu stojíš." (Součást vylistování místnosti, když ho provádí
     *   NPC a vidí přitom PC -- ">Sally, rozhlédni se".)
     *
     *   Tady musím objekt poser na rozdíl od předchozích funkcí uvést, jinak
     *   by se šablony vázaly k PC.
     */
    roomActorHereDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        "{Ty poser} {|[jsi poser]} tu <<actor.posture.participle>>. ";
    }

    /*
     *   mention that an actor is visible at a distance or remotely,
     *   without mentioning the enclosing room, as part of a room
     *   description 
     */
    roomActorThereDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>> poblíž. ";
    }

    /*
     *   Mention that an actor is in a given local room, as part of a room
     *   description.  This is used as a default "special description" for
     *   an actor.  
     *
     *   "Kocour leží na podlaze pod palandou."
     */
    actorInRoom(actor, cont)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>>
        <<cont.objInName>>. ";
    }

    /*
     *   Describe an actor as standing/sitting/lying on something, as part
     *   of the actor's EXAMINE description.  This is additional
     *   information added to the actor's description, so we refer to the
     *   actor with a pronoun ("He's standing here").  
     *
     *   ">sally, rozhlédni se" -> "... Ty tu stojíš."
     *   ">rozhlédni se" -> "... Sally tu stojí."
     */
    actorInRoomPosture(actor, room)
    {
        local poser = actor;
        gMessageParams(poser);

        "\^<<actor.name>> <<actor.posture.participle>>
        <<room.objInName>>. ";
    }

    /*
     *   Describe an actor's posture, as part of an actor's "examine"
     *   description.  If the actor is standing, don't bother mentioning
     *   anything, as standing is the trivial default condition.  
     */
    roomActorPostureDesc(actor)
    {
        local poser = actor;
        gMessageParams(poser);

        if (actor.posture != standing)
            "\^<<actor.name>> <<actor.posture.participle>>. ";
    }

    /*
     *   mention that the given actor is visible, at a distance or
     *   remotely, in the given location; this is used in room
     *   descriptions when an NPC is visible in a remote or distant
     *   location 
     *
     *   Listuje actora, který je v jiné lokaci propojené sense konektorem.
     *   Když napíšu v posture.participle parametr vztahující se k actor, tak
     *   je tím ale implicitně míněn PC vykonávající příkaz a ne proměnná actor
     *   předaná do fukce definující viděnou NPC. Musím si tedy v definici
     *   posture.participle specifikovat parametr jiného názvu a tady si ho
     *   naplnit actorem.
     */
    actorInRemoteRoom(actor, room, pov)
    {
        local poser = actor;
        gMessageParams(poser);

        /* say that the actor is in the room, using its remote in-name */
        "Vid{[íš]|ěl[a] [jsi]} odsud <<room.destName>>. \^<<actor.name>> tam
            <<actor.posture.participle>>. ";
    }

    /*
     *   mention that the given actor is visible, at a distance or
     *   remotely, in the given nested room within the given outer
     *   location; this is used in room descriptions 
     */
    actorInRemoteNestedRoom(actor, inner, outer, pov)
    {
        local poser = actor;
        gMessageParams(poser);

        /*
         *   say that the actor is in the nested room, in the current
         *   posture, and add then add that we're in the outer room as
         *   well 
         */
        "\^<<actor.name>> {je poser} <<outer.inRoomName(pov)>>,
        <<actor.posture.participle>> <<inner.objInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors are in the local room in a nominal
     *   container that we want to mention: "Bob and Bill are sitting on
     *   the couch."
     *
     *   TODO: Jak s parametry posture.participle vs. mnoho postav?
     */
    actorInGroupPrefix(posture, cont, lst) { "\^"; }
    actorInGroupSuffix(posture, cont, lst)
    {
        " <<posture.participle>> <<cont.objInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors are inside a nested room that's inside
     *   a remote location: "Bob and Bill are in the courtyard, sitting on
     *   the bench." 
     */
    actorInRemoteGroupPrefix(pov, posture, cont, remote, lst) { "\^"; }
    actorInRemoteGroupSuffix(pov, posture, cont, remote, lst)
    {
        " <<remote.inRoomName(pov)>>, <<posture.participle>>
        <<cont.objInName>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors' nominal container cannot be seen or is
     *   not to be stated: "Bob and Bill are standing here."
     *   
     *   Note that we don't always want to state the nominal container,
     *   even when it's visible.  For example, when actors are standing on
     *   the floor, we don't bother saying that they're on the floor, as
     *   that's stating the obvious.  The container will decide whether or
     *   not it wants to be included in the message; containers that don't
     *   want to be mentioned will use this form of the message.  
     */
    actorHereGroupPrefix(posture, lst) { "\^"; }
    actorHereGroupSuffix(posture, lst)
    {
        " <<posture.participle>>. ";
    }

    /*
     *   Prefix/suffix messages for listing actors in a room description,
     *   for cases when the actors' immediate container cannot be seen or
     *   is not to be stated, and the actors are in a remote location:
     *   "Bob and Bill are in the courtyard."  
     */
    actorThereGroupPrefix(pov, posture, remote, lst) { "\^"; }
    actorThereGroupSuffix(pov, posture, remote, lst)
    {
        " <<posture.participle>> <<remote.inRoomName(pov)>>. ";
    }

    /*
     *   a traveler is arriving, but not from a compass direction
     *
     *   "Sally přichází na zahradu / do pramice."
     */
    sayArriving(traveler)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerLocName>>. ";
    }

    /*
     *   a traveler is departing, but not in a compass direction
     *
     *   "Sally odchází pryč." Šablona {šel} funguje ke správné postavě, asi
     *   u travelera je to obecně ok.
     *
     *   TODO: Chce to název cíle, viz i sayDepartingLocally. Původně se
     *   vkládalo "leaving travelerLocName". Pritom travelerRemoteLocName neni
     *   nazvem cile.
     */
    sayDeparting(traveler)
    {
        local nm = traveler.travelerLocName();
        nm = ''; // dokud nebudu mit nazev cile, zabijim spatny nazev
        "\^<<traveler.travelerName(nil)>> {odchází|ode[šel]}<<nm != '' ? ' '
            + nm : ''>>. ";
    }

    /*
     *   a traveler is arriving locally (staying within view throughout the
     *   travel, and coming closer to the PC) 
     */
    sayArrivingLocally(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerLocName>>. ";
    }

    /*
     *   a traveler is departing locally (staying within view throughout
     *   the travel, and moving further away from the PC) 
     */
    sayDepartingLocally(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> {odchází|ode[šel]}. ";
    }

    /*
     *   a traveler is traveling remotely (staying within view through the
     *   travel, and moving from one remote top-level location to another) 
     */
    sayTravelingRemotely(traveler, dest)
    {
        "\^<<traveler.travelerName(true)>> {jde|[šel]}
            <<traveler.travelerLocName>>. ";
    }

    /* a traveler is arriving from a compass direction */
    sayArrivingDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerRemoteLocName>> <<dirName>>. ";
    }

    /* a traveler is leaving in a given compass direction */
    /* TODO: Není tu potřeba obrátit logiku místností oproti originálu? */
    sayDepartingDir(traveler, dirName)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> od{chází|[šel]}
        <<dirName>><<nm != '' ? ' na ' + nm : ''>>. ";
    }
    
    /* a traveler is arriving from a shipboard direction */
    sayArrivingShipDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> při{chází|[šel]}
        <<traveler.travelerRemoteLocName>> <<dirName>>. ";
    }

    /* a traveler is leaving in a given shipboard direction */
    sayDepartingShipDir(traveler, dirName)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> od{chází|[šel]}
        <<dirName>><<nm != '' ? ' na ' + nm : ''>>. ";
    }

    /* a traveler is going aft */
    sayDepartingAft(traveler)
    {
        local nm = traveler.travelerRemoteLocName;
        
        "\^<<traveler.travelerName(nil)>> od{chází|[šel]}
        na záď<<nm != '' ? ' z ' + nm : ''>>. ";
    }

    /* a traveler is going fore */
    sayDepartingFore(traveler)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(nil)>> od{chází|[šel]}
        na příď<<nm != '' ? ' z ' + nm : ''>>. ";
    }

    /* a shipboard direction was attempted while not onboard a ship */
    notOnboardShip = "Tenhle směr tu nedáv{á|al} smysl. "

    /* a traveler is leaving via a passage */
    sayDepartingThroughPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(nil)>> {vycház[íš]|vy[šel]}
        <<traveler.travelerRemoteLocName>> <<passage.nameKymCim>>. ";
    }

    /* a traveler is arriving via a passage */
    sayArrivingThroughPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(true)>> {vcház[íš]|ve[šel]}
        <<traveler.travelerRemoteLocName>> <<passage.nameKymCim>>. ";
    }

    /* a traveler is leaving via a path */
    sayDepartingViaPath(traveler, passage)
    {
        "\^<<traveler.travelerName(nil)>> {odcház[íš]|ode[šel]}
        <<traveler.travelerRemoteLocName>> po <<passage.nameKomCem>>. ";
    }

    /* a traveler is arriving via a path */
    sayArrivingViaPath(traveler, passage)
    {
        "\^<<traveler.travelerName(true)>> {přicház[íš]|při[šel]}
        <<traveler.travelerRemoteLocName>> po <<passage.nameKomCem>>. ";
    }

    /* a traveler is leaving up a stairway */
    sayDepartingUpStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> {odcház[íš]|ode[šel]}
        nahoru po <<stairs.nameKomCem>>. ";
    }

    /* a traveler is leaving down a stairway */
    sayDepartingDownStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> {odcház[íš]|ode[šel]}
        dolů po <<stairs.nameKomCem>>. ";
    }

    /* a traveler is arriving by coming up a stairway */
    sayArrivingUpStairs(traveler, stairs)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(true)>> při{cház[íš]|[šel]} po
        <<stairs.nameKomCem>> nahoru<<nm != '' ? ' ' + nm : ''>>. ";
    }

    /* a traveler is arriving by coming down a stairway */
    sayArrivingDownStairs(traveler, stairs)
    {
        local nm = traveler.travelerRemoteLocName;

        "\^<<traveler.travelerName(true)>> při{cház[íš]|[šel]} po
        <<stairs.nameKomCem>> dolů<<nm != '' ? ' ' + nm : ''>>. ";
    }

    /* acompanying another actor on travel */
    sayDepartingWith(traveler, lead)
    {
        gMessageParams(lead);
        "\^<<traveler.travelerName(nil)>> {odchází|ode[šel]} {s/se lead}
            <<lead.nameKymCim>>. ";
    }

    /*
     *   Accompanying a tour guide.  Note the seemingly reversed roles:
     *   the lead actor is the one initiating the travel, and the tour
     *   guide is the accompanying actor.  So, the lead actor is
     *   effectively following the accompanying actor.  It seems
     *   backwards, but really it's not: the tour guide merely shows the
     *   lead actor where to go, but it's up to the lead actor to actually
     *   initiate the travel.  
     */
    sayDepartingWithGuide(guide, lead)
    {
        gMessageParams(guide, lead);
        
        "Nech{[áš lead] se|al[a lead] [ses lead]} vést. ";
    }

    /* note that a door is being opened/closed remotely */
    sayOpenDoorRemotely(door, stat)
    {
        "Někdo <<stat ? 'otev' : 'zav'>>{írá|řel} <<door.nameKohoCo>> z druhé
            strany. ";
    }

    /*
     *   open/closed status - these are simply adjectives that can be used
     *   to describe the status of an openable object 
     */
    openMsg(obj)
    {
        gMessageParams(obj);
        return 'otevřen{ý obj}';
    }
    closedMsg(obj)
    {
        gMessageParams(obj);
        return 'zavřen{ý obj}';
    }

    /* object is currently open/closed */
    currentlyOpen = 'Momentálně {je dobj} otevřen{ý}. '
    currentlyClosed = 'Momentálně {je dobj} zavřen{ý}. '

    /* stand-alone independent clause describing current open status */
    openStatusMsg(obj)
    {
        gMessageParams(obj);
        return '{Je obj} ' + obj.openDesc;
    }

    /* locked/unlocked status - adjectives describing lock states */
    lockedMsg(obj)
    {
        gMessageParams(obj);
        return 'zamčen{ý obj}';
    }
    unlockedMsg(obj)
    {
        gMessageParams(obj);
        return 'odemčen{ý obj}';
    }

    /* object is currently locked/unlocked */
    currentlyLocked = 'Momentálně {je dobj} zamčen{ý}. '
    currentlyUnlocked = 'Momentálně {je dobj} odemčen{ý}. '

    /*
     *   on/off status - these are simply adjectives that can be used to
     *   describe the status of a switchable object 
     */
    onMsg(obj)
    {
        gMessageParams(obj);
        return 'zapnut{ý obj}';
    }
    offMsg(obj)
    {
        gMessageParams(obj);
        return 'vypnut{ý obj}';
    }

    /* daemon report for burning out a match */
    matchBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} dohořel{a obj} a plamen {mizí|zmizel} v obláčku dýmu. ";
    }

    /* daemon report for burning out a candle */
    candleBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} už vyhořel{a obj} až na konec a zrovna
            zhas{íná|l[a obj]}. ";
    }

    /* daemon report for burning out a generic fueled light source */
    objBurnedOut(obj)
    {
        gMessageParams(obj);
        "{Kdoco obj} dohořel{a obj} a zrovna {zhasíná|zhasl[a obj]}. ";
    }

    /* 
     *   Standard dialog titles, for the Web UI.  These are shown in the
     *   title bar area of the Web UI dialog used for inputDialog() calls.
     *   These correspond to the InDlgIconXxx icons.  The conventional
     *   interpreters use built-in titles when titles are needed at all,
     *   but in the Web UI we have to generate these ourselves. 
     */
    dlgTitleNone = 'Note'
    dlgTitleWarning = 'Warning'
    dlgTitleInfo = 'Note'
    dlgTitleQuestion = 'Question'
    dlgTitleError = 'Error'

    /*
     *   Standard dialog button labels, for the Web UI.  These are built in
     *   to the conventional interpreters, but in the Web UI we have to
     *   generate these ourselves.  
     */
    dlgButtonOk = 'OK'
    dlgButtonCancel = 'Cancel'
    dlgButtonYes = 'Yes'
    dlgButtonNo = 'No'

    /* web UI alert when a new user has joined a multi-user session */
    webNewUser(name) { "\b[<<name>> has joined the session.]\n"; }

    /*
     *   Warning prompt for inputFile() warnings generated when reading a
     *   script file, for the Web UI.  The interpreter normally displays
     *   these warnings directly, but in Web UI mode, the program is
     *   responsible, so we need localized messages.  
     */
    inputFileScriptWarning(warning, filename)
    {
        /* remove the two-letter error code at the start of the string */
        warning = warning.substr(3);

        /* build the message */
        return warning + ' Do you wish to proceed?';
    }
    inputFileScriptWarningButtons = [
        '&Yes, use this file', '&Choose another file', '&Stop the script']
;

/* ------------------------------------------------------------------------ */
/*
 *   Player Character messages.  These messages are generated when the
 *   player issues a regular command to the player character (i.e.,
 *   without specifying a target actor).  
 */
playerMessages: libMessages
    /* invalid command syntax */
    commandNotUnderstood(actor)
    {
        "<.parser>Příběh tomuto příkazu nerozum{í|ěl}.<./parser> ";
    }

    /* a special topic can't be used right now, because it's inactive */
    specialTopicInactive(actor)
    {
        "<.parser>{Zrovna teď|V tu chvíli [jsi]} tento příkaz {nemůž[eš]|nemohl[a]} použít.<./parser> ";
    }

    /* no match for a noun phrase */
    noMatch(actor, action, txt) { action.noMatch(self, actor, txt); }

    /*
     *   No match message - we can't see a match for the noun phrase.  This
     *   is the default for most verbs. 
     */
    noMatchCannotSee(actor, txt)
        { "Nic takového jako <q><<txt>></q> {|[jsi]} tu nevid{[íš]|ěl[a]}. "; }

    /*
     *   No match message - we're not aware of a match for the noun phrase.
     *   Some sensory actions, such as LISTEN TO and SMELL, use this
     *   variation instead of the normal version; the things these commands
     *   refer to tend to be intangible, so "you can't see that" tends to
     *   be nonsensical. 
     */
    noMatchNotAware(actor, txt)
        { "Ničeho takového jako <<txt>> {|[jsi]} si {[nejsi/není]|nebyl[a]} vědom{a}. "; }

    /* 'all' is not allowed with the attempted action */
    allNotAllowed(actor)
    {
        "<.parser>Slovo <q>vše</q> nelze použít s tímto slovesem.<./parser> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>Nevid{[íš]|ěl[a] [jsi]} tu nic odpovídajícího.<./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>Nic jiného tu {nevid[íš]|[jsi] neviděl[a]}.<./parser> ";
    }

    /* nothing left in a plural phrase after removing 'except' items */
    noMatchForListBut(actor) { noMatchForAllBut(actor); }

    /* no match for a pronoun */
    noMatchForPronoun(actor, typ, pronounWord)
    {
        /* show the message */
        "<.parser>Slovo <q><<pronounWord>></q> {teď zrovna|v tom okamžiku} na nic
        neodkaz{uje|ovalo}.<./parser> ";
    }

    /*
     *   Ask for a missing object - this is called when a command is
     *   completely missing a noun phrase for one of its objects.  
     *
     *   Funkce action.getQuestionInf(which) byla přepsána, aby se nezabývala
     *   předložkama ani zájmeny a dává jen sloveso s případým complementizerem.
     *
     *   "Čím chceš *vyhrabat díru*?"
     *
     *   Ale pak je tu věta "Ven z čeho se chceš dostat?", kde musí být
     *   "se". To je zvratné osobní zájmeno a používá se, pokud je podmět
     *   i předmět totožný, tj. *ty* chceš *sebe* dostat.
     *
     *   Parametr which má hodnotu DirectObject nebo IndirectObject.
     */
    askMissingObject(actor, action, which)
    {
        local pronoun = action.getOtherMessageObjectPronoun(which);
        local reflexive = action.getReflexivePronoun();
        local question = action.getQuestionInf(which);

        reportQuestion('<.parser>\^'

            /* "*Koho* se chceš zeptat?" */
            + action.whatObj(which)

            + ' {|[jsi]}'

            /* "Koho *se* chceš zeptat?" */
            + (reflexive ? ' ' + reflexive : '')

            /*
             *   Dosadíme zájmeno pro ten druhý objekt, než nám chybí.
             *
             *   ">dej minci" "Komu *ji* chceš dát?"
             *
             *   Pokud ale u slovesa je complementizer, zájmeno vynecháme.
             *
             *   ">vykopej díru do zěmě" "Čím chceš vykopat díru?"
             */
            + (pronoun && !question.find(' ') ? ' ' + pronoun : '')

            + ' {[chceš]|chtěl[a]} {kdoco/postava} '

            /* "Čím chceš *vykopat díru*?" */
            + question + '?<./parser> ');
    }

    /*
     *   An object was missing - this is called under essentially the same
     *   circumstances as askMissingObject, but in cases where interactive
     *   resolution is impossible and we simply wish to report the problem
     *   and do not wish to ask for help.
     */
    missingObject(actor, action, which)
    {
        local reflexive = action.getReflexivePronoun();

        "<.parser>Mus{[íš]|el[a] [jsi]} být specifičtější o tom,
            <<action.whatObj(which)>> {|[jsi]}
            << reflexive ? ' ' + reflexive : ''>>
            {chc[eš]|chtěl[a]} {kdoco/postava}
            <<action.getQuestionInf(which)>>.<./parser> ";
    }

    /*
     *   Ask for a missing literal phrase. 
     */
    askMissingLiteral(actor, action, which)
    {
        /* use the standard missing-object message */
        askMissingObject(actor, action, which);
    }

    /*
     *   Show the message for a missing literal phrase.
     */
    missingLiteral(actor, action, which)
    {
        "<.parser>Buď prosím specifičtější, <<action.whatObj(which)>> {chc[eš]|
            [jsi] chtěl[a]} {kdoco/postava} <<action.getQuestionInf(which)>>.
            Např. <<action.getQuestionInf(which)>> <q>něco</q>.<./parser> ";
    }

    /* reflexive pronoun not allowed */
    reflexiveNotAllowed(actor, typ, pronounWord)
    {
        "<.parser>Příběh nerozum{í|ěl}, jak by mohl tímto způsobem použít slovo
            <q><<pronounWord>></q>.<./parser> ";
    }

    /*
     *   a reflexive pronoun disagrees in gender, number, or something
     *   else with its referent 
     */
    wrongReflexive(actor, typ, pronounWord)
    {
        "<.parser>Příběh nerozumí, na co se odkazuje slovo
            <q><<pronounWord>></q>.<./parser> ";
    }

    /* no match for a possessive phrase */
    noMatchForPossessive(actor, owner, txt)
    {
	gMessageParams(owner);
        "<.parser>\^<<owner.name>> nevypad{[áš owner]|al[a owner]}, že by něco takového
            měl{a}.<./parser> ";
    }

    /*
     *   no match for a plural possessive phrase
     *
     *   TODO: V minulém čase i/y
     */
    noMatchForPluralPossessive(actor, txt)
    {
        "<.parser>\^Nevypada<<tSel('jí', 'li')>>, že by něco takového měli.
            <./parser> ";
    }

    /* no match for a containment phrase */
    noMatchForLocation(actor, loc, txt)
    {
        "<.parser>\^<<actor.name>> {nevidí|neviděl[a]}
        <<loc.childInName(txt)>>.<./parser> ";
    }

    /*
     *   nothing in a container whose contents are specifically requested
     */
    nothingInLocation(actor, loc)
    {
        "<.parser>\^<<actor.namePostava>> {nevid[íš]|neviděl[a]}
        <<loc.childInName('nic neobvyklého')>>.<./parser> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /*
         *   show the message, leaving the <.parser> tag mode open - we
         *   always show another disambiguation prompt after this message,
         *   so we'll let the prompt close the <.parser> mode 
         */
        "<.parser>To nebyla jedna z nabízených možností. ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "<.parser>Vypadá to, že jsi vynechal pár slov.<./parser> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "<.parser>S nulovým množstvím {nemůž[eš]|[jsi] nemohl[a]} nic dělat.
            <./parser> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>Tolik <<txt>> {|[jsi]} tu nevid{[íš]|ěl[a]}.<./parser> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "<.parser>Tady nemůžeš použít více objektů.<./parser> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "<.parser>S tímto příkazem není dovoleno použít více objektů.
            <./parser> ";
    }

    /*
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the <.parser> tag open, for the re-prompt that will follow */
        "<.parser>Tolik možností tu nebylo. ";
    }

    /*
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed; and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.
     *   
     *   If askingAgain is true, it means that we're asking the question
     *   again because we got an invalid response to the previous attempt
     *   at the same prompt.  We will have explained the problem, and now
     *   we're going to give the user another crack at the same response.
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* mark this as a question report with a dummy report */
        reportQuestion('');
        
        /*
         *   Open the "<.parser>" tag, if we're not "asking again."  If we
         *   are asking again, we will already have shown a message
         *   explaining why we're asking again, and that message will have
         *   left us in <.parser> tag mode, so we don't need to open the
         *   tag again. 
         */
        if (!askingAgain)
            "<.parser>";
        
        /*
         *   the question varies depending on whether we want just one
         *   object or several objects in the final result 
         */
        if (requiredNum == 1)
        {
            /*
             *   One object needed - use the original text in the query.
             *   
             *   Note that if we're "asking again," we will have shown an
             *   additional message first explaining *why* we're asking
             *   again, and that message will have left us in <.parser>
             *   tag mode; so we need to close the <.parser> tag in this
             *   case, but we don't need to show a new one. 
             */
            if (askingAgain)
            {
                "Tak co {|[jsi]} tím {m[áš]|měl[a]} na mysli?
                    \^<<askDisambigList(matchList, fullMatchList, nil,
                    dist)>>?";
            }
            else
            {
                "A co {|[jsi]} tím <q><<originalText>></q> {m[áš]|měl[a]} na
                    mysli? \^<<askDisambigList(matchList, fullMatchList, nil,
                    dist)>>?";
            }
        }
        else
        {
            /*
             *   Multiple objects required - ask by number, since we can't
             *   easily guess what the plural might be given the original
             *   text.
             *   
             *   As above, we only need to *close* the <.parser> tag if
             *   we're asking again, because we will already have shown a
             *   prompt that opened the tag in this case.  
             */
            if (askingAgain)
                "Tak které <<spellInt(requiredNum, 2)>> (z <<askDisambigList(
                    matchList, fullMatchList, true, dist)>>) {m[áš]|[jsi]
                    měl[a]} na mysli?";
            else
                "A které <<spellInt(requiredNum, 2)>> (z <<askDisambigList(
                    matchList, fullMatchList, true, dist)>>) {m[áš]|[jsi]
                    měl[a]} na mysli?";
        }

        /* close the <.parser> tag */
        "<./parser> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     *
     *   TODO: Přizpůsobit se rodu slovem který
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>Příběh nev{í|ěděl}, který <<originalText>>
            myslíš.<./parser> ";
    }

    /* the actor is missing in a command */
    missingActor(actor)
    {
        "<.parser>Musíš být specifičtější o tom, koho {chc[eš]|[jsi] chtěl[a]}
            {kdoco/postava} oslovit.<./parser> ";
    }

    /* only a single actor can be addressed at a time */
    singleActorRequired(actor)
    {
        "<.parser>Najednou {můž[eš]|[jsi] mohl[a]} oslovit jen jednu osobou.
            <./parser> ";
    }

    /* cannot change actor mid-command */
    cannotChangeActor()
    {
        "<.parser>V tomto příběhu {nemůž[eš]|[jsi] nemohl[a]} oslovit více než
            jednu postavu na jedné příkazové řádce.<./parser> ";
    }

    /*
     *   tell the user they entered a word we don't know, offering the
     *   chance to correct it with "oops" 
     */
    askUnknownWord(actor, txt)
    {
        /* start the message */
        "<.parser>Slovo <q><<txt>></q> v tomto příběhu {není|nebylo} důležité.<./parser> ";

        /* mention the OOPS command, if appropriate */
        oopsNote();
    }

    /*
     *   tell the user they entered a word we don't know, but don't offer
     *   an interactive way to fix it (i.e., we can't use OOPS at this
     *   point) 
     */
    wordIsUnknown(actor, txt)
    {
        "<.parser>Tomuto příkazu příběh nerozum{í|ěl}.<./parser> ";
    }

    /* the actor refuses the command because it's busy with something else */
    refuseCommandBusy(targetActor, issuingActor)
    {
        gMessageParams(targetActor);
        "{Kdoco targetActor} {je} zaneprázdněn{ý}. ";
    }

    /* cannot speak to multiple actors */
    cannotAddressMultiple(actor)
    {
        "<.parser>{Kdoco} {nemůže|nemohl[a]} oslovit najednou více postav.
            <./parser> ";
    }

    /* 
     *   Remaining actions on the command line were aborted due to the
     *   failure of the current action.  This is just a hook for the game's
     *   use, if it wants to provide an explanation; by default, we do
     *   nothing.  Note that games that override this will probably want to
     *   use a flag property so that they only show this message once -
     *   it's really only desirable to explain the the mechanism, not to
     *   flag it every time it's used.  
     */
    explainCancelCommandLine()
    {
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Non-Player Character (NPC) messages - parser-mediated format.  These
 *   versions of the NPC messages report errors through the
 *   parser/narrator.
 *   
 *   Note that a separate set of messages can be selected to report
 *   messages in the voice of the NPC - see npcMessagesDirect below.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessages: playerMessages
    /* the target cannot hear a command we gave */
    commandNotHeard(actor)
    {
        "{Kdoco} neodpov{ídá|ěděl[a]}. ";
    }

    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
        { "\^<<actor.name>> nevid{í|ěl[a]} nic jako <<txt>>. "; }
    noMatchNotAware(actor, txt)
        { "\^<<actor.name>> si {není|nebyl[a]} ničeho jako <<txt>> vědom{a}. "; }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>\^<<actor.name>> nevid{í|ěl[a]} nic vhodného.<./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>\^<<actor.name>> nevid{[íš]|ěl[a]} nic dalšího.<./parser> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>\^<<actor.name>> nevid{[íš]|ěl[a]} <<txt>> v takovém
            množství.<./parser> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>\^<<actor.name>> nev{[íš]|ěděl[a]}, co {|[jsi]} tím
            <<originalText>> {m[áš]|měl[a]} na mysli.<./parser> ";
    }

    /*
     *   Missing object query and error message templates 
     *
     *   Místo getQuestionInf použijeme getQuestionPast, protože chceme ve větě
     *   "Koho chceš, aby Sally *následovala*?" mít minulý čas.
     *
     *   TODO: Jak to, že i když je tu použit actor.name, tak se neshoduje se
     *   šablonou {kdoco}? Měl jsem pocit, že actor je výchozím, vždy přítomným
     *   parametrem. Přitom zde actorem je jiná postava, než PC. Tohle je asi ten
     *   duvod, proč někde jsou ve zprávách špičaté závorky, i když by zdánlivě
     *   šla použít šablona. Ohlídat - poznám to podle parametru funkce actor,
     *   ktery je explicitně předán?
     */
    askMissingObject(actor, action, which)
    {
        local pc = actor;
        gMessageParams(pc);
        reportQuestion('<.parser>\^' + action.whatObj(which) + ' {chceš pc},
            aby ' + actor.name + ' ' + action.getQuestionPast(which)
            + '?<./parser> ');
    }
    missingObject(actor, action, which)
    {
        "<.parser>Musíš být specifičtější o tom, <<action.whatObj(which)>>
            {chc[eš]|[jsi] chtěl[a]} {kdoco/postava}
        <<action.getQuestionInf(which)>>.<./parser> ";
    }

    /* missing literal phrase query and error message templates */
    missingLiteral(actor, action, which)
    {
        "<.parser>Musíš být specifičtější o tom, <<action.whatObj(which)>>
            {chc[eš]|[jsi] chtěl[a]} {kdoco/postava}
        <<action.getQuestionInf(which)>>. Například: <<actor.name>>,
        <<action.getQuestionInf(which)>> <q>něco</q>.<./parser> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   This default implementation simply doesn't report deferred errors at
 *   all.  The default message voice is the parser/narrator character, and
 *   there is simply no good way for the parser/narrator to say that a
 *   command failed in the past for a given character: "Bob looks like he
 *   didn't know which box you meant" just doesn't work.  So, we'll simply
 *   not report these errors at all.
 *   
 *   To report messages in the NPC's voice directly, modify the NPC's
 *   Actor object, or the Actor base class, to return
 *   npcDeferredMessagesDirect rather than this object from
 *   getParserMessageObj().  
 */
npcDeferredMessages: object
;

/* ------------------------------------------------------------------------ */
/*
 *   NPC messages, reported directly in the voice of the NPC.  These
 *   messages are not selected by default, but a game can use them instead
 *   of the parser-mediated versions by modifying the actor object's
 *   getParserMessageObj() to return these objects.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessagesDirect: npcMessages
    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
    {
        "\^<<actor.name>> se rozhl{íží|édl[a]} kolem. <q>Nevidím nic jako
            <<txt>>.</q> ";
    }
    noMatchNotAware(actor, txt)
    {
        "<q>Nevím o ničem jako <<txt>>,</q> {říká|řekl[a]} {kdoco}. ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Nevidím tu nic vhodného.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Nic dalšího tu nevidím.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>To nemohu udělat s nulovým množstvím.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>\^<<txt>> tu nevidím v takovém množství.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nemůžu tímto způsobem použít více objektů.</q> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nemůžu tímto způsobem použít více objektů.</q> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /* leave the quote open for the re-prompt */
        "{Kdoco} {říká|řekl[a]}:
        <q>To nebyla jedna z možností. ";
    }

    /*
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the quote open for the re-prompt */
        "{Kdoco} {říká|řekl[a]}:
        <q>Tolik možností tu nebylo. ";
    }

    /*
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed, and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.  
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* mark this as a question report */
        reportQuestion('');
        
        /* the question depends on the number needed */
        if (requiredNum == 1)
        {
            /* one required - ask with the original text */
            if (!askingAgain)
                "\^<<actor.name>> se ptá, <q>";
            
            "A co {|[jsi]}  tím <q><<originalText>></q> {m[áš]|měl[a]} na
                mysli, <<askDisambigList(matchList, fullMatchList, nil,
                dist)>>?</q> ";
        }
        else
        {
            /*
             *   more than one required - we can't guess at the plural
             *   given the original text, so just use the number 
             */
            if (!askingAgain)
                "\^<<actor.name>> ask, <q>";
            
            "A které <<spellInt(requiredNum, 2)>> (z <<
            askDisambigList(matchList, fullMatchList, true, dist)>>)
            máš na mysli?</q> ";
        }
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nevím, co tím <q><<originalText>></q> máš na mysli.</q> ";
    }

    /*
     *   Missing object query and error message templates 
     */
    askMissingObject(actor, action, which)
    {
        reportQuestion('{Kdoco} {říká|řekl[a]}: <q>\^' + action.whatObj(which)
            + ' chceš, abych ' + action.getQuestionInf(which) + '?</q> ');
    }
    missingObject(actor, action, which)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Nevím <<action.whatObj(which)>> chceš, abych
            <<action.getQuestionInf(which)>>.</q> ";
    }
    missingLiteral(actor, action, which)
    {
        /* use the same message we use for a missing ordinary object */
        missingObject(actor, action, which);
    }

    /* tell the user they entered a word we don't know */
    askUnknownWord(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Neznám slovo <q><<txt>></q>.</q> ";
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Použil{a} jsi slovo, které neznám.</q> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   Some messages will never be deferred:
 *   
 *   commandNotHeard - if a command is not heard, it will never enter an
 *   actor's command queue; the error is given immediately in response to
 *   the command entry.
 *   
 *   refuseCommandBusy - same as commandNotHeard
 *   
 *   noMatchDisambig - interactive disambiguation will not happen in a
 *   deferred response situation, so it is impossible to have an
 *   interactive disambiguation failure.  
 *   
 *   disambigOrdinalOutOfRange - for the same reason noMatchDisambig can't
 *   be deferred.
 *   
 *   askDisambig - if we couldn't display a message, we definitely
 *   couldn't perform interactive disambiguation.
 *   
 *   askMissingObject - for the same reason that askDisambig can't be
 *   deferred
 *   
 *   askUnknownWord - for the same reason that askDisambig can't be
 *   deferred.  
 */
npcDeferredMessagesDirect: npcDeferredMessages
    /* TODO: pot5ebuji zn8t nejen actor, ale i protistranu. */
    commandNotUnderstood(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nerozuměl{a} jsem, co jsi měl{a} na mysli.</q> ";
    }

    /* no match for a noun phrase */
    noMatchCannotSee(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Nic jako <<txt>> jsem neviděl{a}.</q> ";
    }
    noMatchNotAware(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Ničeho jako <<txt>> jsem si nebyl{a} vědom{a}.</q> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Neviděl{a} jsem nic vhodného.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nevěděl{a} jsem, co jsi měl na mysli.</q> ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Vynechal{a} jsi některá slova.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nerozumím tomu, co jsi měl{a} na mysli.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>\^<<txt>> jsem v takovém množství neviděl{a}.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nerozuměl{a} jsem tomu, co máš na mysli.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    /* TODO: Poslední šablona by měla být podle cílového actora. */
    singleObjectRequired(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}:
        <q>Nerozumím tomu, co jsi myslel{a}.</q> ";
    }

    /*
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     *
     *   TODO: Řídit se rodem podle matchListu, ale bez testu to nedam.
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "{Kdoco} {říká|řekl[a]}:
            <q>Nejsem si jist{ý actor}, který <<originalText>> máš na
            mysli.</q> ";
    }

    /* an object phrase was missing */
    askMissingObject(actor, action, which)
    {
        reportQuestion('{Kdoco} {říká|řekl[a]}: <q>Nevím '
            + action.whatObj(which) + ' jsem měl{a} '
            + action.getQuestionInf(which) + '.</q> ');
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "{Kdoco} {říká|řekl[a]}: <q>Použil jsi slovo, které neznám.</q> ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Verb messages for standard library verb implementations for actions
 *   performed by the player character.  These return strings suitable for
 *   use in VerifyResult objects as well as for action reports
 *   (defaultReport, mainReport, and so on).
 *   
 *   Most of these messages are generic enough to be used for player and
 *   non-player character alike.  However, some of the messages either are
 *   too terse (such as the default reports) or are phrased awkwardly for
 *   NPC use, so the NPC verb messages override those.  
 */
playerActionMessages: MessageHelper
    /*
     *   generic "can't do that" message - this is used when verification
     *   fails because an object doesn't define the action ("doXxx")
     *   method for the verb 
     */
    cannotDoThatMsg = 'To {nemůž[eš]|[jsi] nemohl[a]} udělat. '

    /* must be holding something before a command */
    mustBeHoldingMsg(obj)
    {
        gMessageParams(obj);
        return 'Abys to mohl{a} udělat, musel{a} {[bys]|[jsi]} {kohoco obj}
            držet. ';
    }

    /* it's too dark to do that */
    tooDarkMsg = 'Na to {je tu|tu byla} moc velká tma. '

    /* object must be visible */
    mustBeVisibleMsg(obj)
    {
        gMessageParams(obj);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji obj} vidět. ';
    }

    /* object can be heard but not seen */
    heardButNotSeenMsg(obj)
    {
        gMessageParams(obj);
        return 'Sice {|[jsi]} {kohoco obj} slyš{[íš]|el[a]}, ale nevid{[íš]|ěl[a]
            [jsi]} {ho/ji obj}. ';
    }

    /* object can be smelled but not seen */
    smelledButNotSeenMsg(obj)
    {
        gMessageParams(obj);
        return 'Sice {|[jsi]} {kohoco obj} cít{[íš]|il[a]}, ale nevid{[íš]|ěl[a]
            [jsi]} {ho/ji obj}. ';
    }

    /* cannot hear object */
    cannotHearMsg(obj)
    {
        gMessageParams(obj);
        return 'Neslyš{[íš]|el[a] [jsi]} {ho/ji obj}. ';
    }

    /* cannot smell object */
    cannotSmellMsg(obj)
    {
        gMessageParams(obj);
        return 'Necít{[íš]|il[a] [jsi]} {ho/ji obj}. ';
    }

    /* cannot taste object */
    cannotTasteMsg(obj)
    {
        gMessageParams(obj);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji obj} ochutnat. ';
    }

    /* must remove an article of clothing before a command */
    cannotBeWearingMsg(obj)
    {
        gMessageParams(obj);
        return 'Než {bud[eš] moci|[jsi] mohl[a]} tohle udělat,
            mus{[íš]|el[a] [jsi]} nejprve sundat {kohoco obj}. ';
    }

    /* all contents must be removed from object before doing that */
    mustBeEmptyMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {mus[íš]|[jsi] musel[a]} všechno {z/ze obj} {kohočeho obj}
            vyndat, než {bud[eš] moci|[jsi] mohl[a]} tohle udělat. ';
    }

    /* object must be opened before doing that */
    mustBeOpenMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {mus[íš]|[jsi] musel[a]} otevřít {kohoco obj}, než
            {bud[eš] moci|[jsi] mohl[a]} tohle udělat. ';
    }

    /* object must be closed before doing that */
    mustBeClosedMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {mus[íš]|[jsi] musel[a]} zavřít {kohoco obj}, než
            {bud[eš] moci|[jsi] mohl[a]} tohle udělat. ';
    }

    /* object must be unlocked before doing that */
    mustBeUnlockedMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {mus[íš]|[jsi] musel[a]} odemknout {kohoco obj}, než
            {bud[eš] moci|[jsi] mohl[a]} tohle udělat. ';
    }

    /* no key is needed to lock or unlock this object */
    noKeyNeededMsg = '{Kdoco dobj} {není|nebyl[a dobj]} na klíč. '

    /* actor must be standing before doing that */
    mustBeStandingMsg = 'Nejprve {mus[íš]|[jsi] musel[a]} vstát, než {bud[eš]
        moci|[jsi] mohl[a]} tohle udělat. '

    /* must be sitting on/in chair */
    mustSitOnMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {si mus[íš]|[sis] musel[a]} sednout {v obj}. ';
    }

    /* must be lying on/in object */
    mustLieOnMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {si mus[íš]|[sis] musel[a]} lehnout {v obj}. ';
    }

    /* must get on/in object */
    mustGetOnMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {se mus[íš]|[ses] musel[a]} dostat {v obj}. ';
    }

    /* object must be in loc before doing that */
    mustBeInMsg(obj, loc)
    {
        gMessageParams(obj, loc);
        return 'Než {bud[eš] moci|[jsi] mohl[a]} tohle udělat, {kdoco obj}
            {musí|musel[a obj]} být {v loc}. ';
    }

    /* actor must be holding the object before we can do that */
    mustBeCarryingMsg(obj, actor)
    {
        gMessageParams(obj, actor);
        return '{Kdoco actor} {musí|musel[a actor]} nejprve {kohoco obj} držet,
            než {bude moci|mohl[a actor]} tohle udělat. ';
    }

    /* generic "that's not important" message for decorations */
    decorationNotImportantMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {není} důležit{ý}. ';
    }

    /* generic "you don't see that" message for "unthings" */
    unthingNotHereMsg(obj)
    {
        gMessageParams(obj);
        return 'Nevid{[íš]|el[a] [jsi]} {ho/ji obj} tady. ';
    }

    /* generic "that's too far away" message for Distant items */
    tooDistantMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} příliš daleko. ';
    }

    /* generic "no can do" message for intangibles */
    notWithIntangibleMsg(obj)
    {
        gMessageParams(obj);
        return 'To {|[jsi]} {s/se} {kýmčím} nem{ůž[eš]|ohl[a]} udělat. ';
    }

    /* generic failure message for varporous objects */
    notWithVaporousMsg(obj)
    {
        gMessageParams(obj);
        return 'To {|[jsi]} {s/se} {kýmčím} nem{ůž[eš]|ohl[a]} udělat. ';
    }

    /* look in/look under/look through/look behind/search vaporous */
    lookInVaporousMsg(obj)
    {
        gMessageParams(obj);
        return 'Vid{[íš]|ěl[a] [jsi]} prostě jen {kohoco obj}. ';
    }

    /*
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObjectMsg(obj)
    {
        gMessageParams(obj);
        return 'Na {kohoco obj} {nedosáhn[eš]|[jsi] nedosáhl[a]}. ';
    }

    /* cannot reach an object through an obstructor */
    cannotReachThroughMsg(obj, loc)
    {
        gMessageParams(obj, loc);
        return '\^{kdoco/postava} {nemůž[eš]|[jsi] nemohl[a]} skrz {kohoco loc}
            šahat na {kohoco obj}.';
    }

    /* generic long description of a Thing */
    thingDescMsg(obj)
    {
        gMessageParams(obj);
        return 'Nevid{[íš]|ěl[a] [jsi]} na {něm/ní obj} nic mimořádného. ';
    }

    /* generic LISTEN TO description of a Thing */
    thingSoundDescMsg(obj)
        { return 'Neslyš{[íš]|el[a] [jsi]} nic mimořádného. '; }

    /* generic "smell" description of a Thing */
    thingSmellDescMsg(obj)
        { return 'Necít{[íš]|il[a] [jsi]} nic mimořádného. '; }

    /* default description of a non-player character */
    npcDescMsg(npc)
    {
        gMessageParams(npc);
        return 'Nevid{[íš]|ěl[a] [jsi]} na {něm/ní npc} nic neobvyklého. ';
    }

    /* generic messages for looking prepositionally */
    nothingInsideMsg =
        '{V/ve dobj} {komčem dobj} {není|nebylo} nic neobvyklého. '
    nothingUnderMsg =
        'Pod {kýmčím dobj} {není|nebylo} nic neobvyklého. '
    nothingBehindMsg =
        'Za {kýmčím dobj} {není|nebylo} nic neobvyklého. '
    nothingThroughMsg =
        'Skrz {kohoco dobj} {|[jsi]} nic nevid{[íš]|ěl[a]}. '

    /* this is an object we can't look behind/through */
    cannotLookBehindMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat za {kohoco dobj}. '
    cannotLookUnderMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat pod {kohoco dobj}. '
    cannotLookThroughMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} se podívat skrz
        {kohoco dobj}. '

    /* looking through an open passage */
    nothingThroughPassageMsg = 'Odsud {|[jsi]} toho skrz {kohoco dobj} moc
        nevid{[íš]|ěl[a]}. '

    /* there's nothing on the other side of a door we just opened */
    nothingBeyondDoorMsg = 'Otevření {kohočeho dobj} neodhalilo nic
        zvláštního. '

    /* there's nothing here with a specific odor */
    nothingToSmellMsg = 'Necít{[íš]|il[a] [jsi]} nic neobvyklého. '

    /* there's nothing here with a specific noise */
    nothingToHearMsg = 'Neslyš{[íš]|el[a] [jsi]} nic neobvyklého. '

    /* a sound appears to be coming from a source */
    noiseSourceMsg(src)
    {
        return '{Kdoco dobj} zřejmě vycház{[íš dobj]|zel[a dobj]} '
            + src.prepZ() + ' ' + src.nameKohoCeho + '. ';
    }

    /* an odor appears to be coming from a source */
    odorSourceMsg(src)
    {
        return '{Kdoco dobj} zřejmě vycház{[íš dobj]|zel[a dobj]} '
            + src.prepZ() + ' ' + src.nameKohoCeho + '. ';
    }

    /* an item is not wearable */
    notWearableMsg =
        'To {není} něco, co {[bys]|[jsi]} mohl{a} na sebe vzít. '

    /* doffing something that isn't wearable */
    notDoffableMsg =
        'To {není} něco, co {[bys]|[jsi]} mohl{a} sundat. '

    /* already wearing item */
    alreadyWearingMsg = 'Už {|[jsi]} {ho/ji dobj} {m[áš]|měl[a]} na sobě. '

    /* not wearing (item being doffed) */
    notWearingMsg = '{Kohoco dobj} nem{áš} na sobě. '

    /* default response to 'wear obj' */
    okayWearMsg = 'Hotovo, {m[áš]|měl[a] [jsi]} na sobě {kohoco dobj}. '

    /* default response to 'doff obj' */
    okayDoffMsg = 'Hotovo, už {|[jsi]} na sobě ne{m[áš]|měl[a]} {kohoco dobj}. '

    /* default response to open/close */
    okayOpenMsg = shortTMsg(
        'Hotovo. ', 'Otev{ír[áš]|řel[a] [jsi]} {kohoco dobj}. ')
    okayCloseMsg = shortTMsg(
        'Hotovo. ', 'Zav{ír[áš]|řel[a] [jsi]} {kohoco dobj}. ')

    /* default response to lock/unlock */
    okayLockMsg = shortTMsg(
        'Hotovo. ', 'Zam{yk[áš]|kl[a] [jsi]} {kohoco dobj}. ')
    okayUnlockMsg = shortTMsg(
        'Hotovo. ', 'Odem{yk[áš]|kl[a] [jsi]} {kohoco dobj}. ')

    /* cannot dig here */
    cannotDigMsg = 'Ke kopání do {něho/ní dobj} {nem[áš]|[jsi] neměl[a]} žádný
        důvod. '

    /* not a digging implement */
    cannotDigWithMsg =
        'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {ho/ji iobj} použít jako
            lopatu. '

    /* taking something already being held */
    alreadyHoldingMsg = '{Kohoco dobj} {|[jsi]} už drž{[íš]|el[a]}. '

    /* actor taking self ("take me") */
    takingSelfMsg = 'Nemůž{eš} vzít sebe. '

    /* dropping an object not being carried */
    notCarryingMsg = '{Kohoco dobj} nem{áš} u sebe. '

    /* actor dropping self */
    droppingSelfMsg = 'Nemůž{eš} položit sebe. '

    /* actor putting self in something */
    puttingSelfMsg = 'Tohle se sebou udělat nem{ůž[eš]|ohl[a]}. '

    /* actor throwing self */
    throwingSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} hodit sebe. '

    /* we can't put the dobj in the iobj because it's already there */
    alreadyPutInMsg = '{Kdoco dobj} už {je} {v/ve iobj} {komčem iobj}. '

    /* we can't put the dobj on the iobj because it's already there */
    alreadyPutOnMsg = '{Kdoco dobj} už {je} na {komčem iobj}. '

    /* we can't put the dobj under the iobj because it's already there */
    alreadyPutUnderMsg = '{Kdoco dobj} už {je} pod {kýmčím iobj}. '

    /* we can't put the dobj behind the iobj because it's already there */
    alreadyPutBehindMsg = '{Kdoco dobj} už {je} za {kýmčím iobj}. '

    /*
     *   trying to move a Fixture to a new container by some means (take,
     *   drop, put in, put on, etc) 
     */
    cannotMoveFixtureMsg = '{S/se dobj} {kýmčím dobj} se ned{á|al[a dobj]} hýbat. '

    /* trying to take a Fixture */
    cannotTakeFixtureMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} vzít. '

    /* trying to put a Fixture in something */
    cannotPutFixtureMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} nikam dát. '

    /* trying to take/move/put an Immovable object */
    cannotTakeImmovableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} vzít. '
    cannotMoveImmovableMsg = '{S/se dobj} {kýmčím dobj} se ned{á|al[a dobj]} hýbat. '
    cannotPutImmovableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} nikam dát. '

    /* trying to take/move/put a Heavy object */
    cannotTakeHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '
    cannotMoveHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '
    cannotPutHeavyMsg = '{Kdoco dobj} {je} příliš těžk{ý}. '

    /* trying to move a component object */
    cannotMoveComponentMsg(loc)
    {
        return '{Kdoco dobj} {je} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* trying to take a component object */
    cannotTakeComponentMsg(loc)
    {
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} mít, protože
            {je dobj} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* trying to put a component in something */
    cannotPutComponentMsg(loc)
    {
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} nikam dát, protože {je}
            součástí ' + loc.nameKohoCeho + '. ';
    }

    /* specialized Immovable messages for TravelPushables */
    cannotTakePushableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} vzít, ale
        mohl{a} {[bys]|[jsi]} {ho/ji dobj} zkusit někam odtlačit. '
    cannotMovePushableMsg = 'Bezcílným posouváním {kohočeho dobj} {[bys]|[jsi]}
        ničeho nedosáhl{a}, ale mohl{a} {[bys]|[jsi]} {ho/ji dobj} zkusit
        odtlačit určitým směrem. '
    cannotPutPushableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} nikam dát,
        ale mohl{a} {[bys]|[jsi]} {ho/ji dobj} zkusit někam odtlačit. '

    /* can't take something while occupying it */
    cannotTakeLocationMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} vzít, dokud
        {|[jsi]} {ho/ji dobj} zabír{[áš]|al[a]}. '

    /* can't REMOVE something that's being held */
    cannotRemoveHeldMsg = '{Není|Nebylo} z čeho {kohoco dobj} vyndat. '

    /* default 'take' response */
    /*
     *   Mělo by dávat smysl i na objekty sebrané ze země/lokace, i vyndané
     *   z kontejneru, který držím. Mám pocit, že nedokonavé fungovaly trochu
     *   víc přirozeně "Beru.", než přesnější překlad "Uchopeno.". "Vzato." je
     *   divné a "Sebráno." nefunguje na kontejner. Kromě toho řada hlášek
     *   obsahuje jen "Hotovo.", které zase nemá nedokonavý tvar a mělo by to
     *   všechno působit jednotně.
     */
    okayTakeMsg = shortTMsg(
        'Hotovo. ', 'Ber{eš} {kohoco dobj}. ')

    /* default 'drop' response */
    okayDropMsg = shortTMsg(
        'Hotovo. ', 'Od{klád[áš]|ložil[a] [jsi]} {kohoco dobj}. ')

    /* dropping an object */
    droppingObjMsg(dropobj)
    {
        gMessageParams(dropobj);
        return 'Od{klád[áš]|ložil[a] [jsi]} {kohoco dropobj}. ';
    }

    /* default receiveDrop suffix for floorless rooms */
    floorlessDropMsg(dropobj)
    {
        gMessageParams(dropobj);
        return 'Pad{[áš dropobj]|al[a dropobj]} kamsi dolů z dohledu. ';
    }

    /* default successful 'put in' response */
    okayPutInMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} do {kohočeho iobj}. ')

    /* default successful 'put on' response */
    okayPutOnMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} na {kohoco iobj}. ')

    /* default successful 'put under' response */
    okayPutUnderMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} pod {kohoco iobj}. ')

    /* default successful 'put behind' response */
    okayPutBehindMsg = shortTIMsg(
        'Hotovo. ', '{Dáv[áš]|Dal[a] [jsi]} {kohoco dobj} za {kohoco iobj}. ')

    /* try to take/move/put/taste an untakeable actor */
    cannotTakeActorMsg = 'To {ti/jí} {kdoco dobj} nedovolí. '
    cannotMoveActorMsg = 'To {ti/jí} {kdoco dobj} nedovolí. '
    cannotPutActorMsg = 'To {ti/jí} {kdoco dobj} nedovolí. '
    cannotTasteActorMsg = 'To {ti/jí} {kdoco dobj} nedovolí. '

    /* trying to take/move/put/taste a person */
    cannotTakePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotMovePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotPutPersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '
    cannotTastePersonMsg =
        'To by se {komučemu dobj} nejspíš nelíbilo. '

    /* cannot move obj through obstructor */
    cannotMoveThroughMsg(obj, obs)
    {
        gMessageParams(obj, obs);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji obj} sunout skrz '
            + '{kohoco obs}. ';
    }

    /* cannot move obj in our out of container cont */
    cannotMoveThroughContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji obj} sunout skrz '
            + '{kohoco cont}. ';
    }

    /* cannot move obj because cont is closed */
    cannotMoveThroughClosedMsg(obj, cont)
    {
        gMessageParams(cont);
        return 'To nem{ůž[eš]|ohl[a] [jsi]} udělat, protože {kdoco cont} '
            + '{je} zavřen{ý}. ';
    }

    /* cannot fit obj into cont through cont's opening */
    cannotFitIntoOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To nem{ůž[eš]|ohl[a] [jsi]} udělat, protože {kdoco obj} '
            + '{je} příliš velk{ý} a do {kohoco cont} se '
            + 'neve{jd[eš obj]|šl[a obj]}. ';
    }

    /* cannot fit obj out of cont through cont's opening */
    cannotFitOutOfOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} {je}
            příliš velký na vyndání {z/ze cont} {kohočeho cont}. ';
    }

    /* actor 'obj' cannot reach in our out of container 'cont' */
    cannotTouchThroughContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} nem{ůže|ohl[a obj]} skrz {kohoco cont} s ničím
            manipulovat. ';
    }

    /* actor 'obj' cannot reach through cont because cont is closed */
    cannotTouchThroughClosedMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco obj} nem{ůže|ohl[a obj]} udělat, protože {kdoco cont}
            {je} zavřen{ý}. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachIntoOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Komučemu obj} se do {kohočeho cont} neve{jde|šla} ruka. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachOutOfOpeningMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} neprotáhn{e|ul[a obj]} ruku ven {z/ze cont}
            {kohočeho cont}. ';
    }

    /* the object is too large for the actor to hold */
    tooLargeForActorMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} na {tebe/ni} moc velký. ';
    }

    /* the actor doesn't have room to hold the object */
    handsTooFullForMsg(obj)
    {
        return '{Tvoje} ruce {jsou|byly} přeplněné, nezvládneš držet '
            + obj.nameKohoCo + '. ';
    }

    /* the object is becoming too large for the actor to hold */
    becomingTooLargeForActorMsg(obj)
    {
        gMessageParams(obj);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} by
            byl{a obj} moc velk{ý} a neuzdvihl{a} bys {ho/ji obj}. ';
    }

    /* the object is becoming large enough that the actor's hands are full */
    handsBecomingTooFullForMsg(obj)
    {
        gMessageParams(obj);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {tvoje} ruce by
            pak byly moc plné a neudržel{a} bys {ho/ji obj}. ';
    }

    /* the object is too heavy (all by itself) for the actor to hold */
    tooHeavyForActorMsg(obj)
    {
        gMessageParams(obj);
        return '{Kohoco obj} {nezdvihn[eš]|[jsi] neuzdvihl[a]}, protože {je obj}
            moc těžk{ý}. ';
    }

    /*
     *   the object is too heavy (in combination with everything else
     *   being carried) for the actor to pick up 
     */
    totalTooHeavyForMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} moc těžk{ý}, {bud[eš] muset|musel[a] [jsi]}
            nejprve něco odložit. ';
    }

    /* object is too large for container */
    tooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, do {kohoco cont} se
            neve{jde obj}. ';
    }

    /* object is too large to fit under object */
    tooLargeForUndersideMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, pod {kohoco cont} se
            neve{jde obj}. ';
    }

    /* object is too large to fit behind object */
    tooLargeForRearMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco obj} {je} moc velk{ý}, za {kohoco cont} se
            neve{jde obj}. ';
    }

    /* container doesn't have room for object */
    containerTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{Kdoco cont} {je} moc pln{ý}, {kdoco obj} se už do {něho/ní cont}
            neve{jde obj}. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Na {komčem cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* underside doesn't have room for object */
    undersideTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Pod {kýmčím cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* rear surface/space doesn't have room for object */
    rearTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'Za {kýmčím cont} {není|nebylo} pro {kohoco obj} dost místa. ';
    }

    /* the current action would make obj too large for its container */
    becomingTooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj} by
            {pak|} byl{a obj} moc velk{ý} na {kohoco cont}. ';
    }

    /*
     *   the current action would increase obj's bulk so that container is
     *   too full 
     */
    containerBecomingTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco obj}
            by se už neve{šel obj} do {kohočeho cont}. ';
    }

    /* trying to put an object in a non-container */
    notAContainerMsg = 'Do {kohočeho iobj} se ned{á|alo} nic dát. '

    /* trying to put an object on a non-surface */
    notASurfaceMsg = 'Na {komčem iobj} {není|nebyla} žádná vhodná plocha. '

    /* can't put anything under iobj */
    cannotPutUnderMsg =
        'Pod {něj/ni iobj} {nemůž[eš]|[jsi] nemohl[a]} nic dát. '

    /* nothing can be put behind the given object */
    cannotPutBehindMsg = 'Za {kohoco iobj} {nemůž[eš]|[jsi] nemohl[a]} nic
        dát. '

    /* trying to put something in itself */
    cannotPutInSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} do sebe sama. '

    /* trying to put something on itself */
    cannotPutOnSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} na sebe sama. '

    /* trying to put something under itself */
    cannotPutUnderSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} pod seebe
        sama. '

    /* trying to put something behind itself */
    cannotPutBehindSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} za
        sebe sama. '

    /* can't put something in/on/etc a restricted container/surface/etc */
    cannotPutInRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} dát do {kohočeho iobj}. '
    cannotPutOnRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} dát na {kohoco iobj}. '
    cannotPutUnderRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} dát pod {kohoco iobj}. '
    cannotPutBehindRestrictedMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} dát za {kohoco iobj}. '

    /* trying to return something to a remove-only dispenser */
    cannotReturnToDispenserMsg = '{Kohoco dobj} {nemůž[eš]|[jsi] nemohl[a]} vrátit
        zpátky do {kohočeho iobj}. '

    /* wrong item type for dispenser */
    cannotPutInDispenserMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} dát {kohoco dobj} do {kohočeho iobj}. '

    /* the dobj doesn't fit on this keyring */
    objNotForKeyringMsg = '{Kdoco dobj} {nepasuje|nepasoval[a dobj]} na
        {kohoco iobj}. '

    /* the dobj isn't on the keyring */
    keyNotOnKeyringMsg = '{Kdoco dobj} {není} navlečen{ý} na {komčem iobj}. '

    /* can't detach key (with no iobj specified) because it's not on a ring */
    keyNotDetachableMsg = '{Kdoco dobj} {není} na ničem navlečen{ý}. '

    /* we took a key and attached it to a keyring */
    takenAndMovedToKeyringMsg(keyring)
    {
        gMessageParams(keyring);
        return '{Ber[eš]|Vzal[a] [jsi]} {kohoco dobj} a navlék{[áš]|l[a] [jsi]}
            {ho/ji dobj} na {kohoco keyring}. ';
    }

    /* we attached a key to a keyring automatically */
    movedKeyToKeyringMsg(keyring)
    {
        gMessageParams(keyring);
        return 'Navlék{[áš]|l[a] [jsi]} {kohoco dobj} na {kohoco keyring}. ';
    }

    /* we moved several keys to a keyring automatically */
    movedKeysToKeyringMsg(keyring, keys)
    {
        gMessageParams(keyring);
        return 'Navlék{[áš]|l[a] [jsi]} ' + (keys.length() > 1
            ? 'své volné klíče' : 'svůj volný klíč') + ' na
            {kohoco keyring}. ';
    }

    /* putting y in x when x is already in y */
    circularlyInMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            {v/ve y} {komčem y}. ';
    }

    /* putting y in x when x is already on y */
    circularlyOnMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            na {komčem y}. ';
    }

    /* putting y in x when x is already under y */
    circularlyUnderMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            pod {kýmčím y}. ';
    }

    /* putting y in x when x is already behind y */
    circularlyBehindMsg(x, y)
    {
        gMessageParams(x, y);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je x} {kdoco x}
            za {kýmčím y}. ';
    }

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotInMsg = '{Kdoco dobj} v {něm/ní iobj} {není dobj}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOnMsg = '{Kdoco dobj} na {něm/ní iobj} {není}. '

    /* taking dobj from under something, but dobj isn't under iobj */
    takeFromNotUnderMsg = '{Kdoco dobj} pod {ním/ní iobj} {není}. '

    /* taking dobj from behind something, but dobj isn't behind iobj */
    takeFromNotBehindMsg = '{Kdoco dobj} za {ním/ní iobj} {není}. '

    /* taking dobj from an actor, but actor doesn't have iobj */
    takeFromNotInActorMsg = '{Kdoco iobj} {ho/ji dobj}
        ne{m[áš iobj]|měl[a iobj]}. '

    /* actor won't let go of a possession */
    willNotLetGoMsg(holder, obj)
    {
        gMessageParams(holder, obj);
        return 'Nedovol{[íš holder]|il[a holder] [jsi holder]} {komučemu} {ho/ji obj} vzít. ';
    }

    /* must say which way to go */
    whereToGoMsg = 'Budeš muset říci, kterým směrem se vydat. '

    /* travel attempted in a direction with no exit */
    cannotGoThatWayMsg = 'Tímto směrem {se|[ses]} {kdoco/postava} pohybovat
        {nemůž[eš]|nemohl[a]}. '

    /* travel attempted in the dark in a direction with no exit */
    cannotGoThatWayInDarkMsg = '{Je|Byla} tu příliš velká tma, než a{bys}
        viděl{a} na cestu. '

    /* we don't know the way back for a GO BACK */
    cannotGoBackMsg = 'Nev{[íš]|ěděl[a] [jsi]}, jak se odsud dostat zpátky. '

    /* cannot carry out a command from this location */
    cannotDoFromHereMsg = 'Odtud {|[jsi]} to nem{ůž[eš]|ohl[a]} udělat. '

    /* can't travel through a close door */
    cannotGoThroughClosedDoorMsg(door)
    {
        gMessageParams(door);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, protože {kdoco door}
               {je} zavřen{ý}. ';
    }

    /* cannot carry out travel while 'dest' is within 'cont' */
    invalidStagingContainerMsg(cont, dest)
    {
        gMessageParams(cont, dest);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {kdoco dest} {je}
            {v/ve cont} {komčem cont}. ';
    }

    /* cannot carry out travel while 'cont' (an actor) is holding 'dest' */
    invalidStagingContainerActorMsg(cont, dest)
    {
        gMessageParams(cont, dest);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {kdoco cont}
            drž{í|el[a cont]} {kohoco dest}. ';
    }

    /* can't carry out travel because 'dest' isn't a valid staging location */
    invalidStagingLocationMsg(dest)
    {
        gMessageParams(dest);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} se dostat {do dest}. ';
    }

    /* destination is too high to enter from here */
    nestedRoomTooHighMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} {je} příliš vysoko, odsud {|[jsi]} na {něj/ni obj}
            nedosáh{n[eš]|l[a]}. ';
    }

    /* enclosing room is too high to reach by GETTING OUT OF here */
    nestedRoomTooHighToExitMsg(obj)
    {
        return 'Odsud by to byl pád z moc velké výšky. ';
    }

    /* cannot carry out a command from a nested room */
    cannotDoFromMsg(obj)
    {
        gMessageParams(obj);
        return 'To {|[jsi]} {z/ze obj} {kohočeho obj} nem{ůž[eš]|ohl[a]} udělat. ';
    }

    /* cannot carry out a command from within a vehicle in a nested room */
    vehicleCannotDoFromMsg(obj)
    {
        local loc = obj.location;
        gMessageParams(obj, loc);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {je obj}
            {kdoco obj} {v loc}. ';
    }

    /* cannot go that way in a vehicle */
    cannotGoThatWayInVehicleMsg(traveler)
    {
        gMessageParams(traveler);
        return 'To {nemůž[eš]|[jsi] nemohl[a]} udělat {v traveler}. ';
    }

    /* cannot push an object that way */
    cannotPushObjectThatWayMsg(obj)
    {
        gMessageParams(obj);
        return 'Nem{ůž[eš]|ohl[a] [jsi]} tlačit {kohoco obj} tím směrem. ';
    }

    /* cannot push an object to a nested room */
    cannotPushObjectNestedMsg(obj)
    {
        gMessageParams(obj);
        return 'Tam {nemůž[eš]|[jsi] nemohl[a]} tlačit {kohoco obj}. ';
    }

    /* cannot enter an exit-only passage */
    cannotEnterExitOnlyMsg(obj)
    {
        gMessageParams(obj);
        return 'Odsud {nemůž[eš]|[jsi] nemohl[a]} vstoupit do {kohočeho obj}. ';
    }

    /* must open door before going that way */
    mustOpenDoorMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejdříve {mus[íš]|[jsi] musel[a]} {kohoco obj} otevřít. ';
    }

    /* door closes behind actor during travel through door */
    doorClosesBehindMsg(obj)
    {
        gMessageParams(obj);
        return '<.p>Jakmile {jsi} pro{šel} {kýmčím obj}, tak se za
            {tebou/ním actor} zavřel{a obj}. ';
    }

    /* the stairway does not go up/down */
    stairwayNotUpMsg = '{Kdoco dobj} odsud {vede|vedl[a dobj]} jen dolů. '
    stairwayNotDownMsg = '{Kdoco dobj} odsud {vede|vedl[a dobj]} jen nahoru. '

    /* "wait" */
    timePassesMsg = 'Čas plyn{e|ul}... '

    /* "hello" with no target actor */
    sayHelloMsg = (addressingNoOneMsg)

    /* "goodbye" with no target actor */
    sayGoodbyeMsg = (addressingNoOneMsg)

    /* "yes"/"no" with no target actor */
    sayYesMsg = (addressingNoOneMsg)
    sayNoMsg = (addressingNoOneMsg)

    /* an internal common handler for sayHelloMsg, sayGoodbyeMsg, etc */
    addressingNoOneMsg
    {
        return 'Musíš být specifičtější o tom, koho {|jsi} {[chceš]|chtěl[a]}
            {kdoco/postava} oslovit. ';
    }

    /* "yell" */
    okayYellMsg = 'Křič{[íš]|el[a] [jsi]} tak hlasitě, jak {|[jsi]} jen
        doved{[eš]|l[a]}. '

    /* "jump" */
    okayJumpMsg = 'Kousek {vyskaku[ješ]|[jsi] vyskočil[a]} a
        dopad{[áš]|l[a]} zpátky na stejném místě. '

    /* cannot jump over object */
    cannotJumpOverMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} přeskočit. '

    /* cannot jump off object */
    cannotJumpOffMsg = 'Z {něho/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} seskočit. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHereMsg = 'Odsud {není|nebylo} kam vyskočit. '

    /* failed to find a topic in a consultable object */
    cannotFindTopicMsg =
        'To se {ti/jí} ne{daří|podařilo} najít {v/ve dobj} {komčem dobj}. '

    /* an actor doesn't accept a command from another actor */
    refuseCommand(targetActor, issuingActor)
    {
        gMessageParams(targetActor, issuingActor);
        return '{Kdoco targetActor} odmít{[áš targetActor]|nul[a targetActor]} {tvoji}
            žádost. ';
    }

    /* cannot talk to an object (because it makes no sense to do so) */
    notAddressableMsg(obj)
    {
        gMessageParams(obj);
        return 'S {ním/ní obj} {nemůž[eš]|[jsi] nemohl[a]} mluvit. ';
    }

    /* actor won't respond to a request or other communicative gesture */
    noResponseFromMsg(other)
    {
        gMessageParams(other);
        return '{Kdoco other} neodpovíd{[áš other]|al[a other]}. ';
    }

    /* trying to give something to someone who already has the object */
    giveAlreadyHasMsg = '{Kdoco iobj} už {ho/ji dobj} {má|měl[a iobj]}. '

    /* can't talk to yourself */
    cannotTalkToSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't ask yourself about anything */
    cannotAskSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't ask yourself for anything */
    cannotAskSelfForMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't tell yourself about anything */
    cannotTellSelfMsg = 'Mluvením se sebou {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't give yourself something */
    cannotGiveToSelfMsg = 'Podáním {kohočeho dobj} {sám} sobě {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't give something to itself */
    cannotGiveToItselfMsg = 'Podáním {kohočeho dobj} {komučemu dobj} {|[bys]}
        ničeho nedosáh{n[eš]|l[a]}. '

    /* can't show yourself something */
    cannotShowToSelfMsg = 'Ukázáním {kohočeho dobj} {sám} sobě {|[bys]} ničeho
        nedosáh{n[eš]|l[a]}. '

    /* can't show something to itself */
    cannotShowToItselfMsg = 'Ukázáním {kohočeho dobj} {komučemu dobj} {|[bys]}
        ničeho nedosáh{n[eš]|l[a]}. '

    /* can't give/show something to a non-actor */
    cannotGiveToMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {komučemu iobj} nic dát. '
    cannotShowToMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {komučemu iobj} nic ukázat. '

    /* actor isn't interested in something being given/shown */
    notInterestedMsg(actor)
    {
	gMessageParams(actor);
        return '\^' + actor.name + ' vypad{[áš actor]|al[a actor]} nezaujatě. ';
    }

    /* vague ASK/TELL (for ASK/TELL <actor> <topic> syntax errors) */
    askVagueMsg = '<.parser>Příběh tomuto příkazu nerozumí. Použij prosím
        ZEPTEJ SE POSTAVY NA TÉMA.<./parser> '
    tellVagueMsg = '<.parser>Příběh tomuto příkazu nerozumí. Použij prosím
        ŘEKNI POSTAVĚ O TÉMATU.<./parser> '

    /* object cannot hear actor */
    objCannotHearActorMsg(obj)
    {
	gMessageParams(obj);
        return '\^' + obj.name + ' vypad{[áš obj]|al[a obj]}, že {tě/ji}
            neslyš{í|el[a obj]}. ';
    }

    /* actor cannot see object being shown to actor */
    actorCannotSeeMsg(actor, obj)
    {
	gMessageParams(actor, obj);
        return '\^' + actor.name + ' vypad{[áš obj]|al[a obj]}, že ' + obj.nameKohoCo
            + ' nevid{í|ěl[a actor]}. ';
    }

    /* not a followable object */
    notFollowableMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} následovat. '

    /* cannot follow yourself */
    cannotFollowSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} následovat sebe. '

    /* following an object that's in the same location as the actor */
    followAlreadyHereMsg = '{Kdoco dobj} {je} přímo tady. '

    /*
     *   following an object that we *think* is in our same location (in
     *   other words, we're already in the location where we thought we
     *   last saw the object go), but it's too dark to see if that's
     *   really true 
     */
    followAlreadyHereInDarkMsg = '{Kdoco dobj} by tu měl{a dobj} být, ale
        nevid{[íš]|ěl[a] [jsi]} {ho/ji dobj}. '

    /* trying to follow an object, but don't know where it went from here */
    followUnknownMsg = '{[Nejsi/není] si|Nebyl[a] [sis]} jist{ý}, kam {kdoco dobj} odsud {šel dobj}. '

    /*
     *   we're trying to follow an actor, but we last saw the actor in the
     *   given other location, so we have to go there to follow 
     *
     *   Poměrně šroubovaná věta, abychom si vystačili s názvem místnosti.
     */
    cannotFollowFromHereMsg(srcLoc)
    {
        return '\^' + srcLoc.name + ' bylo poslední místo, kde {jsi}
            {kohoco dobj} naposledy viděl{a}. ';
    }

    /* acknowledge a 'follow' for a target that was in sight */
    okayFollowInSightMsg(loc)
    {
        return 'Násled{u[ješ]|oval[a] [jsi]} {kohoco dobj} '
            + loc.objIntoName + '. ';
    }

    /* obj is not a weapon */
    notAWeaponMsg = '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} zaútočit. '

    /* no effect attacking obj */
    uselessToAttackMsg = '\^{kdoco/postava} na {něj/ni dobj} {nemůž[eš]|[jsi]
        nemohl[a]} zaútočit. '

    /* pushing object has no effect */
    pushNoEffectMsg = 'Zmáčknutí {kohočeho dobj} {nemá|nemělo} žádný efekt. '

    /* default 'push button' acknowledgment */
    okayPushButtonMsg = '<q>Cvak.</q> '

    /* lever is already in pushed state */
    alreadyPushedMsg =
        'Už {je|byl[a dobj]} zmáčknut{ý dobj} až na doraz. '

    /* default acknowledgment to pushing a lever */
    okayPushLeverMsg = '{Mačk[áš]|Zmáčkl[a] [jsi]} {kohoco dobj} až na doraz. '

    /* pulling object has no effect */
    pullNoEffectMsg = 'Zatáhnutí za {kohoco dobj} {nemá|nemělo} žádný efekt. '

    /* lever is already in pulled state */
    alreadyPulledMsg =
        'Už {je|byl[a dobj]} zatáhnut{ý dobj} až na doraz. '

    /* default acknowledgment to pulling a lever */
    okayPullLeverMsg =
        '{Táh[neš]|Zatáhl[a] [jsi]} za {kohoco dobj} až na doraz. '

    /* default acknowledgment to pulling a spring-loaded lever */
    okayPullSpringLeverMsg = 'Zatáhl{a} {jsi} za {kohoco dobj}, kter{ý dobj}
        se následně {vrac[íš]|vrátila} do původní polohy, jakmile {jsi}
        {ho/ji dobj} pustil{a}. '

    /* moving object has no effect */
    moveNoEffectMsg = 'Posunutí {kohočeho dobj} by nemělo žádný '
        + 'efekt. '

    /* cannot move object to other object */
    moveToNoEffectMsg = 'Tím {bys} ničeho nedosáhl{a}.  '

    /* cannot push an object through travel */
    cannotPushTravelMsg = 'To by nemělo žádný efekt. '

    /* acknowledge pushing an object through travel */
    okayPushTravelMsg(obj)
    {
        return '<.p>Tlač{[íš]|il[a] [jsi]} ' + obj.nameKohoCo
            + ' do volného prostoru. ';
    }

    /* cannot use object as an implement to move something */
    cannotMoveWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic posunout. '

    /* cannot set object to setting */
    cannotSetToMsg = 'Na {něm/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} nic nastavit. '

    /* invalid setting for generic Settable */
    setToInvalidMsg = '{Kdoco dobj} {nemá|neměl[a dobj]} takové nastavení. '

    /* default 'set to' acknowledgment */
    okaySetToMsg(val)
        { return 'Fajn, {kdoco dobj} {je} {teď|} nastaven{a dobj} na ' + val + '. '; }

    /* cannot turn object */
    cannotTurnMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {jím/jí dobj} otočit. '

    /* must specify setting to turn object to */
    mustSpecifyTurnToMsg = 'Musíš specifikovat polohu, do které {ho/ji dobj}
        {chc[eš]|[jsi] chtěl[a]} {kdoco/postava} natočit. '

    /* cannot turn anything with object */
    cannotTurnWithMsg =
        '{Tím dobj} {nemůž[eš]|[jsi] nemohl[a]} nic otočit. '

    /* invalid setting for dial */
    turnToInvalidMsg = '{Kdoco dobj} {nemá|neměl[a dobj]} takovou polohu. '

    /* default 'turn to' acknowledgment */
    okayTurnToMsg(val)
        { return 'Fajn, {kdoco dobj} {je} {teď|} natočen{a dobj} na ' + val + '. '; }

    /* switch is already on/off */
    alreadySwitchedOnMsg = '{Kdoco dobj} už {je} zapnut{ý}. '
    alreadySwitchedOffMsg = '{Kdoco dobj} už {je} vypnut{ý}. '

    /* default acknowledgment for switching on/off */
    okayTurnOnMsg = 'Fajn, {kdoco dobj} {je} {teď|od té chvíle} zapnut{ý}. '
    okayTurnOffMsg = 'Fajn, {kdoco dobj} {je} {teď|od té chvíle} vypnut{ý}. '

    /* flashlight is on but doesn't light up */
    flashlightOnButDarkMsg = 'Zapnul{a} {jsi} {kohoco dobj}, ale nezdá se, '
        + 'že by se cokoliv stalo. '

    /* default acknowledgment for eating something */
    okayEatMsg = 'Snědl{a} {jsi} {kohoco dobj}. '

    /* object must be burning before doing that */
    mustBeBurningMsg(obj)
    {
        return 'Nejprve {mus[íš]|[jsi] musel[a]} zapálit ' + obj.nameKohoCo
            + ', než {můž[eš]|[jsi] mohl[a]} tohle udělat. ';
    }

    /* match not lit */
    matchNotLitMsg = '{Kdoco dobj} {není} zapálen{ý}. '

    /* lighting a match */
    okayBurnMatchMsg =
        'Škrtnutím {kohočeho dobj} {zapaluj[eš]|[jsi] zapálil[a]} malý plamínek. '

    /* extinguishing a match */
    okayExtinguishMatchMsg = 'Zhas{ín[áš]|il[a] [jsi]} {kohoco dobj} a plamen '
        + 'se rozpl{ývá|ynul} v obláčku dýmu. '

    /* trying to light a candle with no fuel */
    candleOutOfFuelMsg =
        '{Kdoco dobj} {je} vyhořel{ý}, {nemůž[eš]|mohl[a] [jsi]} {ho/ji dobj} zapálit. '

    /* lighting a candle */
    okayBurnCandleMsg = 'Zap{aluj[eš]|álil[a] [jsi]} {kohoco dobj}. '

    /* extinguishing a candle that isn't lit */
    candleNotLitMsg = '{Kdoco dobj} {není} zapálen{ý}. '

    /* extinguishing a candle */
    okayExtinguishCandleMsg = 'Hotovo. '

    /* cannot consult object */
    cannotConsultMsg =
        'To {není|nebylo} něco, s čím {[bys]|[jsi]} se mohl{a} poradit. '

    /* cannot type anything on object */
    cannotTypeOnMsg = 'Na {něm/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} nic napsat. '

    /* cannot enter anything on object */
    cannotEnterOnMsg = 'Na {něm/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} nic zadat. '

    /* cannot switch object */
    cannotSwitchMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} přepnout. '

    /* cannot flip object */
    cannotFlipMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} obrátit. '

    /* cannot turn object on/off */
    cannotTurnOnMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} zapnout. '
    cannotTurnOffMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} vypnout. '

    /* cannot light */
    cannotLightMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} zapálit. '

    /* cannot burn */
    cannotBurnMsg = 'To {není|nebylo} něco, co {|by} mohlo hořet. '
    cannotBurnWithMsg =
        '{Tím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic spálit. '

    /* cannot burn this specific direct object with this specific iobj */
    cannotBurnDobjWithMsg = '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} zapálit '
        + '{kohoco dobj}. '

    /* object is already burning */
    alreadyBurningMsg = '{Kdoco dobj} už {hoří|hořel[a dobj]}. '

    /* cannot extinguish */
    cannotExtinguishMsg = '{Tím dobj} {nemůž[eš]|[jsi] nemohl[a]} nic uhasit. '

    /* cannot pour/pour in/pour on */
    cannotPourMsg = 'To {není} něco, co {[bys]|[jsi]} mohl{a} lít. '
    cannotPourIntoMsg =
        'Do {něho/ní iobj} {nemůž[eš]|[jsi] nemohl[a]} nic nalít. '
    cannotPourOntoMsg =
        'Na {něj/ni iobj} {nemůž[eš]|[jsi] nemohl[a]} nic vylít. '

    /* cannot attach object to object */
    cannotAttachMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} k ničemu připevnit. '
    cannotAttachToMsg =
        'K {němu/ní iobj} {nemůž[eš]|[jsi] nemohl[a]} nic připevnit. '

    /* cannot attach to self */
    cannotAttachToSelfMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} připevnit {kohoco dobj} {sám dobj} k sobě. '

    /* cannot attach because we're already attached to the given object */
    alreadyAttachedMsg = '{Kdoco dobj} už {je} připevněný {k/ke iobj} {komučemu iobj}. '

    /*
     *   dobj and/or iobj can be attached to certain things, but not to
     *   each other 
     */
    wrongAttachmentMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} připevnit {k/ke iobj} {komučemu iobj}. '

    /* dobj and iobj are attached, but they can't be taken apart */
    wrongDetachmentMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} uvolnit od {kohočeho iobj}. '

    /* must detach the object before proceeding */
    mustDetachMsg(obj)
    {
        gMessageParams(obj);
        return 'Nejprve {mus[íš]|[jsi] musel[a]} {kohoco obj} uvolnit než
            {bud[eš] moci|[jsi] mohl[a]} tohle udělat. ';
    }

    /* default message for successful Attachable attachment */
    okayAttachToMsg = 'Hotovo. '

    /* default message for successful Attachable detachment */
    okayDetachFromMsg = 'Hotovo. '

    /* cannot detach object from object */
    cannotDetachMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} oddělat. '
    cannotDetachFromMsg =
        'Z {něho/ní iobj} {nemůž[eš]|[jsi] nemohl[a]} nic oddělat. '

    /* no obvious way to detach a permanent attachment */
    cannotDetachPermanentMsg = 'Ne{ní|bylo} {ti/jí} jasné, jak {ho/ji dobj}
        oddělat. '

    /* dobj isn't attached to iobj */
    notAttachedToMsg = '{Kdoco dobj} k {němu/ní iobj} {není dobj} připevněn. '

    /* breaking object would serve no purpose */
    shouldNotBreakMsg = '{Jeho/její dobj} rozbití {by|} bylo k ničemu. '

    /* cannot cut that */
    cutNoEffectMsg = '{Kdoco iobj} {kohoco dobj} nepřeříz{ne|l[a iobj]}. '

    /* can't use iobj to cut anything */
    cannotCutWithMsg = '{Kýmčím iobj} {|[jsi]} nic
        {nepřeřízn[eš]|nemohl[a] přeříznout}. '

    /* cannot climb object */
    cannotClimbMsg = 'To {není|nebylo} něco, po čem {by|} se dalo šplhat. '

    /* object is not openable/closable */
    cannotOpenMsg = 'To {není|nebylo} něco, co {by|} se dalo otevřít. '
    cannotCloseMsg = 'To {není|nebylo} něco, co {by|} se dalo zavřít. '

    /* already open/closed */
    alreadyOpenMsg = '{Kdoco dobj} už {je} otevřen{ý}. '
    alreadyClosedMsg = '{Kdoco dobj} už {je} zavřen{ý}. '

    /* already locked/unlocked */
    alreadyLockedMsg = '{Kdoco dobj} už {je} zamčen{ý}. '
    alreadyUnlockedMsg = '{Kdoco dobj} už {je} odemčen{ý}. '

    /* cannot look in container because it's closed */
    cannotLookInClosedMsg = '{Kdoco dobj} {je} zavřen{ý}. '

    /* object is not lockable/unlockable */
    cannotLockMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} zamknout. '
    cannotUnlockMsg =
        'To {není|nebylo} něco, co {[bys]|[jsi]} mohl{a} odemknout. '

    /* attempting to open a locked object */
    cannotOpenLockedMsg = '{Kdoco dobj} {vypad[áš dobj]|vypadal[a dobj]}
        zamčeně. '

    /* object requires a key to unlock */
    unlockRequiresKeyMsg = 'Vypad{á|alo} to, že
        {potřebuj[eš]|[jsi] potřeboval[a]} k odemčení {kohočeho dobj} klíč. '

    /* object is not a key */
    cannotLockWithMsg =
        '{Kdoco iobj} nevypad{[áš iobj]|al[a iobj]} vhodně na zamykání. '
    cannotUnlockWithMsg =
        '{Kdoco iobj} nevypad{[áš iobj]|al[a iobj]} vhodně na odemykání. '

    /* we don't know how to lock/unlock this */
    unknownHowToLockMsg =
        '{Není|Nebylo} {ti/jí} jasné, jak zamknout {kohoco dobj}. '
    unknownHowToUnlockMsg =
        '{Není|Nebylo} {ti/jí} jasné, jak odemknout {kohoco dobj}. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLockMsg = '{Kdoco iobj} nepas{uje|oval[a iobj]} do zámku. '

    /* found key on keyring */
    foundKeyOnKeyringMsg(ring, key)
    {
        gMessageParams(ring, key);
        return 'Zkusil{a} {jsi} každý klíč na {komčem ring} a zjistil{a} {jsi},
            že {kdoco key} pas{uje|oval[a key]} do zámku. ';
    }

    /* failed to find a key on keyring */
    foundNoKeyOnKeyringMsg(ring)
    {
        gMessageParams(ring);
        return 'Zk{ouš[íš]|usil[a] [jsi]} každý klíč na {komčem ring}, ale
            nem{ůž[eš]|ohl[a] [jsi]} najít žádný, který by pasoval do zámku. ';
    }

    /* not edible/drinkable */
    cannotEatMsg = '{Kdoco dobj} nevypad{[áš dobj]|al[a dobj]} poživatelně. '
    cannotDrinkMsg = '{Kdoco dobj} nevypad{[áš dobj]|al[a dobj]} pitně. '

    /* cannot clean object */
    cannotCleanMsg = 'Nev{[íš]|ěděl[a] [jsi]} jak {ho/ji dobj} vyčistit. '
    cannotCleanWithMsg = '{Tím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic vyčistit. '

    /* cannot attach key (dobj) to (iobj) */
    cannotAttachKeyToMsg =
        'Na {něj/ni iobj} {nemůž[eš]|[jsi] nemohl[a]} {kohoco dobj} navéknout. '

    /* actor cannot sleep */
    cannotSleepMsg = 'Nech{ce|tělo} se {ti/jí} spát. '

    /* cannot sit/lie/stand/get on/get out of */
    cannotSitOnMsg =
        'To {není|nebylo} něco, na čem by se dalo sedět. '
    cannotLieOnMsg =
        'To {není|nebylo} něco, na čem by se dalo ležet. '
    cannotStandOnMsg = 'Na {něm/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} stát. '
    cannotBoardMsg = 'Na {něj/ni dobj} {nemůž[eš]|[jsi] nemohl[a]} nastoupit. '
    cannotUnboardMsg = 'Z {něho/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} vystoupit. '
    cannotGetOffOfMsg = 'Z {něho/ní dobj} {nemůž[eš]|[jsi] nemohl[a]} sestoupit. '

    /* standing on a PathPassage */
    cannotStandOnPathMsg = 'Pokud {chc[eš]|[jsi] chtěl[a]} následovat
        {kohoco dobj}, odpověz ne. '

    /* cannot sit/lie/stand on something being held */
    cannotEnterHeldMsg = 'To {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud {|[jsi]}
        {kohoco dobj} drž{[íš]|el[a]}. '

    /* cannot get out (of current location) */
    cannotGetOutMsg = '{[Nejsi/není]|Nebyl[a] [jsi]} v ničem, co {[bys]|[jsi]}
        mohl{a} opustit. '

    /* actor is already in a location */
    alreadyInLocMsg = 'Už {jsi/je} {v dobj}. '

    /* actor is already standing/sitting on/lying on */
    alreadyStandingMsg = '{Ty} už {stoj[íš]|[jsi] stál[a]}. '
    alreadyStandingOnMsg = 'Už {|[jsi]} st{oj[íš]|ál[a]} {v dobj}. '
    alreadySittingMsg = '{Ty} už {sed[íš]|[jsi] seděl[a]}. '
    alreadySittingOnMsg = 'Už {|[jsi]} sed{[íš]|ěl[a]} {v dobj}. '
    alreadyLyingMsg = '{Ty} už {lež[íš]|[jsi] ležel[a]}. '
    alreadyLyingOnMsg = 'Už {|[jsi]} lež{[íš]|el[a]} {v dobj}. '

    /* getting off something you're not on */
    notOnPlatformMsg = '{Nejsi/není} {v dobj}. '

    /* no room to stand/sit/lie on dobj */
    noRoomToStandMsg =
        '{Na dobj} {není|nebylo} pro {tebe/ni} dost místa ke stání. '
    noRoomToSitMsg =
        '{Na dobj} {není|nebylo} pro {tebe/ni} dost místa k sezení. '
    noRoomToLieMsg =
        '{Na dobj} {není|nebylo} pro {tebe/ni} dost místa k ležení. '

    /* default report for standing up/sitting down/lying down */
    /*
     *   Trochu špinavý trik - v posture.participle je šablona parametrizovaná
     *   poserem, ale tady nemam předaného actora, abych ho nabídl, tak objekt
     *   ze šablony zahodím a tím se bude vztahovat automaticky k actorovi.
     */
    okayPostureChangeMsg(posture)
    {
        return 'Fajn, {teď|} ' + posture.participle.findReplace(' poser', '',
            ReplaceAll) + ' {|[jsi]}. ';
    }

    /* default report for standing/sitting/lying in/on something */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        return 'Fajn, {teď|} ' + posture.participle.findReplace(' poser', '',
            ReplaceAll) + ' {|[jsi]} {v obj}. ';
    }

    /* default report for getting off of a platform */
    okayNotStandingOnMsg = 'Fajn, už {[nejsi/není]|[jsi] nebyl[a]} {v dobj}. '

    /* cannot fasten/unfasten */
    cannotFastenMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} zapnout. '
    cannotFastenToMsg =
        '{K/ke iobj} {komučemu iobj} {nemůž[eš]|[jsi] nemohl[a]} nic připoutat. '
    cannotUnfastenMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {kohoco dobj} rozepnout. '
    cannotUnfastenFromMsg =
        'Od {kohočeho iobj} {nemůž[eš]|[jsi] nemohl[a]} nic odpoutat. '

    /* cannot plug/unplug */
    cannotPlugInMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        zapojit. '
    cannotPlugInToMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak cokoliv
        zapojit do {kohočeho iobj}. '
    cannotUnplugMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        vypojit. '
    cannotUnplugFromMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak cokoliv
        vypojit {z/ze iobj} {kohočeho iobj}. '

    /* cannot screw/unscrew */
    cannotScrewMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak {kohoco dobj}
        zašroubovat. '
    cannotScrewWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic přišroubovat. '
    cannotUnscrewMsg = 'Nevid{[íš]|ěl[a] [jsi]} žádnou možnost, jak
        {kohoco dobj} vyšroubovat. '
    cannotUnscrewWithMsg =
        '{Kýmčím iobj} {nemůž[eš]|[jsi] nemohl[a]} nic odšroubovat. '

    /* cannot enter/go through */
    cannotEnterMsg =
        'To {není|nebylo} něco, do čeho {[bys]|[jsi]} mohl{a} vstoupit. '
    cannotGoThroughMsg =
        'To {není|nebylo} něco, čím {[bys]|[jsi]} mohl{a} projít. '

    /* can't throw something at itself */
    cannotThrowAtSelfMsg =
        'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} hodit na sebe sama. '

    /* can't throw something at an object inside itself */
    cannotThrowAtContentsMsg = 'Než {bud[eš] moci|[jsi] mohl[a]} tohle udělat,
        {mus[íš]|musel[a] [jsi]} odstranit {kohoco iobj} {z/ze dobj} {kohočeho dobj}. '

    /* can't throw through a sense connector */
    cannotThrowThroughMsg(target, loc)
    {
        gMessageParams(target, loc);
        return '{You/he} {cannot} throw anything through {the loc/him}. ';
    }

    /* shouldn't throw something at the floor */
    shouldNotThrowAtFloorMsg =
        'Raději {[bys]|[jsi]} {ho/ji dobj} měl{a} prostě položit. '

    /* THROW <obj> <direction> isn't supported; use THROW AT instead */
    dontThrowDirMsg =
        '<.parser>Musíš být specifičtější, po čem {chc[eš]|[jsi] chtěl[a]}
            {kdoco/postava} {kohoco dobj} vrhnout.<./parser> '

    /* thrown object bounces off target (short report) */
    throwHitMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhnul[a projectile]}
            {kohoco target} bez znatelného efektu. ';
    }

    /* thrown object lands on target */
    throwFallMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhl[a projectile]}
            {kohoco target}. ';
    }

    /* thrown object bounces off target and falls to destination */
    throwHitFallMsg(projectile, target, dest)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} zas{ahuje|áhl[a projectile]} bez znatelného
            efektu {kohoco target} a {padá|spadl[a]} ' + dest.putInName + '. ';
    }

    /* thrown object falls short of distant target (sentence prefix only) */
    throwShortMsg(projectile, target)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} nedoletěl{a projectile} až {k/ke target}
            {komučemu target}. ';
    }

    /* thrown object falls short of distant target */
    throwFallShortMsg(projectile, target, dest)
    {
        gMessageParams(projectile, target);
        return '{Kdoco projectile} nedoletěl{a projectile} až {k/ke target}
            {komučemu target} a {padá|spadla} ' + dest.putInName + '. ';
    }

    /* target catches object */
    throwCatchMsg(obj, target)
    {
        gMessageParams(obj, target);
        return '{Kdoco target} chyt{[áš target]|il[a target]} {kohoco obj}. ';
    }

    /* we're not a suitable target for THROW TO (because we're not an NPC) */
    cannotThrowToMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} nic hodit na {kohoco iobj}. '

    /* target does not want to catch anything */
    willNotCatchMsg(catcher)
    {
        gMessageParams(catcher);
        return '{Kdoco catcher} nevypad{[áš catcher]|al[a catcher]}, že {by|}
            chtěl{a catcher} něco chytat. ';
    }

    /* cannot kiss something */
    cannotKissMsg = 'Políbení {kohočeho dobj} {nemá|nemělo} znatelný efekt. '

    /* person uninterested in being kissed */
    cannotKissActorMsg
        = 'To by se {komučemu dobj} nejspíš nelíbilo. '

    /* cannot kiss yourself */
    cannotKissSelfMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} políbit sebe. '

    /* it is now dark at actor's location */
    newlyDarkMsg = 'Najednou tu {je|byla} naprostá tma. '
;

/*
 *   Non-player character verb messages.  By default, we inherit all of
 *   the messages defined for the player character, but we override some
 *   that must be rephrased slightly to make sense for NPC's.
 */
npcActionMessages: playerActionMessages
    /* "wait" */
    timePassesMsg = '{Kdoco} ček{[áš]|al[a]}... '

    /* trying to move a Fixture/Immovable */
    cannotMoveFixtureMsg = '{Kdoco} nem{ůže|ohl[a]} pohnout {kýmčím dobj}. '
    cannotMoveImmovableMsg = '{Kdoco} nem{ůže|ohl[a]} pohnout {kýmčím dobj}. '

    /* trying to take/move/put a Heavy object */
    cannotTakeHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj}, nezdvih{n[eš]|l[a]} {|[bys]} {tebe/ji dobj}. '
    cannotMoveHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj}, neposu{n[eš]|l[a]} {|[bys]} {tebe/ji dobj}. '
    cannotPutHeavyMsg =
        '{Kdoco dobj} {je} pro {kohoco} příliš těžk{ý dobj}, nepoh{n[eš]|l[a]} {|[bys]} s {tebou/ním dobj}. '

    /* trying to move a component object */
    cannotMoveComponentMsg(loc)
    {
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco dobj}
            {je} součástí ' + loc.nameKohoCeho + '. ';
    }

    /* default successful 'take' response */
    okayTakeMsg = '{Kdoco} {bere|vzal[a]} {kohoco dobj}. '

    /* default successful 'drop' response */
    okayDropMsg = '{Kdoco} {pokádá|položil[a]} {kohoco dobj}. '

    /* default successful 'put in' response */
    okayPutInMsg = '{Kdoco} {vkládá|vložil[a]} {kohoco dobj} do {kohočeho iobj}. '

    /* default successful 'put on' response */
    okayPutOnMsg = '{Kdoco} {pokádá|položil[a]} {kohoco dobj} na {kohoco iobj}. '

    /* default successful 'put under' response */
    okayPutUnderMsg =
        '{Kdoco} {dává|dal[a]} {kohoco dobj} pod {kohoco iobj}. '

    /* default successful 'put behind' response */
    okayPutBehindMsg =
        '{Kdoco} {dává|dal[a]} {kohoco dobj} za {kohoco iobj}. '

    /* default succesful response to 'wear obj' */
    okayWearMsg =
        '{Kdoco} {bere|vzal[a]} {kohoco dobj} na sebe. '

    /* default successful response to 'doff obj' */
    okayDoffMsg = '{Kdoco} {sundavá|sundal[a]} {kohoco dobj}. '

    /* default successful responses to open/close */
    okayOpenMsg = '{Kdoco} otev{írá|řel[a]} {kohoco dobj}. '
    okayCloseMsg = '{Kdoco} zav{írá|řel[a]} {kohoco dobj}. '

    /* default successful responses to lock/unlock */
    okayLockMsg = '{Kdoco} zam{yká|kl[a]} {kohoco dobj}. '
    okayUnlockMsg = '{Kdoco} odem{yká|kl[a]} {kohoco dobj}. '

    /* push/pull/move with no effect */
    pushNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} tlačit {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    pullNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} táhnout {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    moveNoEffectMsg = '{Kdoco} {zkouší|zkusil[a]} posunout {kohoco dobj}, '
        + 'ale bez viditelného úspěchu. '
    moveToNoEffectMsg = '{Kdoco} {nechává|zanechal[a]} {kohoco dobj}, '
        + 'kde byl{a dobj}. '

    whereToGoMsg =
        'Měl{a} {[bys]|[jsi]} říci, kterým směrem {má|měl[a]} {kdoco} jít. '

    /* object is too large for container */
    tooLargeForContainerMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se do
            {kohočeho cont} neve{jde|[šel obj]}. ';
    }

    /* object is too large for underside */
    tooLargeForUndersideMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se pod
            {kohoco cont} neve{jde|[šel obj]}. ';
    }

    /* object is too large to fit behind something */
    tooLargeForRearMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco obj} se za
            {kohoco cont} neve{jde|[šel obj]}. ';
    }

    /* container doesn't have room for object */
    containerTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože {kdoco cont} {je}
            už moc plný a {kdoco obj} se do {něho/ní cont} nevejde. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFullMsg(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'To {kdoco} nem{ůže|ohl[a]} udělat, protože na {komčem cont}
            pro {kohoco obj} {není|nebylo} místo. ';
    }

    /* the dobj doesn't fit on this keyring */
    objNotForKeyringMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} nepas{uje|oval[a dobj]} do {kohočeho iobj}. '

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotInMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} v {něm/ní iobj} {není}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOnMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} na {něm/ní iobj} {není}. '

    /* taking dobj under something, but dobj isn't under iobj */
    takeFromNotUnderMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} pod {ním/ní iobj} {není}. '

    /* taking dobj from behind something, but dobj isn't behind iobj */
    takeFromNotBehindMsg = 'To {kdoco} nem{ůže|ohl[a]} udělat, protože
        {kdoco dobj} {není} za {ním/ní iobj}. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHereMsg = '{Kdoco} nem{[áš]|ěl[a]} kam vyskočit. '

    /* should not break object */
    shouldNotBreakMsg = '{Kdoco} {ho/ji dobj} nech{ce|těl[a]} rozbít. '

    /* report for standing up/sitting down/lying down */
    okayPostureChangeMsg(posture)
        { return '{Kdoco} ' + posture.msgVerbI + '. '; }

    /*
     *   report for standing/sitting/lying in/on something
     *
     *   >sally, stoupni si na židli
     *   Sally si stoupá na dřevěnou židli.
     */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        return '{Kdoco} si ' + posture.msgVerbT + ' {do obj}. ';
    }

    /* report for getting off a platform */
    okayNotStandingOnMsg = '{Kdoco} sest{upuje|oupil[a]} {z dobj}. '

    /* default 'turn to' acknowledgment */
    okayTurnToMsg(val)
        { return '{Kdoco} {otáčí|natočil[a]} {kohoco dobj} na ' + val + '. '; }

    /* default 'push button' acknowledgment */
    okayPushButtonMsg = '{Kdoco} {mačká|zmáčkl[a]} {kohoco dobj}. '

    /* default acknowledgment for switching on/off */
    okayTurnOnMsg = '{Kdoco} zap{íná|nul[a]} {kohoco dobj}. '
    okayTurnOffMsg = '{Kdoco} vyp{íná|nul[a]} {kohoco dobj}. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLockMsg = '{Kdoco} zk{ouší|usil[a]} {kohoco iobj}, ale '
        + 'do zámku nepas{uje|oval[a iobj]}. '

    /* acknowledge entering "follow" mode */
    /* TODO: Tady by mohla být potřeba vokalizace. */
    okayFollowModeMsg = '<q>Dobrá, půjdu {s/se dobj} {tebou/ním dobj}.</q> '

    /* note that we're already in "follow" mode */
    alreadyFollowModeMsg = '<q>Já už přeci jdu {s/se dobj} {tebou/ním dobj}.</q> '

    /* extinguishing a candle */
    okayExtinguishCandleMsg = '{Kdoco} zh{áší|asil[a]} {kohoco dobj}. '

    /* acknowledge attachment */
    okayAttachToMsg =
        '{Kdoco} přip{íná|nul[a]} {kohoco dobj} {k/ke iobj} {komučemu iobj}. '

    /* acknowledge detachment */
    okayDetachFromMsg =
        '{Kdoco} od{píná|epnul[a]} {kohoco dobj} od {kohočeho iobj}. '

    /*
     *   the PC's responses to conversational actions applied to oneself
     *   need some reworking for NPC's 
     */
    cannotTalkToSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotAskSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotAskSelfForMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotTellSelfMsg = '{Kdoco} ničeho nedosáhne, když si promluví se sebou. '
    cannotGiveToSelfMsg = '{Kdoco} ničeho nedosáhne, když dá {kohoco dobj}
        {sám} sobě. '
    cannotShowToSelfMsg = '{Kdoco} ničeho nedosáhne, když ukáže {kohoco dobj}
        {sám} sobě. '
;

/* ------------------------------------------------------------------------ */
/*
 *   Standard tips
 */

scoreChangeTip: Tip
    "Pokud bys příště raději neslyšel{a} o změnách skóre, napiš
    <<aHref('vypni upozorňování', 'VYPNI UPOZORŇOVÁNÍ', 'Vypnout upozorňování
    na změnu skóre')>>."
;

footnotesTip: Tip
    "Číslo v [hranatých závorkách] jako to výše odkazuje na poznámku pod čarou,
       která se dá přečíst zadáním POZNÁMKA a čísla, např.:
       <<aHref('poznámka 1', 'POZNÁMKA 1', 'Ukázat poznámku č. [1]')>>.
       Poznámky obvykle obsahují další informace o pozadí příběhu, které by
       mohly být zajímavé, ale nejsou nezbytně nutné pro samotný příběh. Pokud
       si nepřeješ vidět poznámky, můžeš je omezit zadáním <<aHref('poznámky',
       'POZNÁMKY', 'Změnit zobrazování poznámek')>>."
;

oopsTip: Tip
    "Pokud jste se omylem překlepli, můžete to teď napravit zadáním OPRAVA
    (nebo jen O) následovaném opraveným slovem. Kdykoliv vás příběh upozorní
    na neznámé slovo, můžete se opravit použitím slova OPRAVA jako dalšího
    zadaného příkazu."
;

fullScoreTip: Tip
    "Chceš-li vidět detailní informace o dosaženém skóre, napiš
        <<aHref('detailní skóre', 'DETAILNÍ SKÓRE')>>."
;

exitsTip: Tip
    "Vypisování směrů můžeš ovlivnit příkazem SMĚRY. <<aHref('směry v záhlaví',
        'SMĚRY V ZÁHLAVÍ', 'Zapnout zobrazení směrů ve stavovém řádku')>>
        zobrazí směry v záhlaví na stavovém řádku, <<aHref('směry v popisu',
        'SMĚRY V POPISU', 'Vypisovat směry v popisu místnosti')>> zobrazí možné
        směry jako součást popisu každé místnosti, <<aHref('zapnout směry',
        'ZAPNOUT SMĚRY', 'Zapnout výpisy směrů na všech místech')>> zobrazí
        směry v obou místech a <<aHref('vypnout směry', 'VYPNOUT SMĚRY',
        'Vypnout všechny zobrazení směrů')>> vypne zobrazení směrů úplně."
;

undoTip: Tip
    "Pokud se příběh nevyvinul tak, jak jsi čekal, nezapomeň, že můžeš vzít
    svůj tah zpátky zadáním <<aHref('vrať tah', 'VRAŤ TAH', 'Vrátit zpátky
    poslední zadaný příkaz')>>. Vrácení tahu můžeš zopakovat i několikrát za
    sebou a vrátit tak více tahů v řadě. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Listers
 */

/*
 *   The basic "room lister" object - this is the object that we use by
 *   default with showList() to construct the listing of the portable
 *   items in a room when displaying the room's description.  
 */
roomLister: Lister
    /* show the prefix/suffix in wide mode */
    showListPrefixWide(itemCount, pov, parent, selector) { "Vid{[íš]|ěl[a] [jsi]} tu "; }
    showListSuffixWide(itemCount, pov, parent, selector) { ". "; }

    /* show the tall prefix */
    showListPrefixTall(itemCount, pov, parent, selector)
        { "Vid{[íš]|ěl[a] [jsi]} tu:"; }
;

/*
 *   The basic room lister for dark rooms 
 */
darkRoomLister: Lister
    showListPrefixWide(itemCount, pov, parent, selector)
        { "V přítmí {vid[íš]|[jsi] viděl[a]} "; }

    showListSuffixWide(itemCount, pov, parent, selector) { ". "; }

    showListPrefixTall(itemCount, pov, parent, selector)
        { "V přítmí {vid[íš]|[jsi] viděl[a]}:"; }
;

/*
 *   A "remote room lister".  This is used to describe the contents of an
 *   adjoining room.  For example, if an actor is standing in one room,
 *   and can see into a second top-level room through a window, we'll use
 *   this lister to describe the objects the actor can see through the
 *   window. 
 */
class RemoteRoomLister: Lister
    construct(room) { remoteRoom = room; }
    
    showListPrefixWide(itemCount, pov, parent, selector)
        { "\^<<remoteRoom.inRoomName(pov)>> {vid[íš]|[jsi] viděl[a]} "; }
    showListSuffixWide(itemCount, pov, parent, selector)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent, selector)
        { "\^<<remoteRoom.inRoomName(pov)>> {vid[íš]|[jsi] viděl[a]}:"; }

    /* the remote room we're viewing */
    remoteRoom = nil
;

/*
 *   A simple customizable room lister.  This can be used to create custom
 *   listers for things like remote room contents listings.  We act just
 *   like any ordinary room lister, but we use custom prefix and suffix
 *   strings provided during construction.  
 */
class CustomRoomLister: Lister
    construct(prefix, suffix)
    {
        prefixStr = prefix;
        suffixStr = suffix;
    }

    showListPrefixWide(itemCount, pov, parent, selector) { "<<prefixStr>> "; }
    showListSuffixWide(itemCount, pov, parent, selector) { "<<suffixStr>> "; }
    showListPrefixTall(itemCount, pov, parent, selector) { "<<prefixStr>>:"; }

    /* our prefix and suffix strings */
    prefixStr = nil
    suffixStr = nil
;

/*
 *   Single-list inventory lister.  This shows the inventory listing as a
 *   single list, with worn items mixed in among the other inventory items
 *   and labeled "(being worn)".  
 */
actorSingleInventoryLister: InventoryLister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "{Kdoco parent} nes{e|l[a parent]} ";
    }

    showListSuffixWide(itemCount, pov, parent, selector)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "{Kdoco parent} nes{e|l[a parent]}:";
    }

    /* TODO: kterí */
    showListContentsPrefixTall(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "{Kdoco parent}, kter{ý} nes{e|l[a parent]}:";
    }

    showListEmpty(pov, parent)
    {
        gMessageParams(parent);
        "{Kdoco parent} {má|měl[a parent]} prázdné ruce. ";
    }
;

/*
 *   Standard inventory lister for actors - this will work for the player
 *   character and NPC's as well.  This lister uses a "divided" format,
 *   which segregates the listing into items being carried and items being
 *   worn.  We'll combine the two lists into a single sentence if the
 *   overall list is short, otherwise we'll show two separate sentences for
 *   readability.  
 */
actorInventoryLister: DividedInventoryLister
    /*
     *   Show the combined inventory listing, putting together the raw
     *   lists of the items being carried and the items being worn. 
     */
    showCombinedInventoryList(parent, carrying, wearing)
    {
        /* if one or the other sentence is empty, the format is simple */
        if (carrying == '' && wearing == '')
        {
            /* the parent is completely empty-handed */
            showInventoryEmpty(parent);
        }
        else if (carrying == '')
        {
            /* the whole list is being worn */
            showInventoryWearingOnly(parent, wearing);
        }
        else if (wearing == '')
        {
            /* the whole list is being carried */
            showInventoryCarryingOnly(parent, carrying);
        }
        else
        {
            /*
             *   Both listings are populated.  Count the number of
             *   comma-separated or semicolon-separated phrases in each
             *   list.  This will give us an estimate of the grammatical
             *   complexity of each list.  If we have very short lists, a
             *   single sentence will be easier to read; if the lists are
             *   long, we'll show the lists in separate sentences.  
             */
            if (countPhrases(carrying) + countPhrases(wearing)
                <= singleSentenceMaxNouns)
            {
                /* short enough: use a single-sentence format */
                showInventoryShortLists(parent, carrying, wearing);
            }
            else
            {
                /* long: use a two-sentence format */
                showInventoryLongLists(parent, carrying, wearing);
            }
        }
    }

    /*
     *   Count the noun phrases in a string.  We'll count the number of
     *   elements in the list as indicated by commas and semicolons.  This
     *   might not be a perfect count of the actual number of noun phrases,
     *   since we could have commas setting off some other kind of clauses,
     *   but it nonetheless will give us a good estimate of the overall
     *   complexity of the text, which is what we're really after.  The
     *   point is that we want to break up the listings if they're long,
     *   but combine them into a single sentence if they're short.  
     */
    countPhrases(txt)
    {
        local cnt;
        
        /* if the string is empty, there are no phrases at all */
        if (txt == '')
            return 0;

        /* a non-empty string has at least one phrase */
        cnt = 1;

        /* scan for commas and semicolons */
        for (local startIdx = 1 ;;)
        {
            local idx;
            
            /* find the next phrase separator */
            idx = rexSearch(phraseSepPat, txt, startIdx);

            /* if we didn't find it, we're done */
            if (idx == nil)
                break;

            /* count it */
            ++cnt;

            /* continue scanning after the separator */
            startIdx = idx[1] + idx[2];
        }

        /* return the count */
        return cnt;
    }

    phraseSepPat = static new RexPattern(',(?! a )|;| a |<rparen>')

    /*
     *   Once we've made up our mind about the format, we'll call one of
     *   these methods to show the final sentence.  These are all separate
     *   methods so that the individual formats can be easily tweaked
     *   without overriding the whole combined-inventory-listing method. 
     */
    showInventoryEmpty(parent)
    {
        /* empty inventory */
        "{Nem[áš]|Neměl[a] [jsi]} u sebe nic. ";
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "Nic {nenes[eš]|[jsi] nenesl[a]}, {m[áš]|měl[a] [jsi]} ale na sobě <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        /* short lists - combine carried and worn in a single sentence */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>> a na sobě {m[áš]|[jsi] měl[a]} <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        /* long lists - show carried and worn in separate sentences */
        "{Nes[eš]|Nesl[a] [jsi]} <<carrying>>.
        Na sobě {m[áš]|[jsi] měl[a]} <<wearing>>. ";
    }

    /*
     *   For 'tall' listings, we'll use the standard listing style, so we
     *   need to provide the framing messages for the tall-mode listing.  
     *
     *   TODO: kterí
     */
    showListPrefixTall(itemCount, pov, parent, selector)
        { "U sebe {m[áš]|[jsi] měl[a]}:"; }
    showListContentsPrefixTall(itemCount, pov, parent, selector)
        { "<<buildSynthParam('Kdoco', parent)>>, kter{ý parent} nes{e|l[a parent]}:"; }
    showListEmpty(pov, parent)
        { "<<buildSynthParam('Kdoco', parent)>> {má|měl[a parent]} prázdné ruce. "; }
;

/*
 *   Special inventory lister for non-player character descriptions - long
 *   form lister.  This is used to display the inventory of an NPC as part
 *   of the full description of the NPC.
 *   
 *   This long form lister is meant for actors with lengthy descriptions.
 *   We start the inventory listing on a new line, and use the actor's
 *   full name in the list preface.  
 */
actorHoldingDescInventoryListerLong: actorInventoryLister
    showInventoryEmpty(parent)
    {
        /* empty inventory - saying nothing in an actor description */
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "<.p><<buildSynthParam('Kdoco', parent)>> <<tSel('má', 'měl'
            + buildSynthParam('a', parent))>> na sobě <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "<.p><<buildSynthParam('Kdoco', parent)>> <<tSel('nes'
            + buildSynthParam('eš', parent), 'nesl'
            + buildSynthParam('a', parent))>> <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* short lists - combine carried and worn in a single sentence */
        "<.p><<buildParam('Kdoco', nm)>> <<tSel('nes' + buildSynthParam('eš',
            nm), 'nesl' + buildSynthParam('a', nm))>> <<carrying>> a na sobě
            <<tSel('m' + buildSynthParam('áš', nm), 'měl' + buildSynthParam('a',
            nm))>> <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* long lists - show carried and worn in separate sentences */
        "<.p><<buildParam('Kdoco', nm)>> <<tSel('nes' + buildSynthParam('eš',
            nm), 'nesl' + buildSynthParam('a', nm))>> <<carrying>>. Na sobě
            <<tSel('m' + buildSynthParam('áš', nm), 'měl' + buildSynthParam('a',
            nm))>> <<wearing>>. ";
    }
;

/* short form of non-player character description inventory lister */
actorHoldingDescInventoryListerShort: actorInventoryLister
    showInventoryEmpty(parent)
    {
        /* empty inventory - saying nothing in an actor description */
    }
    showInventoryWearingOnly(parent, wearing)
    {
        /* we're carrying nothing but wearing some items */
        "<<buildSynthParam('Kdoco', parent)>> <<tSel('má', 'měl'
            + buildSynthParam('a', parent))>> na sobě <<wearing>>. ";
    }
    showInventoryCarryingOnly(parent, carrying)
    {
        /* we have only carried items to report */
        "<<buildSynthParam('Kdoco', parent)>> <<tSel('nes'
            + buildSynthParam('eš', parent), 'nesl'
            + buildSynthParam('a', parent))>> <<carrying>>. ";
    }
    showInventoryShortLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* short lists - combine carried and worn in a single sentence */
        "<<buildParam('Kdoco', nm)>> <<tSel('nes' + buildSynthParam('eš',
            nm), 'nesl' + buildSynthParam('a', nm))>> <<carrying>> a na sobě
            <<tSel('m' + buildSynthParam('áš', nm), 'měl' + buildSynthParam('a',
            nm))>> <<wearing>>. ";
    }
    showInventoryLongLists(parent, carrying, wearing)
    {
        local nm = gSynthMessageParam(parent);

        /* long lists - show carried and worn in separate sentences */
        "<<buildParam('Kdoco', nm)>> <<tSel('nes' + buildSynthParam('eš',
            nm), 'nesl' + buildSynthParam('a', nm))>> <<carrying>>. Na sobě
            <<tSel('m' + buildSynthParam('áš', nm), 'měl' + buildSynthParam('a',
            nm))>> <<wearing>>. ";
    }
;

/*
 *   Base contents lister for things.  This is used to display the contents
 *   of things shown in room and inventory listings; we subclass this for
 *   various purposes 
 *
 *   TODO: kterí
 */
class BaseThingContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "\^<<parent.name>> obsah{uje|oval[a parent]} ";
    }
    showListSuffixWide(itemCount, pov, parent, selector)
        { ". "; }
    showListPrefixTall(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "\^<<parent.name>> obsah{uje|oval[a parent]}:";
    }
    showListContentsPrefixTall(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "<<parent.name>>, kter{ý parent} obsah{uje|oval[a parent]}:";
    }
;

/*
 *   Contents lister for things.  This is used to display the second-level
 *   contents lists for things listed in the top-level list for a room
 *   description, inventory listing, or object description.  
 */
thingContentsLister: ContentsLister, BaseThingContentsLister
;

/*
 *   Contents lister for descriptions of things - this is used to display
 *   the contents of a thing as part of the long description of the thing
 *   (in response to an "examine" command); it differs from a regular
 *   thing contents lister in that we use a pronoun to refer to the thing,
 *   rather than its full name, since the full name was probably just used
 *   in the basic long description.  
 */
thingDescContentsLister: DescContentsLister, BaseThingContentsLister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "\^Obsah{uje|oval[a parent]} ";
    }
;

/*
 *   Contents lister for openable things.
 */
openableDescContentsLister: thingDescContentsLister
    showListEmpty(pov, parent)
    {
        "\^<<parent.openStatus>>. ";
    }
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);
        "\^<<parent.openStatus>> a obsah{uje|oval[a parent]} ";
    }
;

/*
 *   Base contents lister for "LOOK <PREP>" commands (LOOK IN, LOOK UNDER,
 *   LOOK BEHIND, etc).  This can be subclasses for the appropriate LOOK
 *   <PREP> command matching the container type - LOOK UNDER for
 *   undersides, LOOK BEHIND for rear containers, etc.  To use this class,
 *   combine it via multiple inheritance with the appropriate
 *   Base<Prep>ContentsLister for the preposition type.  
 */
class LookWhereContentsLister: DescContentsLister
    showListEmpty(pov, parent)
    {
        /* show a default message indicating the surface is empty */
        gMessageParams(parent);
        defaultDescReport('\^' + parent.objInPrep + ' {komčem parent}
            {|[jsi]} nic nevid{[íš]|ěl[a]}. ');
    }
;

/*
 *   Contents lister for descriptions of things whose contents are
 *   explicitly inspected ("look in").  This differs from a regular
 *   contents lister in that we explicitly say that the object is empty if
 *   it's empty.
 */
thingLookInLister: LookWhereContentsLister, BaseThingContentsLister
    showListEmpty(pov, parent)
    {
        /*
         *   Indicate that the list is empty, but make this a default
         *   descriptive report.  This way, if we report any special
         *   descriptions for items contained within the object, we'll
         *   suppress this default description that there's nothing to
         *   describe, which is clearly wrong if there are
         *   specially-described contents. 
         */
        gMessageParams(pov, parent);
        defaultDescReport('Nevid{[íš pov]|ěl[a pov] [jsi pov]} '
            + parent.objInPrep + ' {komčem parent} nic neobvyklého. ');
    }
;

/*
 *   Default contents lister for newly-revealed objects after opening a
 *   container.  
 */
openableOpeningLister: BaseThingContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        /*
         *   This list is, by default, generated as a replacement for the
         *   default "Opened" message in response to an OPEN command.  We
         *   therefore need different messages for PC and NPC actions,
         *   since this serves as the description of the entire action.
         *   
         *   Note that if you override the Open action response for a given
         *   object, you might want to customize this lister as well, since
         *   the message we generate (especially for an NPC action)
         *   presumes that we'll be the only response the command.  
         */
        gMessageParams(pov, parent);
        if (pov.isPlayerChar())
            "Otevření {kohočeho parent} odhalilo ";
        else
            "{Kdoco pov} otevřel[a pov] {kohoco parent} a odhalil[a pov] ";
    }
;

/*
 *   Base contents lister.  This class handles contents listings for most
 *   kinds of specialized containers - Surfaces, RearConainers,
 *   RearSurfaces, and Undersides.  The main variation in the contents
 *   listings for these various types of containers is simply the
 *   preposition that's used to describe the relationship between the
 *   container and the contents, and for this we can look to the objInPrep
 *   property of the container.
 *
 *   V češtině v minulém čase se musí rozlišit ještě rod objektu, resp. prvního
 *   objektu v seznamu více objektů. Např. "Na staré kredenci je/byla malá
 *   plechovka." vs. "Na pultu je/byl mosazný zvonek."
 */
class BaseContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        "\^<<parent.objInName>> <<tSel(selector < 4 ? 'je'
            : 'jsou', ['byl', 'byla', 'bylo', 'byli', 'byly', 'byla'][selector])
            >> ";
    }
    showListSuffixWide(itemCount, pov, parent, selector)
    {
        ". ";
    }
    showListPrefixTall(itemCount, pov, parent, selector)
    {
        "\^<<parent.objInName>> <<tSel(selector < 4 ? 'je'
            : 'jsou', ['byl', 'byla', 'bylo', 'byli', 'byly', 'byla'][selector])
            >>:";
    }
    showListContentsPrefixTall(itemCount, pov, parent, selector)
    {
        gMessageParams(parent);

        "{kohoco parent}, <<parent.objInPrep>> ";

        if(parent.objInPad == 6) "kter{ém parent} ";
        if(parent.objInPad == 7) "kter{ým parent} ";

        "<<tSel(selector < 4 ? 'je' : 'jsou', ['byl', 'byla', 'bylo', 'byli',
            'byly', 'byla'][selector])>>:";
    }
;


/*
 *   Base class for contents listers for a surface 
 *
 *   Tato třída se stará o vypisování objeků, pokud jsou položeny
 *   na nějakém povrchu. Proto si zde vlajkou zasignalizujeme, že chceme
 *   vypsat položené objekty v prvním pádu, místo obvyklého čtvrtého.
 *   "Stará kredenc obsahuje malou modrou svítilnu. Na staré kredenci je
 *   *malá plechovka*."
 */
class BaseSurfaceContentsLister: BaseContentsLister
    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKdoCo, pov, infoTab);
    }
;

/*
 *   Contents lister for a surface 
 */
surfaceContentsLister: ContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for explicitly looking in a surface 
 */
surfaceLookInLister: LookWhereContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for a surface, used in a long description. 
 */
surfaceDescContentsLister: DescContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for room parts
 */
roomPartContentsLister: surfaceContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    /* the part I'm listing */
    part_ = nil
;

/*
 *   contents lister for room parts, used in a long description 
 */
roomPartDescContentsLister: surfaceDescContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;

/*
 *   Look-in lister for room parts.  Most room parts act like surfaces
 *   rather than containers, so base this lister on the surface lister.  
 */
roomPartLookInLister: surfaceLookInLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;
                         
/*
 *   Base class for contents listers for an Underside.  
 */
class BaseUndersideContentsLister: BaseContentsLister
;

/* basic contents lister for an Underside */
undersideContentsLister: ContentsLister, BaseUndersideContentsLister
;

/* contents lister for explicitly looking under an Underside */
undersideLookUnderLister: LookWhereContentsLister, BaseUndersideContentsLister
;

/* contents lister for moving an Underside and abandoning its contents */
undersideAbandonContentsLister: undersideLookUnderLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, selector)
        { "Odsunutí <<parent.nameKohoCeho>> odhalilo "; }
    showListSuffixWide(itemCount, pov, parent, selector)
        { " pod <<parent.zajmenoTebouN>>. "; }
    showListPrefixTall(itemCount, pov, parent, selector)
        { "Odsunutí <<parent.nameKohoCeho>> odhalilo:"; }
;
 
/* contents lister for an Underside, used in a long description */
undersideDescContentsLister: DescContentsLister, BaseUndersideContentsLister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        "Pod <<parent.nameKymCim>> <<tSel(selector < 4 ? 'je' : 'jsou',
            ['byl', 'byla', 'bylo', 'byli', 'byly', 'byla'][selector])>> ";
    }
;

/*
 *   Base class for contents listers for an RearContainer or RearSurface 
 */
class BaseRearContentsLister: BaseContentsLister
;

/* basic contents lister for a RearContainer or RearSurface */
rearContentsLister: ContentsLister, BaseRearContentsLister
;

/* contents lister for explicitly looking behind a RearContainer/Surface */
rearLookBehindLister: LookWhereContentsLister, BaseRearContentsLister
;
 
/* lister for moving a RearContainer/Surface and abandoning its contents */
rearAbandonContentsLister: undersideLookUnderLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent, selector)
        { "Odsunutí <<parent.nameKohoCeho>> odhalilo "; }
    showListSuffixWide(itemCount, pov, parent, selector)
        { " za <<parent.zajmenoTebouN>>. "; }
    showListPrefixTall(itemCount, pov, parent, selector)
        { "Odsunutí <<parent.nameKohoCeho>> odhalilo:"; }
;
 
/* long-description contents lister for a RearContainer/Surface */
rearDescContentsLister: DescContentsLister, BaseRearContentsLister
    showListPrefixWide(itemCount, pov, parent, selector)
    {
        "Za <<parent.nameKymCim>> <<tSel(selector < 4 ? 'je' : 'jsou',
            ['byl', 'byla', 'bylo', 'byli', 'byly', 'byla'][selector])>> ";
    }
;


/*
 *   Base class for specialized in-line contents listers.  This shows the
 *   list in the form "(<prep> which is...)", with the preposition obtained
 *   from the container's objInPrep property.  
 */
class BaseInlineContentsLister: ContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(cnt, pov, parent, selector)
    {
        " (<<parent.objInPrep>> kter<<buildSynthParam('ém', parent)>>
            <<tSel(selector < 4 ? 'je' : 'jsou', ['byl', 'byla', 'bylo', 'byli',
            'byly', 'byla'][selector])>> ";
    }
    showListSuffixWide(itemCount, pov, parent, selector)
        { ")"; }
;

/*
 *   Contents lister for a generic in-line list entry.  We customize the
 *   wording slightly here: rather than saying "(in which...)" as the base
 *   class would, we use the slightly more readable "(which contains...)".
 *
 *   TODO: kterí
 */
inlineListingContentsLister: BaseInlineContentsLister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        " (kter<<buildSynthParam('ý', parent)>> <<tSel('obsahuje', 'obsahoval'
            + buildSynthParam('a', parent))>> ";
    }
;

/* in-line contents lister for a surface */
surfaceInlineContentsLister: BaseInlineContentsLister
;

/* in-line contents lister for an Underside */
undersideInlineContentsLister: BaseInlineContentsLister
;

/* in-line contents lister for a RearContainer/Surface */
rearInlineContentsLister: BaseInlineContentsLister
;

/*
 *   Contents lister for keyring list entry.  This is used to display a
 *   keyring's contents in-line with the name of the keyring,
 *   parenthetically. 
 */
keyringInlineContentsLister: inlineListingContentsLister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        " (s navlečen<<['ým', 'ou', 'ým', 'ými', 'ými', 'ými'][selector]>> ";
    }
    showListSuffixWide(cnt, pov, parent, selector)
        { ")"; }
    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKymCim, pov, infoTab);
    }
;

/*
 *   Contents lister for "examine <keyring>" 
 */
keyringExamineContentsLister: DescContentsLister
    showListEmpty(pov, parent)
    {
        "\^<<parent.name>> {je} prázdn{ý}. ";
    }
    showListPrefixWide(cnt, pov, parent, selector)
    {
        "Na <<parent.nameKomCem>> <<tSel(selector < 4 ? 'je' : 'jsou', 'byl' +
            ['', 'a', 'o', 'i', 'y', 'a'][selector])>> navlečen<<['', 'a', 'o',
            'i', 'y', 'a'][selector]>> ";
    }
    showListSuffixWide(itemCount, pov, parent, selector)
    {
        ". ";
    }
;

/*
 *   Lister for actors aboard a traveler.  This is used to list the actors
 *   on board a vehicle when the vehicle arrives or departs a location.  
 */
aboardVehicleLister: Lister
    showListPrefixWide(itemCount, pov, parent, selector)
        { " (vez{e|l[a parent]} "; }
    showListSuffixWide(itemCount, pov, parent, selector)
        { ")"; }

    /* list anything whose isListedAboardVehicle returns true */
    isListed(obj) { return obj.isListedAboardVehicle; }
;

/*
 *   A simple lister to show the objects to which a given Attachable
 *   object is attached.  This shows the objects which have symmetrical
 *   attachment relationships to the given parent object, or which are
 *   "major" items to which the parent is attached.  
 *
 *   "Provaz byl pripevněn k tyči a háku."
 *   "Vlajka byla připevněna ke stožáru."
 */
class SimpleAttachmentLister: Lister
    construct(parent) { parent_ = parent; }
    
    showListEmpty(pov, parent)
        { /* say nothing when there are no attachments */ }
    
    showListPrefixWide(cnt, pov, parent, selector)
    {
        "<.p>\^<<parent.name>> <<parent.slovesoJe>>
            <<parent.ofKind(PlugAttachable) ? 'připojen' : 'připevněn'
            >><<parent.verbPastEnding>> k ";//TODO: k/ke
    }
    
    showListSuffixWide(cnt, pov, parent, selector)
        { ". "; }

    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKomuCemu, pov, infoTab);
    }

    /* ask the parent if we should list each item */
    isListed(obj) { return parent_.isListedAsAttachedTo(obj); }

    /*
     *   the parent object - this is the object whose attachments are being
     *   listed 
     */
    parent_ = nil
;

/*
 *   The "major" attachment lister.  This lists the objects which are
 *   attached to a given parent Attachable, and for which the parent is
 *   the "major" item in the relationship.  The items in the list are
 *   described as being attached to the parent.  
 *
 *   "K tyči byla připevněna cedulka."
 *   "K tyči byl připevněn zvonek."
 */
class MajorAttachmentLister: SimpleAttachmentLister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        gMessageParams(parent);
        "<.p>{K/ke parent} <<parent.nameKomuCemu>> <<tSel(selector < 4 ? 'je' : 'jsou',
            ['byl', 'byla', 'bylo', 'byli', 'byly', 'byla'][selector])>>
            <<parent.ofKind(PlugAttachable) ? 'připojen' : 'připevněn'
            >><<['', 'a', 'o', 'i', 'y', 'a'][selector]>> ";
    }
    showListSuffixWide(cnt, pov, parent, selector)
        { ". "; }

    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options | ListKdoCo, pov, infoTab);
    }

    /* ask the parent if we should list each item */
    isListed(obj) { return parent_.isListedAsMajorFor(obj); }
;

/*
 *   Finish Options lister.  This lister is used to offer the player
 *   options in finishGame(). 
 */
finishOptionsLister: Lister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        "<.p>Chceš ";
    }
    showListSuffixWide(cnt, pov, parent, selector)
    {
        /* end the question, add a blank line, and show the ">" prompt */
        "?\b&gt;";
    }
    
    isListed(obj) { return obj.isListed; }
    listCardinality(obj) { return 1; }
    
    showListItem(obj, options, pov, infoTab)
    {
        /* obj is a FinishOption object; show its description */
        obj.desc;
    }
    
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   for the last separator, show "or" rather than "and"; for
         *   others, inherit the default handling 
         */
        if (curItemNum + 1 == totalItems)
        {
                " nebo ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }
;

/*
 *   Equivalent list state lister.  This shows a list of state names for a
 *   set of otherwise indistinguishable items.  We show the state names in
 *   parentheses, separated by commas only (i.e., no "and" separating the
 *   last two items); we use this less verbose format so that we blend
 *   into the larger enclosing list more naturally.
 *   
 *   The items to be listed are EquivalentStateInfo objects.  
 */
equivalentStateLister: Lister
    showListPrefixWide(cnt, pov, parent, selector) { " ("; }
    showListSuffixWide(cnt, pov, parent, selector) { ")"; }
    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }
    showListItem(obj, options, pov, infoTab)
    {
        "<<spellIntBelow(obj.getEquivCount(), obj.gender, 100)>> <<obj.getName()>>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        if (curItemNum < totalItems)
            ", ";
    }
;

/* in case the exits module isn't included in the build */
property destName_, destIsBack_, others_, enableHyperlinks;

/*
 *   Basic room exit lister.  This shows a list of the apparent exits from
 *   a location.
 *   
 *   The items to be listed are DestInfo objects.  
 */
class ExitLister: Lister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        if (cnt == 1)
            "Jediný zřetelný směr ved{e|l} ";
        else
            "Zřetelné směry ved{ou|ly} ";
    }
    showListSuffixWide(cnt, pov, parent, selector) { ". "; }

    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }

    showListItem(obj, options, pov, infoTab)
    {
        /*
         *   Show the back-to-direction prefix, if we don't know the
         *   destination name but this is the back-to direction: "back to
         *   the north" and so on. 
         */
        if (obj.destIsBack_ && obj.destName_ == nil)
            say(obj.dir_.backToPrefix + ' ');
        
        /* show the direction */
        showListItemDirName(obj, nil);

        /* if the destination is known, show it as well */
        if (obj.destName_ != nil)
        {
            /*
             *   if we have a list of other directions going to the same
             *   place, show it parenthetically 
             */
            otherExitLister.showListAll(obj.others_, 0, 0);

            /* if this is the way back, say so */
            if (obj.destIsBack_)
                " zpátky";

            /* show the destination */
            " <<obj.destName_>>";
        }
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   if we have a "back to" name, use the "long" notation - this is
         *   important because we'll use commas in the directions with
         *   known destination names 
         */
        if ((options & hasBackNameFlag) != 0)
            options |= ListLong;

        /*
         *   for a two-item list, if either item has a destination name,
         *   show a comma or semicolon (depending on 'long' vs 'short' list
         *   mode) before the "and"; for anything else, use the default
         *   handling 
         */
        if (curItemNum == 1
            && totalItems == 2
            && (options & hasDestNameFlag) != 0)
        {
            if ((options & ListLong) != 0)
                " a ";
            else
                " a ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }

    /* show a direction name, hyperlinking it if appropriate */
    showListItemDirName(obj, initCap)
    {
        local dirname;
        
        /* get the name */
        dirname = obj.dir_.name;

        /* capitalize the first letter, if desired */
        if (initCap)
            dirname = dirname.substr(1,1).toUpper() + dirname.substr(2);

        /* show the name with a hyperlink or not, as configured */
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(dirname, dirname));
        else
            say(dirname);
    }

    /* this lister shows destination names */
    listerShowsDest = true

    /*
     *   My special options flag: at least one object in the list has a
     *   destination name.  The caller should set this flag in our options
     *   if applicable. 
     */
    hasDestNameFlag = ListerCustomFlag(1)
    hasBackNameFlag = ListerCustomFlag(2)
    nextCustomFlag = ListerCustomFlag(3)
;

/*
 *   Show a list of other exits to a given destination.  We'll show the
 *   list parenthetically, if there's a list to show.  The objects to be
 *   listed are Direction objects.  
 */
otherExitLister: Lister
    showListPrefixWide(cnt, pov, parent, selector) { " (nebo "; }
    showListSuffixWide(cnt, pov, parent, selector) { ")"; }

    isListed(obj) { return true; }
    listCardinality(obj) { return 1; }

    showListItem(obj, options, pov, infoTab)
    {
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(obj.name, obj.name));
        else
            say(obj.name);
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /*
         *   simply show "or" for all items (these lists are usually
         *   short, so omit any commas) 
         */
        if (curItemNum != totalItems)
            " nebo ";
    }
;

/*
 *   Show room exits as part of a room description, using the "verbose"
 *   sentence-style notation.  
 */
lookAroundExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        /* add a paragraph break before the exit listing */
        "<.roompara>";

        /* inherit default handling */
        inherited(cnt, pov, parent, selector);
    }    
;

/*
 *   Show room exits as part of a room description, using the "terse"
 *   notation. 
 */
lookAroundTerseExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent, selector)
    {
        "<.roompara><.parser>Očividné směry: ";
    }
    showListItem(obj, options, pov, infoTab)
    {
        /* show the direction name */
        showListItemDirName(obj, nil);
    }
    showListSuffixWide(cnt, pov, parent, selector)
    {
        "<./parser> ";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a comma between items */
        if (curItemNum != totalItems)
            ", ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

/*
 *   Show room exits in response to an explicit request (such as an EXITS
 *   command).  
 */
explicitExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "{Nejsou|Nebyly tu} žádné očividné východy. ";
    }
;

/*
 *   Show room exits in the status line (used in HTML mode only)
 */
statuslineExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "<<statusHTML(3)>><b>Směry:</b> <i>žádné</i><<statusHTML(4)>>";
    }
    showListPrefixWide(cnt, pov, parent, selector)
    {
        "<<statusHTML(3)>><b>Směry:</b> ";
    }
    showListSuffixWide(cnt, pov, parent, selector)
    {
        "<<statusHTML(4)>>";
    }
    showListItem(obj, options, pov, infoTab)
    {
        "<<aHref(obj.dir_.name, obj.dir_.name, 'Jdi na ' + obj.dir_.name,
            AHREF_Plain)>>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a space between items */
        if (curItemNum != totalItems)
            " &nbsp; ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

/*
 *   Implied action announcement grouper.  This takes a list of
 *   ImplicitActionAnnouncement reports and returns a single message string
 *   describing the entire list of actions.  
 */
implicitAnnouncementGrouper: object
    /*
     *   Configuration option: keep all failures in a list of implied
     *   announcements.  If this is true, then we'll write things like
     *   "trying to unlock the door and then open it"; if nil, we'll
     *   instead write simply "trying to unlock the door".
     *   
     *   By default, we keep only the first of a group of failures.  A
     *   group of failures is always recursively related, so the first
     *   announcement refers to the command that actually failed; the rest
     *   of the announcements are for the enclosing actions that triggered
     *   the first action.  All of the enclosing actions failed as well,
     *   but only because the first action failed.
     *   
     *   Announcing all of the actions is too verbose for most tastes,
     *   which is why we set the default here to nil.  The fact that the
     *   first action in the group failed means that we necessarily won't
     *   carry out any of the enclosing actions, so the enclosing
     *   announcements don't tell us much.  All they really tell us is why
     *   we're running the action that actually failed, but that's almost
     *   always obvious, so suppressing them is usually fine.  
     */
    keepAllFailures = nil

    /* build the composite message */
    compositeMessage(lst)
    {
        local txt;
        local ctx = new ListImpCtx();

        /* add the text for each item in the list */
        for (txt = '', local i = 1, local len = lst.length() ; i <= len ; ++i)
        {
            local curTxt;

            /* get this item */
            local cur = lst[i];

            /* we're not in a 'try' or 'ask' sublist yet */
            ctx.isInSublist = nil;

            /* set the underlying context according to this item */
            ctx.setBaseCtx(cur);

            /*
             *   Generate the announcement for this element.  Generate the
             *   announcement from the message property for this item using
             *   our running list context.  
             */
            curTxt = cur.getMessageText(
                cur.getAction().getOriginalAction(), ctx);

            /*
             *   If this one is an attempt only, and it's followed by one
             *   or more other attempts, the attempts must all be
             *   recursively related (in other words, the first attempt was
             *   an implied action required by the second attempt, which
             *   was required by the third, and so on).  They have to be
             *   recursively related, because otherwise we wouldn't have
             *   kept trying things after the first failed attempt.
             *   
             *   To make the series of failed attempts sound more natural,
             *   group them into a single "trying to", and keep only the
             *   first failure: rather than "trying to unlock the door,
             *   then trying to open the door", write "trying to unlock the
             *   door and then open it".
             *   
             *   An optional configuration setting makes us keep only the
             *   first failed operation, so we'd instead write simply
             *   "trying to unlock the door".
             *   
             *   Do the same grouping for attempts interrupted for an
             *   interactive question.  
             */
            while ((cur.justTrying && i < len && lst[i+1].justTrying)
                   || (cur.justAsking && i < len && lst[i+1].justAsking))
            {
                local addTxt;
                
                /*
                 *   move on to the next item - we're processing it here,
                 *   so we don't need to handle it again in the main loop 
                 */
                ++i;
                cur = lst[i];

                /* remember that we're in a try/ask sublist */
                ctx.isInSublist = true;

                /* add the list entry for this action, if desired */
                if (keepAllFailures)
                {
                    /* get the added text */
                    addTxt = cur.getMessageText(
                        cur.getAction().getOriginalAction(), ctx);

                    /*
                     *   if both the text so far and the added text are
                     *   non-empty, string them together with 'and then';
                     *   if one or the other is empty, use the non-nil one 
                     */
                    if (addTxt != '' && curTxt != '')
                        curTxt += ' and then ' + addTxt;
                    else if (addTxt != '')
                        curTxt = addTxt;
                }
            }

            /* add a separator before this item if it isn't the first */
            if (txt != '' && curTxt != '')
                txt += ', potom {|[jsi]} ';

            /* add the current item's text */
            txt += curTxt;
        }

        /* if we ended up with no text, the announcement is silent */
        if (txt == '')
            return '';

        /* wrap the whole list in the usual full standard phrasing */
        return standardImpCtx.buildImplicitAnnouncement(txt);
    }
;

/*           
 *   Suggested topic lister. 
 */
class SuggestedTopicLister: Lister
    construct(asker, askee, explicit)
    {
        /* remember the actors */
        askingActor = asker;
        targetActor = askee;

        /* remember whether this is explicit or implicit */
        isExplicit = explicit;

        /* cache the actor's scope list */
        scopeList = asker.scopeList();
    }
    /* Přetěžujeme kvůli náhradám "bys se" na "by ses". */
    showList(pov, parent, lst, options, indent, infoTab, parentGroup)
    {
        local groups;
        local groupTab;
        local singles;
        local origLst;
        local itemCount;

        /* remember the original list */
        origLst = lst;

        /* filter the list to get only the items we actually will list */
        lst = getFilteredList(lst, infoTab);

        /* create a lookup table to keep track of the groups we've seen */
        groupTab = new LookupTable();
        groups = new Vector(10);

        /* set up a vector to keep track of the singles */
        singles = new Vector(10);

        /* figure the groupings */
        itemCount = getListGrouping(groupTab, groups, singles,
                                    lst, parentGroup);

        /*
         *   Now that we've figured out what's in the list and how it's
         *   arranged into groups, show the list.
         *
         *   Celou tuto funkci přetěžuji proto, abych mohl provést náhradu
         *   "bys se" na "by ses" v seznamech doporučených témat rozhovoru.
         *   I když by se zdálo, že toto místo je ideální k provedení náhrady,
         *   není tomu tak, protože ještě nejsou přepsané parametry zpráv,
         *   a tak není ještě určeno, zda ve větě bude "bych", "bys" nebo "by".
         *
         *   Vše by tedy měly zařídit přepisy v typografickém výstupním filtru,
         *   ale typografický filtr dostává texty po malých kouscích a tato
         *   náhrada vždy vyjde přes hranici a proto se neprovede.
         *
         *   Kód níže zařídí, že se text prožene typografickým filtrem ještě
         *   jednou a tentokrát celý najednou a na něj teprve přepisy zaberou.
         */
        local text = outputManager.curOutputStream.captureOutput({:
            showArrangedList(pov, parent, lst, options, indent, infoTab,
                         itemCount, singles, groups, groupTab, origLst)});
        say(text);

        /*
         *   If the list is recursive, mention the contents of any items
         *   that weren't listed in the main list, and of any contents
         *   that are specifically to be listed out-of-line.  Don't do
         *   this if we're already recursively showing such a listing,
         *   since if we did so we could show items at recursive depths
         *   more than once; if we're already doing a recursive listing,
         *   our caller will itself recurse through all levels of the
         *   tree, so we don't have to recurse any further ourselves.  
         */
        if ((options & ListRecurse) != 0
            && indent == 0
            && (options & ListContents) == 0)
        {
            /* show the contents of each object we didn't list */
            showSeparateContents(pov, origLst,
                                 options | ListContents, infoTab);
        }
    }

    showListPrefixWide(cnt, pov, parent, selector)
    {
        /* add the asking and target actors as global message parameters */
        gMessageParams(askingActor, targetActor);

        /* show the prefix; include a paren if not in explicit mode */
        "<<isExplicit ? '' : '('>>Mohl{a} {[bys]|[jsi]} ";
    }

    showListSuffixWide(cnt, pov, parent, selector)
    {
        /* end the sentence; include a paren if not in explicit mode */
        ".<<isExplicit? '' : ')'>> ";
    }

    showListEmpty(pov, parent)
    {
        /*
         *   say that the list is empty if it was explicitly requested;
         *   say nothing if the list is being added by the library 
         */
        if (isExplicit)
        {
            gMessageParams(askingActor, targetActor);
            "<<isExplicit ? '' : '('>>Nem{[áš]|ěl[a] [jsi]} na mysli nic konkrétního, o čem
            by {ses} chtěl{a} bavit {s/se targetActor} {kýmčím targetActor}.<<isExplicit ? '' : ')'>> ";
        }
    }

    showListSeparator(options, curItemNum, totalItems)
    {
        /* use "or" as the conjunction */
        if (curItemNum + 1 == totalItems)
            " nebo ";
        else
            inherited(options, curItemNum, totalItems);
    }

    /* list suggestions that are currently active */
    isListed(obj) { return obj.isSuggestionActive(askingActor, scopeList); }

    /* each item counts as one item grammatically */
    listCardinality(obj) { return 1; }

    /* suggestions have no contents */
    contentsListed(obj) { return nil; }

    /* get the list group */
    listWith(obj) { return obj.suggestionGroup; }

    /* mark as seen - nothing to do for suggestions */
    markAsSeen(obj, pov) { }

    /* show the item - show the suggestion's theName */
    showListItem(obj, options, pov, infoTab)
    {
        /* note that we're showing the suggestion */
        obj.noteSuggestion();

        /* show the name */
        say(obj.fullName);
    }

    /* don't use semicolons, even in long lists */
    longListSepTwo { listSepTwo; }
    longListSepMiddle { listSepMiddle; }
    longListSepEnd { listSepEnd; }

    /* flag: this is an explicit listing (i.e., a TOPICS command) */
    isExplicit = nil

    /* the actor who's asking for the topic list (usually the PC) */
    askingActor = nil

    /* the actor we're talking to */
    targetActor = nil

    /* our cached scope list for the actor */
    scopeList = nil
;

/* ASK/TELL suggestion list group base class */
class SuggestionListGroup: ListGroupPrefixSuffix
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /*
         *   show the short name of the item - the group prefix will have
         *   shown the appropriate long name 
         */
        say(obj.name);
    }
;

/* ASK ABOUT suggestion list group */
suggestionAskGroup: SuggestionListGroup
    groupPrefix = "se {tě/jí targetActor} zeptat "
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /*
         *   show the short name of the item - the group prefix will have
         *   shown the appropriate long name 
         */
        say('na ' + obj.name);
    }
;

/* TELL ABOUT suggestion list group */
suggestionTellGroup: SuggestionListGroup
    groupPrefix = "{ti/jí targetActor} říci "
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /*
         *   show the short name of the item - the group prefix will have
         *   shown the appropriate long name 
         */
        say('o ' + obj.name);
    }
;

/* ASK FOR suggestion list group */
suggestionAskForGroup: SuggestionListGroup
    groupPrefix = "{tě/jí targetActor} požádat "
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /*
         *   show the short name of the item - the group prefix will have
         *   shown the appropriate long name 
         */
        say('o ' + obj.name);
    }
;

/* GIVE TO suggestions list group */
suggestionGiveGroup: SuggestionListGroup
    groupPrefix = "{ti/jí targetActor} dát "
;

/* SHOW TO suggestions */
suggestionShowGroup: SuggestionListGroup
    groupPrefix = "{ti/jí targetActor} ukázat "
;

/* YES/NO suggestion group */
suggestionYesNoGroup: SuggestionListGroup
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /*
         *   if we have one each of YES and NO responses, make the entire
         *   list "say yes or no"; otherwise, use the default behavior 
         */
        if (lst.length() == 2
            && lst.indexWhich({x: x.ofKind(SuggestedYesTopic)}) != nil
            && lst.indexWhich({x: x.ofKind(SuggestedNoTopic)}) != nil)
        {
            /* we have a [yes, no] group - use the simple message */
            "odpovědět ano nebo ne";
        }
        else
        {
            /* inherit the default behavior */
            inherited(pov, lister, lst, options, indent, infoTab);
        }
    }
    groupPrefix = "odpovědět";
;
