#charset "utf-8"

/*
 *   TADS 3 Library - Czech language implementation
 *   Copyright 2010-2013 Tomas Blaha. All Rights Reserved.
 *
 *   Based on English language module by Michael J. Roberts with permission.
 *
 *.                                  -----
 *
 *   Copyright 2000, 2006 Michael J. Roberts.  All Rights Reserved.
 *.  Past-tense extensions written by Michel Nizette, and incorporated by
 *   permission.
 *   
 *   TADS 3 Library - English (United States variant) implementation
 *   
 *   This defines the parts of the TADS 3 library that are specific to the
 *   English language as spoken (and written) in the United States.
 *   
 *   We have attempted to isolate here the parts of the library that are
 *   language-specific, so that translations to other languages or dialects
 *   can be created by replacing this module, without changing the rest of
 *   the library.
 *   
 *   In addition to this module, a separate set of US English messages are
 *   defined in the various msg_xxx.t modules.  Those modules define
 *   messages in English for different stylistic variations.  For a given
 *   game, the author must select one of the message modules - but only
 *   one, since they all define variations of the same messages.  To
 *   translate the library, a translator must create at least one module
 *   defining those messages as well; only one message module is required
 *   per language.
 *   
 *   The past-tense system was contributed by Michel Nizette.
 *   
 *.                                  -----
 *   
 *   "Watch an immigrant struggling with a second language or a stroke
 *   patient with a first one, or deconstruct a snatch of baby talk, or try
 *   to program a computer to understand English, and ordinary speech
 *   begins to look different."
 *   
 *.         Stephen Pinker, "The Language Instinct"
 */

#include "tads.h"
#include "tok.h"
#include "adv3.h"
#include "cs_cz.h"
#include <vector.h>
#include <dict.h>
#include <gramprod.h>
#include <strcomp.h>


/* ------------------------------------------------------------------------ */
/*
 *   Fill in the default language for the GameInfo metadata class.
 */
modify GameInfoModuleID
    languageCode = 'cs-CZ'
;

/* ------------------------------------------------------------------------ */
/*
 *   Simple yes/no confirmation.  The caller must display a prompt; we'll
 *   read a command line response, then return true if it's an affirmative
 *   response, nil if not.
 */
yesOrNo()
{
    /* switch to no-command mode for the interactive input */
    "<.commandnone>";

    /*
     *   Read a line of input.  Do not allow real-time event processing;
     *   this type of prompt is used in the middle of a command, so we
     *   don't want any interruptions.  Note that the caller must display
     *   any desired prompt, and since we don't allow interruptions, we
     *   won't need to redisplay the prompt, so we pass nil for the prompt
     *   callback.
     */
    local str = inputManager.getInputLine(nil, nil);

    /* switch back to mid-command mode */
    "<.commandmid>";

    /*
     *   If they answered with something starting with 'Y', it's
     *   affirmative, otherwise it's negative.  In reading the response,
     *   ignore any leading whitespace.
     */
    return rexMatch('<space>*[aA]', str) != nil;
}

/* ------------------------------------------------------------------------ */
/*
 *   During start-up, install a case-insensitive truncating comparator in
 *   the main dictionary.
 */
PreinitObject
    /*
     *   String komparátoru poskytneme mapu ekvivalentních znaků, aby poznal
     *   objekty i tehdy, když hráč bude psát bez háčků a čárek.
     *   http://www.tads.org/t3doc/doc/sysman/strcomp.htm
     */
    mappings =
    [
        ['ř', 'r', 0, 0],
        ['Ř', 'R', 0, 0],
        ['í', 'i', 0, 0],
        ['Í', 'I', 0, 0],
        ['š', 's', 0, 0],
        ['Š', 'S', 0, 0],
        ['ž', 'z', 0, 0],
        ['Ž', 'Z', 0, 0],
        ['ť', 't', 0, 0],
        ['Ť', 'T', 0, 0],
        ['č', 'c', 0, 0],
        ['Č', 'C', 0, 0],
        ['ý', 'y', 0, 0],
        ['Ý', 'Y', 0, 0],
        ['ů', 'u', 0, 0],
        ['Ů', 'U', 0, 0],
        ['ň', 'n', 0, 0],
        ['Ň', 'N', 0, 0],
        ['ú', 'u', 0, 0],
        ['Ú', 'U', 0, 0],
        ['ě', 'e', 0, 0],
        ['Ě', 'E', 0, 0],
        ['ď', 'd', 0, 0],
        ['Ď', 'D', 0, 0],
        ['á', 'a', 0, 0],
        ['Á', 'A', 0, 0],
        ['é', 'e', 0, 0],
        ['É', 'E', 0, 0],
        ['ó', 'o', 0, 0],
        ['Ó', 'O', 0, 0]
    ]
    execute()
    {
        /* set up the main dictionary's comparator */
        languageGlobals.setStringComparator(
            new StringComparator(gameMain.parserTruncLength, nil, mappings));
    }

    /*
     *   Make sure we run BEFORE the main library preinitializer, so that
     *   we install the comparator in the dictionary before we add the
     *   vocabulary words to the dictionary.  This doesn't make any
     *   difference in terms of the correctness of the dictionary, since
     *   the dictionary will automatically rebuild itself whenever we
     *   install a new comparator, but it makes the preinitialization run
     *   a tiny bit faster by avoiding that rebuild step.
     */
    execAfterMe = [adv3LibPreinit]
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific globals
 */
languageGlobals: object
    /*
     *   Set the StringComparator object for the parser.  This sets the
     *   comparator that's used in the main command parser dictionary. 
     */
    setStringComparator(sc)
    {
        /* remember it globally, and set it in the main dictionary */
        dictComparator = sc;
        cmdDict.setComparator(sc);
    }

    /*
     *   The character to use to separate groups of digits in large
     *   numbers.  US English uses commas; most Europeans use periods.
     *
     *   Note that this setting does not affect system-level BigNumber
     *   formatting, but this information can be passed when calling
     *   BigNumber formatting routines.
     */
    digitGroupSeparator = '.'

    /*
     *   The decimal point to display in floating-point numbers.  US
     *   English uses a period; most Europeans use a comma.
     *
     *   Note that this setting doesn't affect system-level BigNumber
     *   formatting, but this information can be passed when calling
     *   BigNumber formatting routines.
     */
    decimalPointCharacter = ','

    /* the main dictionary's string comparator */
    dictComparator = nil
    
    /* we maintain state for quotes replacement filter */
    quotesLevel = 0
;


/* ------------------------------------------------------------------------ */
/*
 *   Language-specific extension of the default gameMain object
 *   implementation.
 */
modify GameMainDef
    /*
     *   Option setting: the parser's truncation length for player input.
     *   As a convenience to the player, we can allow the player to
     *   truncate long words, entering only the first, say, 6 characters.
     *   For example, rather than typing "x flashlight", we could allow the
     *   player to simply type "x flashl" - truncating "flashlight" to six
     *   letters.
     *   
     *   We use a default truncation length of 6, but games can change this
     *   by overriding this property in gameMain.  We use a default of 6
     *   mostly because that's what the old Infocom games did - many
     *   long-time IF players are accustomed to six-letter truncation from
     *   those games.  Shorter lengths are superficially more convenient
     *   for the player, obviously, but there's a trade-off, which is that
     *   shorter truncation lengths create more potential for ambiguity.
     *   For some games, a longer length might actually be better for the
     *   player, because it would reduce spurious ambiguity due to the
     *   parser matching short input against long vocabulary words.
     *   
     *   If you don't want to allow the player to truncate long words at
     *   all, set this to nil.  This will require the player to type every
     *   word in its entirety.
     *   
     *   Note that changing this property dynamicaly will have no effect.
     *   The library only looks at it once, during library initialization
     *   at the very start of the game.  If you want to change the
     *   truncation length dynamically, you must instead create a new
     *   StringComparator object with the new truncation setting, and call
     *   languageGlobals.setStringComparator() to select the new object.  
     */
    parserTruncLength = 6

    /*
     *   Option: are we currently using a past tense narrative?  By
     *   default, we aren't.
     *
     *   This property can be reset at any time during the game in order to
     *   switch between the past and present tenses.  The macro
     *   setPastTense can be used for this purpose: it just provides a
     *   shorthand for setting gameMain.usePastTense directly.
     *
     *   Authors who want their game to start in the past tense can achieve
     *   this by overriding this property on their gameMain object and
     *   giving it a value of true.
     */
    usePastTense = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for ThingState.
 */
modify ThingState
    /*
     *   Our state-specific tokens.  This is a list of vocabulary words
     *   that are state-specific: that is, if a word is in this list, the
     *   word can ONLY refer to this object if the object is in a state
     *   with that word in its list.
     *   
     *   The idea is that you set up the object's "static" vocabulary with
     *   the *complete* list of words for all of its possible states.  For
     *   example:
     *   
     *.     + Matchstick 'lit unlit match';
     *   
     *   Then, you define the states: in the "lit" state, the word 'lit' is
     *   in the stateTokens list; in the "unlit" state, the word 'unlit' is
     *   in the list.  By putting the words in the state lists, you
     *   "reserve" the words to their respective states.  When the player
     *   enters a command, the parser will limit object matches so that the
     *   reserved state-specific words can only refer to objects in the
     *   corresponding states.  Hence, if the player refers to a "lit
     *   match", the word 'lit' will only match an object in the "lit"
     *   state, because 'lit' is a reserved state-specific word associated
     *   with the "lit" state.
     *   
     *   You can re-use a word in multiple states.  For example, you could
     *   have a "red painted" state and a "blue painted" state, along with
     *   an "unpainted" state.
     */
    stateTokens = []

    /*
     *   Match the name of an object in this state.  We'll check the token
     *   list for any words that apply only to *other* states the object
     *   can assume; if we find any, we'll reject the match, since the
     *   phrase must be referring to an object in a different state.
     */
    matchName(obj, origTokens, adjustedTokens, states)
    {
        /* scan each word in our adjusted token list */
        for (local i = 1, local len = adjustedTokens.length() ;
             i <= len ; i += 2)
        {
            /* get the current token */
            local cur = adjustedTokens[i];

            /*
             *   If this token is in our own state-specific token list,
             *   it's acceptable as a match to this object.  (It doesn't
             *   matter whether or not it's in any other state's token list
             *   if it's in our own, because its presence in our own makes
             *   it an acceptable matching word when we're in this state.) 
             */
            if (stateTokens.indexWhich({t: t == cur}) != nil)
                continue;

            /*
             *   It's not in our own state-specific token list.  Check to
             *   see if the word appears in ANOTHER state's token list: if
             *   it does, then this word CAN'T match an object in this
             *   state, because the token is special to that other state
             *   and thus can't refer to an object in a state without the
             *   token. 
             */
            if (states.indexWhich(
                {s: s.stateTokens.indexOf(cur) != nil}) != nil)
                return nil;
        }

        /* we didn't find any objection, so we can match this phrase */
        return obj;
    }

    /*
     *   Check a token list for any tokens matching any of our
     *   state-specific words.  Returns true if we find any such words,
     *   nil if not.
     *
     *   'toks' is the *adjusted* token list used in matchName().
     */
    findStateToken(toks)
    {
        /*
         *   Scan the token list for a match to any of our state-specific
         *   words.  Since we're using the adjusted token list, every
         *   other entry is a part of speech, so work through the list in
         *   pairs.
         */
        for (local i = 1, local len = toks.length() ; i <= len ; i += 2)
        {
            /*
             *   if this token matches any of our state tokens, indicate
             *   that we found a match
             */
            if (stateTokens.indexWhich({x: x == toks[i]}) != nil)
                return true;
        }

        /* we didn't find a match */
        return nil;
    }

    /* get our name */
    listName(lst) { return listName_; }

    /*
     *   our list name setting - we define this so that we can be easily
     *   initialized with a template (we can't initialize listName()
     *   directly in this manner because it's a method, but we define the
     *   listName() method to simply return this property value, which we
     *   can initialize with a template)
     */
    listName_ = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for VocabObject.
 */
modify VocabObject
    /*
     *   The vocabulary initializer string for the object - this string
     *   can be initialized (most conveniently via a template) to a string
     *   of this format:
     *
     *   'adj adj adj noun/noun/noun*plural plural plural'
     *
     *   The noun part of the string can be a hyphen, '-', in which case
     *   it means that the string doesn't specify a noun or plural at all.
     *   This can be useful when nouns and plurals are all inherited from
     *   base classes, and only adjectives are to be specified.  (In fact,
     *   any word that consists of a single hyphen will be ignored, but
     *   this is generally only useful for the adjective-only case.)
     *
     *   During preinitialization, we'll parse this string and generate
     *   dictionary entries and individual vocabulary properties for the
     *   parts of speech we find.
     *
     *   Note that the format described above is specific to the English
     *   version of the library.  Non-English versions will probably want
     *   to use different formats to conveniently encode appropriate
     *   language-specific information in the initializer string.  See the
     *   comments for initializeVocabWith() for more details.
     *
     *   You can use the special wildcard # to match any numeric
     *   adjective.  This only works as a wildcard when it stands alone,
     *   so a string like "7#" is matched as that literal string, not as a
     *   wildcard.  If you want to use a pound sign as a literal
     *   adjective, just put it in double quotes.
     *
     *   You can use the special wildcard "\u0001" (include the double
     *   quotes within the string) to match any literal adjective.  This
     *   is the literal adjective equivalent of the pound sign.  We use
     *   this funny character value because it is unlikely ever to be
     *   interesting in user input.
     *
     *   If you want to match any string for a noun and/or adjective, you
     *   can't do it with this property.  Instead, just add the property
     *   value noun='*' to the object.
     */
    vocabWords = ''
    gcVocab = ''

    /*
     *   On dynamic construction, initialize our vocabulary words and add
     *   them to the dictionary.
     */
    construct()
    {
        /* initialize our vocabulary words from vocabWords */
        initializeVocab();

        /* add our vocabulary words to the dictionary */
        addToDictionary(&noun);
        addToDictionary(&adjective);
        addToDictionary(&plural);
        addToDictionary(&adjApostS);
        addToDictionary(&literalAdjective);
    }

    /* add the words from a dictionary property to the global dictionary */
    addToDictionary(prop)
    {
        /* if we have any words defined, add them to the dictionary */
        if (self.(prop) != nil)
            cmdDict.addWord(self, self.(prop), prop);
    }

    /* initialize the vocabulary from vocabWords */
    initializeVocab()
    {
        /* inherit vocabulary from this class and its superclasses */
        inheritVocab(self, new Vector(10));
    }

    /*
     *   Inherit vocabulary from this class and its superclasses, adding
     *   the words to the given target object.  'target' is the object to
     *   which we add our vocabulary words, and 'done' is a vector of
     *   classes that have been visited so far.
     *
     *   Since a class can be inherited more than once in an inheritance
     *   tree (for example, a class can have multiple superclasses, each
     *   of which have a common base class), we keep a vector of all of
     *   the classes we've visited.  If we're already in the vector, we'll
     *   skip adding vocabulary for this class or its superclasses, since
     *   we must have already traversed this branch of the tree from
     *   another subclass.
     */
    inheritVocab(target, done)
    {
        /*
         *   if we're in the list of classes handled already, don't bother
         *   visiting me again
         */
        if (done.indexOf(self) != nil)
            return;

        /* add myself to the list of classes handled already */
        done.append(self);

        /* 
         *   add words from our own vocabWords to the target object (but
         *   only if it's our own - not if it's only inherited, as we'll
         *   pick up the inherited ones explicitly in a bit) 
         */
        if (propDefined(&vocabWords, PropDefDirectly))
            target.initializeVocabWith(vocabWords);
        if (propDefined(&gcVocab, PropDefDirectly))
            target.initializeVocabWith(gcVocab);

        /*
         *   TODO: Zde je to tematicky spatne, nepotrebuji mechanismus
         *   dedicnosti, staci vzit na nejvyssi urovni.
         */
        if (propDefined(&gcName, PropDefDirectly))
            target.initializeNameWith(gcName);

        /* add vocabulary from each of our superclasses */
        foreach (local sc in getSuperclassList())
            sc.inheritVocab(target, done);
    }

    /*
     *   Initialize our vocabulary from the given string.  This parses the
     *   given vocabulary initializer string and adds the words defined in
     *   the string to the dictionary.
     *
     *   Note that this parsing is intentionally located in the
     *   English-specific part of the library, because it is expected that
     *   other languages will want to define their own vocabulary
     *   initialization string formats.  For example, a language with
     *   gendered nouns might want to use gendered articles in the
     *   initializer string as an author-friendly way of defining noun
     *   gender; languages with inflected (declined) nouns and/or
     *   adjectives might want to encode inflected forms in the
     *   initializer.  Non-English language implementations are free to
     *   completely redefine the format - there's no need to follow the
     *   conventions of the English format in other languages where
     *   different formats would be more convenient.
     */
    initializeVocabWith(str)
    {
        local sectPart;
        local modList = [];

        /* start off in the adjective section */
        sectPart = &adjective;

        /* scan the string until we run out of text */
        while (str != '')
        {
            local len;
            local cur;

            /*
             *   if it starts with a quote, find the close quote;
             *   otherwise, find the end of the current token by seeking
             *   the next delimiter
             */
            if (str.startsWith('"'))
            {
                /* find the close quote */
                len = str.find('"', 2);
            }
            else
            {
                /* no quotes - find the next delimiter */
                len = rexMatch('<^space|star|/>*', str);
            }

            /* if there's no match, use the whole rest of the string */
            if (len == nil)
                len = str.length();

            /* if there's anything before the delimiter, extract it */
            if (len != 0)
            {
                /* extract the part up to but not including the delimiter */
                cur = str.substr(1, len);

                /*
                 *   if we're in the adjectives, and either this is the
                 *   last token or the next delimiter is not a space, this
                 *   is implicitly a noun
                 */
                if (sectPart == &adjective
                    && (len == str.length()
                        || str.substr(len + 1, 1) != ' '))
                {
                    /* move to the noun section */
                    sectPart = &noun;
                }

                /*
                 *   if the word isn't a single hyphen (in which case it's
                 *   a null word placeholder, not an actual vocabulary
                 *   word), add it to our own appropriate part-of-speech
                 *   property and to the dictionary
                 */
                if (cur != '-')
                {
                    /*
                     *   by default, use the part of speech of the current
                     *   string section as the part of speech for this
                     *   word
                     */
                    local wordPart = sectPart;

                    /*
                     *   Check for parentheses, which indicate that the
                     *   token is "weak."  This doesn't affect anything
                     *   about the token or its part of speech except that
                     *   we must include the token in our list of weak
                     *   tokens.
                     */
                    if (cur.startsWith('(') && cur.endsWith(')'))
                    {
                        /* it's a weak token - remove the parens */
                        cur = cur.substr(2, cur.length() - 2);

                        /*
                         *   if we don't have a weak token list yet,
                         *   create the list
                         */
                        if (weakTokens == nil)
                            weakTokens = [];

                        /* add the token to the weak list */
                        weakTokens += cur;
                    }

                    /*
                     *   Check for special formats: quoted strings,
                     *   apostrophe-S words.  These formats are mutually
                     *   exclusive.
                     */
                    if (cur.startsWith('"'))
                    {
                        /*
                         *   It's a quoted string, so it's a literal
                         *   adjective.
                         */

                        /* remove the quote(s) */
                        if (cur.endsWith('"'))
                            cur = cur.substr(2, cur.length() - 2);
                        else
                            cur = cur.substr(2);

                        /* change the part of speech to 'literal adjective' */
                        wordPart = &literalAdjective;
                    }
                    else if (cur.endsWith('\'s'))
                    {
                        /*
                         *   It's an apostrophe-s word.  Remove the "'s"
                         *   suffix and add the root word using adjApostS
                         *   as the part of speech.  The grammar rules are
                         *   defined to allow this part of speech to be
                         *   used exclusively with "'s" suffixes in input.
                         *   Since the tokenizer always pulls the "'s"
                         *   suffix off of a word in the input, we have to
                         *   store any vocabulary words with "'s" suffixes
                         *   the same way, with the "'s" suffixes removed.
                         */

                        /* change the part of speech to adjApostS */
                        wordPart = &adjApostS;

                        /* remove the "'s" suffix from the string */
                        cur = cur.substr(1, cur.length() - 2);
                    }

                    /* add the word to our own list for this part of speech */
                    if (self.(wordPart) == nil)
                        self.(wordPart) = [cur];
                    else
                        self.(wordPart) += cur;

                    /* add it to the dictionary */
                    cmdDict.addWord(self, cur, wordPart);

                    if (cur.endsWith('.'))
                    {
                        local abbr;

                        /*
                         *   It ends with a period, so this is an
                         *   abbreviated word.  Enter the abbreviation
                         *   both with and without the period.  The normal
                         *   handling will enter it with the period, so we
                         *   only need to enter it specifically without.
                         */
                        abbr = cur.substr(1, cur.length() - 1);
                        self.(wordPart) += abbr;
                        cmdDict.addWord(self, abbr, wordPart);
                    }

                    /* note that we added to this list */
                    if (modList.indexOf(wordPart) == nil)
                        modList += wordPart;
                }
            }

            /* if we have a delimiter, see what we have */
            if (len + 1 < str.length())
            {
                /* check the delimiter */
                switch(str.substr(len + 1, 1))
                {
                case ' ':
                    /* stick with the current part */
                    break;

                case '*':
                    /* start plurals */
                    sectPart = &plural;
                    break;

                case '/':
                    /* start alternative nouns */
                    sectPart = &noun;
                    break;
                }

                /* remove the part up to and including the delimiter */
                str = str.substr(len + 2);

                /* skip any additional spaces following the delimiter */
                if ((len = rexMatch('<space>+', str)) != nil)
                    str = str.substr(len + 1);
            }
            else
            {
                /* we've exhausted the string - we're done */
                break;
            }
        }

        /* uniquify each word list we updated */
        foreach (local p in modList)
            self.(p) = self.(p).getUnique();
    }
    initializeNameWith(str)
    {
        local sectPart;
        
        /* start off in the second grammatical case section */
        sectPart = &nameKohoCeho;

        /* scan the string until we run out of text */
        while (str != '')
        {
            local len;
            local cur;

            /*
             *   if it starts with a quote, find the close quote;
             *   otherwise, find the end of the current token by seeking
             *   the next delimiter
             */
            if (str.startsWith('"'))
            {
                /* find the close quote */
                len = str.find('"', 2);
            }
            else
            {
                /* no quotes - find the next delimiter */
                len = rexMatch('<^,>*', str);
            }

            /* if there's no match, use the whole rest of the string */
            if (len == nil)
                len = str.length();
            
            /* if there's anything before the delimiter, extract it */
            if (len != 0)
            {
                /* extract the part up to but not including the delimiter */
                cur = str.substr(1, len);
                
                /*
                 *   if the word isn't a single hyphen (in which case it's
                 *   a null word placeholder, not an actual vocabulary
                 *   word), add it to our own appropriate part-of-speech
                 *   property and to the dictionary
                 */
                if (cur != '-')
                {
                    /*
                     *   by default, use the part of speech of the current
                     *   string section as the part of speech for this
                     *   word
                     */
                    local wordPart = sectPart;

                    /*
                     *   Check for special formats: quoted strings,
                     *   apostrophe-S words.  These formats are mutually
                     *   exclusive.
                     */
                    if (cur.startsWith('"'))
                    {
                        /*
                         *   It's a quoted string, so it's a literal
                         *   adjective.
                         */

                        /* remove the quote(s) */
                        if (cur.endsWith('"'))
                            cur = cur.substr(2, cur.length() - 2);
                        else
                            cur = cur.substr(2);
                    }

                    /* add the word to our own list for this part of speech */
                    self.(wordPart) = cur;
                }
            }

            /* if we have a delimiter, move to the next grammatical case */
            if (len + 1 < str.length())
            {
                /* switch to the next case */
                if(sectPart == &nameKohoCeho) sectPart = &nameKomuCemu;
                else if(sectPart == &nameKomuCemu) sectPart = &nameKohoCo;
                else if(sectPart == &nameKohoCo) sectPart = &nameKomCem;
                else if(sectPart == &nameKomCem) sectPart = &nameKymCim;
                else if(sectPart == &nameKymCim) sectPart = &pluralName;
                else if(sectPart == &pluralName) sectPart = &pluralNameKohoCeho;
                else if(sectPart == &pluralNameKohoCeho) sectPart = &pluralNameKomuCemu;
                else if(sectPart == &pluralNameKomuCemu) sectPart = &pluralNameKohoCo;
                else if(sectPart == &pluralNameKohoCo) sectPart = &pluralNameKomCem;
                else if(sectPart == &pluralNameKomCem) sectPart = &pluralNameKymCim;
                else if(sectPart == &pluralNameKymCim) break;

                /* remove the part up to and including the delimiter */
                str = str.substr(len + 2);
                
                /* skip any additional spaces following the delimiter */
                if ((len = rexMatch('<space>+', str)) != nil)
                    str = str.substr(len + 1);
            }
            else
            {
                /* we've exhausted the string - we're done */
                break;
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for Thing.  This class contains the
 *   methods and properties of Thing that need to be replaced when the
 *   library is translated to another language.
 *   
 *   The properties and methods defined here should generally never be used
 *   by language-independent library code, because everything defined here
 *   is specific to English.  Translators are thus free to change the
 *   entire scheme defined here.  For example, the notions of number and
 *   gender are confined to the English part of the library; other language
 *   implementations can completely replace these attributes, so they're
 *   not constrained to emulate their own number and gender systems with
 *   the English system.  
 */
modify Thing
    /*
     *   Flag that this object's name is rendered as a plural (this
     *   applies to both a singular noun with plural usage, such as
     *   "pants" or "scissors," and an object used in the world model to
     *   represent a collection of real-world objects, such as "shrubs").
     *
     *   V češtině jsou to podstatná jména pomnožná, tedy jména, která mají
     *   pouze tvar množného čísla. Např. ústa, dveře, nůžky.
     */
    isPlural = nil

    /*
     *   Flag that this is object's name is a "mass noun" - that is, a
     *   noun denoting a continuous (effectively infinitely divisible)
     *   substance or material, such as water, wood, or popcorn; and
     *   certain abstract concepts, such as knowledge or beauty.  Mass
     *   nouns are never rendered in the plural, and use different
     *   determiners than ordinary ("count") nouns: "some popcorn" vs "a
     *   kernel", for example.
     *
     *   V češtině jsou to jména, která mají pouze nebo převážně tvary čísla
     *   jednotného, tj. abstraktní jména, jména látková a hromadná. Např.
     *   lidstvo, voda, písek.
     */
    isMassNoun = nil

    /*
     *   Flags indicating that the object should be referred to with
     *   gendered pronouns (such as 'he' or 'she' rather than 'it').
     *
     *   Note that these flags aren't mutually exclusive, so it's legal
     *   for the object to have both masculine and feminine usage.  This
     *   can be useful when creating collective objects that represent
     *   more than one individual, for example.
     *
     *   Čeština rozlišuje rod mužský (životný a neživotný), ženský
     *   a střední. Postupně nahradíme původní isHim/isHer českým rodem.
     *   Vlastnost gender bude kódována číslem (abychom případně mohli
     *   číslo použít jako index do nějaké tabulky) následovně:
     *
     *   1 - mužský životný
     *   2 - mužský neživotný
     *   3 - ženský
     *   4 - střední
     *
     *   Jako výchozí hodnotu zvolíme střední rod, protože zkrátka nějakou
     *   výchozí hodnotu potřebujeme a žádný lepší klíč nemáme.
     */
    gender = 4
    parsedAs = nil
    changeGender = ''

    /*
     *   Test to see if we can match the pronouns 'him', 'her', 'it', and
     *   'them'.  By default, these simply test the corresponding isXxx
     *   flags (except 'canMatchThem', which tests 'isPlural' to see if the
     *   name has plural usage).
     */
    canMatchHim = (gender == 1 || gender == 2 ? true : nil)
    canMatchHer = (gender == 3 ? true : nil)
    canMatchIt = (gender == 4 ? true : nil)
    canMatchThem = (isPlural)

    /* can we match the given PronounXxx pronoun type specifier? */
    canMatchPronounType(typ)
    {
        /* check the type, and return the appropriate indicator property */
        switch (typ)
        {
        case PronounHim:
            return canMatchHim;

        case PronounHer:
            return canMatchHer;

        case PronounIt:
            return canMatchIt;

        case PronounThem:
            return canMatchThem;

        default:
            return nil;
        }
    }

    /*
     *   The grammatical cardinality of this item when it appears in a
     *   list.  This is used to ensure verb agreement when mentioning the
     *   item in a list of items.  ("Cardinality" is a fancy word for "how
     *   many items does this look like").
     *
     *   English only distinguishes two degrees of cardinality in its
     *   grammar: one, or many.  That is, when constructing a sentence, the
     *   only thing the grammar cares about is whether an object is
     *   singular or plural: IT IS on the table, THEY ARE on the table.
     *   Since English only distinguishes these two degrees, two is the
     *   same as a hundred is the same as a million for grammatical
     *   purposes, so we'll consider our cardinality to be 2 if we're
     *   plural, 1 otherwise.
     *
     *   Some languages don't express cardinality at all in their grammar,
     *   and others distinguish cardinality in greater detail than just
     *   singular-vs-plural, which is why this method has to be in the
     *   language-specific part of the library.
     */
    listCardinality(lister) { return isPlural ? 2 : 1; }

    /*
     *   Proper name flag.  This indicates that the 'name' property is the
     *   name of a person or place.  We consider proper names to be fully
     *   qualified, so we don't add articles for variations on the name
     *   such as 'theName'.
     *
     *   CZ: We don't use it in any way, but is quite common in examples for
     *   further funcionality built on top of it.
     */
    isProperName = nil

    /*
     *   The name of the object - this is a string giving the object's
     *   short description, for constructing sentences that refer to the
     *   object by name.  Each instance should override this to define the
     *   name of the object.  This string should not contain any articles;
     *   we use this string as the root to generate various forms of the
     *   object's name for use in different places in sentences.
     *
     *   V češtině samozřejmě potřebujeme název objektu v mnoha pádech pro
     *   použití v různých situacích. Proto přidáme vlastnosti pokrývající
     *   další pády a jako výchozí hodnotu do nich vložíme první pád
     *   z vlastnosti name.
     */
    name = ''
    nameKohoCeho = (name)
    nameKomuCemu = (name)
    nameKohoCo = (name)
    nameKomCem = (name)
    nameKymCim = (name)
    
    nameCase(gc)
    {
        switch(gc)
    	{
        case 1:
        default:
            return name;
        case 2:
            return nameKohoCeho;
    	case 3:
            return nameKomuCemu;
        case 4:
            return nameKohoCo;
    	case 6:
            return nameKomCem;
        case 7:
            return nameKymCim;
    	}
    }

    /*
     *   The name of the object, for the purposes of disambiguation
     *   prompts.  This should almost always be the object's ordinary
     *   name, so we return self.name by default.
     *
     *   In rare cases, it might be desirable to override this.  In
     *   particular, if a game has two objects that are NOT defined as
     *   basic equivalents of one another (which means that the parser
     *   will always ask for disambiguation when the two are ambiguous
     *   with one another), but the two nonetheless have identical 'name'
     *   properties, this property should be overridden for one or both
     *   objects to give them different names.  This will ensure that we
     *   avoid asking questions of the form "which do you mean, the coin,
     *   or the coin?".  In most cases, non-equivalent objects will have
     *   distinct 'name' properties to begin with, so this is not usually
     *   an issue.
     *
     *   When overriding this method, take care to override
     *   theDisambigName, aDisambigName, countDisambigName, and/or
     *   pluralDisambigName as needed.  Those routines must be overridden
     *   only when the default algorithms for determining articles and
     *   plurals fail to work properly for the disambigName (for example,
     *   the indefinite article algorithm fails with silent-h words like
     *   "hour", so if disambigName is "hour", aDisambigName must be
     *   overridden).  In most cases, the automatic algorithms will
     *   produce acceptable results, so the default implementations of
     *   these other routines can be used without customization.
     *
     *   Při řešení nejasností se ptáme na objekt čtvrtým pádem.
     *   (Koho/co máš na mysli? Nejprve bereš koho/co?)
     */
    disambigName = (name)
    disambigNameKohoCeho = (nameKohoCeho)
    disambigNameKomuCemu = (nameKomuCemu)
    disambigNameKohoCo = (nameKohoCo)
    disambigNameKomCem = (nameKomCem)
    disambigNameKymCim = (nameKymCim)
    
    disambigNameCase(gc)
    {
        switch(gc)
    	{
        case 1:
        default:
            return disambigName;
        case 2:
            return disambigNameKohoCeho;
    	case 3:
            return disambigNameKomuCemu;
        case 4:
            return disambigNameKohoCo;
    	case 6:
            return disambigNameKomCem;
        case 7:
            return disambigNameKymCim;
    	}
    }

    /*
     *   The "equivalence key" is the value we use to group equivalent
     *   objects.  Note that we can only treat objects as equivalent when
     *   they're explicitly marked with isEquivalent=true, so the
     *   equivalence key is irrelevant for objects not so marked.
     *   
     *   Since the main point of equivalence is to allow creation of groups
     *   of like-named objects that are interchangeable in listings and in
     *   command input, we use the basic disambiguation name as the
     *   equivalence key.  
     */
    equivalenceKey = (disambigName)

    /*
     *   The counted name for disambiguation prompts.  We use the same
     *   logic here as in theDisambigName.
     */
    countDisambigName(cnt)
    {
        return (name == disambigName && pluralName == pluralDisambigName
                ? countName(cnt)
                : countNameFrom(cnt, disambigName, pluralDisambigName));
    }

    /*
     *   The plural name for disambiguation prompts.  We use the same
     *   logic here as in theDisambigName.
     *
     *   V češtině nemůžeme nagenerovat plurál automaticky, takže použijeme
     *   pluralName a autor musí předefinovat stejně, jako u disambigName.
     *
     *   TODO: Mělo by tu být spíš pluralNameKohoCo, ale ty asi zatim vubec
     *   nepouzivam? A k čemu dalšímu se pluralName vůbec používá?
     */
    pluralDisambigName = (pluralName)

    /*
     *   The name of the object, for the purposes of disambiguation prompts
     *   to disambiguation among this object and basic equivalents of this
     *   object (i.e., objects of the same class marked with
     *   isEquivalent=true).
     *
     *   This is used in disambiguation prompts in place of the actual text
     *   typed by the user.  For example, suppose the user types ">take
     *   coin", then we ask for help disambiguating, and the player types
     *   ">gold".  This narrows things down to, say, three gold coins, but
     *   they're in different locations so we need to ask for further
     *   disambiguation.  Normally, we ask "which gold do you mean",
     *   because the player typed "gold" in the input.  Once we're down to
     *   equivalents, we don't have to rely on the input text any more,
     *   which is good because the input text could be fragmentary (as in
     *   our present example).  Since we have only equivalents, we can use
     *   the actual name of the objects (they're all the same, after all).
     *   This property gives the name we use.
     *
     *   For English, this is simply the object's ordinary disambiguation
     *   name.  This property is separate from 'name' and 'disambigName'
     *   for the sake of languages that need to use an inflected form in
     *   this context.
     */
    disambigEquivName = (disambigName)

    /*
     *   Single-item listing description.  This is used to display the
     *   item when it appears as a single (non-grouped) item in a list.
     *   By default, we just show the indefinite article description.
     *
     *   Ve vetsine seznamu je vhodny 4. pad ("U sebe mas koho/co", "vidis
     *   tu koho/co". Jenze "Na povrchu je kdo/co." a "Kroužek na klíče
     *   s navlečeným kým/čím."
     */
    listName = (name)
    listNameKohoCo = (nameKohoCo)
    listNameKomuCemu = (nameKomuCemu)
    listNameKymCim = (nameKymCim)

    /*
     *   Return a string giving the "counted name" of the object - that is,
     *   a phrase describing the given number of the object.  For example,
     *   for a red book, and a count of 5, we might return "five red
     *   books".  By default, we use countNameFrom() to construct a phrase
     *   from the count and either our regular (singular) 'name' property
     *   or our 'pluralName' property, according to whether count is 1 or
     *   more than 1.  
     */
    /*
     *   TODO: Velmi prozatimni, nektere listery nemusi chtit 4. pad. Puvodne
     *   tu bylo: return countNameFrom(count, name, pluralName);
     */
    countName(count) { return countNameCase(count, 4); }
    countNameCase(count, gc)
    {
        return countNameFrom(count, nameCase(gc),
            pluralNameCase(count >= 5 ? 2 : gc));
    }

    /*
     *   Returns a string giving a count applied to the name string.  The
     *   name must be given in both singular and plural forms.
     */
    countNameFrom(count, singularStr, pluralStr)
    {
        /* if the count is one, use 'one' plus the singular name */
        if(count == 1) return ['jeden', 'jeden', 'jedna', 'jedno'][gender]
            + ' ' + singularStr;

        /*
         *   Get the number followed by a space - spell out numbers below
         *   100, but use numerals to denote larger numbers.  Append the
         *   plural name to the number and return the result.
         */
        return spellIntBelowExt(count, gender, 100, 0, DigitFormatGroupSep)
            + ' ' + pluralStr;
    }

    getGender()
    {
        local g = gender;
        local lst = changeGender.split(R',<space>*');

        if(parsedAs != nil)
        {
            foreach(local cur in lst)
            {
                local a = cur.split(':');

                /* Je pravidlo negativní? */
                local neg = nil;
                if(a[1].startsWith('!'))
                {
                    neg = true;
                    a[1] = a[1].substr(2);
                }

                /*
                 *   parsedAs muze byt nekdy take prazdne, pokud napr. dam prikaz:
                 *
                 *   >řekni jí o počasí
                 */
                local res = parsedAs.find(a[1]);
                if(!neg && res != nil || neg && res == nil)
                {
                    g = toInteger(a[2]);
                }
            }
        }

        return g;
    }

    /*
     *   Get the 'pronoun selector' for the various pronoun methods.  This
     *   returns:
     *   
     *.  - singular neuter = 1
     *.  - singular masculine = 2
     *.  - singular feminine = 3
     *.  - plural = 4
     *
     *   Přeházíme na logičtější mužský = 1, ženský = 2, střední = 3, plurál = 4
     */
    pronounSelector
    {
        local g = getGender();

        return (isPlural || g > 4 ? 4 : g == 3 ? 2 : g == 4 ? 3 : 1);
    }

    /*
     *   get a string with the appropriate pronoun for the object for the
     *   nominative case, objective case, possessive adjective, possessive
     *   noun
     *
     *   Osobní zájmena:
     *
     *   1. pád - zajmenoTy
     *   2. pád - zajmenoTebe2J/N, zajmenoTe2J/N
     *   3. pád - zajmenoTobe3J/N, zajmenoTiJ/N
     *   4. pád - zajmenoTebe4J/N, zajmenoJej4J/N, zajmenoTe4J/N
     *   6. pád - zajmenoTobe6
     *   7. pád - zajmenoTebouJ/N
     */
    zajmenoTen
    {
        local g = getGender();

        return isPlural ? ['ti', 'ty', 'ty', 'ta'][g]
            : ['ten', 'ten', 'ta', 'to'][g];
    }

    zajmenoTy
    {
        return isPlural && gender == 3 ? 'ony'
            : isPlural && gender == 4 ? 'ona'
            : ['on', 'ona', 'ono', 'oni'][pronounSelector];
    }
    zajmenoTebe2J { return ['jeho', 'jí', 'jeho', 'jich'][pronounSelector]; }
    zajmenoTebe2N { return ['něho', 'ní', 'něho', 'nich'][pronounSelector]; }
    zajmenoTe2J { return ['ho', 'jí', 'ho', 'jich'][pronounSelector]; }
    zajmenoTe2N { return ['ho', 'ní', 'ho', 'nich'][pronounSelector]; }
    zajmenoTobe3J { return ['jemu', 'jí', 'jemu', 'jim'][pronounSelector]; }
    zajmenoTobe3N { return ['němu', 'ní', 'němu', 'nim'][pronounSelector]; }
    zajmenoTiJ { return ['mu', 'jí', 'mu', 'jim'][pronounSelector]; }
    zajmenoTiN { return ['mu', 'ní', 'mu', 'nim'][pronounSelector]; }
    zajmenoTebe4J { return ['jeho', 'ji', 'jej', 'je'][pronounSelector]; }
    zajmenoTebe4N { return ['něho', 'ni', 'něj', 'ně'][pronounSelector]; }
    zajmenoJej4J { return ['jej', 'ji', 'jej', 'je'][pronounSelector]; }
    zajmenoJej4N { return ['něj', 'ni', 'něj', 'ně'][pronounSelector]; }
    zajmenoTe4J { return ['ho', 'ji', 'ho', 'je'][pronounSelector]; }
    zajmenoTe4N { return ['ho', 'ni', 'ho', 'ně'][pronounSelector]; }
    zajmenoTobe6 { return ['něm', 'ní', 'něm', 'nich'][pronounSelector]; }
    zajmenoTebouJ { return ['jím', 'jí', 'jím', 'jimi'][pronounSelector]; }
    zajmenoTebouN { return ['ním', 'ní', 'ním', 'nimi'][pronounSelector]; }

    /*
     *   Zvratná osobní zájmena:
     *
     *   2. pád - zajmenoSebe
     *   3. pád - zajmenoSobe, zajmenoSi
     *   4. pád - zajmenoSebe, zajmenoSe
     *   6. pád - zajmenoSobe
     *   6. pád - zajmenoSebou
     */
    zajmenoSebe { return 'sebe'; }
    zajmenoSobe { return 'sobě'; }
    zajmenoSi { return 'si'; }
    zajmenoSe { return 'se'; }
    zajmenoSebou { return 'sebou'; }

    /*
     *   Přivlastňovací zájmena
     *
     *   1. pád - zajmenoTvuj (m), zajmenoTvoje (ž, s - ale to se ma rozlisit BorisovA zena / BorisovO mesto), TODO: tvá (ž), tvé (s)
     *   4. pád - zajmenoTvoji (ž)
     */
    zajmenoTvuj { return ['jeho', 'její', 'jeho', 'jejich'][pronounSelector]; }
    zajmenoTvoje { return ['jeho', 'její', 'jeho', 'jejich'][pronounSelector]; }
    zajmenoTvoji { return ['jeho', 'její', 'jeho', 'jejich'][pronounSelector]; }

    /* ukazovací zájmena */
    zajmenoSam
    {
        return ['sám', 'sám', 'sama', 'samo', 'sami', 'samy', 'samy', 'sama']
            [gender + (isPlural ? 4 : 0)];
    }

    /*
     *   Přídavná jména:
     *
     *   1. pád - pridavneJmenoMlady
     *   3. pád - pridavneJmenoMlademu
     *   6. pád - pridavneJmenoMladem
     *   7. pád - pridavneJmenoMladym
     *
     *   http://cs.wikipedia.org/wiki/%C4%8Cesk%C3%A1_p%C5%99%C3%ADdavn%C3%A1_jm%C3%A9na
     */
    pridavneJmenoMlady
    {
        return ['ý', 'ý', 'á', 'é', 'í', 'é', 'é', 'á'][gender + (isPlural ? 4
            : 0)];
    }
    pridavneJmenoMlademu
    {
        return ['ému', 'ému', 'é', 'ému', 'ým', 'ým', 'ým', 'ým'][gender +
            (isPlural ? 4 : 0)];
    }
    pridavneJmenoMladem
    {
        return ['ém', 'ém', 'é', 'ém', 'ých', 'ých', 'ých', 'ých'][gender +
            (isPlural ? 4 : 0)];
    }
    pridavneJmenoMladym
    {
        return ['ým', 'ým', 'ou', 'ým', 'ými', 'ými', 'ými', 'ými'][gender +
            (isPlural ? 4 : 0)];
    }
    
    /* Např. Borisův kafovak - http://prirucka.ujc.cas.cz/?id=400 */
    pridavneJmenoPrivlastnovaci(selector)
    {
        if(name == zajmenoTy) return zajmenoTvuj;

        if(gender <= 2)
            return name + ['ův', 'ův', 'ova', 'ovo', 'ovi', 'ovy', 'ovy', 'ova'][selector];
        else
            return name + ['in', 'in', 'ina', 'ivo', 'ini', 'iny', 'iny', 'ina'][selector];
    }
    zvratnePridavneJmenoPrivlastnovaci(selector, gc)
    {
        if(name == zajmenoTy)
            return ['svůj', 'svého', 'svému', 'svůj', 'svůj', 'svém', 'svým'][gc];

        if(gender <= 2)
            return name + ['ův', 'ův', 'ova', 'ovo', 'ovi', 'ovy', 'ovy', 'ova'][selector];
        else
            return name + ['in', 'in', 'ina', 'ivo', 'ini', 'iny', 'iny', 'ina'][selector];
    }

    /*
     *   Get the name with a definite article ("the box").  By default, we
     *   use our standard definite article algorithm to apply an article
     *   to self.name.
     *
     *   The name returned must be in the nominative case (which makes no
     *   difference unless the name is a pronoun, since in English
     *   ordinary nouns don't vary according to how they're used in a
     *   sentence).
     *
     *   V češtine vůbec nepotřebujeme.
     */

    /*
     *   theName with my nominal owner explicitly stated, if we have a
     *   nominal owner: "your backpack," "Bob's flashlight."  If we have
     *   no nominal owner, this is simply my theName.
     */
    nameWithOwner()
    {
        local owner;

        /*
         *   if we have a nominal owner, show with our owner name;
         *   otherwise, just show our regular theName
         *
         *   TODO: Else vrací "Červený bonbón na *zemi*" Je potřeba vymyslet
         *   jinak, tohle nedává smysl.
         */
        if ((owner = getNominalOwner()) != nil)
            /* vybitý akumulátor v *tvém laseru* */
            return (gender == 3 ? owner.zajmenoTve : owner.zajmenoTvem) + ' '
                + nameKomCem;
        else
            return nameKomCem;
    }

    /*
     *   Default preposition to use when an object is in/on this object.
     *   By default, we use 'in' as the preposition; subclasses can
     *   override to use others (such as 'on' for a surface).
     *
     *   Default preposition to use when an actor is in/on this object (as
     *   a nested location), and full prepositional phrase, with no article
     *   and with an indefinite article.  By default, we use the objInPrep
     *   for actors as well.
     *
     *   Důležité je si uvědomit, že objInPrep nemá konkrétně znamenat jen "v",
     *   ale i "na", "pod", "za" apod. Vyjadřuje přítomnost jiného objektu
     *   v tomto, ať už je to kontejner, surface apod.
     *
     *   Vyjádření vhodného pádu provedeme novou vlastností. V anglické verzi
     *   to není potřeba, protože se přizpůsobuje jen objIn/OutOf/IntoName,
     *   ale v češtině se hodí k ovlivnění pádu zájmena "který" (např. ve kterém
     *   vs. pod kterým).
     */
    objInPrep = 'v'
    objInCase = 6

    /* preposition to use when an actor is being removed from this location */
    objOutOfPrep = 'ven z'

    /* preposition to use when an actor is being moved into this location */
    /* TODO: mam k dispozici objIntoCase */
    objIntoPrep
    {
        if (objInPrep is in ('v', 've'))
            return 'do';
        else
            return objInPrep;
    }
    objIntoCase = 2

    /*
     *   describe an actor as being in/being removed from/being moved into
     *   this location
     */
    objInName = (objInPrep + ' ' + nameCase(objInCase))
    objOutOfName = (objOutOfPrep + ' ' + nameKohoCeho)
    objIntoName = (objIntoPrep + ' ' + nameCase(objIntoCase))

    /*
     *   A prepositional phrase that can be used to describe things that
     *   are in this room as seen from a remote point of view.  This
     *   should be something along the lines of "in the airlock", "at the
     *   end of the alley", or "on the lawn".
     *
     *   'pov' is the point of view from which we're seeing this room;
     *   this might be
     *
     *   We use this phrase in cases where we need to describe things in
     *   this room when viewed from a point of view outside of the room
     *   (i.e., in a different top-level room).  By default, we'll use our
     *   objInName.
     */
    inRoomName(pov) { return objInName; }

    /*
     *   Provide the prepositional phrase for an object being put into me.
     *   For a container, for example, this would say "into the box"; for
     *   a surface, it would say "onto the table."  By default, we return
     *   our library message given by our putDestMessage property; this
     *   default is suitable for most cases, but individual objects can
     *   customize as needed.  When customizing this, be sure to make the
     *   phrase suitable for use in sentences like "You put the book
     *   <<putInName>>" and "The book falls <<putInName>>" - the phrase
     *   should be suitable for a verb indicating active motion by the
     *   object being received.
     */
    putInName() { return gLibMessages.(putDestMessage)(self); }

    /*
     *   Get a description of an object within this object, describing the
     *   object's location as this object.  By default, we'll append "in
     *   <theName>" to the given object name.
     */
    childInName(childName)
        { return childInNameGen(childName, nameCase(objInCase)); }

    /*
     *   Get a description of an object within this object, showing the
     *   owner of this object.  This is similar to childInName, but
     *   explicitly shows the owner of the containing object, if any: "the
     *   flashlight in bob's backpack".
     *
     *   "Který červený bonbón máš na mysli: *červený bonbón na zemi*, nebo
     *   tvůj červený bonbón?" a nameWithOwner je právě to "zemi" a zřejmě
     *   může pojmenovávat i vlastníka.
     */
    childInNameWithOwner(childName)
        { return childInNameGen(childName, nameWithOwner); }

    /*
     *   get a description of an object within this object, as seen from a
     *   remote location
     */
    childInRemoteName(childName, pov)
        { return childInNameGen(childName, inRoomName(pov)); }

    /*
     *   Base routine for generating childInName and related names.  Takes
     *   the name to use for the child and the name to use for me, and
     *   combines them appropriately.
     *
     *   In most cases, this is the only one of the various childInName
     *   methods that needs to be overridden per subclass, since the others
     *   are defined in terms of this one.  Note also that if the only
     *   thing you need to do is change the preposition from 'in' to
     *   something else, you can just override objInPrep instead.
     */
    childInNameGen(childName, myName)
        { return childName + ' ' + objInPrep + ' ' + myName; }

    /*
     *   Get my name (in various forms) distinguished by my owner or
     *   location.
     *
     *   If the object has an owner, and either we're giving priority to
     *   the owner or our immediate location is the same as the owner,
     *   we'll show using a possessive form with the owner ("bob's
     *   flashlight").  Otherwise, we'll show the name distinguished by
     *   our immediate container ("the flashlight in the backpack").
     *
     *   These are used by the ownership and location distinguishers to
     *   list objects according to owners in disambiguation lists.  The
     *   ownership distinguisher gives priority to naming by ownership,
     *   regardless of the containment relationship between owner and
     *   self; the location distinguisher gives priority to naming by
     *   location, showing the owner only if the owner is the same as the
     *   location.
     *
     *   We will presume that objects with proper names are never
     *   indistinguishable from other objects with proper names, so we
     *   won't worry about cases like "Bob's Bill".  This leaves us free
     *   to use appropriate articles in all cases.
     */
    nameOwnerLoc(ownerPriority, gc)
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil
            && (ownerPriority || isDirectlyIn(owner)))
        {
            /*
             *   we have an owner - show as "one of Bob's items" (or just
             *   "Bob's items" if this is a mass noun or a proper name)
             *
             *   Tohle bylo v aNameOwnerLoc, jinak stejne jako theNameOwnerLoc
             *
             *   ret = owner.nameKohoCeho + ' ' + pluralName;
             *   if (!isMassNoun && !isPlural)
             *       ret = 'one of ' + ret;
             */
            /*
             *   we have an owner - show as "Bob's item"
             *
             *   "A který zelený bonbón má na mysli? Zelený bonbón na zemi,
             *   nebo *její zelený bonbón*?"
             *
             *   Rozlišujeme rod cílového objektu - tvůj hrad, tvoje růže.
             *   TODO: Nema tu byt misto zajmena spise jmeno?
             */
            if(gc > 1)
                return owner.zvratnePridavneJmenoPrivlastnovaci(gender + (isPlural ? 4 : 0), gc)
                    + ' ' + nameCase(gc);
            else
                return owner.pridavneJmenoPrivlastnovaci(gender + (isPlural ? 4 : 0))
                    + ' ' + name;
        }
        else
        {
            /*
             *   we have no owner - show as "the item in the location"
             *
             *   "A který zelený bonbón má na mysli? *Zelený bonbón na zemi*,
             *   nebo she’s zelený bonbón?"
             */
            return location.childInNameWithOwner(name);
        }
    }
    countNameOwnerLoc(cnt, ownerPriority, gc)
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil
            && (ownerPriority || isDirectlyIn(owner)))
        {
            /* we have an owner - show as "Bob's five items" */
            /* TODO: Tohle je špatně postavený fragment, má být přídavné jméno? */
            return owner.nameKohoCeho + ' ' + countName(cnt);
        }
        else
        {
            /* we have no owner - show as "the five items in the location" */
            return location.childInNameWithOwner(countName(cnt));
        }
    }

    /*
     *   Note that I'm being used in a disambiguation prompt by
     *   owner/location.  If we're showing the owner, we'll set the
     *   antecedent for the owner's pronoun, if the owner is a 'him' or
     *   'her'; this allows the player to refer back to our prompt text
     *   with appropriate pronouns.
     */
    notePromptByOwnerLoc(ownerPriority)
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil
            && (ownerPriority || isDirectlyIn(owner)))
        {
            /* we are showing by owner - let the owner know about it */
            owner.notePromptByPossAdj();
        }
    }

    /*
     *   Note that we're being used in a prompt question with our
     *   possessive adjective.  If we're a 'him' or a 'her', set our
     *   pronoun antecedent so that the player's response to the prompt
     *   question can refer back to the prompt text by pronoun.
     */
    notePromptByPossAdj()
    {
        if (gender == 1 || gender == 2)
            gPlayerChar.setHim(self);
        if (gender == 3)
            gPlayerChar.setHer(self);
    }

    /*
     *   Get the default plural name.  By default, we'll use the
     *   algorithmic plural determination, which is based on the spelling
     *   of the name.
     *
     *   The algorithm won't always get it right, since some English
     *   plurals are irregular ("men", "children", "Attorneys General").
     *   When the name doesn't fit the regular spelling patterns for
     *   plurals, the object should simply override this routine to return
     *   the correct plural name string.
     *
     *   V češtině se množné číslo nedá nagenerovat automaticky, takže
     *   použijeme jednotné a autoři si musí správný tvar explicitně
     *   zadat, pokud se ve hře vyskytuje objekt vícekrát. Viz též
     *   pluralDisambigName.
     */
    pluralName = (name)
    pluralNameKohoCeho = (pluralName)
    pluralNameKomuCemu = (pluralName)
    pluralNameKohoCo = (pluralName)
    pluralNameKomCem = (pluralName)
    pluralNameKymCim = (pluralName)

    pluralNameCase(gc)
    {
        switch(gc)
        {
        case 1:
        default:
            return pluralName;
        case 2:
            return pluralNameKohoCeho;
        case 3:
            return pluralNameKomuCemu;
        case 4:
            return pluralNameKohoCo;
        case 6:
            return pluralNameKomCem;
        case 7:
            return pluralNameKymCim;
        }
    }

    /*
     *   Koncovka slovesa v minulém čase
     *
     *   U slovesa v minulém čase je situace jednodušší, na slovesné třídě vůbec
     *   nezáleží a rozhoduje jen číslo a rod.
     */
    verbPastEnding
    {
        return (isPlural ? ['i', 'y', 'y', 'a'][gender] : ['', '', 'a', 'o'][gender]);
    }

    /* being verb agreeing with this object as subject */
    slovesoJe
    {
        return tSel(isPlural ? 'jsou' : 'je', ['byl', 'byl', 'byla', 'bylo',
            'byli', 'byly', 'byly', 'byla'][gender + (isPlural ? 4 : 0)]);
    }
    
    /* past tense being verb agreeing with object as subject */
    slovesoBytProMinulyCas
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'jsme' : 'jsem') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'jste' : 'jsi') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? '' : '') : '');
    }

    /*
     *   A few common irregular verbs and name-plus-verb constructs,
     *   defined for convenience.
     */
    slovesoNeni
    {
        return tSel(isPlural ? 'nejsou' : 'není',
            ['nebyl', 'nebyl', 'nebyla', 'nebylo', 'nebyli', 'nebyly',
            'nebyly', 'nebyla'][gender + (isPlural ? 4 : 0)]);
    }

    slovesoByt
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'jsme' : 'jsem') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'jste' : 'jsi') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'jsou' : 'je') : '');
    }
    slovesoNebyt
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'nejsme' : 'nejsem') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'nejste' : 'nejsi') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'nejsou' : 'není') : '');
    }
    slovesoBytPodmin
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'bychom' : 'bych') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'byste' : 'bys') : '')
            + (referralPerson == ThirdPerson ? 'by' : '');
    }
    slovesoChces
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'chceme' : 'chci') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'chcete' : 'chceš') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'chtějí' : 'chce') : '');
    }
    slovesoSel
    {
        return isPlural ? 'šl' + ['i', 'y', 'y', 'a'][gender]
            : ['šel', 'šel', 'šla', 'šlo'][gender];
    }
    slovesoJde
    {
        return tSel(isPlural ? 'jdou' : 'jde', slovesoSel());
    }

    /*
     *   Verb endings for regular '-s' verbs, agreeing with this object as
     *   the subject.  We define several methods each of which handles the
     *   past tense differently.
     *
     *   verbEndingS doesn't try to handle the past tense at all - use it
     *   only in places where you know for certain that you'll never need
     *   the past tense form, or in expressions constructed with the tSel
     *   macro: use verbEndingS as the macro's first argument, and specify
     *   the past tense ending explicitly as the second argument.  For
     *   example, you could generate the correctly conjugated form of the
     *   verb "to fit" for an object named "key" with an expression such
     *   as:
     *
     *   'fit' + tSel(key.verbEndingS, 'ted')
     *
     *   This would generate 'fit', 'fits', or 'fitted' according to number
     *   and tense.
     *
     *   verbEndingSD and verbEndingSEd return 'd' and 'ed' respectively in
     *   the past tense.
     *
     *   verbEndingSMessageBuilder_ is for internal use only: it assumes
     *   that the correct ending to be displayed in the past tense is
     *   stored in langMessageBuilder.pastEnding_.  It is used as part of
     *   the string parameter substitution mechanism.
     *
     *   Pravidelná slovesa
     *
     *   Mají pět slovesných tříd, podle kterých je ovlivněna koncovka slovesa
     *   a tak můžeme poskytnout substituční parametr. U Actora jsou tyto funkce
     *   přetížené a rozlišují ještě referralPerson, ale na neživé objekty se
     *   odkazujeme výhradně ve třetí osobě.
     */
    verbEndingEs { return isPlural ? 'ou' : 'e'; }
    verbEndingNes { return isPlural ? 'nou' : 'ne'; }
    verbEndingJes { return isPlural ? 'jí' : 'je'; }
    verbEndingIs { return isPlural ? 'í' : 'í'; }
    verbEndingAs { return isPlural ? 'ají' : 'á'; }

    /*
     *   Preposition vocalization
     */
    vocalize = []
    
    prepK { return vocalize.indexOf('k') != nil ? 'ke' : 'k'; }
    prepS { return vocalize.indexOf('s') != nil ? 'se' : 's'; }
    prepV { return vocalize.indexOf('v') != nil ? 've' : 'v'; }
    prepZ { return vocalize.indexOf('z') != nil ? 'ze' : 'z'; }

    makeVocalized(prep)
    {
        if(prep == 'k') return prepK;
        if(prep == 's') return prepS;
        if(prep == 'v') return prepV;
        if(prep == 'z') return prepZ;
        return prep;
    }

    /*
     *   Dummy name - this simply displays nothing; it's used for cases
     *   where messageBuilder substitutions want to refer to an object (for
     *   internal bookkeeping) without actually showing the name of the
     *   object in the output text.  This should always simply return an
     *   empty string.
     */
    dummyName = ''

    /*
     *   Invoke a property (with an optional argument list) on this object
     *   while temporarily switching to the present tense, and return the
     *   result.
     */
    propWithPresent(prop, [args])
    {
        return withPresent({: self.(prop)(args...)});
    }

    /*
     *   Method for internal use only: invoke on this object the property
     *   stored in langMessageBuilder.fixedTenseProp_ while temporarily
     *   switching to the present tense, and return the result.  This is
     *   used as part of the string parameter substitution mechanism.
     */
    propWithPresentMessageBuilder_
    {
        return propWithPresent(langMessageBuilder.fixedTenseProp_);
    }

    /*
     *   For the most part, "strike" has the same meaning as "hit", so
     *   define this as a synonym for "attack" most objects.  There are a
     *   few English idioms where "strike" means something different, as
     *   in "strike match" or "strike tent."
     */
    dobjFor(Strike) asDobjFor(Attack)

    showListItemGen(options, pov, infoTab, stateNameProp)
    {
        local info;
        local st;
        local stName;

        /* get my visual information from the point of view */
        info = infoTab[self];

        /*
         *   show the item's list name
         *
         *   Zde si vybereme správný pád k zobrazení položky seznamu. Dostaneme
         *   ho prostřednictvím vlajky z Listeru.
         */
        if(options & ListKdoCo)
            say(withVisualSenseInfo(pov, info, &listName));
        else if(options & ListKomuCemu)
            say(withVisualSenseInfo(pov, info, &listNameKomuCemu));
        else if(options & ListKymCim)
            say(withVisualSenseInfo(pov, info, &listNameKymCim));
        else
            say(withVisualSenseInfo(pov, info, &listNameKohoCo));

        /* 
         *   If we have a list state with a name, show it.  Note that to
         *   obtain the state name, we have to pass a list of the objects
         *   being listed to the stateNameProp method of the state object;
         *   we're the only object we're showing for this particular
         *   display list element, so we simply pass a single-element list
         *   containing 'self'.  
         */
        if ((st = getStateWithInfo(info, pov)) != nil
            && (stName = st.(stateNameProp)([self])) != nil)
        {
            /* we have a state with a name - show it */
            gLibMessages.showListState(stName);
        }
    }

    /* V originalu se Set pouziva ve smyslu polozit na, coz nechceme. */
    dobjFor(Set)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotSetToMsg); }
        action() { }
    }
;

modify ThingState
    inventoryNameKohoCo(lst) { return listNameKohoCo(lst); }
;

/* ------------------------------------------------------------------------ */
/*
 *   An object that uses the same name as another object.  This maps all of
 *   the properties involved in supplying the object's name, number, and
 *   other usage information from this object to a given target object, so
 *   that all messages involving this object use the same name as the
 *   target object.  This is a mix-in class that can be used with any other
 *   class.
 *   
 *   Note that we map only the *reported* name for the object.  We do NOT
 *   give this object any vocabulary from the other object; in other words,
 *   we don't enter this object into the dictionary with the other object's
 *   vocabulary words.  
 */
class NameAsOther: object
    /* the target object - we'll use the same name as this object */
    targetObj = nil

    /* map our naming and usage properties to the target object */
    isPlural = (targetObj.isPlural)
    isMassNoun = (targetObj.isMassNoun)
    gender = (targetObj.gender)
    name = (targetObj.name)
    nameKohoCeho = (targetObj.nameKohoCeho)
    nameKomuCemu = (targetObj.nameKomuCemu)
    nameKohoCo = (targetObj.nameKohoCo)
    nameKomCem = (targetObj.nameKomCem)
    nameKymCim = (targetObj.nameKymCim)

    /* map the derived name properties as well, in case any are overridden */
    disambigName = (targetObj.disambigName)
    countDisambigName(cnt) { return targetObj.countDisambigName(cnt); }
    disambigEquivName = (targetObj.disambigEquivName)
    listName = (targetObj.listName)
    listNameKomuCemu = (targetObj.listNameKomuCemu)
    listNameKohoCo = (targetObj.listNameKohoCo)
    listNameKymCim = (targetObj.listNameKymCim)
    countName(cnt) { return targetObj.countName(cnt); }
    countNameCase(cnt) { return targetObj.countNameCase(cnt); }

    /* map the pronoun properites, in case any are overridden */
    zajmenoSam = (targetObj.zajmenoSam)
    zajmenoTy = (targetObj.zajmenoTy) // TODO: Přidat ostatní zájmena

    nameWithOwner = (targetObj.nameWithOwner)
    nameOwnerLoc(ownerPri, gc)
        { return targetObj.nameOwnerLoc(ownerPri, gc); }
    countNameOwnerLoc(cnt, ownerPri, gc)
        { return targetObj.countNameOwnerLoc(cnt, ownerPri, gc); }
    notePromptByOwnerLoc(ownerPri)
        { targetObj.notePromptByOwnerLoc(ownerPri); }
    notePromptByPossAdj()
        { targetObj.notePromptByPossAdj(); }
    pluralName = (targetObj.pluralName)
    slovesoJe = (targetObj.slovesoJe)
;

/*
 *   Name as Parent - this is a special case of NameAsOther that uses the
 *   lexical parent of a nested object as the target object.  (The lexical
 *   parent is the enclosing object in a nested object definition; in other
 *   words, it's the object in which the nested object is embedded.)  
 */
class NameAsParent: NameAsOther
    targetObj = (lexicalParent)
;

/*
 *   ChildNameAsOther is a mix-in class that can be used with NameAsOther
 *   to add the various childInXxx naming to the mapped properties.  The
 *   childInXxx names are the names generated when another object is
 *   described as located within this object; by mapping these properties
 *   to our target object, we ensure that we use exactly the same phrasing
 *   as we would if the contained object were actually contained by our
 *   target rather than by us.
 *   
 *   Note that this should always be used in combination with NameAsOther:
 *   
 *   myObj: NameAsOther, ChildNameAsOther, Thing ...
 *   
 *   You can also use it the same way in combination with a subclass of
 *   NameAsOther, such as NameAsParent.  
 */
class ChildNameAsOther: object
    objInPrep = (targetObj.objInPrep)
    objInCase = (targetObj.objInCase)
    objOutOfPrep = (targetObj.objOutOfPrep)
    objIntoPrep = (targetObj.objIntoPrep)
    objIntoCase = (targetObj.objIntoCase)
    childInName(childName) { return targetObj.childInName(childName); }
    childInNameWithOwner(childName)
        { return targetObj.childInNameWithOwner(childName); }
    childInNameGen(childName, myName)
        { return targetObj.childInNameGen(childName, myName); }
    objInName = (targetObj.objInName)
    objOutOfName = (targetObj.objOutOfName)
    objIntoName = (targetObj.objIntoName)
;

/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for the specialized container types
 */
modify Surface
    /*
     *   objects contained in a Surface are described as being on the
     *   Surface
     */
    objInPrep = 'na'
    objOutOfPrep = 'z'
;

modify Underside
    objInPrep = 'pod'
    objInCase = 7
    objOutOfPrep = 'z pod'
;

modify RearContainer
    objInPrep = 'za'
    objInCase = 7
    objOutOfPrep = 'z poza'
;

/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for Actor.
 *   
 *   An Actor has a "referral person" setting, which determines how we
 *   refer to the actor; this is almost exclusively for the use of the
 *   player character.  The typical convention is that we refer to the
 *   player character in the second person, but a game can override this on
 *   an actor-by-actor basis.  
 */
modify Actor
    /* by default, use my pronoun for my name */
    /*
     *   Tyhle výchozí jména se typicky vztahují jen k postavě hráče. Z nich
     *   jsou pak nagenerovány věty "Nedovolíš Sally ho mít." vs. "Nedovolí
     *   *ti* ho mít" ze šablony "Nedovol{[íš holder]|il[a holder]
     *   [jsi holder]} {komučemu} {ho/ji obj} mít. "
     *
     *   Ale v nekterych situacich se nehodi "jím", ale melo by byt "ním":
     *
     *   >﻿mluv
     *   (s jím) 
     *   >﻿t
     *   Nemáš na mysli nic konkrétního, o čem by ses chtěl bavit s jím.
     *
     *   Otázka je, zda se někde naopak hodí to jím.
     *
     *   (zajmenoTe4N) -> (zajmenoTebe4N) kvuli:
     *   "Na lavici není pro tě -> tebe dost místa."
     */
    name = (zajmenoTy)
    nameKohoCeho = (zajmenoTebe2N)
    nameKomuCemu = (zajmenoTiN)
    nameKohoCo = (zajmenoTebe4N)
    nameKomCem = (zajmenoTobe6)
    nameKymCim = (zajmenoTebouN)

    namePostava = (referralPerson == ThirdPerson ? name : '')

    /*
     *   Pronoun selector.  This returns an index for selecting pronouns
     *   or other words based on number and gender, taking into account
     *   person, number, and gender.  The value returned is the sum of the
     *   following components:
     *
     *   number/gender:
     *.  - singular neuter = 1
     *.  - singular masculine = 2
     *.  - singular feminine = 3
     *.  - plural = 4
     *
     *   person:
     *.  - first person = 0
     *.  - second person = 4
     *.  - third person = 8
     *
     *   The result can be used as a list selector as follows (1=first
     *   person, etc; s=singular, p=plural; n=neuter, m=masculine,
     *   f=feminine):
     *
     *   [1/s/n, 1/s/m, 1/s/f, 1/p, 2/s/n, 2/s/m, 2/s/f, 2/p,
     *.  3/s/n, 3/s/m, 3/s/f, 3/p]
     *
     *   Přeházíme na logičtější mužský = 1, ženský = 2, střední = 3, plurál = 4
     */
    pronounSelector
    {
        return ((referralPerson - FirstPerson)*4
                + (isPlural ? 4 : gender == 3 ? 2 : gender == 4 ? 3 : 1));
    }

    /*
     *   Osobní zájmena:
     *
     *   1. pád - zajmenoTy
     *   2. pád - zajmenoTebe2J/N, zajmenoTe2J/N
     *   3. pád - zajmenoTobe3J/N, zajmenoTiJ/N
     *   4. pád - zajmenoTebe4J/N, zajmenoTe4J/N
     *   6. pád - zajmenoTobe6
     *   7. pád - zajmenoTebouJ/N
     */
    zajmenoTy
    {
        return isPlural && referralPerson == ThirdPerson && gender == 3 ? 'ony'
            : isPlural && referralPerson == ThirdPerson && gender == 4 ? 'ona'
            : ['já', 'já', 'já', 'my',
               'ty', 'ty', 'ty', 'vy',
               'on', 'ona', 'ono', 'oni'][pronounSelector];
    }
    zajmenoTebe2J
    {
        return ['mne', 'mne', 'mne', 'nás',
                'tebe', 'tebe', 'tebe', 'vás',
                'jeho', 'jí', 'jeho', 'jich'][pronounSelector];
    }
    zajmenoTebe2N
    {
        return ['mne', 'mne', 'mne', 'nás',
                'tebe', 'tebe', 'tebe', 'vás',
                'něho', 'ní', 'něho', 'nich'][pronounSelector];
    }
    zajmenoTe2J
    {
        return ['mě', 'mě', 'mě', 'nás',
                'tě', 'tě', 'tě', 'vás',
                'ho', 'jí', 'ho', 'jich'][pronounSelector];
    }
    zajmenoTe2N
    {
        return ['mě', 'mě', 'mě', 'nás',
                'tě', 'tě', 'tě', 'vás',
                'ho', 'ní', 'ho', 'nich'][pronounSelector];
    }
    zajmenoTobe3J
    {
        return ['mně', 'mně', 'mně', 'nám',
                'tobě', 'tobě', 'tobě', 'vám',
                'jemu', 'jí', 'jemu', 'jim'][pronounSelector];
    }
    zajmenoTobe3N
    {
        return ['mně', 'mně', 'mně', 'nám',
                'tobě', 'tobě', 'tobě', 'vám',
                'němu', 'ní', 'němu', 'nim'][pronounSelector];
    }
    zajmenoTiJ
    {
        return ['mi', 'mi', 'mi', 'nám',
                'ti', 'ti', 'ti', 'vám',
                'mu', 'jí', 'mu', 'jim'][pronounSelector];
    }
    zajmenoTiN
    {
        return ['mi', 'mi', 'mi', 'nám',
                'ti', 'ti', 'ti', 'vám',
                'mu', 'ní', 'mu', 'nim'][pronounSelector];
    }
    zajmenoTebe4J
    {
        return ['mne', 'mne', 'mne', 'nás',
                'tebe', 'tebe', 'tebe', 'vás',
                'jeho', 'ji', 'jej', 'je'][pronounSelector];
    }
    zajmenoTebe4N
    {
        return ['mne', 'mne', 'mne', 'nás',
                'tebe', 'tebe', 'tebe', 'vás',
                'něho', 'ni', 'něj', 'ně'][pronounSelector];
    }
    zajmenoJej4J
    {
	return ['mne', 'mne', 'mne', 'nás',
		'tebe', 'tebe', 'tebe', 'vás',
		'jej', 'ji', 'jej', 'je'][pronounSelector]; 
    }
    zajmenoJej4N
    {
        return ['mne', 'mne', 'mne', 'nás',
    		'tebe', 'tebe', 'tebe', 'vás',
    		'něj', 'ni', 'něj', 'ně'][pronounSelector];
    }
    zajmenoTe4J
    {
        return ['mě', 'mě', 'mě', 'nás',
                'tě', 'tě', 'tě', 'vás',
                'ho', 'ji', 'ho', 'je'][pronounSelector];
    }
    zajmenoTe4N
    {
        return ['mě', 'mě', 'mě', 'nás',
                'tě', 'tě', 'tě', 'vás',
                'ho', 'ni', 'ho', 'ně'][pronounSelector];
    }
    zajmenoTobe6
    {
        return ['mně', 'mně', 'mně', 'nás',
                'tobě', 'tobě', 'tobě', 'vás',
                'něm', 'ní', 'něm', 'nich'][pronounSelector];
    }
    zajmenoTebouJ
    {
        return ['mnou', 'mnou', 'mnou', 'námi',
                'tebou', 'tebou', 'tebou', 'vámi',
                'jím', 'jí', 'jím', 'jimi'][pronounSelector];
    }
    zajmenoTebouN
    {
        return ['mnou', 'mnou', 'mnou', 'námi',
                'tebou', 'tebou', 'tebou', 'vámi',
                'ním', 'ní', 'ním', 'nimi'][pronounSelector];
    }

    /*
     *   Přivlastňovací zájmena
     *
     *   1. pád - zajmenoTvuj (m), zajmenoTvoje (ž, s), TODO: tvá (ž), tvé (s)
     *   4. pád - zajmenoTvoji (ž)
     *   6. pád - zajmenoTvém (m)
     */
    zajmenoTvuj
    {
        return ['můj', 'můj', 'můj', 'náš',
                'tvůj', 'tvůj', 'tvůj', 'váš',
                'jeho', 'její', 'jeho', 'jejich'][pronounSelector];
    }
    zajmenoTvoje
    {
        return ['moje', 'moje', 'moje', 'naše',
                'tvoje', 'tvoje', 'tvoje', 'vaše',
                'jeho', 'její', 'jeho', 'jejich'][pronounSelector];
    }

    zajmenoTvoji
    {
        return ['moji', 'moji', 'moji', 'naši',
                'tvoji', 'tvoji', 'tvoji', 'vaši',
                'jeho', 'její', 'jeho', 'jejich'][pronounSelector];
    }

    zajmenoTvem
    {
        return ['mém', 'mém', 'mém', 'naší',
                'tvém', 'tvém', 'tvém', 'vašem',
                'jejím', 'její', 'jejím', 'jejích'][pronounSelector];
    }
    zajmenoTve
    {
        return ['mém', 'mém', 'mém', 'naší',
                'tvé', 'tvé', 'tvé', 'vašem',
                'jejím', 'její', 'jejím', 'jejích'][pronounSelector];
    }

    /*
     *   Pravidelná slovesa
     *
     *   Mají pět slovesných tříd, podle kterých je ovlivněna koncovka slovesa
     *   a tak můžeme poskytnout substituční parametr. Přetížíme tyto funkce
     *   ze třídy Thing, kde jsou všechny jen ve třetí os. pro neživé objekty.
     */
    verbEndingEs
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'eme' : 'u') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'ete' : 'eš') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'ou' : 'e') : '');
    }
    verbEndingNes
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'neme' : 'nu') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'nete' : 'neš') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'nou' : 'ne') : '');
    }
    verbEndingJes
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'jeme' : 'ji') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'jete' : 'ješ') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'jí' : 'je') : '');
    }
    verbEndingIs
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'íme' : 'ím') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'íete' : 'íš') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'í' : 'í') : '');
    }
    verbEndingAs
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'áme' : 'ám') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'áte' : 'áš') : '')
            + (referralPerson == ThirdPerson ? (isPlural ? 'ají' : 'á') : '');
    }

    /*
     *   Spojení slovesa být a zvratného zájmena
     */
    spojeniSis
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'jsme si' : 'jsem si') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'jste si' : 'sis') : '')
            + (referralPerson == ThirdPerson ? 'si' : '');
    }
    spojeniSes
    {
        return (referralPerson == FirstPerson ? (isPlural ? 'jsme se' : 'jsem se') : '')
            + (referralPerson == SecondPerson ? (isPlural ? 'jste se' : 'ses') : '')
            + (referralPerson == ThirdPerson ? 'se' : '');
    }

    /* being verb agreeing with this object as subject */
    slovesoJe
    {
        return tSel(['jsem', 'jsi', 'je', 'jsme', 'jste', 'jsou'][referralPerson
            + (isPlural ? 3 : 0)], ['byl', 'byl', 'byla', 'bylo', 'byli',
            'byly', 'byly', 'byla'][gender + (isPlural ? 4 : 0)]);
    }

    /*
     *   Show my name for an arrival/departure message.  If we've been seen
     *   before by the player character, we'll show our definite name,
     *   otherwise our indefinite name.
     *
     *   V češtině nepotřebujeme rozlišovat člen, takže zobrazíme prosté name.
     *   Úplně zlikvidovat nejde, protože některé třídy, např. vozidlo, vypisují
     *   své cestující.
     */
    travelerName(arriving)
        { say(name); }

    /*
     *   Test to see if we can match the third-person pronouns.  We'll
     *   match these if our inherited test says we match them AND we can
     *   be referred to in the third person.
     */
    canMatchHim = (inherited && canMatch3rdPerson)
    canMatchHer = (inherited && canMatch3rdPerson)
    canMatchIt = (inherited && canMatch3rdPerson)
    canMatchThem = (inherited && canMatch3rdPerson)

    /*
     *   Test to see if we can match a third-person pronoun ('it', 'him',
     *   'her', 'them').  We can unless we're the player character and the
     *   player character is referred to in the first or second person.
     */
    canMatch3rdPerson = (!isPlayerChar || referralPerson == ThirdPerson)

    /*
     *   Set a pronoun antecedent to the given list of ResolveInfo objects.
     *   Pronoun handling is language-specific, so this implementation is
     *   part of the English library, not the generic library.
     *
     *   If only one object is present, we'll set the object to be the
     *   antecedent of 'it', 'him', or 'her', according to the object's
     *   gender.  We'll also set the object as the single antecedent for
     *   'them'.
     *
     *   If we have multiple objects present, we'll set the list to be the
     *   antecedent of 'them', and we'll forget about any antecedent for
     *   'it'.
     *
     *   Note that the input is a list of ResolveInfo objects, so we must
     *   pull out the underlying game objects when setting the antecedents.
     *
     *   TODO: Vyzkoušet, jak je to s reagováním na "ho" apod. v případě,
     *   že tím může být myšlena postava i neživý objekt. Myslím, že když
     *   jsem zkoušel "zeptej se ho na něco", tak že to směrovalo na neživý
     *   objekt.
     */
    setPronoun(lst)
    {
        /* if the list is empty, ignore it */
        if (lst == [])
            return;

        /*
         *   if we have multiple objects, the entire list is the antecedent
         *   for 'them'; otherwise, it's a singular antecedent which
         *   depends on its gender
         */
        if (lst.length() > 1)
        {
            local objs = lst.mapAll({x: x.obj_});

            /* it's 'them' */
            setThem(objs);

            /* forget any 'it' */
            setIt(nil);
        }
        else if (lst.length() == 1)
        {
            /*
             *   We have only one object, so set it as an antecedent
             *   according to its gender.
             */
            setPronounObj(lst[1].obj_);
        }
    }

    /*
     *   Set a pronoun to refer to multiple potential antecedents.  This is
     *   used when the verb has multiple noun slots - UNLOCK DOOR WITH KEY.
     *   For verbs like this, we have no way of knowing in advance whether
     *   a future pronoun will refer back to the direct object or the
     *   indirect object (etc) - we could just assume that 'it' will refer
     *   to the direct object, but this won't always be what the player
     *   intended.  In natural English, pronoun antecedents must often be
     *   inferred from context at the time of use - so we use the same
     *   approach.
     *
     *   Pass an argument list consisting of ResolveInfo lists - that is,
     *   pass one argument per noun slot in the verb, and make each
     *   argument a list of ResolveInfo objects.  In other words, you call
     *   this just as you would setPronoun(), except you can pass more than
     *   one list argument.
     *
     *   We'll store the multiple objects as antecedents.  When we need to
     *   resolve a future singular pronoun, we'll figure out which of the
     *   multiple antecedents is the most logical choice in the context of
     *   the pronoun's usage.
     */
    setPronounMulti([args])
    {
        local lst, subLst;
        local gotThem;

        /*
         *   If there's a plural list, it's 'them'.  Arbitrarily use only
         *   the first plural list if there's more than one.
         */
        if ((lst = args.valWhich({x: x.length() > 1})) != nil)
        {
            /* set 'them' to the plural list */
            setPronoun(lst);

            /* note that we had a clear 'them' */
            gotThem = true;
        }

        /* from now on, consider only the sublists with exactly one item */
        args = args.subset({x: x.length() == 1});

        /* get a list of the singular items from the lists */
        lst = args.mapAll({x: x[1].obj_});

        /*
         *   Set 'it' to all of the items that can match 'it'; do likewise
         *   with 'him' and 'her'.  If there are no objects that can match
         *   a given pronoun, leave that pronoun unchanged.  
         */
        if ((subLst = lst.subset({x: x.canMatchIt})).length() > 0)
            setIt(subLst);
        if ((subLst = lst.subset({x: x.canMatchHim})).length() > 0)
            setHim(subLst);
        if ((subLst = lst.subset({x: x.canMatchHer})).length() > 0)
            setHer(subLst);

        /*
         *   set 'them' to the potential 'them' matches, if we didn't
         *   already find a clear plural list
         */
        if (!gotThem
            && (subLst = lst.subset({x: x.canMatchThem})).length() > 0)
            setThem(subLst);
    }

    /*
     *   Set a pronoun antecedent to the given ResolveInfo list, for the
     *   specified type of pronoun.  We don't have to worry about setting
     *   other types of pronouns to this antecedent - we specifically want
     *   to set the given pronoun type.  This is language-dependent
     *   because we still have to figure out the number (i.e. singular or
     *   plural) of the pronoun type.
     */
    setPronounByType(typ, lst)
    {
        /* check for singular or plural pronouns */
        if (typ == PronounThem)
        {
            /* it's plural - set a list antecedent */
            setPronounAntecedent(typ, lst.mapAll({x: x.obj_}));
        }
        else
        {
            /* it's singular - set an individual antecedent */
            setPronounAntecedent(typ, lst[1].obj_);
        }
    }

    setIt(obj)
    {
        setPronounAntecedent(PronounIt, obj);
        setPronounAntecedent(PronounHim, obj);
    }

    /* set the antecedent for the masculine singular ("him") */
    setHim(obj)
    {
        setPronounAntecedent(PronounHim, obj);
        setPronounAntecedent(PronounIt, obj);
    }

    /*
     *   Set a pronoun antecedent to the given simulation object (usually
     *   an object descended from Thing).
     */
    setPronounObj(obj)
    {
        /*
         *   Actually use the object's "identity object" as the antecedent
         *   rather than the object itself.  In some cases, we use multiple
         *   program objects to represent what appears to be a single
         *   object in the game; in these cases, the internal program
         *   objects all point to the "real" object as their identity
         *   object.  Whenever we're manipulating one of these internal
         *   program objects, we want to make sure that its the
         *   player-visible object - the identity object - that appears as
         *   the antecedent for subsequent references.
         */
        obj = obj.getIdentityObject();

        /*
         *   Set the appropriate pronoun antecedent, depending on the
         *   object's gender.
         *
         *   Note that we'll set an object to be the antecedent for both
         *   'him' and 'her' if the object has both masculine and feminine
         *   usage.
         */

        /* check for masculine usage */
        if (obj.canMatchHim)
            setHim(obj);

        /* check for feminine usage */
        if (obj.canMatchHer)
            setHer(obj);

        /* check for neuter usage */
        if (obj.canMatchIt)
            setIt(obj);

        /* check for third-person plural usage */
        if (obj.canMatchThem)
            setThem([obj]);
    }

    /* set a possessive anaphor */
    setPossAnaphorObj(obj)
    {
        /* check for each type of usage */
        if (obj.canMatchHim)
            possAnaphorTable[PronounHim] = obj;
        if (obj.canMatchHer)
            possAnaphorTable[PronounHer] = obj;
        if (obj.canMatchIt)
            possAnaphorTable[PronounIt] = obj;
        if (obj.canMatchThem)
            possAnaphorTable[PronounThem] = [obj];
    }
    iobjFor(GiveTo)
    {
        verify()
        {
            /* it makes no sense to give something to myself */
            //verifyNotSelf(&cannotGiveToSelfMsg);

            /* it also makes no sense to give something to itself */
            if (gDobj == gIobj)
                illogicalSelf(&cannotGiveToItselfMsg);
        }
        action()
        {
            if(self == gActor)
                replaceAction(Take, DirectObject);
            else
                inherited();
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Give the postures some additional attributes
 */

modify Posture

    /*
     *   Intransitive and transitive forms of the verb, for use in library
     *   messages.  Each of these methods simply calls one of the two
     *   corresponding fixed-tense properties, depending on the current
     *   tense.
     */
    msgVerbI = (tSel(msgVerbIPresent, msgVerbIPast))
    msgVerbT = (tSel(msgVerbTPresent, msgVerbTPast))

    /*
     *   Fixed-tense versions of the above properties, to be defined
     *   individually by each instance of the Posture class.
     */

    /* our present-tense intransitive form ("he stands up") */
    // msgVerbIPresent = 'stand{s} up'

    /* our past-tense intransitive form ("he stood up") */
    // msgVerbIPast = 'stood up'

    /* our present-tense transitive form ("he stands on the chair") */
    // msgVerbTPresent = 'stand{s}'

    /* our past-tense transitive form ("he stood on the chair") */
    // msgVerbTPast = 'stood'

    /* our participle form */
    // participle = 'standing'
;

/*
 *   Intransitivní použití je bez objektu: "Vstal".
 *   Transitivní použití je takové, které má objekt "Stoupnul na židli".
 */
modify standing
    msgVerbIPresent = 'vstává'
    msgVerbIPast = 'vstal{a}'
    msgVerbTPresent = 'stoupá'
    msgVerbTPast = 'stoupl{a}'
    participle = '{stoj[íš poser]|stál[a poser]}'
;

modify sitting
    msgVerbIPresent = 'sedá'
    msgVerbIPast = 'sedl{a}'
    msgVerbTPresent = 'sedá'
    msgVerbTPast = 'sedl{a}'
    participle = '{sed[íš poser]|seděl[a poser]}'
;

modify lying
    msgVerbIPresent = 'ulehá'
    msgVerbIPast = 'ulehl{a}'
    msgVerbTPresent = 'ulehá'
    msgVerbTPast = 'ulehl{a}'
    participle = '{lež[íš poser]|ležel[a poser]}'
;

/* ------------------------------------------------------------------------ */
/*
 *   For our various topic suggestion types, we can infer the full name
 *   from the short name fairly easily.
 */
modify SuggestedTopic
    nameKohoCo = name
    nameKomCem = name
;

modify SuggestedAskTopic
    fullName = ('se {tě/jí targetActor} zeptat na ' + nameKohoCo)
;

modify SuggestedTellTopic
    fullName = ('{ti/jí targetActor} říci o ' + nameKomCem)
;

modify SuggestedAskForTopic
    fullName = ('{tě/jí targetActor} požádat o ' + nameKohoCo)
;

modify SuggestedGiveTopic
    fullName = ('{ti/jí targetActor} dát ' + nameKohoCo)
;

modify SuggestedShowTopic
    fullName = ('{ti/jí targetActor} ukázat ' + nameKohoCo)
;

modify SuggestedYesTopic
    name = 'ano'
    fullName = 'říci ano'
;

modify SuggestedNoTopic
    name = 'ne'
    fullName = 'říci ne'
;

/* ------------------------------------------------------------------------ */
/*
 *   Provide custom processing of the player input for matching
 *   SpecialTopic patterns.  When we're trying to match a player's command
 *   to a set of active special topics, we'll run the input through this
 *   processing to produce the string that we actually match against the
 *   special topics.
 *
 *   First, we'll remove any punctuation marks.  This ensures that we'll
 *   still match a special topic, for example, if the player puts a period
 *   or a question mark at the end of the command.
 *
 *   Second, if the user's input starts with "A" or "T" (the super-short
 *   forms of the ASK ABOUT and TELL ABOUT commands), remove the "A" or "T"
 *   and keep the rest of the input.  Some users might think that special
 *   topic suggestions are meant as ask/tell topics, so they might
 *   instinctively try these as A/T commands.
 *
 *   Users *probably* won't be tempted to do the same thing with the full
 *   forms of the commands (e.g., ASK BOB ABOUT APOLOGIZE, TELL BOB ABOUT
 *   EXPLAIN).  It's more a matter of habit of using A or T for interaction
 *   that would tempt a user to phrase a special topic this way; once
 *   you're typing out the full form of the command, it generally won't be
 *   grammatical, as special topics generally contain the sense of a verb
 *   in their phrasing.
 */
modify specialTopicPreParser
    processInputStr(str)
    {
        /*
         *   remove most punctuation from the string - we generally want to
         *   ignore these, as we mostly just want to match keywords
         */
        str = rexReplace(punctPat, str, '', ReplaceAll);

        /* if it starts with "A" or "T", strip off the leading verb */
        if (rexMatch(aOrTPat, str) != nil)
            str = rexGroup(1)[3];

        /* return the processed result */
        return str;
    }

    /* pattern for string starting with "A" or "T" verbs */
    aOrTPat = static new RexPattern(
        '<nocase><space>*(zep|[řr]ek)<space>+(<^space>.*)$')

    /* pattern to eliminate punctuation marks from the string */
    punctPat = static new RexPattern('[.?!,;:]');
;

/*
 *   For SpecialTopic matches, treat some strings as "weak": if the user's
 *   input consists of just one of these weak strings and nothing else,
 *   don't match the topic.
 */
modify SpecialTopic
    matchPreParse(str, procStr)
    {
        /* if it's one of our 'weak' strings, don't match */
        if (rexMatch(weakPat, str) != nil)
            return nil;

        /* it's not a weak string, so match as usual */
        return inherited(str, procStr);
    }

    /*
     *   Our "weak" strings - 'i', 'l', 'look': these are weak because a
     *   user typing one of these strings by itself is probably actually
     *   trying to enter the command of the same name, rather than entering
     *   a special topic.  These come up in cases where the special topic
     *   is something like "say I don't know" or "tell him you'll look into
     *   it".
     */
    weakPat = static new RexPattern('<nocase><space>*(i|l|look)<space>*$')
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific Traveler changes
 */
modify Traveler
    /*
     *   Get my location's name, from the PC's perspective, for describing
     *   my arrival to or departure from my current location.  We'll
     *   simply return our location's destName, or "the area" if it
     *   doesn't have one.
     */
    travelerLocName()
    {
        /* get our location's name from the PC's perspective */
        local nm = location.getDestName(gPlayerChar, gPlayerChar.location);

        /* if there's a name, return it; otherwise, use "the area" */
        return (nm != nil ? nm : 'do prostoru');
    }
    travelerLocFromName()
    {
        /* get our location's name from the PC's perspective */
        local nm = location.getSrcName(gPlayerChar, gPlayerChar.location);

        /* if there's a name, return it; otherwise, use "the area" */
        return (nm != nil ? nm : 'z prostoru');
    }

    /*
     *   Get my "remote" location name, from the PC's perspective.  This
     *   returns my location name, but only if my location is remote from
     *   the PC's perspective - that is, my location has to be outside of
     *   the PC's top-level room.  If we're within the PC's top-level
     *   room, we'll simply return an empty string.
     *
     *   Používá se ve větě: "Sally odešla *z náměstí* po Široké ulici."
     *   ale i při příchodu "Sally přišla *na náměstí* po Široké ulici."
     */
    travelerRemoteLocName()
    {
        /*
         *   if my location is outside of the PC's outermost room, we're
         *   remote, so return my location name; otherwise, we're local,
         *   so we don't need a remote name at all
         */
        if (isIn(gPlayerChar.getOutermostRoom()))
            return '';
        else
            return travelerLocName;
    }
    travelerRemoteLocFromName()
    {
        /*
         *   if my location is outside of the PC's outermost room, we're
         *   remote, so return my location name; otherwise, we're local,
         *   so we don't need a remote name at all
         */
        if (isIn(gPlayerChar.getOutermostRoom()))
            return '';
        else
            return travelerLocFromName;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific Vehicle changes
 */
modify Vehicle
    /*
     *   Display the name of the traveler, for use in an arrival or
     *   departure message.
     */
    travelerName(arriving)
    {
        /*
         *   By default, start with the indefinite name if we're arriving,
         *   or the definite name if we're leaving.
         *
         *   If we're leaving, presumably they've seen us before, since we
         *   were already in the room to start with.  Since we've been
         *   seen before, the definite is appropriate.
         *
         *   If we're arriving, even if we're not being seen for the first
         *   time, we haven't been seen yet in this place around this
         *   time, so the indefinite is appropriate.
         */
        say(name);

        /* show the list of actors aboard */
        aboardVehicleListerObj.showList(
            libGlobal.playerChar, nil, allContents(), 0, 0,
            libGlobal.playerChar.visibleInfoTable(), nil);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific PushTraveler changes
 */
modify PushTraveler
    /*
     *   When an actor is pushing an object from one room to another, show
     *   its name with an additional clause indicating the object being
     *   moved along with us.
     */
    travelerName(arriving)
    {
        "<<name>>,
        {tlač[íš]|tlačil[a]} <<obj_.nameKohoCo>>,";
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   English-specific travel connector changes
 */

modify PathPassage
    /* treat "take path" the same as "enter path" or "go through path" */
    dobjFor(Take) maybeRemapTo(
        gAction.getEnteredVerbPhrase() == 'take (dobj)', TravelVia, self)

    dobjFor(Enter)
    {
        verify() { logicalRank(50, 'enter path'); }
    }
    dobjFor(GoThrough)
    {
        verify() { logicalRank(50, 'enter path'); }
    }
;

modify AskConnector
    /*
     *   This is the noun phrase we'll use when asking disambiguation
     *   questions for this travel connector: "Which *one* do you want to
     *   enter..."
     */
    travelObjsPhrase = 'one'
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific changes for various nested room types.
 *
 *   ">ven (z dřevěné prkno)"
 */
modify BasicChair
    /* by default, one sits *on* a chair */
    objInPrep = 'na'
    objOutOfPrep = 'z'
    objIntoCase = 4
;

Armchair: Chair
    objInPrep = 'v'
    objIntoCase = 2
;

modify BasicPlatform
    /* by default, one stands *on* a platform */
    objInPrep = 'na'
    objOutOfPrep = 'z'
    objIntoCase = 4
;

modify Platform
    /* by default, one stands *on* a platform */
    objInPrep = 'na'
    objOutOfPrep = 'z'
    objIntoCase = 4
;

modify Booth
    /* by default, one is *in* a booth */
    /* TODO: vokalizace */
    objInPrep = 'v'
    objOutOfPrep = 'z'
;

/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for Matchstick
 */
modify Matchstick
    /* "strike match" means "light match" */
    dobjFor(Strike) asDobjFor(Burn)

    /* "light match" means "burn match" */
    dobjFor(Light) asDobjFor(Burn)
;

/* ------------------------------------------------------------------------ */
/*
 *   Jazyková modifikace pro Keyring
 */
modify Keyring
    /* Kroužek na klíče je pravděpodobně mužský životný, tak změníme výchozí */
    gender = 1
;

/*
 *   Match state objects.  We show "lit" as the state for a lit match,
 *   nothing for an unlit match.
 */
matchStateLit: ThingState 'lit'
    stateTokens = ['lit']
;
matchStateUnlit: ThingState
    stateTokens = ['unlit']
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific modifications for Room.
 */
modify Room
    /*
     *   The ordinary 'name' property is used the same way it's used for
     *   any other object, to refer to the room when it shows up in
     *   library messages and the like: "You can't take the hallway."
     *
     *   By default, we derive the name from the roomName by converting
     *   the roomName to lower case.  Virtually every room will need a
     *   custom room name setting, since the room name is used mostly as a
     *   title for the room, and good titles are hard to generate
     *   mechanically.  Many times, converting the roomName to lower case
     *   will produce a decent name to use in messages: "Ice Cave" gives
     *   us "You can't eat the ice cave."  However, games will want to
     *   customize the ordinary name separately in many cases, because the
     *   elliptical, title-like format of the room name doesn't always
     *   translate well to an object name: "West of Statue" gives us the
     *   unworkable "You can't eat the west of statue"; better to make the
     *   ordinary name something like "plaza".  Note also that some rooms
     *   have proper names that want to preserve their capitalization in
     *   the ordinary name: "You can't eat the Hall of the Ancient Kings."
     *   These cases need to be customized as well.
     */
    name = (roomName.toLower())

    /*
     *   The "destination name" of the room.  This is primarily intended
     *   for use in showing exit listings, to describe the destination of
     *   a travel connector leading away from our current location, if the
     *   destination is known to the player character.  We also use this
     *   as the default source of the name in similar contexts, such as
     *   when we can see this room from another room connected by a sense
     *   connector.
     *
     *   The destination name usually mirrors the room name, but we use
     *   the name in prepositional phrases involving the room ("east, to
     *   the alley"), so this name should include a leading article
     *   (usually definite - "the") unless the name is proper ("east, to
     *   Dinsley Plaza").  So, by default, we simply use the "theName" of
     *   the room.  In many cases, it's better to specify a custom
     *   destName, because this name is used when the PC is outside of the
     *   room, and thus can benefit from a more detailed description than
     *   we'd normally use for the basic name.  For example, the ordinary
     *   name might simply be something like "hallway", but since we want
     *   to be clear about exactly which hallway we're talking about when
     *   we're elsewhere, we might want to use a destName like "the
     *   basement hallway" or "the hallway outside the operating room".
     *
     *   Název místnosti má odpovídat na otázku "kam?". Výchozí hodnota je jen
     *   nuozovka, nebude většinou znít dobře a tak musí být u každé místnosti
     *   zadáno.
     */
    destName = (name)
    srcName = (name)

    /*
     *   Get my "source name," as seen by the given actor from the
     *   given origin location.  This gives the name we can use to
     *   describe this location from the perspective of an actor in an
     *   adjoining location looking at a travel connector from that
     *   location to here.
     *
     *   By default, we simply return our srcName property.  This default
     *   behavior can be overridden if it's necessary for a location to
     *   have different source names in different adjoining
     *   locations, or when seen by different actors.
     *
     *   If this location's name cannot or should not be described from an
     *   adjoining location, this should simply return nil.
     */
    getSrcName(actor, origin) { return srcName; }
    
    /*
     *   For top-level rooms, describe an object as being in the room by
     *   describing it as being in the room's nominal drop destination,
     *   since that's the nominal location for the objects directly in the
     *   room.  (In most cases, the nominal drop destination for a room is
     *   its floor.)
     *
     *   If the player character isn't in the same outermost room as this
     *   container, use our remote name instead of the nominal drop
     *   destination.  The nominal drop destination is usually something
     *   like the floor or the ground, so it's only suitable when we're in
     *   the same location as what we're describing.
     */
    childInName(childName)
    {
        /* if the PC isn't inside us, we're viewing this remotely */
        if (!gPlayerChar.isIn(self))
            return childInRemoteName(childName, gPlayerChar);
        else
            return getNominalDropDestination().childInName(childName);
    }
    childInNameWithOwner(chiName)
    {
        /* if the PC isn't inside us, we're viewing this remotely */
        if (!gPlayerChar.isIn(self))
            return inherited(chiName);
        else
            return getNominalDropDestination().childInNameWithOwner(chiName);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific modifications for the default room parts.
 */

modify Floor
    childInNameGen(childName, myName) { return childName + ' na ' + myName; }
    objInPrep = 'na'
    objOutOfPrep = 'z'
    objIntoCase = 4
    gender = 3
    noun = 'podlaha' 'podlahy' 'podlaze' 'podlahu' 'zem' 'země' 'zemi' 'zemí'
;

modify defaultFloor
    name = 'podlaha'
    nameKohoCeho = 'podlahy'
    nameKomuCemu = 'podlaze'
    nameKohoCo = 'podlahu'
    nameKomCem = 'podlaze'
    nameKymCim = 'podlahou'
;

modify defaultGround
    name = 'zem'
    nameKohoCeho = 'země'
    nameKomuCemu = 'zemi'
    nameKohoCo = 'zem'
    nameKomCem = 'zemi'
    nameKymCim = 'zemí'

    objOutOfPrep = 'ze'
;

modify DefaultWall
    childInNameGen(childName, myName) { return childName + ' na ' + myName; }
    objInPrep = 'na'
    objOutOfPrep = 'ze'
    objIntoCase = 4

    gender = 3
    noun = 'stěna' 'stěny' 'stěně' 'stěnu' 'stěnou' 'zeď' 'zdi' 'zdí'
    plural = 'stěny' 'stěn' 'stěnám' 'stěnách' 'stěnami' 'zdi' 'zdí' 'zdím'
        'zdech' 'zdmi'

    name = 'stěna'
    nameKohoCeho = 'stěny'
    nameKomuCemu = 'stěně'
    nameKohoCo = 'stěnu'
    nameKomCem = 'stěně'
    nameKymCim = 'stěnou'
;

modify defaultCeiling
    gender = 2
    noun = 'strop' 'stropu' 'stropem'

    name = 'strop'
    nameKohoCeho = 'stropu'
    nameKomuCemu = 'stropu'
    nameKohoCo = 'strop'
    nameKomCem = 'stropě'
    nameKymCim = 'stropem'
;

modify defaultNorthWall
    adjective = 's' 'severní'

    name = 'severní stěna'
    nameKohoCeho = 'severní stěny'
    nameKomuCemu = 'severní stěně'
    nameKohoCo = 'severní stěnu'
    nameKomCem = 'severní stěně'
    nameKymCim = 'severní stěnou'
;

modify defaultSouthWall
    adjective = 'j' 'jižní'

    name = 'jižní stěna'
    nameKohoCeho = 'jižní stěny'
    nameKomuCemu = 'jižní stěně'
    nameKohoCo = 'jižní stěnu'
    nameKomCem = 'jižní stěně'
    nameKymCim = 'jižní stěnou'
;

modify defaultEastWall
    adjective = 'v' 'východní'

    name = 'východní stěna'
    nameKohoCeho = 'východní stěny'
    nameKomuCemu = 'východní stěně'
    nameKohoCo = 'východní stěnu'
    nameKomCem = 'východní stěně'
    nameKymCim = 'východní stěnou'
;

modify defaultWestWall
    adjective = 'z' 'západní'

    name = 'západní stěna'
    nameKohoCeho = 'západní stěny'
    nameKomuCemu = 'západní stěně'
    nameKohoCo = 'západní stěnu'
    nameKomCem = 'západní stěně'
    nameKymCim = 'západní stěnou'
;

modify defaultSky
    gender = 3
    noun='obloha' 'oblohy' 'obloze' 'oblohu' 'oblohou' 'nebe' 'nebi' 'nebem'
        'nebesa' 'nebes' 'nebesům' 'nebesech' 'nebesy'

    name = 'obloha'
    nameKohoCeho = 'oblohy'
    nameKomuCemu = 'obloze'
    nameKohoCo = 'oblohu'
    nameKomCem = 'obloze'
    nameKymCim = 'oblohou'
;


/* ------------------------------------------------------------------------ */
/*
 *   The English-specific modifications for directions.
 */
modify Direction
    /* describe a traveler arriving from this direction */
    sayArriving(traveler)
    {
        /* show the generic arrival message */
        gLibMessages.sayArriving(traveler);
    }

    /* describe a traveler departing in this direction */
    sayDeparting(traveler)
    {
        /* show the generic departure message */
        gLibMessages.sayDeparting(traveler);
    }
;

/*
 *   The English-specific modifications for compass directions.
 */
modify CompassDirection
    dirFromName = name
    /* describe a traveler arriving from this direction */
    sayArriving(traveler)
    {
        /* show the generic compass direction description */
        gLibMessages.sayArrivingDir(traveler, nameFrom);
    }

    /* describe a traveler departing in this direction */
    sayDeparting(traveler)
    {
        /* show the generic compass direction description */
        gLibMessages.sayDepartingDir(traveler, name);
    }
;

/*
 *   The English-specific definitions for the compass direction objects.
 *   In addition to modifying the direction objects to define the name of
 *   the direction, we add a 'directionName' grammar rule.
 *
 *   Tady pro češtinu musíme přidat nový druhý sloupeček, ve kterém bude
 *   znění směru pro vylistování východů, např. "Zřetelné východy vedou
 *   na východ, na severozápad a dovnitř." Původně se totiž název směru
 *   odvozoval z anglického názvu směru, který je zároveň identifikátorem
 *   směru v programu. Naopak zrušíme backToPrefix, který se neliší.
 */
#define DefineLangDir(root, dirName, dirFromName, dirNames) \
grammar directionName(root): dirNames: DirectionProd \
   dir = root##Direction \
; \
\
modify root##Direction \
   name = dirName \
   nameFrom = dirFromName

DefineLangDir(north, 'na sever', 'ze severu', 'sever' | 'severně' | 's');
DefineLangDir(south, 'na jih', 'z jihu', 'jih' | 'jižně' | 'j');
DefineLangDir(east, 'na východ', 'z východu', 'východ' | 'východně' | 'v');
DefineLangDir(west, 'na západ', 'ze západu', 'západ' | 'západně' | 'z');
DefineLangDir(northeast, 'na severovýchod', 'ze severovýchodu', 'severovýchod'
    | 'severovýchodně' | 'sv');
DefineLangDir(northwest, 'na severozápad', 'ze severozápadu', 'severozápad'
    | 'severozápadně' | 'sz');
DefineLangDir(southeast, 'na jihovýchod', 'z jihovýchodu', 'jihovýchod'
    | 'jihovýchodně' | 'jv');
DefineLangDir(southwest, 'na jihozápad', 'z jihozápadu', 'jihozápad'
    | 'jihozápadně' | 'jz');
DefineLangDir(up, 'nahoru', 'ze shora', 'nahoru' | 'n');
DefineLangDir(down, 'dolů', 'ze zdola', 'dolu' | 'dolů' | 'd');
DefineLangDir(in, 'dovnitř', 'zevnitř', 'dovnitř' | 'dov');
DefineLangDir(out, 'ven', 'zvenku', 'ven');

/*
 *   The English-specific shipboard direction modifications.  Certain of
 *   the ship directions have no natural descriptions for arrival and/or
 *   departure; for example, there's no good way to say "arriving from
 *   fore."  Others don't fit any regular pattern: "he goes aft" rather
 *   than "he departs to aft."  As a result, these are a bit irregular
 *   compared to the compass directions and so are individually defined
 *   below.
 *
 *   Což pro češtinu neplatí, takže s tím můžeme zacházet stejně jako
 *   s CompassDirection.
 */
modify ShipboardDirection
    dirFromName = name
    /* describe a traveler arriving from this direction */
    sayArriving(traveler)
    {
        /* show the generic compass direction description */
        gLibMessages.sayArrivingDir(traveler, nameFrom);
    }
    
    /* describe a traveler departing in this direction */
    sayDeparting(traveler)
    {
        /* show the generic compass direction description */
        gLibMessages.sayDepartingDir(traveler, name);
    }
;

DefineLangDir(port, 'na levobok', 'z levoboku', 'levobok' | 'lb');
DefineLangDir(starboard, 'na pravobok', 'z pravoboku', 'pravobok' | 'pb');
DefineLangDir(aft, 'na záď', 'ze zádi', 'záď' | 'dozadu' | 'zd');
DefineLangDir(fore, 'na příď', 'z přídi', 'příď' | 'dopředu' | 'pd');

/* ------------------------------------------------------------------------ */
/*
 *   Some helper routines for the library messages.
 */
class MessageHelper: object
    /*
     *   Show a list of objects for a disambiguation query.  If
     *   'showIndefCounts' is true, we'll show the number of equivalent
     *   items for each equivalent item; otherwise, we'll just show an
     *   indefinite noun phrase for each equivalent item.
     */
    askDisambigList(matchList, fullMatchList, showIndefCounts, dist)
    {
        /* show each item */
        for (local i = 1, local len = matchList.length() ; i <= len ; ++i)
        {
            local equivCnt;
            local obj;

            /* get the current object */
            obj = matchList[i].obj_;

            /*
             *   if this isn't the first, add a comma; if this is the
             *   last, add an "or" as well
              */
            if (i == len)
                ", nebo ";
            else if (i != 1)
                ", ";

            /*
             *   Check to see if more than one equivalent of this item
             *   appears in the full list.
             */
            for (equivCnt = 0, local j = 1,
                 local fullLen = fullMatchList.length() ; j <= fullLen ; ++j)
            {
                /*
                 *   if this item is equivalent for the purposes of the
                 *   current distinguisher, count it
                 */
                if (!dist.canDistinguish(obj, fullMatchList[j].obj_))
                {
                    /* it's equivalent - count it */
                    ++equivCnt;
                }
            }

            /*
             *   we have multiple equivalents - show either with or without
             *   a count, depending on the flags the caller provided
             *
             *   Bylo zjednodušeno o rozhodování určitého a neurčitého členu.
             */
            if (equivCnt > 1 && showIndefCounts)
            {
                /* a count is desired for each equivalent group */
                say(dist.countName(obj, equivCnt, 4));
            }
            else
            {
                /* there's only one */
                say(dist.name(obj, 4));
            }
        }
    }

    /*
     *   For a TAction result, select the short-form or long-form message,
     *   according to the disambiguation status of the action.  This is for
     *   the ultra-terse default messages, such as "Taken" or "Dropped",
     *   that sometimes need more descriptive variations.
     *   
     *   If there was no disambiguation involved, we'll use the short
     *   version of the message.
     *   
     *   If there was unclear disambiguation involved (meaning that there
     *   was more than one logical object matching a noun phrase, but the
     *   parser was able to decide based on likelihood rankings), we'll
     *   still use the short version, because we assume that the parser
     *   will have generated a parenthetical announcement to point out its
     *   choice.
     *   
     *   If there was clear disambiguation involved (meaning that more than
     *   one in-scope object matched a noun phrase, but there was only one
     *   choice that passed the logicalness tests), AND the announcement
     *   mode (in gameMain.ambigAnnounceMode) is DescribeClear, we'll
     *   choose the long-form message.  
     */
    shortTMsg(short, long)
    {
        /* check the disambiguation flags and the announcement mode */
        if ((gAction.getDobjFlags() & (ClearDisambig | AlwaysAnnounce))
            == ClearDisambig
            && gAction.getDobjCount() == 1
            && gameMain.ambigAnnounceMode == DescribeClear)
        {
            /* clear disambig and DescribeClear mode - use the long message */
            return long;
        }
        else
        {
            /* in other cases, use the short message */
            return short;
        }
    }

    /*
     *   For a TIAction result, select the short-form or long-form message.
     *   This works just like shortTIMsg(), but takes into account both the
     *   direct and indirect objects. 
     */
    shortTIMsg(short, long)
    {
        /* check the disambiguation flags and the announcement mode */
        if (((gAction.getDobjFlags() & (ClearDisambig | AlwaysAnnounce))
             == ClearDisambig
             || (gAction.getIobjFlags() & (ClearDisambig | AlwaysAnnounce))
             == ClearDisambig)
            && gAction.getDobjCount() == 1
            && gAction.getIobjCount() == 1
            && gameMain.ambigAnnounceMode == DescribeClear)
        {
            /* clear disambig and DescribeClear mode - use the long message */
            return long;
        }
        else
        {
            /* in other cases, use the short message */
            return short;
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Custom base resolver
 */
modify Resolver
    /*
     *   Get the default in-scope object list for a given pronoun.  We'll
     *   look for a unique object in scope that matches the desired
     *   pronoun, and return a ResolveInfo list if we find one.  If there
     *   aren't any objects in scope that match the pronoun, or multiple
     *   objects are in scope, there's no default.
     */
    getPronounDefault(typ, np)
    {
        local map = [PronounHim, &canMatchHim,
                     PronounHer, &canMatchHer,
                     PronounIt, &canMatchIt];
        local idx = map.indexOf(typ);
        local filterProp = (idx != nil ? map[idx + 1] : nil);
        local lst;

        /* if we couldn't find a filter for the pronoun, ignore it */
        if (filterProp == nil)
            return [];

        /*
         *   filter the list of all possible defaults to those that match
         *   the given pronoun
         */
        lst = getAllDefaults.subset({x: x.obj_.(filterProp)});

        /*
         *   if the list contains exactly one element, then there's a
         *   unique default; otherwise, there's either nothing here that
         *   matches the pronoun or the pronoun is ambiguous, so there's
         *   no default
         */
        if (lst.length() == 1)
        {
            /*
             *   we have a unique object, so they must be referring to it;
             *   because this is just a guess, though, mark it as vague
             */
            lst[1].flags_ |= UnclearDisambig;

            /* return the list */
            return lst;
        }
        else
        {
            /*
             *   the pronoun doesn't have a unique in-scope referent, so
             *   we can't guess what they mean
             */
            return [];
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Custom interactive resolver.  This is used for responses to
 *   disambiguation questions and prompts for missing noun phrases.
 */
modify InteractiveResolver
    /*
     *   Resolve a pronoun antecedent.  We'll resolve a third-person
     *   singular pronoun to the target actor if the target actor matches
     *   in gender, and the target actor isn't the PC.  This allows
     *   exchanges like this:
     *
     *.  >bob, examine
     *.  What do you want Bob to look at?
     *.
     *.  >his book
     *
     *   In the above exchange, we'll treat "his" as referring to Bob, the
     *   target actor of the action, because we have referred to Bob in
     *   the partial command (the "BOB, EXAMINE") that triggered the
     *   interactive question.
     */
    resolvePronounAntecedent(typ, np, results, poss)
    {
        local lst;

        /* try resolving with the target actor as the antecedent */
        if ((lst = resolvePronounAsTargetActor(typ)) != nil)
            return lst;

        /* use the inherited result */
        return inherited(typ, np, results, poss);
    }

    /*
     *   Get the reflexive third-person pronoun binding (himself, herself,
     *   itself, themselves).  If the target actor isn't the PC, and the
     *   gender of the pronoun matches, we'll consider this as referring
     *   to the target actor.  This allows exchanges of this form:
     *
     *.  >bob, examine
     *.  What do you want Bob to examine?
     *.
     *.  >himself
     */
    getReflexiveBinding(typ)
    {
        local lst;

        /* try resolving with the target actor as the antecedent */
        if ((lst = resolvePronounAsTargetActor(typ)) != nil)
            return lst;

        /* use the inherited result */
        return inherited(typ);
    }

    /*
     *   Try matching the given pronoun type to the target actor.  If it
     *   matches in gender, and the target actor isn't the PC, we'll
     *   return a resolve list consisting of the target actor.  If we
     *   don't have a match, we'll return nil.
     */
    resolvePronounAsTargetActor(typ)
    {
        /*
         *   if the target actor isn't the player character, and the
         *   target actor can match the given pronoun type, resolve the
         *   pronoun as the target actor
         */
        if (actor_.canMatchPronounType(typ) && !actor_.isPlayerChar())
        {
            /* the match is the target actor */
            return [new ResolveInfo(actor_, 0)];
        }

        /* we didn't match it */
        return nil;
    }
;

/*
 *   Custom disambiguation resolver.
 */
modify DisambigResolver
    /*
     *   Perform special resolution on pronouns used in interactive
     *   responses.  If the pronoun is HIM or HER, then look through the
     *   list of possible matches for a matching gendered object, and use
     *   it as the result if we find one.  If we find more than one, then
     *   use the default handling instead, treating the pronoun as
     *   referring back to the simple antecedent previously set.
     */
    resolvePronounAntecedent(typ, np, results, poss)
    {
        /* if it's a non-possessive HIM or HER, use our special handling */
        if (!poss && typ is in (PronounHim, PronounHer))
        {
            local prop;
            local sub;

            /* get the gender indicator property for the pronoun */
            prop = (typ == PronounHim ? &canMatchHim : &canMatchHer);

            /*
             *   Scan through the match list to find the objects that
             *   match the gender of the pronoun.  Note that if the player
             *   character isn't referred to in the third person, we'll
             *   ignore the player character for the purposes of matching
             *   this pronoun - if we're calling the PC 'you', then we
             *   wouldn't expect the player to refer to the PC as 'him' or
             *   'her'.
             */
            sub = matchList.subset({x: x.obj_.(prop)});

            /* if the list has a single entry, then use it as the match */
            if (sub.length() == 1)
                return sub;

            /*
             *   if it has more than one entry, it's still ambiguous, but
             *   we might have narrowed it down, so throw a
             *   still-ambiguous exception and let the interactive
             *   disambiguation ask for further clarification
             */
            results.ambiguousNounPhrase(nil, ResolveAsker, 'one',
                                        sub, matchList, matchList,
                                        1, self);
            return [];
        }

        /*
         *   if we get this far, it means we didn't use our special
         *   handling, so use the inherited behavior
         */
        return inherited(typ, np, results, poss);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Distinguisher customizations for English.
 *   
 *   Each distinguisher must provide a method that gets the name of an item
 *   for a disamgiguation query.  Since these are inherently
 *   language-specific, they're defined here.  
 */

/*
 *   The null distinguisher tells objects apart based strictly on the name
 *   string.  When we list objects, we simply show the basic name - since
 *   we can tell apart our objects based on the base name, there's no need
 *   to resort to other names.  
 */
modify nullDistinguisher
    /* we can tell objects apart if they have different base names */
    canDistinguish(a, b) { return a.name != b.name; }

    name(obj, gc) { return obj.nameCase(gc); }
    countName(obj, cnt, gc) { return obj.countName(cnt, gc); }
;


/*
 *   The basic distinguisher can tell apart objects that are not "basic
 *   equivalents" of one another.  Thus, we need make no distinctions when
 *   listing objects apart from showing their names.  
 */
modify basicDistinguisher
    name(obj, gc) { return obj.disambigNameCase(gc); }
    countName(obj, cnt, gc) { return obj.countDisambigName(cnt, gc); }
;

/*
 *   The ownership distinguisher tells objects apart based on who "owns"
 *   them, so it shows the owner or location name when listing the object. 
 */
modify ownershipDistinguisher
    name(obj, gc) { return obj.nameOwnerLoc(true, gc); }
    countName(obj, cnt, gc) { return obj.countNameOwnerLoc(cnt, true, gc); }

    /* note that we're prompting based on this distinguisher */
    notePrompt(lst)
    {
        /*
         *   notify each object that we're referring to it by
         *   owner/location in a disambiguation prompt
         */
        foreach (local cur in lst)
            cur.obj_.notePromptByOwnerLoc(true);
    }
;

/*
 *   The location distinguisher tells objects apart based on their
 *   containers, so it shows the location name when listing the object. 
 */
modify locationDistinguisher
    name(obj, gc) { return obj.nameOwnerLoc(nil, gc); }
    countName(obj, cnt, gc) { return obj.countNameOwnerLoc(cnt, nil, gc); }

    /* note that we're prompting based on this distinguisher */
    notePrompt(lst)
    {
        /* notify the objects of their use in a disambiguation prompt */
        foreach (local cur in lst)
            cur.obj_.notePromptByOwnerLoc(nil);
    }
;

/*
 *   The lit/unlit distinguisher tells apart objects based on whether
 *   they're lit or unlit, so we list objects as lit or unlit explicitly.  
 */
modify litUnlitDistinguisher
    name(obj, gc) { return obj.nameLit(gc); }
    countName(obj, cnt, gc) { return obj.pluralNameLit(gc); }
;

/* ------------------------------------------------------------------------ */
/*
 *   Enligh-specific light source modifications
 *   TODO: přeložit
 */
modify LightSource
    /* provide lit/unlit names for litUnlitDistinguisher */
    nameLit(gc) { return (isLit ? 'lit ' : 'unlit ') + nameCase(gc); }
    aNameLit()
    {
        /*
         *   if this is a mass noun or a plural name, just use the name
         *   with lit/unlit; otherwise, add "a"
         */
        if (isPlural || isMassNoun)
            return (isLit ? 'lit ' : 'unlit ') + name;
        else
            return (isLit ? 'a lit ' : 'an unlit ') + name;
    }
    pluralNameLit(gc)
        { return (isLit ? 'lit ' : 'unlit ') + pluralNameCase(gc); }

    /*
     *   Allow 'lit' and 'unlit' as adjectives - but even though we define
     *   these as our adjectives in the dictionary, we'll only accept the
     *   one appropriate for our current state, thanks to our state
     *   objects.
     */
    adjective = 'rozsvícený' 'rozsvícená' 'rozsvícené' 'rozsvícení' 'zhasnutý'
	'zhasnutá' 'zhasnuté' 'zhasnutí' 'nerozsvícený' 'nerozsvícená'
	'nerozsvícené' 'nerozsvícení'
;

/*
 *   Light source list states.  An illuminated light source shows its
 *   status as "providing light"; an unlit light source shows no extra
 *   status.
 *
 *   TODO: přebírá rod z postavy, ne předmětu!
 */
lightSourceStateOn: ThingState 'svít{í|il[a]}'
    stateTokens = ['rozsvícený', 'rozsvícená', 'rozsvícené', 'rozsvícení']
;
lightSourceStateOff: ThingState
    stateTokens = ['zhasnutý', 'zhasnutá', 'zhasnuté', 'zhasnutí',
	'nerozsvícený', 'nerozsvícená', 'nerozsvícené', 'nerozsvícení']
;

/* ------------------------------------------------------------------------ */
/*
 *   Wearable states - a wearable item can be either worn or not worn.
 */

/* "worn" */
wornState: ThingState 'máš na sobě'
    /*
     *   In listings of worn items, don't bother mentioning our 'worn'
     *   status, as the entire list consists of items being worn - it
     *   would be redundant to point out that the items in a list of items
     *   being worn are being worn.
     */
    wornName(lst) { return nil; }
;

/*
 *   "Unworn" state.  Don't bother mentioning the status of an unworn item,
 *   since this is the default for everything.  
 */
unwornState: ThingState;

/*
 *   Lister musíme modifikovat kvůli funkci na vypisování seznamů. V této
 *   funkci se volá showListPrefixWide, kde si potřebujeme předat první
 *   objekt v následujícím seznamu, abychom mohli přizpůsobit rod slovesa
 *   být. Např. "Na staré kredenci *byla* malá plechovka a bonbón." vs. 
 *   "Na pultu *byl* mosazný zvonek a malá plechovka."
 *
 *   Další důležitá modifikace se týká zkracování zájmen. Protože v návrhu
 *   témat k rozhovoru se skupina sestaví podle vzoru "se ho zeptat na x",
 *   což správně funguje v případě "...nebo se ho zeptat na x". Jenže na
 *   začátku věty to sestaví "Mohl bys se ho zeptat na oheň." A my právě
 *   potřebujeme postprocesingem predělat "bys se" na "by ses", event.
 *   ještě "bys si!" na "by sis", ale to normalne nevyuziju.
 */
modify Lister
    /*
     *   Show the list.  This is called after we've figured out which items
     *   we intend to display, and after we've arranged the items into
     *   groups.  In rare cases, listers might want to override this, to
     *   customize the way the way the list is displayed based on the
     *   internal arrangement of the list.  
     */
    showArrangedList(pov, parent, lst, options, indent, infoTab, itemCount,
                     singles, groups, groupTab, origLst)
    {
        /*
         *   We now know how many items we're listing (grammatically
         *   speaking), so we're ready to display the list prefix.  If
         *   we're displaying nothing at all, just display the "empty"
         *   message, and we're done.  
         */
        if (itemCount == 0)
        {
            /* show the empty list */
            showListEmpty(pov, parent);
        }
        else
        {
            local i;
            local cnt;
            local sublists;
            local origOptions = options;
            local itemOptions;
            local groupOptions;
            local listCount;
            local dispCount;
            local cur;
            local selector;

            /* 
             *   Check to see if we have one or more group sublists - if
             *   we do, we must use the "long" list format for our overall
             *   list, otherwise we can use the normal "short" list
             *   format.  The long list format uses semicolons to separate
             *   items.  
             */
            for (i = 1, cnt = groups.length(), sublists = nil ;
                 i <= cnt ; ++i)
            {
                /* 
                 *   if this group's lister item displays a sublist, we
                 *   must use the long format 
                 */
                if (groups[i].groupDisplaysSublist)
                {
                    /* note that we are using the long format */
                    sublists = true;
                    
                    /* 
                     *   one is enough to make us use the long format, so
                     *   we need not look any further 
                     */
                    break;
                }
            }
            
            /* generate the prefix message if we're in a 'tall' listing */
            if ((options & ListTall) != 0)
            {
                /* indent the prefix */
                showListIndent(options, indent);
                
                /*
                 *   Určíme selector pro výběr tvaru slovesa v minulém čase.
                 *   Přísudek musí být ve shodě s podměty, tj. ve shodě s
                 *   objekty seznamu. Protože listujeme objekty na výšku, tedy
                 *   v seznamu, dáme přednost určení tvaru slovesa podle všech
                 *   objektů, místo podle prvního. Viz bod 2. na
                 *   http://prirucka.ujc.cas.cz/?id=601
                 *
                 *   jednotné číslo:
                 *      1 = mužský rod - tvar bez koncovky (byl)
                 *      2 = ženský rod - koncovka 'a' (byla)
                 *      3 = střední rod - koncovka 'o' (bylo)
                 *   množné číslo:
                 *      4 = mužský životný rod - koncovka 'i' (byli)
                 *      5 = mužský neživotný a ženský rod - koncovka 'y' (byly)
                 *      6 = střední rod - koncovka 'a' (byla)
                 */
                if(itemCount > 1 || lst[1].isPlural)
                {
                    /* Přednost má mužský živ., potom neživotný a ženský */
                    local m = 0, n = 0;
                    foreach(cur in lst)
                    {
                        if(cur.gender == 1) m++;
                        if(cur.gender == 2 || cur.gender == 3) n++;
                    }
                    selector = m ? 4 : (n ? 5 : 6);
                }
                else
                {
                    /* Pokud je v seznamu jediný objekt, určíme podle něj. */
                    selector = lst[1].gender == 1 || lst[1].gender == 2 ? 1
                        : (lst[1].gender == 3 ? 2 : 3);
                }
                
                /* 
                 *   Show the prefix.  If this is a contents listing, and
                 *   it's not at the top level, show the contents prefix;
                 *   otherwise show the full list prefix.  Note that we can
                 *   have a contents listing at the top level, since some
                 *   lists are broken out for separate listing.  
                 */
                if ((options & ListContents) != 0 && indent != 0)
                    showListContentsPrefixTall(itemCount, pov, parent, selector);
                else
                    showListPrefixTall(itemCount, pov, parent, selector);
                
                /* go to a new line for the list contents */
                "\n";
                
                /* indent the items one level now, since we showed a prefix */
                ++indent;
            }
            else
            {
                /*
                 *   Protože listujeme na sirku, tedy inline, tak nagenerujeme
                 *   selektor pro tvar slovesa v minulém čase ve shodě s prvním
                 *   objektem v seznamu.
                 *
                 *   Jestlize budu misto objektu listovat skupinu, tak je
                 *   potreba ji brat jako mnoznou.
                 *
                 *   Na stole *jsou* dvě figurky.
                 *   Na stole je figurka a figurka.
                 */
                if(singles.countOf(lst[1]) && lst[1].isPlural
                        || !singles.countOf(lst[1]))
                    selector = lst[1].gender == 1 ? 4 : lst[1].gender == 2
                    || lst[1].gender == 3 ? 5 : 6;
                else
                    selector = lst[1].gender == 1 || lst[1].gender == 2 ? 1
                    : lst[1].gender == 3 ? 2 : 3;

                /* show the prefix */
                showListPrefixWide(itemCount, pov, parent, selector);
            }
            
            /* 
             *   regardless of whether we're adding long formatting to the
             *   main list, display the group sublists with whatever
             *   formatting we were originally using 
             */
            groupOptions = options;
            
            /* show each item with our current set of options */
            itemOptions = options;
            
            /* 
             *   if we're using sublists, show "long list" separators in
             *   the main list 
             */
            if (sublists)
                itemOptions |= ListLong;
            
            /* 
             *   calculate the number of items we'll show in the list -
             *   each group shows up as one list entry, so the total
             *   number of list entries is the number of single items plus
             *   the number of groups 
             */
            listCount = singles.length() + groups.length();

            /*
             *   Show the items.  Run through the (filtered) original
             *   list, so that we show everything in the original sorting
             *   order.  
             */
            dispCount = 0;
            foreach (cur in lst)
            {
                local group;
                local displayedCur;
                
                /* presume we'll display this item */
                displayedCur = true;
                
                /*
                 *   Figure out how to show this item: if it's in the
                 *   singles list, show it as a single item; if it's in
                 *   the group list, show its group; if it's in a group
                 *   we've previously shown, show nothing, as we showed
                 *   the item when we showed its group.  
                 */
                if (singles.indexOf(cur) != nil)
                {
                    /*
                     *   It's in the singles list, so show it as a single
                     *   item.
                     *   
                     *   If the item has contents that we'll display in
                     *   'tall' mode, show the item with its contents - we
                     *   don't need to show the item separately, since it
                     *   will provide a 'tall' list prefix showing itself.
                     *   Otherwise, show the item singly.  
                     */
                    if ((options & ListTall) != 0
                        && (options & ListRecurse) != 0
                        && contentsListed(cur)
                        && getListedContents(cur, infoTab) != [])
                    {
                        /* show the item with its contents */
                        showContentsList(pov, cur, origOptions | ListContents,
                                         indent, infoTab);
                    }
                    else
                    {
                        /* show the list indent if necessary */
                        showListIndent(itemOptions, indent);

                        /* show the item */
                        showListItem(cur, itemOptions, pov, infoTab);

                        /* 
                         *   if we're in wide recursive mode, show the
                         *   item's contents as an in-line parenthetical 
                         */
                        if ((options & ListTall) == 0
                            && (options & ListRecurse) != 0
                            && contentsListed(cur)
                            && !contentsListedSeparately(cur))
                        {
                            /* show the item's in-line contents */
                            showInlineContentsList(pov, cur,
                                origOptions | ListContents,
                                indent + 1, infoTab);
                        }
                    }
                }
                else if ((group = groups.valWhich(
                    {g: groupTab[g].indexOf(cur) != nil})) != nil)
                {
                    /* show the list indent if necessary */
                    showListIndent(itemOptions, indent);

                    /* we found the item in a group, so show its group */
                    group.showGroupList(pov, self, groupTab[group],
                                        groupOptions, indent, infoTab);

                    /* 
                     *   Forget this group - we only need to show each
                     *   group once, since the group shows every item it
                     *   contains.  Since we'll encounter the groups other
                     *   members as we continue to scan the main list, we
                     *   want to make sure we don't show the group again
                     *   when we reach the other items.  
                     */
                    groups.removeElement(group);
                }
                else
                {
                    /* 
                     *   We didn't find the item in the singles list or in
                     *   a group - it must be part of a group that we
                     *   already showed previously, so we don't need to
                     *   show it again now.  Simply make a note that we
                     *   didn't display it.  
                     */
                    displayedCur = nil;
                }

                /* if we displayed the item, show a suitable separator */
                if (displayedCur)
                {
                    /* count another list entry displayed */
                    ++dispCount;

                    /* show an appropriate separator */
                    showListSeparator(itemOptions, dispCount, listCount);
                }
            }

            /* 
             *   if we're in 'wide' mode, finish the listing (note that if
             *   this is a 'tall' listing, we're already done, because a
             *   tall listing format doesn't make provisions for anything
             *   after the item list) 
             */
            if ((options & ListTall) == 0)
            {
                /*
                 *   Určíme selector pro výběr tvaru slovesa v minulém čase.
                 *   Přísudek musí být ve shodě s podměty, tj. ve shodě s
                 *   objekty seznamu. Zde generujeme tento tvar pro objekty
                 *   listované na šířku, tedy ve větě a to do suffixu, takže
                 *   použijeme algoritmus pro přísudek za několikanásobným
                 *   pomětem. Viz bod 1. na http://prirucka.ujc.cas.cz/?id=601
                 *
                 *   jednotné číslo:
                 *      1 = mužský rod - tvar bez koncovky (byl)
                 *      2 = ženský rod - koncovka 'a' (byla)
                 *      3 = střední rod - koncovka 'o' (bylo)
                 *   množné číslo:
                 *      4 = muž. živ. rod, stř. j.č. - koncovka 'i' (byli)
                 *      5 = mužský neživotný a ženský rod - koncovka 'y' (byly)
                 *      6 = střední rod, vše množné č. - koncovka 'a' (byla)
                 */
                if(itemCount > 1 || lst[1].isPlural)
                {
                    /* Přednost má mužský živ., potom neživotný a ženský */
                    local m = 0, n = 0, o = 0;
                    foreach(cur in lst)
                    {
                        if(cur.gender == 1) m++;
                        if(cur.gender == 2 || cur.gender == 3) n++;
                        if(cur.gender == 4 && !cur.isPlural) o++;
                    }
                    selector = m ? 4 : (n ? 5 : (o ? 5 : 6));
                }
                else
                {
                    /* Pokud je v seznamu jediný objekt, určíme podle něj. */
                    selector = lst[1].gender == 1 || lst[1].gender == 2 ? 1
                        : (lst[1].gender == 3 ? 2 : 3);
                }

                /* show the wide-mode list suffix */
                showListSuffixWide(itemCount, pov, parent, selector);
            }
        }
    }
    /*
     *   Kvůli změně parametrů předávaných funkcím pro výpis prefixů a postfixů
     *   musíme upravit parametry funkce tady a v několika potomcích třídy
     *   Lister, které jsou i s funkcemi explicitně přetížené v jazykově
     *   nezávislé části knihovny.
     */
    showListPrefixWide(itemCount, pov, parent, selector) { }
    showListPrefixTall(itemCount, pov, parent, selector) { }
    showListSuffixWide(itemCount, pov, parent, selector) { }
;

modify plainLister
    showListPrefixWide(itemCount, pov, parent, selector) { }
    showListPrefixTall(itemCount, pov, parent, selector) { }
    showListSuffixWide(itemCount, pov, parent, selector) { }
;

modify GroupSublister
    showListPrefixWide(itemCount, pov, parent, selector) { }
    showListPrefixTall(itemCount, pov, parent, selector) { }
    showListSuffixWide(itemCount, pov, parent, selector) { }
;

modify ParagraphLister
    showListPrefixWide(itemCount, pov, parent, selector) { "<.p>"; }
;

modify InventorySublister
    showListPrefixWide(itemCount, pov, parent, selector) { }
    showListSuffixWide(itemCount, pov, parent, selector) { }
;

modify WearingSublister
    showListPrefixWide(itemCount, pov, parent, selector) { }
    showListSuffixWide(itemCount, pov, parent, selector) { }
;

modify fullScoreLister
    showListPrefixTall(itemCount, pov, parent, selector)
    {
        /* showt he full score list intro message */
        gLibMessages.showFullScorePrefix;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Typographical effects output filter.  This filter looks for certain
 *   sequences in the text and converts them to typographical equivalents.
 *   Authors could simply write the HTML for the typographical markups in
 *   the first place, but it's easier to write the typewriter-like
 *   sequences and let this filter convert to HTML.
 *
 *   We perform the following conversions:
 *
 *   '---' -> &zwnbsp;&mdash;
 *.  '--' -> &zwnbsp;&ndash;
 *.  sentence-ending punctuation -> same + &ensp;
 *
 *   Since this routine is called so frequently, we hard-code the
 *   replacement strings, rather than using properties, for slightly faster
 *   performance.  Since this routine is so simple, games that want to
 *   customize the replacement style should simply replace this entire
 *   routine with a new routine that applies the customizations.
 *
 *   Note that we define this filter in the English-specific part of the
 *   library, because it seems almost certain that each language will want
 *   to customize it for local conventions.
 *
 *   V češtině není english spacing používaný a celkově se od něj spíše
 *   ustupuje, tak jsem ho vyhodil a zůstaly jen pomlčky.
 */
typographicalOutputFilter: OutputFilter
    filterText(ostr, val)
    {
        /*
         *   Replace dashes with typographical hyphens.  Three hyphens in a
         *   row become an em-dash, and two in a row become an en-dash.
         *   Note that we look for the three-hyphen sequence first, because
         *   if we did it the other way around, we'd incorrectly find the
         *   first two hyphens of each '---' sequence and replace them with
         *   an en-dash, causing us to miss the '---' sequences entirely.
         *   
         *   We put a no-break marker (\uFEFF) just before each hyphen, and
         *   an okay-to-break marker (\u200B) just after, to ensure that we
         *   won't have a line break between the preceding text and the
         *   hyphen, and to indicate that a line break is specifically
         *   allowed if needed to the right of the hyphen.  
         */
        val = val.findReplace(['---', '--'],
                              ['\uFEFF&mdash;\u200B', '\uFEFF&ndash;\u200B']);

        val = val.findReplace('---', '\uFEFF&mdash;\u200B', ReplaceAll);
        val = val.findReplace('--',  '\uFEFF&ndash;\u200B', ReplaceAll);
        
        local quotes_callback = function(matchString, matchIndex, originalString)
        {
    	    if(rexGroup(1) == nil)
	    {
                languageGlobals.quotesLevel++;
                if(languageGlobals.quotesLevel == 1) return '„';
                if(languageGlobals.quotesLevel == 2) return '‚';
                return '»';
            }
	    else
	    {
		languageGlobals.quotesLevel--;
                if(languageGlobals.quotesLevel == 0) return '“';
                if(languageGlobals.quotesLevel == 1) return '‘';
                return '«';
            }
        };
        val = rexReplace('<(/)?[qQ]>', val, quotes_callback, ReplaceAll);

        /*
         *   Šablona {jsi} ve třetí osobě minuého času negeneruje žádné slovo.
         *   (Např. Mohl jsem udělat / Mohl jsi udělat / Mohl udělat) Tato
         *   úprava odstraní vzniklou mezeru před tečkou, pokud se šablona byla
         *   na konci věty.
         */
        val = val.findReplace(' . ',  '. ', ReplaceAll);
        val = val.findReplace(' , ',  ', ', ReplaceAll);

        val = val.findReplace('by jsem',  'bych', ReplaceAll);
        val = val.findReplace('by jsi',  'bys', ReplaceAll);
        val = val.findReplace('by jsme',  'bychom', ReplaceAll);
        val = val.findReplace('by jste',  'byste', ReplaceAll);

        val = rexReplace(bysSePat, val, 'by ses', ReplaceAll);
        val = rexReplace(bysSiPat, val, 'by sis', ReplaceAll);
        val = rexReplace(jsiSePat, val, 'ses', ReplaceAll);
        val = rexReplace(jsiSiPat, val, 'sis', ReplaceAll);

        /* vokalizace předložek */
        val = val.findReplace('s mnou',  'se mnou', ReplaceAll);
        val = val.findReplace('S mnou',  'Se mnou', ReplaceAll);
        val = val.findReplace('k mně',  'ke mně', ReplaceAll);
        val = val.findReplace('K mně',  'Ke mně', ReplaceAll);
        
        /*
         *   Konzervativni vokalizatory
         *
         *     - přidal jsem "s" za "z" kvůli "ze společenské"
         */
        val = rexReplace('%<(k) +([kg]|[sšzž][^aáeéěiíoóuúůyý])', val,
            '%1e %2', ReplaceAll | ReplaceIgnoreCase);
        val = rexReplace('%<(v) +([vwf]|č[lmt]|[sšzž][^aáeéěiíoóuúůyý])', val,
            '%1e %2', ReplaceAll | ReplaceIgnoreCase);
        val = rexReplace('%<(s) +(psem|[sz]|č[lmt]|[šž][^aáeéěiíoóuúůyý])', val,
            '%1e %2', ReplaceAll | ReplaceIgnoreCase);
        val = rexReplace('%<(z) +([zžsš])', val,
            '%1e %2', ReplaceAll | ReplaceIgnoreCase);

        /* Vratit zmenu */
        val = val.findReplace('se sebou',  's sebou', ReplaceAll);
        val = val.findReplace('Se sebou',  'S sebou', ReplaceAll);


//        val = rexReplace('%<(s) ([szžš])', val, '%1e %2', ReplaceAll);
//        val = rexReplace('%<(z) ([zsšž])', val, '%1e %2', ReplaceAll);
//        val = rexReplace('%<([ksvz]) ([bcčdďfghjklmnňpqrřsštťvwxzž][bcčdďfghjkmnňpqsštťvwxzž][aáeéěiíoóuúů])', val, '%1e %2', ReplaceAll);
//        val = rexReplace('%<([ksvz]) (tř|sl|dř|zr|zl)', val, '%1e %2', ReplaceAll);
//        val = rexReplace('%<([ksvz]) ([bcčdďfghjklmnňpqrřsštťvwxzž][bcčdďfghjklmnňpqrřsštťvwxzž][bcčdďfghjklmnňpqrřsštťvwxzž])', val, '%1e %2', ReplaceAll);

        /* return the result */
        return val;
    }
    innerQuotesPat = static new RexPattern('(<langle>q<rangle>[^<langle>'
        + '<rangle>]*)<langle>q<rangle>([^<langle><rangle>]*)<langle>/q<rangle>'
        + '([^<langle><rangle>]*<langle>/q<rangle>)')

    bysSePat = static new RexPattern('%<(bys se)%>')
    bysSiPat = static new RexPattern('%<(bys si)%>')
    jsiSePat = static new RexPattern('%<(jsi se)%>')
    jsiSiPat = static new RexPattern('%<(jsi si)%>')
;

/* ------------------------------------------------------------------------ */
/*
 *   The English-specific message builder.
 */
langMessageBuilder: MessageBuilder

    /*
     *   The English message substitution parameter table.
     *
     *   Note that we specify two additional elements for each table entry
     *   beyond the standard language-independent complement:
     *
     *   info[4] = reflexive property - this is the property to invoke
     *   when the parameter is used reflexively (in other words, its
     *   target object is the same as the most recent target object used
     *   in the nominative case).  If this is nil, the parameter has no
     *   reflexive form.
     *
     *   info[5] = true if this is a nominative usage, nil if not.  We use
     *   this to determine which target objects are used in the nominative
     *   case, so that we can remember those objects for subsequent
     *   reflexive usages.
     *
     *   Pokud v rámci parametru není určen objekt, tak se podle třetího
     *   parametru rozhoduje, zda se použije actor a nebo předchozí objekt.
     *   '{Kdoco dobj} těžk{ý} {je|byl[a]} těžk{ý} pro {kohočeho} těžk{ý dobj}. '
     *   "Hrad těžký (není zadán třetí parametr, zvtahuje se tedy k předchozímu
     *   objektu) byla (parametr {a} se implicitne vaze k actoru) těžká
     *   (v předchozím parametru se zapamatoval actor jako objekt) pro Sally
     *   příliš těžký (explicitně jsem zvolil objekt) na zdvihnutí."
     */
    paramList_ =
    [
        /* podstatná jména */
        ['kdoco', &name, 'actor', nil, true],
        ['kohočeho', &nameKohoCeho, 'actor', &zajmenoSebe, nil],
        ['komučemu', &nameKomuCemu, 'actor', &zajmenoSobe, nil],
        ['kohoco', &nameKohoCo, 'actor', &zajmenoSebe, nil],
        ['komčem', &nameKomCem, 'actor', &zajmenoSobe, nil],
        ['kýmčím', &nameKymCim, 'actor', &zajmenoSebou, nil],

        ['kdoco/postava', &namePostava, 'actor', nil, true],

        /* přídavná jména */
        ['ý', &pridavneJmenoMlady, nil, nil, nil],
        ['ému', &pridavneJmenoMlademu, nil, nil, nil],
        ['ém', &pridavneJmenoMladem, nil, nil, nil],
        ['ým', &pridavneJmenoMladym, nil, nil, nil],

        /* osobní zájmena */
        ['ten', &zajmenoTen, nil, nil, true],

        ['ty', &zajmenoTy, 'actor', nil, true],

        ['tebe/jí', &zajmenoTebe2J, 'actor', &zajmenoSebe, nil],
        ['tebe/ní', &zajmenoTebe2N, 'actor', &zajmenoSebe, nil],
        ['tě/jí', &zajmenoTe2J, 'actor', &zajmenoSebe, nil],
        ['tě/ní', &zajmenoTe2N, 'actor', &zajmenoSebe, nil],

        ['tobě/jemu', &zajmenoTobe3J, 'actor', &zajmenoSobe, nil],
        ['tobě/němu', &zajmenoTobe3N, 'actor', &zajmenoSobe, nil],
        ['ti/jí', &zajmenoTiJ, 'actor', &zajmenoSi, nil],
        ['ti/ní', &zajmenoTiN, 'actor', &zajmenoSi, nil],

        ['tebe/ji', &zajmenoTebe4J, 'actor', &zajmenoSebe, nil],
        ['tebe/ni', &zajmenoTebe4N, 'actor', &zajmenoSebe, nil],
        ['tě/ji', &zajmenoTe4J, 'actor', &zajmenoSe, nil],
        ['tě/ni', &zajmenoTe4N, 'actor', &zajmenoSe, nil],

        ['tobě/něm', &zajmenoTobe6, 'actor', &zajmenoSobe, nil],

        ['tebou/jím', &zajmenoTebouJ, 'actor', &zajmenoSebou, nil],
        ['tebou/ním', &zajmenoTebouN, 'actor', &zajmenoSebou, nil],

        ['on/ona', &zajmenoTy, nil, nil, true],

        ['jeho/jí', &zajmenoTebe2J, nil, &zajmenoSebe, nil],
        ['něho/ní', &zajmenoTebe2N, nil, &zajmenoSebe, nil],
        ['ho/jí', &zajmenoTe2J, nil, &zajmenoSebe, nil],
        ['ho/ní', &zajmenoTe2N, nil, &zajmenoSebe, nil],

        ['jemu/jí', &zajmenoTobe3J, nil, &zajmenoSobe, nil],
        ['němu/ní', &zajmenoTobe3N, nil, &zajmenoSobe, nil],
        ['mu/jí', &zajmenoTiJ, nil, &zajmenoSi, nil],
        ['mu/ní', &zajmenoTiN, nil, &zajmenoSi, nil],

        ['jeho/ji', &zajmenoTebe4J, nil, &zajmenoSebe, nil],
        ['něho/ni', &zajmenoTebe4N, nil, &zajmenoSebe, nil],
        ['ho/ji', &zajmenoTe4J, nil, &zajmenoSe, nil],
        ['ho/ni', &zajmenoTe4N, nil, &zajmenoSe, nil],
        ['jej/ji', &zajmenoJej4J, nil, &zajmenoSebe, nil],
        ['něj/ni', &zajmenoJej4N, nil, &zajmenoSebe, nil],

        ['něm/ní', &zajmenoTobe6, nil, &zajmenoSobe, nil],

        ['jím/jí', &zajmenoTebouJ, nil, &zajmenoSebou, nil],
        ['ním/ní', &zajmenoTebouN, nil, &zajmenoSebou, nil],

        /*
         *   Přivlastňovací zájmena
         *
         *   1. pád - zajmenoTvuj (m), zajmenoTvoje (ž, s), TODO: tvá (ž), tvé (s)
         *   4. pád - zajmenoTvoji (ž)
         */
        ['tvůj', &zajmenoTvuj, 'actor', nil, nil],
        ['tvoje', &zajmenoTvoje, 'actor', nil, nil],
        ['jeho/její', &zajmenoTvoje, 'actor', nil, nil],

        ['tvoji', &zajmenoTvoji, 'actor', nil, nil],

        /* ukazovací zájmena */
        ['sám', &zajmenoSam, 'actor', nil, nil],

        /* sloveso být s osobním zvratným zájmenem v minulém čase */
        ['ses', &spojeniSes, 'actor', nil, nil],
        ['sis', &spojeniSis, 'actor', nil, nil],

        /* slovesa pravidelná */
        ['eš', &verbEndingEs, 'actor', nil, true],
        ['neš', &verbEndingNes, 'actor', nil, true],
        ['ješ', &verbEndingJes, 'actor', nil, true],
        ['íš', &verbEndingIs, 'actor', nil, true],
        ['áš', &verbEndingAs, 'actor', nil, true],

        /* Koncovka pravidelných sloves v minulém čase. */
        ['a', &verbPastEnding, 'actor', nil, true],

        /* některá nepravidelná slovesa */
        ['je', &slovesoJe, nil, nil, nil],
        ['není', &slovesoNeni, nil, nil, nil],
        ['jsi', &slovesoBytProMinulyCas, 'actor', nil, true],
        ['jsi/je', &slovesoByt, 'actor', nil, true],
        ['nejsi/není', &slovesoNebyt, 'actor', nil, true],
        ['bys', &slovesoBytPodmin, 'actor', nil, true],
        ['chceš', &slovesoChces, 'actor', nil, true],
        ['šel', &slovesoSel, 'actor', nil, true],
        ['jde', &slovesoJde, 'actor', nil, true],

        /* default preposition for standing in/on something */
        ['v', &objInName, nil, nil, nil],
        ['do', &objIntoName, nil, nil, nil],
        ['z', &objOutOfName, nil, nil, nil],

        /* preposition vocalization */
        ['k/ke', &prepK, 'actor', nil, nil],
        ['s/se', &prepS, 'actor', nil, nil],
        ['v/ve', &prepV, 'actor', nil, nil],
        ['z/ze', &prepZ, 'actor', nil, nil],

        /*
         *   The special invisible subject marker - this can be used to
         *   mark the subject in sentences that vary from the
         *   subject-verb-object structure that most English sentences
         *   take.  The usual SVO structure allows the message builder to
         *   see the subject first in most sentences naturally, but in
         *   unusual sentence forms it is sometimes useful to be able to
         *   mark the subject explicitly.  This doesn't actually result in
         *   any output; it's purely for marking the subject for our
         *   internal book-keeping.
         *
         *   (The main reason the message builder wants to know the subject
         *   in the first place is so that it can use a reflexive pronoun
         *   if the same object ends up being used as a direct or indirect
         *   object: "you can't open yourself" rather than "you can't open
         *   you.")
         */
        ['subj', &dummyName, nil, nil, true]
    ]

    /*
     *   Add a hook to the generateMessage method, which we use to
     *   pre-process the source string before expanding the substitution
     *   parameters.
     */
    generateMessage(orig) { return inherited(processOrig(orig)); }

    /*
     *   Pre-process a source string containing substitution parameters,
     *   before generating the expanded message from it.
     *
     *   We use this hook to implement the special tense-switching syntax
     *   {<present>|<past>}.  Although it superficially looks like an
     *   ordinary substitution parameter, we actually can't use the normal
     *   parameter substitution framework for that, because we want to
     *   allow the <present> and <past> substrings themselves to contain
     *   substitution parameters, and the normal framework doesn't allow
     *   for recursive substitution.
     *
     *   We simply replace every sequence of the form {<present>|<past>}
     *   with either <present> or <past>, depending on the current
     *   narrative tense.  We then substitute braces for square brackets in
     *   the resulting string.  This allows treating every bracketed tag
     *   inside the tense-switching sequence as a regular substitution
     *   parameter.
     *
     *   For example, the sequence "{take[s]|took}" appearing in the
     *   message string would be replaced with "take{s}" if the current
     *   narrative tense is present, and would be replaced with "took" if
     *   the current narrative tense is past.  The string "take{s}", if
     *   selected, would in turn be expanded to either "take" or "takes",
     *   depending on the grammatical person of the subject, as per the
     *   regular substitution mechanism.
     */
    processOrig(str)
    {
        local idx = 1;
        local len;
        local match;
        local replStr;

        /*
         *   Keep searching the string until we run out of character
         *   sequences with a special meaning (specifically, we look for
         *   substrings enclosed in braces, and stuttered opening braces).
         */
        for (;;)
        {
            /*
             *   Find the next special sequence.
             */
            match = rexSearch(patSpecial, str, idx);

            /*
             *   If there are no more special sequence, we're done
             *   pre-processing the string.
             */
            if (match == nil) break;

            /*
             *   Remember the starting index and length of the special
             *   sequence.
             */
            idx = match[1];
            len = match[2];

            /*
             *   Check if this special sequence matches our tense-switching
             *   syntax.
             */
            if (rexMatch(patTenseSwitching, str, idx) == nil)
            {
                /*
                 *   It doesn't, so forget about it and continue searching
                 *   from the end of this special sequence.
                 */
                idx += len;
                continue;
            }

            /*
             *   Extract either the first or the second embedded string,
             *   depending on the current narrative tense.
             */
            match = rexGroup(tSel(1, 2));
            replStr = match[3];

            /*
             *   Convert all square brackets to braces in the extracted
             *   string.
             */
            replStr = replStr.findReplace('[', '{', ReplaceAll);
            replStr = replStr.findReplace(']', '}', ReplaceAll);

            /*
             *   In the original string, replace the tense-switching
             *   sequence with the extracted string.
             */
            str = str.substr(1, idx - 1) + replStr + str.substr(idx + len);

            /*
             *   Move the index at the end of the substituted string.
             */
            idx += match[2];
        }

        /*
         *   We're done - return the result.
         */
        return str;
    }

    /*
     *   Pre-compiled regular expression pattern matching any sequence with
     *   a special meaning in a message string.
     *
     *   We match either a stuttered opening brace, or a single opening
     *   brace followed by any sequence of characters that doesn't contain
     *   a closing brace followed by a closing brace.
     */
    patSpecial = static new RexPattern
        ('<lbrace><lbrace>|<lbrace>(?!<lbrace>)((?:<^rbrace>)*)<rbrace>')

    /*
     *   Pre-compiled regular expression pattern matching our special
     *   tense-switching syntax.
     *
     *   We match a single opening brace, followed by any sequence of
     *   characters that doesn't contain a closing brace or a vertical bar,
     *   followed by a vertical bar, followed by any sequence of characters
     *   that doesn't contain a closing brace or a vertical bar, followed
     *   by a closing brace.
     */
    patTenseSwitching = static new RexPattern
    (
        '<lbrace>(?!<lbrace>)((?:<^rbrace|vbar>)*)<vbar>'
                          + '((?:<^rbrace|vbar>)*)<rbrace>'
    )

    /*
     *   The most recent target object used in the nominative case.  We
     *   note this so that we can supply reflexive mappings when the same
     *   object is re-used in the objective case.  This allows us to map
     *   things like "you can't take you" to the better-sounding "you
     *   can't take yourself".
     */
    lastSubject_ = nil

    /* the parameter name of the last subject ('dobj', 'actor', etc) */
    lastSubjectName_ = nil

    /*
     *   Get the target object property mapping.  If the target object is
     *   the same as the most recent subject object (i.e., the last object
     *   used in the nominative case), and this parameter has a reflexive
     *   form property, we'll return the reflexive form property.
     *   Otherwise, we'll return the standard property mapping.
     *
     *   Also, if there was an exclamation mark at the end of any word in
     *   the tag, we'll return a property returning a fixed-tense form of
     *   the property for the tag.
     */
    getTargetProp(targetObj, paramObj, info)
    {
        local ret;

        /*
         *   If this target object matches the last subject, and we have a
         *   reflexive rendering, return the property for the reflexive
         *   rendering.
         *
         *   Only use the reflexive rendering if the parameter name is
         *   different - if the parameter name is the same, then presumably
         *   the message will have been written with a reflexive pronoun or
         *   not, exactly as the author wants it.  When the author knows
         *   going in that these two objects are structurally the same,
         *   they want the exact usage they wrote.
         */
        if (targetObj == lastSubject_
            && paramObj != lastSubjectName_
            && info[4] != nil)
        {
            /* use the reflexive rendering */
            ret = info[4];
        }
        else
        {
            /* no special handling; inherit the default handling */
            ret = inherited(targetObj, paramObj, info);
        }

        /* if this is a nominative usage, note it as the last subject */
        if (info[5])
        {
            lastSubject_ = targetObj;
            lastSubjectName_ = paramObj;
        }

        /*
         *   If there was an exclamation mark at the end of any word in the
         *   parameter string (which we remember via the fixedTenseProp_
         *   property), store the original target property in
         *   fixedTenseProp_ and use &propWithPresentMessageBuilder_ as the
         *   target property instead.  propWithPresentMessageBuilder_ acts
         *   as a wrapper for the original target property, which it
         *   invokes after temporarily switching to the present tense.
         */
        if (fixedTenseProp_)
        {
            fixedTenseProp_ = ret;
            ret = &propWithPresentMessageBuilder_;
        }

        /* return the result */
        return ret;
    }

    /* end-of-sentence match pattern */
    patEndOfSentence = static new RexPattern('[.;:!?]<^alphanum>')

    /*
     *   Process result text.
     */
    processResult(txt)
    {
        /*
         *   If the text contains any sentence-ending punctuation, reset
         *   our internal memory of the subject of the sentence.  We
         *   consider the sentence to end with a period, semicolon, colon,
         *   question mark, or exclamation point followed by anything
         *   other than an alpha-numeric.  (We require the secondary
         *   character so that we don't confuse things like "3:00" or
         *   "7.5" to contain sentence-ending punctuation.)
         */
        if (rexSearch(patEndOfSentence, txt) != nil)
        {
            /*
             *   we have a sentence ending in this run of text, so any
             *   saved subject object will no longer apply after this text
             *   - forget our subject object
             */
            lastSubject_ = nil;
            lastSubjectName_ = nil;
        }

        /* return the inherited processing */
        return inherited(txt);
    }

    /* some pre-compiled search patterns we use a lot */
    patIdObjSlashIdApostS = static new RexPattern(
        '(<^space>+)(<space>+<^space>+)\'s(/<^space>+)$')
    patIdObjApostS = static new RexPattern(
        '(?!<^space>+\'s<space>)(<^space>+)(<space>+<^space>+)\'s$')
    patParamWithExclam = static new RexPattern('.*(!)(?:<space>.*|/.*|$)')

    /*
     *   Rewrite a parameter string for a language-specific syntax
     *   extension.
     *
     *   For English, we'll handle the possessive apostrophe-s suffix
     *   specially, by allowing the apostrophe-s to be appended to the
     *   target object name.  If we find an apostrophe-s on the target
     *   object name, we'll move it to the preceding identifier name:
     *
     *   the dobj's -> the's dobj
     *.  the dobj's/he -> the's dobj/he
     *.  he/the dobj's -> he/the's dobj
     *
     *   We also use this method to check for the presence of an
     *   exclamation mark at the end of any word in the parameter string
     *   (triggering the fixed-tense handling), and to detect a parameter
     *   string matching the {s/?ed} syntax, where ? is any letter, and
     *   rewrite it literally as 's/?ed' literally.
     */
    langRewriteParam(paramStr)
    {
        /*
         *   Check for an exclamation mark at the end of any word in the
         *   parameter string, and remember the result of the test.
         */
        local exclam = rexMatch(patParamWithExclam, paramStr);
        fixedTenseProp_ = exclam;

        /*
         *   Remove the exclamation mark, if any.
         */
        if (exclam)
        {
            local exclamInd = rexGroup(1)[1];
            paramStr = paramStr.substr(1, exclamInd - 1)
                       + paramStr.substr(exclamInd + 1);
        }

        /* look for "id obj's" and "id1 obj's/id2" */
        if (rexMatch(patIdObjSlashIdApostS, paramStr) != nil)
        {
            /* rewrite with the "'s" moved to the preceding parameter name */
            paramStr = rexGroup(1)[3] + '\'s'
                       + rexGroup(2)[3] + rexGroup(3)[3];
        }
        else if (rexMatch(patIdObjApostS, paramStr) != nil)
        {
            /* rewrite with the "'s" moved to the preceding parameter name */
            paramStr = rexGroup(1)[3] + '\'s' + rexGroup(2)[3];
        }

        /* return our (possibly modified) result */
        return paramStr;
    }

    /*
     *   This property is used to temporarily store either a boolean value
     *   indicating whether the last encountered parameter string had an
     *   exclamation mark at the end of any word, or a property to be
     *   invoked by Thing.propWithPresentMessageBuilder_.  This field is
     *   for internal use only; authors shouldn't have any reason to access
     *   it directly.
     */
    fixedTenseProp_ = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   Temporarily override the current narrative tense and invoke a callback
 *   function.
 */
withTense(usePastTense, callback)
{
    /*
     *   Remember the old value of the usePastTense flag.
     */
    local oldUsePastTense = gameMain.usePastTense;
    /*
     *   Set the new value.
     */
    gameMain.usePastTense = usePastTense;
    /*
     *   Invoke the callback (remembering the return value) and restore the
     *   usePastTense flag on our way out.
     */
    local ret;
    try { ret = callback(); }
    finally { gameMain.usePastTense = oldUsePastTense; }
    /*
     *   Return the result.
     */
    return ret;
}


/* ------------------------------------------------------------------------ */
/*
 *   Functions for spelling out numbers.  These functions take a numeric
 *   value as input, and return a string with the number spelled out as
 *   words in English.  For example, given the number 52, we'd return a
 *   string like 'fifty-two'.
 *
 *   These functions obviously have language-specific implementations.
 *   Note also that even their interfaces might vary by language.  Some
 *   languages might need additional information in the interface; for
 *   example, some languages might need to know the grammatical context
 *   (such as part of speech, case, or gender) of the result.
 *
 *   Note that some of the spellIntXxx flags might not be meaningful in all
 *   languages, because most of the flags are by their very nature
 *   associated with language-specific idioms.  Translations are free to
 *   ignore flags that indicate variations with no local equivalent, and to
 *   add their own language-specific flags as needed.
 */

/*
 *   Spell out an integer number in words.  Returns a string with the
 *   spelled-out number.
 *
 *   Note that this simple version of the function uses the default
 *   options.  If you want to specify non-default options with the
 *   SpellIntXxx flags, you can call spellIntExt().
 */
spellInt(val, gender)
{
    return spellIntExt(val, gender, 0);
}

/*
 *   Spell out an integer number in words, but only if it's below the given
 *   threshold.  It's often awkward in prose to spell out large numbers,
 *   but exactly what constitutes a large number depends on context, so
 *   this routine lets the caller specify the threshold.
 *   
 *   If the absolute value of val is less than (not equal to) the threshold
 *   value, we'll return a string with the number spelled out.  If the
 *   absolute value is greater than or equal to the threshold value, we'll
 *   return a string representing the number in decimal digits.  
 */
spellIntBelow(val, gender, threshold)
{
    return spellIntBelowExt(val, gender, threshold, 0, 0);
}

/*
 *   Spell out an integer number in words if it's below a threshold, using
 *   the spellIntXxx flags given in spellFlags to control the spelled-out
 *   format, and using the DigitFormatXxx flags in digitFlags to control
 *   the digit format.  
 */
spellIntBelowExt(val, gender, threshold, spellFlags, digitFlags)
{
    local absval;

    /* compute the absolute value */
    absval = (val < 0 ? -val : val);

    /* check the value to see whether to spell it or write it as digits */
    if (absval < threshold)
    {
        /* it's below the threshold - spell it out in words */
        return spellIntExt(val, gender, spellFlags);
    }
    else
    {
        /* it's not below the threshold - write it as digits */
        return intToDecimal(val, digitFlags);
    }
}

/*
 *   Format a number as a string of decimal digits.  The DigitFormatXxx
 *   flags specify how the number is to be formatted.`
 */
intToDecimal(val, flags)
{
    local str;
    local sep;

    /* perform the basic conversion */
    str = toString(val);

    /* add group separators as needed */
    if ((flags & DigitFormatGroupComma) != 0)
    {
        /* explicitly use a comma as a separator */
        sep = ',';
    }
    else if ((flags & DigitFormatGroupPeriod) != 0)
    {
        /* explicitly use a period as a separator */
        sep = '.';
    }
    else if ((flags & DigitFormatGroupSep) != 0)
    {
        /* use the current languageGlobals separator */
        sep = languageGlobals.digitGroupSeparator;
    }
    else
    {
        /* no separator */
        sep = nil;
    }

    /* if there's a separator, add it in */
    if (sep != nil)
    {
        local i;
        local len;

        /*
         *   Insert the separator before each group of three digits.
         *   Start at the right end of the string and work left: peel off
         *   the last three digits and insert a comma.  Then, move back
         *   four characters through the string - another three-digit
         *   group, plus the comma we inserted - and repeat.  Keep going
         *   until the amount we'd want to peel off the end is as long or
         *   longer than the entire remaining string.
         */
        for (i = 3, len = str.length() ; len > i ; i += 4)
        {
            /* insert this comma */
            str = str.substr(1, len - i) + sep + str.substr(len - i + 1);

            /* note the new length */
            len = str.length();
        }
    }

    /* return the result */
    return str;
}

/*
 *   Spell out an integer number - "extended" interface with flags.  The
 *   "flags" argument is a (bitwise-OR'd) combination of SpellIntXxx
 *   values, specifying the desired format of the result.
 */
spellIntExt(val, gender, flags)
{
    local str;
    local trailingSpace;
    local needAnd;
    local jednicka = nil;
    local powers =
        [1000000000, 3, ' miliarda ', ' miliardy ', ' miliardy ', ' miliard ',
         1000000,    2, ' milión ', ' milióny ', ' milióny ', ' miliónů ',
         1000,       2, ' tisíc ', ' tisíce ', ' tisíce ', ' tisíc ',
         100,        4, ' sto ', ' stě ', ' sta ', ' set ' ];

    /* start with an empty string */
    str = '';
    trailingSpace = nil;
    needAnd = nil;

    /* if it's zero, it's a special case */
    if (val == 0)
        return 'nula';

    /*
     *   if the number is negative, note it in the string, and use the
     *   absolute value
     */
    if (val < 0)
    {
        str = 'mínus ';
        val = -val;
    }

    /* Pokud je to jednička, povolíme úpravy ohledně skloňování. */
    if (val == 1) jednicka = true;
    /*
     *   Přes dvacítku už spadnem do množného čísla i u číslovek 1 a 2,
     *   takže změníme rod na speciální hodnotu 5, címž se indexuje tabulka
     *   nums.
     */
    if (val > 20) gender = 5;

    /* do each named power of ten */
    for (local i = 1 ; val >= 100 && i <= powers.length() ; i += 6)
    {
        /*
         *   if we're in teen-hundreds mode, do the teen-hundreds - this
         *   only works for values from 1,100 to 9,999, since a number like
         *   12,000 doesn't work this way - 'one hundred twenty hundred' is
         *   no good 
         *
         *   V češtině je zvykem jít jen do devatenácti set.
         */
        if ((flags & SpellIntTeenHundreds) != 0
            && val >= 1100 && val < 2000)
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';

            /* spell it out as a number of hundreds */
            str += spellIntExt(val / 100, gender, flags) + ' set ';

            /* take off the hundreds */
            val %= 100;

            /* note the trailing space */
            trailingSpace = true;

            /* we have something to put an 'and' after, if desired */
            needAnd = true;

            /*
             *   whatever's left is below 100 now, so there's no need to
             *   keep scanning the big powers of ten
             */
            break;
        }

        /* if we have something in this power range, apply it */
        if (val >= powers[i])
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';

            /* add the number of multiples of this power and the power name */
            local num = val / powers[i];
            if(num != 1) str += spellIntExt(num, powers[i + 1], flags);
            str += powers[i + (num >= 5 ? 5 : num >= 3 ? 4 : num == 2 ? 3
                : 2)].substr(num == 1 ? 2 : 1);

            /* take it out of the remaining value */
            val %= powers[i];

            /*
             *   note that we have a trailing space in the string (all of
             *   the power-of-ten names have a trailing space, to make it
             *   easy to tack on the remainder of the value)
             */
            trailingSpace = true;

            /* we have something to put an 'and' after, if one is desired */
            needAnd = true;
        }
    }

    /*
     *   if we have anything left, and we have written something so far,
     *   and the caller wanted an 'and' before the tens part, add the
     *   'and'
     */
    if ((flags & SpellIntAndTens) != 0
        && needAnd
        && val != 0)
    {
        /* add the 'and' */
        str += 'a ';
        trailingSpace = true;
    }

    /* do the tens */
    if (val >= 20)
    {
        /* anything above the teens is nice and regular */
        str += ['dvacet', 'třicet', 'čtyřicet', 'padesát', 'šedesát',
                'sedmdesát', 'osmdesát', 'devadesát'][val/10 - 1];
        val %= 10;

        /* if it's non-zero, we'll add the units, so add a hyphen */
        if (val != 0)
            str += ' ';

        /* we no longer have a trailing space in the string */
        trailingSpace = nil;

	/*
	 *   Přes dvacítku už spadnem do množného čísla i u číslovek 1 a 2,
	 *   takže změníme rod na speciální hodnotu 5, címž se indexuje tabulka
	 *   nums.
	 */
	gender = 5;
    }
    else if (val >= 10)
    {
        /* we have a teen */
        str += ['deset', 'jedenáct', 'dvanáct', 'třináct', 'čtrnáct',
                'patnáct', 'šestnáct', 'sedmnáct', 'osmnáct',
                'devatenáct'][val - 9];

        /* we've finished with the number */
        val = 0;

        /* there's no trailing space */
        trailingSpace = nil;
    }

    /* if we have a units value, add it */
    if (val != 0)
    {
        /* add the units name */
	local nums = [
	'jeden', 'dva', 'tři', 'čtyři', 'pět', 'šest', 'sedm', 'osm', 'devět',
	'jeden', 'dva', 'tři', 'čtyři', 'pět', 'šest', 'sedm', 'osm', 'devět',
	'jedna', 'dvě', 'tři', 'čtyři', 'pět', 'šest', 'sedm', 'osm', 'devět',
	'jedno', 'dvě', 'tři', 'čtyři', 'pět', 'šest', 'sedm', 'osm', 'devět',
	'jedna', 'dva', 'tři', 'čtyři', 'pět', 'šest', 'sedm', 'osm', 'devět'
	];

        /* Ve čtvrtém pádu měníme tvar pro 1. */
        if(flags & SpellIntKohoCo && jednicka)
        {
            nums[1] = 'jednoho';
            nums[19] = 'jednu';
        }
        str += nums[val + (gender - 1) * 9];

        /* we have no trailing space now */
        trailingSpace = nil;
    }

    /* if there's a trailing space, remove it */
    if (trailingSpace)
        str = str.substr(1, str.length() - 1);

    /* return the string */
    return str;
}

/*
 *   Return a string giving the numeric ordinal representation of a number:
 *   1st, 2nd, 3rd, 4th, etc.  
 */
intOrdinal(n)
{
    local s;

    /* start by getting the string form of the number */
    s = toString(n);

    /* now add the appropriate suffix */
    if (n >= 10 && n <= 19)
    {
        /* the teens all end in 'th' */
        return s + 'th';
    }
    else
    {
        /*
         *   for anything but a teen, a number whose last digit is 1
         *   always has the suffix 'st' (for 'xxx-first', as in '141st'),
         *   a number whose last digit is 2 always ends in 'nd' (for
         *   'xxx-second', as in '532nd'), a number whose last digit is 3
         *   ends in 'rd' (for 'xxx-third', as in '53rd'), and anything
         *   else ends in 'th'
         */
        switch(n % 10)
        {
        case 1:
            return s + 'st';

        case 2:
            return s + 'nd';

        case 3:
            return s + 'rd';

        default:
            return s + 'th';
        }
    }
}


/*
 *   Return a string giving a fully spelled-out ordinal form of a number:
 *   first, second, third, etc.
 */
spellIntOrdinal(n)
{
    return spellIntOrdinalExt(n, 0);
}

/*
 *   Return a string giving a fully spelled-out ordinal form of a number:
 *   first, second, third, etc.  This form takes the same flag values as
 *   spellIntExt().
 */
spellIntOrdinalExt(n, flags)
{
    local s;

    /* get the spelled-out form of the number itself */
    /* TODO: tady davam ctyrku prozatimne */
    s = spellIntExt(n, 4, flags);

    /*
     *   If the number ends in 'one', change the ending to 'first'; 'two'
     *   becomes 'second'; 'three' becomes 'third'; 'five' becomes
     *   'fifth'; 'eight' becomes 'eighth'; 'nine' becomes 'ninth'.  If
     *   the number ends in 'y', change the 'y' to 'ieth'.  'Zero' becomes
     *   'zeroeth'.  For everything else, just add 'th' to the spelled-out
     *   name
     */
    if (s == 'zero')
        return 'zeroeth';
    if (s.endsWith('one'))
        return s.substr(1, s.length() - 3) + 'first';
    else if (s.endsWith('two'))
        return s.substr(1, s.length() - 3) + 'second';
    else if (s.endsWith('three'))
        return s.substr(1, s.length() - 5) + 'third';
    else if (s.endsWith('five'))
        return s.substr(1, s.length() - 4) + 'fifth';
    else if (s.endsWith('eight'))
        return s.substr(1, s.length() - 5) + 'eighth';
    else if (s.endsWith('nine'))
        return s.substr(1, s.length() - 4) + 'ninth';
    else if (s.endsWith('y'))
        return s.substr(1, s.length() - 1) + 'ieth';
    else
        return s + 'th';
}

/* ------------------------------------------------------------------------ */
/*
 *   Parse a spelled-out number.  This is essentially the reverse of
 *   spellInt() and related functions: we take a string that contains a
 *   spelled-out number and return the integer value.  This uses the
 *   command parser's spelled-out number rules, so we can parse anything
 *   that would be recognized as a number in a command.
 *
 *   If the string contains numerals, we'll treat it as a number in digit
 *   format: for example, if it contains '789', we'll return 789.
 *
 *   If the string doesn't parse as a number, we return nil.
 */
parseInt(str)
{
    try
    {
        /* tokenize the string */
        local toks = cmdTokenizer.tokenize(str);

        /* parse it */
        return parseIntTokens(toks);
    }
    catch (Exception exc)
    {
        /*
         *   on any exception, just return nil to indicate that we couldn't
         *   parse the string as a number
         */
        return nil;
    }
}

/*
 *   Parse a spelled-out number that's given as a token list (as returned
 *   from Tokenizer.tokenize).  If we can successfully parse the token list
 *   as a number, we'll return the integer value.  If not, we'll return
 *   nil.
 */
parseIntTokens(toks)
{
    try
    {
        /*
         *   if the first token contains digits, treat it as a numeric
         *   string value rather than a spelled-out number
         */
        if (toks.length() != 0
            && rexMatch('<digit>+', getTokOrig(toks[1])) != nil)
            return toInteger(getTokOrig(toks[1]));

        /* parse it using the spelledNumber production */
        local lst = spelledNumber.parseTokens(toks, cmdDict);

        /*
         *   if we got a match, return the integer value; if not, it's not
         *   parseable as a number, so return nil
         */
        return (lst.length() != 0 ? lst[1].getval() : nil);
    }
    catch (Exception exc)
    {
        /*
         *   on any exception, just return nil to indicate that it's not
         *   parseable as a number
         */
        return nil;
    }
}


/* ------------------------------------------------------------------------ */
/*
 *   Additional token types for US English.
 *
 *   Cesky preklad tokApostropheS a tokPluralApostrophe vubec nepouziva, ale
 *   napr. rozsireni SpellingCorrector se na symbol odvolava, tak je ponechame.
 */

/* special "apostrophe-s" token */
enum token tokApostropheS;

/* special apostrophe token for plural possessives ("the smiths' house") */
enum token tokPluralApostrophe;

/* special abbreviation-period token */
enum token tokAbbrPeriod;

/* special "#nnn" numeric token */
enum token tokPoundInt;

/*
 *   Command tokenizer for US English.  Other language modules should
 *   provide their own tokenizers to allow for differences in punctuation
 *   and other lexical elements.
 */
cmdTokenizer: Tokenizer
    rules_ = static
    [
        /* skip whitespace */
        ['whitespace', new RexPattern('<Space>+'), nil, &tokCvtSkip, nil],

        /* certain punctuation marks */
        ['punctuation', new RexPattern('[.,;:?!]'), tokPunct, nil, nil],

        /*
         *   We have a special rule for spelled-out numbers from 21 to 99:
         *   when we see a 'tens' word followed by a hyphen followed by a
         *   digits word, we'll pull out the tens word, the hyphen, and
         *   the digits word as separate tokens.
         */
        ['spelled number',
         new RexPattern('<NoCase>(twenty|thirty|forty|fifty|sixty|'
                        + 'seventy|eighty|ninety)-'
                        + '(one|two|three|four|five|six|seven|eight|nine)'
                        + '(?!<AlphaNum>)'),
         tokWord, &tokCvtSpelledNumber, nil],


        /*
         *   Initials.  We'll look for strings of three or two initials,
         *   set off by periods but without spaces.  We'll look for
         *   three-letter initials first ("G.H.W. Billfold"), then
         *   two-letter initials ("X.Y. Zed"), so that we find the longest
         *   sequence that's actually in the dictionary.  Note that we
         *   don't have a separate rule for individual initials, since
         *   we'll pick that up with the regular abbreviated word rule
         *   below.
         *
         *   Some games could conceivably extend this to allow strings of
         *   initials of four letters or longer, but in practice people
         *   tend to elide the periods in longer sets of initials, so that
         *   the initials become an acronym, and thus would fit the
         *   ordinary word token rule.
         */
        ['three initials',
         new RexPattern('<alpha><period><alpha><period><alpha><period>'),
         tokWord, &tokCvtAbbr, &acceptAbbrTok],

        ['two initials',
         new RexPattern('<alpha><period><alpha><period>'),
         tokWord, &tokCvtAbbr, &acceptAbbrTok],

        /*
         *   Abbbreviated word - this is a word that ends in a period,
         *   such as "Mr.".  This rule comes before the ordinary word rule
         *   because we will only consider the period to be part of the
         *   word (and not a separate token) if the entire string
         *   including the period is in the main vocabulary dictionary.
         */
        ['abbreviation',
         new RexPattern('<Alpha|-><AlphaNum|-|squote>*<period>'),
         tokWord, &tokCvtAbbr, &acceptAbbrTok],

        /*
         *   Words - note that we convert everything to lower-case.  A word
         *   must start with an alphabetic character, a hyphen, or an
         *   ampersand; after the initial character, a word can contain
         *   alphabetics, digits, hyphens, ampersands, and apostrophes.
         */
        ['word',
         new RexPattern('<Alpha|-|&><AlphaNum|-|&|squote>*'),
         tokWord, nil, nil],

        /* an abbreviation word starting with a number */
        ['abbreviation with initial digit',
         new RexPattern('<Digit>(?=<AlphaNum|-|&|squote>*<Alpha|-|&|squote>)'
                        + '<AlphaNum|-|&|squote>*<period>'),
         tokWord, &tokCvtAbbr, &acceptAbbrTok],

        /*
         *   A word can start with a number, as long as there's something
         *   other than numbers in the string - if it's all numbers, we
         *   want to treat it as a numeric token.
         */
        ['word with initial digit',
         new RexPattern('<Digit>(?=<AlphaNum|-|&|squote>*<Alpha|-|&|squote>)'
                        + '<AlphaNum|-|&|squote>*'), tokWord, nil, nil],

        /* strings with ASCII "straight" quotes */
        ['string ascii-quote',
         new RexPattern('<min>([`\'"])(.*)%1(?!<AlphaNum>)'),
         tokString, nil, nil],

        /* some people like to use single quotes like `this' */
        ['string back-quote',
         new RexPattern('<min>`(.*)\'(?!<AlphaNum>)'), tokString, nil, nil],

        /* strings with Latin-1 curly quotes (single and double) */
        ['string curly single-quote',
         new RexPattern('<min>\u2018(.*)\u2019'), tokString, nil, nil],
        ['string curly double-quote',
         new RexPattern('<min>\u201C(.*)\u201D'), tokString, nil, nil],

        /*
         *   unterminated string - if we didn't just match a terminated
         *   string, but we have what looks like the start of a string,
         *   match to the end of the line
         */
        ['string unterminated',
         new RexPattern('([`\'"\u2018\u201C](.*)'), tokString, nil, nil],

        /* integer numbers */
        ['integer', new RexPattern('[0-9]+'), tokInt, nil, nil],

        /* numbers with a '#' preceding */
        ['integer with #',
         new RexPattern('#[0-9]+'), tokPoundInt, nil, nil]
    ]

    /*
     *   Handle a spelled-out hyphenated number from 21 to 99.  We'll
     *   return this as three separate tokens: a word for the tens name, a
     *   word for the hyphen, and a word for the units name.
     */
    tokCvtSpelledNumber(txt, typ, toks)
    {
        /* parse the number into its three parts with a regular expression */
        rexMatch(patAlphaDashAlpha, txt);

        /* add the part before the hyphen */
        toks.append([rexGroup(1)[3], typ, rexGroup(1)[3]]);

        /* add the hyphen */
        toks.append(['-', typ, '-']);

        /* add the part after the hyphen */
        toks.append([rexGroup(2)[3], typ, rexGroup(2)[3]]);
    }
    patAlphaDashAlpha = static new RexPattern('(<alpha>+)-(<alpha>+)')

    /*
     *   Check to see if we want to accept an abbreviated token - this is
     *   a token that ends in a period, which we use for abbreviated words
     *   like "Mr." or "Ave."  We'll accept the token only if it appears
     *   as given - including the period - in the dictionary.  Note that
     *   we ignore truncated matches, since the only way we'll accept a
     *   period in a word token is as the last character; there is thus no
     *   way that a token ending in a period could be a truncation of any
     *   longer valid token.
     */
    acceptAbbrTok(txt)
    {
        /* look up the word, filtering out truncated results */
        return cmdDict.isWordDefined(
            txt, {result: (result & StrCompTrunc) == 0});
    }

    /*
     *   Process an abbreviated token.
     *
     *   When we find an abbreviation, we'll enter it with the abbreviated
     *   word minus the trailing period, plus the period as a separate
     *   token.  We'll mark the period as an "abbreviation period" so that
     *   grammar rules will be able to consider treating it as an
     *   abbreviation -- but since it's also a regular period, grammar
     *   rules that treat periods as regular punctuation will also be able
     *   to try to match the result.  This will ensure that we try it both
     *   ways - as abbreviation and as a word with punctuation - and pick
     *   the one that gives us the best result.
     */
    tokCvtAbbr(txt, typ, toks)
    {
        local w;

        /* add the part before the period as the ordinary token */
        w = txt.substr(1, txt.length() - 1);
        toks.append([w, typ, w]);

        /* add the token for the "abbreviation period" */
        toks.append(['.', tokAbbrPeriod, '.']);
    }

    /*
     *   Given a list of token strings, rebuild the original input string.
     *   We can't recover the exact input string, because the tokenization
     *   process throws away whitespace information, but we can at least
     *   come up with something that will display cleanly and produce the
     *   same results when run through the tokenizer.
     */
    buildOrigText(toks)
    {
        local str;

        /* start with an empty string */
        str = '';

        /* concatenate each token in the list */
        for (local i = 1, local len = toks.length() ; i <= len ; ++i)
        {
            /* add the current token to the string */
            str += getTokOrig(toks[i]);

            /*
             *   if this looks like a hyphenated number that we picked
             *   apart into two tokens, put it back together without
             *   spaces
             */
            if (i + 2 <= len
                && rexMatch(patSpelledTens, getTokVal(toks[i])) != nil
                && getTokVal(toks[i+1]) == '-'
                && rexMatch(patSpelledUnits, getTokVal(toks[i+2])) != nil)
            {
                /*
                 *   it's a hyphenated number, all right - put the three
                 *   tokens back together without any intervening spaces,
                 *   so ['twenty', '-', 'one'] turns into 'twenty-one'
                 */
                str += getTokOrig(toks[i+1]) + getTokOrig(toks[i+2]);

                /* skip ahead by the two extra tokens we're adding */
                i += 2;
            }

            /*
             *   if another token follows, and the next token isn't a
             *   punctuation mark, add a space before the next token
             */
            if (i != len && rexMatch(patPunct, getTokVal(toks[i+1])) == nil)
                str += ' ';
        }

        /* return the result string */
        return str;
    }

    /* some pre-compiled regular expressions */
    patSpelledTens = static new RexPattern(
        '<nocase>twenty|thirty|forty|fifty|sixty|seventy|eighty|ninety')
    patSpelledUnits = static new RexPattern(
        '<nocase>one|two|three|four|five|six|seven|eight|nine')
    patPunct = static new RexPattern('[.,;:?!]')
;


/* ------------------------------------------------------------------------ */
/*
 *   Grammar Rules
 */

/*
 *   Command with explicit target actor.  When a command starts with an
 *   actor's name followed by a comma followed by a verb, we take it as
 *   being directed to the actor.
 */
grammar firstCommandPhrase(withActor):
    singleNounOnly->actor_ ',' commandPhrase->cmd_
    : FirstCommandProdWithActor

    /* "execute" the target actor phrase */
    execActorPhrase(issuingActor)
    {
        /* flag that the actor's being addressed in the second person */
        resolvedActor_.commandReferralPerson = SecondPerson;
    }
;

/* V češtině by vyžadovalo jiné tvary příkazů */
//grammar firstCommandPhrase(askTellActorTo):
//    ('požádej' | 'řekni' | 'pož' | 'řek') singleNounOnly->actor_
//    'aby' commandPhrase->cmd_
//    : FirstCommandProdWithActor
//
//    /* "execute" the target actor phrase */
//    execActorPhrase(issuingActor)
//    {
//        /*
//         *   Since our phrasing is TELL <ACTOR> TO <DO SOMETHING>, the
//         *   actor clearly becomes the antecedent for a subsequent
//         *   pronoun.  For example, in TELL BOB TO READ HIS BOOK, the word
//         *   HIS pretty clearly refers back to BOB.
//         */
//        if (resolvedActor_ != nil)
//        {
//            /* set the possessive anaphor object to the actor */
//            resolvedActor_.setPossAnaphorObj(resolvedActor_);
//
//            /* flag that the actor's being addressed in the third person */
//            resolvedActor_.commandReferralPerson = ThirdPerson;
//
//            /*
//             *   in subsequent commands carried out by the issuer, the
//             *   target actor is now the pronoun antecedent (for example:
//             *   after TELL BOB TO GO NORTH, the command FOLLOW HIM means
//             *   to follow Bob)
//             */
//            issuingActor.setPronounObj(resolvedActor_);
//        }
//    }
//;

/*
 *   An actor-targeted command with a bad command phrase.  This is used as
 *   a fallback if we fail to match anything on the first attempt at
 *   parsing the first command on a line.  The point is to at least detect
 *   the target actor phrase, if that much is valid, so that we better
 *   customize error messages for the rest of the command.  
 */
grammar actorBadCommandPhrase(main):
    singleNounOnly->actor_ ',' miscWordList
    | ('ask' | 'tell' | 'a' | 't') singleNounOnly->actor_ 'to' miscWordList
    : FirstCommandProdWithActor

    /* to resolve nouns, we merely resolve the actor */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* resolve the underlying actor phrase */
        return actor_.resolveNouns(getResolver(issuingActor), results);
    }
;


/*
 *   Command-only conjunctions.  These words and groups of words can
 *   separate commands from one another, and can't be used to separate noun
 *   phrases in a noun list.  
 */
grammar commandOnlyConjunction(sentenceEnding):
    '.'
    | '!'
    : BasicProd

    /* these conjunctions end the sentence */
    isEndOfSentence() { return true; }
;

grammar commandOnlyConjunction(nonSentenceEnding):
    ( | 'a' | ',' | ',' 'a') ('potom' | 'pak' | 'následně')
    | ';'
    : BasicProd

    /* these conjunctions do not end a sentence */
    isEndOfSentence() { return nil; }
;


/*
 *   Command-or-noun conjunctions.  These words and groups of words can be
 *   used to separate commands from one another, and can also be used to
 *   separate noun phrases in a noun list.
 */
grammar commandOrNounConjunction(main):
    ','
    | 'a'
    | ',' 'a'
    : BasicProd

    /* these do not end a sentence */
    isEndOfSentence() { return nil; }
;

/*
 *   Noun conjunctions.  These words and groups of words can be used to
 *   separate noun phrases from one another.  Note that these do not need
 *   to be exclusive to noun phrases - these can occur as command
 *   conjunctions as well; this list is separated from
 *   commandOrNounConjunction in case there are conjunctions that can never
 *   be used as command conjunctions, since such conjunctions, which can
 *   appear here, would not appear in commandOrNounConjunctions.  
 */
grammar nounConjunction(main):
    ','
    | 'a'
    : BasicProd

    /* these conjunctions do not end a sentence */
    isEndOfSentence() { return nil; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun list: one or more noun phrases connected with conjunctions.  This
 *   kind of noun list can end in a terminal noun phrase.
 *   
 *   Note that a single noun phrase is a valid noun list, since a list can
 *   simply be a list of one.  The separate production nounMultiList can be
 *   used when multiple noun phrases are required.  
 */

/*
 *   a noun list can consist of a single terminal noun phrase
 */
grammar nounList(terminal): terminalNounPhrase->np_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying noun phrase */
        return np_.resolveNouns(resolver, results);
    }
;

/*
 *   a noun list can consist of a list of a single complete (non-terminal)
 *   noun phrase
 */
grammar nounList(nonTerminal): completeNounPhrase->np_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying noun phrase */
        return np_.resolveNouns(resolver, results);
    }
;

/*
 *   a noun list can consist of a list with two or more entries
 */
grammar nounList(list): nounMultiList->lst_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying list */
        return lst_.resolveNouns(resolver, results);
    }
;

/*
 *   An empty noun list is one with no words at all.  This is matched when
 *   a command requires a noun list but the player doesn't include one;
 *   this construct has "badness" because we only want to match it when we
 *   have no choice.
 */
grammar nounList(empty): [badness 500] : EmptyNounPhraseProd
    responseProd = nounList
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun Multi List: two or more noun phrases connected by conjunctions.
 *   This is almost the same as the basic nounList production, but this
 *   type of production requires at least two noun phrases, whereas the
 *   basic nounList production more generally defines its list as any
 *   number - including one - of noun phrases.
 */

/*
 *   a multi list can consist of a noun multi- list plus a terminal noun
 *   phrase, separated by a conjunction
 */
grammar nounMultiList(multi):
    nounMultiList->lst_ nounConjunction terminalNounPhrase->np_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return a list of all of the objects from both underlying lists */
        return np_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;

/*
 *   a multi list can consist of a non-terminal multi list
 */
grammar nounMultiList(nonterminal): nonTerminalNounMultiList->lst_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying list */
        return lst_.resolveNouns(resolver, results);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A non-terminal noun multi list is a noun list made up of at least two
 *   non-terminal noun phrases, connected by conjunctions.
 *
 *   This is almost the same as the regular non-terminal noun list
 *   production, but this production requires two or more underlying noun
 *   phrases, whereas the basic non-terminal noun list matches any number
 *   of underlying phrases, including one.
 */

/*
 *   a non-terminal multi-list can consist of a pair of complete noun
 *   phrases separated by a conjunction
 */
grammar nonTerminalNounMultiList(pair):
    completeNounPhrase->np1_ nounConjunction completeNounPhrase->np2_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return the combination of the two underlying noun phrases */
        return np1_.resolveNouns(resolver, results)
            + np2_.resolveNouns(resolver, results);
    }
;

/*
 *   a non-terminal multi-list can consist of another non-terminal
 *   multi-list plus a complete noun phrase, connected by a conjunction
 */
grammar nonTerminalNounMultiList(multi):
    nonTerminalNounMultiList->lst_ nounConjunction completeNounPhrase->np_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return the combination of the sublist and the noun phrase */
        return lst_.resolveNouns(resolver, results)
            + np_.resolveNouns(resolver, results);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   "Except" list.  This is a noun list that can contain anything that's
 *   in a regular noun list plus some things that only make sense as
 *   exceptions, such as possessive nouns (e.g., "mine").
 */

grammar exceptList(single): exceptNounPhrase->np_ : ExceptListProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }
;

grammar exceptList(list):
    exceptNounPhrase->np_ nounConjunction exceptList->lst_
    : ExceptListProd
    resolveNouns(resolver, results)
    {
        /* return a list consisting of all of our objects */
        return np_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;

/*
 *   An "except" noun phrase is a normal "complete" noun phrase or a
 *   possessive noun phrase that doesn't explicitly qualify another phrase
 *   (for example, "all the coins but bob's" - the "bob's" is just a
 *   possessive noun phrase without another noun phrase attached, since it
 *   implicitly qualifies "the coins").
 */
grammar exceptNounPhrase(singleComplete): completeNounPhraseWithoutAll->np_
    : ExceptListProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }
;

grammar exceptNounPhrase(singlePossessive): possessiveNounPhrase->poss_
    : ButPossessiveProd
;


/* ------------------------------------------------------------------------ */
/*
 *   A single noun is sometimes required where, structurally, a list is not
 *   allowed.  Single nouns should not be used to prohibit lists where
 *   there is no structural reason for the prohibition - these should be
 *   used only where it doesn't make sense to use a list structurally.  
 */
grammar singleNoun(normal): singleNounOnly->np_ : LayeredNounPhraseProd
;

/*
 *   An empty single noun is one with no words at all.  This is matched
 *   when a command requires a noun list but the player doesn't include
 *   one; this construct has "badness" because we only want to match it
 *   when we have no choice.
 */
grammar singleNoun(empty): [badness 500] : EmptyNounPhraseProd
    /* use a nil responseProd, so that we get the phrasing from the action */
    responseProd = nil

    /* the fallback responseProd, if we can't get one from the action */
    fallbackResponseProd = singleNoun
;

/*
 *   A user could attempt to use a noun list with more than one entry (a
 *   "multi list") where a single noun is required.  This is not a
 *   grammatical error, so we accept it grammatically; however, for
 *   disambiguation purposes we score it lower than a singleNoun production
 *   with only one noun phrase, and if we try to resolve it, we'll fail
 *   with an error.  
 */
grammar singleNoun(multiple): nounMultiList->np_ : SingleNounWithListProd
;


/*
 *   A *structural* single noun phrase.  This production is for use where a
 *   single noun phrase (not a list of nouns) is required grammatically.
 */
grammar singleNounOnly(main):
    terminalNounPhrase->np_
    | completeNounPhrase->np_
    : SingleNounProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Prepositionally modified single noun phrases.  These can be used in
 *   indirect object responses, so allow for interactions like this:
 *   
 *   >unlock door
 *.  What do you want to unlock it with?
 *   
 *   >with the key
 *   
 *   The entire notion of prepositionally qualified noun phrases in
 *   interactive indirect object responses is specific to English, so this
 *   is implemented in the English module only.  However, the general
 *   notion of specialized responses to interactive indirect object queries
 *   is handled in the language-independent library in some cases, in such
 *   a way that the language-specific library can customize the behavior -
 *   see TIAction.askIobjResponseProd.  
 */
class PrepSingleNounProd: SingleNounProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }

    /*
     *   If the response starts with a preposition, it's pretty clearly a
     *   response to the special query rather than a new command.
     */
    isSpecialResponseMatch()
    {
        return (np_ != nil && np_.firstTokenIndex > 1);
    }
;

/*
 *   Same thing for a Topic phrase
 */
class PrepSingleTopicProd: TopicProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }

    /* Take jako u PrepSingleNounProd, napr. "na co se chces zeptat? >na bednu" */
    isSpecialResponseMatch()
    {
        return (np_ != nil && np_.firstTokenIndex > 1);
    }
;

grammar inSingleNoun(main):
    ('v' | 've' | ) singleNoun->np_
    : PrepSingleNounProd
;

grammar forSingleNoun(main):
    ('o' | ) singleNoun->np_ : PrepSingleNounProd
;

/* Umožnit předlozku v odpovědi na dotaz: "Do čeho chceš vstoupit? >do pramice" */
grammar toSingleNoun(main):
    ('do' | ) singleNoun->np_ : PrepSingleNounProd
;

grammar throughSingleNoun(main):
    ('skrz' | 'skrze' | ) singleNoun->np_
    : PrepSingleNounProd
;

grammar fromSingleNoun(main):
    ('od' | ) singleNoun->np_  : PrepSingleNounProd
;

grammar onSingleNoun(main):
    ('na' | 'po' | 'do' | ) singleNoun->np_
    : PrepSingleNounProd
;

grammar withSingleNoun(main):
    ('s' | 'se' | ) singleNoun->np_ : PrepSingleNounProd
;

grammar atSingleNoun(main):
    ('k' | 'ke' | 'na' | 'do' | ) singleNoun->np_ : PrepSingleNounProd
;

grammar alongSingleNoun(main):
    ('po' | 'na' | ) singleNoun->np_ : PrepSingleNounProd
;

grammar outOfSingleNoun(main):
    ('z' | 'ze' | ) singleNoun->np_ : PrepSingleNounProd
;

grammar aboutTopicPhrase(main):
    ('o' | 'na' | ) topicPhrase->np_
    : PrepSingleTopicProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Complete noun phrase - this is a fully-qualified noun phrase that
 *   cannot be modified with articles, quantifiers, or anything else.  This
 *   is the highest-level individual noun phrase.  
 */

grammar completeNounPhrase(main):
    completeNounPhraseWithAll->np_ | completeNounPhraseWithoutAll->np_
    : LayeredNounPhraseProd
;

/*
 *   Slightly better than a purely miscellaneous word list is a pair of
 *   otherwise valid noun phrases connected by a preposition that's
 *   commonly used in command phrases.  This will match commands where the
 *   user has assumed a command with a prepositional structure that doesn't
 *   exist among the defined commands.  Since we have badness, we'll be
 *   ignored any time there's a valid command syntax with the same
 *   prepositional structure.
 */
grammar completeNounPhrase(miscPrep):
    [badness 100] completeNounPhrase->np1_
        ('s' | 'se' | 'do' | 'v' | 'skrz' | 'skrze' | 'pro' | 'na' | 'pod'
        | 'za')
        completeNounPhrase->np2_
    : NounPhraseProd
    resolveNouns(resolver, results)
    {
        /* note that we have an invalid prepositional phrase structure */
        results.noteBadPrep();

        /* resolve the underlying noun phrases, for scoring purposes */
        np1_.resolveNouns(resolver, results);
        np2_.resolveNouns(resolver, results);

        /* return nothing */
        return [];
    }
;


/*
 *   A qualified noun phrase can, all by itself, be a full noun phrase
 */
grammar completeNounPhraseWithoutAll(qualified): qualifiedNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   Pronoun rules.  A pronoun is a complete noun phrase; it does not allow
 *   further qualification.  
 *
 *   "Odemkni dveře. Otveři *je*.", "zeptej se *ho*", "zatáhni za *něj*"
 */
grammar completeNounPhraseWithoutAll(it): 
    'jeho' | 'něho' | 'ho' | 'jemu' | 'němu' | 'mu' | 'jej' | 'něj' | 'něm'
    | 'jím' | 'ním' : ItProd
;

grammar completeNounPhraseWithoutAll(them):
    'jich' | 'nich' | 'jim' | 'nim' | 'je' | 'ně' | 'nich' | 'jimi' | 'nimy'
    : ThemProd
;

grammar completeNounPhraseWithoutAll(him):
    'jeho' | 'něho' | 'ho' | 'jemu' | 'němu' | 'mu' | 'jej' | 'něj' | 'něm'
    | 'jím' | 'ním' : HimProd
;

grammar completeNounPhraseWithoutAll(her):
    'jí' | 'ní' | 'ji' | 'ni' : HerProd
;

/*
 *   Reflexive second-person pronoun, for things like "bob, look at
 *   yourself"
 */
grammar completeNounPhraseWithoutAll(yourself):
    'sebe' | 'sobě' | 'si' | 'se' | 'sebou' : YouProd
;

/*
 *   Reflexive third-person pronouns.  We accept these in places such as
 *   the indirect object of a two-object verb.
 */
grammar completeNounPhraseWithoutAll(itself): 'itself' : ItselfProd
    /* check agreement of our binding */
    checkAgreement(lst)
    {
        /* the result is required to be singular and ungendered */
        return (lst.length() == 1 && lst[1].obj_.canMatchIt);
    }
;

//grammar completeNounPhraseWithoutAll(themselves):
//    'themself' | 'themselves' : ThemselvesProd
//
//    /* check agreement of our binding */
//    checkAgreement(lst)
//    {
//        /*
//         *   For 'themselves', allow anything; we could balk at this
//         *   matching a single object that isn't a mass noun, but that
//         *   would be overly picky, and it would probably reject at least
//         *   a few things that really ought to be acceptable.  Besides,
//         *   'them' is the closest thing English has to a singular
//         *   gender-neutral pronoun, and some people intentionally use it
//         *   as such.
//         */
//        return true;
//    }
//;

/* "zeptej se ho na *něho*" */
grammar completeNounPhraseWithoutAll(himself): 'něj' | 'něho' : HimselfProd
    /* check agreement of our binding */
    checkAgreement(lst)
    {
        /* the result is required to be singular and masculine */
        return (lst.length() == 1 && lst[1].obj_.canMatchHim);
    }
;

grammar completeNounPhraseWithoutAll(herself): 'ni' : HerselfProd
    /* check agreement of our binding */
    checkAgreement(lst)
    {
        /* the result is required to be singular and feminine */
        return (lst.length() == 1 && lst[1].obj_.canMatchHer);
    }
;

/*
 *   First-person pronoun, for referring to the speaker: "bob, look at me"
 */
grammar completeNounPhraseWithoutAll(me): 'mě' | 'mi' : MeProd;

/*
 *   "All" and "all but".
 *   
 *   "All" is a "complete" noun phrase, because there's nothing else needed
 *   to make it a noun phrase.  We make this a special kind of complete
 *   noun phrase because 'all' is not acceptable as a complete noun phrase
 *   in some contexts where any of the other complete noun phrases are
 *   acceptable.
 *   
 *   "All but" is a "terminal" noun phrase - this is a special kind of
 *   complete noun phrase that cannot be followed by another noun phrase
 *   with "and".  "All but" is terminal because we want any and's that
 *   follow it to be part of the exception list, so that we interpret "take
 *   all but a and b" as excluding a and b, not as excluding a but then
 *   including b as a separate list.  
 */
grammar completeNounPhraseWithAll(main):
    'vše' | 'všechno' | 'všechen' | 'všeho' | 'vší' | 'všemu' | 'všechnu'
    | 'vším' | 'všechny' | 'všechna' | 'všech' | 'všem' | 'všechny' | 'všechna'
    | 'všemi'
    : EverythingProd
;

grammar terminalNounPhrase(allBut):
    ('vše' | 'všechno' | 'všechen' | 'všeho' | 'vší' | 'všemu' | 'všechnu'
    | 'vším' | 'všechny' | 'všechna' | 'všech' | 'všem' | 'všechny' | 'všechna'
    | 'všemi') ('kromě' | 'mimo' | 'bez' | 's' 'výjimkou')
        exceptList->except_
    : EverythingButProd
;

/*
 *   Plural phrase with an exclusion list.  This is a terminal noun phrase
 *   because it ends in an exclusion list.
 */
grammar terminalNounPhrase(pluralExcept):
    (qualifiedPluralNounPhrase->np_ | detPluralNounPhrase->np_)
    ('kromě' | 'mimo' | 'bez' | 's' 'výjimkou' | 'ale' ('bez' | 'ne'))
    exceptList->except_
    : ListButProd
;

/*
 *   Qualified singular with an exception
 */
grammar terminalNounPhrase(anyBut):
    ('kteréhokoliv' | 'kterýkoliv' | 'kteroukoliv' | 'kterékoliv' | 'kterákoliv'
    | 'jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv')
     nounPhrase->np_
    ('kromě' | 'mimo' | 'ale' ('bez' | 'ne')) exceptList->except_
    : IndefiniteNounButProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A qualified noun phrase is a noun phrase with an optional set of
 *   qualifiers: a definite or indefinite article, a quantifier, words such
 *   as 'any' and 'all', possessives, and locational specifiers ("the box
 *   on the table").
 *
 *   Without qualification, a definite article is implicit, so we read
 *   "take box" as equivalent to "take the box."
 *
 *   Grammar rule instantiations in language-specific modules should set
 *   property np_ to the underlying noun phrase match tree.
 */

/*
 *   A qualified noun phrase can be either singular or plural.  The number
 *   is a feature of the overall phrase; the phrase might consist of
 *   subphrases of different numbers (for example, "bob's coins" is plural
 *   even though it contains a singular subphrase, "bob"; and "one of the
 *   coins" is singular, even though its subphrase "coins" is plural).
 */
grammar qualifiedNounPhrase(main):
    qualifiedSingularNounPhrase->np_
    | qualifiedPluralNounPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Singular qualified noun phrase.
 */

/*
 *   A singular qualified noun phrase with an implicit or explicit definite
 *   article.  If there is no article, a definite article is implied (we
 *   interpret "take box" as though it were "take the box").
 */
grammar qualifiedSingularNounPhrase(definite):
    ('ten' | 'tu' | ) indetSingularNounPhrase->np_
    : DefiniteNounProd
;

/*
 *   A singular qualified noun phrase with an explicit indefinite article.
 */
grammar qualifiedSingularNounPhrase(indefinite):
    ('nějakého' | 'nějaký' | 'nějakou' | 'nějaké') indetSingularNounPhrase->np_
    : IndefiniteNounProd
;

/*
 *   A singular qualified noun phrase with an explicit arbitrary
 *   determiner.
 */
grammar qualifiedSingularNounPhrase(arbitrary):
    ('kteréhokoliv' | 'kterýkoliv' | 'kteroukoliv' | 'kterékoliv' | 'kterákoliv'
    | 'jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv')
    indetSingularNounPhrase->np_
    : ArbitraryNounProd
;

/*
 *   A singular qualified noun phrase with a possessive adjective.
 *
 *   "Pripoj *sondu unidipu* k modulu"
 */
grammar qualifiedSingularNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetSingularNounPhrase->np_
    | indetSingularNounPhrase->np_ possessiveAdjPhrase->poss_
    : PossessiveNounProd
;

/*
 *   A singular qualified noun phrase that arbitrarily selects from a
 *   plural set.  This is singular, even though the underlying noun phrase
 *   is plural, because we're explicitly selecting one item.
 */
grammar qualifiedSingularNounPhrase(anyPlural):
    ('kteréhokoliv' | 'kterýkoliv' | 'kteroukoliv' | 'kterékoliv' | 'kterákoliv'
    | 'jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv')
    ('z' | 'ze') explicitDetPluralNounPhrase->np_
    : ArbitraryNounProd
;

/*
 *   A singular object specified only by its containment, with a definite
 *   article.
 *
 *   "Který červený bonbón máš na mysli: červený bonbón na podlaze, nebo tvůj
 *   červený bonbón? >
 *
 *   Na rozdíl od angličtiny, která vyžaduje na začátku "the one", tak v češtině
 *   takové nadužívaní ukazovacích zájmen není obvyklé. Proto není potřeba zadat
 *   nic až na předložku a kontejner. Hodí se např.:
 *
 *      >p akumulátor
 *      A co tím „akumulátor“ máš na mysli? Vybitý akumulátor, nebo nabitý
 *      akumulátor?
 *      >ve svítilně
 *
 *
 */
grammar qualifiedSingularNounPhrase(theOneIn):
    ('ten'->mainPhraseText | 'ta'->mainPhraseText | 'to'->mainPhraseText | )
    ((',' | ) ('který' | 'která' | 'které' | 'co')
    ('je' | 'byl' | 'byla' | 'bylo' | 'leží' | 'ležel') | )
    ('v' | 've' | 'uvnitř' | 'na' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : VagueContainerDefiniteNounPhraseProd

    /*
     *   our main phrase is simply 'one' (so disambiguation prompts will
     *   read "which one do you mean...")
     *
     *   Frázi necháme prázdnou, aby proměnná určitě alespoň existovala. A když
     *   hráč použije ukazovací zájmeno, bude naplněno do této proměnné.
     */
    mainPhraseText = ''
;

/*
 *   A singular object specified only by its containment, with an
 *   indefinite article.
 */
grammar qualifiedSingularNounPhrase(anyOneIn):
    ('kteréhokoliv' | 'kterýkoliv' | 'kteroukoliv' | 'kterékoliv' | 'kterákoliv'
    | 'jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv')
    (('který' | 'která' | 'které') ('je' | 'byl' | 'byla' | 'bylo') | )
    ('v' | 've' | 'na' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : VagueContainerIndefiniteNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   An "indeterminate" singular noun phrase is a noun phrase without any
 *   determiner.  A determiner is a phrase that specifies the phrase's
 *   number and indicates whether or not it refers to a specific object,
 *   and if so fixes which object it refers to; determiners include
 *   articles ("the", "a") and possessives.
 *
 *   Note that an indeterminate phrase is NOT necessarily an indefinite
 *   phrase.  In fact, in most cases, we assume a definite usage when the
 *   determiner is omitted: we take TAKE BOX as meaning TAKE THE BOX.  This
 *   is more or less the natural way an English speaker would interpret
 *   this ill-formed phrasing, but even more than that, it's the
 *   Adventurese convention, taking into account that most players enter
 *   commands telegraphically and are accustomed to noun phrases being
 *   definite by default.
 */

/* an indetermine noun phrase can be a simple noun phrase */
grammar indetSingularNounPhrase(basic):
    nounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   An indetermine noun phrase can specify a location for the object(s).
 *   The location must be a singular noun phrase, but can itself be a fully
 *   qualified noun phrase (so it can have possessives, articles, and
 *   locational qualifiers of its own).
 *   
 *   Note that we take 'that are' even though the noun phrase is singular,
 *   because what we consider a singular noun phrase can have plural usage
 *   ("scissors", for example).  
 */
grammar indetSingularNounPhrase(locational):
    nounPhrase->np_
    (('který' | 'která' | 'které') ('je' | 'byl' | 'byla' | 'bylo')
     | ('kteří' | 'které' | 'která') ('jsou' | 'byli' | 'byly' | 'byla')
     | )
    ('v' | 've' | 'uvnitř' | 'na' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : ContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Plural qualified noun phrase.
 */

/*
 *   A simple unqualified plural phrase with determiner.  Since this form
 *   of plural phrase doesn't have any additional syntax that makes it an
 *   unambiguous plural, we can only accept an actual plural for the
 *   underlying phrase here - we can't accept an adjective phrase.
 */
grammar qualifiedPluralNounPhrase(determiner):
    ('kterýkoliv' | ) detPluralOnlyNounPhrase->np_
    : LayeredNounPhraseProd
;

/* plural phrase qualified with a number and optional "any" */
grammar qualifiedPluralNounPhrase(anyNum):
    ('kterýkoliv' | ) numberPhrase->quant_ indetPluralNounPhrase->np_
    | ('kterýkoliv' | ) numberPhrase->quant_ 'z' explicitDetPluralNounPhrase->np_
    : QuantifiedPluralProd
;

/* plural phrase qualified with a number and "all" */
grammar qualifiedPluralNounPhrase(allNum):
    ('všechny' | 'všechna' | 'všech') numberPhrase->quant_
    indetPluralNounPhrase->np_
    | ('všechny' | 'všechna' | 'všech') numberPhrase->quant_ ('z' | 'ze')
    explicitDetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

/* plural phrase qualified with "both" */
grammar qualifiedPluralNounPhrase(both):
    ('oba' ( | 'dva') | 'obě' ( | 'dvě') | 'obojí' ( | 'dvojí')
    | 'oboje' ( | 'dvoje')) detPluralNounPhrase->np_
    | ('oba' ( | 'dva') | 'obě' ( | 'dvě')| 'obojí' ( | 'dvojí')
    | 'oboje' ( | 'dvoje')) ('z' | 'ze') explicitDetPluralNounPhrase->np_
    : BothPluralProd
;

/* plural phrase qualified with "all" */
/* TODO: chtělo by to také slovo každý + jednotné číslo */
grammar qualifiedPluralNounPhrase(all):
    ('všechny' | 'všechna') detPluralNounPhrase->np_
    | ('všechny' | 'všechna' | 'vše') ('z' | 'ze')
    explicitDetPluralNounPhrase->np_
    : AllPluralProd
;

/* vague plural phrase with location specified */
grammar qualifiedPluralNounPhrase(theOnesIn):
    (('ti' | 'ty' | 'ta') ((',' | ) ('kteří' | 'které' | 'která' | 'co')
    ('jsou' | 'byly' | 'byli' | 'byla' | 'leží' | 'leželi' | 'ležely'
    | 'ležela') | )
    | ('všichni' | 'všechny' | 'všechna' | 'všechno' | 'vše') (('kteří'
    | 'které' | 'která' | 'co') ('jsou' | 'je' | 'byly' | 'byli' | 'byla'
    | 'bylo' | 'leží' | 'leželi' | 'ležely' | 'ležela') | ) )
    ('v' | 've' | 'uvnitř' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : AllInContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with a determiner.  The determiner can be
 *   explicit (such as an article or possessive) or it can implied (the
 *   implied determiner is the definite article, so "take boxes" is
 *   understood as "take the boxes").
 */
grammar detPluralNounPhrase(main):
    indetPluralNounPhrase->np_ | explicitDetPluralNounPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A determiner plural phrase with an explicit underlying plural (i.e.,
 *   excluding adjective phrases with no explicitly plural words).
 */
grammar detPluralOnlyNounPhrase(main):
    implicitDetPluralOnlyNounPhrase->np_
    | explicitDetPluralOnlyNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   An implicit determiner plural phrase is an indeterminate plural phrase
 *   without any extra determiner - i.e., the determiner is implicit.
 *   We'll treat this the same way we do a plural explicitly determined
 *   with a definite article, since this is the most natural interpretation
 *   in English.
 *
 *   (This might seem like a pointless extra layer in the grammar, but it's
 *   necessary for the resolution process to have a layer that explicitly
 *   declares the phrase to be determined, even though the determiner is
 *   implied in the structure.  This extra layer is important because it
 *   explicitly calls results.noteMatches(), which is needed for rankings
 *   and the like.)
 */
grammar implicitDetPluralOnlyNounPhrase(main):
    indetPluralOnlyNounPhrase->np_
    : DefinitePluralProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with an explicit determiner.
 */

/* a plural noun phrase with a definite article */
grammar explicitDetPluralNounPhrase(definite):
    indetPluralNounPhrase->np_
    : DefinitePluralProd
;

/* a plural noun phrase with a definite article and a number */
grammar explicitDetPluralNounPhrase(definiteNumber):
    numberPhrase->quant_ indetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

/* a plural noun phrase with a possessive */
grammar explicitDetPluralNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetPluralNounPhrase->np_
    : PossessivePluralProd
;

/* a plural noun phrase with a possessive and a number */
grammar explicitDetPluralNounPhrase(possessiveNumber):
    possessiveAdjPhrase->poss_ numberPhrase->quant_
    indetPluralNounPhrase->np_
    : ExactQuantifiedPossessivePluralProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with an explicit determiner and only an
 *   explicitly plural underlying phrase.
 */
grammar explicitDetPluralOnlyNounPhrase(definite):
    indetPluralOnlyNounPhrase->np_
    : AllPluralProd
;

grammar explicitDetPluralOnlyNounPhrase(definiteNumber):
    numberPhrase->quant_ indetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

grammar explicitDetPluralOnlyNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetPluralOnlyNounPhrase->np_
    : PossessivePluralProd
;

grammar explicitDetPluralOnlyNounPhrase(possessiveNumber):
    possessiveAdjPhrase->poss_ numberPhrase->quant_
    indetPluralNounPhrase->np_
    : ExactQuantifiedPossessivePluralProd
;


/* ------------------------------------------------------------------------ */
/*
 *   An indeterminate plural noun phrase.
 *
 *   For the basic indeterminate plural phrase, allow an adjective phrase
 *   anywhere a plural phrase is allowed; this makes possible the
 *   short-hand of omitting a plural word when the plural number is
 *   unambiguous from context.
 */

/* a simple plural noun phrase */
grammar indetPluralNounPhrase(basic):
    pluralPhrase->np_ | adjPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   A plural noun phrase with a locational qualifier.  Note that even
 *   though the overall phrase is plural (and so the main underlying noun
 *   phrase is plural), the location phrase itself must always be singular.
 */
grammar indetPluralNounPhrase(locational):
    (pluralPhrase->np_ | adjPhrase->np_) ((',' | ) ('kteří' | 'které' | 'která')
    ('jsou' | 'byli' | 'byly' | 'byla') | )
    ('v' | 've' | 'uvnitř' | 'na' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : ContainerNounPhraseProd
;

/*
 *   An indetermine plural noun phrase with only explicit plural phrases.
 */
grammar indetPluralOnlyNounPhrase(basic):
    pluralPhrase->np_
    : LayeredNounPhraseProd
;

grammar indetPluralOnlyNounPhrase(locational):
    pluralPhrase->np_ ((',' | ) ('kteří' | 'které' | 'která') ('jsou' | 'byli'
    | 'byly' | 'byla') | )
    ('v' | 've' | 'uvnitř' | 'na' | 'z' | 'ze')
    completeNounPhraseWithoutAll->cont_
    : ContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun Phrase.  This is the basic noun phrase, which serves as a
 *   building block for complete noun phrases.  This type of noun phrase
 *   can be qualified with articles, quantifiers, and possessives, and can
 *   be used to construct possessives via the addition of "'s" at the end
 *   of the phrase.
 *   
 *   In most cases, custom noun phrase rules should be added to this
 *   production, as long as qualification (with numbers, articles, and
 *   possessives) is allowed.  For a custom noun phrase rule that cannot be
 *   qualified, a completeNounPhrase rule should be added instead.  
 */
grammar nounPhrase(main): compoundNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   Plural phrase.  This is the basic plural phrase, and corresponds to
 *   the basic nounPhrase for plural forms.
 */
grammar pluralPhrase(main): compoundPluralPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Compound noun phrase.  This is one or more noun phrases connected with
 *   'of', as in "piece of paper".  The part after the 'of' is another
 *   compound noun phrase.
 *   
 *   Note that this general rule does not allow the noun phrase after the
 *   'of' to be qualified with an article or number, except that we make an
 *   exception to allow a definite article.  Other cases ("a piece of four
 *   papers") do not generally make sense, so we won't attempt to support
 *   them; instead, games can add as special cases new nounPhrase rules for
 *   specific literal sequences where more complex grammar is necessary.  
 */
grammar compoundNounPhrase(simple): simpleNounPhrase->np_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        return np_.getVocabMatchList(resolver, results, extraFlags);
    }
    getAdjustedTokens()
    {
        return np_.getAdjustedTokens();
    }
;

/*
 *   Protože na rozdíl od angličtiny, kde jsou slova oddělena slovem "of",
 *   v češtině je vyjádřeno jen druhým pádem druhého slova, tak za druhé slovo
 *   dáme simple* místo compound*, aby nevznikla smyčka v parseru.
 */
grammar compoundNounPhrase(of):
    simpleNounPhrase->np1_ simpleNounPhrase->np2_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local lst1;
        local lst2;

        /* resolve the two underlying lists */
        lst1 = np1_.getVocabMatchList(resolver, results, extraFlags);
        lst2 = np2_.getVocabMatchList(resolver, results, extraFlags);

        /*
         *   the result is the intersection of the two lists, since we
         *   want the list of objects with all of the underlying
         *   vocabulary words
         */
        return intersectNounLists(lst1, lst2);
    }
    getAdjustedTokens()
    {
        /* return the full list */
        return np1_.getAdjustedTokens() + np2_.getAdjustedTokens();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Compound plural phrase - same as a compound noun phrase, but involving
 *   a plural part before the 'of'.
 */

/*
 *   just a single plural phrase
 */
grammar compoundPluralPhrase(simple): simplePluralPhrase->np_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        return np_.getVocabMatchList(resolver, results, extraFlags);
    }
    getAdjustedTokens()
    {
        return np_.getAdjustedTokens();
    }
;

/*
 *   <plural-phrase> of <noun-phrase>
 *
 *   Na rozdíl od angličtiny bereme jako druhé slovo i plurál, např.
 *   "prozkoumej kryty kójí". A viz výše simple* místo compound*.
 */
grammar compoundPluralPhrase(of):
    simplePluralPhrase->np1_ simplePluralPhrase->np2_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local lst1;
        local lst2;

        /* resolve the two underlying lists */
        lst1 = np1_.getVocabMatchList(resolver, results, extraFlags);
        lst2 = np2_.getVocabMatchList(resolver, results, extraFlags);

        /*
         *   the result is the intersection of the two lists, since we
         *   want the list of objects with all of the underlying
         *   vocabulary words
         */
        return intersectNounLists(lst1, lst2);
    }
    getAdjustedTokens()
    {
        /* return the full list */
        return np1_.getAdjustedTokens() + np2_.getAdjustedTokens();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Simple noun phrase.  This is the most basic noun phrase, which is
 *   simply a noun, optionally preceded by one or more adjectives.
 */

/*
 *   just a noun
 */
grammar simpleNounPhrase(noun): nounWord->noun_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        return noun_.getVocabMatchList(resolver, results, extraFlags);
    }
    getAdjustedTokens()
    {
        return noun_.getAdjustedTokens();
    }
;

/*
 *   <adjective> <simple-noun-phrase> (this allows any number of adjectives
 *   to be applied) 
 */
grammar simpleNounPhrase(adjNP): adjWord->adj_ simpleNounPhrase->np_
    : NounPhraseWithVocab

    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying noun phrase
         */
        return intersectNounLists(
            adj_.getVocabMatchList(resolver, results, extraFlags),
            np_.getVocabMatchList(resolver, results, extraFlags));
    }
    getAdjustedTokens()
    {
        return adj_.getAdjustedTokens() + np_.getAdjustedTokens();
    }
;

/*
 *   A simple noun phrase can also include a number or a quoted string
 *   before or after a noun.  A number can be spelled out or written with
 *   numerals; we consider both forms equivalent in meaning.
 *   
 *   A number in this type of usage is grammatically equivalent to an
 *   adjective - it's not meant to quantify the rest of the noun phrase,
 *   but rather is simply an adjective-like modifier.  For example, an
 *   elevator's control panel might have a set of numbered buttons which we
 *   want to refer to as "button 1," "button 2," and so on.  It is
 *   frequently the case that numeric adjectives are equally at home before
 *   or after their noun: "push 3 button" or "push button 3".  In addition,
 *   we accept a number by itself as a lone adjective, as in "push 3".  
 */

/*
 *   just a numeric/string adjective (for things like "push 3", "push #3",
 *   'push "G"')
 */
grammar simpleNounPhrase(number): literalAdjPhrase->adj_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   note that this counts as an adjective-ending phrase, since we
         *   don't have a noun involved
         */
        results.noteAdjEnding();

        /* pass through to the underlying literal adjective phrase */
        local lst = adj_.getVocabMatchList(resolver, results,
                                           extraFlags | EndsWithAdj);

        /* if in global scope, also try a noun interpretation */
        if (resolver.isGlobalScope)
            lst = adj_.addNounMatchList(lst, resolver, results, extraFlags);

        /* return the result */
        return lst;
    }
    getAdjustedTokens()
    {
        /* pass through to the underlying literal adjective phrase */
        return adj_.getAdjustedTokens();
    }
;

/*
 *   <literal-adjective> <noun> (for things like "board 44 bus" or 'push
 *   "G" button')
 */
grammar simpleNounPhrase(numberAndNoun):
    literalAdjPhrase->adj_ nounWord->noun_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local nounList;
        local adjList;

        /* get the list of objects matching the rest of the noun phrase */
        nounList = noun_.getVocabMatchList(resolver, results, extraFlags);

        /* get the list of objects matching the literal adjective */
        adjList = adj_.getVocabMatchList(resolver, results, extraFlags);

        /* intersect the two lists and return the results */
        return intersectNounLists(nounList, adjList);
    }
    getAdjustedTokens()
    {
        return adj_.getAdjustedTokens() + noun_.getAdjustedTokens();
    }
;

/*
 *   <noun> <literal-adjective> (for things like "press button 3" or 'put
 *   tab "A" in slot "B"')
 */
grammar simpleNounPhrase(nounAndNumber):
    nounWord->noun_ literalAdjPhrase->adj_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local nounList;
        local adjList;

        /* get the list of objects matching the rest of the noun phrase */
        nounList = noun_.getVocabMatchList(resolver, results, extraFlags);

        /* get the literal adjective matches */
        adjList = adj_.getVocabMatchList(resolver, results, extraFlags);

        /* intersect the two lists and return the results */
        return intersectNounLists(nounList, adjList);
    }
    getAdjustedTokens()
    {
        return noun_.getAdjustedTokens() + adj_.getAdjustedTokens();
    }
;

/*
 *   A simple noun phrase can also end in an adjective, which allows
 *   players to refer to objects using only their unique adjectives rather
 *   than their full names, which is sometimes more convenient: just "take
 *   gold" rather than "take gold key."
 *   
 *   When a particular phrase can be interpreted as either ending in an
 *   adjective or ending in a noun, we will always take the noun-ending
 *   interpretation - in such cases, the adjective-ending interpretation is
 *   probably a weak binding.  For example, "take pizza" almost certainly
 *   refers to the pizza itself when "pizza" and "pizza box" are both
 *   present, but it can also refer just to the box when no pizza is
 *   present.
 *   
 *   Equivalent to a noun phrase ending in an adjective is a noun phrase
 *   ending with an adjective followed by "one," as in "the red one."  
 */
grammar simpleNounPhrase(adj): adjWord->adj_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* note in the results that we end in an adjective */
        results.noteAdjEnding();

        /* generate a list of objects matching the adjective */
        local lst = adj_.getVocabMatchList(
            resolver, results, extraFlags | EndsWithAdj);

        /* if in global scope, also try a noun interpretation */
        if (resolver.isGlobalScope)
            lst = adj_.addNounMatchList(lst, resolver, results, extraFlags);

        /* return the result */
        return lst;
    }
    getAdjustedTokens()
    {
        /* return the adjusted token list for the adjective */
        return adj_.getAdjustedTokens();
    }
;

grammar simpleNounPhrase(adjAndOne): adjective->adj_ 'one'
    : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   This isn't exactly an adjective ending, but consider it as
         *   such anyway, since we're not matching 'one' to a vocabulary
         *   word - we're just using it as a grammatical marker that we're
         *   not providing a real noun.  If there's another match for
         *   which 'one' is a noun, that one is definitely preferred to
         *   this one; the adj-ending marking will ensure that we choose
         *   the other one.
         */
        results.noteAdjEnding();

        /* generate a list of objects matching the adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              extraFlags | EndsWithAdj, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

/*
 *   In the worst case, a simple noun phrase can be constructed from
 *   arbitrary words that don't appear in our dictionary.
 */
grammar simpleNounPhrase(misc):
    [badness 200] miscWordList->lst_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* get the match list from the underlying list */
        local lst = lst_.getVocabMatchList(resolver, results, extraFlags);

        /*
         *   If there are no matches, note in the results that we have an
         *   arbitrary word list.  Note that we do this only if there are
         *   no matches, because we might match non-dictionary words to an
         *   object with a wildcard in its vocabulary words, in which case
         *   this is a valid, matching phrase after all.
         */
        if (lst == nil || lst.length() == 0)
            results.noteMiscWordList(lst_.getOrigText());

        /* return the match list */
        return lst;
    }
    getAdjustedTokens()
    {
        return lst_.getAdjustedTokens();
    }
;

/*
 *   If the command has qualifiers but omits everything else, we can have
 *   an empty simple noun phrase.
 */
grammar simpleNounPhrase(empty): [badness 600] : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* we have an empty noun phrase */
        return results.emptyNounPhrase(resolver);
    }
    getAdjustedTokens()
    {
        return [];
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Tady se snažím zařídit, aby jako přivlastňovací zájmeno mohlo vystupovat
 *   jen zájmeno s vlastností adjApostS.
 */
grammar simplePossessivePhrase(adj): possAdjWord->adj_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* note in the results that we end in an adjective */
        results.noteAdjEnding();

        /* generate a list of objects matching the adjective */
        local lst = adj_.getVocabMatchList(
            resolver, results, extraFlags | EndsWithAdj);

        /* if in global scope, also try a noun interpretation */
        if (resolver.isGlobalScope)
            lst = adj_.addNounMatchList(lst, resolver, results, extraFlags);

        /* return the result */
        return lst;
    }
    getAdjustedTokens()
    {
        /* return the adjusted token list for the adjective */
        return adj_.getAdjustedTokens();
    }
;

grammar possAdjWord(adjApostS): adjApostS->adj_
    : AdjPhraseWithVocab
    /* generate a list of resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* return a list of objects in scope matching our adjective */
        return getWordMatches(adj_, &adjApostS, resolver,
                              extraFlags, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjApostS];
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   An AdjPhraseWithVocab is an English-specific subclass of
 *   NounPhraseWithVocab, specifically for noun phrases that contain
 *   entirely adjectives.  
 */
class AdjPhraseWithVocab: NounPhraseWithVocab
    /* the property for the adjective literal - this is usually adj_ */
    adjVocabProp = &adj_

    /* 
     *   Add the vocabulary matches that we'd get if we were treating our
     *   adjective as a noun.  This combines the noun interpretation with a
     *   list of matches we got for the adjective version.  
     */
    addNounMatchList(lst, resolver, results, extraFlags)
    {
        /* get the word matches with a noun interpretation of our adjective */
        local nLst = getWordMatches(
            self.(adjVocabProp), &noun, resolver, extraFlags, VocabTruncated);

        /* combine the lists and return the result */
        return combineWordMatches(lst, nLst);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A "literal adjective" phrase is a number or string used as an
 *   adjective.
 */
grammar literalAdjPhrase(number):
    numberPhrase->num_ | poundNumberPhrase->num_
    : AdjPhraseWithVocab

    adj_ = (num_.getStrVal())
    getVocabMatchList(resolver, results, extraFlags)
    {
        local numList;

        /*
         *   get the list of objects matching the numeral form of the
         *   number as an adjective
         */
        numList = getWordMatches(num_.getStrVal(), &adjective,
                                 resolver, extraFlags, VocabTruncated);

        /* add the list of objects matching the special '#' wildcard */
        numList += getWordMatches('#', &adjective, resolver,
                                  extraFlags, VocabTruncated);

        /* return the combined lists */
        return numList;
    }
    getAdjustedTokens()
    {
        return [num_.getStrVal(), &adjective];
    }
;

grammar literalAdjPhrase(string): quotedStringPhrase->str_
    : AdjPhraseWithVocab

    adj_ = (str_.getStringText().toLower())
    getVocabMatchList(resolver, results, extraFlags)
    {
        local strList;
        local wLst;

        /*
         *   get the list of objects matching the string with the quotes
         *   removed
         */
        strList = getWordMatches(str_.getStringText().toLower(),
                                 &literalAdjective,
                                 resolver, extraFlags, VocabTruncated);

        /* add the list of objects matching the literal-adjective wildcard */
        wLst = getWordMatches('\u0001', &literalAdjective, resolver,
                              extraFlags, VocabTruncated);
        strList = combineWordMatches(strList, wLst);

        /* return the combined lists */
        return strList;
    }
    getAdjustedTokens()
    {
        return [str_.getStringText().toLower(), &adjective];
    }
;

/*
 *   In many cases, we might want to write what is semantically a literal
 *   string qualifier without the quotes.  For example, we might want to
 *   refer to an elevator button that's labeled "G" as simply "button G",
 *   without any quotes around the "G".  To accommodate these cases, we
 *   provide the literalAdjective part-of-speech.  We'll match these parts
 *   of speech the same way we'd match them if they were quoted.
 */
grammar literalAdjPhrase(literalAdj): literalAdjective->adj_
    : AdjPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local lst;

        /* get a list of objects in scope matching our literal adjective */
        lst = getWordMatches(adj_, &literalAdjective, resolver,
                             extraFlags, VocabTruncated);

        /* if the scope is global, also include ordinary adjective matches */
        if (resolver.isGlobalScope)
        {
            /* get the ordinary adjective bindings */
            local aLst = getWordMatches(adj_, &adjective, resolver,
                                        extraFlags, VocabTruncated);

            /* global scope - combine the lists */
            lst = combineWordMatches(lst, aLst);
        }

        /* return the result */
        return lst;
    }
    getAdjustedTokens()
    {
        return [adj_, &literalAdjective];
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   A noun word.  This can be either a simple 'noun' vocabulary word, or
 *   it can be an abbreviated noun with a trailing abbreviation period.
 */
class NounWordProd: NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        local w;
        local nLst;

        /* get our word text */
        w = getNounText();

        /* get the list of matches as nouns */
        nLst = getWordMatches(w, &noun, resolver, extraFlags, VocabTruncated);

        /*
         *   If the resolver indicates that we're in a "global" scope,
         *   *also* include any additional matches as adjectives.
         *
         *   Normally, when we're operating in limited, local scopes, we
         *   use the structure of the phrasing to determine whether to
         *   match a noun or adjective; if we have a match for a given word
         *   as a noun, we'll treat it only as a noun.  This allows us to
         *   take PIZZA to refer to the pizza (for which 'pizza' is defined
         *   as a noun) rather than to the PIZZA BOX (for which 'pizza' is
         *   a mere adjective) when both are in scope.  It's obvious which
         *   the player means in such cases, so we can be smart about
         *   choosing the stronger match.
         *
         *   In cases of global scope, though, it's much harder to guess
         *   about the player's intentions.  When the player types PIZZA,
         *   they might be thinking of the box even though there's a pizza
         *   somewhere else in the game.  Since the two objects might be in
         *   entirely different locations, both out of view, we can't
         *   assume that one or the other is more likely on the basis of
         *   which is closer to the player's senses.  So, it's better to
         *   allow both to match for now, and decide later, based on the
         *   context of the command, which was actually meant.
         */
        if (resolver.isGlobalScope)
        {
            /* get the list of matching adjectives */
            local aLst = getWordMatches(w, &adjective, resolver,
                                        extraFlags, VocabTruncated);

            /* combine it with the noun list */
            nLst = combineWordMatches(nLst, aLst);
        }

        /* return the match list */
        return nLst;
    }
    getAdjustedTokens()
    {
        /* the noun includes the period as part of the literal text */
        return [getNounText(), &noun];
    }

    /* the actual text of the noun to match to the dictionary */
    getNounText() { return noun_; }
;

grammar nounWord(noun): noun->noun_ : NounWordProd
;

grammar nounWord(nounAbbr): noun->noun_ tokAbbrPeriod->period_
    : NounWordProd

    /*
     *   for dictionary matching purposes, include the text of our noun
     *   with the period attached - the period is part of the dictionary
     *   entry for an abbreviated word
     */
    getNounText() { return noun_ + period_; }
;

/* ------------------------------------------------------------------------ */
/*
 *   An adjective word.  This can be either a simple 'adjective' vocabulary
 *   word, or it can be an 'adjApostS' vocabulary word plus a 's token.  
 */
grammar adjWord(adj): adjective->adj_ : AdjPhraseWithVocab
    /* generate a list of resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* return a list of objects in scope matching our adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              extraFlags, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

/*
 *   Toto vyloučíme, aby přivlastňovací přídavné jméno nemohlo vystupovat v roli
 *   obyčejného zájmena. Na rozdíl od angličtiny totiž je přivlastňovací použití
 *   dané čistě jen tvarem slova a ne použitím 's za obyčejným přídavným jménem.
 */
//grammar adjWord(adjApostS): adjApostS->adj_
//    : AdjPhraseWithVocab
//    /* generate a list of resolved objects */
//    getVocabMatchList(resolver, results, extraFlags)
//    {
//        /* return a list of objects in scope matching our adjective */
//        return getWordMatches(adj_, &adjApostS, resolver,
//                              extraFlags, VocabTruncated);
//    }
//    getAdjustedTokens()
//    {
//        return [adj_, &adjApostS];
//    }
//;

grammar adjWord(adjAbbr): adjective->adj_ tokAbbrPeriod->period_
    : AdjPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   return the list matching our adjective *with* the period
         *   attached; the period is part of the dictionary entry for an
         *   abbreviated word
         */
        return getWordMatches(adj_ + period_, &adjective, resolver,
                              extraFlags, VocabTruncated);
    }
    getAdjustedTokens()
    {
        /* the adjective includes the period as part of the literal text */
        return [adj_ + period_, &adjective];
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Possessive phrase.  This is a noun phrase expressing ownership of
 *   another object.
 *   
 *   Note that all possessive phrases that can possibly be ambiguous must
 *   define getOrigMainText() to return the "main noun phrase" text.  In
 *   English, this means that we must omit any "'s" suffix.  This is needed
 *   only when the phrase can be ambiguous, so pronouns don't need it since
 *   they are inherently unambiguous.  
 *
 *   TADS má možnost definovat vlastnictví předmětu a i když ho momentálně
 *   vlastník nedrží, může se na předmět odvolávat jako na svůj. Problém tu
 *   nastává v tom, že se musí správně zacházet se zájmeny můj vs. svůj:
 *
 *   vem mého kocoura (kocour patřící hráči, který se personalizuje s PC)
 *   vem svého kocoura (kocour patřící PC)
 *   sally, vem mého kocoura (kocour patřící PC)
 *   sally, vem svého kocoura (kocour patřící sally)
 */
grammar possessiveAdjPhrase(its): 'jeho' : ItsAdjProd
    /* we only agree with a singular ungendered noun */
    checkAnaphorAgreement(lst)
        { return lst.length() == 1 && lst[1].obj_.canMatchIt; }
;
grammar possessiveAdjPhrase(his): 'jeho' : HisAdjProd
    /* we only agree with a singular masculine noun */
    checkAnaphorAgreement(lst)
        { return lst.length() == 1 && lst[1].obj_.canMatchHim; }
;
grammar possessiveAdjPhrase(her): 'její' : HerAdjProd
    /* we only agree with a singular feminine noun */
    checkAnaphorAgreement(lst)
        { return lst.length() == 1 && lst[1].obj_.canMatchHer; }
;
grammar possessiveAdjPhrase(their): 'jejich' : TheirAdjProd
    /* we only agree with a single noun that has plural usage */
    checkAnaphorAgreement(lst)
        { return lst.length() == 1 && lst[1].obj_.isPlural; }
;

grammar possessiveAdjPhrase(your):
    'tvůj' | 'tvoje' | 'tvá' | 'tvé' | 'tvého' | 'tvojí' | 'tvému' | 'tvoji'
    | 'tvou' | 'tvém' | 'tvým' | 'tví' | 'tvých' | 'tvými' | 'svůj' | 'svoje'
    | 'svá' | 'své' | 'svého' | 'svojí' | 'svému' | 'svoji' | 'svou' | 'svém'
    | 'svým' | 'sví' | 'svých' | 'svými' : YourAdjProd
    /* we are non-anaphoric */
    checkAnaphorAgreement(lst) { return nil; }
;

grammar possessiveAdjPhrase(my):
    'můj' | 'má' | 'moje' | 'mé' | 'mého' | 'mojí' | 'mému' | 'mou' | 'moji'
    | 'mém' | 'mým' : MyAdjProd
    /* we are non-anaphoric */
    checkAnaphorAgreement(lst) { return nil; }
;

/*
 *   Angličtina zde má nounPhrase->np_ tokApostropheS->apost_, tedy jakákoliv
 *   nounPhrase, která je zakončená apostrofem S, tak se bere jako
 *   přivastňovací. V češtině takovouto symboliku nemáme a jako přivlastňovací
 *   přídavné jméno musíme brát jen specificky vyjmenované tvary slov ve
 *   slovníku zařazené jako adjApostS.
 */
grammar possessiveAdjPhrase(npApostropheS):
    simplePossessivePhrase->np_ : LayeredNounPhraseProd

    /* get the original text without the "'s" suffix */
    getOrigMainText()
    {
        /* return just the basic noun phrase part */
        return np_.getOrigText();
    }
;

grammar possessiveAdjPhrase(ppApostropheS):
    simplePossessivePhrase->np_
    : LayeredNounPhraseProd

    /* get the original text without the "'s" suffix */
    getOrigMainText()
    {
        /* return just the basic noun phrase part */
        return np_.getOrigText();
    }

    resolveNouns(resolver, results)
    {
        /* note that we have a plural phrase, structurally speaking */
        results.notePlural();

        /* inherit the default handling */
        return inherited(resolver, results);
    }

    /* the possessive phrase is plural */
    isPluralPossessive = true
;

/*
 *   Possessive noun phrases.  These are similar to possessive phrases, but
 *   are stand-alone phrases that can act as nouns rather than as
 *   qualifiers for other noun phrases.  For example, for a first-person
 *   player character, "mine" would be a possessive noun phrase referring
 *   to an object owned by the player character.
 *   
 *   Note that many of the words used for possessive nouns are the same as
 *   for possessive adjectives - for example "his" is the same in either
 *   case, as are "'s" words.  However, we make the distinction internally
 *   because the two have different grammatical uses, and some of the words
 *   do differ ("her" vs "hers", for example).  
 */
grammar possessiveNounPhrase(its): 'jeho': ItsNounProd;
grammar possessiveNounPhrase(his): 'jeho': HisNounProd;
grammar possessiveNounPhrase(hers): 'její': HersNounProd;
grammar possessiveNounPhrase(theirs): 'jejich': TheirsNounProd;

grammar possessiveNounPhrase(yours):
    'tvůj' | 'tvoje' | 'tvá' | 'tvé' | 'tvého' | 'tvojí' | 'tvému' | 'tvoji'
    | 'tvou' | 'tvém' | 'tvým' | 'tví' | 'tvých' | 'tvými' | 'svůj' | 'svoje'
    | 'svá' | 'své' | 'svého' | 'svojí' | 'svému' | 'svoji' | 'svou' | 'svém'
    | 'svým' | 'sví' | 'svých' | 'svými' : YoursNounProd;

grammar possessiveNounPhrase(mine):
    'můj' | 'má' | 'moje' | 'mé' | 'mého' | 'mojí' | 'mému' | 'mou' | 'moji'
    | 'mém' | 'mým' : MineNounProd;

grammar possessiveNounPhrase(npApostropheS):
    simplePossessivePhrase->np_ /*| pluralPhrase->np_*/
    : LayeredNounPhraseProd

    /* get the original text without the "'s" suffix */
    getOrigMainText()
    {
        /* return just the basic noun phrase part */
        return np_.getOrigText();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Simple plural phrase.  This is the most basic plural phrase, which is
 *   simply a plural noun, optionally preceded by one or more adjectives.
 *   
 *   (English doesn't have any sort of adjective declension in number, so
 *   there's no need to distinguish between plural and singular adjectives;
 *   this equivalent rule in languages with adjective-noun agreement in
 *   number would use plural adjectives here as well as plural nouns.)  
 */
grammar simplePluralPhrase(plural): plural->plural_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        local lst;

        /* get the list of matching plurals */
        lst = getWordMatches(plural_, &plural, resolver,
                             extraFlags, PluralTruncated);

        /* get the list of matching 'noun' definitions */
        local nLst = getWordMatches(plural_, &noun, resolver,
                                    extraFlags, VocabTruncated);

        /* get the combined list */
        local comboLst = combineWordMatches(lst, nLst);

        /*
         *   If we're in global scope, add in the matches for just plain
         *   'noun' properties as well.  This is important because we'll
         *   sometimes want to define a word that's actually a plural
         *   usage (in terms of the real-world English) under the 'noun'
         *   property.  This occurs particularly when a single game-world
         *   object represents a multiplicity of real-world objects.  When
         *   the scope is global, it's hard to anticipate all of the
         *   possible interactions with vocabulary along these lines, so
         *   it's easiest just to include the 'noun' matches.
         */
        if (resolver.isGlobalScope)
        {
            /* keep the combined list */
            lst = comboLst;
        }
        else if (comboLst.length() > lst.length())
        {
            /*
             *   ordinary scope, so don't include the noun matches; but
             *   since we found extra items to add, at least mark the
             *   plural matches as potentially ambiguous
             */
            lst.forEach({x: x.flags_ |= UnclearDisambig});
        }

        /* return the result list */
        return lst;
    }
    getAdjustedTokens()
    {
        return [plural_, &plural];
    }
;

grammar simplePluralPhrase(adj): adjWord->adj_ simplePluralPhrase->np_ :
    NounPhraseWithVocab

    /* resolve my object list */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying noun phrase
         */
        return intersectNounLists(
            adj_.getVocabMatchList(resolver, results, extraFlags),
            np_.getVocabMatchList(resolver, results, extraFlags));
    }
    getAdjustedTokens()
    {
        return adj_.getAdjustedTokens() + np_.getAdjustedTokens();
    }
;

grammar simplePluralPhrase(poundNum):
    poundNumberPhrase->num_ simplePluralPhrase->np_
    : NounPhraseWithVocab

    /* resolve my object list */
    getVocabMatchList(resolver, results, extraFlags)
    {
        local baseList;
        local numList;

        /* get the base list for the rest of the phrase */
        baseList = np_.getVocabMatchList(resolver, results, extraFlags);

        /* get the numeric matches, including numeric wildcards */
        numList = getWordMatches(num_.getStrVal(), &adjective,
                                 resolver, extraFlags, VocabTruncated)
                  + getWordMatches('#', &adjective,
                                   resolver, extraFlags, VocabTruncated);

        /* return the intersection of the lists */
        return intersectNounLists(numList, baseList);
    }
    getAdjustedTokens()
    {
        return [num_.getStrVal(), &adjective] + np_.getAdjustedTokens();
    }
;

/*
 *   A simple plural phrase can end with an adjective and "ones," as in
 *   "the red ones."
 */
grammar simplePluralPhrase(adjAndOnes): ('ti' | 'ty' | 'ta') adjective->adj_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* generate a list of objects matching the adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              extraFlags | EndsWithAdj, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

/*
 *   If the command has qualifiers that require a plural, but omits
 *   everything else, we can have an empty simple noun phrase.
 */
grammar simplePluralPhrase(empty): [badness 600] : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* we have an empty noun phrase */
        return results.emptyNounPhrase(resolver);
    }
    getAdjustedTokens()
    {
        return [];
    }
;

/*
 *   A simple plural phrase can match unknown words as a last resort.
 */
grammar simplePluralPhrase(misc):
    [badness 300] miscWordList->lst_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* get the match list from the underlying list */
        local lst = lst_.getVocabMatchList(resolver, results, extraFlags);

        /*
         *   if there are no matches, note in the results that we have an
         *   arbitrary word list that doesn't correspond to any object
         */
        if (lst == nil || lst.length() == 0)
            results.noteMiscWordList(lst_.getOrigText());

        /* return the vocabulary match list */
        return lst;
    }
    getAdjustedTokens()
    {
        return lst_.getAdjustedTokens();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   An "adjective phrase" is a phrase made entirely of adjectives.
 */
grammar adjPhrase(adj): adjective->adj_ : AdjPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* note the adjective ending */
        results.noteAdjEnding();

        /* return the match list */
        local lst = getWordMatches(adj_, &adjective, resolver,
                                   extraFlags | EndsWithAdj, VocabTruncated);

        /* if in global scope, also try a noun interpretation */
        if (resolver.isGlobalScope)
            lst = addNounMatchList(lst, resolver, results, extraFlags);

        /* return the result */
        return lst;
    }

    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

grammar adjPhrase(adjAdj): adjective->adj_ adjPhrase->ap_
    : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results, extraFlags)
    {
        /*
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying adjective phrase
         */
        return intersectWordMatches(
            adj_, &adjective, resolver, extraFlags, VocabTruncated,
            ap_.getVocabMatchList(resolver, results, extraFlags));
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective] + ap_.getAdjustedTokens();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   A "topic" is a special type of noun phrase used in commands like "ask
 *   <actor> about <topic>."  We define a topic as simply an ordinary
 *   single-noun phrase.  We distinguish this in the grammar to allow games
 *   to add special syntax for these.  
 */
grammar topicPhrase(main): singleNoun->np_ : TopicProd
;

/*
 *   Explicitly match a miscellaneous word list as a topic.
 *
 *   This might seem redundant with the ordinary topicPhrase that accepts a
 *   singleNoun, because singleNoun can match a miscellaneous word list.
 *   The difference is that singleNoun only matches a miscWordList with a
 *   "badness" value, whereas we match a miscWordList here without any
 *   badness.  We want to be more tolerant of unrecognized input in topic
 *   phrases than in ordinary noun phrases, because it's in the nature of
 *   topic phrases to go outside of what's implemented directly in the
 *   simulation model.  At a grammatical level, we don't want to treat
 *   topic phrases that we can resolve to the simulation model any
 *   differently than we treat those we can't resolve, so we must add this
 *   rule to eliminate the badness that singleNoun associated with a
 *   miscWordList match.
 *
 *   Note that we do prefer resolvable noun phrase matches to miscWordList
 *   matches, but we handle this preference with the resolver's scoring
 *   mechanism rather than with badness.
 */
grammar topicPhrase(misc): miscWordList->np_ : TopicProd
   resolveNouns(resolver, results)
   {
       /* note in the results that we have an arbitrary word list */
       results.noteMiscWordList(np_.getOrigText());

       /* inherit the default TopicProd behavior */
       return inherited(resolver, results);
   }
;

/* ------------------------------------------------------------------------ */
/*
 *   A "quoted string" phrase is a literal enclosed in single or double
 *   quotes.
 *
 *   Note that this is a separate production from literalPhrase.  This
 *   production can be used when *only* a quoted string is allowed.  The
 *   literalPhrase production allows both quoted and unquoted text.
 */
grammar quotedStringPhrase(main): tokString->str_ : LiteralProd
    /*
     *   get my string, with the quotes trimmed off (so we return simply
     *   the contents of the string)
     */
    getStringText() { return stripQuotesFrom(str_); }
;

/*
 *   Service routine: strip quotes from a *possibly* quoted string.  If the
 *   string starts with a quote, we'll remove the open quote.  If it starts
 *   with a quote and it ends with a corresponding close quote, we'll
 *   remove that as well.  
 */
stripQuotesFrom(str)
{
    local hasOpen;
    local hasClose;

    /* presume we won't find open or close quotes */
    hasOpen = hasClose = nil;

    /*
     *   Check for quotes.  We'll accept regular ASCII "straight" single
     *   or double quotes, as well as Latin-1 curly single or double
     *   quotes.  The curly quotes must be used in their normal
     */
    if (str.startsWith('\'') || str.startsWith('"'))
    {
        /* single or double quote - check for a matching close quote */
        hasOpen = true;
        hasClose = (str.length() > 2 && str.endsWith(str.substr(1, 1)));
    }
    else if (str.startsWith('`'))
    {
        /* single in-slanted quote - check for either type of close */
        hasOpen = true;
        hasClose = (str.length() > 2
                    && (str.endsWith('`') || str.endsWith('\'')));
    }
    else if (str.startsWith('\u201C'))
    {
        /* it's a curly double quote */
        hasOpen = true;
        hasClose = str.endsWith('\u201D');
    }
    else if (str.startsWith('\u2018'))
    {
        /* it's a curly single quote */
        hasOpen = true;
        hasClose = str.endsWith('\u2019');
    }

    /* trim off the quotes */
    if (hasOpen)
    {
        if (hasClose)
            str = str.substr(2, str.length() - 2);
        else
            str = str.substr(2);
    }

    /* return the modified text */
    return str;
}

/* ------------------------------------------------------------------------ */
/*
 *   A "literal" is essentially any phrase.  This can include a quoted
 *   string, a number, or any set of word tokens.
 */
grammar literalPhrase(string): quotedStringPhrase->str_ : LiteralProd
    getLiteralText(results, action, which)
    {
        /* get the text from our underlying quoted string */
        return str_.getStringText();
    }

    getTentativeLiteralText()
    {
        /*
         *   our result will never change, so our tentative text is the
         *   same as our regular literal text
         */
        return str_.getStringText();
    }

    resolveLiteral(results)
    {
        /* flag the literal text */
        results.noteLiteral(str_.getOrigText());
    }
;

grammar literalPhrase(miscList): miscWordList->misc_ : LiteralProd
    getLiteralText(results, action, which)
    {
        /* get my original text */
        local txt = misc_.getOrigText();

        /*
         *   if our underlying miscWordList has only one token, strip
         *   quotes, in case that token is a quoted string token
         */
        if (misc_.getOrigTokenList().length() == 1)
            txt = stripQuotesFrom(txt);

        /* return the text */
        return txt;
    }

    getTentativeLiteralText()
    {
        /* our regular text is permanent, so simply use it now */
        return misc_.getOrigText();
    }

    resolveLiteral(results)
    {
        /*
         *   note the length of our literal phrase - when we have a choice
         *   of interpretations, we prefer to choose shorter literal
         *   phrases, since this means that we'll have more of our tokens
         *   being fully interpreted rather than bunched into an
         *   uninterpreted literal
         */
        results.noteLiteral(misc_.getOrigText());
    }
;

/*
 *   In case we have a verb grammar rule that calls for a literal phrase,
 *   but the player enters a command with nothing in that slot, match an
 *   empty token list as a last resort.  Since this phrasing has a badness,
 *   we won't match it unless we don't have any better structural match.
 */
grammar literalPhrase(empty): [badness 400]: EmptyLiteralPhraseProd
    resolveLiteral(results) { }
;

/* ------------------------------------------------------------------------ */
/*
 *   An miscellaneous word list is a list of one or more words of any kind:
 *   any word, any integer, or any apostrophe-S token will do.  Note that
 *   known and unknown words can be mixed in an unknown word list; we care
 *   only that the list is made up of tokWord, tokInt, and/or
 *   abbreviation-period tokens.
 *
 *   Note that this kind of phrase is often used with a 'badness' value.
 *   However, we don't assign any badness here, because a miscellaneous
 *   word list might be perfectly valid in some contexts; instead, any
 *   productions that include a misc word list should specify badness as
 *   desired.
 */
grammar miscWordList(wordOrNumber):
    tokWord->txt_ | tokInt->txt_
    | tokPoundInt->txt_ | tokString->txt_ | tokAbbrPeriod->txt_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* we don't match anything directly with our vocabulary */
        return [];
    }
    getAdjustedTokens()
    {
        /* our token type is the special miscellaneous word type */
        return [txt_, &miscWord];
    }
;

grammar miscWordList(list):
    (tokWord->txt_ | tokInt->txt_ | tokAbbrPeriod->txt_
     | tokPoundInt->txt_ | tokString->txt_) miscWordList->lst_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results, extraFlags)
    {
        /* we don't match anything directly with our vocabulary */
        return [];
    }
    getAdjustedTokens()
    {
        /* our token type is the special miscellaneous word type */
        return [txt_, &miscWord] + lst_.getAdjustedTokens();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A main disambiguation phrase consists of a disambiguation phrase,
 *   optionally terminated with a period.
 */
grammar mainDisambigPhrase(main):
    disambigPhrase->dp_
    | disambigPhrase->dp_ '.'
    : BasicProd
    resolveNouns(resolver, results)
    {
        return dp_.resolveNouns(resolver, results);
    }
    getResponseList() { return dp_.getResponseList(); }
;

/*
 *   A "disambiguation phrase" is a phrase that answers a disambiguation
 *   question ("which book do you mean...").
 *   
 *   A disambiguation question can be answered with several types of
 *   syntax:
 *   
 *.  all/everything/all of them
 *.  both/both of them
 *.  any/any of them
 *.  <disambig list>
 *.  the <ordinal list> ones
 *.  the former/the latter
 *   
 *   Note that we assign non-zero badness to all of the ordinal
 *   interpretations, so that we will take an actual vocabulary
 *   interpretation instead of an ordinal interpretation whenever possible.
 *   For example, if an object's name is actually "the third button," this
 *   will give us greater affinity for using "third" as an adjective than
 *   as an ordinal in our own list.  
 */
grammar disambigPhrase(all):
    'všechny' | 'všechna' | 
    | ('každý' | 'každého' | 'každou' | 'každé' | 'vše') ( | 'z' 'nich')
    | ('z' 'nich') ('každý' | 'každého' | 'každou' | 'každé' | 'vše')
    : DisambigProd
    resolveNouns(resolver, results)
    {
        /* they want everything we proposed - return the whole list */
        return removeAmbigFlags(resolver.getAll(self));
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(both): 'oba' ( | 'dva') | 'obě' ( | 'dvě')
    | 'obojí' ( | 'dvojí') | 'oboje' ( | 'dvoje') : DisambigProd
    resolveNouns(resolver, results)
    {
        /*
         *   they want two items - return the whole list (if it has more
         *   than two items, we'll simply act as though they wanted all of
         *   them)
         */
        return removeAmbigFlags(resolver.getAll(self));
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(any):
    ('jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv'
    | 'libovolného' | 'libovolný' | 'libovolnou' | 'libovolné' | 'libovolná')
    ( | 'z' 'nich')
    | ('z' 'nich')
    ('jakéhokoliv' | 'jakýkoliv' | 'jakoukoliv' | 'jakékoliv' | 'jakákoliv'
    | 'libovolného' | 'libovolný' | 'libovolnou' | 'libovolné' | 'libovolná')
    : DisambigProd
    resolveNouns(resolver, results)
    {
        local lst;

        /* they want any item - arbitrarily pick the first one */
        lst = resolver.matchList.sublist(1, 1);

        /*
         *   add the "unclear disambiguation" flag to the item we picked,
         *   to indicate that the selection was arbitrary
         */
        if (lst.length() > 0)
            lst[1].flags_ |= UnclearDisambig;

        /* return the result */
        return lst;
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(list): disambigList->lst_ : DisambigProd
    resolveNouns(resolver, results)
    {
        return removeAmbigFlags(lst_.resolveNouns(resolver, results));
    }

    /* there's only me in the response list */
    getResponseList() { return lst_.getResponseList(); }
;

grammar disambigPhrase(ordinalList):
    ('ten' | 'ta' | 'to' | 'ti' | 'ty' | 'ta' | ) disambigOrdinalList->lst_
    : DisambigProd

    resolveNouns(resolver, results)
    {
        /* return the list with the ambiguity flags removed */
        return removeAmbigFlags(lst_.resolveNouns(resolver, results));
    }

    /* the response list consists of my single ordinal list item */
    getResponseList() { return [lst_]; }
;

/*
 *   A disambig list consists of one or more disambig list items, connected
 *   by noun phrase conjunctions.  
 */
grammar disambigList(single): disambigListItem->item_ : DisambigProd
    resolveNouns(resolver, results)
    {
        return item_.resolveNouns(resolver, results);
    }

    /* the response list consists of my single item */
    getResponseList() { return [item_]; }
;

grammar disambigList(list):
    disambigListItem->item_ commandOrNounConjunction disambigList->lst_
    : DisambigProd

    resolveNouns(resolver, results)
    {
        return item_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }

    /* my response list consists of each of our list items */
    getResponseList() { return [item_] + lst_.getResponseList(); }
;

/*
 *   Base class for ordinal disambiguation items
 */
class DisambigOrdProd: DisambigProd
    resolveNouns(resolver, results)
    {
        /* note the ordinal match */
        results.noteDisambigOrdinal();

        /* select the result by the ordinal */
        return selectByOrdinal(ord_, resolver, results);
    }

    selectByOrdinal(ordTok, resolver, results)
    {
        local idx;
        local matchList = resolver.ordinalMatchList;

        /*
         *   look up the meaning of the ordinal word (note that we assume
         *   that each ordinalWord is unique, since we only create one of
         *   each)
         */
        idx = cmdDict.findWord(ordTok, &ordinalWord)[1].numval;

        /*
         *   if it's the special value -1, it indicates that we should
         *   select the *last* item in the list
         */
        if (idx == -1)
            idx = matchList.length();

        /* if it's outside the limits of the match list, it's an error */
        if (idx > matchList.length())
        {
            /* note the problem */
            results.noteOrdinalOutOfRange(ordTok);

            /* no results */
            return [];
        }

        /* return the selected item as a one-item list */
        return matchList.sublist(idx, 1);
    }
;

/*
 *   A disambig vocab production is the base class for disambiguation
 *   phrases that involve vocabulary words.
 */
class DisambigVocabProd: DisambigProd
;

/*
 *   A disambig list item consists of:
 *
 *.  first/second/etc
 *.  the first/second/etc
 *.  first one/second one/etc
 *.  the first one/the second one/etc
 *.  <compound noun phrase>
 *.  possessive
 */

grammar disambigListItem(ordinal):
    ('ten' | 'ta' | 'to' | 'ti' | 'ty' | 'ta' | ) ordinalWord->ord_
    : DisambigOrdProd
;

/*
 *   Hráč může vytrvat ve tvaru odpovědi z předchozí otázky:
 *
 *   >vlož laser
 *   Do čeho ho chceš dát?
 *   >do přihrádky
 *   A co tím „přihrádky“ máš na mysli? Pravou přihrádku, nebo levou přihrádku?
 *   >do levé přihrádky  <- zde použije předložku
 */
grammar disambigListItem(noun):
    ('v' | 'o' | 'do' | 'skrz' | 'skrze' | 'od' | 'na' | 'po' | 's' | 'se'
    | 'k' | 'ke' | 'z' | 'ze' | )
    completeNounPhraseWithoutAll->np_ | terminalNounPhrase->np_
    : DisambigVocabProd
    resolveNouns(resolver, results)
    {
        /* get the matches for the underlying noun phrase */
        local lst = np_.resolveNouns(resolver, results);

        /* note the matches */
        results.noteMatches(lst);

        /* return the match list */
        return lst;
    }
;

grammar disambigListItem(plural):
    pluralPhrase->np_
    : DisambigVocabProd
    resolveNouns(resolver, results)
    {
        local lst;

        /*
         *   get the underlying match list; since we explicitly have a
         *   plural, the result doesn't need to be unique, so simply
         *   return everything we find
         */
        lst = np_.resolveNouns(resolver, results);

        /*
         *   if we didn't get anything, it's an error; otherwise, take
         *   everything, since we explicitly wanted a plural usage
         */
        if (lst.length() == 0)
            results.noMatch(resolver.getAction(), np_.getOrigText());
        else
            results.noteMatches(lst);

        /* return the list */
        return lst;
    }
;

grammar disambigListItem(possessive): possessiveNounPhrase->poss_
    : DisambigPossessiveProd
;

/*
 *   A disambig ordinal list consists of two or more ordinal words
 *   separated by noun phrase conjunctions.  Note that there is a minimum
 *   of two entries in the list.
 */
grammar disambigOrdinalList(tail):
    ordinalWord->ord1_ ('a' | ',') ordinalWord->ord2_ : DisambigOrdProd
    resolveNouns(resolver, results)
    {
        /* note the pair of ordinal matches */
        results.noteDisambigOrdinal();
        results.noteDisambigOrdinal();

        /* combine the selections of our two ordinals */
        return selectByOrdinal(ord1_, resolver, results)
            + selectByOrdinal(ord2_, resolver, results);
    }
;

grammar disambigOrdinalList(head):
    ordinalWord->ord_ ('a' | ',') disambigOrdinalList->lst_
    : DisambigOrdProd
    resolveNouns(resolver, results)
    {
        /* note the ordinal match */
        results.noteDisambigOrdinal();

        /* combine the selections of our ordinal and the sublist */
        return selectByOrdinal(ord_, resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Ordinal words.  We define a limited set of these, since we only use
 *   them in a few special contexts where it would be unreasonable to need
 *   even as many as define here.
 */
#define defOrdinal(str, val) object ordinalWord=#@str numval=val

/*
defOrdinal(první, 1);
defOrdinal(druhý, 2);
defOrdinal(druhá, 2);
defOrdinal(druhé, 2);
defOrdinal(druzí, 2);
defOrdinal(třetí, 3);
defOrdinal(čtvrtý, 4);
defOrdinal(čtvrtá, 4);
defOrdinal(čtvrté, 4);
defOrdinal(čtvrtí, 4);
defOrdinal(pátý, 5);
defOrdinal(pátá, 5);
defOrdinal(páté, 5);
defOrdinal(pátí, 5);
defOrdinal(šestý, 6);
defOrdinal(šestá, 6);
defOrdinal(šesté, 6);
defOrdinal(šestí, 6);
defOrdinal(sedmý, 7);
defOrdinal(sedmá, 7);
defOrdinal(sedmé, 7);
defOrdinal(sedmí, 7);
defOrdinal(osmý, 8);
defOrdinal(osmá, 8);
defOrdinal(osmé, 8);
defOrdinal(osmí, 8);
defOrdinal(devátý, 9);
defOrdinal(devátá, 9);
defOrdinal(deváté, 9);
defOrdinal(devátí, 9);
defOrdinal(desátý, 10);
defOrdinal(desátá, 10);
defOrdinal(desáté, 10);
defOrdinal(desátí, 10);
defOrdinal(jedenáctý, 11);
defOrdinal(jedenáctá, 11);
defOrdinal(jedenácté, 11);
defOrdinal(jedenáctí, 11);
defOrdinal(dvanáctý, 12);
defOrdinal(dvanáctá, 12);
defOrdinal(dvanácté, 12);
defOrdinal(dvanáctí, 12);
defOrdinal(třináctý, 13);
defOrdinal(třináctá, 13);
defOrdinal(třinácté, 13);
defOrdinal(třináctí, 13);
defOrdinal(čtrnáctý, 14);
defOrdinal(čtrnáctá, 14);
defOrdinal(čtrnácté, 14);
defOrdinal(čtrnáctí, 14);
defOrdinal(patnáctý, 15);
defOrdinal(patnáctá, 15);
defOrdinal(patnácté, 15);
defOrdinal(patnáctí, 15);
defOrdinal(šestnáctý, 16);
defOrdinal(šestnáctá, 16);
defOrdinal(šestnácté, 16);
defOrdinal(šestnáctí, 16);
defOrdinal(sedmnáctý, 17);
defOrdinal(sedmnáctá, 17);
defOrdinal(sedmnácté, 17);
defOrdinal(sedmnáctí, 17);
defOrdinal(osmnáctý, 18);
defOrdinal(osmnáctá, 18);
defOrdinal(osmnácté, 18);
defOrdinal(osmnáctí, 18);
defOrdinal(devatenáctý, 19);
defOrdinal(devatenáctá, 19);
defOrdinal(devatenácté, 19);
defOrdinal(devatenáctí, 19);
defOrdinal(dvacátý, 20);
defOrdinal(dvacátá, 20);
defOrdinal(dvacáté, 20);
defOrdinal(dvacátí, 20);
defOrdinal(1., 1);
defOrdinal(2., 2);
defOrdinal(3., 3);
defOrdinal(4., 4);
defOrdinal(5., 5);
defOrdinal(6., 6);
defOrdinal(7., 7);
defOrdinal(8., 8);
defOrdinal(9., 9);
defOrdinal(10., 10);
defOrdinal(11., 11);
defOrdinal(12., 12);
defOrdinal(13., 13);
defOrdinal(14., 14);
defOrdinal(15., 15);
defOrdinal(16., 16);
defOrdinal(17., 17);
defOrdinal(18., 18);
defOrdinal(19., 19);
defOrdinal(20., 20);
*/
object ordinalWord='první' numval=1;
object ordinalWord='druhý' numval=2;
object ordinalWord='druhá' numval=2;
object ordinalWord='druhé' numval=2;
object ordinalWord='druzí' numval=2;
object ordinalWord='třetí' numval=3;
object ordinalWord='čtvrtý' numval=4;
object ordinalWord='čtvrtá' numval=4;
object ordinalWord='čtvrté' numval=4;
object ordinalWord='čtvrtí' numval=4;
object ordinalWord='pátý' numval=5;
object ordinalWord='pátá' numval=5;
object ordinalWord='páté' numval=5;
object ordinalWord='pátí' numval=5;
object ordinalWord='šestý' numval=6;
object ordinalWord='šestá' numval=6;
object ordinalWord='šesté' numval=6;
object ordinalWord='šestí' numval=6;
object ordinalWord='sedmý' numval=7;
object ordinalWord='sedmá' numval=7;
object ordinalWord='sedmé' numval=7;
object ordinalWord='sedmí' numval=7;
object ordinalWord='osmý' numval=8;
object ordinalWord='osmá' numval=8;
object ordinalWord='osmé' numval=8;
object ordinalWord='osmí' numval=8;
object ordinalWord='devátý' numval=9;
object ordinalWord='devátá' numval=9;
object ordinalWord='deváté' numval=9;
object ordinalWord='devátí' numval=9;
object ordinalWord='desátý' numval=10;
object ordinalWord='desátá' numval=10;
object ordinalWord='desáté' numval=10;
object ordinalWord='desátí' numval=10;
object ordinalWord='jedenáctý' numval=11;
object ordinalWord='jedenáctá' numval=11;
object ordinalWord='jedenácté' numval=11;
object ordinalWord='jedenáctí' numval=11;
object ordinalWord='dvanáctý' numval=12;
object ordinalWord='dvanáctá' numval=12;
object ordinalWord='dvanácté' numval=12;
object ordinalWord='dvanáctí' numval=12;
object ordinalWord='třináctý' numval=13;
object ordinalWord='třináctá' numval=13;
object ordinalWord='třinácté' numval=13;
object ordinalWord='třináctí' numval=13;
object ordinalWord='čtrnáctý' numval=14;
object ordinalWord='čtrnáctá' numval=14;
object ordinalWord='čtrnácté' numval=14;
object ordinalWord='čtrnáctí' numval=14;
object ordinalWord='patnáctý' numval=15;
object ordinalWord='patnáctá' numval=15;
object ordinalWord='patnácté' numval=15;
object ordinalWord='patnáctí' numval=15;
object ordinalWord='šestnáctý' numval=16;
object ordinalWord='šestnáctá' numval=16;
object ordinalWord='šestnácté' numval=16;
object ordinalWord='šestnáctí' numval=16;
object ordinalWord='sedmnáctý' numval=17;
object ordinalWord='sedmnáctá' numval=17;
object ordinalWord='sedmnácté' numval=17;
object ordinalWord='sedmnáctí' numval=17;
object ordinalWord='osmnáctý' numval=18;
object ordinalWord='osmnáctá' numval=18;
object ordinalWord='osmnácté' numval=18;
object ordinalWord='osmnáctí' numval=18;
object ordinalWord='devatenáctý' numval=19;
object ordinalWord='devatenáctá' numval=19;
object ordinalWord='devatenácté' numval=19;
object ordinalWord='devatenáctí' numval=19;
object ordinalWord='dvacátý' numval=20;
object ordinalWord='dvacátá' numval=20;
object ordinalWord='dvacáté' numval=20;
object ordinalWord='dvacátí' numval=20;
object ordinalWord='1.' numval=1;
object ordinalWord='2.' numval=2;
object ordinalWord='3.' numval=3;
object ordinalWord='4.' numval=4;
object ordinalWord='5.' numval=5;
object ordinalWord='6.' numval=6;
object ordinalWord='7.' numval=7;
object ordinalWord='8.' numval=8;
object ordinalWord='9.' numval=9;
object ordinalWord='10.' numval=10;
object ordinalWord='11.' numval=11;
object ordinalWord='12.' numval=12;
object ordinalWord='13.' numval=13;
object ordinalWord='14.' numval=14;
object ordinalWord='15.' numval=15;
object ordinalWord='16.' numval=16;
object ordinalWord='17.' numval=17;
object ordinalWord='18.' numval=18;
object ordinalWord='19.' numval=19;
object ordinalWord='20.' numval=20;

/*
 *   the special 'last' ordinal - the value -1 is special to indicate the
 *   last item in a list
 */
//defOrdinal(poslední, -1);
object ordinalWord='poslední' numval=-1;


/* ------------------------------------------------------------------------ */
/*
 *   A numeric production.  These can be either spelled-out numbers (such
 *   as "fifty-seven") or numbers entered in digit form (as in "57").
 */
class NumberProd: BasicProd
    /* get the numeric (integer) value */
    getval() { return 0; }

    /*
     *   Get the string version of the numeric value.  This should return
     *   a string, but the string should be in digit form.  If the
     *   original entry was in digit form, then the original entry should
     *   be returned; otherwise, a string should be constructed from the
     *   integer value.  By default, we'll do the latter.
     */
    getStrVal() { return toString(getval()); }
;

/*
 *   A quantifier is simply a number, entered with numerals or spelled out.
 */
grammar numberPhrase(digits): tokInt->num_ : NumberProd
    /* get the numeric value */
    getval() { return toInteger(num_); }

    /*
     *   get the string version of the numeric value - since the token was
     *   an integer to start with, return the actual integer value
     */
    getStrVal() { return num_; }
;

grammar numberPhrase(spelled): spelledNumber->num_ : NumberProd
    /* get the numeric value */
    getval() { return num_.getval(); }
;

/*
 *   A number phrase preceded by a pound sign.  We distinguish this kind of
 *   number phrase from plain numbers, since this kind has a somewhat more
 *   limited set of valid contexts.  
 */
grammar poundNumberPhrase(main): tokPoundInt->num_ : NumberProd
    /*
     *   get the numeric value - a tokPoundInt token has a pound sign
     *   followed by digits, so the numeric value is the value of the
     *   substring following the '#' sign
     */
    getval() { return toInteger(num_.substr(2)); }

    /*
     *   get the string value - we have a number token following the '#',
     *   so simply return the part after the '#'
     */
    getStrVal() { return num_.substr(2); }
;


/*
 *   Number literals.  We'll define a set of special objects for numbers:
 *   each object defines a number and a value for the number.
 */
#define defDigit(num, val) object digitWord=#@num numval=val
#define defTeen(num, val)  object teenWord=#@num numval=val
#define defTens(num, val)  object tensWord=#@num numval=val

/*
defDigit(jeden, 1);
defDigit(jedna, 1);
defDigit(jedno, 1);
defDigit(jedni, 1);
defDigit(jedny, 1);
defDigit(dva, 2);
defDigit(dvě, 2);
defDigit(oba, 2);
defDigit(obě, 2);
defDigit(tři, 3);
defDigit(čtyři, 4);
defDigit(pět, 5);
defDigit(šest, 6);
defDigit(sedm, 7);
defDigit(osm, 8);
defDigit(devět, 9);
defTeen(deset, 10);
defTeen(jedenáct, 11);
defTeen(dvanáct, 12);
defTeen(třináct, 13);
defTeen(čtrnáct, 14);
defTeen(patnáct, 15);
defTeen(šestnáct, 16);
defTeen(sedmnáct, 17);
defTeen(osmnáct, 18);
defTeen(devatenáct, 19);
defTens(dvacet, 20);
defTens(třicet, 30);
defTens(ctyřicet, 40);
defTens(padesát, 50);
defTens(šedesát, 60);
defTens(sedmdesát, 70);
defTens(osmdesát, 80);
defTens(devadesát, 90);
*/

object digitWord='jeden' numval=1;
object digitWord='jedna' numval=1;
object digitWord='jedno' numval=1;
object digitWord='jedni' numval=1;
object digitWord='jedny' numval=1;
object digitWord='dva' numval=2;
object digitWord='dvě' numval=2;
object digitWord='oba' numval=2;
object digitWord='obě' numval=2;
object digitWord='tři' numval=3;
object digitWord='čtyři' numval=4;
object digitWord='pět' numval=5;
object digitWord='šest' numval=6;
object digitWord='sedm' numval=7;
object digitWord='osm' numval=8;
object digitWord='devět' numval=9;

object teenWord='deset' numval=10;
object teenWord='jedenáct' numval=11;
object teenWord='dvanáct' numval=12;
object teenWord='třináct' numval=13;
object teenWord='čtrnáct' numval=14;
object teenWord='patnáct' numval=15;
object teenWord='šestnáct' numval=16;
object teenWord='sedmnáct' numval=17;
object teenWord='osmnáct' numval=18;
object teenWord='devatenáct' numval=19;

object tensWord='dvacet' numval=20;
object tensWord='třicet' numval=30;
object tensWord='ctyřicet' numval=40;
object tensWord='padesát' numval=50;
object tensWord='šedesát' numval=60;
object tensWord='sedmdesát' numval=70;
object tensWord='osmdesát' numval=80;
object tensWord='devadesát' numval=90;

grammar spelledSmallNumber(digit): digitWord->num_ : NumberProd
    getval()
    {
        /*
         *   Look up the units word - there should be only one in the
         *   dictionary, since these are our special words.  Return the
         *   object's numeric value property 'numval', which gives the
         *   number for the name.
         */
        return cmdDict.findWord(num_, &digitWord)[1].numval;
    }
;

grammar spelledSmallNumber(teen): teenWord->num_ : NumberProd
    getval()
    {
        /* look up the dictionary word for the number */
        return cmdDict.findWord(num_, &teenWord)[1].numval;
    }
;

grammar spelledSmallNumber(tens): tensWord->num_ : NumberProd
    getval()
    {
        /* look up the dictionary word for the number */
        return cmdDict.findWord(num_, &tensWord)[1].numval;
    }
;

grammar spelledSmallNumber(tensAndUnits):
    tensWord->tens_ '-'->sep_ digitWord->units_
    | tensWord->tens_ digitWord->units_
    : NumberProd
    getval()
    {
        /* look up the words, and add up the values */
        return cmdDict.findWord(tens_, &tensWord)[1].numval
            + cmdDict.findWord(units_, &digitWord)[1].numval;
    }
;

grammar spelledSmallNumber(zero): 'nula' | 'nic' | 'žádný' | 'žádná' | 'žádné' | 'žádní' : NumberProd
    getval() { return 0; }
;

grammar spelledHundred(small): spelledSmallNumber->num_ : NumberProd
    getval() { return num_.getval(); }
;

grammar spelledHundred(hundreds): spelledSmallNumber->hun_ 'hundred'
    : NumberProd
    getval() { return hun_.getval() * 100; }
;

grammar spelledHundred(hundredsPlus):
    spelledSmallNumber->hun_ 'hundred' spelledSmallNumber->num_
    | spelledSmallNumber->hun_ 'hundred' 'and'->and_ spelledSmallNumber->num_
    : NumberProd
    getval() { return hun_.getval() * 100 + num_.getval(); }
;

grammar spelledHundred(aHundred): 'sto' : NumberProd
    getval() { return 100; }
;

grammar spelledHundred(aHundredPlus):
    'sto' ('a' | ) spelledSmallNumber->num_
    : NumberProd
    getval() { return 100 + num_.getval(); }
;

grammar spelledThousand(thousands): spelledHundred->thou_ 'thousand'
    : NumberProd
    getval() { return thou_.getval() * 1000; }
;

grammar spelledThousand(thousandsPlus):
    spelledHundred->thou_ 'thousand' spelledHundred->num_
    : NumberProd
    getval() { return thou_.getval() * 1000 + num_.getval(); }
;

grammar spelledThousand(thousandsAndSmall):
    spelledHundred->thou_ 'thousand' 'and' spelledSmallNumber->num_
    : NumberProd
    getval() { return thou_.getval() * 1000 + num_.getval(); }
;

grammar spelledThousand(aThousand): 'a' 'thousand' : NumberProd
    getval() { return 1000; }
;

grammar spelledThousand(aThousandAndSmall):
    'a' 'thousand' 'and' spelledSmallNumber->num_
    : NumberProd
    getval() { return 1000 + num_.getval(); }
;

grammar spelledMillion(millions): spelledHundred->mil_ 'million': NumberProd
    getval() { return mil_.getval() * 1000000; }
;

grammar spelledMillion(millionsPlus):
    spelledHundred->mil_ 'million'
    (spelledThousand->nxt_ | spelledHundred->nxt_)
    : NumberProd
    getval() { return mil_.getval() * 1000000 + nxt_.getval(); }
;

grammar spelledMillion(aMillion): 'a' 'million' : NumberProd
    getval() { return 1000000; }
;

grammar spelledMillion(aMillionAndSmall):
    'a' 'million' 'and' spelledSmallNumber->num_
    : NumberProd
    getval() { return 1000000 + num_.getval(); }
;

grammar spelledMillion(millionsAndSmall):
    spelledHundred->mil_ 'million' 'and' spelledSmallNumber->num_
    : NumberProd
    getval() { return mil_.getval() * 1000000 + num_.getval(); }
;

grammar spelledNumber(main):
    spelledHundred->num_
    | spelledThousand->num_
    | spelledMillion->num_
    : NumberProd
    getval() { return num_.getval(); }
;


/* ------------------------------------------------------------------------ */
/*
 *   "OOPS" command syntax
 */
grammar oopsCommand(main):
    oopsPhrase->oops_ | oopsPhrase->oops_ '.' : BasicProd
    getNewTokens() { return oops_.getNewTokens(); }
;

grammar oopsPhrase(main):
    'oprava' miscWordList->lst_ | 'oprava' ',' miscWordList->lst_
    | 'opr' miscWordList->lst_ | 'opr' ',' miscWordList->lst_
    | 'op' miscWordList->lst_ | 'op' ',' miscWordList->lst_
    : BasicProd
    getNewTokens() { return lst_.getOrigTokenList(); }
;

grammar oopsPhrase(missing):
    'oprava' | 'opr' | 'op'
    : BasicProd
    getNewTokens() { return nil; }
;

/* ------------------------------------------------------------------------ */
/*
 *   finishGame options.  We provide descriptions and keywords for the
 *   option objects here, because these are inherently language-specific.
 *   
 *   Note that we provide hyperlinks for our descriptions when possible.
 *   When we're in plain text mode, we can't show links, so we'll instead
 *   show an alternate form with the single-letter response highlighted in
 *   the text.  We don't highlight the single-letter response in the
 *   hyperlinked version because (a) if the user wants a shortcut, they can
 *   simply click the hyperlink, and (b) most UI's that show hyperlinks
 *   show a distinctive appearance for the hyperlink itself, so adding even
 *   more highlighting within the hyperlink starts to look awfully busy.  
 */
modify finishOptionQuit
    desc = "dát <<aHrefAlt('konec', 'KONEC', '<b>K</b>ONEC', 'Ukončit příběh')>> hry"
    responseKeyword = 'konec'
    responseChar = 'k'
;

modify finishOptionRestore
    desc = "<<aHrefAlt('nahrát', 'NAHRÁT', '<b>N</b>AHRÁT',
            'Nahrát uloženou pozici')>> uloženou pozici"
    responseKeyword = 'nahrát'
    responseChar = 'n'
;

modify finishOptionRestart
    desc = "<<aHrefAlt('restartovat', 'RESTARTOVAT', '<b>R</b>ESTARTOVAT',
            'Spustit příběh od začátku')>> příběh"
    responseKeyword = 'restartovat'
    responseChar = 'r'
;

modify finishOptionUndo
    desc = "<<aHrefAlt('odvolat', 'ODVOLAT', '<b>O</b>DVOLAT',
            'Odvolat poslední tah')>> poslední tah"
    responseKeyword = 'odvolat'
    responseChar = 'o'
;

modify finishOptionCredits
    desc = "dozvědět se, kdo je <<aHrefAlt('autor', 'AUTOR', '<b>A</b>UTOR',
            'Ukázat autory')>> příběhu"
    responseKeyword = 'autor'
    responseChar = 'a'
;

modify finishOptionFullScore
    desc = "zobrazit své <<aHrefAlt('detailní skóre', 'DETAILNÍ SKÓRE',
            'DETAILNÍ <b>S</b>KÓRE', 'Ukázat detailní skóre')>>"
    responseKeyword = 'detailní skóre'
    responseChar = 's'
;

modify finishOptionAmusing
    desc = "ukázat zábavné <<aHrefAlt('možnosti', 'MOŽNOSTI', '<b>M</b>OŽNOSTI',
            'Ukázat zábavné možnosti k vyzkoušení')>> k vyzkoušení"
    responseKeyword = 'možnosti'
    responseChar = 'm'
;

modify restoreOptionStartOver
    desc = "<<aHrefAlt('start', 'START', '<b>S</b>TART',
            'Start hry od začátku')>> hry od začátku"
    responseKeyword = 'start'
    responseChar = 's'
;

modify restoreOptionRestoreAnother
    desc = "<<aHrefAlt('nahrát', 'NAHRÁT', '<b>N</b>AHRÁT',
            'Nahrát uloženou pozici')>> jinou uloženou pozici"
;

/* ------------------------------------------------------------------------ */
/*
 *   Context for Action.getVerbPhrase().  This keeps track of pronoun
 *   antecedents in cases where we're stringing together a series of verb
 *   phrases.
 */
class GetVerbPhraseContext: object
    /*
     *   get the objective form of an object, using a pronoun as appropriate
     *
     *   V češtině si předáme podle obsahu verbPhrase, který pád chceme získat.
     *
     *   TODO: Po opravě regexpů odstranit varianty s mezerou.
     */
    objName(obj, gc)
    {
        /*
         *   if it's the pronoun antecedent, use the pronoun form;
         *   otherwise, use the full name
         *
         *   Zájmeno se použije, pokud už byl objekt zmíněn.
         *
         *   "(nejprve bereš keyring z pocket, potom odemykáš door 022, potom
         *   otevíráš *je*)"
         */
        if (obj == pronounObj)
        {
            if(gc == 4) return obj.zajmenoTe4J;
//            if(gc == 2) return obj.zajmenoTe2J;
//(nejprve vypínáš UniDiP, potom uvolňuješ vysílací modul od ho) -> nej
            if(gc == 2) return obj.zajmenoTebe2N;
            return 'TODO:obj.zajmeno(' + gc + ')';
        }
        else
        {
            return obj.nameCase(gc);
        }
    }

    /* are we showing the given object pronomially? */
    isObjPronoun(obj) { return (obj == pronounObj); }

    /* set the pronoun antecedent */
    setPronounObj(obj) { pronounObj = obj; }

    /* the pronoun antecedent */
    pronounObj = nil
;

/*
 *   Default getVerbPhrase context.  This can be used when no other context
 *   is needed.  This context instance has no state - it doesn't track any
 *   antecedents.  
 */
defaultGetVerbPhraseContext: GetVerbPhraseContext
    /* we don't remember any antecedents */
    setPronounObj(obj) { }
;

/* ------------------------------------------------------------------------ */
/*
 *   Implicit action context.  This is passed to the message methods that
 *   generate implicit action announcements, to indicate the context in
 *   which the message is to be used.
 */
class ImplicitAnnouncementContext: object
    /*
     *   Should we use the infinitive form of the verb, or the participle
     *   form for generating the announcement?  By default, use use the
     *   participle form: "(first OPENING THE BOX)".
     */
    useInfPhrase = nil

    /* is this message going in a list? */
    isInList = nil

    /*
     *   Are we in a sublist of 'just trying' or 'just asking' messages?
     *   (We can only have sublist groupings one level deep, so we don't
     *   need to worry about what kind of sublist we're in.)
     */
    isInSublist = nil

    /* our getVerbPhrase context - by default, don't use one */
    getVerbCtx = nil

    /* generate the announcement message given the action description */
    buildImplicitAnnouncement(txt)
    {
        /* if we're not in a list, make it a full, stand-alone message */
        if (!isInList)
            txt = '<./p0>\n<.assume>nejprve {|[jsi]} ' + txt + '<./assume>\n';

        /* return the result */
        return txt;
    }
;

/* the standard implicit action announcement context */
standardImpCtx: ImplicitAnnouncementContext;

/* the "just trying" implicit action announcement context */
tryingImpCtx: ImplicitAnnouncementContext
    /*
     *   The action was merely attempted, so use the infinitive phrase in
     *   the announcement: "(first trying to OPEN THE BOX)".
     */
    useInfPhrase = true

    /* build the announcement */
    buildImplicitAnnouncement(txt)
    {
        /*
         *   If we're not in a list of 'trying' messages, add the 'trying'
         *   prefix message to the action description.  This isn't
         *   necessary if we're in a 'trying' list, since the list itself
         *   will have the 'trying' part.
         */
        if (!isInSublist)
        {
            /*
             *   Pokud máme v txt zvratné zájmeno, přendáme ho hned za
             *   "nejprve", protože to tak bude lépe znít. Tedy místo "nejprve
             *   zkoušíš si sundat hodinky" budeme mít "nejprve si zkoušíš
             *   sundat hodinky".
             */
//            local res = rexSearch(siSePat, txt);
//            if(res)
//                txt = res[3] + '{zkouš[íš]|zkusil[a]} '
//                    + txt.substr(res[1] + 3);
//            else
                txt = '{zkouš[íš]|zkusil[a]} ' + txt;
        }

        /* now build the message into the full text as usual */
        return inherited(txt);
    }

    siSePat = static new RexPattern('(si|se) ')
;

/*
 *   The "asking question" implicit action announcement context.  By
 *   default, we generate the message exactly the same way we do for the
 *   'trying' case.
 */
askingImpCtx: tryingImpCtx;

/*
 *   A class for messages appearing in a list.  Within a list, we want to
 *   keep track of the last direct object, so that we can refer to it with
 *   a pronoun later in the list.
 */
class ListImpCtx: ImplicitAnnouncementContext, GetVerbPhraseContext
    /*
     *   Set the appropriate base context for the given implicit action
     *   announcement report (an ImplicitActionAnnouncement object).
     */
    setBaseCtx(ctx)
    {
        /*
         *   if this is a failed attempt, use a 'trying' context;
         *   otherwise, use a standard context
         */
        if (ctx.justTrying)
            baseCtx = tryingImpCtx;
        else if (ctx.justAsking)
            baseCtx = askingImpCtx;
        else
            baseCtx = standardImpCtx;
    }

    /* we're in a list */
    isInList = true

    /* we are our own getVerbPhrase context */
    getVerbCtx = (self)

    /* delegate the phrase format to our underlying announcement context */
    useInfPhrase = (delegated baseCtx)

    /* build the announcement using our underlying context */
    buildImplicitAnnouncement(txt) { return delegated baseCtx(txt); }

    /* our base context - we delegate some unoverridden behavior to this */
    baseCtx = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific Action modifications.
 */
modify Action
    /*
     *   In the English grammar, all 'predicate' grammar definitions
     *   (which are usually made via the VerbRule macro) are associated
     *   with Action match tree objects; in fact, each 'predicate' grammar
     *   match tree is the specific Action subclass associated with the
     *   grammar for the predicate.  This means that the Action associated
     *   with a grammar match is simply the grammar match object itself.
     *   Hence, we can resolve the action for a 'predicate' match simply
     *   by returning the match itself: it is the Action as well as the
     *   grammar match.
     *
     *   This approach ('grammar predicate' matches are based on Action
     *   subclasses) works well for languages like English that encode the
     *   role of each phrase in the word order of the sentence.
     *
     *   Languages that encode phrase roles using case markers or other
     *   devices tend to be freer with word order.  As a result,
     *   'predicate' grammars for such languages should generally not
     *   attempt to capture all possible word orderings for a given
     *   action, but should instead take the complementary approach of
     *   capturing the possible overall sentence structures independently
     *   of verb phrases, and plug in a verb phrase as a component, just
     *   like noun phrases plug into the English grammar.  In these cases,
     *   the match objects will NOT be Action subclasses; the Action
     *   objects will instead be buried down deeper in the match tree.
     *   Hence, resolveAction() must be defined on whatever class is used
     *   to construct 'predicate' grammar matches, instead of on Action,
     *   since Action will not be a 'predicate' match.
     */
    resolveAction(issuingActor, targetActor) { return self; }

    /*
     *   Return the interrogative pronoun for a missing object in one of
     *   our object roles.  In most cases, this is simply "what", but for
     *   some actions, "whom" is more appropriate (for example, the direct
     *   object of "ask" is implicitly a person, so "whom" is most
     *   appropriate for this role).
     */
     whatObj(which)
     {
         /* intransitive verbs have no objects, so there's nothing to show */
     }
     whatCase(which)
     {
         /* intransitive verbs have no objects, so there's nothing to show */
     }

    /*
     *   Return a string with the appropriate pronoun (objective form) for
     *   a list of object matches, with the given resolved cardinality.
     *   This list is a list of ResolveInfo objects.
     *
     *   Podej lopatu. "Komu *ji* chceš dát?"
     */
    objListPronoun(objList, gc, prep)
    {
        local himCnt, herCnt, themCnt;
        local FirstPersonCnt, SecondPersonCnt, ThirdPersonCnt;
        local resolvedNumber;

        /* if there's no object list at all, just use 'it' */
        if (objList == nil || objList == [])
            return 'to';

        /* note the number of objects in the resolved list */
        resolvedNumber = objList.length();

        /*
         *   In the tentatively resolved object list, we might have hidden
         *   away ambiguous matches.  Expand those back into the list so
         *   we have the full list of in-scope matches.
         */
        foreach (local cur in objList)
        {
            /*
             *   if this one has hidden ambiguous objects, add the hidden
             *   objects back into our list
             */
            if (cur.extraObjects != nil)
                objList += cur.extraObjects;
        }

        /*
         *   if the desired cardinality is plural and the object list has
         *   more than one object, simply say 'them'
         *
         *   ">dej mince Komu *je* chceš dát?"
         */
        if (objList.length() > 1 && resolvedNumber > 1)
            return 'je';

        /*
         *   singular cardinality - count masculine and feminine objects,
         *   and count the referral persons
         */
        himCnt = herCnt = themCnt = 0;
        FirstPersonCnt = SecondPersonCnt = ThirdPersonCnt = 0;
        foreach (local cur in objList)
        {
            /* if it's masculine, count it */
            if (cur.obj_.getGender() == 1 || cur.obj_.getGender() == 2
                || cur.obj_.getGender() == 4)
                    ++himCnt;

            /* if it's feminine, count it */
            if (cur.obj_.getGender() == 3)
                ++herCnt;

            /* if it has plural usage, count it */
            if (cur.obj_.isPlural)
                ++themCnt;

            /* if it's first person usage, count it */
            if (cur.obj_.referralPerson == FirstPerson)
                ++FirstPersonCnt;

            /* if it's second person usage, count it */
            if (cur.obj_.referralPerson == SecondPerson)
                ++SecondPersonCnt;

            /* if it's third person usage, count it */
            if (cur.obj_.referralPerson == ThirdPerson)
                ++ThirdPersonCnt;
        }

        /*
         *   if they all have plural usage, show "them"; if they're all of
         *   one gender, show "him" or "her" as appropriate; if they're
         *   all neuter, show "it"; otherwise, show "them"
         *
         *   "Odemkni dveře. Čím *je* chceš odemknout?" (Dveře jsou pomnožné.)
         */
        if (themCnt == objList.length())
            return 'je'; 
        /* Komu *se* chci ukázat? */
        else if (FirstPersonCnt == objList.length())
            return 'se';
        /* Komu *sebe* chceš ukázat? */
        else if (SecondPersonCnt == objList.length())
            return 'sebe';
        /*
         *   >ukaž
         *   (technikovi)
         *   Co *mu* chceš ukázat?
         */
        else if (himCnt == objList.length() && herCnt == 0)
            return
                prep == 'do' ?
                ['!', 'do něho', '!', '!', '!', '!', '!'][gc] :
                prep == 'na' ?
                ['!', 'na něho', '!', 'na něj', '!', 'na něm', '!'][gc] :
                prep == 'k' ?
                ['!', '!', 'k němu', '!', '!', '!', '!'][gc] :
                ['!', 'něho', 'mu', 'ho', '', 'něm', 'jím'][gc];
        else if (herCnt == objList.length() && himCnt == 0)
            return
                prep == 'do' ?
                ['!', 'do ní', '!', '!', '!', '!', '!'][gc] :
                prep == 'na' ?
                ['!', 'na ní', '!', 'na ni', '!', 'na ní', '!'][gc] :
                prep == 'k' ?
                ['!', '!', 'k ní', '!', '!', '!', '!'][gc] :
                ['!', 'ní', 'jí', 'ji', '', 'ní', 'jí'][gc];
        else if (herCnt == 0 && himCnt == 0)
            return '';
        else
            return 'je';
    }

    /*
     *   Announce a default object used with this action.
     *
     *   'resolvedAllObjects' indicates where we are in the command
     *   processing: this is true if we've already resolved all of the
     *   other objects in the command, nil if not.  We use this
     *   information to get the phrasing right according to the situation.
     */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /*
         *   the basic action class takes no objects, so there can be no
         *   default announcement
         */
        return '';
    }

    /*
     *   Announce all defaulted objects in the action.  By default, we
     *   show nothing.
     */
    announceAllDefaultObjects(allResolved) { }

    /*
     *   Return a phrase describing the action performed implicitly, as a
     *   participle phrase.  'ctx' is an ImplicitAnnouncementContext object
     *   describing the context in which we're generating the phrase.
     *
     *   This comes in two forms: if the context indicates we're only
     *   attempting the action, we'll return an infinitive phrase ("open
     *   the box") for use in a larger participle phrase describing the
     *   attempt ("trying to...").  Otherwise, we'll be describing the
     *   action as actually having been performed, so we'll return a
     *   present participle phrase ("opening the box").
     */
    getImplicitPhrase(ctx)
    {
        /*
         *   Get the phrase.  Use the infinitive or participle form, as
         *   indicated in the context.
         */
        return getVerbPhrase(ctx.useInfPhrase, ctx.getVerbCtx);
    }

    /*
     *   Get the infinitive form of the action.  We are NOT to include the
     *   infinitive complementizer (i.e., "to") as part of the result,
     *   since the complementizer isn't used in all contexts in which we
     *   might want to use the infinitive; for example, we don't want a
     *   "to" in phrases involving an auxiliary verb, such as "he can open
     *   the box."
     */
    getInfPhrase()
    {
        /* return the verb phrase in infinitive form */
        return getVerbPhrase(true, nil);
    }

    /*
     *   Get the root infinitive form of our verb phrase as part of a
     *   question in which one of the verb's objects is the "unknown" of
     *   the interrogative.  'which' is one of the role markers
     *   (DirectObject, IndirectObject, etc), indicating which object is
     *   the subject of the interrogative.
     *
     *   For example, for the verb UNLOCK <dobj> WITH <iobj>, if the
     *   unknown is the direct object, the phrase we'd return would be
     *   "unlock": this would plug into contexts such as "what do you want
     *   to unlock."  If the indirect object is the unknown for the same
     *   verb, the phrase would be "unlock it with", which would plug in as
     *   "what do you want to unlock it with".
     *
     *   Note that we are NOT to include the infinitive complementizer
     *   (i.e., "to") as part of the phrase we generate, since the
     *   complementizer isn't used in some contexts where the infinitive
     *   conjugation is needed (for example, "what should I <infinitive>").
     *
     *   Co chceš *odemknout*? Co tím chceš *odemknout*?
     */
    getQuestionInf(which)
    {
        /*
         *   for a verb without objects, this is the same as the basic
         *   infinitive
         */
        return getInfPhrase();
    }

    /*
     *   Get a string describing the full action in present participle
     *   form, using the current command objects: "taking the watch",
     *   "putting the book on the shelf"
     */
    getParticiplePhrase()
    {
        /* return the verb phrase in participle form */
        return getVerbPhrase(nil, nil);
    }

    /*
     *   Get the full verb phrase in either infinitive or participle
     *   format.  This is a common handler for getInfinitivePhrase() and
     *   getParticiplePhrase().
     *
     *   'ctx' is a GetVerbPhraseContext object, which lets us keep track
     *   of antecedents when we're stringing together multiple verb
     *   phrases.  'ctx' can be nil if the verb phrase is being used in
     *   isolation.
     */
    getVerbPhrase(inf, ctx)
    {
        /*
         *   parse the verbPhrase into the parts before and after the
         *   slash, and any additional text following the slash part
         *
         *   Tahle funkce vybírá sloveso do hlášky "(nejprve vstáváš)".
         *   Přidal jsem extra lomítko pro tvar minulého času a měl bych tu
         *   rozhodnout, zda chci přítomný a nebo minulý čas, popř. si tu
         *   informaci nechat předat parametrem.
         */
        rexMatch('(.*)/(<alphanum|lbrace|rbrace>+)/(<alphanum|lbrace|rbrace>+)'
            + '(.*)', verbPhrase);

        /* return the appropriate parts */
        if (inf)
        {
            /*
             *   infinitive - we want the part before the slash, plus the
             *   extra prepositions (or whatever) after the switched part
             */
            return rexGroup(1)[3] + rexGroup(4)[3];
        }
        else
        {
            /* participle - it's the part after the slash */
            return tSel(rexGroup(2)[3] + rexGroup(4)[3],
                rexGroup(3)[3] + rexGroup(4)[3]);
        }
    }

    /*
     *   Show the "noMatch" library message.  For most verbs, we use the
     *   basic "you can't see that here".  Verbs that are mostly used with
     *   intangible objects, such as LISTEN TO and SMELL, might want to
     *   override this to use a less visually-oriented message.
     */
    noMatch(msgObj, actor, txt) { msgObj.noMatchCannotSee(actor, txt); }

    /*
     *   Verb flags - these are used to control certain aspects of verb
     *   formatting.  By default, we have no special flags.
     */
    verbFlags = 0

    /* add a space prefix/suffix to a string if the string is non-empty */
    spPrefix(str) { return (str == '' ? str : ' ' + str); }
    spSuffix(str) { return (str == '' ? str : str + ' '); }
;

/*
 *   English-specific additions for single-object verbs.
 */
modify TAction
    /* return an interrogative word for an object of the action */
    whatObj(which)
    {
        /*
         *   Show the interrogative for our direct object - this is the
         *   last word enclosed in parentheses in our verbPhrase string.
         *
         *   Vrací první část pro větu typu "*Do čeho* chceš vstoupit?"
         *   Pro češtinu je to zde evidentně jednodušší, protože chceme
         *   zobrazit i předložku, která se v anglitině házela až na konec
         *   věty. 
         */
        rexSearch('<lparen>(.*)<rparen>', verbPhrase);
        return rexGroup(1)[3];
    }

    whatCase(which)
    {
        local gc;

        rexSearch('<lparen>.*?(<space>*<alpha>+)<rparen>', verbPhrase);
        gc = decodeGrammaticalCase(rexGroup(1)[3]);

        /* show the group match */
        return gc;
    }

    decodeGrammaticalCase(text, pref = 4)
    {
        local gc = nil;
        
        text = text.findReplace(' ', '');
//        say('((' + text + '))');
        
        if(text == 'koho' && pref == 2 || text == 'čeho') gc = 2;
        else if(text == 'komu' || text == 'čemu') gc = 3;
        else if(text == 'koho' && pref == 4 || text == 'co') gc = 4;
        else if(text == 'kom' || text == 'čem') gc = 6;
        else if(text == 'kým' || text == 'čím') gc = 7;
        else say('nezmany pad:((' + text + '))');
        
        return gc;
    }

    /* announce a default object used with this action */
    /*
     *   Zde bude největší problém s rozlišením koho/čeho vs. koho/co.
     *   Přitom "co" je vždy ve významu koho/co, nikdy není použito jako kdo/co.
     */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        local prep, pronoun;
        local nm;
        
        /*
         *   get any direct object preposition - this is the part inside
         *   the "(what)" specifier parens, excluding the last word
         */
        rexSearch('<lparen>(.*<space>+)?(<alpha>+)<rparen>', verbPhrase);
        prep = (rexGroup(1) == nil ? '' : rexGroup(1)[3]);
        pronoun = rexGroup(2)[3];

        /* do any verb-specific adjustment of the preposition */
        if (prep != nil)
            prep = adjustDefaultObjectPrep(prep, obj);
        
        /* do any verb-specific adjustment of the grammatical case */
        local gc = decodeGrammaticalCase(pronoun);
        if (gc != nil)
            gc = adjustDefaultObjectCase(gc, obj);

        /* Vokalizujeme predlozku s ohledem na objekt za ni. */
        prep = obj.makeVocalized(prep);

        /* 
         *   get the object name - we need to distinguish from everything
         *   else in scope, since we considered everything in scope when
         *   making our pick 
         */
        nm = obj.getAnnouncementDistinguisher(gActor.scopeList())
            .name(obj, gc);

        /* show the preposition (if any) and the object */
        return (prep == '' ? nm : prep + nm);
    }

    /*
     *   Adjust the preposition.  In some cases, the verb will want to vary
     *   the preposition according to the object.  This method can return a
     *   custom preposition in place of the one in the verbPhrase.  By
     *   default, we just use the fixed preposition from the verbPhrase,
     *   which is passed in to us in 'prep'.  
     */
    adjustDefaultObjectPrep(prep, obj) { return prep; }
    adjustDefaultObjectCase(gc, obj) { return gc; }

    /* announce all defaulted objects */
    announceAllDefaultObjects(allResolved)
    {
        /* announce a defaulted direct object if appropriate */
        maybeAnnounceDefaultObject(dobjList_, DirectObject, allResolved);
    }

    /* show the verb's basic infinitive form for an interrogative */
    /*
     *   Zobrazuje poslední část v "Do čeho chceš *vstoupit*?" Musel jsem
     *   odstranit předložku, protože v angličtině je na konci "What do you
     *   want *enter to*?"
     */
    getQuestionInf(which)
    {
        /*
         *   Show the present-tense verb form (removing the participle
         *   part - the "/xxxing" part).  Include any prepositions
         *   attached to the verb itself or to the direct object (inside
         *   the "(what)" parens).
         *
         *   Tohle generuje např. "Do čeho chceš *vstoupit*?"
         */
        rexSearch('(.*)/<alphanum|lbrace|rbrace>+/<alphanum|lbrace|rbrace>+'
                  + '(.*?)<space>+<lparen>(.*?)<space>*?<alpha>+<rparen>',
                  verbPhrase);
        return rexGroup(1)[3] + spPrefix(rexGroup(2)[3]);
    }

    /*
     *   Tahle nová funkce získává minulý čas do věty typu "Koho chceš, aby
     *   Sally *následovala*?" Jinak je to obdoba předchozí funkce
     */
    getQuestionPast(which)
    {
        rexSearch('<alphanum>*/<alphanum|lbrace|rbrace>*/'
            + '(<alphanum|lbrace|rbrace>*)<space>+', verbPhrase);
        return rexGroup(1)[3];
    }

    /* get the verb phrase in infinitive or participle form */
    getVerbPhrase(inf, ctx)
    {
        local dobj;
        local dobjText;
        local dobjIsPronoun;
        local ret;

        /* use the default pronoun context if one wasn't supplied */
        if (ctx == nil)
            ctx = defaultGetVerbPhraseContext;

        /* get the direct object */
        dobj = getDobj();

        /* note if it's a pronoun */
        dobjIsPronoun = ctx.isObjPronoun(dobj);

        /* get the direct object name */
        /* "nejprve bereš *lopatu*" - získává pádovou otázku pro objName */
        rexMatch('(.*)/(<alphanum|-|squote>+)(.*) <lparen>(.*?)'
            + '(<space>*?<alpha>+)<rparen>(.*)', verbPhrase);
        
        /* do any verb-specific adjustment of the grammatical case */
        local gc = decodeGrammaticalCase(rexGroup(5)[3]);
        if (gc != nil)
            gc = adjustDefaultObjectCase(gc, dobj);

        dobjText = ctx.objName(dobj, gc);

        /* get the phrasing */
        /* "nejprve *bereš lopatu*" */
        ret = getVerbPhrase1(inf, verbPhrase, dobjText, dobjIsPronoun);

        /* set the pronoun antecedent to my direct object */
        ctx.setPronounObj(dobj);

        /* return the result */
        return ret;
    }

    /*
     *   Given the text of the direct object phrase, build the verb phrase
     *   for a one-object verb.  This is a class method that can be used by
     *   other kinds of verbs (i.e., non-TActions) that use phrasing like a
     *   single object.
     *
     *   'inf' is a flag indicating whether to use the infinitive form
     *   (true) or the present participle form (nil); 'vp' is the
     *   verbPhrase string; 'dobjText' is the direct object phrase's text;
     *   and 'dobjIsPronoun' is true if the dobj text is rendered as a
     *   pronoun.
     *
     *   Výstup této funkce navazuje na slova "nejprve", "potom" v implicitních
     *   hlášeních.
     */
    getVerbPhrase1(inf, vp, dobjText, dobjIsPronoun)
    {
        local ret;
        local dprep;
        local vcomp;

        /*
         *   parse the verbPhrase: pick out the 'infinitive/participle'
         *   part, the complementizer part up to the '(what)' direct
         *   object placeholder, and any preposition within the '(what)'
         *   specifier
         *
         *   V češtině "squote" a "-" není potřeba, zato však povolím
         *   složené závorky na parametry.
         */
        rexMatch('(.*)/(<alphanum|lbrace|rbrace>+)/'
                 + '(<alphanum|lbrace|rbrace>+)(.*) '
                 + '<lparen>(.*?)<space>*?<alpha>+<rparen>(.*)',
                 vp);

        ret = '';

        /*
         *   Pokud dobjIsPronoun, tak ho teď zařadíme do věty. Jedná se
         *   o případ, kdy už jednou je objekt v hlášení zmíněn a tak podruhé
         *   je nahražen zájmenem.
         *
         *   "Nejprve odemykáš dveře, potom *je* otevíráš."
         */
        if (dobjIsPronoun)
            ret += dobjText + ' ';

        /*
         *   if there's any suffix following the direct object
         *   placeholder, add it at the end of the phrase
         *
         *   V našem případě je to případné "se", "si".
         *   "(nejprve *se* dostáváš ven ze stínu)"
         */
        ret += spSuffix(rexGroup(6)[3]);

        /*
         *   Přidáme do věty sloveso. Pokud není infinitiv, tak si vybereme
         *   správnou verzi času. Ostatní parametry se posunuly o jeden dál,
         *   protože máme oproti angličitě dvě lomítka ve verbPhrase.
         *
         *   "*odemykáš* dveře", "jsi *odemykla* dveře"
         */
        if (inf)
            ret += rexGroup(1)[3];
        else
            ret += tSel(rexGroup(2)[3], rexGroup(3)[3]);

        /* get the prepositional complementizer */
        vcomp = rexGroup(4)[3];

        /* get the direct object preposition */
        dprep = rexGroup(5)[3];

        /* do any verb-specific adjustment of the preposition */
        if (dprep != nil)
            dprep = adjustDefaultObjectPrep(dprep, getDobj());

        /*
         *   if the direct object is not a pronoun, put the complementizer
         *   BEFORE the direct object (the 'up' in "PICKING UP THE BOX")
         */
        if (!dobjIsPronoun)
            ret += spPrefix(vcomp);

        /* add the direct object preposition */
        ret += spPrefix(dprep);

        /*
         *   add the direct object, using the pronoun form if applicable
         *
         *   Pouze pokud už jsme nepřidali zájmeno dříve, viz výše.
         */
        if (!dobjIsPronoun)
            ret += ' ' + dobjText;

        /*
         *   if the direct object is a pronoun, put the complementizer
         *   AFTER the direct object (the 'up' in "PICKING IT UP")
         */
        if (dobjIsPronoun)
            ret += spPrefix(vcomp);

        /* return the complete phrase string */
        return ret;
    }
    /*
     *   Zde budeme hledat případné "se" za závorkou ve verbPhrase, abychom ho
     *   umístili do věty "Z čeho *se* chceš dostat ven?"
     */
    getReflexivePronoun()
    {
        rexMatch('(.*)/(<alphanum|-|squote>+)(.*) <lparen>(.*?)'
            + '(<space>*?<alpha>+)<rparen> *(.*)', verbPhrase);
        return rexGroup(6)[3];
    }
;

/*
 *   English-specific additions for two-object verbs.
 */
modify TIAction
    /*
     *   Flag: omit the indirect object in a query for a missing direct
     *   object.  For many verbs, if we already know the indirect object
     *   and we need to ask for the direct object, the query sounds best
     *   when it includes the indirect object: "what do you want to put in
     *   it?"  or "what do you want to take from it?".  This is the
     *   default phrasing.
     *
     *   However, the corresponding query for some verbs sounds weird:
     *   "what do you want to dig in with it?" or "whom do you want to ask
     *   about it?".  For such actions, this property should be set to
     *   true to indicate that the indirect object should be omitted from
     *   the queries, which will change the phrasing to "what do you want
     *   to dig in", "whom do you want to ask", and so on.
     */
    omitIobjInDobjQuery = nil

    /*
     *   For VerbRules: does this verb rule have a prepositional or
     *   structural phrasing of the direct and indirect object slots?  That
     *   is, are the object slots determined by a prepositional marker, or
     *   purely by word order?  For most English verbs with two objects,
     *   the indirect object is marked by a preposition: GIVE BOOK TO BOB,
     *   PUT BOOK IN BOX.  There are a few English verbs that don't include
     *   any prespositional markers for the objects, though, and assign the
     *   noun phrase roles purely by the word order: GIVE BOB BOOK, SHOW
     *   BOB BOOK, THROW BOB BOOK.  We define these phrasings with separate
     *   verb rules, which we mark with this property.
     *
     *   We use this in ranking verb matches.  Non-prepositional verb
     *   structures are especially prone to matching where they shouldn't,
     *   because we can often find a way to pick out words to fill the
     *   slots in the absence of any marker words.  For example, GIVE GREEN
     *   BOOK could be interpreted as GIVE BOOK TO GREEN, where GREEN is
     *   assumed to be an adjective-ending noun phrase; but the player
     *   probably means to give the green book to someone who they assumed
     *   would be filled in as a default.  So, whenever we find an
     *   interpretation that involves a non-prespositional phrasing, we'll
     *   use this flag to know we should be suspicious of it and try
     *   alternative phrasing first.
     *
     *   Most two-object verbs in English use prepositional markers, so
     *   we'll set this as the default.  Individual VerbRules that use
     *   purely structural phrasing should override this.
     */
    isPrepositionalPhrasing = true

    /* resolve noun phrases */
    resolveNouns(issuingActor, targetActor, results)
    {
        /*
         *   If we're a non-prepositional phrasing, it means that we have
         *   the VERB IOBJ DOBJ word ordering (as in GIVE BOB BOX or THROW
         *   BOB COIN).  For grammar match ranking purposes, give these
         *   phrasings a lower match probability when the dobj phrase
         *   doesn't have a clear qualifier.  If the dobj phrase starts
         *   with 'the' or a qualifier word like that (GIVE BOB THE BOX),
         *   then it's pretty clear that the structural phrasing is right
         *   after all; but if there's no qualifier, we could reading too
         *   much into the word order.  We could have something like GIVE
         *   GREEN BOX, where we *could* treat this as two objects, but we
         *   could just as well have a missing indirect object phrase.
         */
        if (!isPrepositionalPhrasing)
        {
            /*
             *   If the direct object phrase starts with 'a', 'an', 'the',
             *   'some', or 'any', the grammar is pretty clearly a good
             *   match for the non-prepositional phrasing.  Otherwise, it's
             *   suspect, so rank it accordingly.
             */
            if (rexMatch('(a|an|the|some|any)<space>',
                         dobjMatch.getOrigText()) == nil)
            {
                /* note this as weak phrasing level 100 */
                results.noteWeakPhrasing(100);
            }
        }

        /* inherit the base handling */
        inherited(issuingActor, targetActor, results);
    }

    /*
     *   get the interrogative for one of our objects
     *
     *   Vrací první část pro větu typu "*V čem* chceš vyhledat?" Pro češtinu
     *   je to zde evidentně jednodušší, protože chceme zobrazit i předložku,
     *   která se v anglitině házela až na konec věty.
     */
     whatObj(which)
     {
         switch (which)
         {
         case DirectObject:
             /*
              *   the direct object interrogative is the first word in
              *   parentheses in our verbPhrase string
              */
             rexSearch('<lparen>(.*)<rparen>.*<lparen>', verbPhrase);
             break;

         case IndirectObject:
             /*
              *   the indirect object interrogative is the second
              *   parenthesized word in our verbPhrase string
              */
             rexSearch('<rparen>.*<lparen>(.*)<rparen>', verbPhrase);
             break;
         }

         /* show the group match */
         return rexGroup(1)[3];
     }

    whatCase(which)
    {
        local gc;

        switch (which)
        {
        case DirectObject:
            rexSearch('<lparen>.*?(<space>*<alpha>+)<rparen>', verbPhrase);
            gc = decodeGrammaticalCase(rexGroup(1)[3]);
            break;

        case IndirectObject:
            rexSearch('<rparen>.*<lparen>.*?(<space>*<alpha>+)<rparen>',
                verbPhrase);
            gc = decodeGrammaticalCase(rexGroup(1)[3]);
            break;
        }

        /* show the group match */
        return gc;
    }

    whatPrep(which)
    {
        switch (which)
        {
        case DirectObject:
            rexSearch('<lparen>(.*?)<space>*<alpha>+<rparen>', verbPhrase);
            return rexGroup(1)[3];

        case IndirectObject:
            rexSearch('<rparen>.*<lparen>(.*?)<space>*<alpha>+<rparen>',
                verbPhrase);
            return rexGroup(1)[3];
        }

        /* show the group match */
        return '';
    }

    /* announce a default object used with this action */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        local verb;
        local prep;
        local gc;

        /* presume we won't have a verb or preposition */
        verb = '';
        prep = '';

        /*
         *   Check the full phrasing - if we're showing the direct object,
         *   but an indirect object was supplied, use the verb's
         *   participle form ("asking bob") in the default string, since
         *   we must clarify that we're not tagging the default string on
         *   to the command line.  Don't include the participle form if we
         *   don't know all the objects yet, since in this case we are in
         *   fact tagging the default string onto the command so far, as
         *   there's nothing else in the command to get in the way.
         *
         *   >zeptej se na kouř
         *   (*ptáš* se uhlíře)
         */
        if (whichObj == DirectObject && resolvedAllObjects)
        {
            /*
             *   extract the verb's participle form (including any
             *   complementizer phrase)
             *
             *   TODO: Jeden regexp selhává, vyzkoušet v 3.0.19
             */
            rexSearch('/(<alpha|lbrace|rbrace>+)/', verbPhrase);
            local a = rexGroup(1)[3];

            rexSearch('/(<alpha|lbrace|rbrace>+) ', verbPhrase);
            local b = rexGroup(1)[3];

            rexSearch('/<alpha|lbrace|rbrace>+ (<alpha>+)', verbPhrase);
            local c = rexGroup(1) ? rexGroup(1)[3] : '';

            verb = tSel(a, b) + '{| [jsi]}' + spPrefix(spSuffix(c));
        }

        /*
         *   Za sloveso přidávám ještě zvratné zájmeno, pokud se ke slovesu
         *   váže. Podmíněno je to stejným způsobem, jako získání slovesa. Tj.
         *   buď se vypíše celé "(*ptáš se* uhlíře)" a nebo jen "(uhlíře)".
         */
        local zvratneZajmeno = '';
        if (whichObj == DirectObject && resolvedAllObjects)
        {
            rexSearch('<rparen>.*<rparen><space>*(<alpha>+)', verbPhrase);
            if(rexGroup(1)) zvratneZajmeno = rexGroup(1)[3];
        }

        /*
         *   Máme nějakou preferenci pro pád podle zvratného zájmena, která by
         *   pomohla rozhodnout mezi "koho" jako 2. vs. 4. pád?
         */
        local pref;
        if(zvratneZajmeno == 'se') pref = 2;
        else pref = 4;

        /* get the preposition to use, if any */
        switch(whichObj)
        {
        case DirectObject:
            /* use the preposition in the first "(what)" phrase */
            rexSearch('<lparen>(.*?)<space>*<alpha>+<rparen>', verbPhrase);
            prep = rexGroup(1)[3];
            rexSearch('<lparen>.*?(<space>*<alpha>+)<rparen>', verbPhrase);
            gc = decodeGrammaticalCase(rexGroup(1)[3], pref);
            break;

        case IndirectObject:
            /* use the preposition in the second "(what)" phrase */
            rexSearch('<rparen>.*<lparen>(.*?)<space>*<alpha>+<rparen>',
                      verbPhrase);
            prep = rexGroup(1)[3];
            rexSearch('<rparen>.*<lparen>.*?(<space>*<alpha>+)<rparen>',
                verbPhrase);
            gc = decodeGrammaticalCase(rexGroup(1)[3], pref);
            break;
        }

        /* Vokalizujeme predlozku s ohledem na objekt za ni. */
        prep = obj.makeVocalized(prep);

        /* 
         *   get the object name - we need to distinguish from everything
         *   else in scope, since we considered everything in scope when
         *   making our pick 
         */
        
        /* build and return the complete phrase */
        return spSuffix(verb) + spSuffix(zvratneZajmeno) + spSuffix(prep)
            + obj.getAnnouncementDistinguisher(gActor.scopeList())
            .name(obj, gc);
    }

    /* announce all defaulted objects */
    announceAllDefaultObjects(allResolved)
    {
        /* announce a defaulted direct object if appropriate */
        maybeAnnounceDefaultObject(dobjList_, DirectObject, allResolved);

        /* announce a defaulted indirect object if appropriate */
        maybeAnnounceDefaultObject(iobjList_, IndirectObject, allResolved);
    }

    /* show the verb's basic infinitive form for an interrogative */
    getQuestionInf(which)
    {
        local ret;
        local vcomp;

        /*
         *   Our verb phrase can one of three formats, depending on which
         *   object role we're asking about (the part in <angle brackets>
         *   is the part we're responsible for generating).  In these
         *   formats, 'verb' is the verb infinitive; 'comp' is the
         *   complementizer, if any (e.g., the 'up' in 'pick up'); 'dprep'
         *   is the direct object preposition (the 'in' in 'dig in x with
         *   y'); and 'iprep' is the indirect object preposition (the
         *   'with' in 'dig in x with y').
         *
         *   asking for dobj: verb vcomp dprep iprep it ('what do you want
         *   to <open with it>?', '<dig in with it>', '<look up in it>').
         *
         *   asking for dobj, but suppressing the iobj part: verb vcomp
         *   dprep ('what do you want to <turn>?', '<look up>?', '<dig
         *   in>')
         *
         *   asking for iobj: verb dprep it vcomp iprep ('what do you want
         *   to <open it with>', '<dig in it with>', '<look it up in>'
         *
         *   V češtině se mnoho z této funkcionality přesouvá do msg_neu.t
         *   a funkcí askMissingObject, missingObject a missingLiteral,
         *   protože předložky i zájmena se staví do věty a jiné místo.
         *
         *   "Na co se ho chceš *zeptat*?"
         *   "Čím chceš *vyhrabat díru*?"
         */

        /* parse the verbPhrase into its component parts */
        rexMatch('(.*)/<alphanum|lbrace|rbrace>+/<alphanum|lbrace|rbrace>+(?:<space>+(<^lparen>*))?'
                 + '<space>+<lparen>(.*?)<space>*<alpha>+<rparen>'
                 + '<space>+<lparen>(.*?)<space>*<alpha>+<rparen>',
                 verbPhrase);

        /*
         *   pull out the verb complementizer
         *
         *   Complementizer je část verbPhrase mezi slovesy a zástupci objektů
         *   v závorkách:
         *
         *   verbPhrase = 'vyhrabat/hrab{eš}/vyhrabal{a} *díru* (v čem) (čím)'
         *
         *   Ve větě patří přímo za sloveso: "V čem chceš vyhrabat *díru*?"
         */
        vcomp = (rexGroup(2) == nil ? '' : rexGroup(2)[3]);

        /* pull out the verb */
        ret = rexGroup(1)[3];

        /* add the <vcomp> part in all cases */
        ret += spPrefix(vcomp);

        /* return the result */
        return ret;
    }

    /*
     *   Get the pronoun for the message object in the given role.
     */
    getOtherMessageObjectPronoun(which)
    {
        local lst;

        /*
         *   Get the resolution list (or tentative resolution list) for the
         *   *other* object, since we want to show a pronoun representing
         *   the other object.  If we don't have a fully-resolved list for
         *   the other object, use the tentative resolution list, which
         *   we're guaranteed to have by the time we start resolving
         *   anything (and thus by the time we need to ask for objects).
         */
        lst = (which == DirectObject ? iobjList_ : dobjList_);
        if (lst == nil || lst == [])
            lst = (which == DirectObject
                   ? tentativeIobj_ : tentativeDobj_);

        /* if we found an object list, use the pronoun for the list */
        if (lst != nil && lst != [])
        {
            /* we got a list - return a suitable pronoun for this list */
            return objListPronoun(lst, whatCase(which == DirectObject
                ? IndirectObject : DirectObject), whatPrep(which == DirectObject
                ? IndirectObject : DirectObject));
        }
        else
        {
            /* there's no object list, so there's no pronoun */
            return nil;
        }
    }

    /* get the verb phrase in infinitive or participle form */
    getVerbPhrase(inf, ctx)
    {
        local dobj, dobjText, dobjIsPronoun;
        local iobj, iobjText;
        local ret;

        /* use the default context if one wasn't supplied */
        if (ctx == nil)
            ctx = defaultGetVerbPhraseContext;

        /*
         *   get the direct object information
         *
         *   Vezmeme pádovou otázku a podle ní skloňujeme objekt.
         *
         *   TODO: bez .* před <lparen> nefunguje, znovu zkusit v 3.0.19 a
         *   případně poslat další bugreport, viz i níže.
         */
        dobj = getDobj();
        rexMatch('.*<lparen>(<alpha>*?)(<space>?<alpha>+)<rparen><space>', verbPhrase);
        dobjText = ctx.objName(dobj, decodeGrammaticalCase(rexGroup(2)[3]));
        dobjIsPronoun = ctx.isObjPronoun(dobj);

        /* get the indirect object information */
        iobj = getIobj();
        rexMatch('.*<rparen><space>*<lparen>(.*?<space>|)(<alpha>+)', verbPhrase);
        iobjText = (iobj != nil ? ctx.objName(iobj, decodeGrammaticalCase(rexGroup(2)[3])) : nil);

        /* get the phrasing */
        ret = getVerbPhrase2(inf, verbPhrase,
                             dobjText, dobjIsPronoun, iobjText);

        /*
         *   Set the antecedent for the next verb phrase.  Our direct
         *   object is normally the antecedent; however, if the indirect
         *   object matches the current antecedent, keep the current
         *   antecedent, so that 'it' (or whatever) remains the same for
         *   the next verb phrase.
         */
        if (ctx.pronounObj != iobj)
            ctx.setPronounObj(dobj);

        /* return the result */
        return ret;
    }

    /*
     *   Get the verb phrase for a two-object (dobj + iobj) phrasing.  This
     *   is a class method, so that it can be reused by unrelated (i.e.,
     *   non-TIAction) classes that also use two-object syntax but with
     *   other internal structures.  This is the two-object equivalent of
     *   TAction.getVerbPhrase1().
     *
     *   Tahle funkce už má dané tvary popisu objektů a jen najde předložky
     *   a sestaví do věty.
     */
    getVerbPhrase2(inf, vp, dobjText, dobjIsPronoun, iobjText)
    {
        local ret;
        local vcomp;
        local dprep, iprep;

        /* parse the verbPhrase into its component parts */
        rexMatch('(.*)/(<alphanum|lbrace|rbrace>+)/(<alphanum|lbrace|rbrace>+)(?:<space>+(<^lparen>*))?'
                 + '<space>+<lparen>(.*?)<space>*<alpha>+<rparen>'
                 + '<space>+<lparen>(.*?)<space>*<alpha>+<rparen>',
                 vp);

        ret = '';

        /* get the direct and indirect object prepositions */
        dprep = rexGroup(5)[3];
        iprep = rexGroup(6)[3];

        /*
         *   Pokud dobjIsPronoun, tak ho teď zařadíme do věty. Jedná se
         *   o případ, kdy už jednou je objekt v hlášení zmíněn a tak podruhé
         *   je nahražen zájmenem.
         */
        if (dobjIsPronoun)
            ret += spPrefix(dprep) + ' ' + dobjText + ' ';

        /* start off with the infinitive or participle, as desired */
        if (inf)
            ret += rexGroup(1)[3];
        else
            ret += tSel(rexGroup(2)[3], rexGroup(3)[3]);

        /* get the complementizer */
        vcomp = (rexGroup(4) == nil ? '' : rexGroup(4)[3]);

        /*
         *   add the complementizer BEFORE the direct object, if the
         *   direct object is being shown as a full name ("PICK UP BOX")
         */
        if (!dobjIsPronoun)
            ret += spPrefix(vcomp);

        /*
         *   add the direct object and its preposition, using a pronoun if
         *   applicable
         *
         *   Pouze pokud už jsme nepřidali zájmeno dříve, viz výše.
         */
        if (!dobjIsPronoun)
            ret += spPrefix(dprep) + ' ' + dobjText;

        /*
         *   add the complementizer AFTER the direct object, if the direct
         *   object is shown as a pronoun ("PICK IT UP")
         */
        if (dobjIsPronoun)
            ret += spPrefix(vcomp);

        /* if we have an indirect object, add it with its preposition */
        if (iobjText != nil)
            ret += spPrefix(iprep) + ' ' + iobjText;

        /* return the result phrase */
        return ret;
    }
    
    preferredIobj = nil
    replace doActionMain()
    {
        local lst;
        local preAnnouncedDobj;
        local preAnnouncedIobj;
       
        /*
         *   Get the list of resolved objects for the multiple object.  If
         *   neither has multiple objects, it doesn't matter which is
         *   iterated, since we'll just do the command once anyway. 
         */
        lst = (iobjList_.length() > 1 ? iobjList_ : dobjList_);

        /*
         *   Set the pronoun antecedents, using the game-specific pronoun
         *   setter.  Don't set an antecedent for a nested command.
         */
        if (parentAction == nil)
        {
           /*
            *   Set both direct and indirect objects as potential
            *   antecedents.  Rather than trying to figure out right now
            *   which one we might want to refer to in the future, remember
            *   both - we'll decide which one is the logical antecedent
            *   when we find a pronoun to resolve in a future command. 
            */
           gActor.setPronounMulti(dobjList_, iobjList_);

            /*
             *   If one or the other object phrase was specified in the
             *   input as a pronoun, keep the meaning of that pronoun the
             *   same, overriding whatever we just did.  Note that the
             *   order we use here doesn't matter: if a given pronoun
             *   appears in only one of the two lists, then the list where
             *   it's not set has no effect on the pronoun, hence it
             *   doesn't matter which comes first; if a pronoun appears in
             *   both lists, it will have the same value in both lists, so
             *   we'll just do the same thing twice, so, again, order
             *   doesn't matter. 
             */
            setPronounByInput(dobjList_);
            setPronounByInput(iobjList_);
        }

        /*
         *   pre-announce the non-list object if appropriate - this will
         *   provide a common pre-announcement if we iterate through
         *   several announcements of the main list objects
         */
        if (lst == dobjList_)
        {
            /* pre-announce the single indirect object if needed */
            preAnnouncedIobj = preAnnounceActionObject(
                iobjList_[1], dobjList_, IndirectObject);

            /* we haven't announced the direct object yet */
            preAnnouncedDobj = nil;

            /* pre-calculate the multi-object announcements */
            cacheMultiObjectAnnouncements(dobjList_, DirectObject);
        }
        else
        {
            /* pre-announce the single direct object if needed */
            preAnnouncedDobj = preAnnounceActionObject(
                dobjList_[1], iobjList_, DirectObject);

            /* we haven't announced the indirect object yet */
            preAnnouncedIobj = nil;

            /* pre-calculate the multi-object announcements */
            cacheMultiObjectAnnouncements(iobjList_, IndirectObject);
        }

        /* we haven't yet canceled the iteration */
        iterationCanceled = nil;

        /* iterate over the resolved list for the multiple object */
        for (local i = 1, local len = lst.length() ;
             i <= len && !iterationCanceled ; ++i)
        {
            local dobjInfo;
            local iobjInfo;

            /*
             *   make the current list item the direct or indirect object,
             *   as appropriate
             */
            if (lst == dobjList_)
            {
                /* the direct object is the multiple object */
                dobjInfo = dobjInfoCur_ = lst[i];
                iobjInfo = iobjInfoCur_ = iobjList_[1];
            }
            else
            {
                /* the indirect object is the multiple object */
                dobjInfo = dobjInfoCur_ = dobjList_[1];
                iobjInfo = iobjInfoCur_ = lst[i];
            }

            /*
             *  Get the current dobj and iobj from the resolve info.
             *
             *  In Czech there are few nonprepositional action phrases where it
             *  is not possible to distinguish order of iobj and dobj in parser,
             *  because player can enter them in either order equaly well and
             *  when parser is parsing a command it didn't have object resolved
             *  yet so it can't guess what is what by logicalness.
             *
             *  Actions affected are GiveTo, ShowTo, ThrowTo, MoveWith,
             *  TurnWith, BurnWith, CutWith, CleanWith, (Un)LockWith,
             *  (Un)ScrewWith. For these actions we should define preferredIobj
             *  property hinting which object should be in iobj role, so we can
             *  swap objects if they came from the parser in wrong order.
             */
            if (preferredIobj != nil && dobjInfo.obj_.ofKind(preferredIobj))
            {
                /* if preferredIobj is in dobj slot, then swap */
                dobjCur_ = iobjInfo.obj_;
                iobjCur_ = dobjInfo.obj_;
            }
            else
            {
                dobjCur_ = dobjInfo.obj_;
                iobjCur_ = iobjInfo.obj_;
            }   
           
            /*
             *   if the action was remapped, and we need to announce
             *   anything, announce the entire action
             */
            if (isRemapped())
            {
                /*
                 *   We were remapped.  The entire phrasing of the new
                 *   action might have changed from what the player typed,
                 *   so it might be nonsensical to show the objects as we
                 *   usually would, as sentence fragments that are meant
                 *   to combine with what the player actually typed.  So,
                 *   instead of showing the usual sentence fragments, show
                 *   the entire phrasing of the command.
                 *   
                 *   Only show the announcement if we have a reason to: we
                 *   have unclear disambiguation in one of the objects, or
                 *   one of the objects is defaulted.
                 *   
                 *   If we don't want to announce the remapped action,
                 *   still consider showing a multi-object announcement,
                 *   if we would normally need to do so. 
                 */
                if (needRemappedAnnouncement(dobjInfo)
                    || needRemappedAnnouncement(iobjInfo))
                {
                    /* show the remapped announcement */
                    gTranscript.announceRemappedAction();
                }
                else
                {
                    /* announce the multiple dobj if necessary */
                    if (!preAnnouncedDobj)
                        maybeAnnounceMultiObject(
                            dobjInfo, dobjList_.length(), DirectObject);

                    /* announce the multiple iobj if necessary */
                    if (!preAnnouncedIobj)
                        maybeAnnounceMultiObject(
                            iobjInfo, iobjList_.length(), IndirectObject);
                }
            }
            else
            {
                /* announce the direct object if appropriate */
                if (!preAnnouncedDobj)
                    announceActionObject(dobjInfo, dobjList_.length(),
                                         DirectObject);

                /* announce the indirect object if appropriate */
                if (!preAnnouncedIobj)
                    announceActionObject(iobjInfo, iobjList_.length(),
                                         IndirectObject);
            }

            /* run the execution sequence for the current direct object */
            doActionOnce();

            /* if we're top-level, count the iteration in the transcript */
            if (parentAction == nil)
                gTranscript.newIter();
        }
    }
;

class PreferredIobj: object;

/*
 *   English-specific additions for verbs taking a literal phrase as the
 *   sole object.
 */
modify LiteralAction
    /* provide a base verbPhrase, in case an instance leaves it out */
    verbPhrase = 'verb/verbing (what)'

    /* get an interrogative word for an object of the action */
    whatObj(which)
    {
        /* use the same processing as TAction */
        return delegated TAction(which);
    }
    whatCase(which)
    {
        /* use the same processing as TAction */
        return delegated TAction(which);
    }

    getVerbPhrase(inf, ctx)
    {
        /* handle this as though the literal were a direct object phrase */
        return TAction.getVerbPhrase1(inf, verbPhrase, gLiteral, nil);
    }

    getQuestionInf(which)
    {
        /* use the same handling as for a regular one-object action */
        return delegated TAction(which);
    }

;

/*
 *   English-specific additions for verbs of a direct object and a literal
 *   phrase.
 */
modify LiteralTAction
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /*
         *   Use the same handling as for a regular two-object action.  We
         *   can only default the actual object in this kind of verb; the
         *   actual object always fills the DirectObject slot, but in
         *   message generation it might use a different slot, so use the
         *   message generation slot here.
         */
        return delegated TIAction(obj, whichMessageObject,
                                  resolvedAllObjects);
    }

    whatObj(which)
    {
        /* use the same handling we use for a regular two-object action */
        return delegated TIAction(which);
    }
    whatCase(which)
    {
        /* use the same handling we use for a regular two-object action */
        return delegated TIAction(which);
    }

    getQuestionInf(which)
    {
        /*
         *   use the same handling as for a two-object action (but note
         *   that we override getMessageObjectPronoun(), which will affect
         *   the way we present the verb infinitive in some cases)
         */
        return delegated TIAction(which);
    }

    /*
     *   When we want to show a verb infinitive phrase that involves a
     *   pronoun for the literal phrase, refer to the literal as 'that'
     *   rather than 'it' or anything else.
     */
    getOtherMessageObjectPronoun(which)
    {
        /*
         *   If we're asking about the literal phrase, then the other
         *   pronoun is for the resolved object: so, return the pronoun
         *   for the direct object phrase, because we *always* store the
         *   non-literal in the direct object slot, regardless of the
         *   actual phrasing of the action.
         *
         *   If we're asking about the resolved object (i.e., not the
         *   literal phrase), then return 'that' as the pronoun for the
         *   literal phrase.
         */
        if (which == whichMessageLiteral)
        {
            /*
             *   we're asking about the literal, so the other pronoun is
             *   for the resolved object, which is always in the direct
             *   object slot (so the 'other' slot is effectively the
             *   indirect object)
             */
            return delegated TIAction(IndirectObject);
        }
        else
        {
            /*
             *   We're asking about the resolved object, so the other
             *   pronoun is for the literal phrase: always use 'that' to
             *   refer to the literal phrase.
             *
             *   >napiš KBR-3867
             *   Na čem *to* chceš napsat?
             */
            return 'to';
        }
    }

    getVerbPhrase(inf, ctx)
    {
        local dobj, dobjText, dobjIsPronoun;
        local litText;
        local ret;

        /* use the default context if one wasn't supplied */
        if (ctx == nil)
            ctx = defaultGetVerbPhraseContext;

        /* get the direct object information */
        dobj = getDobj();
        dobjText = ctx.objName(dobj, 1);
        dobjIsPronoun = ctx.isObjPronoun(dobj);

        /* get our literal text */
        litText = gLiteral;

        /*
         *   Use the standard two-object phrasing.  The order of the
         *   phrasing depends on whether our literal phrase is in the
         *   direct or indirect object slot.
         */
        if (whichMessageLiteral == DirectObject)
            ret = TIAction.getVerbPhrase2(inf, verbPhrase,
                                          litText, nil, dobjText);
        else
            ret = TIAction.getVerbPhrase2(inf, verbPhrase,
                                          dobjText, dobjIsPronoun, litText);

        /* use the direct object as the antecedent for the next phrase */
        ctx.setPronounObj(dobj);

        /* return the result */
        return ret;
    }
;

/*
 *   English-specific additions for verbs taking a topic phrase as the sole
 *   object.  
 */
modify TopicAction
    /* get an interrogative word for an object of the action */
    whatObj(which)
    {
        /* use the same processing as TAction */
        return delegated TAction(which);
    }
    whatCase(which)
    {
        /* use the same processing as TAction */
        return delegated TAction(which);
    }

    getVerbPhrase(inf, ctx)
    {
        /* handle this as though the topic text were a direct object phrase */
        return TAction.getVerbPhrase1(
            inf, verbPhrase, getTopic().getTopicText().toLower(), nil);
    }

    getQuestionInf(which)
    {
        /* use the same handling as for a regular one-object action */
        return delegated TAction(which);
    }

;

/*
 *   English-specific additions for verbs with topic phrases.
 */
modify TopicTAction
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /*
         *   Use the same handling as for a regular two-object action.  We
         *   can only default the actual object in this kind of verb; the
         *   actual object always fills the DirectObject slot, but in
         *   message generation it might use a different slot, so use the
         *   message generation slot here.
         */
        return delegated TIAction(obj, whichMessageObject,
                                  resolvedAllObjects);
    }

    whatObj(which)
    {
        /* use the same handling we use for a regular two-object action */
        return delegated TIAction(which);
    }
    whatCase(which)
    {
        /* use the same handling we use for a regular two-object action */
        return delegated TIAction(which);
    }

    getQuestionInf(which)
    {
        /* use the same handling as for a regular two-object action */
        return delegated TIAction(which);
    }

    getOtherMessageObjectPronoun(which)
    {
        /*
         *   If we're asking about the topic, then the other pronoun is
         *   for the resolved object, which is always in the direct object
         *   slot.  If we're asking about the resolved object, then return
         *   a pronoun for the topic.
         */
        if (which == whichMessageTopic)
        {
            /*
             *   we want the pronoun for the resolved object, which is
             *   always in the direct object slot (so the 'other' slot is
             *   effectively the indirect object)
             */
             // zrejme: "na co *se* chceš zeptat"/"o co chceš požádat"
            return delegated TIAction(IndirectObject);
        }
        else
        {
            /* return a generic pronoun for the topic */
            /*
             *   "Koho *se* chceš zeptat?" Ale také "Koho se chceš požádat?"
             *   Proto tady nebudu vracet nic (jinak je tenhle mechanismus
             *   vyuzivan na "Komu *ji* chces dat", tak ho nemuzu cely
             *   odstranit.
             */
            
            return '';
        }
    }

    getVerbPhrase(inf, ctx)
    {
        local dobj, dobjText, dobjIsPronoun;
        local topicText;
        local ret;

        /* use the default context if one wasn't supplied */
        if (ctx == nil)
            ctx = defaultGetVerbPhraseContext;

        /* get the direct object information */
        dobj = getDobj();
        dobjText = ctx.objName(dobj, 1);
        dobjIsPronoun = ctx.isObjPronoun(dobj);

        /* get our topic phrase */
        topicText = getTopic().getTopicText().toLower();

        /*
         *   Use the standard two-object phrasing.  The order of the
         *   phrasing depends on whether our topic phrase is in the direct
         *   or indirect object slot.
         */
        if (whichMessageTopic == DirectObject)
            ret = TIAction.getVerbPhrase2(inf, verbPhrase,
                                          topicText, nil, dobjText);
        else
            ret = TIAction.getVerbPhrase2(inf, verbPhrase,
                                          dobjText, dobjIsPronoun, topicText);

        /* use the direct object as the antecedent for the next phrase */
        ctx.setPronounObj(dobj);

        /* return the result */
        return ret;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Verbs.
 *
 *   The actual body of each of our verbs is defined in the main
 *   language-independent part of the library.  We only define the
 *   language-specific grammar rules here.
 */

VerbRule(Take)
    ('vezmi' | 'vem' | 'vzít' | 'seber' | 'ber' | 'seb' | 'sebrat' | 'zvedni'
    | 'zvednout' | 'uchop' | 'uchopit' | 'popadni' | 'popadnout') ( | 'si')
    dobjList
    : TakeAction
    verbPhrase = 'vzít/ber{eš}/vzal{a} (co)'
;

VerbRule(TakeFrom)
    ('vezmi' | 'vem' | 'vzít' | 'vyndej' | 'vyndat' | 'vyjmi' | 'vyjmout'
    'vysuň' | 'vysunout' | 'odstraň' | 'odstranit') ( | 'si') dobjList
    ('z' | 'ze') singleIobj
    | ('vezmi' | 'vem' | 'vzít' | 'vyndej' | 'vyndat' | 'vyjmi' | 'vyjmout'
    | 'vysuň' | 'vysunout' | 'odstraň' | 'odstranit') ( | 'si') ('z' | 'ze')
    singleIobj dobjList
    : TakeFromAction
    verbPhrase = 'vzít/ber{eš}/vzal{a} (co) (z čeho)'
    askIobjResponseProd = outOfSingleNoun
;

VerbRule(Remove)
    ('vyndej' | 'vyndat' | 'vyjmi' | 'vyjmout' | 'vysuň' | 'vysunout'
    | 'odstraň' | 'odstranit') dobjList
    : RemoveAction
    verbPhrase = 'vyndat/vyndav{áš}/vyndal{a} (co)'
    askIobjResponseProd = inSingleNoun
;

VerbRule(Drop)
    ('polož' | 'položit' | 'pol' | 'odlož' | 'odložit' | 'upusť' | 'upustit')
    ( | 'si') dobjList
    : DropAction
    verbPhrase = 'odložit/odklád{áš}/odložil{a} (co)'
;

VerbRule(Examine)
    ('p' | 'pro' | 'prozkoumej' | 'prozkoumat' | 'zkoumej' | 'zkoumat'
    | 'pozoruj' | 'pozorovat' 
    | ('podívej' | 'podívat' | 'koukni' | 'koukej' | 'kouknout') ('se' | ) 'na'
    | ('prohlédni' | 'prohlédnout') ('si' | ))
    dobjList
    : ExamineAction
    verbPhrase = 'prozkoumat/zkoum{áš}/prozkoumal{a} (co)'
;

VerbRule(Read)
    ('přečti' | 'přečíst' | 'pře' | 'pročti' | 'pročíst' | 'čti' | 'číst')
    ( | 'si') dobjList
    : ReadAction
    verbPhrase = 'přečíst/čt{eš}/přečetl{a} (co)'
;

VerbRule(LookIn)
    ('koukni' | 'kouknout' | 'koukej' | 'koukat' | 'podívej' | 'podívat'
    | 'dívej' | 'dívat') ('se' | ) 'do' dobjList
    | ('nahlédni' | 'nahlížej' | 'nahlédnout') 'do' dobjList
    : LookInAction
    verbPhrase = 'kouknout/kouk{áš}/kouknul{a} (do čeho)'
;

VerbRule(Search)
    ('prohledej' | 'prohledat') dobjList
    : SearchAction
    verbPhrase = 'prohledat/prohledáv{áš}/prohledal{a} (co)'
;

/*
 *   TODO: "Koukni *do* okna" je také logické, přitom se ale plete s LookIn.
 *   Budeme to řešit, nebo bude na autorovi si udělat remap?
 */
VerbRule(LookThrough)
    ('koukni' | 'kouknout' | 'koukej' | 'koukat' | 'nahlédni' | 'nahlédnout'
    | 'podívej' | 'podívat' | 'pohlédni' | 'pohlédnout') ('se' | )
    ('skrz' | 'skrze' | ) dobjList
    | ('vykoukni' | 'vykouknout' | 'vyhlédni' | 'vyhlédnout') ('ven' | )
    ('z' | 'ze') dobjList
    : LookThroughAction
    verbPhrase = 'kouknout/kouk{áš}/koukl{a} (skrz co)'
;

VerbRule(LookUnder)
    ('podívej' | 'podívat' | 'koukni' | 'kouknount' | 'koukej' | 'koukat' |
    'nahlédni' | 'nahlédnout') ( | 'se') 'pod' dobjList
    : LookUnderAction
    verbPhrase = 'kouknout/kouk{áš}/koukl{a} (pod co)'
;

VerbRule(LookBehind)
    ('koukni' | 'kouknount' | 'koukej' | 'koukat' | 'nahlédni' | 'nahlédnout')
    'za' dobjList
    : LookBehindAction
    verbPhrase = 'kouknout/kouk{áš}/koukl{a} (za co)'
;

VerbRule(Feel)
    ('dotkni' | 'dotknout') 'se' dobjList
    | ('osahej' | 'osahat' | 'ohmatej' | 'ohmatat' | 'prohmatej' | 'prohmatat')
    dobjList
    | ('sáhni' | 'sáhnout') ('si' | ) ('na' | 'do' | ) dobjList
    : FeelAction
    verbPhrase = 'dotknout/dotýk{áš}/dotkl{a} (čeho) se'
    askDobjResponseProd = onSingleNoun
;

VerbRule(FeelWhat)
    ('dotkni' | 'dotknout') 'se'
    | ('osahej' | 'osahat' | 'ohmatej' | 'ohmatat' | 'prohmatej' | 'prohmatat')
    | ('sáhni' | 'sáhnout') ('si' | )
    : FeelAction
    verbPhrase = 'dotknout/dotýk{áš}/dotkl{a} (čeho) se'
    askDobjResponseProd = onSingleNoun
    construct()
    {
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = onSingleNoun;
    }
;


VerbRule(Taste)
    ('ochutnej' | 'ochutnat') dobjList
    : TasteAction
    verbPhrase = 'ochutnat/ochutnáv{áš}/ochutnal{a} (co)'
;

VerbRule(Smell)
    ('čichej' | 'čichni' | 'čichat' | 'čuchej' | 'čuchat' | 'očichej'
    | 'očichat' | 'čmuchej' | 'čmuchat' | 'čenichej' | 'čenichat' | 'přivoň'
    | 'přivonět' | 'přičichni' | 'přičichnout') ('si' | ) ('k' | 'ke' | )
    dobjList
    : SmellAction
    verbPhrase = 'přičichnout/cít{íš}/cítil{a} (k čemu)'

    /*
     *   use the "not aware" version of the no-match message - the object
     *   of SMELL is often intangible, so the default "you can't see that"
     *   message is often incongruous for this verb
     */
    noMatch(msgObj, actor, txt) { msgObj.noMatchNotAware(actor, txt); }
;

VerbRule(SmellImplicit)
    ('čichej' | 'čichni' | 'čichat' | 'čuchej' | 'čuchat' | 'čmuchej'
    | 'čmuchat' | 'čenichej' | 'čenichat') ('si' | )
    : SmellImplicitAction
    verbPhrase = 'čichat/cít{íš}/cítil{a}'
;

VerbRule(ListenTo)
    ('poslouchej' | 'poslouchat' | 'naslouchej' | 'naslouchat' | 'slyš'
    | 'slyšet') ('k' | 'ke' | ) dobjList
    : ListenToAction
    verbPhrase = 'naslouchat/naslouch{áš}/naslouchal{a} (čemu)'

    /*
     *   use the "not aware" version of the no-match message - the object
     *   of LISTEN TO is often intangible, so the default "you can't see
     *   that" message is often incongruous for this verb
     */
    noMatch(msgObj, actor, txt) { msgObj.noMatchNotAware(actor, txt); }
;

VerbRule(ListenImplicit)
    'poslouchej' | 'poslouchat' | 'naslouchej' | 'naslouchat'
    : ListenImplicitAction
    verbPhrase = 'naslouchat/naslouch{áš}/naslouchal{a}'
;

VerbRule(PutIn)
    ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol' | 'vlož'
    | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč' | 'strčit'
    | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    dobjList 'do' singleIobj
    | ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol' 
    | 'vlož' | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč' 
    | 'strčit' | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    'do' singleIobj dobjList
    : PutInAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (do čeho)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(PutOn)
    ('dej' | 'dát' | 'polož' | 'položit' | 'pol'  | 'umísti' | 'umístit'
    | 'přidej' | 'přidat') dobjList 'na' singleIobj
    | ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol'
    | 'umísti' | 'umístit' | 'přidej' | 'přidat') 'na' singleIobj dobjList
    : PutOnAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (na co)'
    askIobjResponseProd = onSingleNoun
;

VerbRule(PutUnder)
    ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol' | 'vlož'
    | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč' | 'strčit'
    | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    dobjList 'pod' singleIobj
    | ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol' 
    | 'vlož' | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč'
    | 'strčit' | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    'pod' singleIobj dobjList
    : PutUnderAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (pod co)'
;

VerbRule(PutBehind)
    ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol' | 'vlož'
    | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč' | 'strčit'
    | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    dobjList 'za' singleIobj
    | ('dej' | 'dát' | 'zandej' | 'zandat' | 'polož' | 'položit' | 'pol'
    | 'vlož' | 'vložit' | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč'
    | 'strčit' | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit')
    'za' singleIobj dobjList    : PutBehindAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (za co)'
;

VerbRule(PutInWhat)
    [badness 500] ('dej' | 'dát' | 'zandej' | 'zandat' | 'vlož' | 'vložit'
    | 'umísti' | 'umístit' | 'přidej' | 'přidat' | 'strč' | 'strčit'
    | 'vsuň' | 'vsunout' | 'zasuň' | 'zasunout' | 'vraž' | 'vrazit') dobjList
    : PutInAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (do čeho)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = toSingleNoun;
    }
;

/* Dame lehky badness, aby "obleci se" bylo odchyceno WearWhat. */
VerbRule(Wear)
    [badness 10] ('oblékni' | 'obléknout' | 'obleč' | 'obléci' | 'obléct'
    | 'navlékni' | 'navkléknout' | 'obuj' | 'obout' | 'nazuj' | 'nazout')
    ('se' ('do' | ) | 'si' ('na' 'sebe' | ) | ) dobjList
    | ('nandej' | 'nandat' | 'nasaď' | 'nasadit' )
    ('si' ('na' 'sebe' | ) | ) dobjList
    | ('vezmi' | 'vem' | 'vzít' | 'dej' | 'dát') ('si' | ) 'na' 'sebe' dobjList
    : WearAction
    verbPhrase = 'nandat/nandav{áš}/nandal{a} (co) si'
;

VerbRule(WearWhat)
    ('oblékni' | 'obléknout' | 'obleč' | 'obléci' | 'obléct' | 'navlékni'
    | 'navkléknout' | 'obuj' | 'obout' | 'nazuj' | 'nazout') 'se'
    | ('nandej' | 'nandat' | 'nasaď' | 'nasadit') 'si' 
    : WearAction
    verbPhrase = 'nandat/nandav{áš}/nandal{a} (co) si'

    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(Doff)
    [badness 10] ('svlékni' | 'svléknout' | 'svleč' | 'svléci' | 'svléct')
    ('se' ('z' | 'ze' | ) | 'si' ('ze' 'sebe' | ) | ) dobjList
    | ('sundej' | 'sundat' | 'zuj' | 'zout')
    ('si' ('ze' 'sebe' | ) | ) dobjList
    : DoffAction
    verbPhrase = 'sundat/sundav{áš}/sundal{a} (co) si'
;

VerbRule(DoffWhat)
    ('svlékni' | 'svléknout' | 'svleč' | 'svléci' | 'svléct') 'se'
    | ('sundej' | 'sundat' | 'zuj' | 'zout') 'si'
    : DoffAction
    verbPhrase = 'sundat/sundav{áš}/sundal{a} (co) si'
    
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(Kiss)
    ('polib' | 'políbit') singleDobj
    : KissAction
    verbPhrase = 'políbit/líb{áš}/líbal{a} (koho)'
;

VerbRule(AskFor)
    ('požádej' | 'požádat' | 'popros' | 'poprosit') singleDobj 'o' singleTopic
    | ('řekni' | 'řeknout' | 'říci' | 'říct') 'si' 'o' singleTopic singleDobj
    | ('půjč' | 'půjčit' | 'vypůjč' | 'vypůjčit') 'si' singleTopic 'od' singleDobj
    | ('půjč' | 'půjčit' | 'vypůjč' | 'vypůjčit') 'si' 'od' singleDobj singleTopic
    : AskForAction
    verbPhrase = 'požádat/žád{áš}/požádal{a} (koho) (o co)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = forSingleNoun
;

VerbRule(AskWhomFor)
    ('požádej' | 'požádat' | 'popros' | 'poprosit' | ('řekni' | 'řeknout' | 'říci' | 'říct' ) 'si')
    'o' singleTopic
    | ('půjč' | 'půjčit' | 'vypůjč' | 'vypůjčit') 'si' singleTopic
    : AskForAction
    verbPhrase = 'požádat/žád{áš}/požádal{a} (koho) (o co)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

/*
 *   Zde musi byt 'se' povinne, protoze jinak by mohl ">zeptej se na loď" chapat
 *   jako ze 'se' je singleDobj, tj. odkazuje na hrace a prohlasit, ze mluvenim
 *   se sebou samym niceho nedosahnes.
 *
 *   TODO: Dame velmi mirny badness proto, aby ">zeptat se na identifikacni
 *   kartu" nepochopil identifikacni jako singleTopic a kartu jako singleDobj.
 *   To ale rozbije opacnou situaci ">zeptat se na identifikační kartu borise"
 */

VerbRule(AskAbout)
    ('zeptej' | 'zeptat' | 'poptej' | 'poptat' | 'ptej' | 'ptát')
    'se' singleDobj 'na' singleTopic
    : AskAboutAction
    verbPhrase = 'zeptat/pt{áš}/zeptal{a} (koho) (na co) se'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
;

VerbRule(AskAboutType2)
    [badness 10] ('zeptej' | 'zeptat' | 'poptej' | 'poptat' | 'ptej' | 'ptát')
    'se' 'na' singleTopic singleDobj
    : AskAboutAction
    verbPhrase = 'zeptat/pt{áš}/zeptal{a} (koho) (na co) se'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
;

VerbRule(AskAboutImplicit)
    ('zeptej' | 'zeptat' | 'poptej' | 'poptat' | 'ptej' | 'ptát') 'se' 'na'
    singleTopic
    | 'zep' ( | 'se') ( | 'na') singleTopic
    : AskAboutAction
    verbPhrase = 'zeptat/pt{áš}/zeptal{a} (koho) (na co) se'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(AskAboutWhat)
    [badness 500] ('zeptej' | 'zeptat' | 'poptej' | 'poptat' | 'ptej' | 'ptát')
    'se' singleDobj
    : AskAboutAction
    verbPhrase = 'zeptat/pt{áš}/zeptal{a} (koho) (na co) se'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty topic phrase */
        topicMatch = new EmptyNounPhraseProd();
        topicMatch.responseProd = aboutTopicPhrase;
    }
;

/* 
 *   Trochu nižší rank, aby dostala přednost intepretace AskFor u fráze "řekni
 *   *si* o bonbóny"
 */
VerbRule(TellAbout)
    [badness 10] ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět')
    (singleDobj 'o' singleTopic | 'o' singleTopic singleDobj)
    : TellAboutAction
    verbPhrase = 'říci/řík{áš}/řekl{a} (komu) (o čem)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

/*
 *   Nižší rank oproti TellAbout, aby "řekni o součástkách veliteli" nebylo
 *   pochopeno jako TellAboutImplicit s topicem "součástkách veliteli".
 */
VerbRule(TellAboutImplicit)
    [badness 15] ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět')
    'o' singleTopic
    | 'řek' ( | 'o') singleTopic
    : TellAboutAction
    verbPhrase = 'říci/řík{áš}/řekl{a} (komu) (o čem)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(TellAboutWhat)
    [badness 500] ('řekni' | 'říci' | 'pověz' | 'povědět') singleDobj
    : TellAboutAction
    verbPhrase = 'říci/řík{áš}/řekl{a} (komu) (o čem)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty topic phrase */
        topicMatch = new EmptyNounPhraseProd();
        topicMatch.responseProd = aboutTopicPhrase;
    }
;

//VerbRule(AskVague)
//    [badness 500] 'ask' singleDobj singleTopic
//    : AskVagueAction
//    verbPhrase = 'ask/asking (whom)'
//;

//VerbRule(TellVague)
//    [badness 500] 'tell' singleDobj singleTopic
//    : AskVagueAction
//    verbPhrase = 'tell/telling (whom)'
//;

VerbRule(TalkTo)
    ('pozdrav' | 'pozdravit' | 'poz' | 'oslov' | 'oslovit' | ('mluv' | 'mluvit'
    | 'promluv' | 'promluvit') ('s' | 'se' | 'na')) singleDobj
    : TalkToAction
    verbPhrase = 'mluvit/mluv{íš}/mluvil{a} (s kým)'
    askDobjResponseProd = singleNoun
;

VerbRule(TalkToWhat)
    [badness 500] 'pozdrav' | 'pozdravit' | 'mluv' | 'mluvit' | 'promluv'
    | 'promluvit' | 'oslov' | 'oslovit'
    : TalkToAction
    verbPhrase = 'mluvit/mluv{íš}/mluvil{a} (s kým)'
    askDobjResponseProd = withSingleNoun

    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = withSingleNoun;
    }
;

VerbRule(Topics)
    ('ukaž' | 'ukázat' | 'zobraz' | 'zobrazit' | 'vypiš' | 'vypsat' | )
    ('témata' | 'téma' | 'tém' | 't' | 'návrhy' | 'doporučení')
    : TopicsAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukazoval{a} témata'
;

VerbRule(Hello)
    ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět' | ) ('ahoj'
    | 'nazdar' | 'čau' | 'dobrý' 'den') | ('pozdrav' | 'pozdravit') ('se' | )
    | ('začni' | 'začít' | 'zahaj' | 'zahájit') ('hovor' | 'rozhovor')
    : HelloAction
    verbPhrase = 'pozdravit/zdrav{íš}/pozdravil{a}'
;

VerbRule(Goodbye)
    ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět' | ) ('nasledanou'
    | 'sbohem') | ('rozluč' | 'rozloučit') ('se' | )
    | ('ukonči' | 'ukončit' | 'skonči' | 'skončit') ('hovor' | 'rozhovor')
    : GoodbyeAction
    verbPhrase = 'rozloučit/louč{íš}/rozloučil{a} se'
;

VerbRule(Yes)
    ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět' | 'odpověz'
    | 'odpovědět' | ) ('ano' | 'jo')
    : YesAction
    verbPhrase = 'říci/řík{áš}/řekl{a} ano'
;

VerbRule(No)
    ('řekni' | 'řeknout' | 'říci' | 'říct' | 'pověz' | 'povědět' | 'odpověz'
    | 'odpovědět' | ) 'ne'
    : NoAction
    verbPhrase = 'říci/řík{áš}/řekl{a} ne'
;

VerbRule(Yell)
    'křič' | 'křičet' | 'zakřič' | 'zakřičet' | 'zaječ' | 'zaječet' | 'zazřvi'
    | 'zařvat' | 'zavolej' | 'zavolat'
    : YellAction
    verbPhrase = 'zakřičet/křič{íš}/zakřičel{a}'
;

VerbRule(GiveTo)
    ('dej' | 'dát' | 'předej' | 'předat' | 'podej' | 'podat' | 'nabídni'
    | 'nabídnout')
    (singleIobj dobjList | dobjList singleIobj)
    : GiveToAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (komu)'
    askIobjResponseProd = onSingleNoun
    isPrepositionalPhrasing = nil
    preferredIobj = Actor
;

/*
 * Tady vlastne (komucemu) me vaze na urceni padu v ">dej minci" "(mladé
 * prodavačce)". To se urcuje kolem radku 8680. Jenze viz 9660 je obsah
 * zavorky pouzivan take ve vetach typu "Do čeho chceš vstoupit?"
 * Druhá závorka (komu) se použije "Komu chceš dát xxx"
 *
 *   Zde umyslne nedame 'dej' a 'dát', misto toho presmerujeme
 *   PutIn(Actor) -> GiveTo(Actor)
 */
VerbRule(GiveToWhom)
    ('dej' | 'dát' | 'předej' | 'předat' | 'podej' | 'podat' | 'nabídni'
    | 'nabídnout') dobjList
    : GiveToAction
    verbPhrase = 'dát/dáv{áš}/dal{a} (co) (komu)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new ImpliedActorNounPhraseProd();
        iobjMatch.responseProd = onSingleNoun;
    }
;

VerbRule(ShowTo)
    ('ukaž' | 'ukázat') (singleIobj dobjList | dobjList singleIobj)
    : ShowToAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} (co) (komu)'
    askIobjResponseProd = singleNoun

    /* this is a non-prepositional phrasing */
    isPrepositionalPhrasing = nil
    preferredIobj = Actor
;

VerbRule(ShowToWhom)
    ('ukaž' | 'ukázat') dobjList
    : ShowToAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} (co) (komu)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new ImpliedActorNounPhraseProd();
        iobjMatch.responseProd = singleNoun;
    }
;

VerbRule(Throw)
    ('hoď' | 'hodit' | 'zahoď' | 'zahodit' | 'vrhni' | 'vrhnout') dobjList
    : ThrowAction
    verbPhrase = 'hodit/ház{íš}/hodil{a} (co)'
;

VerbRule(ThrowAt)
    ('hoď' | 'hodit' | 'vrhni' | 'vrhnout') dobjList
    ('na' | 'po' | 'do') singleIobj
    : ThrowAtAction
    verbPhrase = 'hodit/ház{íš}/hodil{a} (co) (na co)'
    askIobjResponseProd = onSingleNoun
;

VerbRule(ThrowTo)
    ('hoď' | 'hodit') (singleIobj dobjList | dobjList singleIobj)
    : ThrowToAction
    verbPhrase = 'hodit/ház{íš}/hodil{a} (co) (komu)'
    askIobjResponseProd = singleNoun

    /* this is a non-prepositional phrasing */
    isPrepositionalPhrasing = nil
    preferredIobj = Actor
;

VerbRule(ThrowToType2)
    ('hoď' | 'hodit') singleIobj dobjList
    : ThrowToAction
    verbPhrase = 'hodit/ház{íš}/hodil{a} (co) (komu)'
    askIobjResponseProd = singleNoun

    /* this is a non-prepositional phrasing */
    isPrepositionalPhrasing = nil
;

VerbRule(ThrowDir)
    ('hoď' | 'hodit' | 'vrhni' | 'vrhnout') dobjList ('na' | ) singleDir
    : ThrowDirAction
    /* TODO: Je tu směr správně? Nepotřebuje doplnit předložku? */
    verbPhrase = ('hodit/ház{íš}/hodil{a} (co) ' + dirMatch.dir.name)
;

/* a special rule for THROW DOWN <dobj> */
VerbRule(ThrowDirDown)
    ('odhoď' | 'odhodit' | 'shoď' | 'shodit' | 'hoď' | 'hodit')
    ( | 'dolů' | 'dolu') dobjList
    : ThrowDirAction
    verbPhrase = 'odhodit/odhazu{ješ}/odhodil{a} (co)'

    /* the direction is fixed as 'down' for this phrasing */
    getDirection() { return downDirection; }
;

VerbRule(Follow)
    ('následuj' | 'následovat' | 'sleduj' | 'sledovat') singleDobj
    | ('jdi' | 'jít' | 'běž') 'za' singleDobj
    : FollowAction
    verbPhrase = 'následovat/následu{ješ}/následoval{a} (koho)'
    askDobjResponseProd = singleNoun
;

VerbRule(Attack)
    ('zaútoč' 'na' | 'zaútočit' 'na' | 'zabij' | 'zabít' | 'zasáhni'
    | 'zasáhnout' | 'zmlať' | 'zmlátit') singleDobj
    | ('kopni' | 'kopnout' | 'bouchni' | 'bouchnout' | 'udeř' | 'udeřit'
    | 'uhoď' | 'uhodit' ) ('si' | ) ('do' | ) singleDobj
    : AttackAction
    verbPhrase = 'zaútočit/útoč{íš}/zaútočil{a} (na koho)'
    askDobjResponseProd = onSingleNoun
;

VerbRule(AttackWith)
    ('zaútoč' | 'zaútočit') 'na' singleDobj singleIobj
    | ('zaútoč' | 'zaútočit') singleIobj 'na' singleDobj
    : AttackWithAction
    verbPhrase = 'zaútočit/útoč{íš}/zaútočil{a} (na koho) (čím)'
    askDobjResponseProd = onSingleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(AttackWithType2)
    ('zabij' | 'zabít' | 'zasáhni' | 'zasáhnout' | 'udeř' | 'udeřit' | 'kopni'
    | 'kopnout' | 'bouchni' | 'bouchnout')
    (singleIobj singleDobj | singleDobj singleIobj)
    : AttackWithAction
    verbPhrase = 'zaútočit/útoč{íš}/zaútočil{a} (na koho) (čím)'
    askDobjResponseProd = onSingleNoun
    askIobjResponseProd = singleNoun
    preferredIobj = Actor
;

VerbRule(Inventory)
    'i' | 'inv' | 'inventář'
    : InventoryAction
    verbPhrase = 'prohlídnout/prohlíž{íš}/prohlídl{a} i inventář'
;

VerbRule(InventoryTall)
    'i' 'na' 'výšku' | 'inv' 'na' 'výšku' | 'inventář' 'na' 'výšku'
    | 'vysoký' 'inventář' | 'vysoký' 'inv' | 'vysoký' 'i'
    : InventoryTallAction
    verbPhrase = 'prohlídnout/prohlíž{íš}/prohlídl{a} inventář na výšku'
;

VerbRule(InventoryWide)
    'i' 'na' 'šířku' | 'inv' 'na' 'šířku' | 'inventář' 'na' 'šířku'
    | 'široký' 'inventář' | 'široký' 'inv' | 'široký' 'i'
    : InventoryWideAction
    verbPhrase = 'prohlídnout/prohlíž{íš}/prohlídl{a} inventář na šířku'
;

VerbRule(Wait)
    'č' | 'ček' | 'čekej' | 'čekat'
    : WaitAction
    verbPhrase = 'počkat/ček{áš}/počkal{a}'
;

VerbRule(Look)
    ('rozhlédni' | 'rozhlédnout' | 'porozhlédni' | 'porozhlédnout') ('se' | )
    | 'roz' | 'r' | 'situace' | 'sit'
    : LookAction
    verbPhrase = 'rozhlédnout/rozhlíž{íš}/rozhlédl{a} se'
;

VerbRule(Quit)
    'konec' | 'kon'
    : QuitAction
    verbPhrase = 'ukončit/ukonču{ješ}/ukončil{a}'
;

VerbRule(Again)
    'opakuj' | 'o' | 'znovu' | 'zno' | 'zn'
    : AgainAction
    verbPhrase = 'zopakovat/opaku{ješ}/zopakoval{a} poslední příkaz'
;

VerbRule(Footnote)
    ('poznámka' | 'poz') singleNumber
    : FootnoteAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} poznámku pod čarou'
;

VerbRule(FootnotesFull)
    'poznámky' 'naplno'
    : FootnotesFullAction
    verbPhrase = 'zapnout/zapín{áš}/zapnul{a} všechny poznámky'
;

VerbRule(FootnotesMedium)
    'poznámky' 'středně'
    : FootnotesMediumAction
    verbPhrase = 'zapnout/zapín{áš}/zapnul{a} nové poznámky'
;

VerbRule(FootnotesOff)
    'poznámky' 'vypnuté'
    : FootnotesOffAction
    verbPhrase = 'skrýt/skrýv{áš}/skryl{a} poznámky'
;

VerbRule(FootnotesStatus)
    'poznámky' | 'nastavení' 'poznámek'
    : FootnotesStatusAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} nastavení poznámek'
;

VerbRule(TipsOn)
    ('zapni' | 'zapnout' | 'zap') 'tipy'
    : TipModeAction

    stat_ = true

    verbPhrase = 'zapnout/zapín{áš}/zapnul{a} tipy'
;

VerbRule(TipsOff)
    ('vypni' | 'vypnout' | 'vyp') 'tipy'
    : TipModeAction

    stat_ = nil

    verbPhrase = 'vypnout/vypín{áš}/vypnul{a} tipy'
;

VerbRule(Verbose)
    'upovídaný'
    : VerboseAction
    verbPhrase = 'přepnout/přepín{áš}/přepnul{a} na UPOVÍDANÝ režim'
;

VerbRule(Terse)
    'stručný'
    : TerseAction
    verbPhrase = 'přepnout/přepín{áš}/přepnul{a} na STRUČNÝ režim'
;

VerbRule(Score)
    'skóre' | 'body'
    : ScoreAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} skóre'
;

VerbRule(FullScore)
    ('detailní' | 'podrobné' | 'rozepiš') 'skóre'
    : FullScoreAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} detailní skóre'
;

VerbRule(Notify)
    ('nastavení' | ) 'upozorňování'
    : NotifyAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} nastavení upozorňování'
;

VerbRule(NotifyOn)
    ('zapni' | 'zapnout' | 'zap') 'upozorňování'
    : NotifyOnAction
    verbPhrase = 'zapnout/zapín{áš}/zapnul{a} upozorňování na skóre'
;

VerbRule(NotifyOff)
    ('vypni' | 'vypnout' | 'vyp') 'upozorňování'
    : NotifyOffAction
    verbPhrase = 'vypnout/vypín{áš}/vypnul{a} upozorňování na skóre'
;

VerbRule(Save)
    ('ulož' | 'uložit') ('hru' | 'příběh' | 'pozici' | )
    : SaveAction
    verbPhrase = 'uložit/uklád{áš}/uložil{a}'
;

VerbRule(SaveString)
    ('ulož' | 'uložit') ('hru' | 'příběh' | 'pozici' | )
    ('do' ('souboru' | ) | ) quotedStringPhrase->fname_
    : SaveStringAction
    verbPhrase = 'uložit/uklád{áš}/uložil{a}'
;

VerbRule(Restore)
    ('načti' | 'načíst' | 'nahraj' | 'nahrát'| 'nahrej' )
    ('hru' | 'příběh' | 'pozici' | )
    : RestoreAction
    verbPhrase = 'načíst/načít{áš}/načetl{a}'
;

VerbRule(RestoreString)
    ('načti' | 'načíst' | 'nahraj' | 'nahrát' | 'nahrej')
    ('ze' ('souboru' | 'z' | ) | ) quotedStringPhrase->fname_
    : RestoreStringAction
    verbPhrase = 'načíst/načít{áš}/načetl{a}'
;

VerbRule(SaveDefaults)
    ('ulož' | 'uložit') 'nastavení'
    : SaveDefaultsAction
    verbPhrase = 'uložit/uklád{áš}/uložil{a} výchozí nastavení'
;

VerbRule(RestoreDefaults)
    ('načti' | 'načíst' | 'nahraj' | 'nahrát' | 'nahrej') 'nastavení'
    : RestoreDefaultsAction
    verbPhrase = 'načíst/načít{áš}/načetl{a} výchozí nastavení'
;

VerbRule(Restart)
    'restartuj' | 'restartovat'
    : RestartAction
    verbPhrase = 'restartovat/restartu{ješ}/restartoval{a}'
;

VerbRule(Pause)
    'pauza'
    : PauseAction
    verbPhrase = 'pozastavit/pozastavu{ješ}/pozastavil{a} hru'
;

VerbRule(Undo)
    ('vrať' | 'vrátit') 'tah'
    | ('odvolej' | 'odvolat') ('tah' | )
    | 'krok' 'zpět'
    : UndoAction
    verbPhrase = 'vrátit/vrac{íš}/vrátil{a} tah'
;

VerbRule(Version)
    'verze'
    : VersionAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} verzi'
;

VerbRule(Credits)
    'zásluhy' | 'autor' | 'autoři' | 'titulky'
    : CreditsAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} titulky'
;

VerbRule(About)
    'o' 'hře' | 'o' 'příběhu' | 'informace' | 'info'
    : AboutAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} informace o příběhu'
;

VerbRule(Script)
    ('zapni' | 'zapnout' | ) ('zápis' | 'zapisování')
    : ScriptAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} zápis průběhu hry'
;

VerbRule(ScriptString)
    'zápis' quotedStringPhrase->fname_
    : ScriptStringAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} zápis průběhu hry'
;

VerbRule(ScriptOff)
    ('vypni' | 'vypnout') ('zápis' | 'zapisování')
    : ScriptOffAction
    verbPhrase = 'ukončit/ukonču{ješ}/ukončil{a} zápis průběhu hry'
;

VerbRule(Record)
    ('zapni' | 'zapnout' | ) 'záznam'
    : RecordAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} záznam příkazů'
;

VerbRule(RecordString)
    'záznam' quotedStringPhrase->fname_
    : RecordStringAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} záznam příkazů'
;

VerbRule(RecordEvents)
    ('zapni' | 'zapnout' | ) 'záznam' 'událostí'
    : RecordEventsAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} záznam událostí'
;

VerbRule(RecordEventsString)
    'záznam' 'událostí' quotedStringPhrase->fname_
    : RecordEventsStringAction
    verbPhrase = 'zahájit/zahaju{ješ}/zahájil{a} záznam událostí'
;

VerbRule(RecordOff)
    ('vypni' | 'vypnout') 'záznam'
    : RecordOffAction
    verbPhrase = 'ukončit/ukonču{ješ}/ukončil{a} záznam'
;

VerbRule(ReplayString)
    ('přehraj' | 'přehrát') ('quiet'->quiet_ | 'nonstop'->nonstop_ | )
        (quotedStringPhrase->fname_ | )
    : ReplayStringAction
    verbPhrase = 'přehrát/přehráváš/přehrála zaznamenané příkazy'

    /* set the appropriate option flags */
    scriptOptionFlags = ((quiet_ != nil ? ScriptFileQuiet : 0)
                         | (nonstop_ != nil ? ScriptFileNonstop : 0))
;
VerbRule(ReplayQuiet)
    ('přehraj' | 'přehrát') 'potichu' (quotedStringPhrase->fname_ | )
    : ReplayStringAction

    scriptOptionFlags = ScriptFileQuiet
;

VerbRule(VagueTravel) 'jdi' | 'jít' | 'běž' : VagueTravelAction
    verbPhrase = 'jít/jd{eš}/{šel}'
;

VerbRule(Travel)
    ('jdi' | 'jít' | 'běž' | ) ('na' | ) singleDir
    : TravelAction
    verbPhrase = ('jít/jd{eš}/{šel} ' + dirMatch.dir.name)
;

/*
 *   Create a TravelVia subclass merely so we can supply a verbPhrase.
 *   (The parser looks for subclasses of each specific Action class to find
 *   its verb phrase, since the language-specific Action definitions are
 *   always in the language module's 'grammar' subclasses.  We don't need
 *   an actual grammar rule, since this isn't an input-able verb, so we
 *   merely need to create a regular subclass in order for the verbPhrase
 *   to get found.)  
 */
class EnTravelVia: TravelViaAction
    verbPhrase = 'projít/procház{íš}/pro{šel} (čím)'
;

VerbRule(Port)
    (('jdi' | 'jít' | 'běž' | ) 'na' | ) 'levobok'
    : PortAction
    dirMatch: DirectionProd { dir = portDirection }
    verbPhrase = 'jít/jd{eš}/{šel} na levobok'
;

VerbRule(Starboard)
    (('jdi' | 'jít' | 'běž' | ) 'na' | ) 'pravobok'
    : StarboardAction
    dirMatch: DirectionProd { dir = starboardDirection }
    verbPhrase = 'jít/jd{eš}/{šel} na pravobok'
;

VerbRule(In)
    'vejdi' | 'vejít' | 'vstup' | 'vstoupit' | 'do' | 'dov'
    | ('jdi' | 'jít' | 'běž' | ) 'dovnitř'
    : InAction
    dirMatch: DirectionProd { dir = inDirection }
    verbPhrase = 'vejít/vcház{íš}/ve{šel}'
;

VerbRule(Out)
    'odejdi' | 'odejít' | 'vyjdi' | 'vyjít' | 'vylez' | 'vylézt' | 'opusť'
    | 'opustit'
    | ('jdi' | 'jít' | 'běž' | 'běžet' | 'odejdi' | 'odejít' | 'vyjdi' | 'vyjít'
    | 'vylez' | 'vylézt' | ) ('ven' | 'pryč' | 'odsud')
    : OutAction
    dirMatch: DirectionProd { dir = outDirection }
    verbPhrase = 'vyjít/vycház{íš}/vy{šel}'
;

VerbRule(GoThrough)
    ('jdi' | 'jít' | 'běž' | 'lez' | 'lézt' | 'prolez' | 'prolézt'
    | ('plaz' | 'plazit') 'se' | ) ('skrz' | 'skrze' | 'po') singleDobj
    | ('projdi' | 'prijít' | 'prolez' | 'prolézt') singleDobj
    : GoThroughAction
    verbPhrase = 'projít/projd{eš}/pro{šel} (čím)'
    askDobjResponseProd = throughSingleNoun
;

VerbRule(Enter)
    ('jdi' | 'jít' | 'běž' | 'vejdi' | 'vejít' | 'vstup' | 'vstoupit' | 'vlez'
    | 'vlézt' | 'prolez' | 'prolézt' | ) (('dovnitř' | ) 'do' | 'na') singleDobj
    : EnterAction
    verbPhrase = 'vstoupit/vstupu{ješ}/vstoupil{a} (do čeho)'
    /*
     *   Chceme umožnit odpověď s předložkou: ">dovnitř Do čeho chceš vstoupit?
     *   Do čeho chceš vstoupit? >*do* pramice". Proc tady to reaguje na příkaz
     *   "dovnitř", když ten je VerbRule(In)? dobjFor(Board) asDobjFor(Enter)
     */
    askDobjResponseProd = toSingleNoun
;

VerbRule(GoBack)
    ('jdi' | 'jít' | 'běž' | ) ('zpátky' | 'zpět')
    | ('vrať' | 'vrátit') 'se'
    : GoBackAction
    verbPhrase = 'jít/jd{eš}/{šel} zpátky'
;

VerbRule(Dig)
    ('vykopej' | 'vykopat' | 'kopej' | 'kopat'
    | 'vyhrabej' | 'vyhrab' | 'vyhrabat' | 'hrabej' | 'hrabat'
    | 'vyhlub' | 'vyhloubit') 'díru' ( | 'v' | 've' | 'do') singleDobj
    : DigAction
    verbPhrase = 'vyhloubit/hloub{íš}/vyhloubil{a} díru (v čem)'
    askDobjResponseProd = inSingleNoun
;

VerbRule(DigWith)
    ('vyhrabej' | 'vyhrab' | 'vyhrabat' | 'hrabej' | 'hrabat') 'díru'
    ('v' | 've' | 'do') singleDobj ('s' | 'se' | 'pomocí' | ) singleIobj
    : DigWithAction
    verbPhrase = 'vyhloubit/hloub{íš}/vyhloubil{a} díru (v čem) (čím)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = inSingleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(Jump)
    'vyskoč' | 'vyskočit' | 'skoč' | 'skočit'
    : JumpAction
    verbPhrase = 'vyskočit/vyskaku{ješ}/vyskočil{a}'
;

VerbRule(JumpOffI)
    'seskoč' | 'seskočit'
    : JumpOffIAction
    verbPhrase = 'seskočit/seskaku{ješ}/seskočil{a}'
;

VerbRule(JumpOff)
    ('seskoč' | 'seskočit') ('z' | 'ze') singleDobj
    : JumpOffAction
    verbPhrase = 'seskočit/seskaku{ješ}/seskočil{a} (z čeho)'
    askDobjResponseProd = outOfSingleNoun
;

VerbRule(JumpOver)
    ('přeskoč' | 'přeskočit') ( | 'přes') singleDobj
    : JumpOverAction
    verbPhrase = 'přeskočit/přeskaku{ješ}/přeskočil{a} (co)'
    askDobjResponseProd = singleNoun
;

VerbRule(Push)
    ('zmáčkni' | 'zmáčknout' | 'zamáčkni' | 'zamáčknout' | 'mačkej' | 'mačkat'
    | 'stiskni' | 'stisknout' | 'stlač' | 'stlačit') dobjList
    | ('zatlač' | 'zatlačit' | 'tlač' | 'tlačit') ( | 'na' | 'do') dobjList
    : PushAction
    verbPhrase = 'zmáčknout/mačk{áš}/zmáčkl{a} (co)'
;

VerbRule(Pull)
    ('zatáhni' | 'zatahej' | 'zatáhnout' | 'táhni' | 'tahej' | 'táhnout'
    | 'vytáhni' | 'vytáhnout') ('za' | ) dobjList
    : PullAction
    verbPhrase = 'zatáhnout/táh{neš}/zatáhl{a} (za co)'
;

VerbRule(Move)
    ('odsuň' | 'odsunout' | 'posuň' | 'posunout' | 'pohni' | 'pohnout')
    ('s' | 'se' | ) dobjList
    : MoveAction
    verbPhrase = 'posunout/posouv{áš}/posunul{a} (co)'
;

VerbRule(MoveTo)
    ('odsuň' | 'odsunout' | 'posuň' | 'posunout' | 'pohni' | 'pohnout')
    ('s' | 'se' | ) dobjList ('do' | 'pod' | 'na') singleIobj
    : MoveToAction
    verbPhrase = 'posunout/posouv{áš}/posunul{a} (co) (do čeho)'
    askIobjResponseProd = toSingleNoun
    omitIobjInDobjQuery = true
;

VerbRule(MoveWith)
    ('posuň' | 'posunout') singleDobj 'pomocí' singleIobj
    | ('posuň' | 'posunout') 'pomocí' singleIobj singleDobj
    : MoveWithAction
    verbPhrase = 'posunout/posouv{áš}/posunul{a} (co) (čím)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(MoveWithType2)
    ('posuň' | 'posunout') (singleIobj singleDobj | singleDobj singleIobj)
    : MoveWithAction
    verbPhrase = 'posunout/posouv{áš}/posunul{a} (co) (čím)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    isPrepositionalPhrasing = nil
    preferredIobj = PreferredIobj
;

VerbRule(Turn)
    ('otoč' | 'otočit' | 'toč' | 'točit') ('s' | 'se' | ) dobjList
    : TurnAction
    verbPhrase = 'otočit/otáč{íš}/otočil{a} (co)'
;

VerbRule(TurnWith)
    ('otoč' | 'otočit' | 'toč' | 'točit') ('s' | 'se' | ) singleDobj 'pomocí' singleIobj
    | ('otoč' | 'otočit' | 'toč' | 'točit') 'pomocí' singleIobj ('s' | 'se' | ) singleDobj
    : TurnWithAction
    verbPhrase = 'otočit/otáč{íš}/otočil{a} (co) (čím)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(TurnWithType2)
    ('otoč' | 'otočit' | 'toč' | 'točit') (singleIobj singleDobj | singleDobj singleIobj)
    : TurnWithAction
    verbPhrase = 'otočit/otáč{íš}/otočil{a} (co) (čím)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
    isPrepositionalPhrasing = nil
    preferredIobj = PreferredIobj
;

VerbRule(TurnTo)
    ('otoč' | 'otočit' | 'natoč' | 'natočit') singleDobj
    ('na' ('číslo' | 'hodnotu' | 'polohu' | ) | 'do' 'polohy') singleLiteral
    : TurnToAction
    verbPhrase = 'natočit/natáč{íš}/natočil{a} (co) (na co)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(Set)
    ('nastav' | 'nastavit') dobjList
    : SetAction
    verbPhrase = 'nastavit/nastavu{ješ}/nastavil{a} (co)'
;

VerbRule(SetTo)
    ('nastav' | 'nastavit') singleDobj
    ('na' ('číslo' | 'hodnotu' | 'polohu' | ) | 'do' 'polohy') singleLiteral
    : SetToAction
    verbPhrase = 'nastavit/nastavu{ješ}/nastavil{a} (co) (na co)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(TypeOn)
    ('piš' | 'psát' | 'napiš' | 'napsat') ('na' | 'do' | 'v' | 've') singleDobj
    : TypeOnAction
    verbPhrase = 'napsat/píš{eš}/napsal{a} (na čem)'
;

VerbRule(TypeLiteralOn)
    ('napiš' | 'napsat') singleLiteral ('na' | 'do' | 'v' | 've') singleDobj
    : TypeLiteralOnAction
    verbPhrase = 'napsat/píš{eš}/napsal{a} (co) (na čem)'
    askDobjResponseProd = singleNoun
;

VerbRule(TypeLiteralOnWhat)
    [badness 500] ('piš' | 'psát' | 'napiš' | 'napsat') singleLiteral
    : TypeLiteralOnAction
    verbPhrase = 'napsat/píš{eš}/napsal{a} (co) (na čem)'
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = onSingleNoun;
    }
;

VerbRule(EnterOn)
    ('zadej' | 'zadat') singleLiteral ('na' | 'do' | 'v' | 've') singleDobj
    : EnterOnAction
    verbPhrase = 'zadat/zadáv{áš}/zadal{a} (co) (na čem)'
    askDobjResponseProd = onSingleNoun
;

VerbRule(EnterOnWhat)
    ('zadej' | 'zadat') singleLiteral
    : EnterOnAction
    verbPhrase = 'zadat/zadáv{áš}/zadal{a} (co) (na čem)'
    construct()
    {
        /*
         *   ENTER <text> is a little special, because it could mean ENTER
         *   <text> ON <keypad>, or it could mean GO INTO <object>.  It's
         *   hard to tell which based on the grammar alone, so we have to
         *   do some semantic analysis to make a good decision about it.
         *
         *   We'll start by assuming it's the ENTER <text> ON <iobj> form
         *   of the command, and we'll look for a suitable default object
         *   to serve as the iobj.  If we can't find a suitable default, we
         *   won't prompt for the missing object as we usually would.
         *   Instead, we'll try re-parsing the command as GO INTO.  To do
         *   this, use our custom "asker" - this won't actually prompt for
         *   the missing object, but will instead retry the command as a GO
         *   INTO command.
         */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.setPrompt(onSingleNoun, enterOnWhatAsker);
    }
;

/* our custom "asker" for the missing iobj in an "ENTER <text>" command */
enterOnWhatAsker: ResolveAsker
    askMissingObject(targetActor, action, which)
    {
        /*
         *   This method is called when the resolver has failed to find a
         *   suitable default for the missing indirect object of ENTER
         *   <text> ON <iobj>.
         *
         *   Instead of issuing the prompt that we'd normally issue under
         *   these circumstances, assume that we're totally wrong about the
         *   way we've been interpreting the command: assume that it's not
         *   meant as ENTER <text> ON <iobj> after all, but was actually
         *   meant as GO IN <object>.  So, rephrase the command as such and
         *   start over with the new phrasing.
         */
        throw new ReplacementCommandStringException(
            'vejdi do ' + action.getLiteral(), gIssuingActor, gActor);
    }
;

VerbRule(Consult)
    ('poraď' | 'poradit') 'se' ('s' | 'se') singleDobj
    | ('hledej' | 'hledat') ('v' | 've') singleDobj
    : ConsultAction
    verbPhrase = 'hledat/hled{áš}/hledal{a} (v čem)'
    askDobjResponseProd = inSingleNoun
;

VerbRule(ConsultAbout)
    ('poraď' | 'poradit') 'se' ('s' | 'se') singleDobj 'o' singleTopic
    | ('najdi' | 'najít' | 'hledej' | 'hledat' | 'vyhledej' | 'vyhledat')
    singleTopic ('na' | 'v' | 've') singleDobj
    | ('najdi' | 'najít' | 'hledej' | 'hledat' | 'vyhledej' | 'vyhledat')
    ('na' | 'v' | 've') singleDobj singleTopic
    : ConsultAboutAction
    verbPhrase = 'hledat/hled{áš}/hledal{a} (v čem) (o čem)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = inSingleNoun
;

VerbRule(ConsultWhatAbout)
    ('najdi' | 'najít' | 'hledej' | 'hledat' | 'vyhledej' | 'vyhledat'
    | ('přečti' | 'přečíst') 'si' 'o') singleTopic
    : ConsultAboutAction
    verbPhrase = 'hledat/hled{áš}/hledal{a} (co) (v čem)'
    whichMessageTopic = DirectObject
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = inSingleNoun;
    }
;

VerbRule(Switch)
    ('přepni' | 'přepnout') dobjList
    : SwitchAction
    verbPhrase = 'přepnout/přepín{áš}/přepnul{a} (co)'
;

VerbRule(Flip)
    ('obrať' | 'obrátit') dobjList
    : FlipAction
    verbPhrase = 'obrátit/obrac{íš}/obrátil{a} (co)'
;

VerbRule(TurnOn)
    ('zapni' | 'zapnout' | 'zap' | 'aktivuj' | 'aktivovat') dobjList
    : TurnOnAction
    verbPhrase = 'zapnout/zapín{áš}/zapnul{a} (co)'
;

VerbRule(TurnOff)
    ('vypni' | 'vypnout' | 'vyp' | 'deaktivuj' | 'deaktivovat') dobjList
    : TurnOffAction
    verbPhrase = 'vypnout/vypín{áš}/vypnul{a} (co)'
;

VerbRule(Light)
    ('rozsviť' | 'rozsvítit') dobjList
    : LightAction
    verbPhrase = 'rozsvítit/rozsvěc{íš}/rozsvítil{a} (co)'
;

DefineTAction(Strike);
VerbRule(Strike)
    ('škrtni' | 'škrtnout') dobjList
    : StrikeAction
    verbPhrase = 'škrtnout/škrt{áš}/škrtnul{a} (čím)'
;

VerbRule(Burn)
    ('zapal' | 'zapálit') dobjList
    : BurnAction
    verbPhrase = 'zapálit/zapalu{ješ}/zapálil{a} (co)'
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(BurnWith)
    ('zapal' | 'zapálit') singleDobj singleIobj
    | ('zapal' | 'zapálit') singleIobj singleDobj
    : BurnWithAction
    verbPhrase = 'zapálit/zapalu{ješ}/zapálil{a} (co) (čím)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(Extinguish)
    ('uhas' | 'uhasit' | 'sfoukni' | 'sfouknout') dobjList
    : ExtinguishAction
    verbPhrase = 'uhasit/has{íš}/uhasil{a} (co)'
;

VerbRule(ExtinguishType2)
    ('zhasni' | 'zhasnout') dobjList
    : ExtinguishAction
    verbPhrase = 'zhasnount/zhasín{áš}/zhasnul{a} (co)'
;

VerbRule(Break)
    ('rozbij' | 'rozbít' | 'rozmlať' | 'rozmlátit' | 'znič' | 'zničit'
    | 'vyraž' | 'vyrazit' | 'rozkopni' | 'rozkopnout') dobjList
    : BreakAction
    verbPhrase = 'rozbít/rozbíj{íš}/rozbil{a} (co)'
;

VerbRule(CutWithWhat)
    [badness 500] ('uřízni' | 'uříznout' | 'přeřízni' | 'přeříznout') singleDobj
    : CutWithAction
    verbPhrase = 'uříznout/řež{eš}/uřízl{a} (co) (čím)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = singleNoun;
    }
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(CutWith)
    ('uřízni' | 'uříznout') singleDobj singleIobj
    | ('uřízni' | 'uříznout') singleIobj singleDobj
    : CutWithAction
    verbPhrase = 'uříznout/řež{eš}/uřízl{a} (co) (čím)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(Eat)
    ('sněz' | 'sníst' | 'jez' | 'jíst') dobjList
    : EatAction
    verbPhrase = 'sníst/j{íš}/snědl{a} (co)'
;

VerbRule(EatType2)
    ('najez' | 'najíst') 'se' ('z' | 'ze' | ) dobjList
    : EatAction
    verbPhrase = 'sníst/j{íš}/snědl{a} (z čeho) se'
;

VerbRule(Drink)
    ('pij' | 'pít' | 'vypij' | 'vypít') dobjList
    : DrinkAction
    verbPhrase = 'vypít/pi{ješ}/vypil{a} (co)'
;

VerbRule(DrinkType2)
    ('napij' | 'napít') 'se' ('z' | 'ze' | ) dobjList
    : DrinkAction
    verbPhrase = 'napít/pi{ješ}/napil{a} (z čeho) se'
;

VerbRule(Pour)
    ('rozlij' | 'rozlít' | 'vylij' | 'vylít' | 'polij' | 'polít' | 'polej'
    | 'polejt' | 'lij' | 'lít') dobjList
    : PourAction
    verbPhrase = 'lít/li{ješ}/lil{a} (co)'
;

VerbRule(PourInto)
    ('nalij' | 'nalít' | 'vlij' | 'vlít' | 'vylij' | 'vylít' | 'lij' | 'lít'
    | 'stříkni' | 'stříknout'| 'vstříkni' | 'vstříknout')
    dobjList ('do' | 'na' | 'po') singleIobj
    : PourIntoAction
    verbPhrase = 'nalít/nalév{áš}/nalil{a} (co) (do čeho)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(PourOnto)
    ('rozlij' | 'rozlít' | 'vylij' | 'vylít' | 'rozstříkni' | 'rozstříknout')
    dobjList ('do' | 'na' | 'po') singleIobj
    : PourOntoAction
    verbPhrase = 'rozlít/rozlév{áš}/rozlil{a} (co) (na co)'
    askIobjResponseProd = onSingleNoun
;

VerbRule(Climb)
    ('šplhej' | 'šplhat' | 'lez' | 'lézt') ('na' | 'po') singleDobj
    : ClimbAction
    verbPhrase = 'šplhat/šplh{áš}/šplhal{a} (po čem)'
    askDobjResponseProd = alongSingleNoun
;

/*
 *   Protoze ve variante s vynechanym slovesem je zamenne s GoThrough a Enter,
 *   tak dame presmerovani akce, kdyby se pochopila blbe.
 */
VerbRule(ClimbUp)
    ('vyšplhej' | 'vyšplhat' | 'vylez' | 'vylézt' | 'vystoupej'
    | 'vystoupat' | 'vystup' | 'vystoupit'/* |*/ ) ('na' | 'po') singleDobj
    | ('vyšplhej' | 'vyšplhat' | 'vylez' | 'vylézt' | 'vystoupej'
    | 'vystoupat' | 'vystup' | 'vystoupit') 'do' singleDobj
    : ClimbUpAction
    verbPhrase = 'vyšplhat/šplh{áš}/vyšplhal{a} (na co)'
    askDobjResponseProd = onSingleNoun
;

VerbRule(ClimbUpWhat)
    [badness 200] 'vyšplhej' | 'vyšplhat' | 'vylez' | 'vylézt' | 'vystoupej'
    | 'vystoupat'
    : ClimbUpAction
    verbPhrase = 'vyšplhat/šplh{áš}/vyšplhal{a} (na co)'
    askDobjResponseProd = onSingleNoun
    construct()
    {
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = onSingleNoun;
    }
;

modify StairwayUp
    dobjFor(Enter) asDobjFor(ClimbUp)
    dobjFor(GoThrough) asDobjFor(ClimbUp)
;

modify BasicChair
    dobjFor(Enter) asDobjFor(SitOn)
;

VerbRule(ClimbDown)
    ('sešplhej' | 'sešplhat' | 'sestoupej' | 'sestoupat' | )
    ('z' | 'ze') singleDobj
    | ('sešplhej' | 'sešplhat' | 'slez' | 'slézt' | 'sestoupej' | 'sestoupat'
    | 'sestup' | 'sestoupit' | ) 'po' singleDobj
    | ('sešplhej' | 'sešplhat' | 'slez' | 'slézt' | 'sestoupej' | 'sestoupat'
    | 'sestup' | 'sestoupit') 'do' singleDobj
    : ClimbDownAction
    verbPhrase = 'sešplhat/sešplháv{áš}/sešplhal{a} (z čeho)'
    askDobjResponseProd = singleNoun
;

VerbRule(ClimbDownWhat)
    [badness 200] 'sešplhej' | 'sešplhat' | 'sestoupej' | 'sestoupat' | 'sestup'
    | 'sestoupit'
    : ClimbDownAction
    verbPhrase = 'sešplhat/sešplháv{áš}/sešplhal{a} (z čeho)'
    askDobjResponseProd = outOfSingleNoun
    construct()
    {
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = outOfSingleNoun;
    }
;

modify StairwayDown
    dobjFor(GoThrough) asDobjFor(ClimbDown)
    dobjFor(GetOffOf) asDobjFor(ClimbDown)
;

VerbRule(Clean)
    ('vyčisti' | 'vyčistit') dobjList
    : CleanAction
    verbPhrase = 'vyčistit/čist{íš}/vyčistil{a} (co)'
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(CleanWith)
    ('vyčisti' | 'vyčistit') dobjList singleIobj
    | ('vyčisti' | 'vyčistit') singleIobj dobjList
    : CleanWithAction
    verbPhrase = 'vyčistit/čist{íš}/vyčistil{a} (co) (čím)'
    askIobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(AttachTo)
    ('připevni' | 'připevnit' | 'přidělej' | 'přidělat' | 'přilož' | 'přiložit'
    | 'přimontuj' | 'přimontovat') dobjList ('k' | 'ke' | 'na' | 'do') singleIobj
    | ('dej' | 'dát') dobjList ('k' | 'ke') singleIobj
    : AttachToAction
    askIobjResponseProd = atSingleNoun
    verbPhrase
    {
        if(dobjList_.length() > 0 && dobjList_[1].obj_.ofKind(PlugAttachable) &&
            iobjList_.length() > 0 && iobjList_[1].obj_.ofKind(PlugAttachable))
            return 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)';
        else
            return 'připevnit/připevňu{ješ}/připevnil{a} (co) (k čemu)';
    }
;

VerbRule(AttachToWhat)
    [badness 500] ('připevni' | 'připevnit' | 'přidělej' | 'přidělat'| 'přilož'
    | 'přiložit' | 'přimontuj' | 'přimontovat') dobjList
    : AttachToAction
    verbPhrase
    {
        if(dobjList_.length() > 0 && dobjList_[1].obj_.ofKind(PlugAttachable) ||
            iobjList_.length() > 0 && iobjList_[1].obj_.ofKind(PlugAttachable))
            return 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)';
        else
            return 'připevnit/připevňu{ješ}/připevnil{a} (co) (k čemu)';
    }
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = atSingleNoun;
    }
;

VerbRule(DetachFrom)
    ('oddělej' | 'oddělat' | 'uvolni' | 'uvolnit' | 'odmontuj' | 'odmontovat')
    dobjList ('od' | 'z' | 'ze') singleIobj
    : DetachFromAction
    verbPhrase
    {
        if(dobjList_.length() > 0 && dobjList_[1].obj_.ofKind(PlugAttachable) &&
            iobjList_.length() > 0 && iobjList_[1].obj_.ofKind(PlugAttachable))
            return 'odpojit/odpoju{ješ}/odpojil{a} (co) (od čeho)';
        else
            return 'uvolnit/uvolňu{ješ}/uvolnil{a} (co) (od čeho)';
    }
    askIobjResponseProd = fromSingleNoun
;

VerbRule(Detach)
    ('oddělej' | 'oddělat' | 'uvolni' | 'uvolnit' | 'odmontuj' | 'odmontovat')
    dobjList
    : DetachAction
    verbPhrase = 'uvolnit/uvolňu{ješ}/uvolnil{a} (co)'
;

VerbRule(Open)
    ('otevři' | 'otevřít' | 'ote' | 'otev') dobjList
    : OpenAction
    verbPhrase = 'otevřít/otevír{áš}/otevřel{a} (co)'
;

VerbRule(Close)
    ('zavři' | 'zavřít' | 'uzavři' | 'uzavřít' | 'zav' | 'uzav') dobjList
    : CloseAction
    verbPhrase = 'zavřít/zavír{áš}/zavřel{a} (co)'
;

VerbRule(Lock)
    ('zamkni' | 'zamknout' | 'zamčít' | 'zam') dobjList
    : LockAction
    verbPhrase = 'zamknout/zamyk{áš}/zamkl{a} (co)'
;

VerbRule(Unlock)
    ('odemkni' | 'odemknout' | 'odemčít' | 'ode') dobjList
    : UnlockAction
    verbPhrase = 'odemknout/odemyk{áš}/odemkl{a} (co)'
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(LockWith)
    ('zamkni' | 'zamknout' | 'zamčít' | 'zam') singleDobj singleIobj
    | ('zamkni' | 'zamknout' | 'zamčít' | 'zam') singleIobj singleDobj
    : LockWithAction
    verbPhrase = 'zamknout/zamyk{áš}/zamkl{a} (co) (čím)'
    isPrepositionalPhrasing = nil
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(UnlockWith)
    ('odemkni' | 'odemknout' | 'odemčít' | 'ode') singleDobj singleIobj
    | ('odemkni' | 'odemknout' | 'odemčít' | 'ode') singleIobj singleDobj
    : UnlockWithAction
    verbPhrase = 'odemknout/odemyk{áš}/odemkl{a} (co) (čím)'
    isPrepositionalPhrasing = nil
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = singleNoun
;

VerbRule(SitOn)
    ('sedni' | 'sednout') ( | 'si') ('na' | 'do') singleDobj
    : SitOnAction
    verbPhrase = 'sednout/sed{áš}/sedl{a} (na co) si'
    askDobjResponseProd = onSingleNoun

    /* use the objInPrep, if there's a direct object available */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objIntoPrep + ' ' : prep); }
    adjustDefaultObjectCase(gc, obj)
        { return (obj != nil ? obj.objIntoCase : gc); }
;

VerbRule(Sit)
    ('sedni' | 'sednout') ( | 'si')
    | ('posaď' | 'posadit') ( | 'se')
    : SitAction
    verbPhrase = 'sednout/sed{áš}/sedl{a} si'
;

VerbRule(LieOn)
    ('lehni' | 'lehnout') ('si' | ) 'na' singleDobj
    : LieOnAction
    verbPhrase = 'lehnout/leh{áš}/lehl{a} (na co) si'
    askDobjResponseProd = onSingleNoun

    /* use the objInPrep, if there's a direct object available */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objIntoPrep + ' ' : prep); }
    /*
     *   Lehnout si lze na neco i do neceho. Proto pri announceDefaultObject
     *   prihledneme k tomu, co si preje objekt.
     */
    adjustDefaultObjectCase(gc, obj)
        { return (obj != nil ? obj.objIntoCase : gc); }
;

VerbRule(LieOnType2)
    ('lehni' | 'lehnout') ('si' | ) 'do' singleDobj
    : LieOnAction
    verbPhrase = 'lehnout/leh{áš}/lehl{a} (do čeho) si'
    askDobjResponseProd = toSingleNoun

    /* use the objInPrep, if there's a direct object available */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objIntoPrep + ' ' : prep); }
    adjustDefaultObjectCase(gc, obj)
        { return (obj != nil ? obj.objIntoCase : gc); }
;

VerbRule(Lie)
    ('lehni' | 'lehnout') ('si' | ) : LieAction
    verbPhrase = 'lehnout/leh{áš}/lehl{a} si'
;

VerbRule(StandOn)
    ('stoupni' | 'stoupnout') ('si' | ) ('na' | 'do') singleDobj
    : StandOnAction
    verbPhrase = 'stoupnout/stoup{áš}/stoupl{a} (na co) si'
    askDobjResponseProd = onSingleNoun

    /* use the objInPrep, if there's a direct object available */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objIntoPrep + ' ' : prep); }
    adjustDefaultObjectCase(gc, obj)
        { return (obj != nil ? obj.objIntoCase : gc); }
;

VerbRule(Stand)
    'vstaň' | 'vstát' | 'stůj' | 'stát' | ('stoupni' | 'stoupnout') ('si' | )
    | ('postav' | 'postavit') ('se' | )
    : StandAction
    verbPhrase = 'vstát/vstáv{áš}/vstal{a}'
;

VerbRule(GetOutOf)
    ('opusť' | 'opustit') singleDobj
    | ('vystup' | 'vystoupit') ('z' | 'ze') singleDobj
    | ('jdi' | 'jít' | 'běž' | 'běžet' | 'vyjdi' | 'vyjít' | 'odejdi' | 'odejít'
    | 'vylez' | 'vylézt') ('pryč' | 'ven' | ) ('z' | 'ze') singleDobj
    | ('pryč' | 'ven') ('z' | 'ze') singleDobj
    : GetOutOfAction
    verbPhrase = 'dostat/dostáv{áš}/dostal{a} ven (z čeho) se'
    askDobjResponseProd = outOfSingleNoun

    /* use the objOutOfPrep, if there's a direct object available */
    /*
     * Tohle se podílí na větě ">ven (z dřevěného prkna)
     */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objOutOfPrep + ' ' : prep); }
;

VerbRule(GetOffOf)
    ('sestup' | 'sestoupit' | 'slez' | 'slézt' | 'vstaň' | 'vstát')
    ('z' | 'ze') singleDobj
    : GetOffOfAction
    verbPhrase = 'dostat/dostáv{áš}/dostal{a} pryč (z čeho) se'
    askDobjResponseProd = outOfSingleNoun

    /* use the objOutOfPrep, if there's a direct object available */
    adjustDefaultObjectPrep(prep, obj)
        { return (obj != nil ? obj.objOutOfPrep + ' ' : prep); }
;

VerbRule(GetOffOfWhat)
    ('sestup' | 'sestoupit' | 'slez' | 'slézt' | 'vstaň' | 'vstát')
    : GetOffOfAction
    verbPhrase = 'dostat/dostáv{áš}/dostal{a} pryč (z čeho) se'
    askDobjResponseProd = outOfSingleNoun
    construct()
    {
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = outOfSingleNoun;
    }
;
/**/
VerbRule(GetOut)
    ('vystup' | 'vystoupit' | 'vys') ( | 'ven')
    : GetOutAction
    verbPhrase = 'dostat/dostáv{áš}/dostal{a} ven'
;

VerbRule(Board)
    ('nastup' | 'nastoupit' | 'nas') ('na' | 'do' | ) singleDobj
    : BoardAction
    verbPhrase = 'nastoupit/nastupu{ješ}/nastoupil{a} (do čeho)'
    askDobjResponseProd = toSingleNoun
;

VerbRule(Sleep)
    ('spi' | 'spát')
    | ('vyspi' | 'vyspat') ('se' | )
    : SleepAction
    verbPhrase = 'spát/sp{íš}/spal{a}'
;

VerbRule(Fasten)
    ('připoutej' | 'připoutat') dobjList
    : FastenAction
    verbPhrase = 'připoutat/připoutáv{áš}/připoutal{a} (co)'
;

VerbRule(FastenTo)
    ('připoutej' | 'připoutat') dobjList ('k' | 'ke') singleIobj
    : FastenToAction
    verbPhrase = 'připoutat/připoutáv{áš}/připoutal{a} (co) (k čemu)'
    askIobjResponseProd = atSingleNoun
;

VerbRule(Unfasten)
    ('odpoutej' | 'odpoutat') dobjList
    : UnfastenAction
    verbPhrase = 'odpoutat/odpoutáv{áš}/odpoutal{a} (co)'
;

VerbRule(UnfastenFrom)
    ('odpoutej' | 'odpoutat') dobjList ('od' | 'z' | 'ze') singleIobj
    : UnfastenFromAction
    verbPhrase = 'odpoutat/odpoutáv{áš}/odpoutal{a} (co) (od čeho)'
    askIobjResponseProd = fromSingleNoun
;

VerbRule(PlugInto)
    ('připoj' | 'připojit' | 'napoj' | 'napojit' | 'zapoj' | 'zapojit'
    | 'nasaď' | 'nasadit') dobjList ('do' | 'k' | 'ke' | 'na') singleIobj
    | ('spoj' | 'spojit' | 'propoj' | 'propojit') dobjList ('s' | 'se')
    singleIobj
    : PlugIntoAction
    verbPhrase = 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)'
    askIobjResponseProd = atSingleNoun
;

VerbRule(PlugIntoWhat)
    [badness 500] ('připoj' | 'připojit' | 'napoj' | 'napojit' | 'zapoj'
    | 'zapojit' | 'spoj' | 'spojit' | 'propoj' | 'propojit' | 'nasaď'
    | 'nasadit') dobjList
    : PlugIntoAction
    verbPhrase = 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = atSingleNoun;
    }
;

VerbRule(PlugIn)
    ('zapoj' | 'zapojit') dobjList
    : PlugInAction
    verbPhrase = 'zapojit/zapoju{ješ}/zapojil{a} (co)'
;

VerbRule(UnplugFrom)
    ('odpoj' | 'odpojit' | 'vypoj' | 'vypojit') dobjList
    ('z' | 'ze' | 'od') singleIobj
    : UnplugFromAction
    verbPhrase = 'odpojit/odpoju{ješ}/odpojil{a} (co) (z čeho)'
    askIobjResponseProd = outOfSingleNoun
;

VerbRule(Unplug)
    ('odpoj' | 'odpojit' | 'vypoj' | 'vypojit') dobjList
    : UnplugAction
    verbPhrase = 'odpojit/odpoju{ješ}/odpojil{a} (co)'
;

VerbRule(Screw)
    ('zašroubuj' | 'zašroubovat') dobjList
    : ScrewAction
    verbPhrase = 'přišroubovat/přišroubováv{áš}/přišrouboval{a} (co)'
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(ScrewWith)
    ('zašroubuj' | 'zašroubovat') dobjList singleIobj
    | ('zašroubuj' | 'zašroubovat') singleIobj dobjList
    : ScrewWithAction
    verbPhrase = 'přišroubovat/přišroubováv{áš}/přišrouboval{a} (co) (čím)'
    omitIobjInDobjQuery = true
    askIobjResponseProd = singleNoun
;

VerbRule(Unscrew)
    ('odšroubuj' | 'odšroubovat') dobjList
    : UnscrewAction
    verbPhrase = 'odšroubovat/odšroubováv{áš}/odšrouboval{a} (co)'
;

/* TODO: rozdělit na prepositional phrasing a nonprep. phrasing */
VerbRule(UnscrewWith)
    ('odšroubuj' | 'odšroubovat') dobjList singleIobj
    | ('odšroubuj' | 'odšroubovat') singleIobj dobjList
    : UnscrewWithAction
    verbPhrase = 'odšroubovat/odšroubováv{áš}/odšrouboval{a} (co) (čím)'
    omitIobjInDobjQuery = true
    askIobjResponseProd = singleNoun
;

VerbRule(PushTravelDir)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit' | 'vytlač' | 'vytlačit' |
    'strkej' | 'strkat' | 'vystrkej' | 'vystrkat') singleDobj
    ( | 'na') singleDir
    : PushTravelDirAction
    verbPhrase = ('tlačit/tlač{íš}/tlačil{a} (co) ' + dirMatch.dir.name)
;

VerbRule(PushTravelThrough)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit' | 'protlač' | 'protlačit'
    | 'prostrč' | 'prostrčit') singleDobj ('skrz' | ) singleIobj
    : PushTravelThroughAction
    verbPhrase = 'tlačit/tlač{íš}/tlačil{a} (co) (skrz co)'
    preferredIobj = NonPortable
;

VerbRule(PushTravelEnter)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit' | 'zatlač' | 'zatlačit')
    singleDobj 'do' singleIobj
    : PushTravelEnterAction
    verbPhrase = 'tlačit/tlač{íš}/tlačil{a} (co) (do čeho)'
;

VerbRule(PushTravelGetOutOf)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit' | 'vytlač' | 'vytlačit')
    singleDobj ('ven' | ) ('z' | 'ze') singleIobj
    : PushTravelGetOutOfAction
    verbPhrase = 'tlačit/tlač{íš}/tlačil{a} (co) (ven z čeho)'
;

/* TODO: Tyhle dvě jsou nějaké divné, nedovedu si představit využití. Mozna na schodech, ktere reaguji na climb? */
VerbRule(PushTravelClimbUp)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit') singleDobj
    'na' singleIobj
    : PushTravelClimbUpAction
    verbPhrase = 'tlačit/tlač{íš}/tlačil{a} (co) (na co)'
    omitIobjInDobjQuery = true
;

VerbRule(PushTravelClimbDown)
    ('tlač' | 'tlačit' | 'odtlač' | 'odtlačit') singleDobj
    'pod' singleIobj
    : PushTravelClimbDownAction
    verbPhrase = 'tlačit/tlač{íš}/tlačil{a} (co) (pod co)'
;

VerbRule(Exits)
    ('ukaž' | 'ukázat' | ) ('mi' | ) ('směry' | 'východy' | 'cesty' | 'pohyby')
    : ExitsAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} směry'
;

VerbRule(ExitsMode)
    ('zapnout'->on_ | 'všechny'->on_ | 'vypnout'->off_ | 'žádné'->off_ )
    ('směry' | 'východy' | 'cesty' | 'pohyby')
    | ('směry' | 'východy' | 'cesty' | 'pohyby')
    'v' ('záhlaví'->stat_ | 'popisu'->look_)
    : ExitsModeAction
    verbPhrase = 'vypnout/vypín{áš}/vypnul{a} zobrazení směrů'
;

VerbRule(HintsOff)
    ('vypni' | 'vypnout' | 'vyp') ('rady')
    | ('rady') ('vypni' | 'vypnout' | 'vyp')
    : HintsOffAction
    verbPhrase = 'vypnout/vypín{áš}/vypnul{a} rady'
;

VerbRule(Hint)
    ('ukaž' | 'ukázat') ('radu' | 'rady')
    | ('poraď' | 'poradit') ( | 'mi')
    | 'rada' | 'rady'
    : HintAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} rady'
;

VerbRule(Oops)
    ('oprava' | 'opr' | 'op') singleLiteral
    : OopsAction
    verbPhrase = 'opravit/opravu{ješ}/opravil{a} (co)'
;

VerbRule(OopsOnly)
    ('oprava' | 'opr' | 'op')
    : OopsIAction
    verbPhrase = 'opravit/opravu{ješ}/opravil{a}'
;


/* ------------------------------------------------------------------------ */
/*
 *   "debug" verb - special verb to break into the debugger.  We'll only
 *   compile this into the game if we're compiling a debug version to begin
 *   with, since a non-debug version can't be run under the debugger.
 */
#ifdef __DEBUG

VerbRule(Debug)
    'debuguj' | 'debugovat' | 'debug'
    : DebugAction
    verbPhrase = 'debugovat/debugu{ješ}/debugoval{a}'
;

#endif /* __DEBUG */

/* ------------------------------------------------------------------------ */
/*
 *   Execute a command line, as issued by the given actor and as given as a
 *   list of tokens.
 *
 *   If 'firstInSentence' is true, we're at the start of a "sentence."  The
 *   meaning and effect of this may vary by language.  In English, a
 *   sentence ends with certain punctuation marks (a period, semicolon,
 *   exclamation mark, or question mark), so anything after one of these
 *   punctuation marks is the start of a new sentence.  Also in English, we
 *   can address a command to an explicit target actor using the "actor,"
 *   prefix syntax, which we can't use except at the start of a sentence.
 *
 *   If the command line consists of multiple commands, we will only
 *   actually execute the first command before returning.  We'll schedule
 *   any additional commands for later execution by putting them into the
 *   target actor's pending command queue before we return, but we won't
 *   actually execute them.
 */
find_np(obj)
{
        if(obj.noun_ != nil) return obj.noun_.noun_;
        if(obj.np_ != nil) return find_np(obj.np_);
        return nil;
}

modify executeCommand(targetActor, issuingActor, toks, firstInSentence)
{
    local actorPhrase;
    local actorSpecified;

    /*
     *   Turn on sense caching while we're working, until execution
     *   begins.  The parsing and resolution phases of command processing
     *   don't involve any changes to game state, so we can safely cache
     *   sense information; caching sense information during these phases
     *   is desirable because these steps (noun resolution in particular)
     *   involve repeated inspection of the current sensory environment,
     *   which can require expensive calculations.  
     */
    libGlobal.enableSenseCache();

    /* we don't have an explicit actor phrase yet */
    actorPhrase = nil;
    
    /* presume an actor will not be specified */
    actorSpecified = nil;
    
    /*
     *   If this is the start of a new sentence, and the issuing actor
     *   wants to cancel any target actor designation at the end of each
     *   sentence, change the target actor back to the issuing actor.  
     */
    if (firstInSentence
        && issuingActor != targetActor
        && issuingActor.revertTargetActorAtEndOfSentence)
    {
        /* switch to the target actor */
        targetActor = issuingActor;

        /* switch to the issuer's sense context */
        senseContext.setSenseContext(targetActor, sight);
    }

    /*
     *   Keep going until we've processed the command.  This might take
     *   several iterations, because we might have replacement commands to
     *   execute.  
     */
parseTokenLoop:
    for (;;)
    {
        local lst;
        local action;
        local match;
        local nextIdx;
        local nextCommandTokens;
        local extraIdx;
        local extraTokens;
        
        /*
         *   Catch any errors that occur while executing the command,
         *   since some of them are signals to us that we should reparse
         *   some new text read or generated deep down inside the command
         *   processing.  
         */
        try
        {
            local rankings;

            /* we have no extra tokens yet */
            extraTokens = [];

            /* 
             *   Parse the token list.  If this is the first command on
             *   the command line, allow an actor prefix.  Otherwise, just
             *   look for a command.  
             */
            lst = (firstInSentence ? firstCommandPhrase : commandPhrase)
                  .parseTokens(toks, cmdDict);

            /*
             *   As a first cut at reducing the list of possible matches
             *   to those that make sense, eliminate from this list any
             *   matches which do not have valid actions.  In grammars for
             *   "scrambling" languages (i.e., languages with flexible
             *   word ordering), it is possible to construct commands that
             *   fit the grammatical rules of sentence construction but
             *   which make no sense because of the specific constraints
             *   of the verbs involved; we can filter out such nonsense
             *   interpretations immediately by keeping only those
             *   structural interpretations that can resolve to valid
             *   actions.  
             */
            lst = lst.subset(
                {x: x.resolveFirstAction(issuingActor, targetActor) != nil});

            /* if we have no matches, the command isn't understood */
            if (lst.length() == 0)
            {
                /* 
                 *   If this is a first-in-sentence phrase, try looking for
                 *   a target actor phrase.  If we can find one, we can
                 *   direct the 'oops' to that actor, to allow the game to
                 *   customize messages more specifically.  
                 */
                if (firstInSentence)
                {
                    local i;
                    
                    /* try parsing an "actor, <unknown>" phrase */
                    lst = actorBadCommandPhrase.parseTokens(toks, cmdDict);

                    /* if we got any matches, try to resolve actors */
                    lst = lst.mapAll({x: x.resolveNouns(
                        issuingActor, issuingActor,
                        new TryAsActorResolveResults())});

                    /* drop any that didn't yield any results */
                    lst = lst.subset({x: x != nil && x.length() != 0});

                    /*
                     *
                     *   if anything's left, and one of the entries 
                     *   resolves to an actor, arbitrarily pick the 
                     *   first such entry use the resolved actor as the 
                     *   new target actor 
                     */
                    if (lst.length() != 0
                        && (i = lst.indexWhich(
                            {x: x[1].obj_.ofKind(Actor)})) != nil)
                        targetActor = lst[i][1].obj_;
                }

                /* 
                 *   We don't understand the command.  Check for unknown
                 *   words - if we have any, give them a chance to use
                 *   OOPS to correct a typo.  
                 */
                tryOops(toks, issuingActor, targetActor,
                        1, toks, rmcCommand);

                /* 
                 *   try running it by the SpecialTopic history to see if
                 *   they're trying to use a special topic in the wrong
                 *   context - if so, explain that they can't use the
                 *   command right now, rather than claiming that the
                 *   command is completely invalid 
                 */
                if (specialTopicHistory.checkHistory(toks))
                {
                    /* the special command is not currently available */
                    targetActor.notifyParseFailure(
                        issuingActor, &specialTopicInactive, []);
                }
                else
                {
                    /* tell the issuer we didn't understand the command */
                    targetActor.notifyParseFailure(
                        issuingActor, &commandNotUnderstood, []);
                }
                
                /* 
                 *   we're done with this command, and we want to abort
                 *   any subsequent commands on the command line 
                 */
                return;
            }

            /* show the matches if we're in debug mode */
            dbgShowGrammarList(lst);
        
            /* 
             *   Perform a tentative resolution on each alternative
             *   structural interpretation of the command, and rank the
             *   interpretations in order of "goodness" as determined by
             *   our ranking criteria encoded in CommandRanking.
             *   
             *   Note that we perform the tentative resolution and ranking
             *   even if we have only one interpretation, because the
             *   tentative resolution is often useful for the final
             *   resolution pass.  
             */
            rankings = CommandRanking
                       .sortByRanking(lst, issuingActor, targetActor);
                
            /* 
             *   Take the interpretation that came up best in the rankings
             *   (or, if we have multiple at the same ranking, arbitrarily
             *   pick the one that happened to come up first in the list)
             *   - they're ranked in descending order of goodness, so take
             *   the first one.
             */
            match = rankings[1].match;

            /* if we're in debug mode, show the winner */
            dbgShowGrammarWithCaption('Winner', match);
            
            /*
             *   Get the token list for the rest of the command after what
             *   we've parsed so far.
             *   
             *   Note that we'll start over and parse these tokens anew,
             *   even though we might have parsed them already into a
             *   subcommand on the previous iteration.  Even if we already
             *   parsed these tokens, we want to parse them again, because
             *   we did not have a suitable context for evaluating the
             *   semantic strengths of the possible structural
             *   interpretations this command on the previous iteration;
             *   for example, it was not possible to resolve noun phrases
             *   in this command because we could not guess what the scope
             *   would be when we got to this point.  So, we'll simply
             *   discard the previous match tree, and start over, treating
             *   the new tokens as a brand new command.  
             */
            nextIdx = match.getNextCommandIndex();
            nextCommandTokens = toks.sublist(nextIdx);

            /* if the pending command list is empty, make it nil */
            if (nextCommandTokens.length() == 0)
                nextCommandTokens = nil;

            /*
             *   Get the part of the token list that the match doesn't use
             *   at all.  These are the tokens following the tokens we
             *   matched. 
             */
            extraIdx = match.tokenList.length() + 1;
            extraTokens = toks.sublist(extraIdx);

            /*
             *   We now have the best match for the command tree.
             *   
             *   If the command has an actor clause, resolve the actor, so
             *   that we can direct the command to that actor.  If the
             *   command has no actor clause, the command is to the actor
             *   who issued the command.
             *   
             *   Do NOT process the actor clause if we've already done so.
             *   If we edit the token list and retry the command after
             *   this point, there is no need to resolve the actor part
             *   again.  Doing so could be bad - if resolving the actor
             *   required player interaction, such as asking for help with
             *   an ambiguous noun phrase, we do not want to go through
             *   the same interaction again just because we have to edit
             *   and retry a later part of the command.  
             */
            if (match.hasTargetActor())
            {
                local actorResults;

                /*
                 *   If we haven't yet explicitly specified a target
                 *   actor, and the default target actor is different from
                 *   the issuing actor, then this is really a new command
                 *   from the issuing actor.
                 *   
                 *   First, an actor change mid-command is allowed only if
                 *   the issuing actor waits for NPC commands to be
                 *   carried out; if not, then we can't change the actor
                 *   mid-command.
                 *   
                 *   Second, since the command comes from the issuing
                 *   actor, not from the default target actor, we want to
                 *   issue the orders on the issuing actor's turn, so put
                 *   the command into the queue for the issuer, and let
                 *   the issuer re-parse the command on the issuer's next
                 *   turn.  
                 */
                if (!actorSpecified && issuingActor != targetActor)
                {
                    /* 
                     *   don't allow the command if the issuing actor
                     *   doesn't wait for orders to be completed 
                     */
                    if (!issuingActor.issueCommandsSynchronously)
                    {
                        /* turn off any sense capturing */
                        senseContext.setSenseContext(nil, sight);

                        /* show the error */
                        issuingActor.getParserMessageObj()
                            .cannotChangeActor();

                        /* done */
                        return;
                    }

                    /* put the command into the issuer's queue */
                    issuingActor.addFirstPendingCommand(
                        firstInSentence, issuingActor, toks);

                    /* done */
                    return;
                }

                /* create an actor-specialized results object */
                actorResults = new ActorResolveResults();
                
                /* set up the actors in the results object */
                actorResults.setActors(targetActor, issuingActor);
                
                /* resolve the actor object */
                match.resolveNouns(issuingActor, targetActor, actorResults);

                /* get the target actor from the command */
                targetActor = match.getTargetActor();

                /* pull out the phrase specifying the actor */
                actorPhrase = match.getActorPhrase();

                /*
                 *   Copy antecedents from the issuing actor to the target
                 *   actor.  Since the issuer is giving us a new command
                 *   here, pronouns will be given from the issuer's
                 *   perspective.  
                 */
                targetActor.copyPronounAntecedentsFrom(issuingActor);

                /* let the actor phrase know we're actually using it */
                match.execActorPhrase(issuingActor);
                
                /* 
                 *   Ask the target actor if it's interested in the
                 *   command at all.  This only applies when the actor was
                 *   actually specified - if an actor wasn't specified,
                 *   the command is either directed to the issuer itself,
                 *   in which case the command will always be accepted; or
                 *   the command is a continuation of a command line
                 *   previously accepted.  
                 */
                if (!targetActor.acceptCommand(issuingActor))
                {
                    /* 
                     *   the command was immediately rejected - abandon
                     *   the command and any subsequent commands on the
                     *   same line 
                     */
                    return;
                }

                /* note that an actor was specified */
                actorSpecified = true;

                /*
                 *   Pull out the rest of the command (other than the
                 *   target actor specification) and start over with a
                 *   fresh parse of the whole thing.  We must do this
                 *   because our tentative resolution pass that we used to
                 *   pick the best structural interpretation of the
                 *   command couldn't go far enough - since we didn't know
                 *   the actor involved, we weren't able to resolve nouns
                 *   in the rest of the command.  Now that we know the
                 *   actor, we can start over and resolve everything in
                 *   the rest of the command, and thus choose the right
                 *   structural match for the command.  
                 */
                toks = match.getCommandTokens();

                /* what follows obviously isn't first in the sentence */
                firstInSentence = nil;

                /* go back to parse the rest */
                continue parseTokenLoop;
            }

            /* pull out the first action from the command */        
            action = match.resolveFirstAction(issuingActor, targetActor);
            
            if(action.dobjList_ != nil)
            {
                if(action.dobjList_.length() > 0)
                {
                    action.dobjList_[1].obj_.parsedAs
                        = find_np(action.dobjList_[1].np_);
                }
            }

            /*
             *   If the winning interpretation had any unknown words, run
             *   a resolution pass to resolve those interactively, if
             *   possible.  We want to do this before doing any other
             *   interactive resolution because OOPS has the unique
             *   property of forcing us to reparse the command; if we
             *   allowed any other interactive resolution to happen before
             *   processing an OOPS, we'd likely have to repeat the other
             *   resolution on the reparse, which would confuse and
             *   irritate users by asking the same question more than once
             *   for what is apparently the same command.  
             */
            if (rankings[1].unknownWordCount != 0)
            {
                /* 
                 *   resolve using the OOPS results gatherer - this runs
                 *   essentially the same preliminary resolution process
                 *   as the ranking results gatherer, but does perform
                 *   interactive resolution of unknown words via OOPS 
                 */
                match.resolveNouns(
                    issuingActor, targetActor,
                    new OopsResults(issuingActor, targetActor));
            }

            /* 
             *   If the command is directed to a different actor than the
             *   issuer, change to the target actor's sense context.  
             */
            if (actorSpecified && targetActor != issuingActor)
                senseContext.setSenseContext(targetActor, sight);

            /*
             *   Vyresetujeme uvozovkový automat, aby případné nespárované
             *   uvozovky se netáhly celým zbytkem hry.
             */
            languageGlobals.quotesLevel = 0;

            /* set up a transcript to receive the command results */
            withCommandTranscript(CommandTranscript, function()
            {
                /* 
                 *   Execute the action.
                 *   
                 *   If a target actor was specified, and it's not the same
                 *   as the issuing actor, this counts as a turn for the
                 *   issuing actor.  
                 */
                executeAction(targetActor, actorPhrase, issuingActor,
                              actorSpecified && issuingActor != targetActor,
                              action);
            });

            /*
             *   If we have anything remaining on the command line, insert
             *   the remaining commands at the start of the target actor's
             *   command queue, since we want the target actor to continue
             *   with the rest of this command line as its next operation.
             *   
             *   Since the remainder of the line represents part of the
             *   work the actor pulled out of the queue in order to call
             *   us, put the remainder back in at the START of the actor's
             *   queue - it was before anything else that might be in the
             *   actor's queue before, so it should stay ahead of anything
             *   else now.  
             */
            if (nextCommandTokens != nil)
            {
                /* 
                 *   Prepend the remaining commands to the actor's queue.
                 *   The next command is the start of a new sentence if
                 *   the command we just executed ends the sentence. 
                 */
                targetActor.addFirstPendingCommand(
                    match.isEndOfSentence(), issuingActor, nextCommandTokens);
            }

            /*
             *   If the command was directed from the issuer to a
             *   different target actor, and the issuer wants to wait for
             *   the full set of issued commands to complete before
             *   getting another turn, tell the issuer to begin waiting.  
             */
            if (actorSpecified && issuingActor != targetActor)
                issuingActor.waitForIssuedCommand(targetActor);

            /* we're done */
            return;
        }
        catch (ParseFailureException rfExc)
        {
            /*
             *   Parsing failed in such a way that we cannot proceed.
             *   Tell the target actor to notify the issuing actor.  
             */
            rfExc.notifyActor(targetActor, issuingActor);

            /* 
             *   the command cannot proceed, so abandon any remaining
             *   tokens on the command line 
             */
            return;
        }
        catch (CancelCommandLineException cclExc)
        {
            /* 
             *   if there are any tokens remaining, the game might want to
             *   show an explanation 
             */
            if (nextCommandTokens != nil)
                targetActor.getParserMessageObj().explainCancelCommandLine();

            /* stop now, abandoning the rest of the command line */	
            return;
        }
        catch (TerminateCommandException tcExc)
        {
            /* 
             *   the command cannot proceed - we can't do any more
             *   processing of this command, so simply return, abandoning
             *   any additional tokens we have 
             */
            return;
        }
        catch (RetryCommandTokensException rctExc)
        {
            /* 
             *   We want to replace the current command's token list with
             *   the new token list - get the new list, and then go back
             *   and start over processing it.
             *   
             *   Note that we must retain any tokens beyond those that the
             *   match tree used.  The exception only edits the current
             *   match tree's matched tokens, since it doesn't have access
             *   to any of the original tokens beyond those, so we must
             *   now add back in any tokens beyond the originals.  
             */
            toks = rctExc.newTokens_ + extraTokens;

            /* go back and process the command again with the new tokens */
            continue parseTokenLoop;
        }
        catch (ReplacementCommandStringException rcsExc)
        {
            local str;
            
            /* retrieve the new command string from the exception */
            str = rcsExc.newCommand_;

            /* 
             *   if the command string is nil, it means that the command
             *   has been fully handled already, so we simply return
             *   without any further work 
             */
            if (str == nil)
                return;

            /* 
             *   Replace the entire command string with the one from the
             *   exception - this cancels any previous command that we
             *   had.
             */
            toks = cmdTokenizer.tokenize(str);

            /* 
             *   we have a brand new command line, so we're starting a
             *   brand new sentence 
             */
            firstInSentence = true;

            /* set the issuing and target actor according to the exception */
            issuingActor = rcsExc.issuingActor_;
            targetActor = rcsExc.targetActor_;

            /* 
             *   Put this work into the target actor's work queue, so that
             *   the issuer will carry out the command at the next
             *   opportunity.  This is a brand new command line, so it
             *   starts a new sentence.  
             */
            targetActor.addPendingCommand(true, issuingActor, toks);

            /* we're done processing this command */
            return;
        }
    }
}

modify SpecialTopic
    matchPreParse(str, procStr)
    {
        /* build the regular expression pattern if there isn't one */
        if (matchPat == nil)
        {
            local pat;

            /* start with the base pattern string */
            pat = '<nocase><space>*(%<';

            /* add the keywords */
            for (local i = 1, local len = keywordList.length() ;
                 i <= len ; ++i)
            {
                /* add this keyword to the pattern */
                for(local j = 1, local len2 = keywordList[i].length() ;
                    j <= len2 ; ++j)
                {
                    local ch = keywordList[i].substr(j, 1);
                    local found = nil;
            	    
                    for(local k = 1; k <= mappings.length(); k++)
                    {
                        if(ch == mappings[k][1])
                        {
                            pat += '[' + ch + mappings[k][2] + ']';
                            found = true;
                            break;
                        }
                    }
            	    
                    if(!found) pat += ch;
                }

                /* add the separator or terminator, as appropriate */
                if (i == len)
                    pat += '%><space>*)+';
                else
                    pat += '%><space>*|%<';
            }

            /* create the pattern object */
            matchPat = new RexPattern(pat);
        }

        /* we have a match if the pattern matches the processed input */
        return rexMatch(matchPat, procStr) == procStr.length();
    }
    mappings =
    [
        ['ř', 'r'],
        ['í', 'i'],
        ['š', 's'],
        ['ž', 'z'],
        ['ť', 't'],
        ['č', 'c'],
        ['ý', 'y'],
        ['ů', 'u'],
        ['ň', 'n'],
        ['ú', 'u'],
        ['ě', 'e'],
        ['ď', 'd'],
        ['á', 'a'],
        ['é', 'e'],
        ['ó', 'o']
    ]
;
