#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 *
 *   Mechanismus pomoci pro začínající hráče pochází z hry Mrs. Pepper's Nasty
 *   Secret od Jima Aikina a Erica Eva. Použito s laskavým svolením.
 */

#include <adv3.h>
#include <cs_cz.h>

/*
 *   Many of the ideas in this file are based on Emily Short's
 *   NewbieGrammar.h extension for Inform 6, itself based on observation of
 *   some of the non-standard things newcomers to IF tend to type. The idea
 *   is to get the parser to recognize these commands, at least to the
 *   extent of giving the player who issues them some help towards typing
 *   more useful commands rather than giving a standard (and probably
 *   unhelpful) parser error message
 *
 *   We also provide some helpful brief IF playing instructions for newbies,
 *   a sample transcript they can view to get the general idea, and a
 *   mechanism to check whether someone is entering an excessive number of
 *   invalid commands (so that we can then offer more help).
 */

//helpMessage: object
//    printMsg {
//        "If you\'re new to interactive fiction, you can type
//        <<aHref('INSTRUCTIONS', 'INSTRUCTIONS', 'Show full instructions')>> at
//        the prompt for a full explanation of how to interact with the software
//        and the story; for a brief introduction to playing this type of game
//        type <<aHref('INTRO', 'INTRO', 'Show a brief introduction')>>, or
//          <<aHref('SAMPLE', 'SAMPLE', 'Show a short sample transcript')>> to
//        show a short sample transcript.
//        If you\'re having trouble moving forward and need help
//        solving a puzzle, type <<aHref('HINT', 'HINT', 'Show hints')>>. ";
//        extraHintManager.explainExtraHints();
//    }

//    briefIntro()
//    {
//        "You play this kind of game by entering brief commands (try to use as
//        few words as possible). Examples include:\b
//        WEST\n
//        X KEYS\n
//        TAKE BOOK\n
//        READ BOOK\n
//        DROP BOOK\n
//        PUT BOOK ON TABLE\n
//        OPEN DOOR\b
//        To move around use compass directions: NORTH, EAST, SOUTH etc. These may
//        be abbreviated to N, E, S, W, NE, NW, SE, SW.
//        You can also use IN, OUT, UP and DOWN.\b
//        EXAMINE (= LOOK AT) can be abbreviated to X; X BOOK, EXAMINE BOOK and
//        LOOK AT BOOK all mean the same thing.\b
//        The command INVENTORY (or simply I) will give you a list of what you're
//        carrying.\n
//        Use LOOK or L to repeat the description of your current location.\n
//        AGAIN or G repeats the previous command.\b
//        For full instructions, use the <<aHref('INSTRUCTIONS', 'INSTRUCTIONS',
//                                              'Show full Instructions')>>
//        command.\n
//        To see a sample transcript, use the <<aHref('SAMPLE', 'SAMPLE', 'See
//            sample transcript')>> command. ";
//    }

//    showSample()
//    {
//        "A typical game (not this particular one) might start something like
//        this:\b";
//        inputManager.pauseForMore(nil);
//        "<b>>x me</b>\n
//        You\'re a handsome enough young lad.\b
//        <b>>i</b>\n
//        You're carrying a flashlight and a treasure map.\b
//        <b>>x map</b>\n
//        So far as you can tell, the map indicates that some treasure was hidden
//        underground somewhere on Horatio Bumblespoon\'s property.\b
//        <b>>l</b>\n
//        <b>Bumblespoon's Back Yard</b>\n
//        Suffering from years of neglect, this back yard has been thoroughly
//        overgrown with a tangle of weeds and briars. A path leads north round
//        the side of the house, and you can just make out a narrow staircase
//        leading down into the darkness.\b
//        <b>>x weeds</b>\n
//        The weeds aren\'t important.\b
//        <b>>down</b>\n
//        <b>In the Dark</b>\n
//        It is pitch dark in here.\b
//        <b>>switch on flashlight</b>\n
//        <b>Damp Cellar</b>\n
//        Green mould covers the walls, and a pile of rusty junk clutters one
//        corner. The only way out appears to be back up the stairs.\b
//        <b>>x junk</b>\n
//        It\'s mostly bits of broken gardening equipment, together with the
//        remains of a metal bed-frame. But underneath it all you can just make
//        out what looks like a large chest.\b
//        <b>>x chest</b>\n
//        It\'s a large wooden chest, reinforced with steel bands. It\'s closed.\b
//        <b>>open chest</b>\n
//        You can\'t really get at it with all that junk in the way.\b
//        <b>>move junk</b>\n
//        You manage to move quite a bit of the junk, clearing your access to
//        the wooden chest beneath.\b
//        <b>>open chest</b>\n
//        (first trying to unlock the wooden chest)\n
//        The wooden chest seems to be locked.\b
//        <b>>unlock chest</b>\n
//        What do you want to unlock it with?\b
//        <b>>key</b>\n
//        You see no key here.\b
//        At this point the player knows he or she has to go in search of the key,
//        and presumably that\'s how this game would continue.\b";
//        inputManager.pauseForMore(nil);
//        "And now back to the game you\'re actually playing....\b";
//        me.lookAround(true);

//    }
//;

 /*
  *   The playerHelper is an object that starts a daemon at start of play.
  *   This daemon checks whether the player is making any progress at all,
  *   and also watches the ratio of commands the parser rejects to the
  *   number of turns. If this ratio becomes too high (as defined by the
  *   errorThreshold property) we offer the player a HELP command. If it
  *   becomes very low (as defined by the ceaseCheckingErrorLevel property)
  *   we cease checking (i.e. stop the daemon) on the grounds that the
  *   player doesn't appear to need the kind of help we want to offer. We
  *   first perform a check after firstCheckAfter turns to see if the player
  *   is making any progress, and then after each errorCheckInterval turns
  *   to see if the player is having difficulty entering valid commands.
  *
  *   The idea is to keep offering HELP to an inexperienced player who
  *   clearly needs it, even if the player declined to read any help text at
  *   the start of the game.
  */

playerHelper: InitObject
    execute()
    {
        new Fuse(self, &firstCheck, firstCheckAfter);

        // new OneTimePromptDaemon(self, &showAbout);

        startLocation = gPlayerChar.roomLocation;
    }

    startLocation = nil

    firstCheckAfter = 20
    firstCheckMsg = "<.p><font color=\"#808080\"><b>Tip:</b> Zatím to nevypadá,
        že bys postupoval vpřed. Pokud máš pocit, že by se ti hodila trocha
        pomoci, můžeš o ni požádat zadáním příkazu <<aHref('nápověda',
        'NÁPOVĚDA', 'Požádat o nápovědu')>>.</font> "

    errorCheckInterval = 20

    /*
     *   The proportion of rejected commands to turns (i.e. accepted
     *   commands) that will trigger an offer of help. We express this
     *   number as a percentag.
     */
    errorThreshold = 50

    ceaseCheckingErrorLevel = 5

    /*
     *   By default we'll make the check for no progress the fact that the
     *   player is still in the starting location. This can be overridden as
     *   appropriate.
     */
    firstCheckCriterion()
    {
        return gPlayerChar.location == startLocation;
    }


    firstCheck()
    {
        /*
         *   If the player is still in the starting location (and so hasn't
         *   visited the endOfDriveway, which would trigger its doScript
         *   method) after 5 turns, s/he may be a novice who's getting stuck,
         *   so we'll offer help.
         */
        if(firstCheckCriterion())
        {

            firstCheckMsg;
            /*
             *   We've just offered help, so we'll wait another
             *   errorCheckInterval turns before seeing whether to offer it
             *   again.
             */
            new Fuse(self, &startErrorDaemon, errorCheckInterval);
        }
        else
            startErrorDaemon();
    }

    startErrorDaemon() { errorDaemonID = new Daemon(self, &errorCheck, 1); }

    /* Watch for a high percentage of errors in user input */
    errorCheck()
    {
        local errorPercent = (100 * errorCount) / libGlobal.totalTurns;
        if(errorPercent > errorThreshold)
        {
            "<.p>Potřebujete-li s hrou pomoci, zadejte <<aHref('nápověda',
                'NÁPOVĚDA', 'Požádat o nápovědu')>>.<.p>";
            /*
             *   We don't want to keep showing this message every turn, so
             *   we'll turn the daemon off for twenty turns.
             */

            stopErrorDaemon();
            new Fuse(self, &startErrorDaemon, errorCheckInterval);

        }

        /*
         *   If there are very few input errors, we're probably not needed
         *   any more
         */
        if(errorPercent < ceaseCheckingErrorLevel)
            stopErrorDaemon();

    }

    stopErrorDaemon()
    {
        if(errorDaemonID != nil)
            errorDaemonID.removeEvent();

        errorDaemonID = nil;
    }

    showAbout()
    {
        "<.p><font size=-1>[First-time players may want to type
        <<aHref('ABOUT', 'ABOUT', 'About this
            game')>><<systemInfo(SysInfoInterpClass) == SysInfoIClassHTML ? '
                (or click the hyperlink in this message)' : ''>>; don't forget
        to press the ENTER (or RETURN) key to execute your command after typing
        it. <<extraHintManager.explainExtraHints()>>]\b\b\b</font>";
    }

    errorDaemonID = nil
    errorCount = 0
;

/*
 *   This game is meant to be for novice players. We'll try to keep track of
 *   how many not-understood commands they enter, in case it helps us decide
 *   whether they need help. The playerHelper object will use this
 *   information to decide whether to offer help.
 */

modify playerMessages
    commandNotUnderstood(actor)
    {
        inherited(actor);
        playerHelper.errorCount++;
    }

    askUnknownWord(actor, txt)
    {
        inherited(actor, txt);
        playerHelper.errorCount++;
    }

    wordIsUnknown(actor, txt)
    {
        inherited(actor, txt);
        playerHelper.errorCount++;
    }
;


modify libMessages
    emptyCommandResponse() { emptyCommandResponseList.doScript(); }
    emptyCommandResponseList: StopEventList {
    [
        {: "<font color=\"#808080\"><b>Tip:</b> Pokud jsi chtěl jeden tah nic
            nedělat, použij příkaz <<aHref('čekej', 'ČEKEJ', 'Jeden tah čekej a
            nic nedělej')>>, který můžeš zkrátit na Č. V ostatních případech bys
            měl něco napsat, než stiskneš enter.</font>" },

        {: "<font color=\"#808080\">Napiš prosím něco, než stiskneš enter, i
            kdyby to mělo být jen <<aHref('č','Č', 'Jeden tah čekej a nic
            nedělej')>>.</font> " }
    ]
    }

    distantThingDesc(obj)
    {
        gMessageParams(obj);
        "Musíš přijít blíž, abys viděl {jeho/její obj} detaily. ";
    }

;

//-------------------------------------------------------------------------------
/*
 *   More newbie-helpful bits and pieces, based on work by Emily Short for I6
 *
 *   The idea is to trap the kind of invalid input a newcomer to IF might
 *   type and respond with something a bit more helpful than a standard
 *   parser error message. In some cases we'll execute the command the
 *   player probably meant, and in others we'll just explain why the command
 *   failed.
 */

/*  First, trap attempts to refer to body parts */
bodyParts: MultiLoc, Unthing 'moji mou tvoji tvojí (jeho) (její) ruka/ruky/ruce/
    ruku/rukou/rukám/rukách/rukami/ucho/ucha/uchu/uchem/uši/uší/uším/uších/
    ušima/pěst/pěsti/pěstí/pěstím/pěstích/pěstmi/prst/prstu/prstem/prsty/prstů/
    prstům/prstech/oko/oka/oku/okem/oči/očí/očím/očích/očima/paže/paži/paží/
    pažím/pažích/pažemi/noha/nohy/noze/nohu/nohou/nohám/nohách/nohami/palec/
    palce/palci/palcem/palců/palcům/palcích/palcemi/chodidlo/chodidla/chodidlu/
    chodidlem/chodidel/chodidlům/chodidlech/chodidly/obličej/obličeje/obličeji/
    obličejem/nos/nosu/nose/nosem/ústa/úst/ústům/ústech/ústy/zub/zubu/zubem/
    zuby/zubů/zubům/zubech/jazyk/jazyku/jazykem/ret/rtu/rtem/rty/rtů/rtům/rtech/
    koleno/kolena/kolenu/kolenem/kolen/kolenům/kolenech/koleny/loket/loktu/
    loktem/lokty/loktů/loktům/loktech'
    'část těla' *3

    notHereMsg = 'V příbězích interaktivní fikce obecně není důvod se odkazovat
        na jednotlivé části svého těla. Například příkaz OBUJ BOTY NA NOHY
        nemusí fungovat, místo něj postačí OBUJ BOTY. A pokud ti hra nenaznačí
        opak, tak nebudeš moci ani zadat OTEVŘI DVEŘE NOHOU či DEJ SAFÍROVÝ
        PRSTEN DO PUSY. '

    canMatchThem = true
    initialLocationClass = Room
    isListedInContents = nil
;

/* Trap the use of vague words like 'someone' or 'something' or 'anyone' */

StringPreParser
    doParsing(str, which)
    {
        local ret = rexSearch(pat, str);
        if(ret != nil)
        {
            /* first check whether the word occurs in the dictionary */
            local lst = cmdDict.findWord(ret[3]);

            /* then see if it matches any objects in scope. */
            lst = lst.intersect(gPlayerChar.scopeList);

            /* if not, display a helpful message */
            if(lst.length == 0)
            {
                "It's usually better not to use vague words like
                <q><<ret[3]>></q> in your commands, because the game won't be
                able to guess what you mean. Be more specific and refer to the
                objects in your vicinity. ";
                playerHelper.errorCount++;
                return nil;
            }
        }
        return str;
    }
    pat = static new RexPattern('<NoCase>(someone|something|anyone|anything)')
;

DefineIAction(Vystridej)
    execAction()
    {
        "<.p>Kapitán je živý člověk. Chceš-li ho vystřídat ve službě, promluv si
            s ním o tom. Začni třeba příkazem MLUV S KAPITÁNEM.<.p>";
    }
;

VerbRule(Vystridej)
    ('vystřídej' | 'vystřídat') 'kapitána'
    : VystridejAction
    verbPhrase = 'vystřídat/stříd{áš}/vystřídal{a}'
;

/*
 *   Trap commands like LOOK HERE or SEARCH THERE. We'll actually carry out
 *   a LOOK command, but we'll also tell the player just to use LOOK in
 *   future.
 */
//DefineIAction(LookHere)
//    execAction()
//    {
//        gActor.lookAround(true);
//        playerHelper.errorCount++;
//        "<.p>[For future reference, you don't need to refer to places in the
//        game with words like <q><<placeName>></q>; a simple LOOK command will
//        suffice]<.p>";
//    }
//;

//VerbRule(LookHere)
//    ('l' | 'look' | 'search') ('here'->placeName|'there'->placeName)
//    : LookHereAction
//    verbPhrase = 'look/looking'
//;

/*
 *   Trap the words KINDLY and PLEASE in a player's command, and explain
 *   that they shouldn't be used, giving examples of the kind of commands to
 *   use instead.
 */

StringPreParser
    doParsing(str, which)
    {
        if(rexMatch(pat, str))
        {
            "To je od tebe velice hezké, že používáš slova <q>prosím</q> či
            <q>laskavě</q>, ale opravdu to není potřeba; brzy přijdeš na to, že
            ti hra bude rozumět mnohem lépe, pokud se přidržíš jednoduchých
            příkazů jako JDI NA SEVER, P KLÍČE nebo POLOŽ KRABICI NA STŮL.<.p>";
            playerHelper.errorCount++;
            return nil;
        }
        return str;
    }

    pat = static new RexPattern('<NoCase>%<(prosím|laskavě)%>')
;

/*
 *   Trap any command beginning with USE, telling the player to be more
 *   specific
 */
StringPreParser
    doParsing (str, which)
    {
        if(rexMatch(pat, str))
        {
            "<.p>Na příkaz POUŽIJ (či jemu podobné) neumím reagovat, protože je
                takový neurčitý. Buď prosím konkrétnější, co chceš udělat.<.p>";

            playerHelper.errorCount++;
            return nil;
        }

        return str;
    }

    pat = static new RexPattern('<NoCase>^(po)?u[žz](ij|[íi]t)%>')
;

/*
 *   Trap commands like KEEP GOING NORTH or CONTINUE HEADING WEST. We'll
 *   carry out the directional movement command obviously intended, but
 *   advise the player on the standard form of such commands.
 */
DefineAction(KeepGoing, TravelAction)
    execAction()
    {
        local action = new TravelDirAction(getDirection);
        action.doActionOnce();
        "<.p><font color=\"#808080\"><b>Tip:</b> Ve vtěšině příběhů interaktivní
            fikce je nejen nutné, ale i vhodnější zadávat příkazy tohoto typu
            ve formě samotných směrů: JDI NA SEVER, SEVER (nebo jen S) apod.,
            než POKRAČUJ SMĚREM NA SERVER, VRAŤ SE ZPÁTKY NA JIH
            atd.</font><.p>";
    }
;

VerbRule(KeepGoing)
    ('pokračuj' | 'stále' | 'neustále')
    ('jdi' | 'jít' | 'běž' | 'běžet' | 'miř')
    ('směrem' | 'cestou' | ) ('na' | ) singleDir
    | ('vrať' | 'vrátit') ('se' | ) ('zpět' | 'zpátky' | ) ('na' | ) singleDir
    : KeepGoingAction
    verbPhrase = 'jít/jd{eš}/{šel} ' + dirMatch.dir.name
;

/*
 *   Trap commands that start with a pronoun (e.g. I AM LOST or YOU ARE
 *   SILLY) and advise the player that they are likely to be unproductive,
 *   suggesting the format of commands that are more likely to work.
 *
 *   Note that we have to make exceptions that allow valid commands starting
 *   with I where I is an abbreviation for INVENTORY, such as I itself, I
 *   TALL and I WIDE.
 */

//StringPreParser
//    doParsing(str, which)
//    {
//        if(rexMatch(pat3, str))
//            return str;

//        if(rexMatch(pat, str) || rexMatch(pat2, str))
//        {
//            "If the game is not understanding you, try issuing your commands in
//            the imperative: e.g., THROW THE KNIFE, but not I WOULD REALLY LIKE
//            TO THROW THE KNIFE.  Chatty sentences such as YOU ARE A VERY STUPID
//            GAME will only prove themselves true.\b
//            If you really feel that the game is looking for a word that is not
//            a verb (as the solution to a riddle, eg.) try some variations, such
//            as SAY FLOOBLE.\b
//            If you need more help, try the <<aHref('HELP','HELP', 'Ask for
//                help')>> command.<.p>";
//            playerHelper.errorCount++;
//            return nil;
//        }
//        return str;
//    }
//    pat = static new RexPattern('<NoCase>^(you|he|she|it|they|we|its|theyre'
//                                + '|youre|hes|shes)<Space|squote>')

//    pat2 = static new RexPattern('<NoCase>^(i|im|i<squote>m)<Space>+%w')

//    pat3 = static new RexPattern('<NoCase>^i<Space>+(wide|tall)')
//;

/*
 *   Strip question marks from commands, so they don't cause them to be
 *   rejected.
 */
StringPreParser
    doParsing(str, which)
    {
        return str.findReplace('?', '', ReplaceAll);
    }
    runOrder = 10 // we'll strip question marks early
;

/*
 *   Trap commands like WHERE AM I or WHERE ARE WE or WHAT IS HERE. Perform
 *   a LOOK command but explain that LOOK is the phrasing to use.
 */
DefineIAction(WhereAmI)
    execAction
    {
        gActor.lookAround(true);
        "<.p><font color=\"#808080\"><b>Tip:</b> Příště napiš příkaz
        <<aHref('ROZHLÉDNI SE','ROZHLÉDNI SE', 'Rozhlédnout se')>>, resp. stačí
        napsat jeho zkratku R.</font><.p>";
    }

;

VerbRule(WhereAmI)
    'kde' ('to' | ) ('jsem' | 'jsme')
    : WhereAmIAction
    verbPhrase = 'rozhlédnout/rozhlíž{íš}/rozhlédl{a} se'
;

VerbRule(WhatsHere)
    'co' (('tu' | 'tady') 'je' | 'je' ('tu' | 'tady'))
    : WhereAmIAction
    verbPhrase = 'rozhlédnout/rozhlíž{íš}/rozhlédl{a} se'
;

/*
 *   Trap commands like WHERE CAN I GO. Perform an EXITS command to list the
 *   exits, but then tell the player to use the wording EXITS in future.
 */
DefineIAction(WhereGo)
    execAction()
    {
        nestedAction(Exits);
        "<.p><font color=\"#808080\"><b>Tip:</b> Příště zadej SMĚRY.</font><.p>";
    }
;

VerbRule(WhereGo)
    'kam' ('můžu' | 'mohu' | 'mám' | 'bych' ('měl' | 'měla') | 'bychom'
    ('měli' | 'měly')) ('jít' | 'dojít' | 'zajít' | 'směrovat' | 'směřovat')
    : WhereGoAction
    verbPhrase = 'ukázat/ukazu{ješ}/ukázal{a} směry'
;

/*
 *   Trap commands like WHERE CAN I GET HELP. Print a suitable Help message,
 *   and then explain the use of the HELP command.
 */
//DefineSystemAction(WhereHelp)
//    execSystemAction()
//    {
//        "<.p><font color=\"#808080\"><b>Tip:</b> Příště stačí napsat 
//            NÁPOVĚDA.</font><.p>";
//        helpMessage.printMsg();
//    }
//;

//VerbRule(WhereHelp)
//    ('kde' | 'jak') ('můžu' | 'mohu' | 'bych' ('mohl' | 'mohla') | 'bychom'
//    ('mohli' | 'mohly') | )
//    ('získat' | 'dostat' | 'najít') ('nějakou' | 'trochu' | )
//    ('pomoc' | 'pomoci' | 'nápovědu' | 'radu' | 'návod') ('k' | 'ke' | 'o' | )
//    (singleLiteral | )
//    : WhereHelpAction
//    verbPhrase = 'žádat/žád{áš}/požádal{a} o nápovědu'
//;
/*
DefineSystemAction(Help)
    execSystemAction()   {   helpMessage.printMsg();   }
;
*/
/* Provide grammar to understand a fairly wide variety of requests for help */
/*
VerbRule(Help)
    ('help' | 'assist' | 'assistance' ) |
    'how' ('do' | 'can' | 'does' | 'will' | 'shall' | 'could' | 'should' | 'may'
           | 'must')
    ('i' | 'me' | 'he' | 'she' | 'it' | 'we' | 'you' | 'they' | 'person' | 'one'
     | 'someone' | 'anyone' | 'somebody' | 'anybody') singleLiteral

    : HelpAction
    verbPhrase = 'help/helping with the software'
;

VerbRule(WhatNext)
    'what' ('next' | 'now') |
    'what' ('should' | 'can' | 'do' | 'does' | 'am' | 'is')
    ('i' | 'one' |'anyone' | 'someone')
    (('meant' 'to')|) ('do' | 'try') ('next' | 'now'|)
    : HelpAction
    verbPhrase = 'help/helping with the software'
;
*/

/*
 *   Provide a command to display a brief introduction to playing IF (as an
 *   alternative to the full INSTRUCTIONS menu provided by the TADS 3 library)
 */
//DefineSystemAction(Intro)
//    execSystemAction() { helpMessage.briefIntro(); }
//;

//VerbRule(Intro)
//    ('show'|'view'|) ('brief'|) ('intro' | 'introduction')
//    :IntroAction
//    verbPhrase = 'show/showing brief introduction'
//;

///* Provide a command to show the player a sample transcript */
//DefineSystemAction(Sample)
//    execSystemAction() { helpMessage.showSample(); }
//;

//VerbRule(Sample)
//    ('show'|'view'|) 'sample' ('transcript' | )
//    : SampleAction
//    verbPhrase = 'show/showing sample transcript'
//;

/*
 *   Provide a response to WHO AM I. We provide a brief explanation and then
 *   perform an EXAMINE ME command to add any game-specific player character
 *   description.
 */
DefineIAction(WhoAmI)
    execAction()
    {
        "<font color=\"#808080\"><b>Tip:</b> Ty jsi hlavní postavou hry. Autor
            hry ti samozřejmě mohl dát nějaký bližší popis. Na něj se můžeš
            podívat tak, že napíšeš <<aHref('p sebe','P SEBE','Prozkoumej sebe')
            >>, což je rychlejší a obvyklejší.</font>\b";
        replaceAction(Examine, gPlayerChar);
    }
;

VerbRule(WhoAmI)
    ('kdo' | 'co') 'jsem' ('já' | ) ('zač' | )
    : WhoAmIAction
    verbPhrase = 'zeptat/pt{áš}/zeptal{a} se kdo jsi'
;

/*
 *   Make WHAT IS X behave like EXAMINE X, but then explain the standard
 *   phrasing of an EXAMINE command.
 */

//VerbRule(WhatIsNoun)
//    ('co' ('je' | 'jsou') ('zač' | )
//    ('ten' | 'ta' | 'to' | 'ti' | 'ty' | 'ta' | )) dobjList
//    : ExamineAction
//    verbPhrase = 'prozkoumat/zkoum{áš}/prozkoumal{a} (co)'
//;

//modify Thing
//    mainExamine()
//    {
//        if(gAction.getEnteredVerbPhrase.startsWith('co'))
//            "<font color=\"#808080\"><b>Tip:</b> Příště můžeš to samé udělat
//                zadáním <<aHref('p '+ nameKohoCo, 'P ' + nameKohoCo.toUpper,
//                'Prozkoumej ' + nameKohoCo)>>, což je o poznání rychlejší a
//                obvyklejší.</font>\b";
//        inherited();
//    }
//;

/*
 *   Trap a variety of commands of the sort WHAT IS GAME ABOUT or WHATS THE
 *   POINT and respond by showing the game's ABOUT text.
 */
//VerbRule(WhatsThePoint)
//    ('whats' | 'what' 'is') ('the'|) ('point' | 'idea' | 'goal' | 'purpose')
//    (singleLiteral | )
//    : AboutAction
//    verbPhrase = 'ask/asking about the point of the game'
//;

//VerbRule(WhatThisGame)
//    ('whats' | 'what' ('is'|'are')) ('the' | 'these' | 'this' |)
//    ('game' | 'story' | 'program' | 'games' | ('interactive' 'fiction'))
//        ('for' | 'about' | )
//    : AboutAction
//    verbPhrase = 'ask/asking what the game is about'
//;


/*
 *   Trap a variety of vague travel commands like GO SOMEWHERE or WALK
 *   AROUND or TURN RIGHT and explain how movement commands should be
 *   phrased. Then display a list of available exits from the current
 *   location.
 */
DefineIAction(GoSomewhere)
    execAction()
    {
        "Pokud chceš někam jít, zadej některý ";
        if(gActor.getOutermostRoom.ofKind(ShipboardRoom))
            "ze směrů na palubě lodi (PŘÍĎ, LEVOBOK, ZD, PB atp.). ";
        else
            "z kompasových směrů (SEVER, VÝCHOD, JZ, Z, J atd.). ";
        
        "Odsud \v";
        gActor.roomLocation.cannotGoShowExits(gActor);
    }
    actionTime = 0
;

VerbRule(GoSomewhere)
    ('go' | 'walk' | 'turn' | 'proceed' | 'run') ( | ('to' 'the') )
    ('left' | 'right' | 'on' | 'onwards' | 'onward' | 'forward' | 'forwards' |
     'around' | 'somewhere' | (('straight'| ) 'ahead'))
    : GoSomewhereAction
    verbPhrase = 'go/going somewhere'
;

modify VagueTravelAction
    execAction()  {  delegated GoSomewhereAction;  }
;

/*
 *   The StringPreparser and the TopicAction which follows are designed to
 *   deal with command like LOOK FOR X, FIND Y or SEARCH FOR Z. The
 *   complication is that in the standard library these are all forms of
 *   LOOK UP X, which prompt the response "What do you want to look that up
 *   in?". This is likely to confuse new players.
 *
 *   The StringPreparser checks to see if there's a Consultable in the
 *   current location. If so, then we use the standard library handling, on
 *   the assumption that the player is trying to looking something up in it.
 *   If not we change the command to SEEK X in order to invoke the new
 *   SeekAction defined below.
 */

//StringPreParser
//    doParsing(str, which)
//    {
//        if(rexMatch(pat, str) && me.getOutermostRoom.allContents.indexWhich({ x:
//            x.ofKind(Consultable)}) == nil)
//        {
//            str = rexReplace(pat, str, 'seek', ReplaceOnce);
//        }
//        return str;
//    }
//    pat = static new RexPattern('^<NoCase>(find|look for|search for|hunt for)')
//;

/*
 *   SeekAction is designed to handle of commands FIND X, LOOK FOR Y or
 *   SEARCH FOR Z, when they don't seem to be intended as attempts to look
 *   something up in a Consultable. We make it a TopicAction so that it will
 *   match whatever the player types, and so not give away any premature
 *   spoilery information by the nature of the parser's response.
 *
 *   The appropriate response then depends on the player character's state of
 *   knowledge. In the most general case the player is simply given
 *   instructions on how to go about looking for things. This hardly seems
 *   appropriate, however if the object requested is in plain sight, in
 *   which case we point this out to the player. As a courtesy to the
 *   player, we also remind him or her of where an object was last seen, if
 *   it has been seen.
 *
 *   One or two complications need to be dealt with. If the player finds
 *   something like FIND SMELL or FIND NOISE then we should describe it as
 *   having been smelt or heard elsewhere, not seen. We also want to make
 *   sure that the command never matches an Unthing in preference to a
 *   Thing, and that if an Unthing is matched it is not described as being
 *   present.
 */

//DefineTopicAction(Seek)
//    execAction()
//    {
//        local obj = getBestMatch();
//        gMessageParams(obj);
//        if(obj && obj.ofKind(Unthing))
//        {
//            say(obj.notHereMsg);
//            return;
//        }

//        if(obj == gPlayerChar)
//        {
//            "If you{'ve|d} managed to lose yourself, things must {be|have been}
//            desperate! But don't worry, you{'re| were} right {|t}here. ";
//            return;
//        }

//        if(obj && obj.ofKind(Thing) && gActor.hasSeen(obj))
//        {
//            local loc = obj.location;
//            if(loc == nil)
//            {
//                if(obj.isIn(gActor.getOutermostRoom))
//                    loc = gActor.getOutermostRoom;
//                else if(obj.ofKind(MultiLoc) && obj.locationList.length > 0)
//                    loc = obj.locationList[1];
//            }


//            if(obj.isIn(gActor))
//                "{You/he} {are} carrying {it obj/him}. ";
//            else if(gActor.canSee(obj) && loc != nil)
//            {
//                "{The obj/he} {is} ";
//                if(loc == gActor.getOutermostRoom)
//                    "right {here|there}. ";
//                else
//                    "<<obj.isIn(gActor.getOutermostRoom) ? '' : 'nearby, '>>
//                    <<locDesc(obj, loc)>>. ";
//            }
//            else
//                "{You/he} {|had} last <<senseDesc(obj)>> {the obj/him}
//                <<locDesc(obj, obj.lastSeenIn)>>. ";
//        }
//        else
//            "If you want to find something you can\'t see, you'll have to hunt
//            for it. If you think it may be in your current location, try
//            examining things, searching things, opening things that open, or
//            looking in, under and behind things. If what you\'re looking for
//            could be elsewhere, you may have to go elsewhere to find it. ";

//    }

//    /*
//     *   gTopic.getBestMatch() may not give the best results for our
//     *   purposes. The following code is designed to prefer Things to
//     *   Unthings, and then to prioritize what the player char can see over
//     *   what s/he has seen, and both over what s/he only knows about. If
//     *   none of these find a match, we then revert to gTopic.getBestMatch.
//     */

//    getBestMatch()
//    {
//        local obj;
//        if((obj = gTopic.inScopeList.valWhich({x: gActor.canSee(x) &&
//                                              !x.ofKind(Unthing)})) != nil)
//            return obj;

//        local lst = gTopic.inScopeList + gTopic.likelyList + gTopic.otherList;
//        if((obj = lst.valWhich({x: gActor.hasSeen(x) && !x.ofKind(Unthing)})) != nil)
//            return obj;

//        if((obj = lst.valWhich({x: gActor.knowsAbout(x)})) != nil)
//            return obj;

//        return gTopic.getBestMatch();

//    }

//    locDesc(obj, loc)
//    {
//        if(obj.ofKind(SensoryEmanation))
//            "coming from <<loc.ofKind(Room) ? loc.destName : loc.name>>";
//        else if(loc.ofKind(Room))
//            "<<loc.inRoomName(gActor)>>";
//        else
//            "<<loc.objInName>>";
//    }

//    senseDesc(obj)
//    {
//        if(obj.ofKind(Noise))
//            return 'heard';
//        if(obj.ofKind(Odor))
//            return 'smelt';
//        return '{saw|seen}';
//    }

//    createTopicResolver(issuingActor, targetActor)
//    {
//        return delegated ConvTopicTAction(issuingActor, targetActor);
//    }
//;

//VerbRule(Seek)
//    ('seek' | ('hunt' 'for')) singleTopic
//    : SeekAction
//    verbPhrase = 'seek/seeking (what)'
//;

//modify Thing
//    lastSeenIn = nil
//    noteSeenBy(actor, prop)
//    {
//        if(actor == gPlayerChar)
//            lastSeenIn = (location == nil ? actor.getOutermostRoom : location);
//        inherited(actor, prop);
//    }
//;
