#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   A special precondition requiring that the actor's hands are empty.
 *
 *   If the actor is carrying the tote bag, we'll try auto-bagging
 *   everything to free up space: the tote bag is wearable, and everything
 *   fits into it, so we might be able to go hands-free that way.
 *   Otherwise, we won't try anything implied; it's not always desirable to
 *   just put everything down, so in this case we'll simply enforce the
 *   requirement that the actor is holding nothing...
 *
 *   Batoh zde nepoužíváme, ale nechtěl jsem funkcionaliu vysloveně mazat.
 *   Řešení pochází ze hry Return to Ditch Day.
 */
toteBag: BagOfHolding, Wearable, ComplexContainer;
handsEmpty: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /*
         *   if they're not holding anything that we don't allow them to be
         *   holding, we can proceed with the command without further ado.
         */
        if (gActor.contents.indexWhich({x: requireDropping(x)}) == nil)
            return nil;

        /*
         *   if implied commands are allowed, and auto-bagging is allowed,
         *   and the actor is carrying the tote bag, try moving things into
         *   bags.
         */
        if (allowImplicit && autoBag && toteBag.isIn(gActor))
        {
            local aff;

            /* get the bag-of-holding affinities for items being held */
            aff = gActor.getBagAffinities(gActor.contents
                                          .subset({x: x.isHeldBy(gActor)}));

            /*
             *   Putting everything away could result in a lot of
             *   individual actions.  Rather than announcing each one
             *   individually, preface the actions with a summary
             *   announcement that says simply "putting everything away";
             *   then, after we're done, we'll remove the individual
             *   announcements that we generated...
             */
            gTranscript.addReport(new ImplicitHandsEmptyAnnouncement(
                gAction, &announceImplicitAction));

            /* move things into their respective bags */
            foreach (local cur in aff)
            {
                /* put it in the bag if possible */
                if (!cur.bag_.isIn(cur.obj_)
                    && !cur.bag_.tryPuttingObjInBag(cur.obj_))
                    break;

                /*
                 *   if we failed to accomplish that, give up without any
                 *   further message (as the failed attempt will have
                 *   generated an appropriate error report).
                 */
                if (!cur.obj_.isIn(cur.bag_))
                    exit;
            }

            /*
             *   Clean up the individual reports - these aren't needed, as
             *   we added the summary report at the beginning of the group
             *   of actions.  For each implicit action we performed
             *   directly, make the action's announcement silent...
             */
            gTranscript.forEachReport(new function(cur)
            {
                /*
                 *   if this is an implied action, and it was triggered
                 *   directly by our action, we can get rid of it.
                 */
                if (cur.ofKind(ImplicitActionAnnouncement)
                    && (cur.getAction().getOriginalAction().parentAction
                        == gAction))
                {
                    /* make this announcement silent */
                    cur.makeSilent();
                }
            });

            /* indicate that we did something implicit */
            return true;
        }

        /* we can't try moving anything around, so simply give up */
        reportFailure(failureMsg);
        exit;
    }

    /* can we try moving things into bags as a first resort? */
    autoBag = true

    /* the failure message */
    failureMsg = 'Nejprve budeš muset vše odložit. '

    /*
     *   Are we required to drop the given object?  By default, we'll
     *   require dropping anything being held...
     */
    requireDropping(obj) { return obj.isHeldBy(gActor); }
;

/* special implied action announcement for emptying hands */
class ImplicitHandsEmptyAnnouncement: ImplicitActionAnnouncement
    /* use custom message text */
    getMessageText([params]) { return 'odkládáš všechno stranou'; }

    /* the bag of holding we're moving things into */
    bag_ = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   PreCondition zajišťující, že postava svlékne či oblékne určité oblečení.
 *   Používáme např. na zouvání a obouvání bot.
 */
objsNotWorn: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /* check each object to see if it's worn */
        foreach (local wornobj in objList)
        {
            /* if it's worn, try taking it off */
            if (wornobj.isWornBy(gActor))
            {
                /* if allowed, take it off implicitly */
                if (allowImplicit
                    && autoDoff
                    && tryImplicitAction(Doff, wornobj))
                {
                    /* make sure it's not being worn */
                    if (wornobj.isWornBy(gActor))
                        exit;

                    /* indicate that we performed an implied command */
                    return true;
                }
                else
                {
                    /* failed */
                    gMessageParams(wornobj);
                    reportFailure(failureMsg);
                    exit;
                }
            }
        }

        /* we didn't find anything worn, so we can proceed */
        return nil;
    }

    /* the objects that can't be worn */
    objList = []

    /* can we automatically remove these items? */
    autoDoff = true

    /* the failure message */
    failureMsg = 'Nejprve {mus[íš]|[jsi] musel[a]} {kohoco wornobj} sundat. '
;

objWorn: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /* if the object is already held, there's nothing we need to do */
        if (gActor != me || obj == nil || obj.isWornBy(gActor))
            return nil;

        /* the object isn't being held - try an implicit 'take' command */
        if (obj.isIn(gActor.getOutermostRoom()) && allowImplicit
            && obj.tryWearing())
        {
            /*
             *   we successfully executed the command; check to make sure
             *   it worked, and if not, abort the command without further
             *   comment (if the command failed, presumably the command
             *   showed an explanation as to why)
             */
            if (!obj.isWornBy(gActor))
                exit;

            /* tell the caller we executed an implicit command */
            return true;
        }

        /* it's not held and we can't take it - fail */
        reportFailure('Měl by sis obout boty, gravitace je tu nepatrná, ale mohl
            by sis při odrážení něco vrazit do chodidla. ', obj);

        /* make it the pronoun */
        gActor.setPronounObj(obj);

        /* abort the command */
        exit;
    }

    /* lower the likelihood rating for anything not being held */
    verifyPreCondition(obj)
    {
        /* if the object isn't being held, reduce its likelihood rating */
        if (obj != nil && !obj.isWornBy(gActor))
            logicalRankOrd(80, 'implied wear', 150);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   PreCondition zajišťující, že postava odloží určité objekty. Používáme např.
 *   aby hráč neodnesl skafandr z kupole a další.
 */
objNotHeld: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /* if the object isn't being worn, we have nothing to do */
        if (obj == nil || !obj.isIn(gActor))
            return nil;

        /* try an implicit 'doff' command */
        if (allowImplicit && tryImplicitAction(Drop, obj))
        {
            /*
             *   we executed the command - make sure it worked, and abort
             *   if it didn't
             */
            if (obj.isIn(gActor))
                exit;

            /* tell the caller we executed an implicit command */
            return true;
        }

        /* report the problem and terminate the command */
        reportFailure('Tohle {nemůž[eš]|[jsi] nemohl[a]} udělat, dokud
            {kohoco obj} {drž[íš]|[jsi] držel[a]}. ');

        /* make it the pronoun */
        gActor.setPronounObj(obj);

        /* abort the command */
        exit;
    }

    /* lower the likelihood rating for anything being worn */
    verifyPreCondition(obj)
    {
        /* if the object is being worn, reduce its likelihood rating */
        if (obj != nil && obj.isIn(gActor))
            logicalRankOrd(80, 'implied drop', 150);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   PreCondition zajišťující automatické vypnutí resp. zapnutí určitého
 *   objektu. Používáme např. pro vypnutí běžeckého pásu, než z něj postava
 *   sleze atp.
 */
objTurnedOff: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /* if we're already turned off, we're good to go */
        if (!obj.isOn)
            return nil;

        /* try turning it off */
        if (allowImplicit && tryImplicitAction(TurnOff, obj))
        {
            /* make sure that worked */
            if (obj.isOn)
                exit;

            /* tell the caller we performed an implied action */
            return true;
        }

        /* couldn't turn it off implicitly, so complain and give up */
        gMessageParams(obj);
        "Nejpve budeš muset {kohoco obj} vypnout. ";
        exit;
    }
;

objTurnedOn: PreCondition
    checkPreCondition(obj, allowImplicit)
    {
        /* if we're already on, we're good to go */
        if (obj.isOn)
            return nil;

        /* try turning it on */
        if (allowImplicit && tryImplicitAction(TurnOn, obj))
        {
            /* make sure that worked */
            if (!obj.isOn)
                exit;

            /* tell the caller we performed an implied action */
            return true;
        }

        /* couldn't turn it off implicitly, so complain and give up */
        gMessageParams(obj);
        reportFailure('Nejprve budeš muset {kohoco obj} zapnout. ');
        exit;
    }
;
