#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <bignum.h>
#include <cs_cz.h>

/*
 *   Několik témat do tabletu, když ho hráč použije jako zdroj informací
 *   k vyhledání v encyklopedii. Některá kritická témata necháme na začátku jako
 *   neznámá, aby se nedala najít dříve, než se o nich hráč ve hře dozví.
 */
tPluto: Topic
    '(let) (na) AV53-81/AV-53-81/AV-5381/pluto'
    isKnown = nil
;

/*
 *   Některá slova ve slovníku jsou natolik obecná či kolidují s něčím dalším,
 *   že je musíme dát do závorek. Ty říkají, že je hráč může použít, ale jen
 *   k některému normálnímu slovu, pokud řekne jen uzávorkované slovo,
 *   nereagujeme.
 */
tCmx12: Topic
    '(havarovaná) (havarované) (rypadla) (rypadlo) cmx12/cmx-12'
    isKnown = nil
;

tRab4: Topic
    '(rypadlo) ra-b4/rab4'
    isKnown = nil
;

tAnomaly: Topic
    'geologická geologické geologickou anomálie/anomálii/anomálií'
;

tBella: Topic
    'ivan ivana ivanovi ivanu ivanem bella/belly/bellovi/bellu/belou'
;

tMicroscope: Topic
    'elektronový mikroskop tem sem rusk-31/zworyk-42'
;

tMoon: Topic
    'měsíc/měsíce/měsíci/měsícem'
;

tVunidip: Topic
    'vunidip/vunidipu/vunidipem'
;

tLagrangianPoint: Topic
    'lagrangeův lagrangeova lagrangeově lagrangeovým (bod)/(bodu)/(bodem)'
;

tSpacepedia: Topic
    'spacepedie/spacepedii/spacepedií'
;

tTychosComplex: Topic
    'tychův tychova tychově tychovým komplex/komplexu/komplexem'
;

tTechnouniverse: Topic
    'technouniverse (corp)/(corporation)/(ltd)'
;

tApl21: Topic
    'apl21/apl-21'
;

tApl22: Topic
    'apl22/apl-22'
;

tRecycledFood: Topic
    'potravinový potravinového potravinovému potravinovém potravinovým recyklát/
        recyklátu/recyklátem'
;

/* ------------------------------------------------------------------------ */
/*
 *   AKCE ZAMÍŘIT
 */
DefineTIAction(AimAt);

/*
 *   Svislítko značí alternativy a závorky tvoří skupinu, která buď omezuje,
 *   kam až sahá alternativa vytvořená svislítkem, pokud je svislítko uvnitř,
 *   nebo pokud le svislítko zvenku, tak závorky tvoří celý nerozdělitelný blok.
 */
VerbRule(AimAt)
    ('zamiř' | 'zamířit' | 'zaměřit' | 'miř' | 'mířit' | 'namiř' | 'namířit'
    | 'ukaž' | 'ukázat') singleDobj 'na' singleIobj
    | ('zamiř' | 'zamířit' | 'zaměřit' | 'miř' | 'mířit' | 'namiř' | 'namířit'
    | 'ukaž' | 'ukázat') 'na' singleIobj singleDobj
    : AimAtAction

    /* Na co chceš zamířit? Hráč může odpovědět s předložkou "na vozítko". */
    askIobjResponseProd = onSingleNoun
    verbPhrase = 'zamířit/zaměřu{ješ}/zaměřil{a} (co) (na co)'
;

/*
 *   S badnessem napíšeme variantu příkazu, ve které chybí iobj slot. Místo toho
 *   zkonstruujeme prázdnou frázi, která vyvolá otázku parseru nebo automatické
 *   doplnění.
 */
VerbRule(AimAtWhat)
    [badness 500] ('zamiř' | 'zamířit' | 'zaměřit' | 'miř' | 'mířit' | 'namiř'
    | 'namířit') singleDobj
    : AimAtAction
    verbPhrase = 'mířit/míř{íš}/mířil{a} (čím) (na co)'

    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = onSingleNoun;
    }
;

/*
 *   Varianta, ve které chybí dobj.
 */
VerbRule(AimAtImplicit)
    ('zamiř' | 'zamířit' | 'zaměřit' | 'miř' | 'mířit' | 'namiř' | 'namířit'
    | 'ukaž' | 'ukázat') 'na' singleIobj
    : AimAtAction
    verbPhrase = 'mířit/míř{íš}/mířil{a} (čím) (na co)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty indirect object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

/*
 *   Když máme akci hotovou, musíme ji zapojit do hry. S většinou objektů by
 *   akce neměla fungovat, protože pro ně nedává smysl. Proto na třídě Thing,
 *   která je předkem všech běžných fyzických objektů ve hře, nadefinujeme akci
 *   tak, že zobrazí patřičnou chybovou zprávu.
 */
modify Thing
    dobjFor(AimAt)
    {
        preCond = [objHeld]
        verify() { illogical('{Kdoco dobj} není něco, co by se dalo
            namířit. '); }
    }
    iobjFor(AimAt)
    {
        preCond = [objVisible]
    }
    cannotAimAtMsg = 'Na {kohoco iobj} nemůžeš {kýmčím dobj} namířit. '
;

/*
 *   Intangible objekty jsou např. zvuky, pachy či v našem případě naoranžovělá
 *   záře v místnosti. Na takové objekty nejde zamířit.
 */
modify Intangible
    iobjFor(AimAt)
    {
        verify() { illogical(cannotAimAtMsg); }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Společná třída pro aplikace naprogramované v tabletu. Zajišťujeme jejich
 *   objevování a skrývání.
 */
class TabletApp: PresentLater, Component
    initiallyPresent = nil
    plKey = 'apps'
    appName = nil
;

/* Kliknutí je totéž, jako zmáčknutí. */
VerbRule(Click)
    ('klikni' | 'kliknout') ( | 'na') dobjList
    : PushAction
    verbPhrase = 'kliknout/klik{áš}/kliknul{a} (na co)'
;

/* ------------------------------------------------------------------------ */
/*
 *   TABLET
 *
 *   Tablet je jeden z nejsložitějších objektů, protože má spoustu funkcí. Dědí
 *   z OnOffControl, aby reagoval na příkazy zapnout/vypnout a rozpoznával tyto
 *   dva své stavy, z TestGearAttachable aby šel připojit k UniDiPu a nakonec
 *   Consultable, aby v něm mohl hráč vyhledávat informace o různých věcech.
 */
tablet: OnOffControl, TestGearAttachable, Consultable
    'tablet/displej/display' 'tablet' *2 @ycRightCompartment
    desc()
    {
        /*
         *   Popis rozlišíme podle toho, zda je hráč na světle nebo v temném
         *   podpalubí. V TADSu jsou čtyři úrovně osvětlení, 0 znamená potmě
         *   bez možnosti manipulace a 1 je takový objekt, který sice potmě,
         *   ale dovoluje s sebou manipulovat po hmatu. Na vizuální popis
         *   potřebujeme osvětlení alespoň 2.
         */
        if(senseAmbientMax([sight]) > 1) "Tablet je docela obyčejný až strohý
            tmavošedý rámeček z neklouzavého plastu okolo nepoškrábatelného
            displeje s jediným tlačítkem, které slouží k zapínání a návratu na
            úvodní obrazovku. Vše na tabletu je přísně účelné. Lze na něm
            používat jednoduché aplikace, hledat v databázi nebo tablet namířit
            kamerou v zadní stěně na zařízení a to tabletem třeba i ovládat.
            <<screenDesc>> ";
        else "Nahmatal jsi tablet, z jedné strany má hladký displej s tlačítkem
            a z druhé zaoblený zadní kryt. ";
    }

    wrongAttachmentMsg = 'Nem{ůž[eš]|ohl[a] [jsi]} {ho/ji dobj} připojit
        {k/ke iobj} {komučemu iobj}, konektory nejsou kompatibilní. '

    tasteDesc = "Olízl jsi plast tabletu. Brr, chutná po potu lidských rukou. "

    /*
     *   Upravujeme jas tabletu podle toho, zda mu svítí displej. Jas úrovně 1
     *   znamená, že tablet sice nic neosvítí, ale postava s ním může
     *   manipulovat, tj. jakoby ho vidí ve tmě.
     */
    brightness { return isOn ? 3 : 1; }

    /*
     *   Při zapínání a vypínání tabletu zajišťujeme démona, který automaticky
     *   zhasíná displej při nečinnosti.
     */
    brightSpan = 0
    brightDaemonId = nil
    makeOn(val)
    {
        if(val)
        {
            "Zmáčkl jsi tlačítko a displej se rozzářil. ";

            /*
             *   Metodu makePresentByKeyIf voláme přímo na třídě PresentLater,
             *   nikoliv na objektu. Zviditelní všechny PresentLater objekty
             *   označené klíčem plKey = 'apps'. Jako podmínku posíláme lambda
             *   funkci (callback), který vrací true nebo nil a tím vybere
             *   aplikaci podle názvu v runningApp proměnné.
             */
            PresentLater.makePresentByKeyIf('apps',
                {x: x.appName == runningApp});

            /* Démon snižuje brightSpan, až dojde na nulu, zhasne. */
            brightSpan = 4;
            brightDaemonId = new Daemon(self, &brightDaemon, 1);
        }
        else
        {
            "Vypnul jsi displej, aby se šetřila energie. ";

            PresentLater.makePresentByKeyIf('apps', nil);

            brightDaemonId.removeEvent();
            brightDaemonId = nil;
        }

        inherited(val);
    }

    /* Tuto funkci voláme vždy, když hráč s tabletem něco dělá. */
    notifyActivity()
    {
        brightSpan = 4;
    }

    brightDaemon()
    {
        /*
         *   Jakmile si už hráč tabletem svítí v podpalubí, tak ho přestaneme
         *   zhasínat.
         */
        if(!self.isIn(lowerDeck)) brightSpan--;

        if(brightSpan == 0)
        {
            /* O zhasnutí neříkáme, když hráč tablet nevidí. */
            if(me.canSee(tablet))
                "<.p>Tablet zhasl displej, aby šetřil energii. ";

            /*
             *   Vypneme ho manuálně, protože chceme jiný popisek, než když se
             *   vypne sám. Nevolame tedy makeOn(nil).
             */
            isOn = nil;
            PresentLater.makePresentByKeyIf('apps', nil);

            brightDaemonId.removeEvent();
            brightDaemonId = nil;
        }
    }

    /*
     *   Pokud hráči dojde, že tablet vydává světlo, musí být schopen zadat
     *   příkaz "rozsviť tablet". V žádném případě ale nechceme ale, aby to
     *   udělala za hráče implicitní akce, proto nonObvious.
     */
    dobjFor(Light)
    {
        verify() { nonObvious; }
        action() { replaceAction(Push, tabletButton); }
    }
    dobjFor(Extinguish)
    {
        verify() { nonObvious; }
        action() { replaceAction(TurnOff, tablet); }
    }

    dobjFor(TurnOn)
    {
        action() { replaceAction(Push, tabletButton); }
    }

    /*
     *   Tohle je spíše pro srandu a tak nechceme, aby se parser ptal, co chce
     *   hráč otevřít a nabízel mu otevření tabletu, proto nonObvious.
     */
    dobjFor(Open)
    {
        verify() { nonObvious; }
        check() { failCheck('Tablet nemůžeš otevřít. Je zakázáno nahlížet do
            výrobků bez podepsání velmi restriktivní NDA. Toto mají dovolenou
            pouze technici žijící v uzavřených opravářských komunitách bez
            kontaktu s okolním světem, aby nemohli vyzradit výrobní
            tajemství.'); }
    }

    dobjFor(Take)
    {
        action()
        {
            achievement.awardPointsOnce();
            inherited;
        }
    }
    achievement: Achievement { +1 "nalezení tabletu" }

    /*
     *   Tablet má na sobě kameru a tak dokáže o různých elektronických
     *   objektech zjišťovat informace a ovládat je tím.
     */
    target = nil
    dobjFor(AimAt)
    {
        /* Namíření tabletu ho zároveň zapne. */
        preCond = inherited + objTurnedOn
        verify()
        {
            if(gIobj != nil && gIobj == self)
                illogicalSelf('Tablet nemůže mířit sám na sebe. ');
        }
        action()
        {
            /*
             *   Namíření tabletu je obsluhováno zde v akci pro dobjFor(AimAt)
             *   i u vybraných objektů v iobjFor(AimAt). Tím, že zde vypíšeme
             *   hlášku přes defaultReport, tak jakákoliv konkrétní hláška iobj
             *   tuto základní přebije. Použije se tedy jen u objektů, které
             *   nemají specifické chování pro namíření tablet.
             */
            defaultReport('Pečlivě {jsi} {kýmčím dobj} zamířil na {kohoco iobj},
                ale nic zvláštního se nestalo. Asi proto, že tablet reaguje jen
                na elektronická zařízení. ');

            target = gIobj;
            notifyActivity();
        }
    }

    /*
     *   Přepínání aplikací na tabletu. Vlastnost runningApp obsahuje název
     *   aplikace, která momentálně na tabletu běží. Metoda runApp přepne
     *   aplikaci a pokud nebude quite true, pak ohlásí obsah displeje.
     */
    runningApp = 'home'
    runApp(app, quite = nil)
    {
        if(!tablet.isOn) makeOn(true);

        if(runningApp != app)
        {
            runningApp = app;

            /*
             *   Vyvoláme všechny PresentLater objekty s klíčem 'apps', jejichž
             *   vlastnost appName má hodnotu spouštěné aplikace. To divné ve
             *   složených závorkách je anonymní funkce (callback) definující
             *   filtr.
             */
            PresentLater.makePresentByKeyIf('apps', {x: x.appName == app});
        }

        if(!quite) tablet.screenDesc();
        notifyActivity();
    }

    /*
     *   Tato metoda popíše obsah obrazovky, tj. nechá aktuální aplikaci popsat
     *   svůj stav.
     */
    screenDesc()
    {
        if(!isOn)
        {
            "Jeho displej je momentálně zhasnutý. ";
        }
        else
        {
            /* Když tablet zkoumáme zapnutý, tak nechceme, aby zhasl. */
            notifyActivity();

            switch(runningApp)
            {
            case 'home':
                appHome.desc();
                break;
            case 'database':
                appDatabase.desc();
                break;
            case 'news':
                appNews.desc();
                break;
            case 'treadmill':
                appTreadmill.desc();
                break;
            case 'robot':
                appRobot.desc();
                break;
            case 'canons':
                appCanons.desc();
                break;
            }
        }
    }

    /* Tablet můžeme také otestovat UniDiPem. */
    probeWithUnidip()
    {
        "<.p>Přístroj chvilí bzučel a poblikával kontrolkami.

            <.p>Na displeji testovaného tabletu se začaly střídat různé barevné
            tvary po chvíli vystřídané počítadlem testované paměti, pak
            ukazateli přenosových rychlostí.

            <.p>Po dlouhé době se na displeji objevila vodorovná lehce se
            vlnící čára a ozval se reproduktor: <q>raz dva tři, raz dva tři,
            zkouška mikrofonu…</q> a během toho se čára měnila v chaotickou
            křivku přesně podle zvuku. <q>Mikrofon funguje,</q> prohlásil tablet
            spokojeně.

            <.p>Pak tablet zhasl a na obrazovce UniDiPu se začal objevovat
            dlouhý výpis provedených testů, všechny zakončené zářivě zeleným OK.

            <.p>Nakonec se zobrazil nápis: <q>Vše v pořádku, závada
            nenalezena.</q> ";
    }

    /*
     *   Tablet obsahuje resp. má přístup do řady databází, ve kterých dokáže
     *   hledat užitečné (i neužitečné) informace.
     */
    dobjFor(ConsultAbout)
    {
        /* Hráč při tom drží tablet v rukou. */
        preCond = inherited + objHeld;
        verify()
        {
            /*
             *   Když hráč v potemnělém podpalubí zadá příkaz "hledej zdroj
             *   světla", tak by postava začala hledat ve Spacepedii a omylem
             *   vyřešila puzzle.
             */
            if(me.isIn(lowerDeck) && lowerDeck.scoreMarker.scoreCount == 0)
                nonObvious;
        }
        action()
        {
            /*
             *   ConsultAbout akce je přímo v tabletu samotném, ale pro efekt
             *   zapneme příslušnou aplikaci na obrazovce, předáním true se to
             *   udělá potichu.
             */
            tablet.runApp('database', true);

            inherited();
        }
    }
    dobjFor(TypeLiteralOn)
    {
        verify() { }
        action()
        {
            /*
             *   Literární akci nemůžeme tak snadno přesměrovat, proto ji
             *   přepíšeme a vrátíme zpátky ke standardnímu zpracování.
             */
            throw new ReplacementCommandStringException('najdi ' + gLiteral
                + ' v tabletu', gIssuingActor, gActor);
        }
    }

    gcName = 'tabletu, tabletu, tablet, tabletu, tabletem'
    gcVocab = 'displeje displeji displejem displaye displayi displayem tabletu/
        tabletem'
;

+ tabletButton: ComponentDeferrer, Button, Component
    'zapínací domů zpět home tlačítko (na) tabletu/tlačítko'
    'zapínací tlačítko' *4
    "Jediné fyzické tlačítko, které na tabletu je. Slouží k zapnutí displeje a
        případně k návratu na úvodní nabídku. "

    /* Tlačítko musí být nahmatatelné i ve tmě. */
    brightness = 1

    /*
     *   Když se bude parser ptát, které tlačítko má hráč na mysli, zvolíme
     *   raději jinou frázi, než "zapínací tlačítko", aby bylo pochopitelné, co
     *   parser nabízí za tlačítko.
     */
     disambigName = 'tlačítko tabletu'
     disambigNameKohoCo = 'tlačítko tabletu'

    dobjFor(Push)
    {
        /*
         *   Při zapnutí tabletu ho hráč automaticky sebere. ObjectPreCondition
         *   zajišťuje aplikaci podmínky na jiný objekt, než ve kterém jsme.
         */
        preCond = inherited + new ObjectPreCondition(tablet, objHeld);
        action()
        {
            if(!tablet.isOn)
            {
                tablet.makeOn(true);

                /*
                 *   Jestliže je zapnutí tabletu implicitní akcí vyvolanou např.
                 *   namířením na něco, tak vynecháme popis předchozího
                 *   programu.
                 */
                if(!gAction.isImplicit) tablet.screenDesc();
            }
            else
            {
                "Stisknutím tlačítka tabletu ses vrátil na úvodní
                    obrazovku. ";

                tablet.runApp('home');
            }
        }
    }
    dobjFor(TurnOn) asDobjFor(Push)

    gcName = 'zapínacího tlačítka, zapínacímu tlačítku, zapínací tlačítko,
        zapínacím tlačítku, zapínacím tlačítkem'
    /*
     *   Úmyslně dáme "tlačítka" jen jako adjective, aby nebylo silnější, než
     *   plurál tlačítek na palubní desce.
     */
    gcVocab = 'zapínacího zapínacímu zapínacím tlačítka tlačítku tlačítkem'
;

+ Component 'zadní kryt/(tabletu)' 'zadní kryt tabletu' *2
    "Zadní kryt je téměř hladký, s velmi jemnou texturou, u krajů zaoblený. V
        horní třetině plochy je objektiv malé univerzální kamerky. "

    gcName = 'zadního krytu tabletu, zadnímu krytu tabletu, zadní kryt tabletu,
        zadním krytu tabletu, zadním krytem tabletu'
    gcVocab = 'zadního zadnímu zadním krytu/krytem'
;

+ Component 'kamera/(tabletu)' 'kamera tabletu' *3
    "Malá zabudovaná univerzální kamera s vysokým rozlišením, jaké běžně bývají
        v přenosných počítačích a telefonech. Vysoce odolný miniaturní objektiv
        nijak nevyčuhuje a zcela splývá s rovnou plochou zadního krytu
        tabletu. "

    gcName = 'kamery tabletu, kameře tabletu, kameru tabletu, kameře tabletu,
        kamerou tabletu'
    gcVocab = 'kamery/kameře/kameru/kamerou'
;

/* ------------------------------------------------------------------------ */
/*
 *   ÚVODNÍ NABÍDKA
 */
+ appHome: TabletApp 'úvodní nabídka' 'úvodní nabídka' *3
    "<<tablet.notifyActivity>>Na displeji tabletu je zobrazena úvodní nabídka s
        ikonami zpravodajství, znalostní databáze a hry, které můžeš stisknout.
        Kromě toho umí tablet i rozpoznat některé elektronické přístroje, když
        ho na ně namíříš. "

    /* Všechny aplikace jsou označené názvem, podle toho je pak hledáme. */
    appName = 'home'

    gcName = 'úvodní nabídky, úvodní nabídce, úvodní nabídku, úvodní nabídce,
        úvodní nabídkou'
    gcVocab = 'nabídky/nabídce/nabídku/nabídkou'
;

/* Společné definice pro každou ikonku. */
class TabletIcon: Button, Component
    gender = 3

    /* Lidé jsou zvyklí na "otevírání" aplikací. */
    dobjFor(Open) asDobjFor(Push)
    hideFromAll(action)
    {
        return action.ofKind(ReadAction) || inherited(action);
    }

    vocabWords = 'ikona*ikony*aplikace'
    gcVocab = 'ikony ikoně ikonu ikonou ikony/ikoně/ikonu/ikonou*ikon*ikonám*
        ikonách*ikonami*aplikací*aplikacím*aplikacích*aplikacemi'
;

++ TabletIcon 'hry/hra' 'ikona hry'
    "Hru symbolizuje obrázek maličkého děla střílejícího přes kopec.
        <<tablet.notifyActivity>> "

    dobjFor(Push)
    {
        action()
        {
            tablet.runApp('canons');
        }
    }

    gcName = 'ikony hry, ikoně hry, ikonu hry, ikoně hry, ikonou hry'
    gcVocab = 'hry/hře/hru/hrou'
;

++ TabletIcon 'znalostní databáze' 'ikona databáze'
    "Ikona databáze je nakreslená jako šuplíky s uloženými lejstry… Hmmm, něco
        takového jsi viděl před mnoha lety ve škole při virtuální návštěvě
        pozemského muzea. No, ještě že se pro psaní textu nepoužívá obrázek
        hliněné tabulky s klínovým písmem.<<tablet.notifyActivity>> "

    dobjFor(Push)
    {
        action()
        {
            tablet.runApp('database');
        }
    }

    gcName = 'ikony databáze, ikoně databáze, ikonu databáze, ikoně databáze,
        ikonou databáze'
    gcVocab = 'databázi/databází'
;

++ TabletIcon 'zpravodajství' 'ikona zpravodajství'
    "Zpravodajství symbolizuje staromódní obrázek tištěných novin.
        <<tablet.notifyActivity>> "

    dobjFor(Push)
    {
        action()
        {
            tablet.runApp('news');
        }
    }
    dobjFor(Read) asDobjFor(Push)

    gcName = 'ikony zpravodajství, ikoně zpravodajství, ikonu zpravodajství,
        ikoně zpravodajství, ikonou zpravodajství'
    gcVocab = 'zpravodajstvím'
;

/* ------------------------------------------------------------------------ */
/*
 *   DATABÁZE
 *
 *   Jen vysvětlíme, jak funguje běžná ConsultAbout akce.
 */
+ appDatabase: TabletApp 'databáze' 'databáze' *3
    "Prohlédl sis displej tabletu. Je na něm jednoduché rozhraní pro hledání
        informací. Stačí zadat HLEDEJ <i>NĚCO</i> a systém to zkusí vyhledat.
        <<tablet.notifyActivity>> "

    appName = 'database'

    gcName = 'databáze, databázi, databázi, databázi, databází'
    gcVocab = 'databázi/databází'
;

/*
 *   Do Spacepedie budeme dávat hodně témat, tak si připravíme šablony na
 *   zadávání (odpověď máme přejmenovanou na myResponse, abychom všude nemuseli
 *   psát inherited) a třídu zajišťující při hledání příkazem "hledej něco"
 *   přepnutí na správnou aplikaci.
 */
DatabaseTopic template
   +matchScore?
   @matchObj | [matchObj] | 'matchPattern'
   "myResponse" | [eventList] ?;

/* we can also include *both* the match object/list *and* pattern */
DatabaseTopic template
   +matchScore?
   @matchObj | [matchObj]
   'matchPattern'
   "myResponse" | [eventList] ?;

class DatabaseTopic: ConsultTopic
    myResponse = ""
    topicResponse()
    {
        tablet.runApp('database', true);
        tablet.notifyActivity();
        myResponse();
    }
;

/* Nejprve dvě nejdůležitější témata, která posouvají hru, pak blbosti :-) */
+ suspiciousTopic: DatabaseTopic @suspiciousContainer
    myResponse()
    {
        "<.p>Ta bedna venku… Pořád se ti k ní vrací myšlenky. Nesedí ti to. Měl
            jsi pocit, že náklad jsi zkontroloval, ale tuhle bednu si vůbec
            nevybavuješ. No, nic, zkontroluješ si alespoň její číslo. Zadal jsi
            do tabletu bednu číslo KBR-3867.

            <.p>Displej zůstal prázdný – ano, měl jsi pravdu, tahle bedna s námi
            přece necestovala. Rozšířil jsi hledání. Po chvíli pípnutí oznámilo
            nalezení výsledku: Transportní bedna KBR-3867 vyřazena při letu
            AV53-81 jako poničená. Při vykládce na Plutu došlo k deformaci celé
            přední stěny a bedna nešla uzavřít. ";

        /* Toto téma nebylo známo od počátku, bylo označeno isKnown = nil. */
        gSetKnown(tPluto);
        achievement.awardPointsOnce();
    }
    achievement: Achievement { +1 "získání informací o podezřelém kontejneru" }
;

+ plutoTopic: DatabaseTopic @tPluto
    myResponse()
    {
        "<.p><blockquote><tt>Let AV53-81
            <br><br>transportní let Měsíc – Pluto
            <br>velitel letu: kpt. Henry Scott
            <br>palubní technik: Chriss Anderson
            <br>účel letu: běžná dodávka zásob na základu na Plutu
            </tt></blockquote>

            <.p><q>Hmm,</q> prohlížíš si údaje, <q>tohle je dobré dva roky
            zpátky. A Chriss bude asi ten chlápek, co tuhle práci dělal přede
            mnou.</q> ";

        achievement.awardPointsOnce();
    }
    achievement: Achievement { +2 "získání stopy vedoucí ke kapitánovi" }

    isActive = tPluto.isKnown
;

+ DatabaseTopic @tCmx12
    "Našel jsi nádherný obrázek CMX12. Krásný kousek dovednosti lidstva.
        Razicí stroje řady CMX jsou určené do dolů na Zemi, na terrestické
        planety nebo na větší měsíce. K provozu potřebují gravitaci alespoň
        0,73 g.

        <.p><q>Hmm,</q> pomyslel sis, <q>tady na asteroidu je gravitace žádná
        celá, skoro nic. Tady nenarazím z CMX ani na odrazku ze zadního
        blinkru.</q><.reveal cmx12-lie> "

     isActive = tCmx12.isKnown
;

+ DatabaseTopic @tRab4
    "Zadal jsi dotaz na rypadlo RA-B4. Na displeji se ukázala informace z
        centrální Spacepedie:

        <.p><blockquote><tt>RA-B4 (Riepadelo asteroidowe) je plnoprofilový
        razící zeminový štít. Je vybavený diamantovou postupovou hlavou s
        absolutní prostupností. Stabilitu zajišťují nezávislé pásové skupiny,
        napínací kola a rolny, umožňující opor po celém obvodu. Je plně
        automatizovaný a při těžbě na asteroidech nemá konkurenci.
        </tt></blockquote> "
;

+ DatabaseTopic @sq71
    "Zadal jsi do vyhledávače oblast SQ71. Otevřela se ti místní databáze:

        <.p><blockquote><tt>Oblast s mírně nadprůměrným výskytem těžené horniny.
        Uzavřena. Pro další informace ťukněte.</tt></blockquote>

        <.p>Mlčky zíráš na obrazovku, kde po ťuknutí do požadovaného tlačítka
        poblikává strohé oznámení, že pro přístup do této databáze nemáš
        práva. "
;

+ DatabaseTopic [brokenComponents, fixedComponents]
    "Zadal jsi dotaz na součástku VIZ-54. Na displeji se ukázala informace z
        centrální Spacepedie:

        <.p><blockquote><tt>VIZ-54 je výkonný integrovaný zesilovač používaný
        ve vysílacích modulech v transportních lodích pouze u typu APL 21 a 22.
        Tuto verzi VIZ-54 nelze nahradit jiným typem.</tt></blockquote>

        <<if gRevealed('database-access')>><.p>Po chvíli tablet pípl a objevila
        se další informace:

        <.p><blockquote><tt>Sklad základny: součástek VIZ-54… počet:
        <<if !newComponent.moved>>1, umístění: balík číslo C3-A24
        <<else>>0<<end>></tt></blockquote><.reveal low-stock><<end>> "

    isActive = brokenComponents.discovered
;

+ DatabaseTopic [c3a24Package, c3a24Unpackage]
    "<q>Ty jo, fakt je v balíku C3-A24 součástka VIZ-54. A balík je na regálu ve
        zdejším skladu. To jsou mi věci!</q> pokýval jsi hlavou. "
;

+ DatabaseTopic @ship
    "Zadal jsi dotaz na naši loď. Na displeji se ukázala informace z centrální
        Spacepedie:

        <.p><blockquote><tt>APL je série transportních lodí určených na převoz
        velkých nákladů. Ve snaze získat co nejpříznivější poměr cena /
        přepravovaná hmotnost, byla část pro posádku upravená do minimálních
        rozměrů. V typech APL-21 a APL-22 byly experimentálně testovány moduly
        součástek, které nejsou kompatibilní s žádnými sériově vyráběnými moduly
        jiných lodí. APL-29 sérii ukončila.</tt></blockquote>

        <.p><blockquote><tt>Některé transportéry jsou dodnes používané na cesty
        v pásmu asteroidů, na běžných trasách je nahrazují modernější BRN s
        pohodlnějším prostorem pro posádku.</tt></blockquote> "
;

+ DatabaseTopic @asteroid
    "Zadal jsi do tabletu heslo asteroid a okamžitě jsi poznal, že se
        připojil do místní databáze:

        <.p><blockquote><tt>Asteroid 287 965 MQ2045 obíhá v pásu planetek mezi
        Marsem a Jupiterem. Díky své velikosti má téměř kulový tvar a měřitelné
        hodnoty gravitace. Opakovaný průzkum potvrdil zajímavou koncentraci
        surovin a před několika lety tu byla zřízena základna a zahájena těžba.
        Vytěžený konglomerát odváží dvakrát až čtyřikrát do roka transportní
        lodě k dalšímu zpracování v blízkosti Země. Transportní lodě zároveň
        dováží veškeré zásoby pro zdejší život. Velitelem základny od samého
        začátku je Ivan Petronov.</tt></blockquote> "
;

+ DatabaseTopic @tAnomaly
    "Zadal jsi do tabletu <q>geologická anomálie</q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Obecně anomálie je odchylka od normálu. V geologii
        se nejčastěji jedná o odchylky od běžných hodnot gravitačního pole. Může
        to být způsobeno tím, že v tom místě je vyšší koncentrace těžkých kovů.
        Obecně jsou anomálie indikátorem výskytu jiných nerostů, než jsou v
        okolí.</tt></blockquote> "
;

+ DatabaseTopic @tBella
    "Zadal jsi do tabletu <q>Ivan Bella</q>. Po chvíli se na displeji objevilo:

        <.p><blockquote><tt>Ivan Bella, narozen 25. května 1964 v Breznu na
        Slovensku. Byl pilotem vojenské stíhačky, armádním generálem a prvním
        kosmonautem nezávislého Slovenska. Do vesmíru letěl v roce 1999 jako
        385. člověk ze Země lodí SOJUZ TM-29 pod velením Rusa V. Afanasieva. Na
        oběžné dráze realizoval čtyři vědecké projekty z oblsati medicíny, jeden
        z biologie a jeden fyzikální projekt. Celkové náklady na jeho cestu byly
        15,6 miliónu Sk. Na oběžné dráze strávil 8 dní.</tt></blockquote> "
;

+ DatabaseTopic @tresor
    "Zadal jsi do tabletu <q>trezor</q>. Po chvíli se na displeji objevilo:

        <.p><blockquote><tt>Trezor, též sejf slouží k bezpečnému, případně i
        dlouhodobému uložení cenných předmětů (peníze, šperky, ale i důležité
        listiny). Zloději specializovaní na sejfy byli označování jako kasaři a 
        trezor jako káča.</tt></blockquote> "
;

+ DatabaseTopic @tMicroscope
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Elektronový mikroskop je zařízení, kde fotony jsou
        nahrazeny elektrony a skleněné čočky elektromagnetickými cívkami. Ty
        vytváří vhodně tvarované magnetické pole. Mezní rozlišovací schopnost
        mikroskopů je úměrná vlnové délce použitého záření a tu mají elektrony
        podstatně kratší než viditelné světlo. Proto elektronový mikroskop může
        tak dosáhnout až milionkrát vyššího zvětšení než světelný mikroskop.
        </tt></blockquote>
        
        <.p><blockquote><tt>Typ TEM sestrojil v roce 1931 Ernst Ruska. Elektrony
        v něm procházejí vzorkem, proto potřebují vysokou energii a velmi tenké
        vzorky.</tt></blockquote>
        <.p><blockquote><tt>Typ SEM zkonstruoval V. K. Zworykin v roce 1942.
        Obraz v něm vzniká pomocí odražených elektronů při postupném rastrování
        vzorku.</tt></blockquote> "
;

+ DatabaseTopic @tProgramming
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Robot Karel vznikl v roce 1981. Vytvořil ho Richard
        E. Pattis na Stanfordově univerzitě jako pomůcku pro výuku studentů.
        Pojmenoval ho na poctu českému spisovateli a dramatiku Karlu Čapkovi.
        </tt></blockquote>
        
        <.p><blockquote><tt>Robot Karel rozumí několika základním příkazům z
        nichž lze snadno sestavit mnohem složitější strukturovaný program. K
        výuce základů programování se jeho nejrůznější varianty používají
        dodnes.</tt></blockquote> "
;

+ DatabaseTopic @garden
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Hydroponie je pěstování rostlin v živném roztoku.
        Oproti pěstování v půdě takový způsob umožňuje mnohem přesněji dávkovat
        živiny a pěstovat více rostlin v menším prostoru. V hydroponii pěstované
        rostliny také rychleji rostou a bývají méně náchylné k chorobám. Růst
        lze řídit, nebo stimulovat i vhodně voleným spektrem světla, kterým jsou
        rostliny osvětleny.</tt></blockquote> "
;

+ DatabaseTopic @tMoon
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Měsíc je přirozená družice Země. Střední vzdálenost
        Měsíce od Země je 384 403 km. Měsíční rovníkový průměr činí 3 476
        km.</tt></blockquote>

        <.p><blockquote><tt>Měsíc fascinoval člověka svou přítomností na nebi od
        nepaměti. Návštěvníci se nicméně shodují na tom, že to je těleso veskrze
        nudné, samé šutry, prach, regolit a sem tam kráter.</tt></blockquote> "
;

+ DatabaseTopic @unidip
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>UniDiP, univerzální diagnostický přístroj vyrobený
        ve společnosti TechnoUniverse Corporation. Byl vyvinut týmy zkušených
        techniků nejprve pro interní použití, později byl ve zjednodušené a
        levnější podobě nabízen na trhu, kde se stal rychle oblíbeným, dokud
        nebyl nahrazen pokročilejším VUniDiPem.</tt></blockquote>

        <.p><blockquote><tt>UniDiP slouží především k diagnostice a testování
        standardizovaných elektronických modulů z nejrůznějších přístrojů
        používaných při dobývání vesmíru.</tt></blockquote> "
;

+ DatabaseTopic @tVunidip
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>VUniDiP je velmi univerzální diagnostický přístroj.
        Prakticky nahradil do té doby oblíbený, komerčně úspešný a rozšířený
        UniDiP.</tt></blockquote>

        <.p><blockquote><tt>Tento článek je pahýl.</tt></blockquote> "
;

+ DatabaseTopic @firstModule
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Přijímací modul radaru je velmi kompaktní zařízení
        zajišťující zesílení a předzpracování slabých odražených signálů
        přijímaných anténami lodi. Modul je umístěn na dobře přístupném místě z
        venku na trupu lodi, aby byl snadno vyměnitelný v případě
        poruchy.</tt></blockquote> "
;

+ DatabaseTopic @secondModule
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Vysílací modul radaru generuje silné pulzy, které
        lze vysílat radarovou anténou. Je velmi kompaktní a je umístěn na
        přístupném místě z venku na trupu lodi. Modul pracuje s velkým
        elektrickým výkonem, neobejde se bez chlazení. Při montáži je třeba dbát
        opatrnosti, aby vlivem chyby nedošlo k ovlivnění dalších systémů
        lodi.</tt></blockquote> "
;

+ DatabaseTopic @tLagrangianPoint
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Lagrangeův bod, taktéž librační centrum, je místo v
        soustavě dvou vesmírných těles, rotujících kolem společného těžiště v
        němž se vyrovnávají jejich gravitační a odstředivé síly. Leží vždy v
        rovině rotace těles a celkem jich je pět.</tt></blockquote>

        <.p><blockquote><tt>Čistě prakticky to znamená, že k udržení tělesa v
        tomto bodě je potřeba nejméně paliva.</tt></blockquote> "
;

+ DatabaseTopic @tSpacepedia
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Spacepedie je nejlepší zdroj informací každého
        vesmířana. Nepropadejte panice!</tt></blockquote> "
;

+ DatabaseTopic @tTychosComplex
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Stabilní a postupně se rozrůstající výzkumná
        základna poblíž kráteru Tycho zabývající se převážně materiálovou
        fyzikou, kosmickou energetikou a průmyslovým zpracováním měsíčního
        regolitu. Některé části komplexu patří velkým korporacím, které tak
        mohou získat zázemí pro svůj výzkum bez nutnosti samostatné a proto
        drahé kolonizace nového neobsazeného území.</tt></blockquote> "
;

+ DatabaseTopic @tTechnouniverse
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Společnost působící na trhu kosmické přepravy, těžby
        a průzkumu nových teritorií již celá desetiletí. Aktuálně ovládá i velmi
        významné procento trhu s dopravními prostředky, provozuje několik
        vesmírných průmyslových komplexů a vlastní velká území na různých
        planetách i mnoho celých menších těles.</tt></blockquote>

        <.p><blockquote><tt>Podrobnější informace na webu společnosti
        http://technouniverse.cz/</tt></blockquote> "
;

+ DatabaseTopic @card
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Stále používaný identifikátor oprávněné osoby
        umožňující přístup do chráněných prostor, nebo využívání různých
        technických prostředků k nimž je třeba přístup
        omezit.</tt></blockquote> "
;

+ DatabaseTopic @tApl21
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Nákladní loď pro přepravu objemných nákladů na
        meziplanetární vzdálenosti velmi modulární konstrukce, vybavena pro
        pobyt jen dvoučlenné posádky.</tt></blockquote>

        <.p><blockquote><tt>Oproti starším typům nebyl tento model příliš
        obchodně úspěšný, ale přesto jich stále zůstává několik v aktivní
        službě, především pro svou snadnou ovladatelnost a nízké nároky na
        posádku.</tt></blockquote> "
;

+ DatabaseTopic @tApl22
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Nákladní loď pro přepravu těžkých nákladů na
        meziplanetární vzdálenosti velmi modulární konstrukce, vybavena pro
        pobyt jen dvoučlenné posádky.</tt></blockquote>

        <.p><blockquote><tt>Koncepčně vychází z APL 21, nákladní prostor i
        pohony lodi modifikovány pro náklady menších objemů s vyšší
        hmotností.</tt></blockquote>

        <.p><blockquote><tt>Oblíbený model nákladní lodi především pro svou
        snadnou ovladatelnost a nízké nároky na posádku.</tt></blockquote> "
;

+ DatabaseTopic @tube
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Vesmírný sendvič™, oblíbená snídaně kosmických
        šampionů. Dodá sílu i energii do dalších dobrodružství.
        </tt></blockquote>

        <.p><blockquote><tt>Disclaimer: Společnost TechnoUniverse Corp. neručí
        za zdravotní potíže způsobené překračováním doporučenné denní
        dávky.</tt></blockquote> "
;

+ DatabaseTopic @tRecycledFood
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevilo:

        <.p><blockquote><tt>Potravinový recyklát, směs vitamínů a minerálů
        extrahovaná z odpadu, která prochází složitým procesem separace,
        čištění, obohacení o chybějící ingredience až se ze směsi stane
        plnohodnotná potravina schopná udržet astronauty naživu i bez přístupu
        k čerstvým potravinám. Někteří tomu posměšně přezdívají tatrgel podle
        jedné povídky z dávných dob. Chutná prý přesně tak.</tt></blockquote> "
;

+ DatabaseTopic @captain
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevil typický souhrn veřejných informací o zaměstnanci společnosti
        TechnoUniverse Corporation.

        <.p>Henry Scott, aktuálně kapitán nákladní lodi APL-22, u společnosti
        pracuje už několik let, bývalý vojenský pilot… přelétl jsi obvyklá
        firemní hodnocení, nic neobvyklého, snad jen to, že letí tuto trasu
        vícekrát. "
;

+ DatabaseTopic @tChriss
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevil typický souhrn veřejných informací o zaměstnanci společnosti
        TechnoUniverse Corporation.

        <.p>Rychle jsi pročetl obvyklý firemní balast, moc informací tu není, o
        nehodě jen datum úmrtí, přesto je zřejmé, že pracoval u společnosti už
        nějaký čas s docela dobrými výsledky, zkušenosti mu nechyběly. Nebýt
        nehody, asi by brzy dosáhl nějakého povýšení.<.reveal chriss-death> "
;

+ DatabaseTopic @technician
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevil typický souhrn veřejných informací o zaměstnanci společnosti
        TechnoUniverse Corporation.

        <.p>Přeskákal jsi očima po tabulkách, grafech a hodnoceních… nemá špatné
        hodnocení, u společnosti pracuje už nějaký čas, jen ta poznámka o
        nadměrné spotřebě povzbuzujících nápojů je hodně zvýrazněná. "
;

+ DatabaseTopic @commander
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevil typický souhrn veřejných informací o zaměstnanci společnosti
        TechnoUniverse Corporation.

        <.p>Pročetl jsi personalistický balast a firemní hodnocení, prý je
        přísný, hodnocení nijak neobvyklé, pracuje u společnosti skoro od
        začátku kariéry, vystřídal spoustu působišť… zdá se ale, že na současném
        místě zůstává o maličko déle, než je obvyklé. "
;

+ DatabaseTopic @me
    "Zadal jsi do tabletu <q><<gTopicText>></q>. Po chvíli se na displeji
        objevil typický souhrn veřejných informací o zaměstnanci společnosti
        TechnoUniverse Corporation.

        <.p>Bleh, manažerský kecy, tabulky, hodnotící kritéria… brr, až se ti z
        toho naježilo strniště. Naštěstí nic neobvyklého na tebe společnost
        neeviduje. By ještě scházelo, jak si jednou personalisti člověka
        všimnou, nepustí, dokud ho nesežvýkají. "
;

/*
 *   Podobně jako u rozhovorů máme téma, které se ukáže, když žádné jiné
 *   odpovídající není nalezeno.
 */
+ DefaultConsultTopic
    "Zadal jsi do tabletu <q><<gTopic.getTopicText>></q>. Po chvíli se na
        displeji objevilo:

        <<if gRevealed('database-access')>><.p><blockquote><tt>žádné záznamy ve
        zdejší databázi</tt></blockquote>

        <.p>a za moment další pípnutí:<<end>>

        <.p><blockquote><tt>globální encyklopedie… téma nenalezeno<br>
        Spacepedie… přístup odepřen</tt></blockquote><<tablet.notifyActivity>> "
;

/* ------------------------------------------------------------------------ */
/*
 *   ZPRAVODAJSTVÍ
 */
+ appNews: TabletApp 'zpravodajství' 'zpravodajství'
    "Prohlédl sis displej tabletu. Zobrazují se na něm nejčersvější články,
        které loď přijala z retranslační stanice, zatímco jsi spal: <q>Konání
        Letních olympijských her ohroženo?</q>, <q>Objev tisíciletí nebo
        dokonalý podvrh?</q>, <q>Kam pokračují Simpsonovi?</q>
        <<tablet.notifyActivity>> "

    appName = 'news'
;

class NewsItem: Readable, Component
    vocabWords = 'článek (o) -*články*článkům*článcích*zprávy*zpráv*zprávám*
        zprávách*zprávami'
    gender = 2

    dobjFor(Push) asDobjFor(Read)

    gcVocab = 'článku článkem -'
;

++ NewsItem
    name = 'článek o konání Letních olympijských her'
    vocabWords = 'konání letních olympijských her ohroženo -'
    desc = "<.p>Přečetl sis článek <i>Konání Letních olympijských her
        ohroženo?</i> Píše se v něm o prvních olympijských hrách konaných mimo
        matičku Zemi, právo uspořádat vyhrála Venuše. Tisíce dělníků ze Země
        mířících do venušského hlavního města Magellam za prací při přípravě
        olympiády nedostávají adekvátní mzdu a je jim odepírán odpočinek. Tvrdí
        to ve své zprávě organizace hájící lidská práva pozemšťanů Terra Human
        Rights Watch (THRW).<<tablet.notifyActivity>> "

    gcVocab = 'konáním letní letním letními olympijské olympijským olympijskými
        hry hrám hrách hrami ohrožení ohrožením -'
;

++ NewsItem
    name = 'článek o objevu tisíciletí'
    vocabWords = 'objev tisíciletí (nebo) dokonalý podvrh -'
    desc = "<.p>Přečetl sis článek <i>Objev tisíciletí nebo dokonalý
        podvrh?</i> Je to rozhovor s vědci z Drážďanského muzea
        specializovanými na mayskou civilizaci. Uvádí, že na světě je jen
        několik mincí z doby Mayů. Tato civilizace zlaté mince zvané itzamy
        nerazila, ale jednotlivě tepala, takže jsou každá trochu jiná. A teď se
        do muzea dostalo několik mincí naprosto a do detailu identických. Žádný
        test zatím nepotvrdil, že by nepocházely z doby Mayů a že by se jednalo
        o padělek.<<tablet.notifyActivity>> "

    gcVocab = 'objevu objevem dokonalého dokonalému dokonalém dokonalým podvrhu
        podvrhem -'
;

++ NewsItem
    name = 'článek o Simpsonech'
    vocabWords = '(kam) pokračují simpsonovi -'
    desc = "<.p>Přečetl sis článek <i>Kam pokračují Simpsonovi?</i> V něm tě
        nejvíc zaujal poslední odstavec: <q>Před natáčením další série sílí
        hlasy, aby postava Barta Simpsona konečně ze seriálu zmizela. Nerudný
        dědek vystupující v posledních deseti sériích už zcela vyčerpal veškeré
        nápady a vtip a ve všech průzkumech je hodnocen nejhůře.</q>
        <<tablet.notifyActivity>> "

    gcVocab = 'pokračování pokračováním simpsnovi simpsonů simpsnů simpsonům
        simpsnům simpsony simpsny simpsonech simpsnech simpsni -'
;

/* ------------------------------------------------------------------------ */
/*
 *   BĚHÁNÍ NA PÁSU
 */
+ appTreadmill: TabletApp 'sport' 'sport' *2
    "Na obrazovce tabletu se zobrazily statistiky tvých sportovních výkonů.
        Průměrné výsledky nevypadají moc povzbudivě, ke splnění minimálních
        požadavků stačí, ale autobus bys možná nedoběhl. Dnes jsi uběhl
        <<treadmillDial.counterDesc()>> km. "

    appName = 'treadmill'

    gcName = 'sportu, sportu, sport, sportu, sportem'
    gcVocab = 'sportu/sportem'
;

/* ------------------------------------------------------------------------ */
/*
 *   ROBOT
 */
+ appRobot: TabletApp
    'ovládací panel ovládání robota -' 'ovládací panel robota' *2
    "Na obrazovce tabletu je zobrazen ovládací panel robota. Je tu prostor pro
        zadání programu, instrukce k ovládání robota, které si můžeš přečíst, a
        tři tlačítka. První je označené <q>start</q> a tím se program spouští,
        druhé je označené <q>chyba</q> a tím se maže poslední zadaný příkaz a to
        třetí má označení <q>smazat</q> a maže se jím celý program.
        <<showProgram>> "

    appName = 'robot'

    /* Pozice a natočení robota, program. */
    x = 3
    y = 3
    dir = 1
    program = []
    showProgram()
    {
        tablet.notifyActivity();
        if(program.length() == 0)
        {
            "<.p>Momentálně není zadaný žádný program. Napiš ZADEJ <i>PŘÍKAZ</i>
                a tím se příkaz zařadí do programu. Seznam dostupných příkazů je
                k dispozici v instrukcích na ovládacím panelu robota, které si
                můžeš přečíst. ";
        }
        else
        {
            "<.p>V prostoru programu je momentálně zadán tento program:

                <pre style=\"color: #060\">";

            /* Uroveň zanoření příkazu OPAKUJ. */
            local level = 1;
            for(local i = 1, local len = program.length(); i <= len; i++)
            {
                if(program[i].startsWith('KONEC'))
                    level--;

                for(local j = 1; j < level; j++)
                    "    ";

                say(program[i]);
                if(i < len) "<br />";

                if(program[i].startsWith('OPAKUJ'))
                    level++;
            }

            "</pre><.p0>";
        }
    }

    /* Smazání celého programu. */
    clearProgram()
    {
        program = [];
        enterLevel = 0;

        "Smazal jsi celý program, nyní můžeš zadat nový program od začátku. ";
    }

    /* Vymazání posledního zadaného příkazu. */
    clearCommand()
    {
        if(program[program.length()].startsWith('KONEC'))
            enterLevel++;

        program = program.removeElementAt(-1);
        if(!program.length())
        {
            "Smazal jsi jediný příkaz, program je teď prázdný. ";
        }
        else
        {
            "Smazal jsi poslední instrukci, program teď obsahuje příkazy: ";
            showProgram();
        }
    }

    inst = nil
    sayNext()
    {
        if(inst == 1)
            "Nejprve ";
        else
            "<<one of>>Potom<<or>>Následně<<or>>Po dokončení předchozí
            instrukce<<or>>Na to<<or>>Pak<<or>>Nyní<<or>>A teď<<half shuffled
            >> ";
    }
    runProgram()
    {
        gReveal('hint-program-run');
        "Robot zablikal zelenou a pomalu se dal do plnění příkazů. ";

        local ret, level = 0, watchdog = 0, stack_inst = [], stack_count = [];

        /* Cyklus pro každý zadaný příkaz. */
        for(inst = 1, local len = program.length(); inst <= len; inst++)
        {
            ret = nil;
            if(program[inst] == 'KROK')
            {
                ret = krok();
            }
            else if(program[inst] == 'VLEVO VBOK')
            {
                vlevovbok();
            }
            else if(program[inst].startsWith('OPAKUJ'))
            {
                /* Regulárním výrazem extrahujeme počet a převedeme na číslo. */
                if(rexMatch(R'^OPAKUJ ([0-9]+) KR[AÁ]T$', program[inst]) != nil)
                {
                    level++;
                    stack_inst += inst;
                    stack_count += parseInt(rexGroup(1)[3]);
                    gReveal('hint-repeat');
                }
            }
            else if(program[inst] == 'KONEC OPAKOVÁNÍ')
            {
                if(--stack_count[level] > 0)
                {
                    inst = stack_inst[level];
                }
                else
                {
                    stack_inst = stack_inst.removeElementAt(-1);
                    stack_count = stack_count.removeElementAt(-1);
                    level--;
                }
            }

            /* Program udelal nejakou chybu a je potreba ho resetovat. */
            if(ret) break;

            /* Pokud by někdo program zacyklil, tak ho zastavíme. */
            if(++watchdog == 100)
            {
                if(x != 3 || y != 3)
                    "Robot se najednou zarazil, rozsvítil napůl žhnoucím světlem
                        majáček na znamení téměř vybitých akumulátorů, a
                        plouživým krokem se vrátil na nabíjecí dok. ";
                else if(dir != 1)
                    "Robot se najednou zarazil, rozsvítil napůl žhnoucím světlem
                        majáček na znamení téměř vybitých akumulátorů,
                        otočil se ve svém nabíjecím doku zpátky na sever a
                        zapnul dobíjení. ";
                else
                    "Robot se najednou zarazil, rozsvítil napůl žhnoucím světlem
                        majáček na znamení téměř vybitých akumulátorů a začal
                        se ve své výchozí pozici na doku zase nabíjet. ";

                x = 3;
                y = 3;
                dir = 1;

                return;
            }
        }

        if(ret == 2) {}
        else if(ret == 1)
        {
            if(x != 3 || y != 3)
                "Rozblikal výstražný majáček a pak se vrátil na svou výchozí
                    pozici v nabíjecím doku. ";
            else if(dir != 1)
                "Rozblikal výstražný majáček a otočil se ve svém nabíjecím doku
                    zpátky na sever. ";
            else
                "Rozblikal na okamžik výstražný majáček a zůstal ve své výchozí
                    pozici na doku. ";
        }
        else
        {
            if(x != 3 || y != 3)
                "Program doběhl úspěšně do konce, ale protože robot neměl další
                    příkazy, tak se zase vrátil na svůj nabíjecí dok. ";
            else if(dir != 1)
                "Program doběhl úspěšně do konce, ale protože robot neměl další
                    příkazy, otočil se ve svém nabíjecím doku zpátky na
                    sever. ";
            else
                "Program doběhl úspěšně do konce, ale protože robot neměl další
                    příkazy, zůstal ve své výchozí pozici na doku. ";
        }
        x = 3;
        y = 3;
        dir = 1;
    }

    krok()
    {
        local nx, ny;

        /* Určíme souřadnice, kam by se měl robot krokem dostat. */
        switch(dir)
        {
        case 0:
            nx = x + 1;
            ny = y;
            break;
        case 1:
            nx = x;
            ny = y - 1;
            break;
        case 2:
            nx = x - 1;
            ny = y;
            break;
        case 3:
            nx = x;
            ny = y + 1;
            break;
        }

        sayNext();

        /* Otestujeme, zda narazil do dvří a vyřešil šifru. */
        if(nx == 0 && ny == 3)
        {
            "se rozjel proti zeleným dveřím a nevybíravě vrazil do jejich
                kliky. Sice se po nárazu s rozblikaným majáčkem vrátil na
                své místo v nabíjecím doku, ale dveře zůstaly pootevřené. ";

            largeDoor.makeLocked(nil);
            gReveal('robot-solved');
            achievement.awardPointsOnce();
            return 2;
        }

        /* Otestujeme, zda tam může jet. */
        else if(nx > 3 || nx < 1 || ny > 3 || ny < 1)
        {
            "se pokusil popojet, ale narazil do stěny místnosti. ";

            return 1;
        }
        else if(nx == 2 && ny >= 2)
        {
            "se pokusil popojet, ale narazil do bedny
                <<if ny == 2>>uprostřed<<else>>na okraji<<end>> místnosti. ";

            return 1;
        }

        /* Můžeme popojet. */
        x = nx;
        y = ny;

        "<<one of>>popojel<<or>>vyrazil<<or>>se rozjel<<shuffled>>
            <<one of>>vpřed<<or>>rovně<<or>>kupředu<<shuffled>> a <<one of
            >>teď je<<or>>ocitl se<<or>>zastavil<<or>>zabrzdil<<shuffled>> ";

        if(y == 1)
        {
            if(x == 1) "v severozápadním rohu";
            if(x == 2) "na prostředním poli u severní stěny";
            if(x == 3) "v severovýchodním rohu";
        }
        if(y == 2)
        {
            if(x == 1) "na prostředním poli u západní stěny";
            if(x == 3) "na prostředním poli u východní stěny";
        }
        if(y == 3)
        {
            if(x == 1) "v jihozápadním rohu";
            if(x == 3) "v jihovýchodním rohu";
        }

        " místnosti. ";

        return 0;
    }
    achievement: Achievement { +5 "získání přístupu k vozítkům" }

    vlevovbok()
    {
        dir++;
        if(dir == 4) dir = 0;

        "Robot se otočil doleva, takže <<one of>>teď směřuje<<or>>aktuálně je
            orientovaný<<or>>míří<<or>>směřuje<<shuffled>> na ";

        switch(dir)
        {
        case 0:
            "východ. ";
            break;
        case 1:
            "sever. ";
            break;
        case 2:
            "západ. ";
            break;
        case 3:
            "jih. ";
            break;
        }
    }

    outOfMemory = 0
    enterLevel = 0
    dobjFor(EnterOn)
    {
        verify()
        {
            /*
             *   Když máme zapnutý tablet s robotí aplikací, tak je vhodnějším
             *   cílem pro zadávání, než počítač.
             */
            if(!gRevealed('robot-solved'))
                logicalRank(110, 'robot');
        }
        action()
        {
            tablet.notifyActivity();

            local lit = gLiteral.toUpper();

            if(lit == 'START')
                replaceAction(Push, start);
            if(lit == 'CHYBA')
                replaceAction(Push, error);
            if(lit == 'SMAZAT' || lit == 'VYMAZAT' || lit == 'SMAŽ')
                replaceAction(Push, erase);

            "Do programu jsi zadal příkaz <q><<gLiteral>></q>, načež se
                zobrazilo: ";

            /*
             *   Úloha se na první pohled zdá jednoduchá, ale na její vyřešení
             *   je potřeba zadat více elementárních příkazů, než se vejde do
             *   paměti robota. Klíčem k vyřešení je objevit opakující se vzor
             *   v posloupnosti příkazů a osvojit si strukturovaný příkaz
             *   OPAKUJ, díky kterému se program zkrátí.
             */
            if(program.length() == 8)
            {
                "<.p><tt><span style=\"color: #800\">ERR: OUT OF MEMORY.</span>
                    </tt> ";
                if(++outOfMemory == 1)
                    "<.p><q>Zatracený výrobce,</q> pomyslel sis, <q>dá do něj
                        malou paměť a její upgrade určitě prodává za cenu celého
                        nového robota.</q> ";
            }

            /* Zápis R'text' je zkratkou pro statický regulární výraz. */
            else if(lit.match(R'^KROK'))
            {
                program += 'KROK';
                "<.p><tt><span style=\"color: #060\">OK.</span></tt> ";
                showProgram();
            }
            else if(lit.match(R'^VLEVO[ -]?VBOK'))
            {
                program += 'VLEVO VBOK';
                "<.p><tt><span style=\"color: #060\">OK.</span></tt> ";
                showProgram();
            }
            else if(lit.match(R'^OPAKUJ [0-9]+ KR[AÁ]T$'))
            {
                program += lit;
                enterLevel++;
                "<.p><tt><span style=\"color: #060\">OK.</span></tt> ";
                showProgram();
            }
            else if(lit.match(R'^KONEC OPAKOV[ÁA]N[ÍI]'))
            {
                if(enterLevel > 0)
                {
                    program += 'KONEC OPAKOVÁNÍ';
                    "<.p><tt><span style=\"color: #060\">OK.</span></tt> ";
                    showProgram();
                    enterLevel--;
                }
                else
                {
                    "<.p><tt><span style=\"color: #800\">ERR: NOT IN A
                        LOOP.</span></tt> ";
                }
            }
            else if(lit.match(R'^OPAKUJ'))
            {
                "<.p><tt><span style=\"color: #800\">ERR: MUST SPECIFY NUMBER OF
                    REPEATS IN LOOP.</span></tt> ";
            }
            else
            {
                "<.p><tt><span style=\"color: #800\">ERR: COMMAND NOT
                    FOUND.</span></tt> ";
            }
        }
    }
    dobjFor(TypeLiteralOn) asDobjFor(EnterOn)

    dobjFor(TypeOn)
    {
        verify() { }
        action() { askForLiteral(TypeLiteralOn); }
    }

    gcName = 'ovládacího panelu robota, ovládacímu panelu robota, ovládací panel
        robota, ovládacím panelu robota, ovládacím panelem robota'
    gcVocab = 'ovládacího ovládacímu ovládacím panelu panelem ovládáním -'
;

++ Readable, Component
    'instrukce příkazy (na) (robotovi) (robotovy) instrukce/příkazy/(robota)'
    'instrukce' *3
    "Přečetl sis návod k robotovi. Přeskočil jsi spoustu zbytečností i licenční
        ujednání a zaměřil ses na základní příkazy, kterým robot rozumí -- KROK,
        VLEVO VBOK, OPAKUJ <i>ČÍSLO</i> KRÁT, KONEC OPAKOVÁNÍ. Do programu se
        dají přidávat pomocí příkazu ZADAT <i>PŘÍKAZ</i> DO PROGRAMU.

        <.p><q>Takže když chci, aby robot udělal jeden krok, napíšu mu ZADEJ
        KROK DO PROGRAMU a potom stisknu tlačítko start,</q> pomyslel sis.
        <q>Nejlépe bude robota řádně vyzkoušet.</q><<tablet.notifyActivity>> "

    isPlural = true

    /*
     *   Aby hráč mohl zadat "robotovy instrukce". Přivlastňovací přídavná jména
     *   máme u robota zadaná. Když ale zadá "instrukce na robotovi", tak nelze
     *   využít lokační kvalifikace, protože instrukce ve skutečnosti nejsou na
     *   robotovi, ale v tabletu. Proto toto pořešíme zde ve slovníku.
     */
    owner = robot
    gcName = 'instrukcí, instrukcím, instrukce, instrukcích, instrukcemi'
    gcVocab = 'instrukcí instrukcím instrukcích instrukcemi příkazů příkazům
        příkazech instrukcí/instrukcím/instrukcích/instrukcemi/příkazů/příkazům/
        příkazech'
;

++ Readable, Component
    'prostor (pro) zadání zadaný program/programu' 'program' *3
    "<<appRobot.showProgram>> "

    dobjFor(EnterOn) remapTo(EnterOn, appRobot, IndirectObject)
    dobjFor(TypeLiteralOn) remapTo(TypeLiteralOn, appRobot, IndirectObject)

    dobjFor(TypeOn)
    {
        verify() { }
        action() { askForLiteral(TypeLiteralOn); }
    }

    isPlural = true
    gcName = 'programu, programu, program, programu, programem'
    gcVocab = 'prostoru prostorem zadaného zadanému zadaném zadaným programem'
;

++ start: Button, Component
    'tlačítko start*tlačítka' 'tlačítko start' *4
    "Tímto tlačítkem se zadaný program spouští. "

    dobjFor(Push)
    {
        verify()
        {
            if(!appRobot.program.length())
                illogicalNow('Není co spustit, nejprve zadej program. ');
        }
        action() { appRobot.runProgram(); }
    }
    dobjFor(Enter) asDobjFor(Push)

    gcName = 'tlačítka start, tlačítku start, tlačítko start, tlačítku start,
        tlačítkem start'
    gcVocab = 'tlačítka tlačítku tlačítkem -*tlačítek*tlačítkům*tlačítkách*
        tlačítky'
;

++ error: Button, Component
    'tlačítko chyba/chybu*tlačítka' 'tlačítko chyba' *4
    "Tímto tlačítkem se maže poslední zadaný příkaz. "

    dobjFor(Push)
    {
        verify()
        {
            if(!appRobot.program.length())
                illogicalNow('Není co smazat, program je úplně prázdný. ');
        }
        action() { appRobot.clearCommand(); }
    }
    dobjFor(Enter) asDobjFor(Push)

    gcName = 'tlačítka chyba, tlačítku chyba, tlačítko chyba, tlačítku chyba,
        tlačítkem chyba'
    gcVocab = 'tlačítka tlačítku tlačítkem chyby/chybě/chybou*tlačítek*
        tlačítkům*tlačítkách*tlačítky'
;

++ erase: Button, Component
    'tlačítko smazat/vymazat*tlačítka' 'tlačítko vymazat' *4
    "Tímto tlačítkem se maže poslední zadaný příkaz. "

    dobjFor(Push)
    {
        verify()
        {
            if(!appRobot.program.length())
                illogicalNow('Není co smazat, program je úplně prázdný. ');
        }
        action() { appRobot.clearProgram(); }
    }
    dobjFor(Enter) asDobjFor(Push)

    gcName = 'tlačítka vymazat, tlačítku vymazat, tlačítko vymazat, tlačítku
        vymazat, tlačítkem vymazat'
    gcVocab = 'tlačítka tlačítku tlačítkem -*tlačítek*tlačítkům*tlačítkách*
        tlačítky'
;
/* ------------------------------------------------------------------------ */
/*
 *   DĚLOSTŘELECKÝ SOUBOJ
 */
+ appCanons: TabletApp
    'hra -' 'hra' *3
    "Na obrazovce tabletu je zobrazena hra Dělostřelecký souboj. Na nádherně
        malované krajině v hyperrealistickém 2D zobrazení stojí dvě děla.
        <<showState>> "

    appName = 'canons'

    /*
     *   Canonicalize a proposed setting value.  For numbers, strip off any
     *   leading zeros, since these don't change the meaning of the value.
     */
    canonicalizeInput(val)
    {
        local num;

        /* try parsing it as a digit string or a spelled-out number */
        if ((num = parseInt(val)) != nil)
        {
            /*
             *   we parsed it successfully - return the string
             *   representation of the numeric value
             */
            return toString(num);
        }

        /* it didn't parse as a number, so just return it as-is */
        return val;
    }

    state = 1
    first = true

    /*
     *   BigNumber je třída pro výpočty s plovoucí řádovou čárkou a téměř
     *   neomezenou resp. definovanou přesností. Na rozdíl od typu float známého
     *   z běžných programovacích jazyků pracuje BigNumber přímo v dekadické
     *   reprezentaci.
     */
    distance = static new BigNumber(10000, 10)
    angle = static new BigNumber(0, 10)
    power = static new BigNumber(0, 10)

    showState()
    {
        tablet.notifyActivity();

        switch(state)
        {
        case 0:
            "Hra ukazuje, že děla jsou od sebe <<distance / 1000>> km daleko a
                čeká, až zadáš, o kolik metrů popojedeš. <<if first>>(např.
                ZADEJ 100) ";
            break;
        case 1:
            "Hra <<if first>>ukazuje, že děla jsou od sebe <<distance / 1000>>
                km daleko a <<end>>čeká, až zadáš úhel hlavně, pod kterým chceš
                vystřelit. <<if first>>(např. ZADEJ 45) ";
            break;
        case 2:
            "Teď bys měl zadat množství střelného prachu v gramech. <<if first>>
                (např. ZADEJ 5000) ";
            break;
        }
    }

    dobjFor(EnterOn)
    {
        verify() {}
        action()
        {
            tablet.notifyActivity();

            local val = toInteger(canonicalizeInput(gLiteral));

            switch(state)
            {
            case 0:
                if(val < -250 || val > 250)
                {
                    "Tak daleko popojet nemůžeš, maximální vzdálenost je 250 m.
                        Hra čeká, až opravíš zadání. ";
                    break;
                }

                if(val != 0)
                {
                    distance -= val;
                    "Ok, vzdálenost od soupeře je nyní <<distance / 1000>>
                        km. ";
                }

                "Hra nyní čeká, až zadáš úhel hlavně, pod kterým chceš
                    vystřelit. <<if first>>(např. ZADEJ 45) ";

                state = 1;
                break;
            case 1:
                if(val < 0 || val > 180)
                {
                    "Úhel musíš zadat ve stupních od 0 do 180. Hra čeká, až se
                        opravíš. ";
                    break;
                }

                angle = val;

                "Teď bys měl zadat množství střelného prachu v gramech.
                    <<if first>>(např. ZADEJ 5000) ";

                state = 2;
                break;
            case 2:
                if(val < 0 || val > 10000)
                {
                    "Hmotnost střelného prachu musí být od 0 do 10000 g. Hra
                        čeká, až se opravíš. ";
                    break;
                }

                power = val;
                power = power / 1000.0;

                local a = angle / 180.0 * 6.28;
                local length = (150.0 * power * power * a.sine())
                    .roundToDecimal(0);

                "Z hlavně děla vylétl veliký projektil a zanechávaje za sebou
                    výraznou kouřovou stopu opsal ladnou balistickou křivku. ";

                if(distance - length < 50 && distance - length > -50)
                {
                    "Projektil dopadl na soupeře s drtivou silou a masivní
                    výbuch ho vymazal z existence.
                    
                    <.p>No teda, takhle už sis dlouho nezahrál, to napětí… do
                    poslední chvilky jsi trnul hrůzou a nakonec ten blažený
                    pocit vítězství.
                    
                    <.p>A grafika, ta byla úplně perfektní, jako bys celou dobu
                    koukal na skutečnou krajinu. Ačkoli… odněkud se ti vkrádá
                    nepatrná pochybnost, jako bys už hru na podobné téma kdysi
                    někde viděl. Snad v muzeu? Ne, to není možné. Tak dobré hry
                    tehdy určitě nemohly existovat. ";
                    
                    state = 1;
                    break;
                }
                else
                    "Projektil dopadl <<length>> metrů daleko. ";

                first = nil;

                /*
                 *   Tohle je taková nějaká umělá inteligence, ktará hraje proti
                 *   hráči. Záměrně není moc chytrá a trochu záhadná, dokonce
                 *   tak záhadná, že ani autor už si nepamatuje, jak pracuje.
                 *   Ale určitý efekt vytváří a to je hlavní.
                 */
                local aiLength = new BigNumber(0, 10);
                local aiPower = new BigNumber(power, 10);
                local aiRnd = new BigNumber(rand(1000), 10);
                aiRnd = aiRnd / 2000 + 0.25;
                aiLength = (length + (distance - length) * aiRnd) / 150;

                local iter = 0;
                local target = distance - length;
                do
                {
                    aiPower = aiPower + aiRnd * sgn(target);
                    aiPower = aiPower.roundToDecimal(1);
                    if(aiPower > 10) aiPower = 10;
                    if(aiPower < 2) aiPower = 2;
                    iter++;
                    target = aiLength / aiPower / aiPower;
                }
                while((target < 0 || target > 1) && iter < 20);

                aiLength = aiLength / aiPower / aiPower;

                local aiAngle = new BigNumber(aiLength.arcsine(), 10);
                aiAngle = (aiAngle / 6.28 * 180.0).roundToDecimal(0);

                aiLength = (150.0 * aiPower * aiPower * (aiAngle / 180.0
                    * 6.28).sine()).roundToDecimal(0);

                if(distance - aiLength < 50 && distance - aiLength > -50)
                {
                    "<.p>Soupeř naplnil projektil <<aiPower * 1000>> gramů
                        střelného prachu a vystřelil pod úhlem <<aiAngle>>
                        stupňů. Nepřátelský projektil dopadl a žhavá exploze
                        proměnila tvé dělo i okolní skálu ve žhavou taveninu.
                        Game Over. ";
                    state = 1;
                    break;
                }
                else
                    "<.p>Soupeř naplnil projektil <<aiPower * 1000>> gramů
                        střelného prachu a vystřelil pod úhlem <<aiAngle>>
                        stupňů. Střela dopadla <<aiLength>> metrů daleko. ";

                "<.p>Nyní jsi opět na tahu ty. Hra čeká, až zadáš úhel
                    hlavně. ";

                /* A zase je na tahu hráč. */
                state = 1;
                break;
            }
        }
    }
    dobjFor(TypeLiteralOn) asDobjFor(EnterOn)

    dobjFor(TypeOn)
    {
        verify() { }
        action() { askForLiteral(TypeLiteralOn); }
    }

    gcName = 'hry, hře, hru, hře, hrou'
    gcVocab = 'hry hře hru hrou -'
;

++ Component
    'zvlněná křivka jemná mlha jasné zeleň krajina/kopec/kopce/kopců/mraků/
        mraky/déšť/slunce/porostu/porost' 'krajina' *3
    "Dramaticky zvlněná křivka kopců na nichž jsou děla umístěna nebezpečně
        blízko na dostřel se chvílemi skrývá v jemné mlze mraků skrápěna deštěm,
        jindy zas jasné slunce rozzáří zeleň porostu. "

    changeGender = 'kop:2, mrak:2, déš:2, slun:4, porost:2'
    gcName = 'krajiny, krajině, krajinu, krajině, krajinou'
    gcVocab = 'krajiny/krajině/krajinu/krajinou/kopci/kopcem/kopcům/kopcích/
        kopci/mrakům/mracích/deště/dešti/deštěm/slunci/sluncem/porost/porostem'
;

++ Component
    'dělo/děla/hlaveň/hlavně' 'děla' *4
    "Z ostrých hran masivních maskovaných pancířů samochodného děla vykukuje
        hlaveň namířená k nepříteli. "

    isPlural = true

    gcName = 'děl, dělům, děla, dělech, děly'
    gcVocab = 'děla/dělu/dělech/děly/děl/dělům/hlavni/hlavní/hlavním/hlavních/
        hlavněmi'
;
