#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   Třída pro místnosti na základně jen defnuje, že je v nich atmosferický
 *   tlak. Ten slouží ke kontrolám u skafandru.
 */
class BaseRoom: Room
{
    pressure = 1
}

/* ------------------------------------------------------------------------ */
/*
 *   Kdekoliv ve vnitřních prostorech základny může hráč zadat příkaz prozkoumej
 *   základnu. Odpovíme v podstatě informací o vnitřním uspořádání, aby se hráč
 *   zorientoval.
 */
base: MultiLoc, CustomFixture 'základna' 'základna' *3
    "Převážně pod povrchem ukrytá základna poskytuje nezbytné zázemí
        astronautům, kteří tu pracují a dohlížejí na bezproblémový provoz
        techniky. Z celé základny je na povrchu zejména vstupní kupole s
        přechodovou komorou. Pod povrchem se ukrývají kajuty s velínem, jídelna,
        sklad s dílnou a toaleta, které propojuje centrální chodba. Někde by
        měla být i hydroponická zahrádka a další technické prostory související
        s prací na základně. "

    /* U MultiLoc se určuje, v jakých místnostech bude k dispozici. */
    initialLocationClass = BaseRoom
    isListedInContents = nil

    /* Loď by neměla být zahrnuta, když řekneš např. "seber vše" */
    hideFromAll(action) { return true; }

    isKnown = true

    gcName = 'základny, základně, základnu, základně, základnou'
    gcVocab = 'základny/základně/základnu/základnou'
;

/* ------------------------------------------------------------------------ */
/*
 *   Tyto třídy pocházejí z demohry Attachable a implementují logiku přechodové
 *   komory. Dveře v komoře se ovládají pákou. Kontrolujeme, aby si hráč
 *   nevypustil vzduch na dýchání.
 */
class IndirectDoor: IndirectLockable, Door
    dobjFor(Open) { verify { illogical (cannotOpenMsg); } }
    dobjFor(Close) { verify { illogical(cannotCloseMsg); } }
    dobjFor(Lock) { verify { illogical(cannotLockMsg); } }
    dobjFor(LockWith) { verify { illogical(cannotLockMsg); } }
    dobjFor(Unlock) { verify { illogical(cannotUnlockMsg); } }
    dobjFor(UnlockWith) { verify { illogical(cannotUnlockMsg); } }
    cannotOpenMsg = '{Kdoco dobj} se ovládají pákou. '
    cannotCloseMsg = (cannotOpenMsg)
    cannotLockMsg = '{Kdoco dobj} nemají žádný zámek. '
    cannotUnlockMsg = (cannotLockMsg)
;

/*
 *   DoorLever is custom class we use for the code common to the two levers
 *   that control the doors in the airlock.
 */
class DoorLever: SpringLever, CustomFixture

    /* A custom property - the other lever (used for various purposes). */
    otherLever = nil

    /* The door controlled by this lever. */
    myDoor = nil

    dobjFor(Pull)
    {
        verify()
        {
            if(otherLever.myDoor.isOpen)
            {
                local otherDoor = otherLever.myDoor;
                gMessageParams(otherDoor);
                
                illogicalNow('{Kdoco dobj} je dočasně uzamknutá, dokud jsou
                    {kdoco otherDoor} otevřené, aby se předešlo otevření obou
                    dveří přechodové komory najednou. ');
            }
        }

        check()
        {
            /*
             *   Don't let the player open a door if there's a vacuum on the
             *   other side and the player is not wearing both suit and
             *   helmet, since that would be fatal.
             */
            if(spaceSuit.wornBy != gActor && myDoor.destination.pressure == 0)
                failCheck('Otevření ' + myDoor.nameKohoCeho + ' by teď mělo pro
                    tebe fatální důsledky. ');
        }
        action()
        {
            myDoor.makeOpen(!myDoor.isOpen);
            
            "Zatáhl jsi za {kohoco dobj}";

            if(myDoor.isOpen && airlock.pressure > myDoor.destination.pressure)
            {
                ". Atmosféra byla s hlasitým syčením odčerpána a <<myDoor.name>>
                    se otevřely. ";
                
                airlock.pressure = myDoor.destination.pressure;
            }
            else if(myDoor.isOpen && airlock.pressure < myDoor.destination.pressure)
            {
                ". Přechodová komora se naplnila vzduchem a <<myDoor.name>> se
                    otevřely. ";
                    
                airlock.pressure = myDoor.destination.pressure;
            }
            else if(myDoor.isOpen)
            {
                " a <<myDoor.name>> se otevřely. ";
            }
            else if(!myDoor.isOpen)
            {
                " a <<myDoor.name>> se zavřely. ";
            }
        }
    }

    /* Make MOVE LEVER pull it or push is as appropriate. */
    dobjFor(Move) asDobjFor(Pull)
    dobjFor(Push) asDobjFor(Pull)
    dobjFor(Switch) asDobjFor(Pull)
    dobjFor(Flip) asDobjFor(Pull)
    dobjFor(Turn) asDobjFor(Pull)
    dobjFor(TurnOn) asDobjFor(Pull)

    cannotTakeMsg = '{Kdoco dobj} nejde vzít, protože je připevněná na stěně.
        Mohl bys s ní ale pohnout. '
;


/* ------------------------------------------------------------------------ */
/*
 *   Velín i sklad jsou otevřené a od chdby je neoddělují žádné dveře. Místnosti
 *   jsou tedy propojené ve dvou kolmých liniích smyslovými konektory, aby hráč
 *   viděl, co se v další místnosti odehrává.
 */
DistanceConnector [supplyRoom, corridorWest]
    transSensingThru(sense)
    {
        return me.isIn(supplyRoom) || me.isIn(corridorWest) ? distant : opaque;
    }
;

DistanceConnector [controlRoom, corridorWest, corridorEast]
    transSensingThru(sense)
    {
        return me.isIn(controlRoom) || me.isIn(corridorWest)
            || me.isIn(corridorEast) ? distant : opaque;
    }
;
