#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   PILOTNÍ KABINA
 */
cForeWall: DefaultWall
    'přední zeď/přepážka/stěna*zdi*přepážky*stěny' 'přední stěna' *3
    "Celý předek zabírá palubní deska s několika obrazovkami, mnoha tlačítky a
        kniplem. "

    /*
     *   Kabina je spojena se společenskou místností smyslovým konektorem.
     *   Většinu objektů máme označenu, aby na dálku nebyla vidět.
     */
    sightSize = small

    gcName = 'přední stěny, přední stěně, přední stěnu, přední stěně,
        přední stěnou'
    gcVocab = 'zdi/zdí/přepážky/přepážce/přepážku/přepážkou/stěny/stěně/stěnu/
        stěnou*zdí*zdím*přepážek*přepážkám*přepážkách*přepážkami*stěn*stěnám*
        stěnách*stěnami'
;

cAftWall: DefaultWall 'zadní zeď/přepážka/stěna*zdi*přepážky*stěny'
    'zadní stěna' *3
    "Přepážka přepažuje celý trup lodi. Je konstruována jako tlakuvzdorná a v
        nouzi je schopná udržet atmosféru, pokud se zavřou tlakové dveře. "

    sightSize = small

    gcName = 'zadní stěny, zadní stěně, zadní stěnu, zadní stěně, zadní stěnou'
    gcVocab = 'zdi/zdí/přepážky/přepážce/přepážku/přepážkou/stěny/stěně/stěnu/
        stěnou*zdí*zdím*přepážek*přepážkám*přepážkách*přepážkami*stěn*stěnám*
        stěnách*stěnami'
;

cockpit: DisambigDeferrer, ShipboardRoom
    'Pilotní kabina' 'do pilotní kabiny' 'z pilotní kabiny'
    "Kabina je rozdělena na dvě rovnocenné části, levou a pravou, pro dva
        piloty. Celý předek kabiny zabírá palubní deska, nad níž je malý průzor.
        Před palubní deskou jsou dvě anatomická pilotní křesla. Na levoboku jsou
        dveře ven bez přechodové komory. "

    /*
     *   Na palubě lodi (ShipboardRoom) platí speciální směry fore, aft, port,
     *   starboard.
     */
    aft = cDoor

    /*
     *   Kabina je vidět na dálku ze společenské místnosti. DisambigDeferrer
     *   umožňuje ji potlačit ve prospěch uvedeného objektu, když by oba objekty
     *   odpovídaly příkazu hráče, aby se hra nemusela ptát "a co tím máš na
     *   mysli".
     */
    disambigDeferTo = [ lrCockpit ]

    /* Některé části místnosti vyměníme za specializované. */
    roomParts = static inherited
        - foreWall + cForeWall
        - aftWall + cAftWall

    /* Atmosféru vypínáme při odchodu z místnosti. */
    suspendAtmosphere = nil
    beforeTravel(trav, conn) { suspendAtmosphere = true;  }
    afterTravel(trav, conn) { suspendAtmosphere = nil; }

    /*
     *   Střídáme podle situace více různých atmosfér, některé jsou společné pro
     *   více místností, proto jsou deklarovány v regionShip.t.
     */
    atmosphereList = suspendAtmosphere ? nil : (gRevealed('ship-landed') ? nil
        : (gRevealed('ship-malfunction') ? malfunctionAtmosphere
        : (captain.curState == captainDriving ? captainDrivingAtmosphere
        : normalAtmosphere)))

    captainDrivingAtmosphere: ShuffledEventList
    {
        [
            'Kapitán se protáhl a zas se vrátil k svému čtení. ',
            'Kapitán posunul stránku na tabletu. ',
            'Kapitán stiskl tlačítko bdělosti na palubní desce. ',
            'Kapitán zkontroloval stav lodi a mrknul na radar. ',
            'Kapitán se natáhl k nejvzdálenější obrazovce, cosi přepnul a zase
                se usadil v křesle. '
        ]
        eventPercent = 50
        eventReduceAfter = 10
        eventReduceTo = 20
    }
    
    gender = 3
    vocabWords = 'pilotní kabina'
    gcName = 'pilotní kabiny, pilotní kabině, pilotní kabinu, pilotní kabině,
        pilotní kabinou'
    gcVocab = 'kabiny/kabině/kabinu/kabinou'
;

/* ------------------------------------------------------------------------ */

+ cLivingRoom: Enterable -> cDoorOutside
    'společenská místnost' 'společenská místnost' *3
    "Společenská místnost je na zádi. "

    /* Tento objekt není zahrnut, když hráč napíše "vše". */
    hideFromAll(action) { return true; }

    sightSize = small

    dobjFor(Open) remapTo(Open, cDoor)
    dobjFor(Close) remapTo(Close, cDoor)

    gcName = 'společenské místnosti, společenské místnosti, společenskou
        místnost, společenské místnosti, společensko místností'
    gcVocab = 'společenské společenskou místnosti/místností'
;

+ cDoor: Door 'vnitřní tlakové dveře dveře/těsnění*dveře'
    'vnitřní tlakové dveře' *3
    "Jsou to mohutné dveře schopné udržet atmosféru v oddělené části lodi. Po
        obvodu je měkké těsnění, které občas prověřuješ. Z obou stran jsou
        výrazné tlakoměry ukazující tlak na druhé straně dveří. V případě
        potřeby tlakový uzávěr může lehce zaklapnout a udrží i velmi prudkou
        změnu tlaku. "

    /* Dveře normálně bývají na začátku zavřené. */
    initiallyOpen = true

    sightSize = small

    /* Dveře jsou pomnožné, tj. mají vždy tvar množného čísla. */
    isPlural = true

    gcName = 'vnitřních tlakových dveří, vnitřním tlakovým dveřím, vnitřní
        tlakové dveře, vnitřních tlakových dveřích, vnitřními tlakovými dveřmi'
    gcVocab = 'vnitřních vnitřním tlakových vnitřními tlakovým tlakové tlakovými
        dveří/dveřím/dveřích/dveřmi'
;

++ cDoorBarometer: Fixture 'tlakoměr/tlakoměry/tlak' 'tlakoměr' *2
    "Jednoduchý ukazatel, v němž se mechanicky posouvá barevný pruh v okénku.
        Teď je zelený, žlutá  by znamenala malý pokles tlaku (pod 80 kPa, asi
        jako v 2000 m nad mořem), oranžová nebezpečí, červená značí smrtelné
        riziko. "

    sightSize = small

    gcName = 'tlakoměru, tlakoměru, tlakoměr, tlakoměru, tlakoměrem'
    gcVocab = 'tlakoměru/tlakoměrem/tlakoměrů/tlakoměrům/tlakoměry/tlakoměrech/
        tlaků/tlakům/tlaky/tlacích'
;

/* ------------------------------------------------------------------------ */
/*
 *   SwitchableTravelMessage je jednoduchá třída, kterou jsme si do hry dodali,
 *   abychom mohli vypnout travelDesc při prvním vstupu do základny, kdy nás
 *   vede kapitán a máme pro to popisek v rámci captainGuideToDome. Devře jsou
 *   IndirectLockable, protože se odemknou programově při přistání.
 */
+ shipDoor: SwitchableTravelWithMessage, IndirectLockable, Door
    'venkovní vnější tlakové dveře dveře/(ven)/těsnění*dveře'
    'venkovní tlakové dveře' *3
    "Přetlakové dveře z kabiny se používají k výstupu z lodi do prostor s
        atmosférou, třeba do přechodového tunelu. Dokud je v lodi větší tlak než
        venku, tak se samy zablokují a není možné je otevřít. "

    sightSize = small

    travelDesc = "Vklouzl jsi do tunelu, s občasným odrazem a korigujíc pomalý
        pád rukama jsi proplul až do základny. "

    cannotOpenLockedMsg = 'Nemůžeš otevřít vnější dveře do vzduchoprázdna. Jsou
        automaticky blokované. '
    cannotUnlockMsg = 'Dveře nemají žádný zámek. '
    cannotLockMsg = 'Dveře nemají žádný zámek. '
    alreadyLockedMsg = 'Dveře nemají žádný zámek. '
    alreadyUnlockedMsg = 'Dveře nemají žádný zámek. '

    isPlural = true
    gcName = 'venkovních tlakových dveří, venkovním tlakovým dveřím, venkovní
        tlakové dveře, venkovních tlakových dveřích, venkovními tlakovými dveřmi'
    gcVocab = 'venkovních venkovním venkovními vnějších vnějším vnějšími 
        tlakových tlakovým tlakové tlakovými dveří/dveřím/dveřích/dveřmi'
;

/* EntryPortal je jako Enterable, jen se jím nevchází, ale prochází se jím. */
+ PresentLater, EntryPortal -> shipDoor 'přechodový tunel' 'přechodový tunel' *2
    "Přechodový tunel je utěsněný, atmosférou nafouknutý tunel umožňující
        astronautům relativně bezpečný přechod z jednoho kosmického zařízení do
        jiného, aniž by museli používat skafandry. Teď zrovna je jedním koncem
        připevněný ke kabině lodi a druhým koncem ke kupoli základny. "

    /*
     *   Společné označení několika PresetLater objektů, které zviditelňujeme
     *   najednou.
     */
    plKey = 'ship-landed'

    sightSize = small

    gcName = 'přechodového tunelu, přechodovému tunelu, přechodový tunel,
        přechodovém tunelu, přechodovým tunelem'
    gcVocab = 'přechodového přechodovému přechodovém přechodovým tunelu/
        tunelem'
;


/* ------------------------------------------------------------------------ */

+ deck: Fixture 'palubní pilotní přístrojová kovové deska/díly' 'palubní deska' *3
    "Celý předek zabírá palubní deska s několika obrazovkami, mnoha tlačítky a
        kniplem. Na první pohled vypadá kabina velmi používaně. Místa, kde se
        dotýkají ruce pilotů, jsou oleštěná a leckde je prošoupaná barva až na
        podkladový plast a kovové díly. Některé ovládací prvky naopak vypadají
        docela nově. Asi byly nedávno vyměněny. Vespod uprostřed je štěrbina,
        kde <<if manual.isIn(manualHolder)>>je<<else>>obvykle bývá<<end>>
        uložený manuál k lodi. "

    sightSize = small

    iobjFor(AimAt)
    {
        action()
        {
            "Tablet zobrazuje souhrn informací viditelných i na obrazovkách v
                kokpitu.
                <<if gRevealed('ship-landed') && !gRevealed('ship-fixed')>>
                Momentálně ukazuje závadu radarové navigace a vybízí k
                servisnímu zásahu.
                <<else if gRevealed('ship-malfunction')
                && !gRevealed('ship-fixed')>>
                Momentálně je největším problémem porouchaná navigace
                autopilota, takže hrozí, že loď narazí do nějakého šutru, které
                se okolo asteroidu pohybují.
                <<else>>
                Na obrazovce vidíš výpis jednotlivých částí lodi a u každé z
                nich je zelené <q>ok</q>. ";
        }
    }

    gcName = 'palubní desky, palubní desce, palubní desku, palubní desce,
        palubní deskou'
    gcVocab = 'přístrojové přístrojovou kovových kovovým kovovými desky/desce/
        desku/deskou/dílů/dílům/dílech'
    changeGender = 'díl:5'
;

++ Component 'ovládací knipl*prvky' 'knipl' *2
    "Knipl připomíná velký a složitý joystick k dávným počítačovým hrám. Slouží
        k manuálnímu ovládání lodi.<.reveal joystick> "

    sightSize = small

    dobjFor(Push)
    {
        verify() {}
        action()
        {
            if(!gRevealed('ship-landed'))
                "Nemůžeš ovládat loď ručně, dokud je zapnutý autopilot. ";
            else
                "Nic se nestalo. Loď je teď zadokovaná, motory vypnuté a bez
                    kapitánova rozkazu stejně nemůžeš odletět. ";
        }
    }
    dobjFor(Pull) asDobjFor(Push)
    dobjFor(Move) asDobjFor(Push)

    gcName = 'kniplu, kniplu, knipl, kniplu, kniplem'
    gcVocab = 'ovládacích ovládacím ovládacími kniplu/kniplem*prvků*prvkům*
        prvcích'
;

/*
 *   Obrazovky jsou tři, tak si připravíme společné části do třídy. Ve slovníku
 *   máme společná slova a ty se zkombinují se slovníkem konkrétních obrazovek,
 *   které dodají přídavná jména. Jinak deklarace třídy nijak neovlivňuje
 *   kontejnerovou hierarchii objektů definovanou znaménky plus.
 */
class ShipScreen: Fixture 'obrazovka obrazovka*obrazovky'
    gender = 3
    sightSize = small

    iobjFor(AimAt) remapTo(AimAt, DirectObject, deck)

    gcVocab = 'obrazovky obrazovce obrazovku obrazovkou
        obrazovky/obrazovce/obrazovku/obrazovkou*obrazovek*obrazovkám*obrazovky*
        obrazovkách*obrazovkami'
;

++ flightScreen: ShipScreen
    'letového dráhu letu/plánu' 'obrazovka letového plánu'
    "<<if gRevealed('ship-landed')>>Obvykle ukazuje křivky předpokládané a
        skutečné dráhy letu, ale teď, po úspěšném přístání, na ní jen pomalu
        rotuje trojrozměrný obrázek lodi a na dolním řádku svítí nápis
        <q>Ukotveno</q>.
        <<else if gRevealed('ship-malfunction')>>Obrazovka ukazuje plánovanou
        dráhu letu, předpokládanou dráhu letu, místa korekce rychlosti, současný
        směr letu a spoustu dalších údajů. Symbol lodi je zobrazen mimo
        požadovanou dráhu letu a pomaličku se odchyluje, takže sebou buď
        plácneme někam mimo, nebo loď asteroid zcela mine a skončí v prázdnotě,
        nebo na jiném šutru.
        <.p>Na dolním řádku svítí nápis <q><font color=red>Hrozí
        srážka – korekce letu není efektivní</font><.reveal hint-collision></q>
        <<else>>Ukazuje plánovanou dráhu letu, předpokládanou dráhu letu, místa
        korekce rychlosti, současný směr letu a spoustu dalších údajů. Teď se
        požadovaná dráha letu téměř přesně kryje s předpokládanou dráhou a směr
        je tečnou této křivky v místě, kde je zobrazený symbol lodi. Jednoduše
        řečeno, loď letí správně. "

    gcName = 'obrazovky letového plánu, obrazovce letového plánu, obrazovku
        letového plánu, obrazovce letového plánu, obrazovkou letového plánu'
    gcVocab = 'letovému letovém letovým dráhy dráze dráhou -'
;

++ radarScreen: ShipScreen 'radar' 'obrazovka radaru'
    "<<if gRevealed('ship-malfunction') && !gRevealed('ship-fixed')>>
        Okolo malého obrázku lodi by měl zobrazovat objekty v jejím okolí, ale
        teď okolo nezobrazuje vůbec nic, jako by byl vesmír na den letu okolo
        zcela prázdný, a to určitě není.
        <<if !gRevealed('ship-landed')>>
        <.p><q>Jestli nevyřeším potíže s navigací brzy, tak to bude průšvih,</q>
        pomyslel sis. <q>Okolo asteroidu je spousta jiných šutrů, když se
        odchýlíme od pečlivě naplánované trasy, tak bychom mohli mnohem dřív
        trefit něco jiného.</q><<end>>
        <<else>>Okolo malého obrázku lodi zobrazuje všechny větší předměty a
        případné hrozící kolize. Detekuje možné kolize daleko dopředu, umožňuje
        plánovat přesnou trasu s minimální spotřebou paliva i usnadňuje ruční
        ovládání lodi v blízkosti doku. "

    /* Pokud by tohle hráč zadal, musíme mu trochu poradit. */
    dobjFor(Repair)
    {
        action()
        {
            "<<if gRevealed('ship-malfunction') && !gRevealed('ship-fixed')>>
                Usilovně se snažíš vzpomenout, jak se to dělá. Asi jsi měl v
                pilotní škole dávat více pozor. Možná se někde povaluje příručka
                k lodi, tam by mohly být servisní postupy popsané.
                <<else>>Navigace není rozbitá. ";
        }
    }

    gcName = 'obrazovky radaru, obrazovce radaru, obrazovku radaru, obrazovce
        radaru, obrazovkou radaru'
    gcVocab = 'radarová radarové radarovou radaru/radarem/navigace/navigaci/
        navigací'
;

++ stateScreen: ShipScreen 'stav' 'obrazovka stavu'
    "<<if gRevealed('ship-landed') && !gRevealed('ship-fixed')>>Na obrazovce
        stavu lodi vidíš výpis jednotlivých částí lodi, u radarové navigace je
        červený nápis nefunkční, na dolním řádku pod výpisem svítí nápis
        <q>Servisní zásah nezbytný!</q>.
        <<else if gRevealed('ship-malfunction') && !gRevealed('ship-fixed')>>
        Na obrazovce stavu lodi vidíš výpis jednotlivých částí lodi, u některých
        z nich je červený nápis <q>nefunkční</q>. Nefunkční jsou osvětlení,
        topení a radarová navigace, podle které autopilot koriguje dráhu letu a
        vyhýbá se menším kamenům, jaké v blízkosti asteroidů poletují.
        <<else>>Na obrazovce stavu lodi vidíš výpis jednotlivých částí lodi a u
        každé z nich je zelené <q>ok</q>. Jen spotřeba paliva se zdá být
        nepatrně větší, než by odpovídalo hmotnosti nákladu, ale nijak
        znepokojivě.<.reveal fuel-consumption> "

    gcName = 'obrazovky stavu, obrazovce stavu, obrazovku stavu, obrazovce
        stavu, obrazovkou stavu'
    gcVocab = 'stavu/stavem'
;

/*
 *   CollectiveGroup je objekt, který zastoupí individuální tlačítka, když je
 *   hráč prozkoumá najednou (řekne příkaz v množném čísle). Smyslem je vytvořit
 *   přirozenejší popis, než vypsání individuálních popisů každého tlačítka,
 *   které by vypadalo mechanicky a repetivně.
 */
++ buttonsGroup: CollectiveGroup, Component
    'ovládací *tlačítka*prvků*prvkům*prvcích' 'tlačítka' *4
    "Mezi mnoha různými tlačítky tu je i tlačítko autopilota a tlačítko
        bdělosti. "

    iobjFor(AimAt) remapTo(AimAt, DirectObject, deck)

    sightSize = small

    isPlural = true
    gcName = 'tlačítek, tlačítkům, tlačítka, tlačítách, tlačítky'
    gcVocab = 'tlačítek/tlačítkům/tlačítkách/tlačítky*prvků*prvkům*
        prvcích'
;

++ Button, Component
    'šedé šedivé tlačítko bdělosti*tlačítka' 'tlačítko bdělosti' *4
    "Tlačítko bdělosti je nenápadně šedé, svrchu zaoblené, aby se dalo snadno
        stisknout. "

    /* Tlačítko je součástí skupiny výše. */
    collectiveGroups = [buttonsGroup]

    /* Pokud na tlačítko zaměříme tablet, tak ho zaměřujeme na celou palubku. */
    iobjFor(AimAt) remapTo(AimAt, DirectObject, deck)

    dobjFor(Push)
    {
        action() { "Stiskl jsi tlačítko bdělosti. Do protokolu byl zaznamenán
            čas stisknutí s poznámkou, že pilot nespí. "; }
    }

    gcName = 'tlačítka bdělosti, tlačítku bdělosti, tlačítko bdělosti,
        tlačítku bdělosti, tlačítkem bdělosti'
    gcVocab = 'tlačítka tlačítku tlačítkem -'
;

++ Button, Component
    'červené tlačítko autopilota*tlačítka' 'tlačítko autopilota' *4
    "Tlačítko autopilota je červené a je na něm, kromě nápisu <q>autopilot</q>
        namalovaný obrázek robota v pilotním křesle se slunečními brýlemi.
        <<if gRevealed('ship-landed')>>Teď zrovna je autopilot vypnutý a
        tlačítko je zhasnuté.
        <<else>>Teď zrovna je autopilot zapnutý a tlačítko svítí. "

    collectiveGroups = [buttonsGroup]
    iobjFor(AimAt) remapTo(AimAt, DirectObject, deck)

    dobjFor(Push)
    {
        verify()
        {
            if(gRevealed('ship-landed'))
                illogicalNow('Přistáli jste, není důvod zapínat autopilota. ');
            else if(!gRevealed('me-driving'))
                illogicalNow('Teď řídí loď kapitán, asi byses mu neměl plést do
                    řízení. ');
            else if(!gRevealed('ship-malfunction'))
                illogicalNow('Není důvod ho vypínat. ');
        }
        action()
        {
            /*
             *   Konstrukce <<one of>> <<or>> <<stopping>> je do řetězce vložená
             *   forma EventListu, který při každém zobrazení zobrazí jednu z
             *   uvedených variant textu. Konkrétně StopEventList bere varianty
             *   postupně v pořadí a až je vyčerpá, poslední opakuje stále
             *   dokola. Kromě <<atopping>> existují i <<cycling>>, <<shuffled>>
             *   a mnoho dalších variant.
             */
            "<<one of>>Stisknul jsi tlačítko k vypnutí autopilota. Pohoda, tohle
                znáš ze školy. Teď převezmeš řízení… A kruci. A sakra. Kontrolka
                autopilota stále svítí a má pravdu, přechod na manuál se nedaří.
                A sakra. A kruci. Co teď? Pracně lovíš v paměti, co dělat, když
                ovládací prvky na přístrojové desce nereagují.
                <.reveal autopilot-frozen>
                <<or>>Znovu jsi stisknul tlačítko. Ne, opravdu se nic nezměnilo.
                Pořád ta kontrolka svítí, jako by jí za to někdo platil. Co má
                člověk dělat, když si to ze školy nepamatuje?<<stopping>> ";
        }
    }

    dobjFor(TurnOff)
    {
        verify()
        {
            if(gRevealed('ship-landed'))
                illogicalAlready('Autopilot už je vypnutý. ');
        }
        action()
        {
            /* Přesměrování akce, které je na rozdíl od remap až po verify. */
            replaceAction(Push, self);
        }
    }

    dobjFor(TurnOn)
    {
        verify()
        {
            if(!gRevealed('ship-landed'))
                illogicalAlready('Autopilot už je zapnutý. ');
        }
        action()
        {
            replaceAction(Push, self);
        }
    }

    gcName = 'tlačítka autopilota, tlačítku autopilota, tlačítko autopilota,
        tlačítku autopilota, tlačítkem autopilota'
    gcVocab = 'tlačítka tlačítku tlačítkem -'
;

++ manualHolder: RestrictedContainer, Component 'štěrbina/otvor' 'štěrbina' *3
    "Je to obdélníkový otvor k uložení manuálu. "

    sightSize = small

    /* Výchozí maxSingleBulk je 10, ale my máme manuál s velikostí 20. */
    maxSingleBulk = 10000

    validContents = [manual]
    cannotPutInMsg(obj)
    {
        gMessageParams(obj);
        return '{Kdoco obj} tam nepatří, kapitán by tě vytahal za uši. ';
    }

    /* Zmínku o manuálu už máme v samotném popisu palubní desky. */
    contentsListed = nil

    /* Změníme lister, aby mluvil o "zasunutých" objektech. */
    descContentsLister: ContentsLister, BaseThingContentsLister
    {
        showListPrefixWide(itemCount, pov, parent, selector)
        {
            gMessageParams(parent);
            "V {komčem parent} je zasunutý ";
        }

        showListSuffixWide(itemCount, pov, parent, selector) { ". "; }

        showListItem(obj, options, pov, infoTab)
        {
            obj.showListItem(options | ListKdoCo, pov, infoTab);
        }
    }

    changeGender = 'otv:2'
    gcName = 'štěrbiny, štěrbině, štěrbinu, štěrbině, štěrbinou'
    gcVocab = 'štěrbiny/štěrbině/štěrbinu/štěrbinou/otvoru/otvorem'
;

+++ manual: Readable 'manuál (k) (lodi)/manuál' 'manuál k lodi' *2
    desc()
    {
        /* V potemnělém podpalubí upozorníme v popisu na nemožnost číst. */
        if(senseAmbientMax([sight]) > 1)
            "Manuál je tlustá ohmataná kniha, obsahující vše o ovládání a
                základních opravách lodi. ";
        else
            "Po tmě jsi ohmatal obálku manuálu a zkusil zašustit stránkami, ale
                nic jsi nezjistil. ";
    }

    /* Aby ho hráč nemohl dávat do trouby. */
    bulk = 20

    /* Po tmě jde nahmatat. */
    brightness = 1

    sightSize = small

    /* Readable objekty mají zvláštní popis pro akci "přečti". */
    readDesc()
    {
        /* Hráč najde drobnou nápovědu k situaci, ve které zrovna je. */
        if(firstModule.moved && !gRevealed('ship-fixed'))
        {
            "Opět jsi trochu zalistoval manuálem. Bohužel k potížím s navigací v
                něm nic dalšího napsáno není, jen krátká poznámka pod čarou:
                <q>Pokud Vaše potíže přetrvávají, obraťte se na autorizovaný
                servis.</q>

                <.p><q>Ksakru!</q> hlasitě jsi zaklel, <q>nejbližší servis je
                odsud přes 500 milionů kilometrů!</q> ";
        }
        else if(gRevealed('ship-landed') && !gRevealed('ship-fixed'))
        {
            "Zase jsi nakoukl do manuálu, zahlédl jsi zmínku o modulech
                souvisejících s navigací, které jsou přístupné z povrchu trupu v
                přední části lodi. ";
        }
        else if(gRevealed('ship-malfunction') && !gRevealed('ship-fixed'))
        {
            if(panel.described)
            {
                "Opět jsi trochu prolistoval manuálem. Zdá se, že na nouzovém
                    ovládacím panelu lze vypnout autopilota i když jsou
                    přístroje v kokpitu zaseknuté a nereagují. Doslova píší:

                    <.p><tt>Jednotlivé aktivní okruhy řízení lze postupným
                    mačkáním tlačítek odstěhovat z levé zóny autopilota a
                    opačným směrem přesunout neaktivní okruhy. Pozor však při
                    stěhování okruhů na to, že zelený okruh se může buď posunout
                    jen o jednu pozici nebo přeskočit jeden červený okruh, ale
                    ne více.</tt> ";

                gReveal('panel-read');
            }
            else if(gRevealed('autopilot-frozen'))
            {
                "Prolistoval jsi manuál trochu pečlivěji a dozvěděl ses, že
                    nouzový ovládací panel lodních systémů se nachází v
                    servisním podpalubí.<.reveal hint-lower-deck> ";
            }
            else
            {
                "Prolistoval jsi manuál, ale je toho v něm spousta a zatím si
                    nejsi jistý, co vlastně chceš najít. Manuál radí nejprve
                    zkontrolovat obrazovky a zjistit, co loď hlásí, pak je tu i
                    zmínka o možnosti ovládat loď ručně, když se porouchá
                    autopilot, o kousek dál příjemně optimistická rada
                    <q>Nepropadejte panice!</q> a úplně na konci disclaimer
                    prohlašující, že za případné škody, nebo zranění posádky
                    nenese TechnoUniverse Corporation žádnou zodpovědnost. ";
            }
        }
        else
        {
            "Letmo jsi prolistoval tlustým manuálem. Je v něm spousta schémat
                a postupů, na několika místech jsou záložky a upřesňující
                poznámky po okrajích stránek, ale nezaujalo tě nic konkrétního.
                Pro tuto chvíli je to docela nudné čtení. ";
        }
    }

    dobjFor(ConsultAbout)
    {
        verify()
        {
            /* Hledej něco přesměrujeme na čtení manuálu, ale ne automaticky. */
            if(me.isIn(lowerDeck) && lowerDeck.scoreMarker.scoreCount == 0)
                nonObvious;
        }
        action()
        {
            replaceAction(Read, self);
        }
    }

    /* Při čtení hráč vezme manuál do ruky. */
    dobjFor(Read) { preCond = static inherited + objHeld; }
    dobjFor(Open)
    {
        /* Otevři manuál je logické, ale ne tak, jako otevření dveří. */
        verify() { logicalRank(90, 'not door'); }
        action() { replaceAction(Read, self); }
    }
    dobjFor(LookIn) asDobjFor(Read)
    dobjFor(Close)
    {
        verify() { logicalRank(90, 'not door'); }
        check()
        {
            failCheck('Manuál nemusíš zavírat, skončí zavřený tak nějak sám od
                sebe, jakmile dočteš. ');
        }
    }

    gcName = 'manuálu k lodi, manuálu k lodi, manuál k lodi, manuálu k lodi,
        manuálem k lodi'
    gcVocab = 'manuálu (k) manuálu/manuálem'
;

/* ------------------------------------------------------------------------ */

+ cPilotChair: Armchair, CustomFixture
    'pilotní křeslo pilota/křeslo*křesla' 'křeslo pilota' *4
    "Pilotní křesla vypadají docela nově, je to jeden z těch nejrozšířenějších
        typů, které se přizpůsobují anatomii sedícího pilota, po drobném zásahu
        i ve skafandru, kdyby to bylo potřeba. "

    /* Aby hráč mohl mluvit o "kapitánově křesle". */
    owner = captain

    /*
     *   Křesla jsou jedny z mála objektů, které nejsou small a jsou vidět na
     *   dálku. Vzdálený popisek je ale upraven proti plnému popisu.
     */
    distantDesc = "Odsud vidíš z pilotního křesla ve vedlejší místnosti jen
        opěradlo. "

    cannotTakeMsg = 'Křeslo je pevně přidělané k podlaze. '

    gcName = 'křesla pilota, křeslu pilota, křeslo pilota, křesle pilota,
        křeslem pilota'
    gcVocab = 'pilotního pilotnímu pilotním křesla křeslu křesle křeslem
        -*křesel*křeslům*křesla*křeslech*křesly'
;

+ cNavigatorChair: Armchair, CustomFixture
    'navigátorovo navigátorské křeslo navigátora/křeslo*křesla'
    'křeslo navigátora' *4
    "Pilotní křesla vypadají docela nově, je to jeden z těch nejrozšířenějších
        typů, které se přizpůsobují anatomii sedícího pilota, po drobném zásahu
        i ve skafandru, kdyby to bylo potřeba. "

    /* Aby hráč mohl mluvit o "svém křesle". */
    owner = me

    distantDesc = "Odsud vidíš z křesla navigátora ve vedlejší místnosti jen
        opěradlo. "

    cannotTakeMsg = 'Křeslo je pevně přidělané k podlaze. '
    
    dobjFor(GetOutOf)
    {
        check()
        {
            inherited;
            
            /* Po převzetí služby až do nehody mu neumožníme odejít. */
            if(gRevealed('me-driving') && !gRevealed('ship-malfunction'))
                failCheck('Rád bys vstal, ale smysl pro povinnost ti
                    nedovoluje zanedbávat službu jen tak z rozmaru. ');
        }
    }
    
    gcName = 'křesla navigátora, křeslu navigátora, křeslo navigátora, křesle
        navigátora, křeslem navigátora'
    gcVocab = 'navigátorova navigátorovu navigátorově navigátorovým
        navigátorským navigátorskému navigátorském křesla křeslu křeslem
        -*křesel*křeslům*křesla*křeslech*křesly'
;

+ cWindow: Fixture 'pancéřované malý průzor/okénko/okno/žaluzie/kryt' 'průzor' *2
    "Průzor je malé okénko uprostřed mezi hlavními obrazovkami. Zvenku ho lze
        zakrýt pancéřovanými žaluziemi a zevnitř utěsnit krytem, který je nyní
        v pohotovostní poloze nad průzorem. Samotný průhledný plast je schopen
        odolat silnému nárazu a zacelit průraz mikrometeoritu. Nepropouští vůbec
        UV a dokonce ani značnou část rentgenového záření. "

    sightSize = small

    /* Podívej se průzorem. */
    dobjFor(LookThrough)
    {
        action()
        {
            if(gRevealed('ship-landed'))
                "Venku je vidět převážně jen temný a chladný vesmír, žádná
                    blízká planeta v dohledu. Ani z asteroidu není vidět mnoho,
                    jen kousek vzdálenějšího zcela nezajímavého obzoru. ";
            else
                "Venku je vidět převážně jen temný a chladný vesmír, žádná
                    blízká planeta v dohledu, jen vzdálené hvězdy a na pozadí
                    prázdnoty maličký asteroid, cíl naší cesty. ";
        }
    }
    dobjFor(LookIn) asDobjFor(LookThrough)
    shouldNotBreakMsg = 'Průzor odolá mnohem drsnějšímu namáhání než je člověk
        schopen. Krom toho bys velmi vážně ohrozil svůj život i celou loď. '

    /* Objekt má synonyma v různých mluvnických rodech. */
    changeGender = 'ok:4, žal:5'
    gcName = 'průzoru, průzoru, průzor, průzoru, průzorem'
    gcVocab = 'pancéřovaných pancéřovaným pancéřovanými malého malému malém
        malým průzoru/průzorem/okénka/okénku/okénku/okénkem/okna/oknu/oknem/
        žaluzií/žaluziím/žaluziích/žaluziemi'
;

+ asteroid: Distant 'asteroid' 'asteroid' *2
    "<<if gRevealed('ship-landed')>>Poloha lodi a výhled z průzoru bohužel
        neposkytují lepší výhled než na kousek vzdálenějšího obzoru, který lze
        popsat snad jen jako prach a šutry.
        <<else>>Z dálky zatím vypadá jako zcela nepatrný kus kamení. K těžbě byl
        vybrán pro svou velikost ve srovnání s jinými asteroidy a značnou
        koncentraci kovů v hornině, aby se zřízení základny vyplatilo. "

    /* I když ho momentálně nevidíš, určitě o něm víš a můžeš o něm mluvit. */
    isKnown = true

    gcName = 'asteroidu, asteroidu, asteroid, asteroidu, asteroidem'
    gcVocab = 'asteroidu/asteroidem'
;
