#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   VELÍN
 */
controlRoom: BaseRoom 'Velín' 'do velínu'
    "Jsi ve velící místnosti, odsud se řídí těžba i všechny důležité činnosti
        související s provozem základny. Uprostřed je pohodlné pracoviště s
        velikým křeslem, velkými obrazovkami a gumovým popruhem na ruce, aby se
        operátor nevznášel od terminálu, když píše na klávesnici. Podél zadní
        stěny stojí několik racků nacpaných servery. "

    east = controlPassage

    /*
     *   Atmosféru v této lokaci vyplňuje Borisova činnost, když je přítomen. A
     *   to buď když pracuje nebo zejména když se později ukáže jeho závislost
     *   na kafi, které chceme, aby si hráč všiml a zneužil ji.
     */
    atmosphereList = technician.curState == technicianWorking
        ? (gRevealed('coffee') ? technicianCoffeeAtmosphere
        : technicianWorkingAtmosphere) : nil

    technicianWorkingAtmosphere: ShuffledEventList
    {
        [
            'Technik ospale zamžoural na monitor, opřel si bradu o dlaně a
                chvíli se pleskal do tváří, zkoušeje vypleskat jednoduchou
                melodii. ',
            'Technik sedí u počítače a mžourá na obrazovku. ',
            '<q>Dvě bedny současně tlačit nejdou… he he,</q> zabrblal Boris. ',
            'Technik se zhoupl v křesle a dlouze zívnul, pak ještě jednou. ',
            'Boris se odrazil nohou od ovládacího pultu a jsa připoután chvíli
                se v křesle otáčel kolem své osy: <q>Huíííí…</q> ',
            'Boris se protáhl, zívl, pak ještě jednou a nakonec dlouze
                zazíval. ',
            'Boris zamrkal na monitor masírujíc si kořen nosu oběma rukama:
                <q>Hele, jak tam ty bedny stěhujou.</q> '
        ]
        eventPercent = 50
        eventReduceAfter = 12
        eventReduceTo = 20
    }
    technicianCoffeeAtmosphere: ShuffledEventList
    {
        [
            'Technik ospale zamžoural na monitor, opřel si bradu o dlaně a
                chvíli se pleskal do tváří, zkoušeje vypleskat jednoduchou
                melodii. ',
            '<q>Ještě jedno kafe a bude mi kolovat i v žilách. Muhehe…,</q>
                zlověstně se zachechtal technik. ',
            'Technik se zhoupl v křesle a dlouze zívnul, pak ještě jednou. ',
            '<q>Černý muž pod bičem otrokáře žil a nás zotročuje kofein,</q>
                zanotoval Boris. ',
            'Boris se protáhl, zívl, pak ještě jednou a nakonec dlouze
                zazíval. ',
            'Technik opět nasál trochu ledové kávy z plastového pytlíku s
                ventilkem. ',
            '<q>Jsme savci,</q> zamrmlal Boris s ventilkem kávového pytlíku v
                koutku úst. ',
            'Technik se zaklonil a chvilku nasával z plastového pytlíku ledovou
                kávu, poválel doušek důkladně po jazyku, prohlédl pytlík s kávou
                proti světlu a s labužnickým zamručením polkl. ',
            '<q>Jednou jsem neměl kafe 3 dny,</q> zabručel Boris. <q>Chápeš?
                Celý dlouhý tři zatracený dny.</q> '
        ]
        eventPercent = 75
    }

    /*
     *   Tady jsme body přidělili až tehdy, co si hráč přečte dvě důležité
     *   informace (v libovolném pořadí) v počítači. Asi se to dá řešit i z obou
     *   míst, ale roomDaemon se spouští po každém tahu v místnosti, tak je
     *   vhodným společným místem, kde se dá splnění úkolu otestovat.
     */
    roomDaemon()
    {
        if(computer.read1 && computer.read2)
        {
            gReveal('computer-solved2');
            achievement.awardPointsOnce();
        }

        inherited;
    }
    achievement: Achievement { +2 "získání informací o havárii a přesunu těžby" }

    gender = 2
    vocabWords = 'velín/velínu/velínem'
    gcName = 'velínu, velínu, velín, velínu, velínem'
;

/*
 *   S chodbou je místnost spojena vstupem, který nemá dveře. ThroughPassage je
 *   něco, čím se "prochází", jako chodba nebo portál.
 */
+ controlPassage: ThroughPassage
    'vstup (na) (chodbu)' 'vstup na chodbu' *2
    "Vstup na východě vede na chodbu. "

    /* Takhle označujeme objekty, které nemají být vidět na dálku. */
    sightSize = small

    gcName = 'vstupu na chodbu, vstupu na chodbu, vstup na chodbu,
        vstupu do na chodbu, vstupem na chodbu'
    gcVocab = 'vstupu vstupem -'
;

/* ------------------------------------------------------------------------ */

+ crChair: Armchair, Fixture 'křeslo/sedátko*pracoviště' 'křeslo' *4
    "Spíš než křeslo je to jednoduché sedátko, do kterého se může obsluha
        připoutat. V tak malé gravitaci netřeba měkkého polstrování, aby sezení
        bylo pohodlné. "

    /* Popisek, který se použije při prozkoumání na dálku (z chodby). */
    distantDesc = "Je uprostřed velínu u pracovního pultíku. "

    /*
     *   Jak hra popíše postavu v křesle viděnou na dálku z chodby (nikoho
     *   jiného, než Borise vidět takto nemůžeme, není třeba dál rozlišovat).
     */
    roomActorThereDesc(actor)
    {
        "Boris sedí ve velínu u počítače. ";
    }

    gcName = 'křesla, křeslu, křeslo, křesle, křeslem'
    gcVocab = 'křesla/křeslu/křesle/křeslem/sedátka/sedátku/sedátkem*pracovišti*
        pracovištěm'
;

+ controlTable: Surface, Fixture 'pultík*pracoviště' 'pultík' *2
    "Pultík<<if technician.isIn(controlRoom)>>, u kterého technik
        pracuje,<<end>> zabírá především klávesnice s popruhem a velké obrazovky
        počítačového terminálu a kamerového systému. "

    /* Výchozí hodnota je 10 a my jsme měnili velikost některých objektů. */
    maxSingleBulk = 10000

    gcName = 'pultíku, pultíku, pultík, pultíku, pultíkem'
    gcVocab = 'pultíku/pultíkem*pracovišti*pracovištěm'
;

/*
 *   Až nastane správná doba, bude na pultíku kafovak, ze kterého Boris pije. Ve
 *   hře máme spoustu PresentLater objektů označených stejným plKey, které se
 *   pak všechny objeví zavoláním PresentLater.makePresentByKey('coffee').
 */
++ techniciansCoffee: PresentLater, CoffeeBag
    plKey = 'coffee'
    coffeeVol = 1

    /*
     *   Aby hra věděla, že patří Borisovi, i když ho momentálně nedrží v ruce,
     *   a mohla se zeptat "A který kafovak máš na mysli? Svůj kafovak nebo
     *   Borisův kafovak?
     */
    owner = technician
;

/* ------------------------------------------------------------------------ */

++ computer: CustomFixture 'textový počítač/terminál' 'terminál' *2

    /*
     *   Popisek většinou zadáváme jako dvojitými uvozovkami ohraničený řetězec.
     *   Když se nám to ale hodí kvůli přehlednosti, můžeme použít plnohodnotnou
     *   metodu, která popisek vytiskne. Totiž dvojitými uvozovkami ohraničený
     *   řetězec je také příkazem k vytištění, je to vlastně anonymní funkce.
     */
    desc()
    {
        "Je to jeden z nejmodernějších textových terminálů pro náročná prostředí
            přímo připojený k hlavnímu serveru. Odolná obrazovka s připojenou
            klávesnicí má kryty v odstínech vojenské šedi kombinované s
            vojenskou zelení. Velmi vysoké rozlišení výrazně zvyšuje pohodlí
            uživatele při čtení a zelený text umocňuje pocit souznění s
            přírodou. ";
            
        if(!gRevealed('database-access') || technician.isIn(controlRoom))
        {
            "Zobrazuje se na něm několik složitých, neustále přibývajících
                textových výpisů. ";
        }
        else
        {
            "Na obrazovce vidíš: <<showScreen>> ";
            
            if(!gRevealed('computer-solved'))
                "<.p>a za <<if pos > input.length()>>dvojtečkou<<else
                    >>šipečkou<<end>> bliká kurzor. ";
        }
    }

    /*
     *   Protože by někdo mohl zkusit počítač zapínat či vypínat, poskytneme
     *   raději vysvětlení. Protože počítač není odvozen ze třídy OnOffControl,
     *   standardní hláška by říkala, že to není něco, co by mohl hráč zapnout
     *   či vypnout, což by bylo hodně matoucí.
     */
    cannotTurnOnMsg = 'Počítač je zapnutý trvale. '
    cannotTurnOffMsg = 'Počítač je zapnutý trvale. '

    /*
     *   Zde začíná řešení šifry. Počítač reaguje na zadání slov ze žlutých
     *   papírků tím, že odpoví podobnými shluky znaků. Vyžaduje od hráče, aby
     *   postupně zjistil správné pořadí, ve kterém má papírky zadávat. Zde si
     *   připravíme seznam odpovědí a seznam vstupních slov ve správném pořadí.
     */
    output = ['SŽPP-L', 'UVÍČA-', 'PTTTÁO', 'ELPČÍÍ', 'RKD-HE',
        'U-<span></span>-OS-']
    input = ['IŘOZA', 'ASÍDK', 'EUANŘ', 'S-E-N', 'ÝOPEK']
    input2 = ['IROZA', 'ASIDK', 'EUANR', 'S-E-N', 'YOPEK']

    /* Kolik toho hráč zatím uhodl (kolik odkrýt, vždy alespoň první). */
    pos = 1

    /* Zde sledujeme, kolik špatných pokusů hráč zadal. */
    numWrong = 0

    /* Popíšeme, co je na obrazovce počítače k vidění. */
    showScreen(appendText = '')
    {
        /* Pokud hráč vyřešil, je zde nabídka interních informací. */
        if(gRevealed('computer-solved'))
        {
            "<pre style=\"color: #060\">Welcome superuser.\nLast login:
                Wed Sep 9 08:53:11 2063 from /dev/ttyS3
                <.p>*******************************
                \n&nbsp;&nbsp;&nbsp;TECHNOUNIVERSE CORP. LTD.
                \n&nbsp;&nbsp;&nbsp;&nbsp;Authorised users only!
                \n*******************************
                <.p>Type R for reports
                \nType S for statistics</pre>";
        }

        /* Pokud nevyřešil, je tu zobrazena šifra. */
        else
        {
            "<.p><tt><span style=\"color: #060\">Crypt mode ON";

            /* Doposud správně zadaná slova a odpovědi. */
            for(local i = 1; i <= pos; i++)
            {
                "<br /><<output[i]>>";
                if(i <= input.length()) "<br />&gt;";
                if(i < pos) say(input[i]);
            }
    
            /* Přidáme případné chybné slovo a otazník. */
            say(appendText);
            "</span></tt><.reveal computer-typing> ";
    
            /*
             *   Jakmile zadá hráč správně celou sekvenci, překvapíme ho otázkou
             *   na heslo.
             */
            if(pos > input.length())
                "<.p><tt><span style=\"color: #060\">Crypt sequence completed.
                    Please enter password now:</span></tt> ";
        }
    }

    /* Kvůli bodování si poznamenáme, zda už přečetl obě důležité informace. */
    read1 = nil
    read2 = nil

    /*
     *   Akce EnterOn je literární akcí, tj. kromě herního objektu, do kterého
     *   se zadává, hráč na příkazové řádce ještě píše libovolný řetězec, který
     *   zadává a ten je pak přísupný jako gLiteral.
     */
    dobjFor(EnterOn)
    {
        /* Povolíme zadávání nezavoláním inherited. */
        verify() { }

        /*
         *   Technik si do počítače nenechá sahat, hráč se ho musí nejprve
         *   zbavit. Plus zabráníme vyřešení ještě před tím, než si Boris sedne
         *   k počítači, hráč by mohl do velínu odbočit, když ho vede do skladu.
         */
        check()
        {
            if(!gRevealed('database-access'))
                failCheck('Nezdá se, že by teď šlo cokoliv do terminálu zadávat,
                    běží tam nějaký program, kterému nerozumíš. ');
            if(technician.isIn(controlRoom))
                failCheck('<.p><q>Pusť mě na chvíli k počítači,</q> zaškemral
                    jsi na Borise.<.p><q>Ani náhodou,</q> odsekl technik,
                    <q>jednak tu mám rozdělanou práci a jednak to, k čemu ty máš
                    přístup, máš ve svým tabletu. Jasný?</q> ');
        }

        action()
        {
            /* Vyřešený počítač reaguje na menu interních informací. */
            if(gRevealed('computer-solved'))
            {
                if(gLiteral.toUpper() == 'R')
                    nestedAction(Examine, computerReports);
                else if(gLiteral.toUpper() == 'S')
                    nestedAction(Examine, computerStatistics);
                else
                    "<.p>Terminál jen nesouhlasně pípl. ";
            }
            
            /* Počítač není zcela vyřešený, takže je možné zadávat. */
            else
            {
                "Na klávesnici jsi vyťukal <q><<gLiteral>></q> a zmáčkl
                    Enter. ";

                /* Při zadání hesla otevřeme přístup k interním informacím. */
                if(gLiteral.toUpper() == 'KORINEK'
                        || gLiteral.toUpper() == 'KOŘÍNEK')
                {
                    computerReports.makePresent();
                    computerStatistics.makePresent();
                    gReveal('computer-solved');
                    achievement.awardPointsOnce();

                    "Skvělé, to evidentně zabralo, protože se na obrazovce
                        počítače objevila následující nabídka:
                        <<showScreen()>> ";
                }

                /*
                 *   Pokud počítač čeká na zadání dalšího slova a hráč ho zadá
                 *   správně, tak ho pochválíme.
                 */
                else if(pos <= input.length()
                        && (gLiteral.toUpper() == input[pos]
                        || gLiteral.toUpper() == input2[pos]))
                {
                    if(pos++ == input.length())
                    {
                        "Zdá se, že počítač nějak zareagoval a tentokrát vypsal
                            výzvu pro zadání hesla: <<showScreen>>
                            <.p>a za dvojtečkou bliká kurzor. ";
                        gReveal('hint-computer-all');
                    }
                    else
                    {
                        "Zdá se, že počítač nějak zareagoval, protože se na
                            obrazovce objevilo: <<showScreen>>
                            <.p>a za šipečkou opět bliká kurzor. ";
                        gReveal('hint-computer-first');
                    }
                    numWrong = 0;
                }

                /* Pokud už počítač čeká na heslo, zopakujeme to. */
                else if(pos > input.length())
                {
                    "Počítač odmítavě pípnul a opět zobrazil výzvu pro zadání
                        hesla: <<showScreen()>> ";
                }

                /* Jiná zadání počítač odmítne. */
                else
                {
                    "Počítač nesouhlasně pípnul: <<showScreen(gLiteral.toUpper()
                        + '<br>?')>> ";

                    /* Pokud už byla zadaná částečná sekence, zrušíme ji. */
                    pos = 1;

                    "<.p>Po chvíli se obrazovka sama smazala a už je na ní zase
                        jen: <<showScreen>><.p>s blikajícím kurzorem. ";

                    /* Přečetl si hráč už papírky? */
                    if(terminalScreen.achievement.scoreCount)
                    {
                        /*
                         *   Po každých třiceti neúspěšných pokusech v řadě
                         *   hintujeme.
                         */
                        numWrong++;
                        if(!(numWrong % 30))
                        {
                            "<.p><q>Kruci,</q> pomyslil sis, <q>neviděl jsem už
                                něco podobného těm podivným znakům?</q> ";
                        }
                    }
                }
            }
        }
    }
    dobjFor(TypeLiteralOn) asDobjFor(EnterOn)

    dobjFor(Unlock)
    {
        verify() { illogical('A jaké heslo zadat? Toť klíčová otázka. '); }
    }
    
    achievement: Achievement { +5 "prolomení hesla do počítače" }

    dobjFor(LookIn) remapTo(Examine, self)

    gcName = 'terminálu, terminálu, terminál, terminálu, terminálem'
    gcVocab = 'textového textovému textovém textovým terminálu/terminálem/
        počítače/počítači/počítačem'
;

+++ Component 'klávesnice' 'klávesnice' *3
    "Klávesnice je součástí terminálu. Pomocí ní můžeš zadat text do počítače. "

    dobjFor(EnterOn) remapTo(EnterOn, computer, IndirectObject)
    dobjFor(TypeLiteralOn) remapTo(TypeLiteralOn, computer, IndirectObject)

    gcName = 'klávesnice, klávesnici, klávesnici, klávesnici, klávesnicí'
    gcVocab = 'klávesnici/klávesnicí'
;

++++ Component 'gumový popruh' 'gumový popruh' *2
    "Pásek gumy, která lehce přitahuje ruce ke klávesnici a tím nahrazuje
        chybějící zemskou přitažlivost. Podobá se gumě do tepláků. "

    gcName = 'gumového popruhu, gumovému popruhu, gumový popruh,
        gumovém popruhu, gumovým popruhem'
    gcVocab = 'gumového gumovému gumovém gumovým popruhu/popruhem'
;

++++ Component 'klávesa s' 'klávesa S' *3
    "Je to klávesa S. "
    
    dobjFor(Push) remapTo(EnterOn, computer, 's')

    gcName = 'klávesy S, klávese S, klávesu S, klávese S, klávesou S'
    gcVocab = 'klávesy klávese klávesu klávesou -'
;

++++ Component 'klávesa r' 'klávesa R' *3
    "Je to klávesa R. "
    
    dobjFor(Push) remapTo(EnterOn, computer, 'r')

    gcName = 'klávesy R, klávese R, klávesu R, klávese R, klávesou R'
    gcVocab = 'klávesy klávese klávesu klávesou -'
;

+++ Component 'ergonomický stojan' 'stojan' *2
    "Stojan umožňuje obrazovky nejrůznějšími způsoby naklánět a posouvat, aby se
        uživateli dobře pracovalo. Teď zrovna je nastavený optimálně,
        nepotřebuješ s ním hýbat. "

    gcName = 'stojanu, stojanu, stojan, stojanu, stojanem'
    gcVocab = 'ergonomického ergonomickému orgonomickém ergonomickým stojanu/
        stojanem'
;

+++ terminalScreen: ComponentDeferrer, RestrictedSurface, Component
    'obrazovka monitor terminálu/počítače/obrazovka/výpisy*obrazovky*monitory'
        'obrazovka terminálu' *3

    desc()
    {
        "Obrazovka teminálu je docela tenká s pevným rámečkem okolo, připevněná
            k pultíku pomocí ergonomického stojanu. ";

        /* Dokud hráč technika neotráví, nevidí nic konkrétního. */
        if(!gRevealed('database-access') || technician.isIn(controlRoom))
        {
            "Zobrazuje se na ní několik složitých, neustále přibývajících
                textových výpisů. ";
        }
        else
        {
            "Na obrazovce vidíš: <<computer.showScreen>> ";
            
            if(!gRevealed('computer-solved'))
                "<.p>a za <<if computer.pos > computer.input.length()
                    >>dvojtečkou<<else>>šipečkou<<end>> bliká kurzor. ";
        }
    }
    
    /*
     *   Na obrazovce je prostor jen pro přilepené papírky, nic jiného zde být
     *   nemůže. Na lístečky máme vlastní třídu Paper, takže testujeme podle ní.
     */
    canPutIn(obj) { return obj.ofKind(Paper); }
    cannotPutInMsg(obj)
    {
        gMessageParams(obj);
        return 'Na obrazovku terminálu můžeš dávat akorát žluté papírky, nic
            jiného se na ní nevejde a nebude držet. ';
    }

    /* Předefinujeme lister, aby hovořil o "přilípnutých" papírcích. */
    contentsLister: ContentsLister, BaseThingContentsLister
    {
        showListPrefixWide(itemCount, pov, parent, selector)
        {
            gMessageParams(parent);
            /*
             *   TODO: Tohle by melo byt osetreno primo v knihovne, aby
             *   generovalo spravny selektor.
             *
             *   "Na {komčem parent} <<selector < 4 ? 'je' : 'jsou'>>
             *   přilípnut<<['ý', 'á', 'é', 'í', 'é', 'á'][selector]>> ";
             */
//            "Na {komčem parent} <<selector < 4 ? 'je' : 'jsou'>> přilípnut<<
//                ['ý', 'á', 'é', 'í', 'é', 'á'][selector]>> ";
            "Na {komčem parent} <<itemCount == 1 || itemCount >= 5 ? 'je'
                : 'jsou'>> přilípnut<<itemCount == 1 ? 'ý' : (itemCount >= 5
                ? 'ých' : 'é')>> ";
        }
    }
    descContentsLister = contentsLister

    /* Podívej se do obrazovky směrujeme na prozkoumání počítače. */
    dobjFor(LookIn) remapTo(Examine, computer)

    achievement: Achievement { +1 "přečtení přilepených lístečků" }

    iobjFor(AimAt)
    {
        action()
        {
            "Namířil jsi tablet na terminál. Software tabletu se pokusil výpis
                terminálu analyzovat, ale nepodařilo se mu to. Pouze zobrazil
                varování, že společnost Technouniverse Corporation nedoporučuje
                psát hesla na žluté papírky přilepené k monitoru. ";
        }
    }

    changeGender = 'monit:2, výpis:5'
    gcName = 'obrazovky terminálu, obrazovce terminálu, obrazovku terminálu,
        obrazovce terminálu, obrazovkou terminálu'
    gcVocab = 'obrazovky obrazovce obrazovku obrazovkou monitoru monitorem
        obrazovky/obrazovce/obrazovku/obrazovkou/monitoru/monitorem/výpisů/
        výpisům/výpisech*obrazovek*obrazovkám*obrazovkách*obrazovkami*monitorů*
        monitorům*monitorech'
;

++++ computerReports: PresentLater, Component, Readable
    'reporty hlášení (ze) (základny)' 'reporty ze základny' *2

    desc()
    {
        "Listuješ pravidelnými reporty ze základny. Jsou si podobné jako vejce
            vejci, liší se jenom datem. Tady musí být ale nuda. Tady…!

            <.p><tt><span style=\"color: #060\">Při těžbě v severním kvadrantu
            došlo k havárii těžebního stroje RA-B4. Příčinou nehody byla neznámá
            geologická anomálie v souřadnicích SQ71. Anomálie nebyla uvedena na
            žádných snímcích, ani ve výsledcích geoprůzkumu. Vzhledem k tomu, že
            došlo k poničení razícího štítu, je oprava nerentabilní, a štít byl
            ponechán na místě. Celá oblast byla uzavřena a těžba přesunuta do
            jihozápadní části i z důvodu vyšší výtěžnosti.</span></tt> ";

        /*
         *   Pokud už byl ve vozítku a viděl na navigaci, kam míří kapitán s
         *   velitelem, připomene mu to, aby si informace spojil dohromady.
         */
        if(gRevealed('robot-solved2'))
            "<.p>Tyhle souřadnice jsi už někde viděl, zablesklo se ti hlavou,
                čísla sis vždycky pamatoval dobře. Kde to bylo? Ano, ubezpečuješ
                se, byl to cíl té údajné inspekční cesty kapitána s velitelem,
                místo, kam odjeli vozítkem. ";

        "<.p>Pod zprávou je kurzívou poznámka velitele základny:

            <.p><tt><span style=\"color: #060\">Veškeré informace o GA na SQ71
            prohlásit za tajné. Oblast označit jako nebezpečnou a
            uzavřít.</span></tt>

            <.p><q>A hele, tady je datum předchozího příletu naší lodi. Co se
            tehdy stalo?</q>

            <.p><tt><span style=\"color: #060\">Přílet pravidelného letu z
            Měsíce, nakládka vytěženého materiálu. Zahynul technik Chriss
            Anderson.</span></tt><<gReveal('chriss-death')>>

            <.p>Mlčky zíráš na ty dva řádky. Nic víc? ";

        /* Odkryjeme několik témat, která by mohl hledat v tabletu. */
        gSetKnown(sq71);
        gSetKnown(tRab4);

        computer.read1 = true;
    }

    isPlural = true
    gcName = 'reportů ze základny, reportům ze základny, reporty ze základny,
        reportech ze základny, reporty ze základny'
    gcVocab = 'reportů reportům reportech -'
;

++++ computerStatistics: PresentLater, Component, Readable
    'výsledky těžby' 'výsledky těžby' *2

    desc()
    {
        "Prohlížíš výsledky těžby několik období zpět. Aha, tady došlo k té
            havárce, skoro úplně se přestalo těžit. Ale hned v dalším období se
            těžba rozjíždí. Zajímavé, nikdy se nedostala na úroveň před tou
            nehodou. Přitom na novém místě měla být ruda bohatší, byl to přece
            jeden z důvodů změny místa těžby. ";

        computer.read2 = true;
    }

    isPlural = true
    gcName = 'výsledků těžby, výsledkům těžby, výsledky těžby, výsledcích těžby,
        výsledky těžby'
    gcVocab = 'výsledků výsledkům výsledcích -'
;

/* ------------------------------------------------------------------------ */

+++ cctvScreen: ComponentDeferrer, Component
    'obrazovka monitor kamerového kamer kamerový systému/obrazovka/kamery/
        systém/monitor*obrazovky*monitory' 'obrazovka kamerového systému' *3
    "Obrazovka kamerového systému je docela tenká s pevným rámečkem okolo,
        připevněná k pultíku pomocí ergonomického stojanu.
        <<if !gRevealed('database-access') || technician.isIn(controlRoom)>>
        Zobrazují se na ní záběry z kamer z různých míst této základny.
        <<if technician.isIn(controlRoom)>> Vypadá to na druhou část základny,
        protože ti místnosti nejsou povědomé, ale než sis je stačil pořádně
        prohlédnout, Boris znervózněl:
        <<technician.nervousList.doScript>><.reveal guard><<end>>
        <<else>><<showScreen>> "

    /* Popis toho, co hráč vidí na obrazovce. */
    showScreen()
    {
        /* Pokud se podíval poprvé, navážeme vizuální spojení. */
        if(!cctvLink.isIn(self))
        {
            /* Objekt cctvLink je MultiLoc, proto moveIntoAdd(). */
            cctvLink.moveIntoAdd(self);
            gSetSeen(robot);
        }

        /* Popíšeme, vzdálenou místnost, kam přes kameru nahlíží. */
        "<.p> ";
        robotRoom.roomRemoteDesc(me);
        gReveal('hint-seen-room');
    }

    iobjFor(AimAt)
    {
        verify()
        {
            /* Dokud hráč Borise nevyřadí, jsou na obrazovce jiné záběry. */
            if(!gRevealed('database-access') || technician.isIn(controlRoom))
                illogicalNow('Nic se nestalo, na obrazovce není vidět nic
                    zajímavého. ');
        }
        action()
        {
            if(!cctvLink.isIn(self))
            {
                cctvLink.moveIntoAdd(self);
                gSetSeen(robot);
            }

            /* Hráč namíří tablet ve skutečnosti na robota v záběru. */
            replaceAction(AimAt, tablet, robot);
        }
    }

    changeGender = 'monit:2'
    gcName = 'obrazovky kamerového systému, obrazovce kamerového systému,
        obrazovku kamerového systému, obrazovce kamerového systému,
        obrazovkou kamerového systému'
    gcVocab = 'obrazovky obrazovce obrazovku obrazovkou monitoru monitorem
        kamerovému kamerovém kamerovým obrazovky/obrazovce/obrazovku/obrazovkou/
        kamerám/kamerách/kamerami/systémem/monitoru/monitorem*obrazovek*
        obrazovkám*obrazovkách*obrazovkami*monitorů*monitorům*monitorech'
;

/* ------------------------------------------------------------------------ */

+ RestrictedSurface, Fixture 'rack/racky' 'racky' *2
    "Racky jsou standardizované plechové konstrukce k ukládání nejrůznější
        výpočetní techniky a příslušenství k ní. Počítačové součásti jsou uvnitř
        přišroubované a chráněné před neopatrností obsluhy. "

    objInPrep = 'na'

    contentsLister: ContentsLister, BaseThingContentsLister
    {
        showListPrefixWide(itemCount, pov, parent, selector)
        {
            gMessageParams(parent);
            "Na {komčem parent} je zavěšený ";
        }
    }
    descContentsLister: DescContentsLister, BaseSurfaceContentsLister
    {
        showListPrefixWide(itemCount, pov, parent, selector)
        {
            gMessageParams(parent);
            "K {komučemu parent} je stahovacím páskem přichycen ";
        }
        showListSuffixWide(itemCount, pov, parent, selector)
        {
            ", který ční do místnosti. ";
        }
    }

    isPlural = true
    gcName = 'racků, rackům, racky, raccích, racky'
    gcVocab = 'racků/rackům/racky/raccích/racky/racku/rackem'
;

++ Decoration 'servery' 'servery' *2
    "V racích za zády obsluhy jsou namontované servery, které řídí chod
        základny. "

    isPlural = true
    gcName = 'serverů, serverům, servery, serverech, servery'
    gcVocab = 'serverů/serverům/serverech'
;

++ PresentLater, Component 'stahovací pásek' 'stahovací pásek' *2
    "Stahovací pásek z velice odolného plastu držící zásobník kafovaků na
        místě."

    plKey = 'coffee'

    gcName = 'stahovacího pásku, stahovacímu pásku, stahovací pásek,
        stahovacím pásku, stahovacím páskem'
    gcVocab = 'stahovacího stahovacímu stahovacím pásku/páskem'
;

++ coffeeBox: PresentLater, Dispenser, CustomFixture
    'zásobník/krabice' 'zásobník' *2
    "Zásobník je jen podlouhlá krabice, do které si Boris připravil pár
        kafovaků, aby se mu nerozlétaly po velínu. Je přichycená stahovacím
        páskem k jednomu ze serverů<<if technician.isIn(controlRoom)>> za
        Borisovými zády<<end>>. Jeden z kafovaků je právě navrchu, ostatní jsou
        schované pod ním. "

    plKey = 'coffee'

    /*
     *   Třída Dispenser je zvláštní v tom, že věci v ní umístěné jsou mírně
     *   upozaděny, aby se nepletly s těmi věcmi, které hráč už vyndal.
     */
    myItemClass = CoffeeBag
    canReturnItem = true
    canPutIn(obj) { return obj.ofKind(CoffeeBag); }
    cannotPutInMsg(obj)
    {
        gMessageParams(obj);
        return 'Boris by tě hnal, kdybys mu do zásobníku strkal {kohoco obj}. ';
    }
    cannotPutInDispenserMsg = 'Boris by tě hnal, kdybys mu do zásobníku strkal
        věci, které tam nepatří. '

    isListed = true
    isListedInContents = true
    isListedInInventory = true

    /*
     *   By default, if the object's contents would be listed in a direct
     *   examination, then also list them when showing an inventory list,
     *   or describing the enclosing room or an enclosing object.
     */
    contentsListed = (contentsListedInExamine)

    cannotTakeMsg = 'Nemůžeš vzít zásobník, je připevněn opravdu opravdu
        důkladně. '

    changeGender = 'krab:3'
    gcName = 'zásobníku, zásobníku, zásobník, zásobníku, zásobníkem'
    gcVocab = 'zásobníku/zásobníkem'
;

/* Dispensable na rozdíl od Thing odebere "(který obsahuje tři kafovaky)" */
class CoffeeBag: SimpleAttachable, PlugAttachable, Dispensable
    'kafovak/káva/kafe/pytlík*kafovaky' 'kafovak' *2
    desc()
    {
        "Plastový pytlík s ventilkem";
        if(coffeeVol == 3)
            " naplněný ledovou kávou připravenou k pití. ";
        else if(coffeeVol == 2)
            " kterou Boris načal, než se mu udělalo nevolno. ";
        else if(coffeeVol == 1)
            ". Boris z něj pravidelně ucucává, takže bude jistě brzy prázdný. ";
        else if(coffeeVol == 0)
            " od ledové kávy, kterou Boris do dna vypil. ";
    }

    isEquivalent = true
    isListedInContents = (!isIn(myDispenser))

    coffeeVol = 3

    /*
     *   Kafovak je major attachable, což znamená, že všechny připojené věci se
     *   budou pohybovat s ním, když ho hráč přemístí. Proto zde vyjmenujeme
     *   věci, které mohou být připojeny jako minoritní položky.
     */
    minorAttachmentItems = [syringe]

    iobjFor(AttachTo)
    {
        check()
        {
            inherited;

            if(gDobj == syringe && owner == technician
                && technician.isIn(controlRoom))
                failCheck('Injekci nemůžeš připojit ke kafovaku, ze kterého
                    technik právě pije. Viděl by tě a kdyby zjistit, co chystáš,
                    možná bys skončil za dveřmi bez skafandru. ');
        }
        action()
        {
            "Injekci jsi nasadil na ventilek kafovaku a teď z něj kouká ven. ";
            inherited;
        }
    }
    dobjFor(AttachTo)
    {
        check()
        {
            inherited;

            if(gIobj == syringe && owner == technician
                && technician.isIn(controlRoom))
                failCheck('Injekci nemůžeš připojit ke kafovaku, ze kterého
                    technik právě pije. Viděl by tě a kdyby zjistit, co chystáš,
                    možná bys skončil za dveřmi bez skafandru. ');
        }
        action()
        {
            "Injekci jsi nasadil na ventilek kafovaku a teď z něj kouká ven. ";
            inherited;
        }
    }

    iobjFor(PutIn)
    {
        verify()
        {
            /* Když je to injekce nebo projímadlo v injekci. */
            if(gDobj == laxative || gDobj == syringe && laxative.isIn(syringe))
            {
                if(owner == technician) illogical('Nemůžeš nic přidávat do
                    kafovaku, ze kterého technik právě pije. Viděl by tě a
                    kdyby zjistit, co chystáš, možná bys skončil za dveřmi bez skafandru. ');
		
                else if(!coffeeVol) illogical('Tento kafovak je prázdný, není
                    v čem projímadlo rozpustit. ');
                else nonObvious;
            }
            else inherited();
        }
        action()
        {
            laxative.moveInto(self);

            /* Byla v některém předchozím kroku připojena injekce? */
            if(attachedObjects.length() > 0)
            {
                "Pomalu jsi vymáčkl celý obsah injekce dovnitř a lehce jsi
                    kafovakem zatřásl, aby se projímadlo zamíchalo v
                    nápoji. ";

                /*
                 *   Když hráč kafovak před vymáčknutím injekce vyndal ze
                 *   zásobníku, neřekneme o otrávení kávy hned, ale počkáme, až
                 *   kafovak zase do zásobníku nastraží zpět.
                 */
                if(isIn(coffeeBox))
                {
                    sayTechnicianPoisoned();
                }

                /* Injekci zase odpojíme. */
                syringe.detachFrom(self);
                syringe.moveInto(me);
            }

            /* Pokud nebyla, tak vysvětlíme trochu podrobněji. */
            else
            {
                /* Pokud je kafovak v zásobníku. */
                if(isIn(coffeeBox))
                {
                    "<.p>Potichu za Borisovými zády jsi vzal do ruky vrchní
                        kafovak z jeho zásob, nasadil na něj injekci s
                        projímadlem a pomalu vymáčkl celý její obsah dovnitř.

                        <.p>Lehce jsi kafovakem zatřásl, aby se projímadlo
                        zamíchalo v nápoji a opět potichu a opatrně vrátil tam,
                        kde jsi ho vzal. ";

                    sayTechnicianPoisoned();
                }

                /*
                 *   Když hráč kafovak před vymáčknutím injekce vyndal ze
                 *   zásobníku, neřekneme o otrávení kávy hned, ale počkáme, až
                 *   kafovak zase do zásobníku nastraží zpět.
                 */
                else
                {
                    "<.p>Injekci s projímadlem jsi nasadil na kafovak a pomalu
                        vymáčkl celý její obsah dovnitř.

                        <.p>Lehce jsi kafovakem zatřásl, aby se projímadlo
                        zamíchalo v nápoji. ";
                }
            }
        }
    }
    iobjFor(PourInto) asIobjFor(PutIn)

    dobjFor(PutIn)
    {
        action()
        {
            /*
             *   Otrávený kafovak může hráč nastražit zpátky do zásobníku nebo
             *   na pultík.
             */
            if(gIobj == coffeeBox && laxative.isIn(self))
            {
                "<.p>Kafovak jsi nastražil do zásobníku pěkně na vrch. ";
                sayTechnicianPoisoned();
            }
            if(gIobj == controlTable && laxative.isIn(self))
            {
                "<.p>Když byl Boris zabrán do práce, tak jsi nenápadně nastražil
                    kafovak na kraj pultíku. ";
                sayTechnicianPoisoned();
            }
            inherited;
        }
    }
    dobjFor(PutOn) asDobjFor(PutIn)

    /* Společná část, která se může vyvolat z několika různých situací. */
    sayTechnicianPoisoned()
    {
        "<.p>Pro jistotu jsi odplul kousek stranou.

            <.p><.p>Boris právě do sucha vysál poslední kapky nápoje z kafovaku
            na pultíku a natáhl se pro další. Po paměti si srovnal ventil a s
            náruživostí těžkého závisláka se přisál. Ááááá, první lok. Božský
            nektar a ambrosie nemohou být slastnější. Každá buňka těla sleduje
            cestu kávového megaloku a žaludek je plně připraven kofeinovou drogu
            expedovat dál do těla.

            <.p><q>Uáááh,</q> nečekaná křeč zkroutila technikovo tělo tak
            prudce, že upustil kafovak, a rukama se pokusil obejmout celé
            tělo. Zezelenal a na čele mu vyrazily obrovské krůpěje potu. A
            další nápor střevní peristaltiky ho vyhnal z místnosti jednoznačným
            směrem.

            <.p><q>Chudák Boris,</q> pomyslel sis, <q>ale na druhou stranu, tu
            konečně přestal hlídat. Ale kruci, zamkl počítač.</q> ";

        /*
         *   Vyprázdníme původní technikův kafovak a tenhle trochu načneme, aby
         *   nebyl úplně plný.
         */
        techniciansCoffee.coffeeVol = 0;
        coffeeVol = 2;

        /* Technika přemístíme na toaletu a zamkneme dveře. */
        technician.moveIntoForTravel(toilet);
        technician.setCurState(technicianOnToilet);

        toiletDoor.makeOpen(nil);
        toiletDoor.makeLocked(true);

        scoreMarker.awardPointsOnce();
        gReveal('hint-poisoned');
        clock.eventReached();
    }
    clock: ClockEvent { eventTime = [2, 10, 12] }
    scoreMarker: Achievement { +3 "odlákání Borise z velínu" }

    dobjFor(Take)
    {
        check()
        {
            /* Své kafe si Boris hlídá. */
            if(owner == technician && technician.isIn(controlRoom))
                failCheck('<q>Hele, packy pryč,</q> zvýšil hlas Boris, <q>na
                    <i>moje</i> kafe si nedovolila sáhnout ani moje
                    máma.</q> ');

            /* Ze zásobníku nám stačí vzít jen jeden kafovak. */
            if(location == coffeeBox && coffeeBox.contents.length() < 3)
                failCheck('Asi bys tu měl Borisovi nějaké kafovaky nechat, aby
                    nedostal absťák. ');

            inherited;
        }
    }

    /* Ochutnání */
    dobjFor(Taste)
    {
        preCond = static inherited + objHeld
        verify()
        {
            if(spaceSuit.isWornBy(me))
                illogicalNow('Na hlavě máš přilbu vesmírného skafandru, takže
                    teď pít nemůžeš. ');
            if(coffeeVol == 0)
                illogical('Kafovak je prázdný, není co pít. ');
        }

        check()
        {
            if(owner == technician && technician.isIn(controlRoom))
                failCheck('<q>Hele, packy pryč,</q> zvýšil hlas Boris, <q>na
                    <i>moje</i> kafe si nedovolila sáhnout ani moje máma. ');
            if(laxative.isIn(self))
                failCheck('Do tohoto kafovaku jsi přidal projímadlo, napít se by
                    bylo pošetilé. ');

            inherited;
        }

        action()
        {
            "Zvedl jsi kafovak k ústům a opatrně nasál lok údajného lahodného
                moku. Tvé chuťové buňky se vzepjaly a pokusily se uprchnout ti z
                jazyku. Krk se prudce sevřel a pokusil se zabránit ve vstupu
                mletého hnoje smíchaného s asfaltem do vnitřku tvého těla. ";
        }
    }
    dobjFor(Drink) asDobjFor(Taste)

    cannotPourMsg = 'Není dobrý nápad vylít kafovak jen tak. '
    dobjFor(PourInto)
    {
        verify() { if(gIobj != slot2) inherited(); }
    }
    dobjFor(PourOnto) asDobjFor(PourInto)

    /* Předložku "k" vokalizujeme, tj. píšeme ji jako "ke kafovaku". */
    vocalize = ['k']

    changeGender = 'pytlík:2'
    gcName = 'kafovaku, kafovaku, kafovak, kafovaku, kafovakem,
        kafovaky, kafovaků, kafovakům, kafovaky, kafovacích, kafovaky,
        kafovaků, kafovaků, kafovakům, kafovaků, kafovacích, kafovaky'

    /*
     *   Zde nedáme "*kafovaků", protože ">napij se z kafovaku" by bral
     *   přednostně jako odkroužkovaný plurál.
     */
    gcVocab = 'kafovaku/kafovakem/kávy/kávě/kávu/kávou/kafi/kafem/pytlíku/
        pytlíkem*kafovakům*kafovacích'
;

+++ coffee1: CoffeeBag;
+++ coffee2: CoffeeBag;
+++ coffee3: CoffeeBag;

/*
 *   Ventilek je na každém kafovaku, přesto však to je v podstatě furt ten
 *   stejný ventilek.
 */
MultiFaceted
    instanceObject: Component
    {
        vocabWords = 'ventil/ventilek'
        name = 'ventilek'
        gender = 2
        desc = "Ventilek slouží k plnění vodou z vodomatu, k přidávání
            ingrediencí, nebo přímo k nasávání nápoje. Např. mléko z koncentrátu
            lze do kávy přidat injekční stříkačkou. "

        isListedInContents = nil

        gcName = 'ventilku, ventilku, ventilek, ventilku, ventilkem'
        gcVocab = 'ventilku/ventilkem'
    }
    initialLocationClass = CoffeeBag
;

/*
 *   Zavedeme sloveso "bodni", protože hráči mají tendenci injekci do zásobníku
 *   zabodnout.
 */
VerbRule(StabTo)
    ('bodni' | 'bodnout' | 'píchni' | 'píchnout') dobjList
    ('k' | 'ke' | 'na' | 'do') singleIobj
    : AttachToAction
    askIobjResponseProd = atSingleNoun
    verbPhrase
    {
        if(dobjList_.length() > 0 && dobjList_[1].obj_.ofKind(PlugAttachable) &&
            iobjList_.length() > 0 && iobjList_[1].obj_.ofKind(PlugAttachable))
            return 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)';
        else
            return 'připevnit/připevňu{ješ}/připevnil{a} (co) (k čemu)';
    }
;

VerbRule(StabToWhat)
    [badness 500] ('bodni' | 'bodnout' | 'píchni' | 'píchnout') dobjList
    : AttachToAction
    verbPhrase
    {
        if(dobjList_.length() > 0 && dobjList_[1].obj_.ofKind(PlugAttachable) ||
            iobjList_.length() > 0 && iobjList_[1].obj_.ofKind(PlugAttachable))
            return 'připojit/připoju{ješ}/připojil{a} (co) (k čemu)';
        else
            return 'připevnit/připevňu{ješ}/připevnil{a} (co) (k čemu)';
    }
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = atSingleNoun;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Společná třída pro papírky přilípnuté na monitoru počítače. Slovník
 *   obsahuje slova společná pro všechny papírky.
 */
class Paper: PresentLater, Thing
    vocabWords = 'papírek*papírky'
    gender = 2
    plKey = 'paper'

    /*
     *   Papírky tvoří skupinu, která bude listovaná v inventáři atp. společně.
     *   Hra napíše, že máš určitý počet papírků a v závorce vyjmenuje jen
     *   přídavná jména.
     */
    listWith = [paperGroup]
    paperGroupName = ''

    gcVocab = 'papírku/papírkem*papírků*papírkům*papírky*papírkách'
    pluralName = 'žluté papírky'
    pluralNameKohoCeho = 'žlutých papírků'
    pluralNameKomuCemu = 'žlutým papírkům'
    pluralNameKohoCo = 'žluté papírky'
    pluralNameKomCem = 'žlutých papírkách'
    pluralNameKymCim = 'žlutými papírky'
    
    dobjFor(Take)
    {
        check()
        {
            if(technician.isIn(controlRoom))
            {
                /*
                 *   U papírků je typické, že uživatel mohl zadat příkaz, který
                 *   způsobí braní jednoho papírku za druhým. Proto zastavíme
                 *   případné opakování stejné neúspěšné akce na zbylých
                 *   papírcích.
                 */
                gAction.cancelIteration();

                failCheck('Natáhnul ses pro žlutý papírek, ale jak Boris viděl,
                    k čemu se chystáš, tak se ohradil: <q>Hej, to je
                    moje.</q> ');
            }
        }
    }
    dobjFor(Doff) asDobjFor(Take)
    dobjFor(Remove) asDobjFor(Take)
;

/*
 *   Při listování skupiny budeme uvádět jen přídavná jména místo celého názvu.
 *   Právě ta máme nastavena u každého papírku ve vlastnosti paperGroupName.
 */
paperGroup: ListGroupParen
    showGroupItem(lister, obj, options, pov, info)
        { say(obj.paperGroupName); }
;

/*
 *   Nyní vytvoříme každý z pěti papírků. Ve slovníku uvádíme jen rozlišující
 *   přídavné jméno, zbytek slovníku je zděděn ze společné třídy.
 */
paper1: Paper 'zmuchlaný -' 'zmuchlaný papírek' @terminalScreen
    desc()
    {
        /* Dokud Boris hlídá, nedovolíme hráči si papírky pořádně přečíst. */
        if(technician.isIn(controlRoom))
        {
            "Je to žlutý lepivý papírek a propiskou je na něm něco napsáno, ale
                než sis ho stihnul pořádně prohlédnout, Boris znervózněl: ";
    
            /* Položku z EventListu zobrazíme zavoláním doScript(). */
            technician.nervousList.doScript();

            gReveal('guard');

            /*
             *   Když by hráč zadal příkaz v množném čísle, tak by vypadalo
             *   hloupě, kdybychom opakovali Brisovu nerovozitu za každým
             *   papírkem, proto raději stornujeme zbytek příkazové řádky.
             */
            gAction.cancelIteration();
        }
        else
        {
            "Je to žlutý lepivý papírek a propiskou je na něm napsáno:
                <<computer.input[1]>> ";
            terminalScreen.achievement.awardPointsOnce();
        }
    }

    /* Název v rámci listování skupiny. */
    paperGroupName = 'zmuchlaný'

    gcName = 'zmuchlaného papírku, zmuchlanému papírku, zmuchlaný papírek,
        zmuchlaném papírku, zmuchlaným papírkem'
    gcVocab = 'zmuchlaného zmuchlanému zmuchlaném zmuchlaným -'
;

paper2: Paper 'ohmataný -' 'ohmataný papírek' @terminalScreen
    desc()
    {
        if(technician.isIn(controlRoom))
        {
            "Je to žlutý lepivý papírek a propiskou je na něm něco napsáno, ale
                než sis ho stihnul pořádně prohlédnout, Boris znervózněl: ";

            technician.nervousList.doScript();
            gReveal('guard');
            gAction.cancelIteration();
        }
        else
        {
            "Je to žlutý lepivý papírek a propiskou je na něm napsáno:
                <<computer.input[2]>> ";
            terminalScreen.achievement.awardPointsOnce();
        }
    }
    
    paperGroupName = 'ohmataný'

    gcName = 'ohmataného papírku, ohmatanému papírku, ohmataný papírek,
        ohmataném papírku, ohmataným papírkem'
    gcVocab = 'ohmataného ohmatanému ohmataném ohmataným'
;

paper3: Paper 'natrhnutý -' 'natrhnutý papírek' @terminalScreen
    desc()
    {
        if(technician.isIn(controlRoom))
        {
            "Je to žlutý lepivý papírek a propiskou je na něm něco napsáno, ale
                než sis ho stihnul pořádně prohlédnout, Boris znervózněl: ";

            technician.nervousList.doScript();
            gReveal('guard');
            gAction.cancelIteration();
        }
        else
        {
            "Je to žlutý lepivý papírek a propiskou je na něm napsáno:
                <<computer.input[3]>> ";
            terminalScreen.achievement.awardPointsOnce();
        }
    }
    
    paperGroupName = 'natrhnutý'

    gcName = 'natrhnutého papírku, natrhnutému papírku, natrhnutý papírek,
        natrhnutém papírku, natrhnutým papírkem'
    gcVocab = 'natrhnutého natrhnutému natrhnutém natrhnutým'
;

paper4: Paper 'přeložený -' 'přeložený papírek' @terminalScreen
    desc()
    {
        if(technician.isIn(controlRoom))
        {
            "Je to žlutý lepivý papírek a propiskou je na něm něco napsáno, ale
                než sis ho stihnul pořádně prohlédnout, Boris znervózněl: ";

            technician.nervousList.doScript();
            gReveal('guard');
            gAction.cancelIteration();
        }
        else
        {
            "Je to žlutý lepivý papírek a propiskou je na něm napsáno:
                <<computer.input[4]>> ";
            terminalScreen.achievement.awardPointsOnce();
        }
    }

    paperGroupName = 'přeložený'

    gcName = 'přeloženého papírku, přeloženému papírku, přeložený papírek,
        přeloženém papírku, přeloženým papírkem'
    gcVocab = 'přeloženého přeloženému přeloženém přeloženým'
;

paper5: Paper 'srolovaný -' 'srolovaný papírek' @terminalScreen
    desc()
    {
        if(technician.isIn(controlRoom))
        {
            "Je to žlutý lepivý papírek a propiskou je na něm něco napsáno, ale
                než sis ho stihnul pořádně prohlédnout, Boris znervózněl: ";

            technician.nervousList.doScript();
            gReveal('guard');
            gAction.cancelIteration();
        }
        else
        {
            "Je to žlutý lepivý papírek a propiskou je na něm napsáno:
                <<computer.input[5]>> ";
            terminalScreen.achievement.awardPointsOnce();
        }
    }

    paperGroupName = 'srolovaný'
    vocalize = ['z']

    gcName = 'srolovaného papírku, srolovanému papírku, srolovaný papírek,
        srolovaném papírku, srolovaným papírkem'
    gcVocab = 'srolovaného srolovanému srolovaném srolovaným'
;
