#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   VE VOZÍTKU
 */
roverDoorProxy: UnlistedProxyConnector
    primaryConn = roverDoor
;

roverFloor: Floor 'podlaha/zem' 'podlaha' *3
    "Podlaha je pokrytá tmavým neklouzavým materiálem. "

    objIntoCase = 4

    gcName = 'podlahy, podlaze, podlahu, podlaze, podlahou'
    gcVocab = 'podlahy/podlaze/podlahu/podlahou/panelu/panelem/polyuretanu/
        polyuretanem/země/zemi/zemí'
;

roverCeiling: RoomPart 'strop' 'strop' *2
    "Strop vozítka zaoblený, plynule přecházející v stěny, pokrytý světlým
        polstrováním. "

    gcName = 'stropu, stropu, strop, stropu, stropem'
    gcVocab = 'stropu/stropem'
;

roverPortWall: DefaultWall
    'levá zeď/stěna*zdi*stěny' 'levá stěna' *3
    "Levá stěna vozítka je povětšinou hladká, mírně zaoblená podle vnějšího
        tvaru vozítka, pokrytá světlým polstrováním. "

    gcVocab = 'levé levou zdi/zdí/stěny/stěně/stěnu/stěnou*zdí*zdím*stěn*stěnám*
        stěnách*stěnami'
;

roverStarboardWall: DefaultWall
    'pravá zeď/stěna*zdi*stěny' 'pravá stěna' *3
    "Pravá stěna vozítka je povětšinou hladká, mírně zaoblená podle vnějšího
        tvaru vozítka, pokrytá světlým polstrováním. "

    gcVocab = 'pravé pravou zdi/zdí/stěny/stěně/stěnu/stěnou*zdí*zdím*stěn*
        stěnám*stěnách*stěnami'
;

roverAftWall: DefaultWall
    'zadní zeď/stěna*zdi*stěny' 'zadní stěna' *3
    "Zadní stěna téměř plochá, část z ní zabírají dva průlezy do skafandrů. "

    gcName = 'zadní stěny, zadní stěně, zadní stěnu, zadní stěně, zadní stěnou'
    gcVocab = 'zdi/zdí/stěny/stěně/stěnu/stěnou*zdí*zdím*stěn*stěnám*stěnách*
        stěnami'
;

roverForeWall: DefaultWall
    'přední zeď/stěna*zdi*stěny' 'přední stěna' *3
    "Přední stěna vozítka má tvar vypuklé polokoule a většinu z ní zabírají
        průzory, takže zvenku vypadá vozítko tak trochu jako oko mouchy. "

    gcName = 'přední stěny, přední stěně, přední stěnu, přední stěně,
        přední stenou'
    gcVocab = 'zdi/zdí/stěny/stěně/stěnu/stěnou*zdí*zdím*stěn*stěnám*stěnách*
        stěnami'
;

rover: BaseRoom 'Ve vozítku' 'do vozítka'
    "Vnitřní vybavení je velice strohé, před čelní stěnou z mnoha menších
        průzorů a jednoduchým ovládacím pultem s navigací jsou tu dvě křesla
        pro posádku, vzadu pak další nezbytné vybavení v bočních komorách.
        Úplně vzadu jsou dva průlezy do skafandrů zavěšených zvenku. "

    out = gRevealed('rover-destination') ? suitPort : (roverDoor.isOpen()
        ? roverDoorProxy : nil)
    west = roverDoor.isOpen() ? roverDoor : nil

    /* Před popis předmětů, když se kouká do vozítka průzory. */
    inRoomName(pov) { return 've vozítku'; }

    roomParts = [roverForeWall, roverAftWall, roverStarboardWall, roverPortWall,
        roverFloor, roverCeiling]

    atmosphereList = gRevealed('rover-driving') ? drivingAtmosphere : nil
    drivingAtmosphere: EventList
    {
        [
            '',
            'Celým vozítkem to lehce zahoupalo, jak jsi několikrát změnil směr
                při vyhýbání drobným nerovnostem. ',
            'Okolo se ozývá jen lehké bzučení elektromotorů a šumění recyklace
                vzduchu. '
        ]
    }

    /* Tři tahy po odjetí se spustí rozhovor. */
    startRadioFuse() { new Fuse(self, &initiateRadio, 3); }

    initiateRadio()
    {
        commander.moveIntoForTravel(radioRoom);
        commander.initiateConversation(commanderOnRadio, 'do-you-read-me');
    }

    /*
     *   Tohle tu musi byt ne pro prikaz zeptej se na xxx, ale pro prikaz zeptej
     *   se velitele na xxx.
     */
    getExtraScopeItems(actor)
    {
        if(commander.curState == commanderOnRadio)
            return inherited(actor) + commander;
        else
            return inherited(actor);
    }

    roomDaemon()
    {
        if(gRevealed('lunch'))
            gReveal('want-drive');

        inherited;
    }

    dobjFor(Move) remapTo(Move, roverStick)

    gender = 4
    vocabWords = 'vozítko/vozítka/vozítku/vozítkem/vozidlo/vozidla/vozidlu/
        vozidlem/vozík/vozíku/vozíkem'
    gcName = 'vozítka, vozítku, vozítko, vozítku, vozítkem'
;

+ roverDoor: IndirectLockable, Door
    'tlakové dveře*průlezy' 'tlakové dveře' *3
    "Dveře vozítka slouží k nastupování v doku, jsou akorát tak velké, aby se
        jimi bez potíží protáhl kosmonaut. "

    isPlural = true

    initiallyOpen = true
    initiallyLocked = nil
    cannotOpenLockedMsg = 'Nemůžeš otevřít dveře do vzduchoprázdna. Jsou
        automaticky blokované. '
    cannotUnlockMsg = 'Dveře nemají žádný zámek. '
    cannotLockMsg = 'Dveře nemají žádný zámek. '
    alreadyLockedMsg = 'Dveře nemají žádný zámek. '
    alreadyUnlockedMsg = 'Dveře nemají žádný zámek. '

    gcName = 'tlakových dveří, tlakovým dveřím, tlakové dveře,
        tlakových dveřích, tlakovými dveřmi'
    gcVocab = 'tlakových tlakovým tlakové tlakovými dveří/dveřím/dveřích/dveřmi'
;

+ suitPort: TravelWithMessage, ThroughPassage
    'skafandrový průlez/skafandr/skafandry*průlezy' 'skafandrový průlez' *2
    "Nepoužívané skafandry jsou zavěšené zády na vnější stěně vozítka a jejich
        zavíratelná pouzdra s podporou života otevřená dovnitř vozítka slouží
        místo průlezu. Po oblečení skafandru, když se astronaut nasouká průlezem
        dovnitř, se pouzdro uzavře, utěsní a uvolní, takže je možné opustit
        vozítko i bez přechodové komory a tím šetřit cenný vzduch. "

    dobjFor(TravelVia)
    {
        preCond = inherited
            + canTravelerPassPreCondition
            + handsEmptyForSuit
            + new ObjectPreCondition(shoes, objNotHeld)
            + new ObjectPreCondition(stethoscope, objNotHeld)
    }
    dobjFor(Enter) remapTo(TravelVia, self)
    dobjFor(Wear) remapTo(TravelVia, self)

    /*
     *   Tato PreCondition funguje podobně, jako check() akce, tj. má za úkol
     *   zkontrolovat okolnosti a případně odmítnout provedení příkazu. Proč ale
     *   PreCondition? Tyto kontroly sice perfektně zapadají do check() metody,
     *   ale protože pomocí jiných PreCondition vynucujeme další okolnosti, není
     *   výstup ze hry elegantní.
     *
     *   Tady vynucujeme, aby hráč odložil nesené věci a aby si zul boty, než se
     *   do skafandru obleče. Když se hráč rozhodne vystoupit za jízdy a my tomu
     *   zabráníme kontrolou v check(), potom hráče nejprve donutíme podstoupit
     *   odkládání věcí a potom teprve dojde ke kontrole a řekneme mu, proč
     *   nemůže vystoupit. Proto tuto kontrolu přesouváme do PreCondition běžící
     *   dříve, než triviální přípravy.
     */
    canTravelerPassPreCondition: PreCondition
    {
        checkPreCondition(obj, allowImplicit)
        {
            if (gRevealed('rover-destination'))
                return nil;
    
            /* we can't try moving anything around, so simply give up */
            if(gRevealed('rover-driving'))
                reportFailure('Nemůžeš vystoupit za jízdy. ');
            else
                reportFailure('Teď opravdu není dobrý nápad se procházet mezi
                    zaparkovanými vozítky. ');
            exit;
        }
    }

    /* a customized hands-empty condition for wearing the suit */
    handsEmptyForSuit: handsEmpty
    {
        failureMsg = 'Nejprve bys měl odložit všechno, co máš u sebe, aby ses
            mohl do skafandru nasoukat. '

        /*
         *   Require dropping everything except the chicken suit (we
         *   obviously wouldn't want to require dropping the suit itself,
         *   since we need to be holding it to put it on).  Include even
         *   worn items that aren't AlwaysWorn's.
         */
        requireDropping(obj)
        {
            /*
             *   We have to drop everything that we're either holding or
             *   wearing, except for the chicken suit itself and any
             *   AlwaysWorn items.
             */
            return
            (
                obj != overal && obj != tshirt && obj != shoes
                &&
                (
                    obj.isHeldBy(gActor)
                    || (obj.ofKind(Wearable)
                        && !obj.ofKind(AlwaysWorn)
                        && obj.isWornBy(gActor))
                )
            );
        }

        /*
         *   don't try auto-bagging anything; since we need to drop
         *   everything, including the tote bag, there's no point in trying
         *   to free up space that way
         */
        autoBag = nil
    }

    travelDesc = "Nasoukal ses do skafandru, zkontroloval zásoby vzduchu a
        spustil odpojení. Zadní kryt se pomalu uzavřel, utěsnil vozítko a potom
        se zádová část skafandru uvolnila.<<roverSuit.moveInto(me)>>
        <<roverSuit.makeWornBy(me)>> "

    gcName = 'skafandrového průlezu, skafandrovému průlezu, skafandrový průlez,
        skafandrovém průlezu, skafandrovým průlezem'
    gcVocab = 'skafandrového skafandrovému skafandrovém skafandrovým průlezu/
        průlezem/skafandru/skafandrem/skafandrů/skafandrům/skafandrech'
;

/* ------------------------------------------------------------------------ */

+ Armchair, CustomFixture 'křeslo' 'křeslo' *4
    "Docela obyčejná anatomická křesla pro člověka, který vozítko ovládá, nebo
        se v něm veze. "

    cannotTakeMsg = 'Křeslo je pevně přidělané k podlaze. '

    gcName = 'křesla, křeslu, křeslo, křesle, křeslem'
    gcVocab = 'křesla/křeslu/křesle/křeslem'
;

+ CustomFixture 'ovládací pult (s) pult/panel/(navigací)' 'ovládací pult' *2
    "Ovládací pult není velký, většinu plochy zabírá obrazovka navigace, před
        ní je malá řídicí páka a slot na kartu. "

    iobjFor(PutIn) remapTo(PutIn, DirectObject, slot)
    iobjFor(AimAt) remapTo(AimAt, DirectObject, navScreen)

    gcName = 'ovládacího pultu, ovládacímu pultu, ovládací pult,
        ovládacím pultu, ovládacím pultem'
    gcVocab = 'ovládacího ovládacímu ovládacím pultu/pultem/panelu/panelem'
;

++ slot: RestrictedContainer, Component
    'slot čtečka (na) (kartu)/slot/čtečka' 'slot na kartu' *2
    "Je to úzká štěrbina na identifikační kartu, bez karty nelze vozítko
        ovládat. "

    validContents = [card]
    cannotPutInMsg(obj)
    {
        gMessageParams(obj);
        return 'Čtečka je určena jen na identifikační karty, {kohoco obj} do
            ní vložit nemůžeš. ';
    }

    vocalize = ['z']
    gcName = 'slotu na kartu, slotu na kartu, slot na kartu, slotu na kartu,
        slotem na kartu'
    gcVocab = 'slotu slotem čtečky čtečce čtečku čtečkou slotu/slotem/čtečky/
        čtečce/čtečku/čtečkou'
;

++ roverStick: Component 'řídicí páka/knipl' 'řídicí páka' *3
    "Řídicí pákou posádka může plynule měnit směr pohybu vozítka, potřebuje-li
        ho ovládat ručně. "

    dobjFor(Push)
    {
        verify() {}
        check()
        {
            if(!card.isIn(slot)) failCheck('Vůbec nic se nestalo. Jízda je
                zřejmě možná jen se zasunutou identifikační kartou ve slotu. ');

            inherited;
        }
        action()
        {
            "Přitáhl jsi řídicí páku k sobě a vyrazil na cestu.
                <.p>Razantní přitažení páky sice nezpůsobilo zaskřípění
                pneumatik a okamžité zrychlení v oblacích prachu, místo toho se
                vozítko rozjelo díky své vlastní automatice jen velmi pozvolna a
                opatrně, ale přesto ti teď už nic nemůže stát v cestě k
                tajemství na konci cesty. ";

            roverDoor.makeOpen(nil);
            roverDoor.makeLocked(true);

            rover.startRadioFuse();

            gReveal('rover-driving');
            clock.eventReached();
        }
    }
    clock: ClockEvent { eventTime = [2, 13, 41] }

    dobjFor(Pull) asDobjFor(Push)
    dobjFor(Move) asDobjFor(Push)

    gcName = 'řídicí páku, řídicí páce, řídicí páku, řídicí páce, řídicí pákou'
    gcVocab = 'páku/páce/pákou/kniplu/kniplem'
;

++ navScreen: Component
    'obrazovka navigace/mapa/obrazovka' 'obrazovka navigace' *3
    desc()
    {
        "Navigace je dost velká, aby se na ní vešla mapa okolí. ";

        if(gRevealed('rover-destination'))
        {
            "Tvé vozítko právě stojí v zakázaném kvadrantu SQ71. Symbol druhého
                vozítka se zvolna a neodvratně přibližuje po stejné trase. ";
        }
        else if(gRevealed('rover-driving'))
        {
            "Přibližuješ se ke kvadrantu SQ71 přesně podle záznamu minulé cesty
                v navigaci. ";
        }
        else if(gRevealed('lunch'))
        {
            "Na navigaci zůstala šedivě zakreslená nedávno projetá trasa druhého
                průzkumného vozítka z kvadrantu SQ71 zpět k základně. Co jen
                mohli v zakázané oblasti dělat? ";
        }
        else
        {
            "Na mapě je vidět poloha druhého vozítka, ";

            if(gRevealed('computer-solved2') && gRevealed('tresor-solved2'))
            {
                "právě teď se blíží zpět k základně z kvadrantu SQ71. ";

                if(gRevealed('computer-solved2'))
                    "<.p>Tyhle souřadnice jsi už někde viděl, problesklo ti
                        hlavou, čísla sis vždycky pamatoval dobře. Kde to bylo?
                        Ano, ubezpečuješ se, jsou to souřadnice té havárky,
                        místa, kde byla přerušena těžba. ";

                "<.p>Už je docela blízko, raději bys měl jít odsud pryč, abys
                    nebyl přistižen v místech, kde nemáš co dělat.
                    <.reveal robot-solved2> ";

                gSetKnown(sq71);
                sq71.discover();
                scoreMarker.awardPointsOnce();
            }
            else if(gRevealed('computer-solved2') || gRevealed('tresor-solved2'))
            {
                "právě teď stojí docela daleko od základny v kvadrantu SQ71.
                <.reveal robot-solved2> ";

                if(gRevealed('computer-solved2'))
                    "<.p>Tyhle souřadnice jsi už někde viděl, problesklo ti
                        hlavou, čísla sis vždycky pamatoval dobře. Kde to bylo?
                        Ano, ubezpečuješ se, jsou to souřadnice té havárky,
                        místa, kde byla přerušena těžba. ";

                gSetKnown(sq71);
                sq71.discover();
                scoreMarker.awardPointsOnce();
            }
            else
            {
                "které se vzdaluje od základny.

                    <.p>Proč jedou tímhle směrem? Podle navigace se tam netěží.
                    Kouknu se později, kam vlastně jeli, ukládáš si do
                    paměti. ";
            }
        }
    }
    scoreMarker: Achievement { +2 "zjištění cíle cesty vozítka" }

    iobjFor(AimAt)
    {
        action()
        {
            "Na displeji tabletu se zobrazilo totéž, jako právě zobrazuje
                navigace. Jen ve vyšším rozlišení a u spodního okraje září
                hláška <q>Společnost TechnoUniverse Corporation Vám přeje
                bezpečnou cestu</q>. ";
        }
    }

    gcName = 'obrazovky navigace, obrazovce navigace, obrazovku navigace,
        obrazovce navigace, obrazovkou navigace'
    gcVocab = 'obrazovky obrazovce obrazovku obrazovkou navigaci/navigací/
        mapy/mapě/mapu/mapou/obrazovky/obrazovce/obrazovku/obrazovkou'
;

+++ sq71: Hidden, Component
    'zakázaná oblast souřadnice kvadrant sq71/sq-71' 'SQ71' *2
    "Na mapě navigace je kvadrant SQ71 vyznačen přeškrtnutým červeným rámečkem.
        V něm končí i jedna z těžebních šachet. "

    changeGender = 'souř: 5'
    gcVocab = 'zakázané zakázanou souřadnic souřadnicím souřadnicích
        souřadnicemi oblasti oblastí kvadrantu kvadrantem -'
;

/* ------------------------------------------------------------------------ */

+ Decoration 'boční komory' 'boční komory' *3
    "V komorách moc užitečného není a beztak si nic z toho nemůžeš vzít ven, do
        skafandru se s věcmi nenasoukáš. "

    isPlural = true
    gcName = 'bočních komor, bočním komorám, boční komory, bočních komorách,
        bočními komorami'
    gcVocab = 'bočních bočním bočními komor/komorám/komorách/komorami'
;

/* ------------------------------------------------------------------------ */

Occluder, SenseConnector, Fixture 'průzor/průzory' 'průzor' *2
    "Průzory pokrývají téměř celou zaoblenou přední část vozítka a poskytují
        posádce rozhled skoro všemi směry. "

    locationList = [rover, tunnel]
    connectorMaterial = adventium
    isListedInContents = nil

    /* Hráč by neměl předními průzory vidět na plošinu na zádi vozítka. */
    occludeObj(obj, sense, pov)
    {
        if(obj.isIn(roverPlatform) && pov.isOrIsIn(rover)) return true;
        return inherited(obj, sense, pov);
    }

    dobjFor(LookThrough)
    {
        /*
         *   Jestliže je hráč venku z roveru a nahlíží průzory dovnitř vozítka,
         *   potřebujeme zařídit, aby nestál na plošině (tedy na opačné straně
         *   vozítka, než jsou průzory). K tomu slouží actorDirectlyInRoom.
         *   Je to precondition, která přemístí hráče z jakékoliv případné
         *   "nested" lokace přímo do určené vnější místnosti.
         *
         *   Průzory jsou však obousměrné a pokud je hráč uvnitř vozítka, žadné
         *   actorDirectlyInRoom nevynucujeme, protože nechceme, aby vstal z
         *   křesla, když se podívá průzory ven. Vyrobíme proto podmíněnou
         *   precondition. Závorky v definici vlastně znamenají, že hodnotou
         *   preCond je callback funkce spuštěná v okamžiku vyhodnocení.
         */
        preCond =
        (
            gActor.location.getOutermostRoom() == tunnel
            ? [objVisible, new ObjectPreCondition(tunnel, actorDirectlyInRoom)]
            : [objVisible]
        )
        action()
        {
            local otherLocation = nil;

            connectorMaterial = glass;
            if(gActor.isIn(rover))
            {
                if(gRevealed('rover-destination'))
                {
                    otherLocation = tunnel;

                    "Rozhlédl ses průzory po okolí. Vozítko očividně stojí v
                        mnohem širší šachtě, než kterou jsi sem přijel, ale
                        nikde není vidět další průjezdná šachta, kterou bys mohl
                        pokračovat dál. ";
                }
                else if(gRevealed('rover-driving'))
                {
                    "Vozítko se pomalu pohybuje skalním tunelem. Skála v dosahu
                        reflektorů osvětlujících cestu před tebou je většinou
                        nudně šedá a nezajímavá. ";
                }
                else
                {
                    "Průzkumné vozítko parkuje bokem ke stěně nepříliš vysoké
                        chodby. Před vozítkem, kam už světlo z kabiny nedosáhne,
                        mizí ve tmě tunel, který vede do těžebních šachet. Hrubé
                        skalní stěny jsou na několika místech vyztužené
                        konstrukční pěnou, nejsou tu žádné výdřevy, podpěry ani
                        nic podobného. ";
                }
            }
            else
            {
                otherLocation = rover;

                "Nahlédl jsi do kabiny vozítka čelními průzory. Ovládací panel
                    ti zakrývá výhled, takže toho kromě křesel pro posádku moc
                    nevidíš. ";
            }

            if(otherLocation)
                gActor.location.listRemoteContents(otherLocation);

            connectorMaterial = adventium;
        }
    }
    dobjFor(LookIn) asDobjFor(LookThrough)

    isPlural = true
    gcName = 'průzorů, průzorům, průzory, průzorech, průzory'
    gcVocab = 'průzoru/průzorem/průzorů/průzorům/průzorech'
;

/* ------------------------------------------------------------------------ */

/* Pomocná místnost. */
radioRoom: Room;

/* Vysílačka spojuje hlasem velitele s hráčem ve vozítku. */
SenseConnector, Fixture 'vysílačka' 'vysílačka' *3
    "Je to palubní vysílačka pro komunikaci na krátké vzdálenosti, obvykle s
        blízkou základnou, běžná výbava všech průzkumných vozítek. "

    /* Vozítko je spojeno s pomocnou místností, kde je velitel. */
    locationList = [rover, radioRoom]

    /* Propouštíme jen zvuk, hráč může komunikovat, ne vidět velitele. */
    transSensingThru(sense) { return sense == sound ? transparent : opaque; }
    isListedInContents = nil

    gcName = 'vysílačky, vysílačce, vysílačku, vysílačce, vysílačkou'
    gcVocab = 'vysílačky/vysílačce/vysílačku/vysílačkou'
;
