#charset "utf-8"

/*  
 *   ATTACHMENT
 *
 *   A demonstration of TADS 3 Attachable classes.
 *
 *   Handling Attachables is not always straightforward, to say the least, so
 *   this is the most complex of the demo games. If you have not already 
 *   done so, you may want to become familiar with the other demo games first.
 *
 *   Attachables are complicated because there is so many different ways in 
 *   which they could behave that the library classes can only provide a 
 *   general framework which the game author must customize to suit each 
 *   particular case. For example, if I attach a rope to a heavy box lying 
 *   on the floor and then try to walk out of the room still holding the 
 *   rope, a number of different things might happen, including:
 *
 *   (1)    I walk into the next room, still holding the rope, dragging the 
 *   box along behind me. 
 *
 *   (2)    I walk into the next room, still holding the rope, but it 
 *   becomes detached from the box. 
 *
 *   (3)    I walk into the next room, still holding the rope, which is so 
 *   long that it has not yet become taut. 
 *
 *   (4)    I am jerked to a halt by the rope, which has become taut, and 
 *   cannot leave the room while I am holding the rope. 
 *
 *   (5)    I walk into the room, but am forced to let go of the rope in the 
 *   process.
 *
 *   And those are just the possibilitis involving a rope attached to a box; 
 *   many other kind of attachment relationships are possible.
 *
 *   Given the vast number of possibilities, this demonstration game cannot 
 *   hope to cover them all, and will only attempt a small range by way of 
 *   illustration.
 *
 *   Also, because this game is already quite complex enough, and is intended
 *   primarily to illustrate the use of Attachables, this comments in the 
 *   source code will be largely restricted to those parts of the code 
 *   relating to the implementation of Attachable objects.
 *
 *   Finally, this demonstration game makes use of a custom SimpleAttachable 
 *   class which is supplied in the accompanying class. For details of 
 *   SimpleAttachable, see the comments in SimpleAttachable.t.
 */



/*
 *   Copyright (c) 1999, 2002 by Michael J. Roberts.  Permission is
 *   granted to anyone to copy and use this file for any purpose.  
 *   
 *   This is a starter TADS 3 source file.  This is a complete TADS game
 *   that you can compile and run.
 *   
 *   To compile this game in TADS Workbench, open the "Build" menu and
 *   select "Compile for Debugging."  To run the game, after compiling it,
 *   open the "Debug" menu and select "Go."
 *   
 *   This is the "advanced" starter game - it has only the minimum set of
 *   definitions needed for a working game.  If you would like some more
 *   examples, create a new game, and choose the "introductory" version
 *   when asked for the type of starter game to create.  
 */

/* 
 *   Include the main header for the standard TADS 3 adventure library.
 *   Note that this does NOT include the entire source code for the
 *   library; this merely includes some definitions for our use here.  The
 *   main library must be "linked" into the finished program by including
 *   the file "adv3.tl" in the list of modules specified when compiling.
 *   In TADS Workbench, simply include adv3.tl in the "Source Files"
 *   section of the project.
 *   
 *   Also include the US English definitions, since this game is written
 *   in English.  
 */
#include <adv3.h>
#include <cs_cz.h>

/*
 *   Our game credits and version information.  This object isn't required
 *   by the system, but our GameInfo initialization above needs this for
 *   some of its information.
 *   
 *   You'll have to customize some of the text below, as marked: the name
 *   of your game, your byline, and so on.
 */
versionInfo: GameID
    IFID = '55e2c590-f3b5-749e-e06a-3e3555b45329'
    name = 'Attachment (česká verze)'
    byline = 'napsal Eric Eve'
    htmlByline = 'napsal <a href="mailto:eric.eve@hmc.ox.ac.uk">
                  Eric Eve</a>'
    version = '0.2'
    authorEmail = 'Eric Eve <eric.eve@hmc.ox.ac.uk>'
    desc = 'Demonstrace třídy Attachable v TADS 3.'
    htmlDesc = 'Demonstrace třídy Attachable v TADS 3.'
    
    showAbout()
    {
	"Toto je primárně demonstrace třídy Attachable v TADS 3. Hra je
        navržena hlavně pro demonstraci připevňování a připojování a ne už tak
        moc jako zábavná hra, přesto se ale dá vyhrát (nebo prohrát!).\b
        Pokud si chcete hru zahrát a zaseknete se, tak si vzpoměňte, že hra je
        o připevňování a připojování objektů. Většina věcí, které musíte ve hře
        udělat se nějakým působem dotýká propojování nebo rozpojování objektů
        jeden od druhého.<.p>";
    }
;

/*
 *   The "gameMain" object lets us set the initial player character and
 *   control the game's startup procedure.  Every game must define this
 *   object.  For convenience, we inherit from the library's GameMainDef
 *   class, which defines suitable defaults for most of this object's
 *   required methods and properties.  
 */
gameMain: GameMainDef
    /* the initial player character is 'me' */
    initialPlayerChar = me
    
    showIntro()
    {
        "Nebyl jsi zrovna nadšený, že jsi byl tím, kdo dostal rozkaz jít ven a
        opravit komunikační anténu. Ale tahle procházka do vesmíru ti nejspíše
        zachránila život.\b
        Válečná loď Federace (alespoň předpokládáš, že to byla válečná loď
        Federace) se objevila znenadání a bez varování udeřila na drobné
        špionážní plavidlo, které provrtala jediným výstřelem svého laserového
        děla a odhodila tě z tvé vratké pozice. Velice pochybuješ, že by někdo
        uvnitř lodi mohl přežít -- nikdo další na sobě neměl skafandr a
        dekomprese by je zabila téměř okamžitě.\b
        Naštěstí nepřátelská loď zmizela stejně rychle, jako se objevila. Zřejmě
        předpokládala, že její práce je dokončena a ani se neunavovala podívat
        se po přeživších. Vždyť tu také málem žádní nebyli -- zabralo ti přes
        hodinu se dostat zpátky k lodi do přechodové komory, když ti téměř došel
        kyslík v lahvi. Sundal sis helmu a zhluboka ses nadýchnul vzduchu v
        komoře. Ještě sis vzpoměl vypnout baterku na helmě, protože také začala
        pohasínat.\b
        Ale tím tvé trable zdaleka ještě nekončí.\b";
    }
;

/*  
 *   The game takes place entirely aboard a spaceship, so compass directions 
 *   will have no meaning, which also mean we can't refer to the 'north 
 *   wall' the 'east wall' and the like. We therefore override ShipboardRoom 
 *   to disallow movement in compass directions, and provide it with a 
 *   suitable set of room parts suitable for a ship.
 */
modify ShipboardRoom
    north: NoTravelMessage { "Tento směr tu nedává smysl. Na palubě lodi můžeš
        jít na levobok (LB), pravobok (PB), příď (PD) nebo na záď (ZD). "}
    south asExit(north)
    east asExit(north)
    west asExit(north)
    northeast asExit(north)
    northwest asExit(north)
    southeast asExit(north)
    southwest asExit(north)
    
    /* At the start of the game the power is off and all rooms are dark. */
    brightness = (powerSwitch.isOn ? 3 : 0)
    
    /* 
     *   A custom property representing the air pressure in the room (in 
     *   bar). In this game this will be either 0 or 1.
     */
    pressure = 0
    
    roomParts = [deck, defaultCeiling, portWall, starboardWall, aftWall,
        foreWall]
;

/* Custom room parts for a ship. */

portWall: DefaultWall 'lb levá stěna/zeď*stěny zdi trup' 'levá stěna' *3
    gcName = 'levé stěny, levé stěně, levou stěnu, levé stěně, levou stěnou'
    gcVocab = 'levé levou stěny/stěně/stěnu/stěnou/zdi/zdí*stěny stěn stěnám
        stěnách stěnami zdi zdí zdím zdích zdmi trupu trupe trupem'
;

starboardWall: DefaultWall 'pb pravá stěna/zeď*stěny zdi trup' 'pravá stěna' *3
    gcName = 'pravé stěny, pravé stěně, pravou stěnu, pravé stěně, pravou stěnou'
    gcVocab = 'pravé pravou stěny/stěně/stěnu/stěnou/zdi/zdí*stěny stěn stěnám
        stěnách stěnami zdi zdí zdím zdích zdmi trupu trupe trupem'
;

aftWall: DefaultWall
    'zd zadní stěna/zeď/přepážka*stěny zdi přepážky' 'zadní přepážka' *3

    gcName = 'zadní přepážky, zadní přepážce, zadní přepážku, zadní přepážce,
        zadní přepážkou'
    gcVocab = 'zadního zadnímu zadním stěny/stěně/stěnu/stěnou/přepážky/přepážce
        /přepážku/přepážkou/zdi/zdí*stěny stěn stěnám stěnách stěnami zdi zdí
        zdím zdích zdmi přepážky přepážek přepážkám přepážkách přepážkami'
;

foreWall: DefaultWall
    'pd přední stěna/zeď/přepážka*stěny zdi přepážky' 'přední přepážka' *3

    gcName = 'přední přepážky, přední přepážce, přední přepážku, přední přepážce,
        přední přepážkou'
    gcVocab = 'předního přednímu předním stěny/stěně/stěnu/stěnou/přepážky/přepážce
        /přepážku/přepážkou/zdi/zdí*stěny stěn stěnám stěnách stěnami zdi zdí
        zdím zdích zdmi přepážky přepážek přepážkám přepážkách přepážkami'
;

deck: Floor 'podlaha/zem/paluba' 'podlaha' *3
    gcName = 'podlahy, podlaze, podlahu, podlaze, podlahou'
    gcVocab = 'podlahy/podlaze/podlahu/podlaho/podlahou/paluby/palubě/palubu/
	palubo/palubou/země/zemi/zemí*podlahy podlah podlahám podlahách
	podlahami paluby palub palubám palubách palubami země zemí zemím
	zemích zeměmi'
;

/* Define a message that'll be shown just before the first command prompt. */

InitObject
    execute()
    {
        new OneTimePromptDaemon(self, &message);
    }
    
    message = "Osvětlení komory právě zhaslo, to bude další z následků poškození
        zaviněné válečnou lodí Federace. Umělá gravitace přesto zatím, jak se
        zdá, funguje, takže pár záložních baterií musí mít ještě trochu energie
        v zásobě. "
;

/*  
 *   A Custom class for Doors that are opened and closed by an external 
 *   mechanism, not by using OPEN and CLOSE commands.
 */

class IndirectDoor: Door
    dobjFor(Open) { verify { illogical (cannotOpenMsg); } }
    dobjFor(Close) { verify { illogical(cannotCloseMsg); } }
    dobjFor(Lock) { verify { illogical(cannotLockMsg); } }
    dobjFor(LockWith) { verify { illogical(cannotLockMsg); } }
    dobjFor(Unlock) { verify { illogical(cannotUnlockMsg); } }
    dobjFor(UnlockWith) { verify { illogical(cannotUnlockMsg); } }
    cannotOpenMsg = '{Kdoco dobj} se ovládají pákou. '
    cannotCloseMsg = (cannotOpenMsg)
    cannotLockMsg = '{Kdoco dobj} nemají žádný zámek. '
    cannotUnlockMsg = (cannotLockMsg)
;

//------------------------------------------------------------------------------

/* The starting location. */

airlock: ShipboardRoom 'Hlavní přechodová komora'
    "Tahle malá přechodová komora na levoboku lodi je tak akorát pro jednu
    stojící postavu -- dvě by se už nepohodlně mačkaly. Dvojice pák ovládá
    vzduchotěsné dveře a trojice ciferníků měří tlak vzduchu uvnitř a vně
    přechodové komory. "
    starboard = innerDoor
    port = outerDoor
    pressure = 1
    out asExit(starboard)
    roomBeforeAction()
    {
        if(gActionIs(Out) && outerDoor.isOpen)
            replaceAction(Port);    
    }
;

+ redLever: DoorLever, CustomFixture 'červená páka*páky' 'červená páka' *3
    "Je označena <q>venkovní dveře</q>. "
    otherLever = greenLever
    myDoor = outerDoor
    collectiveGroups = [leverGroup]

    gcName = 'červené páky, červené páce, červenou páku, červené páce,
        červenou pákou'
    gcVocab = 'červené červenou páky/páce/páku/pákou*páky pák pákám pákách
        pákami'
;

+ greenLever: DoorLever, CustomFixture 'zelená páka*páky' 'zelená páka' *3
    "je označena <q>vnitřní dveře</q>. "
    otherLever = redLever
    myDoor = innerDoor
    
    dobjFor(Push)
    {
        verify()
        {
            if(hawser.isIn(airlock))
                illogicalNow('Nemůžeš zavřít vnitřní dveře, je dokud vázací lano
                    nataženo skrz ně. ');
            inherited;
        }
    }
    collectiveGroups = [leverGroup]

    gcName = 'zelené páky, zelené páce, zelenou páku, zelené páce, zelenou pákou'
    gcVocab = 'zelené zelenou páky/páce/páku/pákou*pák pákám páky pákách pákami'
;

+ leverGroup: CollectiveGroup, Fixture '*páky' 'páky' *3
    "Je tu červená páka (ovládající venkovní dveře) a zelená páka (ovládající
        vnitřní dveře). "

    gcName = 'pák, pákám, páky, pákách, pákami'
    gcVocab = 'pák/pákám/pákách/pákami'
;

+ portDial: CustomFixture 'levý ciferník*ciferníky' 'levý ciferník' *2
    "Levý ciferník ukazuje tlak vzduchu za levými (venkovními) dveřmi, tj. venku
        mimo loď. Právě ukazuje 0 barů."

    collectiveGroups = [dialGroup]

    gcName = 'levého ciferníku, levému ciferníku, levý ciferník,
        levém ciferníku, levým ciferníkem'
    gcVocab = 'levého levému levém levým ciferníku/ciferníkem*ciferníků
        ciferníkům cifernících'
;

+ centreDial: CustomFixture 'prostřední ciferník*ciferníky' 'prostřední ciferník' *2
    "Prostřední ciferník měří tlak vzduchu v přechodové komoře. Právě ukazuje
        <<airlock.pressure>> bar<<airlock.pressure == 0 ? 'ů' : ''>>. "

    collectiveGroups = [dialGroup]

    gcName = 'prostředního ciferníku, prostřednímu ciferníku, prostřední ciferník,
        prostředním ciferníku, prostředním ciferníkem'
    gcVocab = 'prostředního prostřednímu prostředním ciferníku/ciferníkem*ciferníků
        ciferníkům cifernících'
;

+ starboardDial: CustomFixture 'pravý ciferník*ciferníky' 'pravý ciferník' *2
    "Pravý ciferník měří tlak vzduchu za pravými (vnitřními) dveřmi. Právě
        ukazuje <<storageCompartment.pressure>> bar<<storageCompartment.pressure
        == 0 ? 'ů' : ''>>. "

    collectiveGroups = [dialGroup]

    gcName = 'pravého ciferníku, pravému ciferníku, pravý ciferník,
        pravém ciferníku, pravým ciferníkem'
    gcVocab = 'pravého pravému pravém pravým ciferníku/ciferníkem*ciferníků
        ciferníkům cifernících'

;

+ dialGroup: CollectiveGroup, Fixture '*ciferníky' 'ciferníky' *2
   "Levý ciferník ukazuje tlak vzduchu za levými (venkovními) dveřmi, který je
       momentálně 0 barů. Prostřední ciferník měří tlak vzduchu v přechodové komoře,
       který je momentálně <<airlock.pressure>> bar<<airlock.pressure == 0 ? 'ů'
       : ''>>. Pravý ciferník měří tlak vzduchu za pravými (vnitřními) dveřmi, který
       je momentálně <<storageCompartment.pressure>> bar<<
       storageCompartment.pressure == 0 ? 'ů' : ''>>. "

    gcVocab = '*ciferníků ciferníkům cifernících'
;

+ innerDoor: IndirectDoor 'vnitřní dveře' 'vnitřní dveře' *3
    isPlural = true
    
    gcName = 'vnitřních dveří, vnitřním dveřím, vnitřní dveře, vnitřních dveřích,
        vnitřními dveřmi'
    gcVocab = 'vnitřních vnitřním vnitřními dveří/dveřím/dveřích/dveřmi'
;

+ outerDoor: IndirectDoor, FakeConnector 'venkovní dveře' 'venkovní dveře' *3

    travelDesc = "Jít ven se ti už nechce, pro dnešek máš už dost vesmírných
        procházek. "
    
    destination: OutdoorRoom { pressure = 0 }

    gcName = 'venkovních dveří, venkovním dveřím, venkovní dveře,
        venkovních dveřích, venkovními dveřmi'
    gcVocab = 'venkovních venkovním venkovních venkovními dveří/dveřím/dveřích/
        dveřmi'
;

/*  The Player Character */

+ me: Actor
    gender = 1
;

/*  
 *   SIMPLE ATTACHABLE
 *
 *   SimpleAttachable is a custom class defined in the accompanying file. 
 *
 *   We make the spaceSuit a SimpleAttachable so that an OxygenTank can be 
 *   attached to it.
 *
 *   SimpleAttachable is designed to model an asymmetric attachment, where 
 *   one of the attached objects is the major attachment and all the others 
 *   are its minor attachments (we'd consider a limpet mine to be attached 
 *   to a battleship, not the other way round - the battleship would be the 
 *   major attachment and the mine the minor attachment). If the major 
 *   attachment is moved, its minor attachments move with it. If a minor 
 *   attachment is moved (e.g. by the player character taking it) it becomes 
 *   detached from the major attachment (think of a magnet attached to a 
 *   fridge). 
 *
 *   By default, a SimpleAttachment is designed to be used with other 
 *   SimpleAttachments: both the major attachment and its minor attachments 
 *   should be of class SimpleAttachable.
 */
++ spaceSuit: SimpleAttachable, PlugAttachable, Wearable
    'tmavomodrý vesmírný skafandr/oblek' 'vesmírný skafandr' *2
    "Je tmavomodrý v barvě námořní uniformy. "
    wornBy = me
     
    /* 
     *   The space suit is the major attachable here (any oxygen tank 
     *   attached to it will move round with it). We set it up as the major 
     *   attachments by listing the minor attachments that can be attached 
     *   to it.
     */
    minorAttachmentItems = [emptyTank, fullTank]    
    
    
    /* Prevent the player removing the space suit in a vacuum. */
    dobjFor(Doff)
    {
        check()
        {
            if(gActor.getOutermostRoom.pressure == 0)
                failCheck('To by mělo fatální důsledky, kolem není vůbec žádný
                    vzduch. ');
        }
    }

    gcName = 'vesmírného skafandru, vesmírnému skafandru, vesmírný skafandr,
        vesmírném skafandru, vesmírným skafandrem'
    gcVocab = 'tmavomodrého tmavomodrému tmavomodrém tmavomodrým vesmírného
        vesmírnému vesmírném vesmírným skafandru/skafandrem/obleku/oblekem'
;

/*       
 *   OxygenTank is a custom class defined below; it inherits from 
 *   SimpleAttachable. By locating emptyTank in spaceSuit we ensure that the 
 *   empty tank starts out attached to the space suit at the start of the 
 *   game.
 */

+++ emptyTank : OxygenTank 'prázdná kyslíková lahev' 'prázdná kyslíková lahev' *3
    airLevel = 4

    gcName = 'prázdné kyslíkové lahve, prázdné kyslíkové lahvi, prázdnou
        kyslíkovou lahev, prázdné kyslíkové lahvi, prázdnou kyslíkovou lahví'
    gcVocab = 'prázdné prázdnou kyslíkové kyslíkovou lahve/lahvi/lahví'
;

/*  
 *   The helmet is another SimpleAttachable so we can attach a lamp to it 
 *   (and detach the lamp from it.
 */

++ helmet: SimpleAttachable, Wearable 'běžná vesmírná helma/přilba' 'přilba' *3
    "Je to běžná vesmírná přilba. "
    
    /* The lamp is the only object that can be attached to the helmet. */
    minorAttachmentItems = [lamp]
    
    
    /* 
     *   While the player character is wearing the helmet, he's dependent on 
     *   the air it contains (or can get from the oxygen cylinder) to 
     *   breathe, so we need to model the breathing and air supply. We do 
     *   this with a DAEMON.
     */    
    breathingDaemonID = nil
    
    breathingDaemon()
    {
        /* Reduce the airLevel by one each turn the helmet is worn. */
        airLevel --;
        
        /* 
         *   If there's an oxygen tank attached to the space suit and the 
         *   tank contains enough air, refresh the air supply in the helmet.
         */
        if(myTank && myTank.airLevel > 0)
        {
            local newAir = min(myTank.airLevel, maxLevel - airLevel);
            if(newAir > 1)
                "Do tvé přilby najednou začal proudit čerstvý kyslík. ";
            airLevel += newAir;
            myTank.airLevel -= newAir;
        }
        
        /* 
         *   If the air in the helmet is running out, display a warning 
         *   message.
         */
        switch(airLevel)
        {
            case 4: "Vzduch ve tvé přilbě začíná připadat velmi vyčpělý.
                "; break;
            case 3: "Vzduch ve tvé helmě je sotva dýchatelný. "; break;
            case 2: "Vzduch ve tvé helmě je tak vyčpělý, že začínáš omdlévat. ";
                break;
            case 1: "Můžeš už jen stěží dýchat; brzy ztratíš vědomí. "; break;
            
            /* 
             *   Once there's no air left, the player character dies of 
             *   asphyxiation.
             */
            case 0: "Kvůli chybějícímu vzduchu jsi ztratil vědomí. ";
            finishGameMsg(ftDeath, [finishOptionUndo] );           
        }
        
    }
    
    /* 
     *   A custom property defining which oxygen tank the helmet is getting 
     *   its air supply from. This will be the tank attached to the suit. 
     */
    myTank = (spaceSuit.attachedObjects.valWhich({x: x.ofKind(OxygenTank) }))
    
    /* The amount of air in the helmet (a custom property). */
    airLevel = 5
    
    /* The maximum amount of air the helment can hold. */
    maxLevel = 5
    
    dobjFor(Wear)
    {
        action()
        {            
            inherited;
            /* 
             *   When the helmet is put on, it starts full of air, but we 
             *   then need to start the breathing daemon.
             */
            airLevel = maxLevel;
            breathingDaemonID = new Daemon(self, &breathingDaemon, 1);
        }
        
    }
    
    dobjFor(Doff)
    {
        check()
        {
            /* 
             *   Don't allow the player character to remove the helmet in a 
             *   vacuum.
             */
            if(gActor.getOutermostRoom.pressure == 0)
                failCheck('To by pro tebe znamenalo jistou smrt. V okolí není
                    žádný vzduch. ');
        }
        
        action()
        {
            inherited;
            
            /* When the helmet is removed, stop the breathing daemon. */
            if(breathingDaemonID)
            {
                breathingDaemonID.removeEvent();
                breathingDaemonID = nil;
            }
            
        }
    }

    gcName = 'přilby, přilbě, přilbu, přilbě, přilbou'
    gcVocab = 'běžné běžnou vesmírné vesmírnou helmy/helmě/helmu/helmou/přilby/
        přilbě/přilbu/přilbou'
;

/*  
 *   PLUG ATTACHABLE
 *
 *   PlugAttachable is a mix-in class for use with other Attachable classes 
 *   to make PLUG INTO and UNPLUG FROM behave like ATTACH TO and DETACH FROM.
 *
 *   We make the lamp a PlugAttachable so it can be plugged into a charging 
 *   socket. 
 *
 *   It's also a SimpleAttachable. It can be attached either to the helmet 
 *   or to the charging socket, but that's defined on them.
 */

+++ lamp: PlugAttachable, SimpleAttachable, FueledLightSource, Flashlight
    'baterka' 'baterka' *3
    "Je navržená k připevnění na přilbu, ale dá se oddělat pro nabíjení. Můžeš
        ji také vypínat a zapínat. "    
    
    /*  
     *   Give it a brightness of 1 when off so it's still in scope for the 
     *   player to turn it on.
     */
    brightnessOff = 1
    
    /*  
     *   When the lamp is attached to the socket, it is charged up again. We 
     *   control this with a charging daemon, so that the longer it's plugged
     *   in, the more charge it receives.
     */
    chargeDaemonID = nil
    
    handleAttach(other)
    {
        inherited(other);
        
        /* Start the charging daemon when we're plugged into the socket. */
        if(other == chargingSocket)
        {
            chargeDaemonID = new Daemon(self, &chargeDaemon, 1);
            if(fuelLevel < 10)
            {
                "Jakmile jsi baterku připojil, začala svítit intenzivněji. ";
                fuelLevel = 10;
            }
        }
    }
    
    handleDetach(other)
    {
        inherited(other);
        
        /* Stop the charging daemon when we're removed from the socket. */
        if(other == chargingSocket)
        {
            chargeDaemonID.removeEvent();
            chargeDaemonID = nil;
        }            
    }
    
    chargeDaemon()
    {
        /* Increase the charge in the lamp each turn it's plugged in. */
        if(fuelLevel < maxCharge)
            fuelLevel += 20;
    }
    
    /* burnDaemon() is a library method, standard for a FuelLightSource. */
    
    burnDaemon()
    {       
        /* 
         *   We override burnDaemon to display a series of messages when the 
         *   lamp is about to go out.
         */
        switch(fuelLevel)
        {
            case 5: "Baterka svítí znatelně méně. "; break;
            case 4: "Baterka začíná poblikávat. "; break;
            case 3: "Baterka teď svítí velmi tlumeně. "; break;
            case 2: "Baterka už skoro zhasla. "; break;
            case 1: "Baterka naposledy zablikala. "; break;
        }
        inherited();
    }
    
    maxCharge = 100000
    gcName = 'baterky, baterce, baterku, baterce, baterkou'
    gcVocab = 'baterky/baterce/baterku/baterkou'
;

/*  
 *   DoorLever is custom class we use for the code common to the two levers 
 *   that control the doors in the airlock.
 */
class DoorLever: Lever, CustomFixture
    
    /* A custom property - the other lever (used for various purposes). */
    otherLever = nil
    
    /* The door controlled by this lever. */
    myDoor = nil
    
    dobjFor(Pull)
    {
        verify()
        {
            
            inherited;
            /* 
             *   This lever can't be pulled when the other one is, since we 
             *   shouldn't have both airlock doors open at once.
             */
            gMessageParams(otherLever);
            if(otherLever.isPulled)
                illogicalNow('{Kdoco dobj} je dočasně uzamknutá, dokud
                    je {kdoco otherLever} stáhnutá dolů, aby se předešlo
                    otevření obou dveří přechodové komory najednou. ');
        }
        
        check()
        {
            inherited();
            /* 
             *   Don't let the player open a door if there's a vacuum on the 
             *   other side and the player is not wearing both suit and 
             *   helmet, since that would be fatal.
             */            
            if((helmet.wornBy != gActor || spaceSuit.wornBy != gActor)
               && myDoor.destination.pressure == 0)
                failCheck('Otevření ' + myDoor.gcName + ' by teď mělo pro
                    tebe fatální důsledky. ');
        }
    }
    
    /* Pulling the lever opens the corresponding door; pushing it closes it. */
    
    makePulled(stat)
    {
        inherited(stat);
        myDoor.makeOpen(stat);
        "\^<<myDoor.name>> se <<stat ? 'otevřely' : 'zavřely'>>. ";
        if(stat && airlock.pressure != myDoor.destination.pressure)
        {   
            "Komorou proběhl prudký pohyb vzduchu. ";
            airlock.pressure = myDoor.destination.pressure;
        }
    }
    
    /* Make MOVE LEVER pull it or push is as appropriate. */
    dobjFor(Move) 
    {
        remap = [isPulled ? PushAction : PushAction, self] 
    }
    
    cannotTakeMsg = '{Kdoco dobj} je pevně připevněn{ý dobj} k přepážce. '
;
    
//------------------------------------------------------------------------------

storageCompartment: ShipboardRoom 'Nákladový prostor'
    "Skříňka s vybavením vypadá v pořádku a mrazák s jídlem také. Dveře
    přechodové komory jsou na levobok a ovládají se červeným tlačítkem.
    Strojovna je na zádi a ubikace leží směrem k přídi. Na přepážce je zásuvka
    pro nabíjení a stranou je naviják (používaný k upevnění zásob na palubě
    lodi). " 
    roomFirstDesc = "Tato oblast vypadá výstřelem z nepřátelské lodi víceméně
        nepoškozená, ačkoliv všechno, co nebylo přiděláno, bylo nejspíše
        z lodi vysáto při dekompresi provázející výbuch v jiné části, protože
        přepážky v lodi nejsou utěsněné. <<desc>>"
    aft = engineRoom
    port = airlockDoor
    fore = livingQuarters    
;

+ airlockDoor: IndirectLockable, Door ->innerDoor
    'dveře přechodové komory/dveře' 'dveře přechodové komory' *3
    
    isPlural = true
    gcName = 'dveří přechodové komory, dveřím přechodové komory, dveře
        přechodové komory, dveřích přechodové komory, dveřmi přechodové komory'
    gcVocab = 'dveří dveřím dveřích dveřmi -'
;

+ Button, CustomFixture 'červené tlačítko' 'červené tlačítko' *4
    dobjFor(Push)
    {
        
        action()
        {
            if(hawser.isIn(airlock) && airlockDoor.isOpen)
                failCheck('Nemůžeš zavřít dveře, dokud je skrz ně natažené lano
                    navijáku. ');
            
            airlockDoor.makeOpen(!airlockDoor.isOpen);
            "Dveře přechodové komory se <<airlockDoor.isOpen ? 'otevřely'
                : 'zavřely'>>. ";            
        }
    }

    gcName = 'červeného tlačítka, červenému tlačítku, červené tlačítko,
        červeném tlačítku, červeným tlačítkem'
    gcVocab = 'červeného červenému červeném červeným tlačítka/tlačítku/
        tlačítkem'
;

+ Container, Fixture 'stojan' 'stojan' *2
    gcName = 'stojanu, stojanu, stojan, stojanu, stojanem'
    gcVocab = 'stojanu/stojanem'
;

/* OxygenTank is a custom class defined below. */

++ fullTank: OxygenTank 'plná -' 'plná kyslíková lahev' *3
    initSpecialDesc = "Ve stojanu vedle přechodové komory zbývá poslední
        kyslíková lahev; doufáš, že je stále plná. "
    airLevel = 5000
    gcName = 'plné kyslíkové lahve, plné kyslíkové lahvi, plnou kyslíkovou
        lahev, plné kyslíkové lahvi, plnou kyslíkovou lahví'
    gcVocab = 'plné plnou -'
;

/* 
 *   PLUG ATTACHABLE, SIMPLE ATTACHABLE 
 *
 *   The charging socket is both a PlugAttachable (so we can plug things into
 *   it) and a SimpleAttachable (which means anything attached to it will 
 *   be moved into it, as we'll make it the major attachment).
 */

+ chargingSocket: PlugAttachable, SimpleAttachable, CustomFixture
    'nabíjecí zásuvka' 'nabíjecí zásuvka' *3
    "<< powerSwitch.isOn ? 'S obnovenou dodávkou elektrické energie nebude žádný
        problém získat energii ze zásuvky' : 'Ačkoliv hlavní elektrický okruh
        je vyřazen, zásuvka má záložní akumulátor, který ti snad poskytne
        dostatek energie.' >>."
    
    /* The list of items that can be attached to the charging socket. */
    minorAttachmentItems = [lamp, blackCable]

    gcName = 'nabíjecí zásuvky, nabíjecí zásuvce, nabíjecí zásuvku,
        nabíjecí zásuvce, nabíjecí zásuvkou'
    gcVocab = 'zásuvky/zásuvce/zásuvku/zásuvkou'
;
    
+ equipmentLocker: LockableContainer, CustomFixture
    'skříňka s vybavením/skříňka' 'skříňka s vybavením' *3
    gcName = 'skříňky s vybavením, skříňce s vybavením, skříňku s vybavením,
        skříňce s vybavením, skříňkou s vybavením'
    gcVocab = 'skříňky skříňce skříňku skříňkou skříňky/skříňce/skříňku/skříňkou'
;

++ Decoration 'různé vybavení' 'různé vybavení' *4
   "<<notImportantMsg>>"
    notImportantMsg = (livingQuarters.seen ? 'Nic dalšího, co bys teď
        potřeboval, tu není. ' : 'Jakmile obhlédneš škody, budeš vědět, co
        budeš potřebovat k opravě. ')
    isListed = true
    isListedInContents = true
    isPlural = true

    gcName = 'různého vybavení, různému vybavení, různé vybavení,
        různém vybavení, různým vybavením'
    gcVocab = 'různého/různému/různém/různým vybavením'
;

/*  
 *   CableConnector (NEARBY ATTACHABLE)
 *
 *   A CableConnector is another custom class (defined below). As can be seen
 *   below, CableConnector subclasses from NearbyAttachable. The purpose of 
 *   CableConnectors is to join two lengths of cable together.
 *
 *   The following four items are also defines to be of class PresentLater, 
 *   with a plKey of 'repair', so that they can all be brought into play 
 *   when needed with the singe statement 
 *   PresentLater.makePresentByKey('repair').
 */

++ redConnector: PresentLater, CableConnector 'červený -' 'červený konektor' *2
    plKey = 'repair'

    gcName = 'červeného konektoru, červenému konektoru, červený konektor,
        červeném konektoru, červeným konektorem'
    gcVocab = 'červeného červenému červeném červeným -'
;

++ yellowConnector: PresentLater, CableConnector 'žlutý -' 'žlutý konektor' *2
    plKey = 'repair'

    gcName = 'žlutého konektoru, žlutému konektoru, žlutý konektor,
        žlutém konektoru, žlutým konektorem'
    gcVocab = 'žlutého žlutému žlutém žlutým -'
;

/*  
 *   PLUG ATTACHABLE
 *
 *   The black cable is a PlugAttachable so it can be plugged into things. 
 *   It's also of class Cable, which is defined below. Cable derives from 
 *   NearbyAttachable, which creates a slight complication in that we also 
 *   want to be able to attach the black cable to the charging socket, which 
 *   is a SimpleAttachable. This is dealt with in the canAttachTo() method.
 */

++ blackCable: PresentLater, PlugAttachable, Cable
    'černý kabel' 'kus černého kabelu' *2
    "Je to běžný elektrický kabel, měří několik metrů. "
    plKey = 'repair'
    
    /* 
     *   getNearbyAttachmentLocs() is a standard library method defined on 
     *   NearbyAttachable, from which the blackCable inherits via Cable 
     *   (defined below). 
     *
     *   This method returns a list with three elements. The first element 
     *   is the target location for 'self', and the second is the target 
     *   location for 'other', the object we're attaching to. The third 
     *   element is an integer giving the priority; a higher number means 
     *   higher priority. 
     *
     *   The priority is an arbitrary value that we use to determine which 
     *   of the two objects involved in the attach gets to decide on the 
     *   target locations. We call this method on both of the two objects 
     *   being attached to one another, then we use the target locations 
     *   returned by the object that claims the higher priority. If the two 
     *   priorities are equal, we pick one arbitrarily. 
     *
     *   In this case we want to try to ensure that the blackCable ends up 
     *   in the location of whatever its attached to; the main purpose of 
     *   the black cable is to join two sections of a severed cable running 
     *   through a conduit, so we want the cable to end up in that conduit. 
     *   The default implementation would tend to keep detaching the cable 
     *   and moving back to the player's inventory if the player didn't 
     *   attach everything in exactly the right order, which would be 
     *   needlessly frustrating. 
     *
     */
    
    getNearbyAttachmentLocs(other)
    {
        return [other.location, other.location, 0];
    }
    
    
    /*  
     *   After every ATTACH TO action involving an ElectricalConnector (a 
     *   custom class defined below), check to see whether the action has 
     *   completed an electrical connection between the two sections of 
     *   cable that need to be re-connected.        
     */    
    afterAction()
    {
        if(gActionIs(AttachTo) && gDobj.ofKind(ElectricalConnector) &&
           aftCable.isElectricallyConnectedTo(foreCable))
            "Podařilo se ti dokončit spojení mezi přední a zadní částí
            poškozeného kabelu. ";
    }
    
    /*  
     *   We want it to be possible to attach this black cable to the socket 
     *   and the winch, but the socket and winch are SimpleAttachables, and 
     *   normally we can only attach a SimpleAttachable to another 
     *   SimpleAttachable. We can get round that by invoking 
     *   SimpleAttachable's canAttachTo method here as well as our own to 
     *   test attachability. 
     */    
    canAttachTo(obj)
    {
        return inherited(obj) || delegated SimpleAttachable(obj);
    }

    gcName = 'kusu černého kabelu, kusu černého kabelu, kus černého kabelu,
        kusu černého kabelu, kusem černého kabelu'
    gcVocab = 'černého černému černém černým kabelu/kabelem'
;

++ roll: PresentLater, Thing 'šedivá role/tkanina' 'role šedivé tkaniny' *3
    "Tkanina je šedivá se slabě kovovým vzhledem. Používá se na dočasné opravy
    porušeného trupu. Sice neochrání při srážce s většími objekty nebo výstřely,
    ale alespoň udrží prach venku a odstíní škodlivé kosmické záření. Také
    vytvoří vzduchotěsnou izolaci, takže bude možné obnovit atmosféru v lodi. "
    
    plKey = 'repair'
    dobjFor(Take)
    {
        check()
        {
            if(fabric.moved)
                failCheck('Víc tkaniny teď už nepotřebuješ. ');
        }
        action()
        {
            fabric.moveInto(gActor);
            "Rozroloval jsi tkaninu a uřízl z ní čtverec potřebné velikosti.
                Zbytek jsi zase vrátil do skříňky. ";
        }
    }

    gcName = 'role šedivé tkaniny, roli šedivé tkaniny, roli šedivé tkaniny,
        roli šedivé tkaniny, rolí šedivé tkaniny'
    gcVocab = 'šedivé šedivou roli/rolí/tkaniny/tkanině/tkaninu/tkaninou'
;


+ freezer: LockableContainer, CustomFixture 'velký mrazák' 'mrazák' *2
    "Je to velký mrazák - musí být velký, aby zásoby vydržely posádce na mnoho
        týdnů. "

    gcName = 'mrazáku, mrazáku, mrazák, mrazáku, mrazákem'
    gcVocab = 'velkého velkému velkém velkým mrazáku/mrazákem'
;

++ Decoration 'jídlo' 'jídlo' *4
    "Je tu spousta jídla všech možných chutí, máš-li umřít, tak to rozhodně
        nebude hlady. "
    isListedInContents = true
    isListed = true
    
    notImportantMsg = ( helmet.wornBy == me ? 'Beztak nemůžeš jíst, dokud máš na
        hlavě helmu, takže klidně můžeš nechat jídlo jídlem. ' : 'O jídlo by sis
        měl dělat starosti, až se ti povede dostat loď odsud pryč. ')    
    
    isMassNoun = true

    gcName = 'jídla, jídlu, jídlo, jídle, jídlem'
    gcVocab = 'jídla/jídlu/jídle/jídlem'
;

/*  
 *   PLUG ATTACHABLE     SIMPLE ATTACHABLE
 *
 *   We make the winch a PlugAttachable and the SimpleAttachable so the black
 *   cable can be plugged into it.
 */
+ winch: PlugAttachable, SimpleAttachable, Fixture 'naviják/pouzdro' 'naviják' *2
    "Naviják, který je přidělaný na podlaze, se používá k posunování těžkého
    nákladu po palubě lodi. Je ovládán modrým tlačítkem na svém pouzdru. "
    minorAttachmentItems = [blackCable]

    gcName = 'navijáku, navijáku, naviják, navijáku, navijákem'
    gcVocab = 'navijáku/navijákem/pouzdra/pouzdru/pouzdrem'
;

++ Button, Component 'modré tlačítko*tlačítka' 'modré tlačítko' *4
     dobjFor(Push)
    {
        action()
        {
            /*  
             *   If power hasn't been restored, the only way to get the 
             *   winch to work is to connect it to the charging socket with 
             *   the black cable. For this connection to be made the black 
             *   cable must be attached both to the winch and to the socket.
             */            
            if(!powerSwitch.isOn && !(blackCable.isAttachedTo(winch) &&
                                      blackCable.isAttachedTo(chargingSocket)))
            {
                "Nic se nestalo zřejmě proto, že naviják je bez energie. ";
                return;
            }
            
            if(hawser.isIn(storageCompartment))
                "Naviják krátce zakňučel a lano se párkrát zaškubalo, ale protože
                je naviják úplně navinutý, nic dalšího se nestalo. ";
            else if(hawser.isAttachedTo(debris))
            {
                "Naviják kňčel a lano se napnulo. Tón navijáku se trochu zvýšil
                námahou při pokusu přitáhnout lano. Několik dalších okamžiků se
                nic nedělo, ale pak se z předu ozval hlasitý škrábavý hluk a
                lano začalo pomalu přitahovat masu spečených trosek do
                nákladového prostoru. ";
                debris.moveInto(storageCompartment);
            }
            else
            {
                "Naviják se probudil k životu a přitahuje lano celou cestu zpátky
                do nákladového prostoru. ";
                hawser.moveInto(storageCompartment);
            }
        }
    }

    gcName = 'modrého tlačítka, modrému tlačítku, modré tlačítko, modrém
        tlačítku, modrým tlačítkem'
    gcVocab = 'modrého modrému modrém modrým tlačítka/tlačítku/tlačítkem'
;

/*  
 *   SIMPLE ATTACHABLE 
 *
 *   We make the hawser a SimpleAttachable so that (a) we can attach it to 
 *   things (in this game, only the debris) and (b) so it moves with whatever
 *   its attached to).
 */
+ hawser: SimpleAttachable, Thing 'lano/konec' 'lano' *4
    "<<specialDesc>>"
    
    /* Vary the description of the hawser depending on where it is. */
    specialDesc()
    {
        switch(getOutermostRoom)
        {
            case storageCompartment: "Krátký kus lana se houpe na navijáku. ";
            break;
        case bridge:
            case livingQuarters: "Lano vede směrem na záď. "; break;
            case engineRoom: "Lano vede směrem na příď. "; break;
        case airlock:
            case cabin: "Lano vede dveřmi směrem na levobok. "; break;
        }
    }
    specialDescBeforeContents = nil
    specialDescListingOrder = 100
    getFacets = [proxyHawser1, proxyHawser2]

    gcName = 'lana, lanu, lano, lanu, lanem'
    gcVocab = 'lana/lanu/lanem/konce/konci/koncem'
;

/* 
 *   If the hawser object is not in the storage compartment, there must be a 
 *   length of hawser running from the the winch to wherever the other end 
 *   of the hawser is. In that case we need a proxy object to describe the 
 *   length of hawser that's visible inside the storage compartment.
 *   ProxyHawser is a custom class defined below.
 */
+ proxyHawser1: ProxyHawser 'lano' 'lano' *4
    "Lano z navijáku vede <<cableDir()>>. "
    
    /* 
     *   We want this length of hawser to be visible only when the real 
     *   hawser object is elsewhere.
     */
    discovered = (!hawser.isIn(storageCompartment))
      
    /* 
     *   Describe which way the hawser runs depending on where the other end 
     *   of the hawser is.
     */
    cableDir()
    {
        switch(hawser.getOutermostRoom)
        {
            case engineRoom: "za záď"; break;
            case airlock: "na levobok do přechodové komory"; break;
            default: "na příď"; break;
        }
    }
    
    /* 
     *   The other objects that can represents sections of the hawser are 
     *   facets of this object.
     */
    getFacets = [hawser, proxyHawser2]
    
    gcName = 'lana, lanu, lano, lanu, lanem'
    gcVocab = 'lana/lanu/lanem'
;

//------------------------------------------------------------------------------
/*  
 *   Define our custom CABLE CONNECTOR class.
 *
 *   This descends from our custom ElecticalConnector class (defined below), 
 *   which in turn descends from NearbyAttachable.
 */
class CableConnector: ElectricalConnector
    'kabelový plastový prstenec/konektor*konektory' 'kabelový konektor' *2
    "Vzhledem vypadá jako plastový prstenec. Má za úkol propojit dva kusy
        kabelu. "
    
    /* 
     *   The getNearbyAttachmentLocs() method is defined in the library for 
     *   NearbyAttachable. It controls where a NearbyAttachable ends up when 
     *   its attached to something. For a full description, see the comment 
     *   in blackCable above. 
     *
     *   In this case we need the two cable connectors to end up connected 
     *   to the two segments of cable in the conduit, so if what we're 
     *   connecting to is in the conduit, that's where we want everything to 
     *   end up. We define getNearbyAttachmentLocs accordingly. 
     */
    getNearbyAttachmentLocs(other)
    {
        if (other.isIn(conduit))
        {
            /* the other is where we want it, so use its location */
            return [other.location, other.location, 5];
        }
        else
        {
            /* 
             *   the other can be moved, so use our own location.
             */
            return [location, location, 0];
        }
    }    

    gcVocab = 'kabelového kabelovému kabelovém kabelovým plastového plastovému
        plastovém plastovým prstence/prstenci/prstencem/konektoru/konektorem*
        prstence prstenců prstencům prstencích prstenci konektory konektorů
        konektorům konektorech'
;

/* 
 *   Definition of the custom CABLE class.
 *
 *   Cable derives from our custom ElectricalConnector class (defined 
 *   immediately below). The only customizetion required on this class is to 
 *   define what a Cable can connect to: Cables can connect to 
 *   CableConnectors.
 */
class Cable: ElectricalConnector
    canAttachTo(obj)
    {
        return obj.ofKind(CableConnector);                        
    }   
;

/*   
 *   ELECTRICAL CONNECTOR     NEARBY ATTACHABLE
 *
 *   Our custom ElectricalConnector class derives from the library's 
 *   NearbyAttachable class. A NearbyAttachable is an Attachable that 
 *   enforces the condition that the attached objects must be in a 
 *   particular location. By default this is the location that one of the 
 *   objects is already in, but this can be customised by overriding 
 *   getNearbyAttachmentLocs(). 
 */
class ElectricalConnector: NearbyAttachable, PlugAttachable, Thing
    
    /* 
     *   isElectricallyConnectedTo() is a custom method to test whether an 
     *   electrical connection exists between two ElectricalConnectors. An 
     *   electrical connection exists if the two ElectricalConnectors are 
     *   directly or indirectly attached; they're indirectly attached if 
     *   there's a chain of attached objects between them.    
     */    
    isElectricallyConnectedTo(obj)
    {
        local vec = new Vector(10, [self]);
        local i = 0, cur;
               
        while(i < vec.length)           
        {
            cur = vec[++i];
            vec.appendUnique(cur.attachedObjects);
            if(vec.indexOf(obj))
                return true;                       
        } 
        
        return nil;
    }
    
    /* 
     *   movedWhileAttached() is a library method defined on the Attachable 
     *   class. It's overridden on the NearbyAttachable class to detach 
     *   objects if one of them is moved while they're attached to each 
     *   other. This could be irritating in this game: if, for example the 
     *   player first attached the cable connectors to the black cable and 
     *   then tried to attach the cable connectors to the cable ends in the 
     *   counduit, the cable connectors would become detached from the black 
     *   cable. In this case we'd rather the cable connectors remained 
     *   attached to the black cable and the black cable moved into the 
     *   conduit along with the cable connectors, so we override 
     *   moveWhileAttached() accordingly. 
     */    
    moveWhileAttached(movedObj, newCont)
    {
        /* 
         *   If anything is being moved into the conduit, move its 
         *   attachments there as well, because that's where we want them 
         *   all to end up.
         */
        if(newCont == conduit)
        {
            if(movedObj != self)
                /* Don't trigger any more movement notifications! */
                baseMoveInto(newCont);
        }
        else
            inherited(movedObj, newCont);
        
    }
    
    
;

//------------------------------------------------------------------------------
engineRoom: ShipboardRoom 'Strojovna'
    "Strojovna vypadá nepoškozená. Po krátké obhlídce přístrojů to zatím vypadá,
    že hlavní motor je v pořádku. <<controls.desc>> "
    fore = storageCompartment
    out asExit(fore)
;

+ controls: Decoration
    'spousta přístrojů/přístroje/ovladčů/ovladače' 'přístroje' *2
    "Je tu spousta přístrojů a ovladačů, ale \v<<controls.notImportantMsg>>"
    isPlural = true
    notImportantMsg = 'Jediné, co tě teď zajímá je velký červený vypínač
        ovládající napájení lodi, žlutá páka ovládající přívod vzduchu a
        tlakoměr ukazující tlak vzduchu unvitř lodi.'

    gcName = 'přístrojů, přístrojům, přístroje, přístrojích, přístroji'
    gcVocab = 'spousty spoustě spoustu spoustou přístroji/přístrojí'
;

+ powerSwitch: Switch, CustomFixture
    'velký červený vypínač/jistič' 'velký červený vypínač' *2
    "Vypínač je teď <<onDesc>>. "
    makeOn(stat)
    {
        if(stat)
        {
            if(!aftCable.isElectricallyConnectedTo(foreCable))
                failCheck('Vypínač skočil zpátky do polohy vypnuto. Jistič
                    z důvodu bezpečnosti nezůstane zapnutý, pokud je někde v
                    elektrickém okruhu zkrat. ');
            "Světla po celé lodi se rozsvítila. "; 
        }
        else
            "Světla v lodi se zase vypnula. ";
    
        inherited(stat);
    }

    gcName = 'velkého červeného vypínače, velkému červenému vypínači, velký
        červený vypínač, velkém červeném vypínači, velkým červeným vypínačem'
    gcVocab = 'velkého velkému velkém velkým červeného červenému červeném
        červeným vypínače/vypínačem/jističe/jističem'
;

+ airLever: Lever, CustomFixture 'žlutá páka' 'žlutá páka' *3
    dobjFor(Pull)
    {
        check()
        {
            if(!lqWall.repaired)
                failCheck('Pokud zapneš dodávku vzduchu bez opravení trupu lodi,
                    tak jednoduše vyplýtváš všechen vzduch - vylétne dírou v lodi
                    stejně rychle, jak se bude pokoušet natlakovat loď. ');
        }
    }
    
    makePulled(stat)
    {
        inherited(stat);
        if(stat && location.pressure == 0)
        {
            "Po celé lodi syčí vzduch z ventilů a ručička tlakoměru se začíná
                zvedat. ";
            forEachInstance(ShipboardRoom, { loc: loc.pressure = 1 } );
        }
    }

    gcName = 'žluté páky, žluté páce, žlutou páku, žluté páce, žlutou pákou'
    gcVocab = 'žluté žlutou páky/páce/páku/pákou'
;

+ Fixture 'tlakoměr/ručička' 'tlakoměr' *2
    "Ručička tlakoměru měří tlak v lodi a momentálně ukazuje <<location.pressure>>
        bar<<location.pressure ? '' : 'ů'>>. "

    gcName = 'tlakoměru, tlakoměru, tlakoměr, tlakoměru, tlakoměrem'
    gcVocab = 'tlakoměru/tlakoměrem/ručičky/ručičce/ručičku/ručičkou'
;


//------------------------------------------------------------------------------
/*  
 *   ATTACHABLE
 *
 *   Attachable is the base class for all the other Attachable classes we 
 *   have seen. It is a mix-in class which must be combined with a 
 *   Thing-derived class or object. 
 *
 *   Here we use it to define a wall to which something (namely, a piece of 
 *   fabric) can be attached.
 */ 
lqWall: Attachable, starboardWall
    desc = "<<repaired ? 'Stěna na pravoboku teď vypadá vzduchotěsná' : 'Ve
        stěně je proražená díra'>>. "
    
    /*  
     *   isMajorItemFor() is a standard library method. If it returns true 
     *   for obj, then obj is described as being attached to us, rather than 
     *   vice versa. We want to see messages like 'a piece of fabric is 
     *   attached to the starboard wall'; 'the starboard wall is attached to 
     *   a piece of fabric' would look wrong. We therefore override this 
     *   method accordingly.
     */
    isMajorItemFor(obj) { return obj == fabric; }
    
    /*  
     *   repaired is a custom property to indicate when the wall has been 
     *   repaired by attaching the piece of fabric.
     */
    repaired = (isAttachedTo(fabric))
;

+ gapingHole: Component 'proražená díra' 'proražená díra' *3
    "Je přibližně kruhová a má asi metr v průměru. "
    
    /* 
     *   Make ATTACH FABRIC TO HOLE equivalent to ATTACH FABRIC TO STARBOARD 
     *   WALL.
     */
    iobjFor(AttachTo) remapTo(AttachTo, DirectObject, lqWall)

    gcName = 'proražené díry, proražené díře, proraženou díru, proražené díře,
        proraženou dírou'
    gcVocab = 'proražené proraženou díry/díře/díru/dírou'
;



livingQuarters: ShipboardRoom 'Ubikace'
    "Tato oblast očividně dostala zásah laserovým dělem. I kdyby to nebylo na
    první pohled poznat podle velké díry v trupu na místě, kde by měla být
    kajuta na pravobok, je to jasné podle trosek, které bývaly obytnou místností.
    Jedna kajuta na levoboku by ale stále mohla být obytná. Skladový prostor leží
    směrem na záď a směrem na příď je můstek. "
    
    roomFirstDesc = "<<desc>>\bNejsou tu známky po žádném z ostatních členů
        posádky. Nejspíš byli všichni vycucnuti ven z lodi dírou v trupu při
        náhlé dekompresi.
        <<PresentLater.makePresentByKey('repair')>>"
    aft = storageCompartment
    port = cabinDoor
    fore = bridge
    roomParts = static inherited - starboardWall + lqWall
;

/*  
 *   PERMANENT ATTACHABLE     LOCKABLE
 *
 *   A PermanentAttachment, as its name suggests, is something that's 
 *   described as being permanently attached to something else. As an 
 *   example we'll attach a sing to this door, so we'll make this door a 
 *   PermanentAttachment too.
 */

+ cabinDoor: Lockable, PermanentAttachment, Door
    'dveře kabiny/dveře' 'dveře kabiny' *3
    "Cedule je připevněná ke dveřím. "
    isPlural = true
    vocalize = ['k']
    
    /*  
     *   Normally making both sides of a Door a Lockable (as opposed to 
     *   LockableWithKey or IndirectLockable) doesn't achieve much, since 
     *   the door is simply unlocked with an implicit action when it's 
     *   opened. In this case, however, we can achieve a significant effect 
     *   by using a check condition to restrict unlocking the door - the 
     *   door won't unlock until the ship has been pressurized.
     */         
    dobjFor(Unlock)
    {
        check() 
        {
            if(location.pressure == 0)
            {
                failCheck('Dveře kabiny se nedají odemknout. Bude to jediná
                    tlaková uzávěra, která drží a v tom případě se ti nepovede
                    dveře otevřít, dokud neobnovíš tlak v lodi. Také se mohlo
                    stát, že pokud je někdo v kajutě, tak tlakový uzávěr je ta
                    jediná věc, která ho drží při životě. ');
            }
        }
    }
    
    
    /* The door can't be closed if there's a hawser running through it. */
    dobjFor(Close)
    {
        verify()
        {
            if(hawser.isIn(cabin))
                illogicalNow('Nemůžeš zavřít dveře, když je skrz ně natažené
                    lano navijáku. ');
            inherited;
        }
    }
    
    /* 
     *   The message to display if someone tries to detach the sign from the 
     *   door.
     */
    cannotDetachMsgFor(obj)
    {        
        return 'Cedule je pevně přidělaná ke dveřím, nemůžeš ji sundat. ';
    }

    gcName = 'dveří kabiny, dveřím kabiny, dveře kabiny, dveřích kabiny,
        dveřmi kabiny'
    gcVocab = 'dveře dveří dveřím dveřích dveřmi dveře/dveří/dveřím/dveřích/
        dveřmi/kabiny'
;

/*  
 *   PERMANENT ATTACHMENT CHILD
 *
 *   A PermanentAttachmentChild is something that's permanently attached to 
 *   its parent. By locating the PermanentAttachmentChild in its parent 
 *   object we ensure that the library automatically makes them attached to 
 *   each other.
 *
 *   This keeps the sign attached to the door (in the sense that the library 
 *   will consider them as attached throughout the game), but it doesn't stop
 *   the player from taking the sign, and if we make the sign simply a 
 *   Thing it will be listed as being 'in' the Door. So the sign needs also 
 *   be be some NonPortable class, such as Component.
 *
 *   We could have achieved the same effect by simply making the sign a 
 *   Component and overriding its cannotDetachMsg. About the only gain from 
 *   using PermanentAttachmentChild is that DETACH DOOR FROM SIGN is handled 
 *   the same as DETACH SIGN FROM DOOR.
 */
     


++ PermanentAttachmentChild, Component 'cedule' 'cedule' *3
    "Na ceduli je napsáno <q>KAPITÁN</q>. "
    cannotTakeComponentMsg(obj)
    {
        gMessageParams(obj);
        return 'Nemůžeš vzít ceduli, je připevněná {k/ke obj} {komučemu obj}. ';
    }
    
    gcName = 'cedule, ceduli, ceduli, ceduli, cedulí'
    gcVocab = 'ceduli/cedulí'
;


+ conduit: Container, CustomFixture 'kabelový žlab' 'kabelový žlab' *2
    
    isInInitState = (!aftCable.isElectricallyConnectedTo(foreCable))
    initSpecialDesc = "Výbuch mimo jiné odkryl hlavní kabelový žlab, který se
        táhne podlahou, a je vidět, že část hlavního napájecího kabelu je
        kompletně upálená a chybí. <<isOpen ? '' : 'Naneštěstí to vypadá, že
        trosky z výbuchu ztěžují přístup k žlabu. '>>"
    
    
    /*  
     *   Customise the way our contents are listed, so that when the cables 
     *   are all joined up our listing says so.
     */
    contentsLister: thingContentsLister 
    {
        showListSuffixWide(itemCount, pov, parent, selector)
        { lexicalParent.showListSuffixWide(); }              
    }
    
    descContentsLister:  thingDescContentsLister
    {
        showListSuffixWide(itemCount, pov, parent, selector)
        { lexicalParent.showListSuffixWide(); }
    }

    showListSuffixWide()
    {
        "<< isInInitState ? '' : ' a všechny kabely jsou nyní zapojené
            dohromady'>>. ";
    }
    
    /* 
     *   The conduit starts off covered with debris that makes it difficult 
     *   to get at, although we can see what's inside. To simulate that we 
     *   make it start off as a closed Container (so the Player Character 
     *   can't reach inside) made of glass (so the Player Character can see 
     *   inside). Moving the debris automatically 'opens' the container so 
     *   that its contents become fully accessible.
     */    
    material = glass
    isOpen = (debris.moved)
    
    cannotMoveThroughMsg = 'Trosky zakrývající žlab ti zamezují k němu
        přístup. '

    gcName = 'kabelového žlabu, kabelovému žlabu, kabelový žlab, kabelovém
        žlabu, kabelovým žlabem'
    gcVocab = 'kabelového kabelovému kabelovém kabelovým žlabu/žlabem'
;

/* 
 *   FixedCable is a custom class defined below (inheriting from 
 *   NearbyAttachable). Since the fore and aft sections of the cable are 
 *   meant to be a couple of metres apart, the same CableConnector can't be 
 *   simultaneously attached to both the foreCable and the aftCable.
 */
++ aftCable: FixedCable 'zadní -' 'zadní část kabelu' *3
    "Je to krátký kousek kabelu vedoucího ze zadní části kabelového žlabu a
        končí asi dva metry před přední části, protože prostředek kabelu odhořel. "

    canAttachTo(obj)
    {
        return inherited(obj) && !obj.isAttachedTo(foreCable);
    }
    
    gcVocab = 'zadního zadnímu zadním -'

;

++ foreCable: FixedCable 'přední -' 'přední část kabelu' *3
    "Je to krátký kousek kabelu vedoucího z přední části kabelového žlabu a
        končí asi dva metry před zadní částí, protože prostředek kabelu odhořel. "

    canAttachTo(obj)
    {
        return inherited(obj) && !obj.isAttachedTo(aftCable);
    }
    
    gcVocab = 'předního přednímu předním -'
;

/*  
 *   SIMPLE ATTACHABLE 
 *
 *   The debris is a SimpleAttachable so we can attach the hawser to it to 
 *   drag it out of the way using the winch. We also make it of class Heavy 
 *   so we can't move it by hand.
 */

+ debris: SimpleAttachable, Heavy 'masa hromada spečené kovové trosky/masa/hromada'
    'hromada trosek' *3
    "Je to masa dohromady spečených kovových trosek výstřelem laseru, který
    provrtal loď. Je to asi hlavně zbytek jídelního stolu plus kusy kajuty
    z pravoboku. "
    
    /* Allow the hawser to be attached to the debris. */
    minorAttachmentItems = [hawser]
    
    specialDesc = "Masa spečených kovových trosek je rozptýlena po palubě. "

    gcName = 'hromady trosek, hromadě trosek, hromadu trosek, hromadě trosek,
        hromadou trosek'
    gcVocab = 'masy mase masu masou hromady hromadě hromadu hromadou spečených
        spečeným spečenými kovových kovovým kovovými masy/mase/masu/masou/
        hromady/hromadě/hromadu/hromadou/trosek/troskám/troskách/troskami'
;

/* 
 *   As with proxyHawser1 above, we need an object to represent the section 
 *   of hawser running through the living quarters if the end of the hawser 
 *   has been taken beyond the living quarters to either the bridge or the 
 *   cabin. ProxyHawser is a custom class defined below.
 */
+ proxyHawser2: ProxyHawser
    desc = "Lano vede na záď do nákladového prostoru a <<cableDir()>>. "
    discovered = (hawser.isIn(bridge) || hawser.isIn(cabin))
    
    cableDir()
    {
        switch(hawser.getOutermostRoom)
        {
            case bridge: "na příď na můstek"; break;
            case cabin: "na levobok do kabiny"; break;
            default: "na příď"; break;
        }
    }
    getFacets = [hawser, proxyHawser1]
    
    
;

/*  
 *   Define the custom ProxyHawser class to represent lengths of hawser 
 *   passing through a location when the free end of the hawser is elsewhere.
 */
class ProxyHawser: Hidden, CustomFixture 'lano navijáku/lano' 'lano' *4
    specialDescBeforeContents = nil
    specialDesc = (desc)
    
    cannotTakeMsg = 'Brát prostředek lana není moc užitečné. '
    dobjFor(Pull)
    {
        verify() {}
        action()
        {
            if(hawser.isAttachedTo(debris))
               failCheck('Nemůžeš tahat lano navijáku jen tak rukama, náklad na
                   jeho konci je příliš těžký. ');
            else
            {
                hawser.moveInto(gActor.location);
                "Stále jsi tahal za lano navijáku, až se ukázal jeho volný konec. ";
            }   
        }
    }

    gcName = 'lana, lanu, lano, lanu, lanem'
    gcVocab = 'lana lanu lanem lana/lanu/lanem'
;

/*  
 *   Define the custom FixedCable class, used to define the two ends of the 
 *   cable left in the conduit.
 */
class FixedCable : Cable, CustomFixture 'konec kus část kabel*kabely*konce*kusy'
    
    explainCannotAttachTo (obj)
    {  
        "Přední a zadní kus kabelu jsou příliš daleko od sebe, než aby se
        <<obj.name>> dal{a obj} připojit k oboum koncům najednou. ";
    }
    isListedInContents = true
    isListed = true

    gcVocab = 'konce konci koncem kusu kusem části částí kabelu/kabelem'
;

//------------------------------------------------------------------------------

cabin: ShipboardRoom 'Kajuta'
    "Vypadá to, že kajuta přežila bez vážného poškození. Je tu palanda, na
    které můžeš spát, pokud tedy budeš ještě mít čas spát, s nočním stolkem
    umístěným vedle ní. "
    starboard = cabinDoorInside
    out asExit(starboard)
;

+ cabinDoorInside: Lockable, Door -> cabinDoor
    'dveře kabiny/dveře' 'dveře kabiny' *3
    isPlural = true
    /* We can\'t close the door if the hawser is running through it. */
    dobjFor(Close)
    {
        verify()
        {
            if(hawser.isIn(cabin))
                illogicalNow('Nemůžeš zavřít dveře, když je skrz ně natažené
                lano. ');
            inherited;
        }
    }

    gcName = 'dveří kabiny, dveřím kabiny, dveře kabiny, dveřích kabiny,
        dveřmi kabiny'
    gcVocab = 'dveře dveří dveřím dveřích dveřmi dveře/dveří/dveřím/dveřích/
        dveřmi/kabiny'
;

+ Bed, Fixture 'postel/palanda' 'palanda' *3
    gcName = 'palandy, palandě, palandu, palandě, palandou'
    gcVocab = 'postele/posteli/postelí/palandy/palandě/palandu/palandou'
;

+ ComplexContainer, Fixture 'malý kovový noční stolek' 'noční stolek' *2
    "Je to malý železný noční stolek s dvířky. "
    subSurface: ComplexComponent, Surface {}
    subContainer: ComplexComponent, LockableContainer { }

    gcName = 'nočního stolku, nočnímu stolku, noční stolek, nočním stolku,
        nočním stolkem'
    gcVocab = 'malého malému malém malým kovového kovovému kovovém kovovým
        nočního nočnímu nočním stolku/stolkem'
;

++ ContainerDoor 'dvířka nočního stolku/dvířka' 'dvířka nočního stolku' *3
    isPlural = true

    gcName = 'dvířek nočního stolku, dvířkám nočního stolku, dvířka nočního
        stolku, dvířkách nočního stolku, dvířky nočního stolku'
    gcVocab = 'dvířek dvířkám dvířka dvířkách dvířky nočního dvířek/dvířkám/
        dvířka/dvířkách/dvířky/stolu'
;

/* 
 *   SIMPLE ATTACHMENT
 *
 *   This one really is simple. 
 */

++ securityCard: SimpleAttachable, Thing
    'bílá bezpečnostní fialové karta/značení' 'bezpečnostní karta' *3
    "Je čistě bílá, má zhruba 8x4 cm a fialové značení. "
    subLocation = &subContainer

    gcName = 'bezpečnostní karty, bezpečnostní kartě, bezpečnostní kartu,
        bezpečnostní kartě, bezpečnostní kartou'
    gcVocab = 'bílé bílou bezpečnostního bezpečnostnímu bezpečnostním fialového
        fialovému fialovém fialovým karty/kartě/kartu/kartou/značením'
;

//------------------------------------------------------------------------------

/* 
 *   PERMANENT ATTACHMENT
 *
 *   We make the bridgeFloor a PermanentAttachment so we can attach the 
 *   chair to it.
 */
bridgeFloor: PermanentAttachment, deck
    isMajorItemFor(obj) { return true; }
    attachedObjects = [bridgeChair]
;


bridge: ShipboardRoom 'Můstek'
    "<q>Můstek</q> je asi trochu eufemismus pro malou řídicí kabinku, ale
    funkčně to můstek je, protože se odtud řídí loď. Jedna sedačka je pevně
    přidělaná k podlaze a proti ní přístrojová deska. Bývala tu ještě jedna
    sedačka pro osobu, která vyhodnocovala zpravodajské skeny, ale musela být
    vycucnuta dekompresí, protože cesta na záď je otevřená. "
    aft = livingQuarters
    out asExit(aft)
    roomParts = static inherited - deck + bridgeFloor
;

/* 
 *   PERMANENT ATTACHMENT 
 *
 *   We make the bridgeChair a PermanentAttachment because it's described as 
 *   attached to the deck. Note that in this case we have to set up the 
 *   attachment relationship by hand by defining the attachedObjects on both 
 *   the bridgeChair and the bridgeFloor. 
 */

+ bridgeChair: PermanentAttachment, Chair, CustomFixture
    'velká pilotova sedačka' 'pilotova sedačka' *3
    "Je to velká sedačka, umístěná proti přístrojové desce, kterou se loď pilotuje. "
    attachedObjects = [bridgeFloor]
   
    cannotTakeMsg = 'Sedačka je pevně přidělaná k podlaze, což je důvod, proč
        tu stále je i přes dekompresi. '
    
    /* baseCannotDetachMsg is a library property. */
    baseCannotDetachMsg = 'To by šlo stěží udělat, sedačka vypadá přivařeně k
        palubě. Ale to je jedno, stejně ji nikam nepotřebujeě přemisťovat. '
       
    gcName = 'pilotovy sedačky, pilotově sedačce, pilotovu sedačku,
        pilotově sedačce, pilotovou sedačkou'
    gcVocab = 'velké velkou pilotovy pilotově pilotovu pilotovo pilotovou
        sedačky/sedačce/sedačku/sedačkou'
;

+ Decoration 'mnohabarevné přístrojová přístroje/displaye/obrazovky/tlačítka/
    vypínače/knoflíky/ukazatele/deska' 'přístrojová deska' *3
    "Je tu spousta mnohabarevných displayů, obrazovek, tlačítek, vypínačů,
    knoflíků a ukazatelů, žádný ale není aktivní. Všechny se zapínají zmáčknutím
    zeleného tlačítka přímo uprostřed přístrojové desky<<conditions()>>."
    
    conditions()
    {
        local cardOK = (securityCard.isAttachedTo(cardReader));
        if(powerSwitch.isOn && cardOK)
            return;
        
        ", ale nic se nezapne, dokud ";
        if(!powerSwitch.isOn)
            "nebude hlavní napájecí okruh zapnut <<cardOK ? '' : 'a '>>";
        if(!cardOK)
            "bezpečnostní karta nebude připevněna do čtečky";         
        
    }
    
    notImportantMsg = 'Teď by tě už mělo zajímat jen zelené tlačítko a čtečka
    karet. '
    canMatchThem = true

    gcName = 'přístrojové desky, přístrojové desce, přístrojovou desku,
        přístrojové desce, přístrojovou deskou'
    gcVocab = 'mnohabarevného mnohabarevnému mnohabarevném mnohabarevným
        přístrojové přístrojovou displayi/tlačítky/tlačítku/vypínače/vypínačem/
        ukazateli/ukazatelí/desky/desce/desku/deskou'
;

+ greenButton: Button, Fixture 'zelené tlačítko' 'zelené tlačítko' *4
    dobjFor(Push)
    {
        action()
        {
            if(!powerSwitch.isOn)
                "Nic se nestalo, nefunguje napájení. ";
            else if(!securityCard.isAttachedTo(cardReader))
                "Nic se nestalo, tenhle typ lodi nereaguje na ovládání, dokud
                nezastrčíš bezpečnostní kartu do čtečky. ";
            else
            {
                "Přístroje se probudily k životu a ukazují, že je loď připravená
                k letu. Je nepravděpodobné, že by se válečná loď Federace, která
                tě prve napadla, ještě vrátila na obhlídku. Ale nemá cenu zde
                zůstávat, takže jsi nastavil kurz k nejbližší planetě Impéria a
                vrátil se do bezpečí.\b";
                finishGameMsg(ftVictory, [finishOptionUndo]);
            }
        }
    }

    gcName = 'zeleného tlačítka, zelenému tlačítku, zelené tlačítko, zeleném
	tlačítku, zeleným tlačítkem'
    gcVocab = 'zeleného zelenému zeleném zeleným tlačítka/tlačítku/tlačítkem'
;

/* 
 *   SIMPLE ATTACHMENT 
 *
 *   Another SimpleAttachment that's actuall simple. We just define the 
 *   minorAttachementItems property to contain the list of things that can be
 *   attached to it: in this case, just the securityCard.
 */
+ cardReader: SimpleAttachable, Fixture 'čtečka karet/čtečka' 'čtečka karet' *3
    "Je velká zhruba 8 krát 4 cm. "
    minorAttachmentItems = [securityCard]

    gcName = 'čtečky karet, čtečce karet, čtečku karet, čtečce karet, čtečkou karet'
    gcVocab = 'čtečky čtečce čtečku čtečkou karet/čtečky/čtečce/čtečku/čtečkou'
;

//==============================================================================
/*  
 *   Define the custom OxygenTank class. 
 *
 *   It's another SIMPLE ATTACHABLE, but it's made a bit more complicated by 
 *   the fact that only one OxygenTank can be attached to the space suit at 
 *   a time.
 */
class OxygenTank: SimpleAttachable, PlugAttachable, Thing
    vocabWords = 'stříbrná kovová vzduchová kyslíková lahev/bomba*lahve/bomby'
     
    /* 
     *   If there's another OxygenCylinder attached to the space suit when 
     *   the player tries to attach this one, insist that the other one is 
     *   detached first. 
     */
    handleAttach(other)
    {
        if(other == spaceSuit && other.attachedObjects.indexWhich({x:
            x.ofKind(OxygenTank) && x != self }) != nil)
        {
            /* 
             *   By the time we get here the attachment relationship will 
             *   already have been set up, so we need to undo it again.
             */
            detachFrom(other);
            failCheck('Nejprve budeš muset odendat připojenou lahev. ');
        }
        inherited(other);        
    }
   
;

/*  
 *   ATTACHABLE
 *
 *   The piece of fabric used to repair the ship's hull can be handled quite 
 *   simply. We could have used SimpleAttachable or NearbyAttachable for 
 *   this job, but here we'll illustrate a fairly simple example of using 
 *   the base Attachable class.
 */
fabric: Attachable, Thing 'šedivý kovový čtverec/tkanina' 'čtverec z tkaniny' *2
    "Je to přes metr velký čtverec z kovově šedé tkaniny.
    <<isAttachedTo(lqWall) ? 'Nyní když je' : 'Když by byla'>> připevněná ke
    trupu lodi na levoboku přes díru, měla by zajistit vzduchotěsnou ucpávku. "
    
    /* 
     *   Don't allow the fabric to be detached from the hull once it's 
     *   attached.
     */
    canDetachFrom(obj) { return nil; }
    
    /* 
     *   The only thing we can attach the fabric to is the starboard wall in 
     *   the living quarters.
     */
    canAttachTo(obj) { return obj == lqWall; }
    
    /* 
     *   handleAttach() is a standard library method of Attachable we can 
     *   use to handle the effects of attaching one object to another. Here 
     *   we want the fabric to be moved to the wall (we don't want the 
     *   player character to be left still holding it after it's attached), 
     *   but we don't want the fabric to be listed any more as an 
     *   independent object on the room; we also want to remove the hole, 
     *   since once the fabric is covering it, it's effectively no longer 
     *   there.
     */    
    handleAttach(other)
    {
        if(other == lqWall)
        {
            gapingHole.moveInto(nil);
            moveInto(lqWall);
            isListed = nil;
            "Přiložil jsi tkaninu kolem díry a zcela ji zakryl. Vnější hrany
                tkaniny přilnuly k vnitřní části trupu lodi a vytvořily těsnění,
                které by mělo stačit k obnovení tlaku v lodi. ";
        }
    }
    
    /* Explain why we can't detach the fabric from the wall. */
    cannotDetachMsgFor(obj)
    {
        return obj == lqWall ? 'Teď, když jsi díru zakryl, už ji nechceš znovu
            odkrývat. ' : inherited(obj);

    }
    
    /* 
     *   Attaching at Attachable to something doesn't prevent it from being 
     *   taken. Here we'll make being detached a precondition of being 
     *   taken. An attempt to take the fabric once it's been attached to the 
     *   wall will then be blocked by the failure to detach it.
     */
    dobjFor(Take) { preCond = static inherited + objNotAttached }
    dobjFor(TakeFrom) { preCond = static inherited + objNotAttached }
    
    gcName = 'čtverce z tkaniny, čtverci z tkaniny, čtverec z tkaniny,
	čtverci z tkaniny, čtvercem z tkaniny'
    gcVocab = 'šedivého šedivému šedivém šedivým kovového kovovému kovovém
	kovovým čtverce/čtverci/čtvercem/tkaniny/tkanině/tkaninu/tkaninou'
;

modify Attachable
    
    /*  
     *   Treat FASTEN and UNFASTEN as synonyms for ATTACH and DETACH on all 
     *   Attachables.
     */
    
    dobjFor(FastenTo) asDobjFor(AttachTo)
    iobjFor(FastenTo) asIobjFor(AttachTo)
    dobjFor(Unfasten) asDobjFor(Detach)
    dobjFor(UnfastenFrom) asDobjFor(DetachFrom)
    iobjFor(UnfastenFrom) asIobjFor(DetachFrom)
    
;
