/*
 *   TADS Chat by Tomas Blaha (tomasb at kapsa dot cz)
 *
 *   An integrated chat extension for web multiplayer cooperative playing allows
 *   user to open chat window right inside the running game. Just click on the
 *   "Open Chat Window" command in drop down menu and transcript window is
 *   narrowed to make room for chat beside it. When first player opens chat
 *   window it will be opened in every connected user too.
 *
 *   Based on TADS version 3.1.1
 */

#include <adv3.h>
#include <cs_cz.h>

/* ------------------------------------------------------------------------ */
/*
 *   The chat window shares much of the funcionality with the command
 *   window. We only change processing of received input.
 */
transient chatWin: WebCommandWin
    shown = nil
    /*
     *   Receive input from the client
     */
    receiveInput(req, query)
    {
        /* remember the text */
        lastInput = query['txt'];

        /* remember the source of the input */
        lastInputClient = ClientSession.find(req);

        /* get the user who entered the command */
        local user = (lastInputClient != nil
                      ? lastInputClient.screenName : '');

        /* tell any other windows listening in about the new input */
        sendWinEvent('<closeInputLine>'
                     + (lastInput != nil ?
                        '<text><<lastInput.htmlify()>></text>' : '')
                     + '<user><<user.htmlify()>></user>'
                     + '</closeInputLine>');

        /* reset to the start of the line in the output context */
        sthCtx.resetLine();

        /* add input into the history for late-joining players or reload */
        textbufToScrollback('[' + user + '] ' + lastInput);

        /* send a new prompt */
        write('<br />&gt; ');
        flushWin();

        /* set the UI state */
        mode = 'inputLine';
        isInputOpen = true;

        /* send the inputLine event to the client */
        sendWinEvent('<inputLine/>');
    }

    /* my virtual path, and the actual resource file location */
    vpath = '/chatwin.htm'
    src = 'webuires/chatwin.htm'
;

/* ------------------------------------------------------------------------ */

openChatPage: WebResource
    /* my virtual path */
    vpath = '/webui/showChat'

    processRequest(req, query)
    {
        req.sendReply('<?xml version="1.0"?><reply><ok/></reply>',
            'text/xml', 200);

        webMainWin.createFrame(commandWin, 'command',
            '0, statusline.bottom, 60%, 100%');
        webMainWin.createFrame(chatWin, 'chat',
            '60%, statusline.bottom, 40%, 100%');

        chatWin.shown = true;
        //<form action="/webui/setScreenName?name=xxx" method="get"><input type="submit" value="ok"></form>
        chatWin.write('&gt; ');

        /* flush buffered output */
        chatWin.flushWin();

        /* set the UI state */
        chatWin.mode = 'inputLine';
        chatWin.isInputOpen = true;

        /* send the inputLine event to the client */
        chatWin.sendWinEvent('<inputLine/>');
    }
;

closeChatPage: WebResource
    /* my virtual path */
    vpath = '/webui/hideChat'

    processRequest(req, query)
    {
        req.sendReply('<?xml version="1.0"?><reply><ok/></reply>',
            'text/xml', 200);

        webMainWin.createFrame(commandWin, 'command',
            '0, statusline.bottom, 100%, 100%');
        webMainWin.createFrame(chatWin, 'chat',
            '100%, statusline.bottom, 0%, 100%');

        chatWin.shown = nil;
        //<form action="/webui/setScreenName?name=xxx" method="get"><input type="submit" value="ok"></form>

        /* flush buffered output */
        chatWin.flushWin();

        /* set the UI state */
        chatWin.mode = 'inputLine';
        chatWin.isInputOpen = true;

        /* send the inputLine event to the client */
        chatWin.sendWinEvent('<inputLine/>');
    }
;

/* ------------------------------------------------------------------------ */

/*
 *   Initialize the display.  We call this when we first enter the
 *   interpreter, and again at each RESTART, to set up the main game
 *   window's initial layout.  We set up the conventional IF screen layout,
 *   with the status line across the top and the transcript/command window
 *   filling the rest of the display...
 */
modify initDisplay()
{
    /* set up the command window and status line */
    if(chatWin.shown)
        webMainWin.createFrame(commandWin, 'command',
                           '0, statusline.bottom, 60%, 100%');
    else
        webMainWin.createFrame(commandWin, 'command',
                           '0, statusline.bottom, 100%, 100%');
    webMainWin.createFrame(statuslineBanner, 'statusline',
                           '0, 0, 100%, content.height');

    /* capture the title string */
    local title = mainOutputStream.captureOutput(
        {: gameMain.setGameTitle() });

    /* parse out the contents of the <title> tag */
    if (rexSearch('<nocase>[<]title[>](.*)[<]/title[>]', title))
        title = rexGroup(1)[3];

    /* initialize the statusline window object */
    statuslineBanner.init();
    statusLine.statusDispMode = StatusModeBrowser;

    /* set the title */
    webMainWin.setTitle(title);

    /* get the session parameters from the arguments */
    local arg = libGlobal.getCommandSwitch('-gameid=');
    if (arg != nil && arg != '')
        webSession.launcherGameID = arg;

    arg = libGlobal.getCommandSwitch('-storagesid=');
    if (arg != nil && arg != '')
        webSession.storageSID = arg;

    arg = libGlobal.getCommandSwitch('-username=');
    if (arg != nil && arg != '')
        webSession.launcherUsername = arg;
}